!===========================================================================
!
! Modules:
!
! scalapack_m    Originally By DAS
!
!   Functions, types, and interfaces for ScaLAPACK/BLACS.
!   Interfaces are from http://www.netlib.org/scalapack/tools, double, complex16
!   and from http://www.netlib.org/blacs/BLACS/QRef.html (entered manually...)
!   Every ScaLAPACK/BLACS function used in the code should be listed here, and this
!   module should be used in every routine containing ScaLAPACK/BLACS calls to ensure
!   the argument types are correct.
!
!============================================================================

#include "f_defs.h"

module scalapack_m

  use global_m
  implicit none

  public ::           &
    scalapack,        &
    blacs_setup,      &
    layout_scalapack

!-----------------------------

  type scalapack
    integer :: nprow
    integer :: npcol
    integer :: nbl
    integer :: myprow
    integer :: mypcol
    integer :: npr
    integer :: npc
    integer :: nqrhs
    integer :: icntxt
    integer, pointer :: npcd(:)
    integer, pointer :: nprd(:)
    integer, pointer :: isrtxrow(:)
    integer, pointer :: isrtxcol(:)
    integer, pointer :: imycol(:)
    integer, pointer :: imyrow(:)
    integer, pointer :: imycold(:,:)
    integer, pointer :: imyrowd(:,:)
  end type scalapack

! SCALAPACK
  interface
    INTEGER FUNCTION NUMROC( N, NB, IPROC, ISRCPROC, NPROCS )
      INTEGER              IPROC, ISRCPROC, N, NB, NPROCS
    end FUNCTION NUMROC
  end interface

  interface
    INTEGER FUNCTION ICEIL( INUM, IDENOM )
      INTEGER            IDENOM, INUM
    end FUNCTION ICEIL
  end interface

  interface
    SUBROUTINE DESCINIT( DESC, M, N, MB, NB, IRSRC, ICSRC, ICTXT, LLD, INFO )
      INTEGER            ICSRC, ICTXT, INFO, IRSRC, LLD, M, MB, N, NB
      INTEGER            DESC( * )
    end SUBROUTINE DESCINIT
  end interface

  interface
    SUBROUTINE PDGESV( N, NRHS, A, IA, JA, DESCA, IPIV, B, IB, JB, DESCB, INFO )
      INTEGER            IA, IB, INFO, JA, JB, N, NRHS
      INTEGER            DESCA( * ), DESCB( * ), IPIV( * )
      DOUBLE PRECISION   A( * ), B( * )
    end SUBROUTINE PDGESV
  end interface

  interface
    SUBROUTINE PZGESV( N, NRHS, A, IA, JA, DESCA, IPIV, B, IB, JB, DESCB, INFO )
      INTEGER            IA, IB, INFO, JA, JB, N, NRHS
      INTEGER            DESCA( * ), DESCB( * ), IPIV( * )
      COMPLEX*16         A( * ), B( * )
    end SUBROUTINE PZGESV
  end interface

  interface
    SUBROUTINE PDSYEVX( JOBZ, RANGE, UPLO, N, A, IA, JA, DESCA, VL, &
      VU, IL, IU, ABSTOL, M, NZ, W, ORFAC, Z, IZ, JZ, DESCZ, WORK, LWORK, IWORK, LIWORK, IFAIL, &
      ICLUSTR, GAP, INFO )
      CHARACTER          JOBZ, RANGE, UPLO
      INTEGER            IA, IL, INFO, IU, IZ, JA, JZ, LIWORK, LWORK, M, N, NZ
      DOUBLE PRECISION   ABSTOL, ORFAC, VL, VU
      INTEGER            DESCA( * ), DESCZ( * ), ICLUSTR( * ), IFAIL( * ), IWORK( * )
      DOUBLE PRECISION   A( * ), GAP( * ), W( * ), WORK( * ), Z( * )
    end SUBROUTINE PDSYEVX
  end interface

  interface
    SUBROUTINE PDGEQRF( M, N, A, IA, JA, DESCA, TAU, WORK, LWORK, INFO )
      INTEGER            IA, INFO, JA, LWORK, M, N
      INTEGER            DESCA( * )
      DOUBLE PRECISION   A( * ), TAU( * ), WORK( * )
    end SUBROUTINE PDGEQRF
  end interface

  interface
    SUBROUTINE PZGEQRF( M, N, A, IA, JA, DESCA, TAU, WORK, LWORK, INFO )
      INTEGER            IA, INFO, JA, LWORK, M, N
      INTEGER            DESCA( * )
      COMPLEX*16         A( * ), TAU( * ), WORK( * )
    end SUBROUTINE PZGEQRF
  end interface

  interface
    subroutine PZHEEVX( JOBZ, RANGE, UPLO, N, A, IA, JA, DESCA, VL, &
      VU, IL, IU, ABSTOL, M, NZ, W, ORFAC, Z, IZ,                   &
      JZ, DESCZ, WORK, LWORK, RWORK, LRWORK, IWORK,                 &
      LIWORK, IFAIL, ICLUSTR, GAP, INFO )
      character          JOBZ, RANGE, UPLO
      integer            IA, IL, INFO, IU, IZ, JA, JZ, LIWORK, LRWORK, LWORK, M, N, NZ
      double precision   ABSTOL, ORFAC, VL, VU
      integer            DESCA( * ), DESCZ( * ), ICLUSTR( * ), IFAIL( * ), IWORK( * )
      double precision   GAP( * ), RWORK( * ), W( * )
      complex*16         A( * ), WORK( * ), Z( * )
    end subroutine PZHEEVX
  end interface

! BLACS
  interface
    subroutine blacs_get(icontxt, what, val)
      integer, intent(in)  :: icontxt
      integer, intent(in)  :: what
      integer, intent(out) :: val
    end subroutine blacs_get
  end interface

  interface
    subroutine blacs_gridinit(icontxt, order, nprow, npcol)
      integer,   intent(inout) :: icontxt
      character, intent(in)    :: order
      integer,   intent(in)    :: nprow
      integer,   intent(in)    :: npcol
    end subroutine blacs_gridinit
  end interface

  interface
    subroutine blacs_gridexit(icontxt)
      integer, intent(in)  :: icontxt
    end subroutine blacs_gridexit
  end interface

  interface
    subroutine blacs_exit(icontxt)
      integer, intent(in)  :: icontxt
    end subroutine blacs_exit
  end interface

  interface
    subroutine blacs_gridinfo(icontxt, nprow, npcol, myprow, mypcol)
      integer, intent(in)  :: icontxt
      integer, intent(out) :: nprow
      integer, intent(out) :: npcol
      integer, intent(out) :: myprow
      integer, intent(out) :: mypcol
    end subroutine blacs_gridinfo
  end interface

contains

! Originally by AC, last modified 6/12/2008 (JRD)
!    Figures out a p by q processor grid layout for the scalapack library.
!    This p by q grid is used to partition the matrix with a block size b.
!    The goal is to get a processor grid which is as close to "square" as
!    possible. For more details, see scalapack documentation.
!
!    Input         nproc          number of processors
!                  matsize        size of matrix
!
!    Output        b              block size
!                  p              processor grid row
!                  q              processor grid column
  subroutine layout_scalapack(matsize, b, nproc, p, q)    
    integer, intent(in) :: matsize
    integer, intent(out) :: b
    integer, intent(in) :: nproc
    integer, intent(out) :: p, q
    
    integer :: i
    
    PUSH_SUB(layout_scalapack)
    
!------------------
! Find processor grid

    p = int(sqrt(dble(nproc) + 1.0d-6))
    
    do i = p, 1, -1
      if(mod(nproc, i) .eq. 0) exit
    enddo
    
    p = i
    q = nproc/p

!-------------------
! Now for the block size

    b = min(32, matsize/(max(p, q)))

!-------------------
! Ensure nonzero

    b = max(b, 1)

    POP_SUB(layout_scalapack)

    return
  end subroutine layout_scalapack

  subroutine blacs_setup(scal, size, is_row_order)
    type(scalapack), intent(inout) :: scal ! other elements might have been set earlier
    integer, intent(in) :: size
    logical, intent(in) :: is_row_order 
    
    character :: order

    PUSH_SUB(blacs_setup)

#ifdef USESCALAPACK
    call layout_scalapack(size, scal%nbl, peinf%npes, &
      scal%nprow, scal%npcol)

    if(is_row_order) then
      order = 'r'
    else
      order = 'c'
    endif

    call blacs_get(-1, 0, scal%icntxt)
    call blacs_gridinit(scal%icntxt,order,scal%nprow, scal%npcol)
    call blacs_gridinfo(scal%icntxt, scal%nprow, scal%npcol, scal%myprow, scal%mypcol)

    scal%npr = numroc(size, scal%nbl, scal%myprow, 0, scal%nprow)
    scal%npc = numroc(size, scal%nbl, scal%mypcol, 0, scal%npcol)
    scal%nqrhs = numroc(size, scal%nbl, scal%mypcol, 0, scal%npcol)

    if (peinf%inode.eq.0) then
      write(6,*) ' '
      write(6,'(a,i3,a,i3,a,i3)') 'BLACS PROCESSOR GRID ', scal%nprow, ' x ', scal%npcol, '; BLOCKSIZE = ', scal%nbl
      write(6,*) ' '
    endif
    
    if(scal%myprow.eq.-1) then
      call die('BLACS initialization returned myprow = -1')
    endif
#else
    scal%npr=size
    scal%npc=size
    scal%nqrhs=size
    scal%nbl=size
    scal%nprow=1
    scal%npcol=1
    scal%myprow=0
    scal%mypcol=0
#endif

    POP_SUB(blacs_setup)
    return
  end subroutine blacs_setup

end module scalapack_m
