!===============================================================================
!
! Routines:
!
! 1. checkbz()        Originally By gsm       Last Modified 7/29/2010 (gsm)
!
!    Check that the Brillouin Zone generated by subroutine fullbz is
!    identical to the original full Brillouin Zone. Subroutine fullbz
!    constructs the Brillouin Zone by unfolding the irreducible wedge
!    with all the symmetries of the space group of the crystal. If the
!    irreducible wedge has missing k-points (for example from using too
!    many symmetries in kgrid.x), the full Brillouin Zone will also have
!    missing k-points.
!
!    For the unshifted grid, fullbz generates the original full grid
!    from the irreducible wedge.
!
!    For the grid shifted by half a grid step, fullbz doubles the grid size.
!    For fcc-Si, (4 4 4 0.5 0.5 0.5) becomes (8 8 8 0.0 0.0 0.0) where half
!    the points uniformly distributed across the grid are missing.
!
!    For the randomly-shifted grid, if symmetries are allowed, fullbz
!    generates a non-uniform grid with the points clustered together.
!    You should never allow symmetries for the randomly-shifted grid.
!
!    For the grid shifted by half a grid step, checkbz would print "extra points"
!    warning message. There is nothing to worry about, so we set
!    allow_half_shift to .true. to suppress this warning message in this case.
!    However, if you see "missing points" warning message, it may indicate a
!    problem with your k-point sampling.
!
!===============================================================================

#include "f_defs.h"

subroutine checkbz(nfk,fk,kgrid,kshift,bdot, &
  filename,kqchar,wignerseitz,freplacebz,fwritebz)

  use global_m
  use misc_m
  implicit none

  integer, intent(inout) :: nfk
  real(DP), intent(inout) :: fk(3,nfk)
  integer, intent(inout) :: kgrid(3)
  real(DP), intent(in) :: kshift(3)
  real(DP), intent(in) :: bdot(3,3)
  character(len=16), intent(in) :: filename
  character, intent(in) :: kqchar
  logical, intent(in) :: wignerseitz,freplacebz,fwritebz
  
  logical, parameter :: allow_half_shift = .true.
  integer, parameter :: ncell = 3
  
  logical :: f1,f2,f3,flag_half_shift
  integer :: i,j,i1,i2,i3,n,gpt(3)
  real(DP) :: l1,l2,k1(3),k2(3),kpt(3)
  real(DP), allocatable :: k(:,:)
  real(DP), allocatable :: kref(:,:)
  character(len=128) :: tmpstr
  
  PUSH_SUB(checkbz)

! Identify the grid type
! Print a warning message if the grid type is unknown
! FHJ: Is this really necessary?!

  kpt(:)=kshift(:)
  call k_range(kpt, gpt, TOL_Small)

  f1=.true.
  f2=.true.
  f3=.true.

  do i=1,3
! FHJ: ignore dimensions that have only one kpt
    if (kpt(i)<2) cycle
  ! the unshifted grid
    f1=f1.and.(abs(kpt(i)).lt.TOL_Small)
  ! the grid shifted by half a grid step
    f2=f2.and.(abs(kpt(i)-0.5d0).lt.TOL_Small)
  ! the randomly-shifted grid
    f3=f3.and.(abs(kpt(i)).gt.TOL_Small.and. abs(kpt(1)-0.5d0).gt.TOL_Small)
  enddo


  if (.not.f1.and..not.f2.and..not.f3) then
    if (peinf%inode.eq.0) write(0,901) kqchar, trim(filename)
  endif
  
  flag_half_shift = f2
  
! Find the number of k-points in the full Brillouin Zone

  n=kgrid(1)*kgrid(2)*kgrid(3)
  if (n.le.0) then
    if (peinf%inode.eq.0) write(0,902) kqchar, trim(filename)
    POP_SUB(checkbz)
    return
  endif

! Allocate array for k-points in the full Brillouin Zone

  SAFE_ALLOCATE(k, (3,n))

! Construct k-points in the full Brillouin Zone

  i=0
  do i1=0,kgrid(1)-1
    do i2=0,kgrid(2)-1
      do i3=0,kgrid(3)-1
        i=i+1
        k(1,i)=(dble(i1)+kshift(1))/dble(kgrid(1))
        k(2,i)=(dble(i2)+kshift(2))/dble(kgrid(2))
        k(3,i)=(dble(i3)+kshift(3))/dble(kgrid(3))
        call k_range(k(:,i), gpt, TOL_Small)
      enddo
    enddo
  enddo

! Construct a Wigner-Seitz box

  if (wignerseitz) then
    do i=1,n
      l2=INF
      do i1=-ncell+1,ncell
        k1(1)=k(1,i)-dble(i1)
        do i2=-ncell+1,ncell
          k1(2)=k(2,i)-dble(i2)
          do i3=-ncell+1,ncell
            k1(3)=k(3,i)-dble(i3)
            l1=DOT_PRODUCT(k1,MATMUL(bdot,k1))
            if (l1.lt.l2) then
              l2=l1
              k2(:)=k1(:)
            endif
          enddo
        enddo
      enddo
      k(:,i)=k2(:)
    enddo
  endif

! Write unfolded BZ and full BZ to files

  if (fwritebz) then
    if (peinf%inode.eq.0) then
      write(tmpstr,801) kqchar, trim(filename)
      call open_file(14, tmpstr, status='replace', form='formatted')
      write(14,803) nfk
      do i=1,nfk
        write(14,804) i,fk(:,i)
      enddo
      call close_file(14)
      write(tmpstr,802) kqchar, trim(filename)
      call open_file(14, tmpstr, status='replace', form='formatted')
      write(14,803) n
      do i=1,n
        write(14,804) i,k(:,i)
      enddo
      call close_file(14)
    endif
  endif

! Replace unfolded BZ with full BZ

  if (freplacebz) then
    if (n.le.nfk) then
      nfk=n
      do i=1,n
        fk(:,i)=k(:,i)
      enddo
    else
      call die('checkbz: failed replacebz')
    endif
  endif

! Before comparing k-points translate from Wigner-Seitz box
! to [0,1) interval

  SAFE_ALLOCATE(kref, (3,nfk))
  kref(1:3,1:nfk)=fk(1:3,1:nfk)
  
  if (wignerseitz) then
    do i=1,nfk
      call k_range(kref(:,i), gpt, TOL_Small)
    enddo
    do i=1,n
      call k_range(k(:,i), gpt, TOL_Small)
    enddo
  endif

! Check that kref(1:3,1:nfk) is a subset of k(1:3,1:n)
! Print a warning message otherwise

  f1=.true.
  do i=1,nfk
    f3=.false.
    do j=1,n
      if (all(abs(kref(1:3,i)-k(1:3,j)).lt.TOL_Small)) f3=.true.
    enddo
    if (.not.f3) f1=.false.
  enddo
  if (.not.f1) then
    if (.not.(flag_half_shift.and.allow_half_shift)) then
      if (peinf%inode.eq.0) write(0,903) trim(filename), kqchar
    endif
  endif

! Check that k(1:3,1:n) is a subset of kref(1:3,1:nfk)
! Print a warning message otherwise

  f2=.true.
  do i=1,n
    f3=.false.
    do j=1,nfk
      if (all(abs(k(1:3,i)-kref(1:3,j)).lt.TOL_Small)) f3=.true.
    enddo
    if (.not.f3) f2=.false.
  enddo
  if (.not.f2) then
    if (peinf%inode.eq.0) write(0,904) trim(filename), kqchar
  endif
  
! Deallocate and finish

  SAFE_DEALLOCATE(k)
  SAFE_DEALLOCATE(kref)

  POP_SUB(checkbz)

  return
  
801 format(a,"_",a,"_unfolded.dat")
802 format(a,"_",a,"_full.dat")
803 format(i5)
804 format(i5,3f13.9)
  
901 format(1x,"WARNING: checkbz: unknown",1x,a, &
      "-grid type in",1x,a,/)
902 format(1x,"WARNING: checkbz: zero",1x,a,"-grid in",1x,a,/)
903 format(1x,"WARNING: checkbz: unfolded BZ from",1x,a,1x, &
      "has extra",1x,a,"-points",/)
904 format(1x,"WARNING: checkbz: unfolded BZ from",1x,a,1x, &
      "has missing",1x,a,"-points",/)
  
end subroutine checkbz
