!=======================================================================================
!
! Module inread_m
!
! (1) inread()          Originally By MLT               Last Modified 7/8/2008 (JRD)
!
!     Read input parameters from file absorption.inp / inteqp.inp.
!
!     input: none
!
!     output: many components of xct
!             nmax :            number of iterations to be performed
!                               (Haydock only!)
!             neig :            number of eigenvectors/eigenvalues
!                               to be computed (diagonalization only!)
!
!=============================================================================================

#include "f_defs.h"

module inread_m

  use global_m
  implicit none

  public :: inread

contains

! supply optionals for absorption, and do not for inteqp
subroutine inread(eqp,xct,flag,nmax,neig,diagonalization)
  type (eqpinfo), intent(out) :: eqp
  type (xctinfo), intent(out) :: xct
  type (flags), intent(out) :: flag
  integer, optional, intent(out) :: nmax,neig
  logical, optional, intent(out) :: diagonalization
      
  character*256 :: blockword,keyword,line,errmsg,filename
  integer :: ii,iostat
  logical :: is_absorption, unknown_keyword
  
  PUSH_SUB(inread)

  is_absorption = present(nmax) .and. present(neig) .and. present(diagonalization)
  if(is_absorption .neqv. (present(nmax) .or. present(neig) .or. present(diagonalization))) then
    call die("inread internal error: all or no optional arguments must be supplied.")
  endif

  if(is_absorption) then
    filename = 'absorption.inp'
  else
    filename = 'inteqp.inp'
  endif

#ifdef MPI
  ! Non-root nodes should wait for root to read the whole file.
  ! That way, we can be sure root gets a chance to write errors before
  ! any die call is issued by another node. Root calls MPI_Barrier below.
  if(peinf%inode /= 0) call MPI_Barrier(MPI_COMM_WORLD, mpierr)
#endif

!--------------------------
! Set default values

  xct%nvband=0
  xct%nvb_co=0
  xct%ncband=0
  if(is_absorption) then
    xct%inteqp=0
  else
    xct%inteqp=1
  endif
  xct%ncb_co=0
  eqp%ev0=0.0d0
  eqp%evs=0.0d0
  eqp%evdel=0.0d0
  eqp%ec0=0.0d0
  eqp%ecs=0.0d0
  eqp%ecdel=0.0d0
  eqp%spl_tck%n=0 
  flag%opr=-1
  flag%spec=0
  if(is_absorption) flag%lor=1
  flag%dtm=0
  flag%vm=0
  flag%eps=0
  flag%eig=0
  flag%krnl=1
  flag%bz0=1
  flag%bzq=1
  flag%bzc=1
  if(is_absorption) nmax=0
  xct%skpt=0
  xct%shift(:)=0.d0
  xct%pol(:)=0.d0
  xct%icutv=0
  xct%iscreen=0
  xct%renorm_transf=.true.
  xct%iwritecoul=0
  xct%iskipinterp=0
  xct%iabsorp0=0
  xct%truncval(1:3)=0.0d0
  xct%dynamic_screening=.false.
  xct%dynamic_energy=0.0d0
  xct%wplasmon=0.0d0
  if(is_absorption) neig=0
  xct%vmin=1
  xct%vmax=0
  xct%rgrid=0
  xct%qflag=1
  xct%efermi_input=0.0d0
  xct%rfermi=.true.
  xct%avgcut=TOL_Zero
  xct%scaling=1.0d0
  xct%iwriteint=1
  xct%freplacebz=.false.
  xct%fwritebz=.false.
  ! inteqp will always override degeneracy
  xct%degeneracy_check_override = .not. is_absorption
  if(is_absorption) then
    xct%averagew=.true.
    xct%eta=0.0d0
    xct%sigma=0.0d0
    xct%gamma=0.0d0
  endif
  xct%eqp_corrections=.false.
  ! inteqp does nothing without eqp_co_corrections
  xct%eqp_co_corrections = .not. is_absorption
  if(is_absorption) diagonalization=.true.

!-----------------------------
! Never ending loop...

  do while(0.eq.0)

! Actually the loop ends when the end of the file is reached

    read(8,'(a256)',iostat=iostat) line
    if(iostat < 0) exit

! Skip comment lines

    if(len_trim(line).eq.0 .or. line(1:1).eq.'#') cycle

! Determine keyword:
    
    keyword=line(1:scan(line," ")-1)
    line=adjustl(line(scan(line," ")+1:256))
    
    unknown_keyword = .false.
    if(trim(keyword).eq.'begin') then
      blockword=line(1:scan(line," ")-1)
      ii=0
      do while(trim(line).ne.'end')
        read(8,'(a256)',iostat=iostat) line
        if(iostat /= 0) then
          write(errmsg,'(3a)') 'The end of the file was reached while reading elements of the ', &
            trim(blockword),' block. '
          call die(errmsg, only_root_writes = .true.)
        endif
        if(trim(line).ne.'end') then
          ii=ii+1
          if(trim(blockword).ne.'XXXX') then
            write(errmsg,'(3a)') 'Unexpected blockword ', trim(blockword), ' was found in ' // trim(filename) // '.'
            call die(errmsg, only_root_writes = .true.)
          end if
        end if
      end do
    elseif(trim(keyword).eq.'number_val_bands_fine') then
      read(line,*,err=110) xct%nvband
    elseif(trim(keyword).eq.'number_val_bands_coarse') then
      read(line,*,err=110) xct%nvb_co
    elseif(trim(keyword).eq.'number_cond_bands_fine') then
      read(line,*,err=110) xct%ncband
    elseif(trim(keyword).eq.'number_cond_bands_coarse') then
      read(line,*,err=110) xct%ncb_co
    elseif(trim(keyword).eq.'evs') then
      read(line,*,err=110) eqp%evs
    elseif(trim(keyword).eq.'ev0') then
      read(line,*,err=110) eqp%ev0
    elseif(trim(keyword).eq.'evdel') then
      read(line,*,err=110) eqp%evdel
    elseif(trim(keyword).eq.'ecs') then
      read(line,*,err=110) eqp%ecs
    elseif(trim(keyword).eq.'ec0') then
      read(line,*,err=110) eqp%ec0
    elseif(trim(keyword).eq.'ecdel') then
      read(line,*,err=110) eqp%ecdel
    elseif(trim(keyword).eq.'cvfit') then
      read(line,*,err=110) eqp%evs,eqp%ev0,eqp%evdel, &
        eqp%ecs,eqp%ec0,eqp%ecdel

! Spline scissors
    elseif(trim(keyword).eq.'spline_scissors') then
      if (peinf%inode==0) then
        write(6,*) 'Reading spline coefficients from `spline_scissors.dat`'
        write(6,*)
        ! read number of pts, knots, coefficients, degree
        call open_file(20,file='spline_scissors.dat',form='formatted',status='old')
        read(20,*) eqp%spl_tck%n
        SAFE_ALLOCATE(eqp%spl_tck%t, (eqp%spl_tck%n))
        SAFE_ALLOCATE(eqp%spl_tck%c, (eqp%spl_tck%n))
        read(20,*) (eqp%spl_tck%t(ii), ii=1,eqp%spl_tck%n)
        read(20,*) (eqp%spl_tck%c(ii), ii=1,eqp%spl_tck%n)
        read(20,*) eqp%spl_tck%k
        close(20)
      endif
#ifdef MPI
      call MPI_BCAST(eqp%spl_tck%n, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, mpierr)
      if(peinf%inode/=0) then
        SAFE_ALLOCATE(eqp%spl_tck%t, (eqp%spl_tck%n))
        SAFE_ALLOCATE(eqp%spl_tck%c, (eqp%spl_tck%n))
      endif
      call MPI_BCAST(eqp%spl_tck%t, eqp%spl_tck%n, MPI_REAL_DP, 0, MPI_COMM_WORLD, mpierr)
      call MPI_BCAST(eqp%spl_tck%c, eqp%spl_tck%n, MPI_REAL_DP, 0, MPI_COMM_WORLD, mpierr)
      call MPI_BCAST(eqp%spl_tck%k, 1, MPI_INTEGER, 0, MPI_COMM_WORLD, mpierr)
#endif

    elseif(trim(keyword).eq.'coarse_grid_points') then
      read(line,*,err=110) xct%nkpt_co
    elseif(trim(keyword).eq.'read_kpoints') then
      xct%skpt=1
    elseif(trim(keyword).eq.'fermi_level') then
      read(line,*,err=110) xct%efermi_input
    elseif(trim(keyword).eq.'fermi_level_absolute') then
      xct%rfermi=.false.
    elseif(trim(keyword).eq.'fermi_level_relative') then
      xct%rfermi=.true.
    elseif(trim(keyword).eq.'no_symmetries_fine_grid') then
      flag%bz0 = 1
    elseif(trim(keyword).eq.'use_symmetries_fine_grid') then
      flag%bz0 = 0
    elseif(trim(keyword).eq.'no_symmetries_shifted_grid') then
      flag%bzq = 1
    elseif(trim(keyword).eq.'use_symmetries_shifted_grid') then
      flag%bzq = 0
    elseif(trim(keyword).eq.'no_symmetries_coarse_grid') then
      flag%bzc = 1
    elseif(trim(keyword).eq.'use_symmetries_coarse_grid') then
      flag%bzc = 0
    elseif(trim(keyword).eq.'lowest_occupied_band') then
      read(line,*,err=110) xct%vmin
    elseif(trim(keyword).eq.'highest_occupied_band') then
      read(line,*,err=110) xct%vmax
    elseif(trim(keyword).eq.'regular_grid') then
      read(line,*,err=110) xct%rgrid
    elseif(trim(keyword).eq.'comm_mpi') then
      xct%iwriteint=1
    elseif(trim(keyword).eq.'comm_disk') then
      xct%iwriteint=0
    elseif(trim(keyword).eq.'fullbz_replace') then
      xct%freplacebz=.true.
    elseif(trim(keyword).eq.'fullbz_write') then
      xct%fwritebz=.true.
    else
      unknown_keyword = .true.
    endif

    if(unknown_keyword .and. is_absorption) then
      unknown_keyword = .false.
      if(trim(keyword).eq.'eqp_corrections') then
        xct%eqp_corrections=.true.
      elseif(trim(keyword).eq.'eqp_co_corrections') then
        xct%eqp_co_corrections=.true.
      elseif(trim(keyword).eq.'energy_resolution') then
        read(line,*,err=110) xct%eta
      elseif(trim(keyword).eq.'energy_resolution_sigma') then
        read(line,*,err=110) xct%sigma
      elseif(trim(keyword).eq.'energy_resolution_gamma') then
        read(line,*,err=110) xct%gamma
      elseif(trim(keyword).eq.'use_velocity') then
        flag%opr=0
      elseif(trim(keyword).eq.'use_momentum') then
        flag%opr=1
      elseif(trim(keyword).eq.'use_dos') then
        flag%opr=2
      elseif(trim(keyword).eq.'read_eigenvalues') then
        flag%spec=1
      elseif(trim(keyword).eq.'read_dtmat') then
        flag%dtm=1
      elseif(trim(keyword).eq.'number_iterations') then
        read(line,*,err=110) nmax
      elseif(trim(keyword).eq.'number_eigenvalues') then
        read(line,*,err=110) neig
      elseif(trim(keyword).eq.'cell_average_cutoff') then
        read(line,*,err=110) xct%avgcut
      elseif(trim(keyword).eq.'lorentzian_broadening') then
        flag%lor=0
      elseif(trim(keyword).eq.'gaussian_broadening') then
        flag%lor=1
      elseif(trim(keyword).eq.'voigt_broadening') then
        flag%lor=2
      elseif(trim(keyword).eq.'q_shift') then
        read(line,*,err=110) (xct%shift(ii),ii=1,3)
      elseif(trim(keyword).eq.'polarization') then
        read(line,*,err=110) (xct%pol(ii),ii=1,3)
      elseif(trim(keyword).eq.'read_vmtxel') then
        flag%vm=1
      elseif(trim(keyword).eq.'read_eps2_moments') then
        flag%vm=2
      elseif(trim(keyword).eq.'read_epsdiag') then
        flag%eps=1
      elseif(trim(keyword).eq.'write_eigenvectors') then
        flag%eig=-1
        read(line,*) flag%eig
      elseif(trim(keyword).eq.'diagonalization') then
        diagonalization = .true.
      elseif(trim(keyword).eq.'haydock') then
        diagonalization = .false.
      elseif(trim(keyword).eq.'spin_triplet') then
        flag%krnl=0
      elseif(trim(keyword).eq.'spin_singlet') then
        flag%krnl=1
      elseif(trim(keyword).eq.'local_fields') then
        flag%krnl=2
      elseif(trim(keyword).eq.'noeh_only') then
        xct%iabsorp0=1
      elseif(trim(keyword).eq.'screening_semiconductor') then
        xct%iscreen=0
      elseif(trim(keyword).eq.'screening_graphene') then
        xct%iscreen=1
      elseif(trim(keyword).eq.'screening_metal') then
        xct%iscreen=2
      elseif(trim(keyword).eq.'spherical_truncation') then
        xct%icutv=2
      elseif(trim(keyword).eq.'cell_wire_truncation') then
        xct%icutv=4
      elseif(trim(keyword).eq.'cell_box_truncation') then
        xct%icutv=5
      elseif(trim(keyword).eq.'cell_slab_truncation') then
        xct%icutv=6
      elseif(trim(keyword).eq.'write_vcoul') then
        xct%iwritecoul=1
      elseif(trim(keyword).eq.'skip_interpolation') then
        xct%iskipinterp=1
      elseif(trim(keyword).eq.'kernel_scaling') then
        read(line,*,err=110) xct%scaling
      elseif(trim(keyword).eq.'coulomb_truncation_radius') then
        read(line,*,err=110) xct%truncval(1)
      elseif(trim(keyword).eq.'degeneracy_check_override') then
        xct%degeneracy_check_override=.true.
      elseif(trim(keyword).eq.'average_w') then
        xct%averagew=.true.
      else
        unknown_keyword = .true.
      endif
    endif ! is_absorption

    if(unknown_keyword) then
      write(errmsg,'(3a)') 'Unexpected keyword ', trim(keyword), ' was found in ' // trim(filename) // '.'
      call die(errmsg, only_root_writes = .true.)
    end if
  enddo
  
  if(xct%nvband.eq.0) then
    call die('The number_val_bands_fine keyword could not be found.', only_root_writes = .true.)
  endif
  
  if(xct%ncband.eq.0) then
    call die('The number_cond_bands_fine keyword could not be found.', only_root_writes = .true.)
  endif
  
  if(xct%nvb_co.eq.0) then
    call die('The number_val_bands_coarse keyword could not be found.', only_root_writes = .true.)
  endif
  
  if(xct%ncb_co.eq.0) then
    call die('The number_cond_bands_coarse keyword could not be found.', only_root_writes = .true.)
  endif
  
  if(is_absorption) then ! rest of file does not apply to inteqp
    if(xct%iabsorp0 .eq. 0 .and. flag%spec .eq. 0) then
      if(xct%nkpt_co.eq.0) then
        call die('The coarse_grid_points keyword could not be found.', only_root_writes = .true.)
      endif
    endif

    ! no fundamental reason, but the way the code is written, energy interpolation only occurs if kernel is interpolated
    if(xct%eqp_co_corrections .and. xct%iskipinterp == 1) then
      call die("Cannot do eqp_co_corrections with skip_interpolation. Use eqp_corrections instead.", only_root_writes = .true.)
    endif

! FHJ: noeh_only doesn`t interpolate onto fine grid, so eqp_co won`t work
    if(xct%eqp_co_corrections .and. (xct%iabsorp0==1)) then
      if (peinf%inode==0) then
        write(0,*) 'Can`t use `eqp_co_corrections` with `noeh_only`. Either:'
        write(0,*) ' (1) Run inteqp, then run absorption with eqp_corrections, or'
        write(0,*) ' (2) Remove the `noeh_only` flag and run absorption directly.'
      endif 
      call die('Can`t use `eqp_co_corrections` and `noeh_only` simultaneously.',  only_root_writes = .true.)
    endif

    if ((abs(xct%efermi_input) > TOL_Zero .or. .not. xct%rfermi)) then ! Fermi level is being adjusted
      if((xct%eqp_corrections .neqv. xct%eqp_co_corrections) .and. xct%iskipinterp == 0) then
        if(peinf%inode == 0) then
          write(0,'(a)') "If Fermi level is adjusted and interpolation is used, you must use both or neither"
          write(0,'(a)') "of eqp_corrections and eqp_co_corrections. (You can get eqp.dat by running inteqp.)"
        endif
        call die("Cannot use this combination of Fermi adjustment / eqp_corrections / interpolation.", only_root_writes = .true.)
      endif
    endif
    ! eqp_corrections only cannot be used because we would need to 'anti-interpolate' to the coarse grid
    ! in order to correctly reset occupations (ifmax) on the coarse grid.
    ! eqp_co_corrections only cannot be used because then the Fermi level is set by the coarse grid, and
    ! it is cumbersome to reset the fine grid Fermi level since WFN_fi was already handled. --DAS
    ! But if we are not interpolating, then it is ok, since eqp.dat will be used on both.

    if(flag%opr.eq.0) then
      if(peinf%inode.eq.0) then
        write(6,*) 'Using the velocity operator'
        write(6,*)
      endif
      if (xct%iskipinterp.eq.1) then
        call die('skip_interpolation is incompatible with velocity operator', only_root_writes = .true.)
      endif
    elseif(flag%opr.eq.1) then
      if(peinf%inode.eq.0) then
        write(6,*) 'Using the momentum operator'
        write(6,*)
        write(0,999)
        write(0,*)
999     format(1x,'**************************************************',/, &
          1x,'**                                              **',/, &
          1x,'**                   WARNING:                   **',/, &
          1x,'**      THE MOMENTUM OPERATOR IS NOT EXACT      **',/, &
          1x,'**  ONLY USE IT IF YOU KNOW WHAT YOU ARE DOING  **',/, &
          1x,'**  OTHERWISE YOU SHOULD USE VELOCITY OPERATOR  **',/, &
          1x,'**    for details see equations (29) and (30)   **',/, &
          1x,'**    of Rohlfing & Louie PRB 62, 4927 (2000)   **',/, &
          1x,'**                                              **',/, &
          1x,'**************************************************')
      endif
    elseif(flag%opr.eq.2) then
      if(peinf%inode.eq.0) then
        write(6,*) 'Using the Joint density of states operator'
        write(6,*)
      endif
    else
      call die("No flag for operator", only_root_writes = .true.)
    endif

    ! momentum and JDOS do not have q-shift. specifying one mistakenly will break things without this.
    if(flag%opr /= 0) then
      xct%shift(1:3) = ZERO
    endif
  
    if(abs(xct%eta).lt.TOL_Zero) then
      call die('The energy_resolution keyword could not be found.', only_root_writes = .true.)
    endif
  
    if(flag%lor.eq.2.and.abs(xct%sigma).lt.TOL_Small) then
      call die('Voigt requires energy_resolution_sigma > 0 and energy_resolution_gamma >= 0', &
        only_root_writes = .true.)
    endif

! JRD: Check compatibility

    if (xct%qflag .eq. 0 .and. flag%opr .eq. 1) then
      call die('Momentum and Finite_q are incompatible',  only_root_writes = .true.)
    end if

! JRD: Make a note if we have finite Q

    if(peinf%inode.eq.0 .and. xct%qflag.eq.0) then
      write(6,700)xct%shift(:)
    endif
700 format(1x,'We are doing a Finite Q Calculation',/, &
      1x,'Q =',3f10.6,/)

! JRD: What screening is present?

    if(peinf%inode.eq.0) then
      if(xct%iscreen.eq.0) then
        write(6,800)
      elseif(xct%iscreen.eq.1) then
        write(6,801)
      elseif(xct%iscreen.eq.2) then
        write(6,802)
      else
        call die('Need to specify screening type')
      endif
    endif
800 format(1x,'Running with Semiconductor Screening',/)
801 format(1x,'Running with Graphene Screening',/)
802 format(1x,'Running with Metal Screening',/)

  if(peinf%inode == 0) then
    if(xct%iwriteint == 1) then ! comm_mpi
      if(peinf%npes > 1) then
        write(6,803)
      else
        write(6,805)
      endif
    else ! comm_disk
      write(6,804)
    endif
  endif
803 format(1x,'We are communicating via MPI',/)
804 format(1x,'We are communicating via disk',/)
805 format(1x,'We are not communicating',/)

! JRD: What truncation scheme are we using?

    if(peinf%inode.eq.0) then
      if(xct%icutv.eq.0) then
        write(6,900)
      elseif(xct%icutv.eq.2) then
        write(6,902)
        write(6,912)xct%truncval(1)
      elseif(xct%icutv.eq.4) then
        write(6,904)
      elseif(xct%icutv.eq.5) then
        write(6,905)
      elseif(xct%icutv.eq.6) then
        write(6,906)
      elseif(xct%icutv.eq.7) then
        write(6,907)
      endif
    endif
900 format(1x,'We are using no truncation',/)
902 format(1x,'We are using a truncated Coulomb interaction (spherical)')
904 format(1x,'We are using a truncated Coulomb interaction: Cell Wire',/)
905 format(1x,'We are using a truncated Coulomb interaction: Cell Box',/)
906 format(1x,'We are using a truncated Coulomb interaction: Cell Slab',/)
907 format(1x,'We are using a truncated Coulomb interaction: Supercell Box',/)
912 format(1x,'r_cut =',f10.6,/)

  endif ! is_absorption 

#ifdef MPI
  ! root lets the others go after it is done reading (see beginning of function)
  if(peinf%inode == 0) call MPI_Barrier(MPI_COMM_WORLD, mpierr)
#endif

  POP_SUB(inread)
  
  return
  
110 write(errmsg,'(3a)') 'Unexpected characters were found while reading the value for the keyword ', &
      trim(keyword), '. '
  call die(errmsg, only_root_writes = .true.)
  
end subroutine inread

end module inread_m
