% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/condeff.R, R/condeff_boot.R
\name{cond_effect}
\alias{cond_effect}
\alias{cond_effect_boot}
\title{Conditional Effects}
\usage{
cond_effect(
  output,
  x = NULL,
  w = NULL,
  w_method = c("sd", "percentile"),
  w_percentiles = c(0.16, 0.5, 0.84),
  w_sd_to_percentiles = NA,
  w_from_mean_in_sd = 1,
  w_values = NULL
)

cond_effect_boot(
  output,
  x = NULL,
  w = NULL,
  ...,
  conf = 0.95,
  nboot = 100,
  boot_args = NULL,
  save_boot_est = TRUE,
  full_output = FALSE,
  do_boot = TRUE
)
}
\arguments{
\item{output}{The output from \code{\link[stats:lm]{stats::lm()}}. It can also accept the output
from
\code{\link[=std_selected]{std_selected()}} or \code{\link[=std_selected_boot]{std_selected_boot()}}.}

\item{x}{The focal variable (independent variable), that is, the
variable with its effect on the outcome variable (dependent)
being moderated. It must be a numeric variable.}

\item{w}{The moderator. Both numeric variables and categorical variables
(character or factor) are supported.}

\item{w_method}{How to define "low", "medium", and "high" for the moderator
levels.
Default is in terms of mean and
standard deviation (SD) of the moderator, \code{"sd"}:
"low", "medium", and "high" are one SD below mean,
mean, and one SD above mean, respectively.
If equal to
\code{"percentile"}, then percentiles of the moderator in
the
dataset are used: "low", "medium", and "high" are
16th, 50th (median), and 84th percentiles, respectively.
Ignored if \code{w} is categorical.}

\item{w_percentiles}{If \code{w_method} is \code{"percentile"}, then this
argument
specifies the three percentiles to be used,
divided by 100.
It must be a
vector of two numbers. The default is
\code{c(.16, .50, .84)},
the 16th, 50th, and 84th percentiles,
which corresponds approximately
to one SD below and above mean in a
normal distribution, respectively.
Ignored if \code{w} is categorical.}

\item{w_sd_to_percentiles}{If \code{w_method} is \code{"percentile"} and
this argument is
set to a number, this number will be
used to
to determine the percentiles to be used.
The
lower percentile is the percentile in a
normal
distribution
that is \code{w_sd_to_percentiles} SD below
the mean.
The upper percentile is the percentile in
a normal
distribution that is \code{w_sd_to_percentiles}
SD
above the mean. Therefore, if
\code{w_sd_to_percentiles} is set to 1, then the
lower
and upper percentiles are 16th and 84th,
respectively. Default is \code{NA}.}

\item{w_from_mean_in_sd}{How many SD from mean is used to define
"low" and
"high" for the moderator. Default is 1.
Ignored if \code{w} is categorical.}

\item{w_values}{The values of \code{w} to
be used. Default is \code{NULL}. If a
numeric vector is supplied, these
values will be used to compute the
conditional effects. Other arguments
on generating levels are ignored.
Note that, if \code{w} has been standardized
or centered, these values are for
the standardized or centered \code{w}.
The values will always be sorted.
This argument is ignored if \code{w} is
categorical.}

\item{...}{Arguments to be passed to \code{\link[=cond_effect]{cond_effect()}}.}

\item{conf}{The level of confidence for the confidence interval.
Default is .95, to get 95\% confidence intervals.}

\item{nboot}{The number of bootstrap samples. Default is 100.}

\item{boot_args}{A named list of arguments to be passed to \code{\link[boot:boot]{boot::boot()}}.
Default is \code{NULL}.}

\item{save_boot_est}{If \code{TRUE}, the default, the bootstrap estimates will
be saved in the element
\code{boot_est} of the output.}

\item{full_output}{Whether the full output from \code{\link[boot:boot]{boot::boot()}} will be
returned.
Default is \code{FALSE}. If \code{TRUE}, the full output from
\code{\link[boot:boot]{boot::boot()}} will be saved in the element \code{boot_out}
of the output.}

\item{do_boot}{Whether bootstrapping confidence intervals will be formed.
Default is \code{TRUE}. If \code{FALSE}, all arguments related to
bootstrapping will be ignored.}
}
\value{
\code{\link[=cond_effect]{cond_effect()}} returns a data-frame-like object of the conditional effects.
The class is
\code{cond_effect} and the print method will print additional information of
the conditional effects. Additional information is stored in the
following attributes:
\itemize{
\item \code{call}: The original call.
\item \code{output}: The \code{output} object, such as the output from \code{lm()}.
\item \code{x}, \code{y}, and \code{w}: The three variables used to compute the conditional
effects: focal variable (\code{x}), outcome variable (\code{y}),
and the moderator (\code{w}).
\item \code{w_method}: The method used to determine the values of the moderator
at the selected levels.
\item \code{w_percentiles} The percentiles to use if \code{w_method} = \code{"percentile"}.
\item \code{w_sd_to_percentiles}: If not equal to \code{NA}, this is a scalar, the
number of standard deviation from the mean used to
determine the percentiles for the "low" and "high"
levels of the moderator.
\item \code{w_from_mean_in_sd}: The number of SD above or below the mean, for
determining the "low" and "high" levels of the
moderator if \code{w_method} is \code{"sd"}.
\item \code{w_empirical_percentiles}: The actual percentile levels in the dataset
for the selected
levels of the moderator. A numeric vector.
\item \code{w_empirical_z}: The actual distance from the mean, in SD, of each
selected level of the moderator. A numeric vector.
\item \code{y_standardized}, \code{x_standardized}, and \code{w_standardized}: Each of them
is a logical scalar, indicating whether the outcome
variable, focal variable, and moderator are standardized.
}

\code{\link[=cond_effect_boot]{cond_effect_boot()}} also returns a data-frame-like object of the
conditional effects of the class
\code{cond_effect}, with additional information from the bootstrapping stored
in these attributes:
\itemize{
\item \code{boot_ci}: A data frame of the bootstrap confidence intervals of the
conditional effects.
\item \code{nboot}: The number of bootstrap samples requested.
\item \code{conf}: The level of confidence, in proportion.
\item \code{boot_est}: A matrix of the bootstrap estimates of the conditional effects.
The number of rows equal to \code{nboot}, and the number of columns
equal to the number of levels of the moderator.
\item \code{cond_effect_boot_call}: The call to \code{cond_effect_boot()}.
\item \code{boot_out}: If available, the original output from \code{boot::boot()}.
}
}
\description{
Compute the conditional effects in a moderated regression model.
}
\details{
\code{\link[=cond_effect]{cond_effect()}} uses the centering approach to find the conditional
effect of the focal variable. For each level of the moderator, the value for
this level is subtracted from the moderator scores, and the model is
fitted to the modified data.
The coefficient of the focal variable is then the conditional effect of the
focal variable when the moderator's score is equal this value.

\code{\link[=cond_effect_boot]{cond_effect_boot()}} function is a wrapper of \code{\link[=cond_effect]{cond_effect()}}.
It calls \code{\link[=cond_effect]{cond_effect()}}
once for each bootstrap sample, and then computes the nonparametric
bootstrap percentile confidence intervals (Cheung, Cheung, Lau, Hui,
& Vong, 2022). If the output object is the output of \code{\link[=std_selected]{std_selected()}}
or \code{\link[=std_selected_boot]{std_selected_boot()}}, in which mean-centering and/or standardization
have been conducted, they will be repeated in each bootstrap sample.
Therefore, like \code{\link[=std_selected_boot]{std_selected_boot()}}, it can be used for form
nonparametric bootstrap confidence intervals for standardized
effects, though \code{\link[=cond_effect_boot]{cond_effect_boot()}} does this for the standardized
conditional effects.

This function ignores bootstrapping done by \code{\link[=std_selected_boot]{std_selected_boot()}}. It will
do its own bootstrapping.

If \code{do_boot} is \code{FALSE}, then the object it returns is identical to that
by \code{\link[=cond_effect]{cond_effect()}}.

This function intentionally does not have an argument for setting the seed
for
random number. Users are recommended to set the seed, e.g., using
\code{\link[=set.seed]{set.seed()}}
before calling it, to ensure reproducibility.
}
\section{Functions}{
\itemize{
\item \code{cond_effect_boot()}: A wrapper of \code{\link[=cond_effect]{cond_effect()}} that forms
nonparametric bootstrap confidence intervals.

}}
\examples{

# Load a sample data set

dat <- test_x_1_w_1_v_1_cat1_n_500

# Do a moderated regression by lm
lm_raw <- lm(dv ~ iv*mod + v1 + cat1, dat)
summary(lm_raw)
cond_effect(lm_raw, x = iv, w = mod)

lm_std <- std_selected(lm_raw, to_standardize = ~ iv + mod)
cond_effect(lm_std, x = iv, w = mod)

# Categorical moderator
lm_cat <- lm(dv ~ iv*cat1 + v1, dat)
summary(lm_cat)
cond_effect(lm_cat, x = iv, w = cat1)


# Load a sample data set

dat <- test_x_1_w_1_v_1_cat1_n_500

# Do a moderated regression by lm
lm_raw <- lm(dv ~ iv*mod + v1 + cat1, dat)
summary(lm_raw)

lm_std <- std_selected(lm_raw, to_standardize = ~ iv + mod)
cond_effect(lm_std, x = iv, w = mod)

# Form nonparametric bootstrap confidence intervals
# Use 2000 or even 5000 for nboot in real research
out <- cond_effect_boot(lm_std, x = iv, w = mod, nboot = 50)
out

}
\author{
Shu Fai Cheung \url{https://orcid.org/0000-0002-9871-9448}
}
