# Copyright (c) 2017-2026 Juancarlo Añez (apalala@gmail.com)
# SPDX-License-Identifier: BSD-4-Clause
# from: https://www.unicode.org/charts/nameslist/n_2500.html
from __future__ import annotations

from typing import NamedTuple

""" Box Drawing
    © 2003–2025 Unicode, Inc.
    Unicode and the Unicode Logo are registered trademarks of Unicode, Inc.,
    in the U.S. and other countries. For terms of use and license,
    see https://www.unicode.org/terms_of_use.html.
"""

# NOTE:
#   All of these characters are intended for compatibility
#   with character cell graphic sets in use prior to 1990.


class UC(NamedTuple):
    code: int
    ustr: str
    name: str


# fmt: off
# ruff: noqa: RUF001
BOX_DRAWING = {
    '🔚': UC(0x1F51A, '\U0001F51A', "END WITH LEFTWARDS ARROW ABOVE"),

    '≔': UC(0x2254, '\u2254', "COLON EQUALS"),
    '∅': UC(0x2205, '\u2205', "EMPTY SET"),
    '⚠': UC(0x26A0, '\u26A0', "WARNING SIGN"),
    '→': UC(0x2192, '\u2192', "RIGHTWARDS ARROW"),
    '↱': UC(0x21b1, '\u21b1', "UPWARDS ARROW WITH TIP RIGHTWARDS"),
    '✂': UC(0x2702, '\u2702', "BLACK SCISSORS"),
    '↦': UC(0x21A6, '\u21A6', "RIGHTWARDS ARROW FROM BAR"),
    '⇥': UC(0x21E5, '\u21E5', "RIGHTWARDS ARROW TO BAR"),
    '↠': UC(0x21A0, '\u21A0', "RIGHTWARDS TWO HEADED ARROW"),

    # Light and heavy solid lines
    '─': UC(0x2500, '\u2500', "BOX DRAWINGS LIGHT HORIZONTAL"),
    '━': UC(0x2501, '\u2501', "BOX DRAWINGS HEAVY HORIZONTAL"),
    '│': UC(0x2502, '\u2502', "BOX DRAWINGS LIGHT VERTICAL"),
    '┃': UC(0x2503, '\u2503', "BOX DRAWINGS HEAVY VERTICAL"),

    # Light and heavy dashed lines
    '┄': UC(0x2504, '\u2504', "BOX DRAWINGS LIGHT TRIPLE DASH HORIZONTAL"),
    '┅': UC(0x2505, '\u2505', "BOX DRAWINGS HEAVY TRIPLE DASH HORIZONTAL"),
    '┆': UC(0x2506, '\u2506', "BOX DRAWINGS LIGHT TRIPLE DASH VERTICAL"),
    '┇': UC(0x2507, '\u2507', "BOX DRAWINGS HEAVY TRIPLE DASH VERTICAL"),
    '┈': UC(0x2508, '\u2508', "BOX DRAWINGS LIGHT QUADRUPLE DASH HORIZONTAL"),
    '┉': UC(0x2509, '\u2509', "BOX DRAWINGS HEAVY QUADRUPLE DASH HORIZONTAL"),
    '┊': UC(0x250A, '\u250A', "BOX DRAWINGS LIGHT QUADRUPLE DASH VERTICAL"),
    '┋': UC(0x250B, '\u250B', "BOX DRAWINGS HEAVY QUADRUPLE DASH VERTICAL"),

    # Light and heavy line box components
    '┌': UC(0x250C, '\u250C', "BOX DRAWINGS LIGHT DOWN AND RIGHT"),
    '┍': UC(0x250D, '\u250D', "BOX DRAWINGS DOWN LIGHT AND RIGHT HEAVY"),
    '┎': UC(0x250E, '\u250E', "BOX DRAWINGS DOWN HEAVY AND RIGHT LIGHT"),
    '┏': UC(0x250F, '\u250F', "BOX DRAWINGS HEAVY DOWN AND RIGHT"),
    '┐': UC(0x2510, '\u2510', "BOX DRAWINGS LIGHT DOWN AND LEFT"),
    '┑': UC(0x2511, '\u2511', "BOX DRAWINGS DOWN LIGHT AND LEFT HEAVY"),
    '┒': UC(0x2512, '\u2512', "BOX DRAWINGS DOWN HEAVY AND LEFT LIGHT"),
    '┓': UC(0x2513, '\u2513', "BOX DRAWINGS HEAVY DOWN AND LEFT"),
    '└': UC(0x2514, '\u2514', "BOX DRAWINGS LIGHT UP AND RIGHT"),
    '┕': UC(0x2515, '\u2515', "BOX DRAWINGS UP LIGHT AND RIGHT HEAVY"),
    '┖': UC(0x2516, '\u2516', "BOX DRAWINGS UP HEAVY AND RIGHT LIGHT"),
    '┗': UC(0x2517, '\u2517', "BOX DRAWINGS HEAVY UP AND RIGHT"),
    '┘': UC(0x2518, '\u2518', "BOX DRAWINGS LIGHT UP AND LEFT"),
    '┙': UC(0x2519, '\u2519', "BOX DRAWINGS UP LIGHT AND LEFT HEAVY"),
    '┚': UC(0x251A, '\u251A', "BOX DRAWINGS UP HEAVY AND LEFT LIGHT"),
    '┛': UC(0x251B, '\u251B', "BOX DRAWINGS HEAVY UP AND LEFT"),
    '├': UC(0x251C, '\u251C', "BOX DRAWINGS LIGHT VERTICAL AND RIGHT"),
    '┝': UC(0x251D, '\u251D', "BOX DRAWINGS VERTICAL LIGHT AND RIGHT HEAVY"),
    '┞': UC(0x251E, '\u251E', "BOX DRAWINGS UP HEAVY AND RIGHT DOWN LIGHT"),
    '┟': UC(0x251F, '\u251F', "BOX DRAWINGS DOWN HEAVY AND RIGHT UP LIGHT"),
    '┠': UC(0x2520, '\u2520', "BOX DRAWINGS VERTICAL HEAVY AND RIGHT LIGHT"),
    '┡': UC(0x2521, '\u2521', "BOX DRAWINGS DOWN LIGHT AND RIGHT UP HEAVY"),
    '┢': UC(0x2522, '\u2522', "BOX DRAWINGS UP LIGHT AND RIGHT DOWN HEAVY"),
    '┣': UC(0x2523, '\u2523', "BOX DRAWINGS HEAVY VERTICAL AND RIGHT"),
    '┤': UC(0x2524, '\u2524', "BOX DRAWINGS LIGHT VERTICAL AND LEFT"),
    '┥': UC(0x2525, '\u2525', "BOX DRAWINGS VERTICAL LIGHT AND LEFT HEAVY"),
    '┦': UC(0x2526, '\u2526', "BOX DRAWINGS UP HEAVY AND LEFT DOWN LIGHT"),
    '┧': UC(0x2527, '\u2527', "BOX DRAWINGS DOWN HEAVY AND LEFT UP LIGHT"),
    '┨': UC(0x2528, '\u2528', "BOX DRAWINGS VERTICAL HEAVY AND LEFT LIGHT"),
    '┩': UC(0x2529, '\u2529', "BOX DRAWINGS DOWN LIGHT AND LEFT UP HEAVY"),
    '┪': UC(0x252A, '\u252A', "BOX DRAWINGS UP LIGHT AND LEFT DOWN HEAVY"),
    '┫': UC(0x252B, '\u252B', "BOX DRAWINGS HEAVY VERTICAL AND LEFT"),
    '┬': UC(0x252C, '\u252C', "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL"),
    '┭': UC(0x252D, '\u252D', "BOX DRAWINGS LEFT HEAVY AND RIGHT DOWN LIGHT"),
    '┮': UC(0x252E, '\u252E', "BOX DRAWINGS RIGHT HEAVY AND LEFT DOWN LIGHT"),
    '┯': UC(0x252F, '\u252F', "BOX DRAWINGS DOWN LIGHT AND HORIZONTAL HEAVY"),
    '┰': UC(0x2530, '\u2530', "BOX DRAWINGS DOWN HEAVY AND HORIZONTAL LIGHT"),
    '┱': UC(0x2531, '\u2531', "BOX DRAWINGS RIGHT LIGHT AND LEFT DOWN HEAVY"),
    '┲': UC(0x2532, '\u2532', "BOX DRAWINGS LEFT LIGHT AND RIGHT DOWN HEAVY"),
    '┳': UC(0x2533, '\u2533', "BOX DRAWINGS HEAVY DOWN AND HORIZONTAL"),
    '┴': UC(0x2534, '\u2534', "BOX DRAWINGS LIGHT UP AND HORIZONTAL"),
    '┵': UC(0x2535, '\u2535', "BOX DRAWINGS LEFT HEAVY AND RIGHT UP LIGHT"),
    '┶': UC(0x2536, '\u2536', "BOX DRAWINGS RIGHT HEAVY AND LEFT UP LIGHT"),
    '┷': UC(0x2537, '\u2537', "BOX DRAWINGS UP LIGHT AND HORIZONTAL HEAVY"),
    '┸': UC(0x2538, '\u2538', "BOX DRAWINGS UP HEAVY AND HORIZONTAL LIGHT"),
    '┹': UC(0x2539, '\u2539', "BOX DRAWINGS RIGHT LIGHT AND LEFT UP HEAVY"),
    '┺': UC(0x253A, '\u253A', "BOX DRAWINGS LEFT LIGHT AND RIGHT UP HEAVY"),
    '┻': UC(0x253B, '\u253B', "BOX DRAWINGS HEAVY UP AND HORIZONTAL"),
    '┼': UC(0x253C, '\u253C', "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL"),
    '┽': UC(0x253D, '\u253D', "BOX DRAWINGS LEFT HEAVY AND RIGHT VERTICAL LIGHT"),
    '┾': UC(0x253E, '\u253E', "BOX DRAWINGS RIGHT HEAVY AND LEFT VERTICAL LIGHT"),
    '┿': UC(0x253F, '\u253F', "BOX DRAWINGS VERTICAL LIGHT AND HORIZONTAL HEAVY"),
    '╀': UC(0x2540, '\u2540', "BOX DRAWINGS UP HEAVY AND DOWN HORIZONTAL LIGHT"),
    '╁': UC(0x2541, '\u2541', "BOX DRAWINGS DOWN HEAVY AND UP HORIZONTAL LIGHT"),
    '╂': UC(0x2542, '\u2542', "BOX DRAWINGS VERTICAL HEAVY AND HORIZONTAL LIGHT"),
    '╃': UC(0x2543, '\u2543', "BOX DRAWINGS LEFT UP HEAVY AND RIGHT DOWN LIGHT"),
    '╄': UC(0x2544, '\u2544', "BOX DRAWINGS RIGHT UP HEAVY AND LEFT DOWN LIGHT"),
    '╅': UC(0x2545, '\u2545', "BOX DRAWINGS LEFT DOWN HEAVY AND RIGHT UP LIGHT"),
    '╆': UC(0x2546, '\u2546', "BOX DRAWINGS RIGHT DOWN HEAVY AND LEFT UP LIGHT"),
    '╇': UC(0x2547, '\u2547', "BOX DRAWINGS DOWN LIGHT AND UP HORIZONTAL HEAVY"),
    '╈': UC(0x2548, '\u2548', "BOX DRAWINGS UP LIGHT AND DOWN HORIZONTAL HEAVY"),
    '╉': UC(0x2549, '\u2549', "BOX DRAWINGS RIGHT LIGHT AND LEFT VERTICAL HEAVY"),
    '╊': UC(0x254A, '\u254A', "BOX DRAWINGS LEFT LIGHT AND RIGHT VERTICAL HEAVY"),
    '╋': UC(0x254B, '\u254B', "BOX DRAWINGS HEAVY VERTICAL AND HORIZONTAL"),

    # Light and heavy dashed lines
    '╌': UC(0x254C, '\u254C', "BOX DRAWINGS LIGHT DOUBLE DASH HORIZONTAL"),
    '╍': UC(0x254D, '\u254D', "BOX DRAWINGS HEAVY DOUBLE DASH HORIZONTAL"),
    '╎': UC(0x254E, '\u254E', "BOX DRAWINGS LIGHT DOUBLE DASH VERTICAL"),
    '╏': UC(0x254F, '\u254F', "BOX DRAWINGS HEAVY DOUBLE DASH VERTICAL"),

    # Double lines
    '═': UC(0x2550, '\u2550', "BOX DRAWINGS DOUBLE HORIZONTAL"),
    '║': UC(0x2551, '\u2551', "BOX DRAWINGS DOUBLE VERTICAL"),

    # Light and double line box components
    '╒': UC(0x2552, '\u2552', "BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE"),
    '╓': UC(0x2553, '\u2553', "BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE"),
    '╔': UC(0x2554, '\u2554', "BOX DRAWINGS DOUBLE DOWN AND RIGHT"),
    '╕': UC(0x2555, '\u2555', "BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE"),
    '╖': UC(0x2556, '\u2556', "BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE"),
    '╗': UC(0x2557, '\u2557', "BOX DRAWINGS DOUBLE DOWN AND LEFT"),
    '╘': UC(0x2558, '\u2558', "BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE"),
    '╙': UC(0x2559, '\u2559', "BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE"),
    '╚': UC(0x255A, '\u255A', "BOX DRAWINGS DOUBLE UP AND RIGHT"),
    '╛': UC(0x255B, '\u255B', "BOX DRAWINGS UP SINGLE AND LEFT DOUBLE"),
    '╜': UC(0x255C, '\u255C', "BOX DRAWINGS UP DOUBLE AND LEFT SINGLE"),
    '╝': UC(0x255D, '\u255D', "BOX DRAWINGS DOUBLE UP AND LEFT"),
    '╞': UC(0x255E, '\u255E', "BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE"),
    '╟': UC(0x255F, '\u255F', "BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE"),
    '╠': UC(0x2560, '\u2560', "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT"),
    '╡': UC(0x2561, '\u2561', "BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE"),
    '╢': UC(0x2562, '\u2562', "BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE"),
    '╣': UC(0x2563, '\u2563', "BOX DRAWINGS DOUBLE VERTICAL AND LEFT"),
    '╤': UC(0x2564, '\u2564', "BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE"),
    '╥': UC(0x2565, '\u2565', "BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE"),
    '╦': UC(0x2566, '\u2566', "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL"),
    '╧': UC(0x2567, '\u2567', "BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE"),
    '╨': UC(0x2568, '\u2568', "BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE"),
    '╩': UC(0x2569, '\u2569', "BOX DRAWINGS DOUBLE UP AND HORIZONTAL"),
    '╪': UC(0x256A, '\u256A', "BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE"),
    '╫': UC(0x256B, '\u256B', "BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE"),
    '╬': UC(0x256C, '\u256C', "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL"),

    # Character cell arcs
    '╭': UC(0x256D, '\u256D', "BOX DRAWINGS LIGHT ARC DOWN AND RIGHT"),
    '╮': UC(0x256E, '\u256E', "BOX DRAWINGS LIGHT ARC DOWN AND LEFT"),
    '╯': UC(0x256F, '\u256F', "BOX DRAWINGS LIGHT ARC UP AND LEFT"),
    '╰': UC(0x2570, '\u2570', "BOX DRAWINGS LIGHT ARC UP AND RIGHT"),

    # Character cell diagonals
    # For a more extensive set of legacy terminal graphic character cell diagonals,
    # see also 1FBA0-1FBAE in the Symbols for Legacy Computing block.
    '╱': UC(0x2571, '\u2571', "BOX DRAWINGS LIGHT DIAGONAL UPPER RIGHT TO LOWER LEFT"),
    '╲': UC(0x2572, '\u2572', "BOX DRAWINGS LIGHT DIAGONAL UPPER LEFT TO LOWER RIGHT"),
    '╳': UC(0x2573, '\u2573', "BOX DRAWINGS LIGHT DIAGONAL CROSS"),

    # Light and heavy half lines
    '╴': UC(0x2574, '\u2574', "BOX DRAWINGS LIGHT LEFT"),
    '╵': UC(0x2575, '\u2575', "BOX DRAWINGS LIGHT UP"),
    '╶': UC(0x2576, '\u2576', "BOX DRAWINGS LIGHT RIGHT"),
    '╷': UC(0x2577, '\u2577', "BOX DRAWINGS LIGHT DOWN"),
    '╸': UC(0x2578, '\u2578', "BOX DRAWINGS HEAVY LEFT"),
    '╹': UC(0x2579, '\u2579', "BOX DRAWINGS HEAVY UP"),
    '╺': UC(0x257A, '\u257A', "BOX DRAWINGS HEAVY RIGHT"),
    '╻': UC(0x257B, '\u257B', "BOX DRAWINGS HEAVY DOWN"),

    # Mixed light and heavy lines
    '╼': UC(0x257C, '\u257C', "BOX DRAWINGS LIGHT LEFT AND HEAVY RIGHT"),
    '╽': UC(0x257D, '\u257D', "BOX DRAWINGS LIGHT UP AND HEAVY DOWN"),
    '╾': UC(0x257E, '\u257E', "BOX DRAWINGS HEAVY LEFT AND LIGHT RIGHT"),
    '╿': UC(0x257F, '\u257F', "BOX DRAWINGS HEAVY UP AND LIGHT DOWN"),
}
