/** @file exam_Gt.cpp
 *
 *  Tests for methods of elliptic polylogs. */

/*
 *  GiNaC Copyright (C) 1999-2026 Johannes Gutenberg University Mainz, Germany
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <iostream>
#include <chrono>
#include "ginac.h"
#include "matrix.h"
#include "parser/parse_context.h"
#include "Gt.h"


using namespace GiNaC;
using namespace std;

// To speed up compilation, large expressions are encoded as strings and parsed at runtime (which is much faster then compiling them)
#define DEFINE_SYMBOL(s) const symbol s{#s};
#define DEFINE_SYMBOLS LIST_SYMBOLS(DEFINE_SYMBOL)
#define ADD_PARSER_SYMBOL(s) symbols[#s] = s;
#define MAKE_PARSER \
	symtab symbols; \
	LIST_SYMBOLS(ADD_PARSER_SYMBOL) \
	prototype_table table = GiNaC::get_default_reader(); \
	table.emplace(prototype{"Gt", 3}, [](const exvector& args) -> ex { return Gt(ex_to<lst>(args[0]), args[1], args[2]); }); \
	parser parse(symbols, false, table);


static unsigned exam_zisToFundamental()
{
	unsigned failed = 0;

#define LIST_SYMBOLS(f) f(tau) f(tau2) f(z) f(z1) f(z2) f(z3) f(z4)

	DEFINE_SYMBOLS
	MAKE_PARSER

#undef LIST_SYMBOLS

	const ex points = lst{
		tau==I*(1),
		tau2==numeric(1,3) + I*numeric(1,6),
		z1==0. + I*(1.1),
		z2==numeric(1,5),
		z3==tau,
		z4==numeric(4,3) + numeric(5,6)*tau,
		z==123+456*I // does not matter
	};

	auto test = [&](const size_t id, const Gt&& input, const char* expected_result_raw) {
		const ex output{input.zisToFundamental(&points)};
		const ex expected_result = parse(expected_result_raw);
		const ex diff = (expected_result-output).expand();

		if (diff.is_zero())
			std::cout << "  - Test #" << id << " succeeded" << std::endl;
		else {
			std::cerr << "Test #" << id << " failed" << std::endl;
			std::cerr << "  Input:           "; input.dbgprint();
			std::cerr << "  Expected result: "; expected_result.expand().dbgprint();
			std::cerr << "  Actual result:   "; output.expand().dbgprint();
			std::cerr << "  Difference:      "; diff.dbgprint();
			failed++;
		}
	};

	test(1,Gt(lst{lst{1,z1}},z,tau),
		"(I*(2))*(Pi)*(Gt({{0,0}},z,tau))+Gt({{1,(-1)*(tau)+z1}},z,tau)");
	test(2,Gt(lst{lst{4,z1}},z,tau),
		"((2)*pow(3,-1))*(pow(Pi,4))*(Gt({{0,0}},z,tau))+(I*((-4)*pow(3,-1)))*(pow(Pi,3))*(Gt({{1,(-1)*(tau)+z1}},z,tau))+(-2)*(pow(Pi,2))*(Gt({{2,(-1)*(tau)+z1}},z,tau))+(I*(2))*(Pi)*(Gt({{3,(-1)*(tau)+z1}},z,tau))+Gt({{4,(-1)*(tau)+z1}},z,tau)");
	test(3,Gt(lst{lst{2,z1},lst{1,z4},lst{1,z3}},z,tau),
		"(8)*(pow(Pi,4))*(Gt({{0,0},{0,0},{0,0}},z,tau))+(I*(-4))*(pow(Pi,3))*(Gt({{0,0},{0,0},{1,0}},z,tau))+(I*(-4))*(pow(Pi,3))*(Gt({{0,0},{1,-1+(-1)*(tau)+z4},{0,0}},z,tau))+(-2)*(pow(Pi,2))*(Gt({{0,0},{1,-1+(-1)*(tau)+z4},{1,0}},z,tau))+(I*(-8))*(pow(Pi,3))*(Gt({{1,(-1)*(tau)+z1},{0,0},{0,0}},z,tau))+(-4)*(pow(Pi,2))*(Gt({{1,(-1)*(tau)+z1},{0,0},{1,0}},z,tau))+(-4)*(pow(Pi,2))*(Gt({{1,(-1)*(tau)+z1},{1,-1+(-1)*(tau)+z4},{0,0}},z,tau))+(I*(2))*(Pi)*(Gt({{1,(-1)*(tau)+z1},{1,-1+(-1)*(tau)+z4},{1,0}},z,tau))+(-4)*(pow(Pi,2))*(Gt({{2,(-1)*(tau)+z1},{0,0},{0,0}},z,tau))+(I*(2))*(Pi)*(Gt({{2,(-1)*(tau)+z1},{0,0},{1,0}},z,tau))+(I*(2))*(Pi)*(Gt({{2,(-1)*(tau)+z1},{1,-1+(-1)*(tau)+z4},{0,0}},z,tau))+Gt({{2,(-1)*(tau)+z1},{1,-1+(-1)*(tau)+z4},{1,0}},z,tau)");
	test(4,Gt(lst{lst{4,z2},lst{1,z3},lst{2,z4}},z,tau),
		"(I*(-4))*(pow(Pi,3))*(Gt({{4,z2},{0,0},{0,0}},z,tau))+(-4)*(pow(Pi,2))*(Gt({{4,z2},{0,0},{1,-1+(-1)*(tau)+z4}},z,tau))+(I*(2))*(Pi)*(Gt({{4,z2},{0,0},{2,-1+(-1)*(tau)+z4}},z,tau))+(-2)*(pow(Pi,2))*(Gt({{4,z2},{1,0},{0,0}},z,tau))+(I*(2))*(Pi)*(Gt({{4,z2},{1,0},{1,-1+(-1)*(tau)+z4}},z,tau))+Gt({{4,z2},{1,0},{2,-1+(-1)*(tau)+z4}},z,tau)");
	test(5,Gt(lst{lst{2,z1},lst{1,z2},lst{2,z4}},z,tau),
		"(4)*(pow(Pi,4))*(Gt({{0,0},{1,z2},{0,0}},z,tau))+(I*(-4))*(pow(Pi,3))*(Gt({{0,0},{1,z2},{1,-1+(-1)*(tau)+z4}},z,tau))+(-2)*(pow(Pi,2))*(Gt({{0,0},{1,z2},{2,-1+(-1)*(tau)+z4}},z,tau))+(I*(-4))*(pow(Pi,3))*(Gt({{1,(-1)*(tau)+z1},{1,z2},{0,0}},z,tau))+(-4)*(pow(Pi,2))*(Gt({{1,(-1)*(tau)+z1},{1,z2},{1,-1+(-1)*(tau)+z4}},z,tau))+(I*(2))*(Pi)*(Gt({{1,(-1)*(tau)+z1},{1,z2},{2,-1+(-1)*(tau)+z4}},z,tau))+(-2)*(pow(Pi,2))*(Gt({{2,(-1)*(tau)+z1},{1,z2},{0,0}},z,tau))+(I*(2))*(Pi)*(Gt({{2,(-1)*(tau)+z1},{1,z2},{1,-1+(-1)*(tau)+z4}},z,tau))+Gt({{2,(-1)*(tau)+z1},{1,z2},{2,-1+(-1)*(tau)+z4}},z,tau)");
	test(6,Gt(lst{lst{1,z1},lst{2,numeric(43,10)+I*(-numeric(14,10))}},z,tau),
		"(I*(-4))*(pow(Pi,3))*(Gt({{0,0},{0,0}},z,tau))+(4)*(pow(Pi,2))*(Gt({{0,0},{1,3/10+I*(-14/10)+tau}},z,tau))+(I*(2))*(Pi)*(Gt({{0,0},{2,3/10+I*(-14/10)+tau}},z,tau))+(-2)*(pow(Pi,2))*(Gt({{1,(-1)*(tau)+z1},{0,0}},z,tau))+(I*(-2))*(Pi)*(Gt({{1,(-1)*(tau)+z1},{1,3/10+I*(-14/10)+tau}},z,tau))+Gt({{1,(-1)*(tau)+z1},{2,3/10+I*(-14/10)+tau}},z,tau)");
	test(7,Gt(lst{lst{1,z1}},z,tau2),
		"(I*(14))*(Pi)*(Gt({{0,0}},z,tau2))+Gt({{1,2+(-7)*(tau2)+z1}},z,tau2)");
	test(8,Gt(lst{lst{4,z1}},z,tau2),
		"((4802)*pow(3,-1))*(pow(Pi,4))*(Gt({{0,0}},z,tau2))+(I*((-1372)*pow(3,-1)))*(pow(Pi,3))*(Gt({{1,2+(-7)*(tau2)+z1}},z,tau2))+(-98)*(pow(Pi,2))*(Gt({{2,2+(-7)*(tau2)+z1}},z,tau2))+(I*(14))*(Pi)*(Gt({{3,2+(-7)*(tau2)+z1}},z,tau2))+Gt({{4,2+(-7)*(tau2)+z1}},z,tau2)");
	test(9,Gt(lst{lst{2,z1},lst{1,z4},lst{1,z3}},z,tau2),
		"(11760)*(pow(Pi,4))*(Gt({{0,0},{0,0},{0,0}},z,tau2))+(I*(-980))*(pow(Pi,3))*(Gt({{0,0},{0,0},{1,0}},z,tau2))+(I*(-1176))*(pow(Pi,3))*(Gt({{0,0},{1,(-5)*(tau2)+z4},{0,0}},z,tau2))+(-98)*(pow(Pi,2))*(Gt({{0,0},{1,(-5)*(tau2)+z4},{1,0}},z,tau2))+(I*(-1680))*(pow(Pi,3))*(Gt({{1,2+(-7)*(tau2)+z1},{0,0},{0,0}},z,tau2))+(-140)*(pow(Pi,2))*(Gt({{1,2+(-7)*(tau2)+z1},{0,0},{1,0}},z,tau2))+(-168)*(pow(Pi,2))*(Gt({{1,2+(-7)*(tau2)+z1},{1,(-5)*(tau2)+z4},{0,0}},z,tau2))+(I*(14))*(Pi)*(Gt({{1,2+(-7)*(tau2)+z1},{1,(-5)*(tau2)+z4},{1,0}},z,tau2))+(-120)*(pow(Pi,2))*(Gt({{2,2+(-7)*(tau2)+z1},{0,0},{0,0}},z,tau2))+(I*(10))*(Pi)*(Gt({{2,2+(-7)*(tau2)+z1},{0,0},{1,0}},z,tau2))+(I*(12))*(Pi)*(Gt({{2,2+(-7)*(tau2)+z1},{1,(-5)*(tau2)+z4},{0,0}},z,tau2))+Gt({{2,2+(-7)*(tau2)+z1},{1,(-5)*(tau2)+z4},{1,0}},z,tau2)");
	test(10,Gt(lst{lst{4,z2},lst{1,z3},lst{2,z4}},z,tau2),
		"(I*(-600))*(pow(Pi,3))*(Gt({{4,z2},{0,0},{0,0}},z,tau2))+(-120)*(pow(Pi,2))*(Gt({{4,z2},{0,0},{1,(-5)*(tau2)+z4}},z,tau2))+(I*(12))*(Pi)*(Gt({{4,z2},{0,0},{2,(-5)*(tau2)+z4}},z,tau2))+(-50)*(pow(Pi,2))*(Gt({{4,z2},{1,0},{0,0}},z,tau2))+(I*(10))*(Pi)*(Gt({{4,z2},{1,0},{1,(-5)*(tau2)+z4}},z,tau2))+Gt({{4,z2},{1,0},{2,(-5)*(tau2)+z4}},z,tau2)");
	test(11,Gt(lst{lst{2,z1},lst{1,z2},lst{2,z4}},z,tau2),
		"(4900)*(pow(Pi,4))*(Gt({{0,0},{1,z2},{0,0}},z,tau2))+(I*(-980))*(pow(Pi,3))*(Gt({{0,0},{1,z2},{1,(-5)*(tau2)+z4}},z,tau2))+(-98)*(pow(Pi,2))*(Gt({{0,0},{1,z2},{2,(-5)*(tau2)+z4}},z,tau2))+(I*(-700))*(pow(Pi,3))*(Gt({{1,2+(-7)*(tau2)+z1},{1,z2},{0,0}},z,tau2))+(-140)*(pow(Pi,2))*(Gt({{1,2+(-7)*(tau2)+z1},{1,z2},{1,(-5)*(tau2)+z4}},z,tau2))+(I*(14))*(Pi)*(Gt({{1,2+(-7)*(tau2)+z1},{1,z2},{2,(-5)*(tau2)+z4}},z,tau2))+(-50)*(pow(Pi,2))*(Gt({{2,2+(-7)*(tau2)+z1},{1,z2},{0,0}},z,tau2))+(I*(10))*(Pi)*(Gt({{2,2+(-7)*(tau2)+z1},{1,z2},{1,(-5)*(tau2)+z4}},z,tau2))+Gt({{2,2+(-7)*(tau2)+z1},{1,z2},{2,(-5)*(tau2)+z4}},z,tau2)");
	test(12,Gt(lst{lst{1,z1},lst{2,numeric(43,10)+I*(-numeric(14,10))}},z,tau2),
		"(I*(-1792))*(pow(Pi,3))*(Gt({{0,0},{0,0}},z,tau2))+(224)*(pow(Pi,2))*(Gt({{0,0},{1,-27/10+I*(-14/10)+(8)*(tau2)}},z,tau2))+(I*(14))*(Pi)*(Gt({{0,0},{2,-27/10+I*(-14/10)+(8)*(tau2)}},z,tau2))+(-128)*(pow(Pi,2))*(Gt({{1,2+(-7)*(tau2)+z1},{0,0}},z,tau2))+(I*(-16))*(Pi)*(Gt({{1,2+(-7)*(tau2)+z1},{1,-27/10+I*(-14/10)+(8)*(tau2)}},z,tau2))+Gt({{1,2+(-7)*(tau2)+z1},{2,-27/10+I*(-14/10)+(8)*(tau2)}},z,tau2)");

	return failed;
}

static unsigned exam_regularise()
{
	unsigned failed = 0;

#define LIST_SYMBOLS(f) f(z) f(tau) f(z1) f(z2)

	DEFINE_SYMBOLS
	MAKE_PARSER

#undef LIST_SYMBOLS

	const ex points = lst{z1 == numeric(1,11), z2 == numeric(2,13)}; // values do not matter here

	auto test = [&](const size_t id, const Gt&& input, const char* expected_result_raw) {
		const ex output{input.regularise(&points)};
		const ex expected_result = parse(expected_result_raw);
		const ex diff = (expected_result-output).expand();

		if (diff.is_zero())
			std::printf("  - Test #%lu succeeded\n", id);
		else {
			std::fprintf(stderr, "Test #%lu failed\n", id);
			std::fprintf(stderr, "  Input:           "); input.dbgprint();
			std::fprintf(stderr, "  Expected result: "); expected_result.expand().dbgprint();
			std::fprintf(stderr, "  Actual result:   "); output.expand().dbgprint();
			std::fprintf(stderr, "  Difference:      "); diff.dbgprint();
			failed++;
		}
	};

	test(1,Gt(lst{lst{1,0}},z,tau),
		"Gt({{1,0}},z,tau)");
	test(2,Gt(lst{lst{1,0},lst{1,0}},z,tau),
		"((1)*pow(2,-1))*(pow(Gt({{1,0}},z,tau),2))");
	test(3,Gt(lst{lst{2,0}},z,tau),
		"Gt({{2,0}},z,tau)");
	test(4,Gt(lst{lst{3,z1},lst{1,0}},z,tau),
		"(Gt({{1,0}},z,tau))*(Gt({{3,z1}},z,tau))+(-1)*(Gt({{1,0},{3,z1}},z,tau))");
	test(5,Gt(lst{lst{2,z1},lst{1,z2},lst{1,0}},z,tau),
		"(Gt({{1,0}},z,tau))*(Gt({{2,z1},{1,z2}},z,tau))+(-1)*(Gt({{1,0},{2,z1},{1,z2}},z,tau))+(-1)*(Gt({{2,z1},{1,0},{1,z2}},z,tau))");
	test(6,Gt(lst{lst{2,z1},lst{1,z2},lst{1,0}},z,tau),
		"(Gt({{1,0}},z,tau))*(Gt({{2,z1},{1,z2}},z,tau))+(-1)*(Gt({{1,0},{2,z1},{1,z2}},z,tau))+(-1)*(Gt({{2,z1},{1,0},{1,z2}},z,tau))");
	test(7,Gt(lst{lst{1,z1},lst{1,0},lst{1,0}},z,tau),
		"((1)*pow(2,-1))*(pow(Gt({{1,0}},z,tau),2))*(Gt({{1,z1}},z,tau))+(-1)*(Gt({{1,0}},z,tau))*(Gt({{1,0},{1,z1}},z,tau))+Gt({{1,0},{1,0},{1,z1}},z,tau)");
	test(8,Gt(lst{lst{1,0},lst{1,z1},lst{1,0},lst{1,0}},z,tau),
		"((1)*pow(2,-1))*(pow(Gt({{1,0}},z,tau),2))*(Gt({{1,0},{1,z1}},z,tau))+(-2)*(Gt({{1,0}},z,tau))*(Gt({{1,0},{1,0},{1,z1}},z,tau))+(3)*(Gt({{1,0},{1,0},{1,0},{1,z1}},z,tau))");

	return failed;
}

static unsigned exam_tauToFundemantal()
{
	unsigned failed = 0;

#define LIST_SYMBOLS(f) f(z) f(tau) f(z1) f(z2) f(z3)

	DEFINE_SYMBOLS
	MAKE_PARSER

#undef LIST_SYMBOLS

	const ex points = lst{
		tau == numeric(1,3) + 2*I,
		z1 == numeric(1,11),
		z2 == numeric(2,13),
		z3 == numeric(3,17)
	};

	auto test = [&](const size_t id, const Gt&& input, const matrix&& expected_transform, const char* expected_result_raw){
		const matrix transform = input.findMoebiusTransform(&points);
		if (transform != expected_transform) {
			std::fprintf(stderr, "Got unexpected transformation matrix:\n");
			std::fprintf(stderr, "  computed: ");
			transform.dbgprint();

			std::fprintf(stderr, "  expected: ");
			expected_transform.dbgprint();
		}

		const ex output{input.tauToFundamental(&points)};
		const ex expected_result = parse(expected_result_raw);
		const ex diff = (expected_result-output).expand().normal();

		if (diff.is_zero())
			std::printf("  - Test #%lu succeeded\n", id);
		else {
			std::fprintf(stderr, "Test #%lu failed\n", id);
			std::fprintf(stderr, "  Input:           "); input.dbgprint();
			std::fprintf(stderr, "  Expected result: "); expected_result.expand().dbgprint();
			std::fprintf(stderr, "  Actual result:   "); output.expand().dbgprint();
			std::fprintf(stderr, "  Difference:      "); diff.dbgprint();
			failed++;
		}
	};

	test(1,Gt(lst{lst{1,z1}},z,(-1)*(pow(tau,-1))),matrix{{0,-1},{1,0}},
		"(I*(1))*(Pi)*(tau)*(z)*(z+(-2)*(z1))+Gt({{1,(-1)*(tau)*(z1)}},(-1)*(tau)*(z),tau)");
	test(2,Gt(lst{lst{2,z1}},z,(-1)*(pow(tau,-1))),matrix{{0,-1},{1,0}},
		"((-2)*pow(3,-1))*(pow(Pi,2))*(pow(tau,2))*(z)*(pow(z,2)+(-3)*(z)*(z1)+(3)*(pow(z1,2)))+(I*(2))*(Pi)*(tau)*(z+(-1)*(z1))*(Gt({{1,(-1)*(tau)*(z1)}},(-1)*(tau)*(z),tau))+(-1)*(tau)*(Gt({{2,(-1)*(tau)*(z1)}},(-1)*(tau)*(z),tau))+(I*(2))*(Pi)*(Gt({{0,0},{1,(-1)*(tau)*(z1)}},(-1)*(tau)*(z),tau))");
	test(3,Gt(lst{lst{1,z1},lst{2,z2}},z,(-1)*(pow(tau,-1))),matrix{{0,-1},{1,0}},
		"(I*((-1)*pow(15,-1)))*(pow(Pi,3))*(pow(tau,3))*(pow(z,2))*((4)*(pow(z,3))+(-30)*(z1)*(pow(z2,2))+(20)*(z)*(z2)*(z1+z2)+(-5)*(pow(z,2))*(z1+(3)*(z2)))+((-2)*pow(3,-1))*(pow(Pi,2))*(pow(tau,2))*(z)*(pow(z,2)+(-3)*(z)*(z2)+(3)*(pow(z2,2)))*(Gt({{1,(-1)*(tau)*(z1)}},(-1)*(tau)*(z),tau))+(-2)*(pow(Pi,2))*(tau)*(pow(z+(-1)*(z2),2))*(Gt({{0,0},{1,(-1)*(tau)*(z1)}},(-1)*(tau)*(z),tau))+(4)*(pow(Pi,2))*(tau)*(z+(-1)*(z1))*(z+(-1)*(z2))*(Gt({{0,0},{1,(-1)*(tau)*(z2)}},(-1)*(tau)*(z),tau))+(I*(2))*(Pi)*(tau)*(z+(-1)*(z1))*(Gt({{0,0},{2,(-1)*(tau)*(z2)}},(-1)*(tau)*(z),tau))+(I*(2))*(Pi)*(tau)*(z+(-1)*(z2))*(Gt({{1,(-1)*(tau)*(z1)},{1,(-1)*(tau)*(z2)}},(-1)*(tau)*(z),tau))+(-1)*(tau)*(Gt({{1,(-1)*(tau)*(z1)},{2,(-1)*(tau)*(z2)}},(-1)*(tau)*(z),tau))+(4)*(pow(Pi,2))*((-1)*(z)+z2)*(Gt({{0,0},{0,0},{1,(-1)*(tau)*(z1)}},(-1)*(tau)*(z),tau))+(4)*(pow(Pi,2))*((3)*(z)+(-2)*(z1)+(-1)*(z2))*(Gt({{0,0},{0,0},{1,(-1)*(tau)*(z2)}},(-1)*(tau)*(z),tau))+(I*(2))*(Pi)*(Gt({{0,0},{0,0},{2,(-1)*(tau)*(z2)}},(-1)*(tau)*(z),tau))+(I*(2))*(Pi)*(Gt({{0,0},{1,(-1)*(tau)*(z1)},{1,(-1)*(tau)*(z2)}},(-1)*(tau)*(z),tau))+(I*(2))*(Pi)*(Gt({{1,(-1)*(tau)*(z1)},{0,0},{1,(-1)*(tau)*(z2)}},(-1)*(tau)*(z),tau))+(-4)*(pow(Pi,2))*(pow(tau,-1))*(Gt({{0,0},{0,0},{0,0},{1,(-1)*(tau)*(z1)}},(-1)*(tau)*(z),tau))+(12)*(pow(Pi,2))*(pow(tau,-1))*(Gt({{0,0},{0,0},{0,0},{1,(-1)*(tau)*(z2)}},(-1)*(tau)*(z),tau))");
	test(4,Gt(lst{lst{1,z1}},z,1+tau),matrix{{1,-1},{0,1}},
		"Gt({{1,z1}},z,tau)");
	test(5,Gt(lst{lst{2,z1},lst{1,z2}},z,1+tau),matrix{{1,-1},{0,1}},
		"Gt({{2,z1},{1,z2}},z,tau)");
	test(6,Gt(lst{lst{3,z1}},z,(pow(1+(-1)*(tau),-1))*(-5+(6)*(tau))),matrix{{1,5},{1,6}},
		"(I*((-1)*pow(3,-1)))*(pow(Pi,3))*(pow(-1+tau,3))*(z)*(z+(-2)*(z1))*(pow(z,2)+(-2)*(z)*(z1)+(2)*(pow(z1,2)))+(-2)*(pow(Pi,2))*(pow(-1+tau,2))*(pow(z+(-1)*(z1),2))*(Gt({{1,(1+(-1)*(tau))*(z1)}},(1+(-1)*(tau))*(z),tau))+(I*(-2))*(Pi)*(pow(-1+tau,2))*(z+(-1)*(z1))*(Gt({{2,(1+(-1)*(tau))*(z1)}},(1+(-1)*(tau))*(z),tau))+(pow(-1+tau,2))*(Gt({{3,(1+(-1)*(tau))*(z1)}},(1+(-1)*(tau))*(z),tau))+(-4)*(pow(Pi,2))*(-1+tau)*(z+(-1)*(z1))*(Gt({{0,0},{1,(1+(-1)*(tau))*(z1)}},(1+(-1)*(tau))*(z),tau))+(I*(-2))*(Pi)*(-1+tau)*(Gt({{0,0},{2,(1+(-1)*(tau))*(z1)}},(1+(-1)*(tau))*(z),tau))+(-4)*(pow(Pi,2))*(Gt({{0,0},{0,0},{1,(1+(-1)*(tau))*(z1)}},(1+(-1)*(tau))*(z),tau))");
	test(7,Gt(lst{lst{1,z1},lst{1,z2}},z,(pow(1+(-1)*(tau),-1))*(-5+(6)*(tau))),matrix{{1,5},{1,6}},
		"((-1)*pow(6,-1))*(pow(Pi,2))*(pow(-1+tau,2))*(pow(z,2))*((3)*(pow(z,2))+(12)*(z1)*(z2)+(-4)*(z)*(z1+(2)*(z2)))+(I*(1))*(Pi)*(-1+tau)*(z)*(z+(-2)*(z2))*(Gt({{1,(1+(-1)*(tau))*(z1)}},(1+(-1)*(tau))*(z),tau))+(I*(2))*(Pi)*(z+(-1)*(z2))*(Gt({{0,0},{1,(1+(-1)*(tau))*(z1)}},(1+(-1)*(tau))*(z),tau))+(I*(-2))*(Pi)*(z+(-1)*(z1))*(Gt({{0,0},{1,(1+(-1)*(tau))*(z2)}},(1+(-1)*(tau))*(z),tau))+Gt({{1,(1+(-1)*(tau))*(z1)},{1,(1+(-1)*(tau))*(z2)}},(1+(-1)*(tau))*(z),tau)+(I*(2))*(Pi)*(pow(-1+tau,-1))*(Gt({{0,0},{0,0},{1,(1+(-1)*(tau))*(z1)}},(1+(-1)*(tau))*(z),tau))+(I*(-2))*(Pi)*(pow(-1+tau,-1))*(Gt({{0,0},{0,0},{1,(1+(-1)*(tau))*(z2)}},(1+(-1)*(tau))*(z),tau))");
	test(8,Gt(lst{lst{1,z1},lst{1,z2}},z,(pow(7+(-4)*(tau),-1))*(2+(-1)*(tau))),matrix{{7,-2},{4,-1}},
		"((-8)*pow(3,-1))*(pow(Pi,2))*(pow(7+(-4)*(tau),2))*(pow(z,2))*((3)*(pow(z,2))+(12)*(z1)*(z2)+(-4)*(z)*(z1+(2)*(z2)))+(I*(4))*(Pi)*(-7+(4)*(tau))*(z)*(z+(-2)*(z2))*(Gt({{1,(7+(-4)*(tau))*(z1)}},(7+(-4)*(tau))*(z),tau))+(I*(8))*(Pi)*(z+(-1)*(z2))*(Gt({{0,0},{1,(7+(-4)*(tau))*(z1)}},(7+(-4)*(tau))*(z),tau))+(I*(-8))*(Pi)*(z+(-1)*(z1))*(Gt({{0,0},{1,(7+(-4)*(tau))*(z2)}},(7+(-4)*(tau))*(z),tau))+Gt({{1,(7+(-4)*(tau))*(z1)},{1,(7+(-4)*(tau))*(z2)}},(7+(-4)*(tau))*(z),tau)+(I*(8))*(Pi)*(pow(-7+(4)*(tau),-1))*(Gt({{0,0},{0,0},{1,(7+(-4)*(tau))*(z1)}},(7+(-4)*(tau))*(z),tau))+(I*(8))*(Pi)*(pow(7+(-4)*(tau),-1))*(Gt({{0,0},{0,0},{1,(7+(-4)*(tau))*(z2)}},(7+(-4)*(tau))*(z),tau))");
	test(9,Gt(lst{lst{2,z1}},z,(pow(7+(-4)*(tau),-1))*(2+(-1)*(tau))),matrix{{7,-2},{4,-1}},
		"((-32)*pow(3,-1))*(pow(Pi,2))*(pow(7+(-4)*(tau),2))*(z)*(pow(z,2)+(-3)*(z)*(z1)+(3)*(pow(z1,2)))+(I*(8))*(Pi)*(-7+(4)*(tau))*(z+(-1)*(z1))*(Gt({{1,(7+(-4)*(tau))*(z1)}},(7+(-4)*(tau))*(z),tau))+(7+(-4)*(tau))*(Gt({{2,(7+(-4)*(tau))*(z1)}},(7+(-4)*(tau))*(z),tau))+(I*(8))*(Pi)*(Gt({{0,0},{1,(7+(-4)*(tau))*(z1)}},(7+(-4)*(tau))*(z),tau))");
	test(10,Gt(lst{lst{2,z1},lst{1,z2},lst{3,z3}},z,tau),matrix{{1,0},{0,1}},
		"Gt({{2,z1},{1,z2},{3,z3}},z,tau)");

	return failed;
}

static unsigned exam_cutIntegrationPath()
{
	unsigned failed = 0;

#define LIST_SYMBOLS(f) f(z) f(z1) f(z2) f(z3) f(z4) f(z5) f(tau)

	DEFINE_SYMBOLS
	MAKE_PARSER

#undef LIST_SYMBOLS

	auto test = [&](const size_t id, const char* input_raw, const int nCuts, const char* expected_result_raw){
		// The cut happens at latest at 0.4*tau, set z to just under 0.4*tau*n to ensure that we get n cuts
		const ex points = lst{
			tau == I,
			z == numeric(1,3) + 4*I/10*nCuts*99/100,
			z1 == numeric(1,11),
			z2 == numeric(2,13),
			z3 == numeric(3,17),
			z4 == numeric(4,23),
			z5 == numeric(5,29)
		};

		const ex input = parse(input_raw);
		const ex expected_result = parse(expected_result_raw);

		const ex output{Gt::ex_cutIntegrationPath(input, &points)};
		const ex diff = (expected_result-output).expand().normal();

		if (diff.is_zero())
			std::printf("  - Test #%lu succeeded\n", id);
		else {
			std::fprintf(stderr, "Test #%lu failed\n", id);
			std::fprintf(stderr, "  Input:           "); input.dbgprint();
			std::fprintf(stderr, "  Expected result: "); expected_result.expand().dbgprint();
			std::fprintf(stderr, "  Actual result:   "); output.expand().dbgprint();
			std::fprintf(stderr, "  Difference:      "); diff.dbgprint();
			failed++;
		}
	};

	test(1, "Gt({{2,z1}},z,tau)", 1, "Gt({{2,z1}},z,tau)");
	test(2, "Gt({{2,z1}},z,tau)", 2, "Gt({{2,z1}},1/2*z,tau)+Gt({{2,-1/2*z+z1}},1/2*z,tau)");
	test(3, "Gt({{2,z1}},z,tau)", 3, "Gt({{2,z1}},1/3*z,tau)+Gt({{2,-2/3*z+z1}},1/3*z,tau)+Gt({{2,-1/3*z+z1}},1/3*z,tau)");
	test(4, "Gt({{2,z1}},z,tau)", 4, "Gt({{2,z1}},1/4*z,tau)+Gt({{2,-3/4*z+z1}},1/4*z,tau)+Gt({{2,-1/2*z+z1}},1/4*z,tau)+Gt({{2,-1/4*z+z1}},1/4*z,tau)");
	test(5, "Gt({{2,z1}},z,tau)", 5, "Gt({{2,z1}},1/5*z,tau)+Gt({{2,-4/5*z+z1}},1/5*z,tau)+Gt({{2,-3/5*z+z1}},1/5*z,tau)+Gt({{2,-2/5*z+z1}},1/5*z,tau)+Gt({{2,-1/5*z+z1}},1/5*z,tau)");
	test(6, "Gt({{2,z1}},z,tau)", 6, "Gt({{2,z1}},1/6*z,tau)+Gt({{2,-5/6*z+z1}},1/6*z,tau)+Gt({{2,-2/3*z+z1}},1/6*z,tau)+Gt({{2,-1/2*z+z1}},1/6*z,tau)+Gt({{2,-1/3*z+z1}},1/6*z,tau)+Gt({{2,-1/6*z+z1}},1/6*z,tau)");
	test(7, "Gt({{1,z1},{1,z2}},z,tau)", 1, "Gt({{1,z1},{1,z2}},z,tau)");
	test(8, "Gt({{1,z1},{1,z2}},z,tau)", 2, "(Gt({{1,-1/2*z+z1}},1/2*z,tau))*(Gt({{1,z2}},1/2*z,tau))+Gt({{1,z1},{1,z2}},1/2*z,tau)+Gt({{1,-1/2*z+z1},{1,-1/2*z+z2}},1/2*z,tau)");
	test(9, "Gt({{1,z1},{1,z2}},z,tau)", 3, "(Gt({{1,-2/3*z+z1}},1/3*z,tau))*(Gt({{1,z2}},1/3*z,tau))+(Gt({{1,-1/3*z+z1}},1/3*z,tau))*(Gt({{1,z2}},1/3*z,tau))+(Gt({{1,-2/3*z+z1}},1/3*z,tau))*(Gt({{1,-1/3*z+z2}},1/3*z,tau))+Gt({{1,z1},{1,z2}},1/3*z,tau)+Gt({{1,-2/3*z+z1},{1,-2/3*z+z2}},1/3*z,tau)+Gt({{1,-1/3*z+z1},{1,-1/3*z+z2}},1/3*z,tau)");
	test(10, "Gt({{1,z1},{1,z2}},z,tau)", 4, "(Gt({{1,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,z2}},1/4*z,tau))+(Gt({{1,-1/2*z+z1}},1/4*z,tau))*(Gt({{1,z2}},1/4*z,tau))+(Gt({{1,-1/4*z+z1}},1/4*z,tau))*(Gt({{1,z2}},1/4*z,tau))+(Gt({{1,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,-1/2*z+z2}},1/4*z,tau))+(Gt({{1,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,-1/4*z+z2}},1/4*z,tau))+(Gt({{1,-1/2*z+z1}},1/4*z,tau))*(Gt({{1,-1/4*z+z2}},1/4*z,tau))+Gt({{1,z1},{1,z2}},1/4*z,tau)+Gt({{1,-3/4*z+z1},{1,-3/4*z+z2}},1/4*z,tau)+Gt({{1,-1/2*z+z1},{1,-1/2*z+z2}},1/4*z,tau)+Gt({{1,-1/4*z+z1},{1,-1/4*z+z2}},1/4*z,tau)");
	test(11, "Gt({{1,z1},{1,z2}},z,tau)", 5, "(Gt({{1,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,z2}},1/5*z,tau))+(Gt({{1,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,z2}},1/5*z,tau))+(Gt({{1,-2/5*z+z1}},1/5*z,tau))*(Gt({{1,z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z1}},1/5*z,tau))*(Gt({{1,z2}},1/5*z,tau))+(Gt({{1,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-2/5*z+z2}},1/5*z,tau))+(Gt({{1,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,-2/5*z+z2}},1/5*z,tau))+(Gt({{1,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z2}},1/5*z,tau))+(Gt({{1,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z2}},1/5*z,tau))+(Gt({{1,-2/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z2}},1/5*z,tau))+Gt({{1,z1},{1,z2}},1/5*z,tau)+Gt({{1,-4/5*z+z1},{1,-4/5*z+z2}},1/5*z,tau)+Gt({{1,-3/5*z+z1},{1,-3/5*z+z2}},1/5*z,tau)+Gt({{1,-2/5*z+z1},{1,-2/5*z+z2}},1/5*z,tau)+Gt({{1,-1/5*z+z1},{1,-1/5*z+z2}},1/5*z,tau)");
	test(12, "Gt({{1,z1},{1,z2}},z,tau)", 6, "(Gt({{1,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z2}},1/6*z,tau))+(Gt({{1,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z1}},1/6*z,tau))*(Gt({{1,z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z1}},1/6*z,tau))*(Gt({{1,z2}},1/6*z,tau))+(Gt({{1,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{1,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{1,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{1,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))+Gt({{1,z1},{1,z2}},1/6*z,tau)+Gt({{1,-5/6*z+z1},{1,-5/6*z+z2}},1/6*z,tau)+Gt({{1,-2/3*z+z1},{1,-2/3*z+z2}},1/6*z,tau)+Gt({{1,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau)+Gt({{1,-1/3*z+z1},{1,-1/3*z+z2}},1/6*z,tau)+Gt({{1,-1/6*z+z1},{1,-1/6*z+z2}},1/6*z,tau)");
	test(13, "Gt({{2,z1},{1,z2},{2,z3}},z,tau)", 1, "Gt({{2,z1},{1,z2},{2,z3}},z,tau)");
	test(14, "Gt({{2,z1},{1,z2},{2,z3}},z,tau)", 2, "(Gt({{2,-1/2*z+z1}},1/2*z,tau))*(Gt({{1,z2},{2,z3}},1/2*z,tau))+(Gt({{2,z3}},1/2*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/2*z,tau))+Gt({{2,z1},{1,z2},{2,z3}},1/2*z,tau)+Gt({{2,-1/2*z+z1},{1,-1/2*z+z2},{2,-1/2*z+z3}},1/2*z,tau)");
	test(15, "Gt({{2,z1},{1,z2},{2,z3}},z,tau)", 3, "(Gt({{1,-1/3*z+z2}},1/3*z,tau))*(Gt({{2,-2/3*z+z1}},1/3*z,tau))*(Gt({{2,z3}},1/3*z,tau))+(Gt({{2,-2/3*z+z1}},1/3*z,tau))*(Gt({{1,z2},{2,z3}},1/3*z,tau))+(Gt({{2,-1/3*z+z1}},1/3*z,tau))*(Gt({{1,z2},{2,z3}},1/3*z,tau))+(Gt({{2,-2/3*z+z1}},1/3*z,tau))*(Gt({{1,-1/3*z+z2},{2,-1/3*z+z3}},1/3*z,tau))+(Gt({{2,z3}},1/3*z,tau))*(Gt({{2,-2/3*z+z1},{1,-2/3*z+z2}},1/3*z,tau))+(Gt({{2,-1/3*z+z3}},1/3*z,tau))*(Gt({{2,-2/3*z+z1},{1,-2/3*z+z2}},1/3*z,tau))+(Gt({{2,z3}},1/3*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2}},1/3*z,tau))+Gt({{2,z1},{1,z2},{2,z3}},1/3*z,tau)+Gt({{2,-2/3*z+z1},{1,-2/3*z+z2},{2,-2/3*z+z3}},1/3*z,tau)+Gt({{2,-1/3*z+z1},{1,-1/3*z+z2},{2,-1/3*z+z3}},1/3*z,tau)");
	test(16, "Gt({{2,z1},{1,z2},{2,z3}},z,tau)", 4, "(Gt({{1,-1/2*z+z2}},1/4*z,tau))*(Gt({{2,-3/4*z+z1}},1/4*z,tau))*(Gt({{2,z3}},1/4*z,tau))+(Gt({{1,-1/4*z+z2}},1/4*z,tau))*(Gt({{2,-3/4*z+z1}},1/4*z,tau))*(Gt({{2,z3}},1/4*z,tau))+(Gt({{1,-1/4*z+z2}},1/4*z,tau))*(Gt({{2,-1/2*z+z1}},1/4*z,tau))*(Gt({{2,z3}},1/4*z,tau))+(Gt({{1,-1/2*z+z2}},1/4*z,tau))*(Gt({{2,-3/4*z+z1}},1/4*z,tau))*(Gt({{2,-1/4*z+z3}},1/4*z,tau))+(Gt({{2,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,z2},{2,z3}},1/4*z,tau))+(Gt({{2,-1/2*z+z1}},1/4*z,tau))*(Gt({{1,z2},{2,z3}},1/4*z,tau))+(Gt({{2,-1/4*z+z1}},1/4*z,tau))*(Gt({{1,z2},{2,z3}},1/4*z,tau))+(Gt({{2,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,-1/2*z+z2},{2,-1/2*z+z3}},1/4*z,tau))+(Gt({{2,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,-1/4*z+z2},{2,-1/4*z+z3}},1/4*z,tau))+(Gt({{2,-1/2*z+z1}},1/4*z,tau))*(Gt({{1,-1/4*z+z2},{2,-1/4*z+z3}},1/4*z,tau))+(Gt({{2,z3}},1/4*z,tau))*(Gt({{2,-3/4*z+z1},{1,-3/4*z+z2}},1/4*z,tau))+(Gt({{2,-1/2*z+z3}},1/4*z,tau))*(Gt({{2,-3/4*z+z1},{1,-3/4*z+z2}},1/4*z,tau))+(Gt({{2,-1/4*z+z3}},1/4*z,tau))*(Gt({{2,-3/4*z+z1},{1,-3/4*z+z2}},1/4*z,tau))+(Gt({{2,z3}},1/4*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/4*z,tau))+(Gt({{2,-1/4*z+z3}},1/4*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/4*z,tau))+(Gt({{2,z3}},1/4*z,tau))*(Gt({{2,-1/4*z+z1},{1,-1/4*z+z2}},1/4*z,tau))+Gt({{2,z1},{1,z2},{2,z3}},1/4*z,tau)+Gt({{2,-3/4*z+z1},{1,-3/4*z+z2},{2,-3/4*z+z3}},1/4*z,tau)+Gt({{2,-1/2*z+z1},{1,-1/2*z+z2},{2,-1/2*z+z3}},1/4*z,tau)+Gt({{2,-1/4*z+z1},{1,-1/4*z+z2},{2,-1/4*z+z3}},1/4*z,tau)");
	test(17, "Gt({{2,z1},{1,z2},{2,z3}},z,tau)", 5, "(Gt({{1,-3/5*z+z2}},1/5*z,tau))*(Gt({{2,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,z3}},1/5*z,tau))+(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{2,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,z3}},1/5*z,tau))+(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{2,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,z3}},1/5*z,tau))+(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{2,-3/5*z+z1}},1/5*z,tau))*(Gt({{2,z3}},1/5*z,tau))+(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{2,-3/5*z+z1}},1/5*z,tau))*(Gt({{2,z3}},1/5*z,tau))+(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{2,-2/5*z+z1}},1/5*z,tau))*(Gt({{2,z3}},1/5*z,tau))+(Gt({{1,-3/5*z+z2}},1/5*z,tau))*(Gt({{2,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-2/5*z+z3}},1/5*z,tau))+(Gt({{1,-3/5*z+z2}},1/5*z,tau))*(Gt({{2,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-1/5*z+z3}},1/5*z,tau))+(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{2,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-1/5*z+z3}},1/5*z,tau))+(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{2,-3/5*z+z1}},1/5*z,tau))*(Gt({{2,-1/5*z+z3}},1/5*z,tau))+(Gt({{2,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,z2},{2,z3}},1/5*z,tau))+(Gt({{2,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,z2},{2,z3}},1/5*z,tau))+(Gt({{2,-2/5*z+z1}},1/5*z,tau))*(Gt({{1,z2},{2,z3}},1/5*z,tau))+(Gt({{2,-1/5*z+z1}},1/5*z,tau))*(Gt({{1,z2},{2,z3}},1/5*z,tau))+(Gt({{2,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-3/5*z+z2},{2,-3/5*z+z3}},1/5*z,tau))+(Gt({{2,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-2/5*z+z2},{2,-2/5*z+z3}},1/5*z,tau))+(Gt({{2,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,-2/5*z+z2},{2,-2/5*z+z3}},1/5*z,tau))+(Gt({{2,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z2},{2,-1/5*z+z3}},1/5*z,tau))+(Gt({{2,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z2},{2,-1/5*z+z3}},1/5*z,tau))+(Gt({{2,-2/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z2},{2,-1/5*z+z3}},1/5*z,tau))+(Gt({{2,z3}},1/5*z,tau))*(Gt({{2,-4/5*z+z1},{1,-4/5*z+z2}},1/5*z,tau))+(Gt({{2,-3/5*z+z3}},1/5*z,tau))*(Gt({{2,-4/5*z+z1},{1,-4/5*z+z2}},1/5*z,tau))+(Gt({{2,-2/5*z+z3}},1/5*z,tau))*(Gt({{2,-4/5*z+z1},{1,-4/5*z+z2}},1/5*z,tau))+(Gt({{2,-1/5*z+z3}},1/5*z,tau))*(Gt({{2,-4/5*z+z1},{1,-4/5*z+z2}},1/5*z,tau))+(Gt({{2,z3}},1/5*z,tau))*(Gt({{2,-3/5*z+z1},{1,-3/5*z+z2}},1/5*z,tau))+(Gt({{2,-2/5*z+z3}},1/5*z,tau))*(Gt({{2,-3/5*z+z1},{1,-3/5*z+z2}},1/5*z,tau))+(Gt({{2,-1/5*z+z3}},1/5*z,tau))*(Gt({{2,-3/5*z+z1},{1,-3/5*z+z2}},1/5*z,tau))+(Gt({{2,z3}},1/5*z,tau))*(Gt({{2,-2/5*z+z1},{1,-2/5*z+z2}},1/5*z,tau))+(Gt({{2,-1/5*z+z3}},1/5*z,tau))*(Gt({{2,-2/5*z+z1},{1,-2/5*z+z2}},1/5*z,tau))+(Gt({{2,z3}},1/5*z,tau))*(Gt({{2,-1/5*z+z1},{1,-1/5*z+z2}},1/5*z,tau))+Gt({{2,z1},{1,z2},{2,z3}},1/5*z,tau)+Gt({{2,-4/5*z+z1},{1,-4/5*z+z2},{2,-4/5*z+z3}},1/5*z,tau)+Gt({{2,-3/5*z+z1},{1,-3/5*z+z2},{2,-3/5*z+z3}},1/5*z,tau)+Gt({{2,-2/5*z+z1},{1,-2/5*z+z2},{2,-2/5*z+z3}},1/5*z,tau)+Gt({{2,-1/5*z+z1},{1,-1/5*z+z2},{2,-1/5*z+z3}},1/5*z,tau)");
	test(18, "Gt({{2,z1},{1,z2},{2,z3}},z,tau)", 6, "(Gt({{1,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{2,z3}},1/6*z,tau))+(Gt({{1,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z3}},1/6*z,tau))+(Gt({{1,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z3}},1/6*z,tau))+(Gt({{1,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z3}},1/6*z,tau))+(Gt({{2,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z2},{2,z3}},1/6*z,tau))+(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,z2},{2,z3}},1/6*z,tau))+(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,z2},{2,z3}},1/6*z,tau))+(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{1,z2},{2,z3}},1/6*z,tau))+(Gt({{2,-1/6*z+z1}},1/6*z,tau))*(Gt({{1,z2},{2,z3}},1/6*z,tau))+(Gt({{2,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-2/3*z+z2},{2,-2/3*z+z3}},1/6*z,tau))+(Gt({{2,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/2*z+z2},{2,-1/2*z+z3}},1/6*z,tau))+(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/2*z+z2},{2,-1/2*z+z3}},1/6*z,tau))+(Gt({{2,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z2},{2,-1/3*z+z3}},1/6*z,tau))+(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z2},{2,-1/3*z+z3}},1/6*z,tau))+(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z2},{2,-1/3*z+z3}},1/6*z,tau))+(Gt({{2,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z2},{2,-1/6*z+z3}},1/6*z,tau))+(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z2},{2,-1/6*z+z3}},1/6*z,tau))+(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z2},{2,-1/6*z+z3}},1/6*z,tau))+(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z2},{2,-1/6*z+z3}},1/6*z,tau))+(Gt({{2,z3}},1/6*z,tau))*(Gt({{2,-5/6*z+z1},{1,-5/6*z+z2}},1/6*z,tau))+(Gt({{2,-2/3*z+z3}},1/6*z,tau))*(Gt({{2,-5/6*z+z1},{1,-5/6*z+z2}},1/6*z,tau))+(Gt({{2,-1/2*z+z3}},1/6*z,tau))*(Gt({{2,-5/6*z+z1},{1,-5/6*z+z2}},1/6*z,tau))+(Gt({{2,-1/3*z+z3}},1/6*z,tau))*(Gt({{2,-5/6*z+z1},{1,-5/6*z+z2}},1/6*z,tau))+(Gt({{2,-1/6*z+z3}},1/6*z,tau))*(Gt({{2,-5/6*z+z1},{1,-5/6*z+z2}},1/6*z,tau))+(Gt({{2,z3}},1/6*z,tau))*(Gt({{2,-2/3*z+z1},{1,-2/3*z+z2}},1/6*z,tau))+(Gt({{2,-1/2*z+z3}},1/6*z,tau))*(Gt({{2,-2/3*z+z1},{1,-2/3*z+z2}},1/6*z,tau))+(Gt({{2,-1/3*z+z3}},1/6*z,tau))*(Gt({{2,-2/3*z+z1},{1,-2/3*z+z2}},1/6*z,tau))+(Gt({{2,-1/6*z+z3}},1/6*z,tau))*(Gt({{2,-2/3*z+z1},{1,-2/3*z+z2}},1/6*z,tau))+(Gt({{2,z3}},1/6*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{2,-1/3*z+z3}},1/6*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{2,-1/6*z+z3}},1/6*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{2,z3}},1/6*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{2,-1/6*z+z3}},1/6*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{2,z3}},1/6*z,tau))*(Gt({{2,-1/6*z+z1},{1,-1/6*z+z2}},1/6*z,tau))+Gt({{2,z1},{1,z2},{2,z3}},1/6*z,tau)+Gt({{2,-5/6*z+z1},{1,-5/6*z+z2},{2,-5/6*z+z3}},1/6*z,tau)+Gt({{2,-2/3*z+z1},{1,-2/3*z+z2},{2,-2/3*z+z3}},1/6*z,tau)+Gt({{2,-1/2*z+z1},{1,-1/2*z+z2},{2,-1/2*z+z3}},1/6*z,tau)+Gt({{2,-1/3*z+z1},{1,-1/3*z+z2},{2,-1/3*z+z3}},1/6*z,tau)+Gt({{2,-1/6*z+z1},{1,-1/6*z+z2},{2,-1/6*z+z3}},1/6*z,tau)");
	test(19, "Gt({{4,z1},{1,z1},{2,z2}},z,tau)", 1, "Gt({{4,z1},{1,z1},{2,z2}},z,tau)");
	test(20, "Gt({{4,z1},{1,z1},{2,z2}},z,tau)", 2, "(Gt({{4,-1/2*z+z1}},1/2*z,tau))*(Gt({{1,z1},{2,z2}},1/2*z,tau))+(Gt({{2,z2}},1/2*z,tau))*(Gt({{4,-1/2*z+z1},{1,-1/2*z+z1}},1/2*z,tau))+Gt({{4,z1},{1,z1},{2,z2}},1/2*z,tau)+Gt({{4,-1/2*z+z1},{1,-1/2*z+z1},{2,-1/2*z+z2}},1/2*z,tau)");
	test(21, "Gt({{4,z1},{1,z1},{2,z2}},z,tau)", 3, "(Gt({{1,-1/3*z+z1}},1/3*z,tau))*(Gt({{2,z2}},1/3*z,tau))*(Gt({{4,-2/3*z+z1}},1/3*z,tau))+(Gt({{4,-2/3*z+z1}},1/3*z,tau))*(Gt({{1,z1},{2,z2}},1/3*z,tau))+(Gt({{4,-1/3*z+z1}},1/3*z,tau))*(Gt({{1,z1},{2,z2}},1/3*z,tau))+(Gt({{4,-2/3*z+z1}},1/3*z,tau))*(Gt({{1,-1/3*z+z1},{2,-1/3*z+z2}},1/3*z,tau))+(Gt({{2,z2}},1/3*z,tau))*(Gt({{4,-2/3*z+z1},{1,-2/3*z+z1}},1/3*z,tau))+(Gt({{2,-1/3*z+z2}},1/3*z,tau))*(Gt({{4,-2/3*z+z1},{1,-2/3*z+z1}},1/3*z,tau))+(Gt({{2,z2}},1/3*z,tau))*(Gt({{4,-1/3*z+z1},{1,-1/3*z+z1}},1/3*z,tau))+Gt({{4,z1},{1,z1},{2,z2}},1/3*z,tau)+Gt({{4,-2/3*z+z1},{1,-2/3*z+z1},{2,-2/3*z+z2}},1/3*z,tau)+Gt({{4,-1/3*z+z1},{1,-1/3*z+z1},{2,-1/3*z+z2}},1/3*z,tau)");
	test(22, "Gt({{4,z1},{1,z1},{2,z2}},z,tau)", 4, "(Gt({{1,-1/2*z+z1}},1/4*z,tau))*(Gt({{2,z2}},1/4*z,tau))*(Gt({{4,-3/4*z+z1}},1/4*z,tau))+(Gt({{1,-1/4*z+z1}},1/4*z,tau))*(Gt({{2,z2}},1/4*z,tau))*(Gt({{4,-3/4*z+z1}},1/4*z,tau))+(Gt({{1,-1/2*z+z1}},1/4*z,tau))*(Gt({{2,-1/4*z+z2}},1/4*z,tau))*(Gt({{4,-3/4*z+z1}},1/4*z,tau))+(Gt({{1,-1/4*z+z1}},1/4*z,tau))*(Gt({{2,z2}},1/4*z,tau))*(Gt({{4,-1/2*z+z1}},1/4*z,tau))+(Gt({{4,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,z1},{2,z2}},1/4*z,tau))+(Gt({{4,-1/2*z+z1}},1/4*z,tau))*(Gt({{1,z1},{2,z2}},1/4*z,tau))+(Gt({{4,-1/4*z+z1}},1/4*z,tau))*(Gt({{1,z1},{2,z2}},1/4*z,tau))+(Gt({{4,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,-1/2*z+z1},{2,-1/2*z+z2}},1/4*z,tau))+(Gt({{4,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,-1/4*z+z1},{2,-1/4*z+z2}},1/4*z,tau))+(Gt({{4,-1/2*z+z1}},1/4*z,tau))*(Gt({{1,-1/4*z+z1},{2,-1/4*z+z2}},1/4*z,tau))+(Gt({{2,z2}},1/4*z,tau))*(Gt({{4,-3/4*z+z1},{1,-3/4*z+z1}},1/4*z,tau))+(Gt({{2,-1/2*z+z2}},1/4*z,tau))*(Gt({{4,-3/4*z+z1},{1,-3/4*z+z1}},1/4*z,tau))+(Gt({{2,-1/4*z+z2}},1/4*z,tau))*(Gt({{4,-3/4*z+z1},{1,-3/4*z+z1}},1/4*z,tau))+(Gt({{2,z2}},1/4*z,tau))*(Gt({{4,-1/2*z+z1},{1,-1/2*z+z1}},1/4*z,tau))+(Gt({{2,-1/4*z+z2}},1/4*z,tau))*(Gt({{4,-1/2*z+z1},{1,-1/2*z+z1}},1/4*z,tau))+(Gt({{2,z2}},1/4*z,tau))*(Gt({{4,-1/4*z+z1},{1,-1/4*z+z1}},1/4*z,tau))+Gt({{4,z1},{1,z1},{2,z2}},1/4*z,tau)+Gt({{4,-3/4*z+z1},{1,-3/4*z+z1},{2,-3/4*z+z2}},1/4*z,tau)+Gt({{4,-1/2*z+z1},{1,-1/2*z+z1},{2,-1/2*z+z2}},1/4*z,tau)+Gt({{4,-1/4*z+z1},{1,-1/4*z+z1},{2,-1/4*z+z2}},1/4*z,tau)");
	test(23, "Gt({{4,z1},{1,z1},{2,z2}},z,tau)", 5, "(Gt({{1,-3/5*z+z1}},1/5*z,tau))*(Gt({{2,z2}},1/5*z,tau))*(Gt({{4,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,-2/5*z+z1}},1/5*z,tau))*(Gt({{2,z2}},1/5*z,tau))*(Gt({{4,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,-1/5*z+z1}},1/5*z,tau))*(Gt({{2,z2}},1/5*z,tau))*(Gt({{4,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,-3/5*z+z1}},1/5*z,tau))*(Gt({{2,-2/5*z+z2}},1/5*z,tau))*(Gt({{4,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,-3/5*z+z1}},1/5*z,tau))*(Gt({{2,-1/5*z+z2}},1/5*z,tau))*(Gt({{4,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,-2/5*z+z1}},1/5*z,tau))*(Gt({{2,-1/5*z+z2}},1/5*z,tau))*(Gt({{4,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,-2/5*z+z1}},1/5*z,tau))*(Gt({{2,z2}},1/5*z,tau))*(Gt({{4,-3/5*z+z1}},1/5*z,tau))+(Gt({{1,-1/5*z+z1}},1/5*z,tau))*(Gt({{2,z2}},1/5*z,tau))*(Gt({{4,-3/5*z+z1}},1/5*z,tau))+(Gt({{1,-2/5*z+z1}},1/5*z,tau))*(Gt({{2,-1/5*z+z2}},1/5*z,tau))*(Gt({{4,-3/5*z+z1}},1/5*z,tau))+(Gt({{1,-1/5*z+z1}},1/5*z,tau))*(Gt({{2,z2}},1/5*z,tau))*(Gt({{4,-2/5*z+z1}},1/5*z,tau))+(Gt({{4,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,z1},{2,z2}},1/5*z,tau))+(Gt({{4,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,z1},{2,z2}},1/5*z,tau))+(Gt({{4,-2/5*z+z1}},1/5*z,tau))*(Gt({{1,z1},{2,z2}},1/5*z,tau))+(Gt({{4,-1/5*z+z1}},1/5*z,tau))*(Gt({{1,z1},{2,z2}},1/5*z,tau))+(Gt({{4,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-3/5*z+z1},{2,-3/5*z+z2}},1/5*z,tau))+(Gt({{4,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-2/5*z+z1},{2,-2/5*z+z2}},1/5*z,tau))+(Gt({{4,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,-2/5*z+z1},{2,-2/5*z+z2}},1/5*z,tau))+(Gt({{4,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z1},{2,-1/5*z+z2}},1/5*z,tau))+(Gt({{4,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z1},{2,-1/5*z+z2}},1/5*z,tau))+(Gt({{4,-2/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z1},{2,-1/5*z+z2}},1/5*z,tau))+(Gt({{2,z2}},1/5*z,tau))*(Gt({{4,-4/5*z+z1},{1,-4/5*z+z1}},1/5*z,tau))+(Gt({{2,-3/5*z+z2}},1/5*z,tau))*(Gt({{4,-4/5*z+z1},{1,-4/5*z+z1}},1/5*z,tau))+(Gt({{2,-2/5*z+z2}},1/5*z,tau))*(Gt({{4,-4/5*z+z1},{1,-4/5*z+z1}},1/5*z,tau))+(Gt({{2,-1/5*z+z2}},1/5*z,tau))*(Gt({{4,-4/5*z+z1},{1,-4/5*z+z1}},1/5*z,tau))+(Gt({{2,z2}},1/5*z,tau))*(Gt({{4,-3/5*z+z1},{1,-3/5*z+z1}},1/5*z,tau))+(Gt({{2,-2/5*z+z2}},1/5*z,tau))*(Gt({{4,-3/5*z+z1},{1,-3/5*z+z1}},1/5*z,tau))+(Gt({{2,-1/5*z+z2}},1/5*z,tau))*(Gt({{4,-3/5*z+z1},{1,-3/5*z+z1}},1/5*z,tau))+(Gt({{2,z2}},1/5*z,tau))*(Gt({{4,-2/5*z+z1},{1,-2/5*z+z1}},1/5*z,tau))+(Gt({{2,-1/5*z+z2}},1/5*z,tau))*(Gt({{4,-2/5*z+z1},{1,-2/5*z+z1}},1/5*z,tau))+(Gt({{2,z2}},1/5*z,tau))*(Gt({{4,-1/5*z+z1},{1,-1/5*z+z1}},1/5*z,tau))+Gt({{4,z1},{1,z1},{2,z2}},1/5*z,tau)+Gt({{4,-4/5*z+z1},{1,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau)+Gt({{4,-3/5*z+z1},{1,-3/5*z+z1},{2,-3/5*z+z2}},1/5*z,tau)+Gt({{4,-2/5*z+z1},{1,-2/5*z+z1},{2,-2/5*z+z2}},1/5*z,tau)+Gt({{4,-1/5*z+z1},{1,-1/5*z+z1},{2,-1/5*z+z2}},1/5*z,tau)");
	test(24, "Gt({{4,z1},{1,z1},{2,z2}},z,tau)", 6, "(Gt({{1,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,z2}},1/6*z,tau))*(Gt({{4,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,z2}},1/6*z,tau))*(Gt({{4,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z1}},1/6*z,tau))*(Gt({{2,z2}},1/6*z,tau))*(Gt({{4,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/6*z+z1}},1/6*z,tau))*(Gt({{2,z2}},1/6*z,tau))*(Gt({{4,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{4,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{4,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{4,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z2}},1/6*z,tau))*(Gt({{4,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z2}},1/6*z,tau))*(Gt({{4,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z2}},1/6*z,tau))*(Gt({{4,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,z2}},1/6*z,tau))*(Gt({{4,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z1}},1/6*z,tau))*(Gt({{2,z2}},1/6*z,tau))*(Gt({{4,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/6*z+z1}},1/6*z,tau))*(Gt({{2,z2}},1/6*z,tau))*(Gt({{4,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{4,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z2}},1/6*z,tau))*(Gt({{4,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z2}},1/6*z,tau))*(Gt({{4,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z1}},1/6*z,tau))*(Gt({{2,z2}},1/6*z,tau))*(Gt({{4,-1/2*z+z1}},1/6*z,tau))+(Gt({{1,-1/6*z+z1}},1/6*z,tau))*(Gt({{2,z2}},1/6*z,tau))*(Gt({{4,-1/2*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z2}},1/6*z,tau))*(Gt({{4,-1/2*z+z1}},1/6*z,tau))+(Gt({{1,-1/6*z+z1}},1/6*z,tau))*(Gt({{2,z2}},1/6*z,tau))*(Gt({{4,-1/3*z+z1}},1/6*z,tau))+(Gt({{4,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z1},{2,z2}},1/6*z,tau))+(Gt({{4,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,z1},{2,z2}},1/6*z,tau))+(Gt({{4,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,z1},{2,z2}},1/6*z,tau))+(Gt({{4,-1/3*z+z1}},1/6*z,tau))*(Gt({{1,z1},{2,z2}},1/6*z,tau))+(Gt({{4,-1/6*z+z1}},1/6*z,tau))*(Gt({{1,z1},{2,z2}},1/6*z,tau))+(Gt({{4,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{4,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/2*z+z1},{2,-1/2*z+z2}},1/6*z,tau))+(Gt({{4,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/2*z+z1},{2,-1/2*z+z2}},1/6*z,tau))+(Gt({{4,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z1},{2,-1/3*z+z2}},1/6*z,tau))+(Gt({{4,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z1},{2,-1/3*z+z2}},1/6*z,tau))+(Gt({{4,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z1},{2,-1/3*z+z2}},1/6*z,tau))+(Gt({{4,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z1},{2,-1/6*z+z2}},1/6*z,tau))+(Gt({{4,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z1},{2,-1/6*z+z2}},1/6*z,tau))+(Gt({{4,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z1},{2,-1/6*z+z2}},1/6*z,tau))+(Gt({{4,-1/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z1},{2,-1/6*z+z2}},1/6*z,tau))+(Gt({{2,z2}},1/6*z,tau))*(Gt({{4,-5/6*z+z1},{1,-5/6*z+z1}},1/6*z,tau))+(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{4,-5/6*z+z1},{1,-5/6*z+z1}},1/6*z,tau))+(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{4,-5/6*z+z1},{1,-5/6*z+z1}},1/6*z,tau))+(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{4,-5/6*z+z1},{1,-5/6*z+z1}},1/6*z,tau))+(Gt({{2,-1/6*z+z2}},1/6*z,tau))*(Gt({{4,-5/6*z+z1},{1,-5/6*z+z1}},1/6*z,tau))+(Gt({{2,z2}},1/6*z,tau))*(Gt({{4,-2/3*z+z1},{1,-2/3*z+z1}},1/6*z,tau))+(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{4,-2/3*z+z1},{1,-2/3*z+z1}},1/6*z,tau))+(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{4,-2/3*z+z1},{1,-2/3*z+z1}},1/6*z,tau))+(Gt({{2,-1/6*z+z2}},1/6*z,tau))*(Gt({{4,-2/3*z+z1},{1,-2/3*z+z1}},1/6*z,tau))+(Gt({{2,z2}},1/6*z,tau))*(Gt({{4,-1/2*z+z1},{1,-1/2*z+z1}},1/6*z,tau))+(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{4,-1/2*z+z1},{1,-1/2*z+z1}},1/6*z,tau))+(Gt({{2,-1/6*z+z2}},1/6*z,tau))*(Gt({{4,-1/2*z+z1},{1,-1/2*z+z1}},1/6*z,tau))+(Gt({{2,z2}},1/6*z,tau))*(Gt({{4,-1/3*z+z1},{1,-1/3*z+z1}},1/6*z,tau))+(Gt({{2,-1/6*z+z2}},1/6*z,tau))*(Gt({{4,-1/3*z+z1},{1,-1/3*z+z1}},1/6*z,tau))+(Gt({{2,z2}},1/6*z,tau))*(Gt({{4,-1/6*z+z1},{1,-1/6*z+z1}},1/6*z,tau))+Gt({{4,z1},{1,z1},{2,z2}},1/6*z,tau)+Gt({{4,-5/6*z+z1},{1,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau)+Gt({{4,-2/3*z+z1},{1,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau)+Gt({{4,-1/2*z+z1},{1,-1/2*z+z1},{2,-1/2*z+z2}},1/6*z,tau)+Gt({{4,-1/3*z+z1},{1,-1/3*z+z1},{2,-1/3*z+z2}},1/6*z,tau)+Gt({{4,-1/6*z+z1},{1,-1/6*z+z1},{2,-1/6*z+z2}},1/6*z,tau)");
	test(25, "Gt({{3,z2},{2,z1},{1,z2}},z,tau)", 1, "Gt({{3,z2},{2,z1},{1,z2}},z,tau)");
	test(26, "Gt({{3,z2},{2,z1},{1,z2}},z,tau)", 2, "(Gt({{3,-1/2*z+z2}},1/2*z,tau))*(Gt({{2,z1},{1,z2}},1/2*z,tau))+(Gt({{1,z2}},1/2*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/2*z,tau))+Gt({{3,z2},{2,z1},{1,z2}},1/2*z,tau)+Gt({{3,-1/2*z+z2},{2,-1/2*z+z1},{1,-1/2*z+z2}},1/2*z,tau)");
	test(27, "Gt({{3,z2},{2,z1},{1,z2}},z,tau)", 3, "(Gt({{1,z2}},1/3*z,tau))*(Gt({{2,-1/3*z+z1}},1/3*z,tau))*(Gt({{3,-2/3*z+z2}},1/3*z,tau))+(Gt({{3,-2/3*z+z2}},1/3*z,tau))*(Gt({{2,z1},{1,z2}},1/3*z,tau))+(Gt({{3,-1/3*z+z2}},1/3*z,tau))*(Gt({{2,z1},{1,z2}},1/3*z,tau))+(Gt({{3,-2/3*z+z2}},1/3*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2}},1/3*z,tau))+(Gt({{1,z2}},1/3*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/3*z,tau))+(Gt({{1,-1/3*z+z2}},1/3*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/3*z,tau))+(Gt({{1,z2}},1/3*z,tau))*(Gt({{3,-1/3*z+z2},{2,-1/3*z+z1}},1/3*z,tau))+Gt({{3,z2},{2,z1},{1,z2}},1/3*z,tau)+Gt({{3,-2/3*z+z2},{2,-2/3*z+z1},{1,-2/3*z+z2}},1/3*z,tau)+Gt({{3,-1/3*z+z2},{2,-1/3*z+z1},{1,-1/3*z+z2}},1/3*z,tau)");
	test(28, "Gt({{3,z2},{2,z1},{1,z2}},z,tau)", 4, "(Gt({{1,z2}},1/4*z,tau))*(Gt({{2,-1/2*z+z1}},1/4*z,tau))*(Gt({{3,-3/4*z+z2}},1/4*z,tau))+(Gt({{1,-1/4*z+z2}},1/4*z,tau))*(Gt({{2,-1/2*z+z1}},1/4*z,tau))*(Gt({{3,-3/4*z+z2}},1/4*z,tau))+(Gt({{1,z2}},1/4*z,tau))*(Gt({{2,-1/4*z+z1}},1/4*z,tau))*(Gt({{3,-3/4*z+z2}},1/4*z,tau))+(Gt({{1,z2}},1/4*z,tau))*(Gt({{2,-1/4*z+z1}},1/4*z,tau))*(Gt({{3,-1/2*z+z2}},1/4*z,tau))+(Gt({{3,-3/4*z+z2}},1/4*z,tau))*(Gt({{2,z1},{1,z2}},1/4*z,tau))+(Gt({{3,-1/2*z+z2}},1/4*z,tau))*(Gt({{2,z1},{1,z2}},1/4*z,tau))+(Gt({{3,-1/4*z+z2}},1/4*z,tau))*(Gt({{2,z1},{1,z2}},1/4*z,tau))+(Gt({{3,-3/4*z+z2}},1/4*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/4*z,tau))+(Gt({{3,-3/4*z+z2}},1/4*z,tau))*(Gt({{2,-1/4*z+z1},{1,-1/4*z+z2}},1/4*z,tau))+(Gt({{3,-1/2*z+z2}},1/4*z,tau))*(Gt({{2,-1/4*z+z1},{1,-1/4*z+z2}},1/4*z,tau))+(Gt({{1,z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z2},{2,-3/4*z+z1}},1/4*z,tau))+(Gt({{1,-1/2*z+z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z2},{2,-3/4*z+z1}},1/4*z,tau))+(Gt({{1,-1/4*z+z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z2},{2,-3/4*z+z1}},1/4*z,tau))+(Gt({{1,z2}},1/4*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/4*z,tau))+(Gt({{1,-1/4*z+z2}},1/4*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/4*z,tau))+(Gt({{1,z2}},1/4*z,tau))*(Gt({{3,-1/4*z+z2},{2,-1/4*z+z1}},1/4*z,tau))+Gt({{3,z2},{2,z1},{1,z2}},1/4*z,tau)+Gt({{3,-3/4*z+z2},{2,-3/4*z+z1},{1,-3/4*z+z2}},1/4*z,tau)+Gt({{3,-1/2*z+z2},{2,-1/2*z+z1},{1,-1/2*z+z2}},1/4*z,tau)+Gt({{3,-1/4*z+z2},{2,-1/4*z+z1},{1,-1/4*z+z2}},1/4*z,tau)");
	test(29, "Gt({{3,z2},{2,z1},{1,z2}},z,tau)", 5, "(Gt({{1,z2}},1/5*z,tau))*(Gt({{2,-3/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{2,-3/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{2,-3/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{2,-2/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{2,-2/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{2,-1/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{2,-2/5*z+z1}},1/5*z,tau))*(Gt({{3,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{2,-2/5*z+z1}},1/5*z,tau))*(Gt({{3,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{2,-1/5*z+z1}},1/5*z,tau))*(Gt({{3,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{2,-1/5*z+z1}},1/5*z,tau))*(Gt({{3,-2/5*z+z2}},1/5*z,tau))+(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{2,z1},{1,z2}},1/5*z,tau))+(Gt({{3,-3/5*z+z2}},1/5*z,tau))*(Gt({{2,z1},{1,z2}},1/5*z,tau))+(Gt({{3,-2/5*z+z2}},1/5*z,tau))*(Gt({{2,z1},{1,z2}},1/5*z,tau))+(Gt({{3,-1/5*z+z2}},1/5*z,tau))*(Gt({{2,z1},{1,z2}},1/5*z,tau))+(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{2,-3/5*z+z1},{1,-3/5*z+z2}},1/5*z,tau))+(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{2,-2/5*z+z1},{1,-2/5*z+z2}},1/5*z,tau))+(Gt({{3,-3/5*z+z2}},1/5*z,tau))*(Gt({{2,-2/5*z+z1},{1,-2/5*z+z2}},1/5*z,tau))+(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{2,-1/5*z+z1},{1,-1/5*z+z2}},1/5*z,tau))+(Gt({{3,-3/5*z+z2}},1/5*z,tau))*(Gt({{2,-1/5*z+z1},{1,-1/5*z+z2}},1/5*z,tau))+(Gt({{3,-2/5*z+z2}},1/5*z,tau))*(Gt({{2,-1/5*z+z1},{1,-1/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,-3/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z2},{2,-3/5*z+z1}},1/5*z,tau))+(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z2},{2,-3/5*z+z1}},1/5*z,tau))+(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z2},{2,-3/5*z+z1}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{3,-2/5*z+z2},{2,-2/5*z+z1}},1/5*z,tau))+(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-2/5*z+z2},{2,-2/5*z+z1}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{3,-1/5*z+z2},{2,-1/5*z+z1}},1/5*z,tau))+Gt({{3,z2},{2,z1},{1,z2}},1/5*z,tau)+Gt({{3,-4/5*z+z2},{2,-4/5*z+z1},{1,-4/5*z+z2}},1/5*z,tau)+Gt({{3,-3/5*z+z2},{2,-3/5*z+z1},{1,-3/5*z+z2}},1/5*z,tau)+Gt({{3,-2/5*z+z2},{2,-2/5*z+z1},{1,-2/5*z+z2}},1/5*z,tau)+Gt({{3,-1/5*z+z2},{2,-1/5*z+z1},{1,-1/5*z+z2}},1/5*z,tau)");
	test(30, "Gt({{3,z2},{2,z1},{1,z2}},z,tau)", 6, "(Gt({{1,z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{3,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{3,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1}},1/6*z,tau))*(Gt({{3,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1}},1/6*z,tau))*(Gt({{3,-1/3*z+z2}},1/6*z,tau))+(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,z1},{1,z2}},1/6*z,tau))+(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,z1},{1,z2}},1/6*z,tau))+(Gt({{3,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,z1},{1,z2}},1/6*z,tau))+(Gt({{3,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,z1},{1,z2}},1/6*z,tau))+(Gt({{3,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,z1},{1,z2}},1/6*z,tau))+(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1},{1,-2/3*z+z2}},1/6*z,tau))+(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{3,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{3,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{3,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-1/3*z+z2},{2,-1/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-1/3*z+z2},{2,-1/3*z+z1}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-1/6*z+z2},{2,-1/6*z+z1}},1/6*z,tau))+Gt({{3,z2},{2,z1},{1,z2}},1/6*z,tau)+Gt({{3,-5/6*z+z2},{2,-5/6*z+z1},{1,-5/6*z+z2}},1/6*z,tau)+Gt({{3,-2/3*z+z2},{2,-2/3*z+z1},{1,-2/3*z+z2}},1/6*z,tau)+Gt({{3,-1/2*z+z2},{2,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau)+Gt({{3,-1/3*z+z2},{2,-1/3*z+z1},{1,-1/3*z+z2}},1/6*z,tau)+Gt({{3,-1/6*z+z2},{2,-1/6*z+z1},{1,-1/6*z+z2}},1/6*z,tau)");
	test(31, "Gt({{3,z2},{2,z1},{1,z2},{1,z2}},z,tau)", 1, "Gt({{3,z2},{2,z1},{1,z2},{1,z2}},z,tau)");
	test(32, "Gt({{3,z2},{2,z1},{1,z2},{1,z2}},z,tau)", 2, "(Gt({{1,z2},{1,z2}},1/2*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/2*z,tau))+(Gt({{3,-1/2*z+z2}},1/2*z,tau))*(Gt({{2,z1},{1,z2},{1,z2}},1/2*z,tau))+(Gt({{1,z2}},1/2*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1},{1,-1/2*z+z2}},1/2*z,tau))+Gt({{3,z2},{2,z1},{1,z2},{1,z2}},1/2*z,tau)+Gt({{3,-1/2*z+z2},{2,-1/2*z+z1},{1,-1/2*z+z2},{1,-1/2*z+z2}},1/2*z,tau)");
	test(33, "Gt({{3,z2},{2,z1},{1,z2},{1,z2}},z,tau)", 3, "(Gt({{2,-1/3*z+z1}},1/3*z,tau))*(Gt({{3,-2/3*z+z2}},1/3*z,tau))*(Gt({{1,z2},{1,z2}},1/3*z,tau))+(Gt({{1,z2}},1/3*z,tau))*(Gt({{3,-2/3*z+z2}},1/3*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2}},1/3*z,tau))+(Gt({{1,z2}},1/3*z,tau))*(Gt({{1,-1/3*z+z2}},1/3*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/3*z,tau))+(Gt({{1,z2},{1,z2}},1/3*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/3*z,tau))+(Gt({{1,-1/3*z+z2},{1,-1/3*z+z2}},1/3*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/3*z,tau))+(Gt({{1,z2},{1,z2}},1/3*z,tau))*(Gt({{3,-1/3*z+z2},{2,-1/3*z+z1}},1/3*z,tau))+(Gt({{3,-2/3*z+z2}},1/3*z,tau))*(Gt({{2,z1},{1,z2},{1,z2}},1/3*z,tau))+(Gt({{3,-1/3*z+z2}},1/3*z,tau))*(Gt({{2,z1},{1,z2},{1,z2}},1/3*z,tau))+(Gt({{3,-2/3*z+z2}},1/3*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2},{1,-1/3*z+z2}},1/3*z,tau))+(Gt({{1,z2}},1/3*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1},{1,-2/3*z+z2}},1/3*z,tau))+(Gt({{1,-1/3*z+z2}},1/3*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1},{1,-2/3*z+z2}},1/3*z,tau))+(Gt({{1,z2}},1/3*z,tau))*(Gt({{3,-1/3*z+z2},{2,-1/3*z+z1},{1,-1/3*z+z2}},1/3*z,tau))+Gt({{3,z2},{2,z1},{1,z2},{1,z2}},1/3*z,tau)+Gt({{3,-2/3*z+z2},{2,-2/3*z+z1},{1,-2/3*z+z2},{1,-2/3*z+z2}},1/3*z,tau)+Gt({{3,-1/3*z+z2},{2,-1/3*z+z1},{1,-1/3*z+z2},{1,-1/3*z+z2}},1/3*z,tau)");
	test(34, "Gt({{3,z2},{2,z1},{1,z2},{1,z2}},z,tau)", 4, "(Gt({{1,z2}},1/4*z,tau))*(Gt({{1,-1/4*z+z2}},1/4*z,tau))*(Gt({{2,-1/2*z+z1}},1/4*z,tau))*(Gt({{3,-3/4*z+z2}},1/4*z,tau))+(Gt({{2,-1/2*z+z1}},1/4*z,tau))*(Gt({{3,-3/4*z+z2}},1/4*z,tau))*(Gt({{1,z2},{1,z2}},1/4*z,tau))+(Gt({{2,-1/4*z+z1}},1/4*z,tau))*(Gt({{3,-3/4*z+z2}},1/4*z,tau))*(Gt({{1,z2},{1,z2}},1/4*z,tau))+(Gt({{2,-1/4*z+z1}},1/4*z,tau))*(Gt({{3,-1/2*z+z2}},1/4*z,tau))*(Gt({{1,z2},{1,z2}},1/4*z,tau))+(Gt({{2,-1/2*z+z1}},1/4*z,tau))*(Gt({{3,-3/4*z+z2}},1/4*z,tau))*(Gt({{1,-1/4*z+z2},{1,-1/4*z+z2}},1/4*z,tau))+(Gt({{1,z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z2}},1/4*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/4*z,tau))+(Gt({{1,-1/4*z+z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z2}},1/4*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/4*z,tau))+(Gt({{1,z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z2}},1/4*z,tau))*(Gt({{2,-1/4*z+z1},{1,-1/4*z+z2}},1/4*z,tau))+(Gt({{1,z2}},1/4*z,tau))*(Gt({{3,-1/2*z+z2}},1/4*z,tau))*(Gt({{2,-1/4*z+z1},{1,-1/4*z+z2}},1/4*z,tau))+(Gt({{1,z2}},1/4*z,tau))*(Gt({{1,-1/2*z+z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z2},{2,-3/4*z+z1}},1/4*z,tau))+(Gt({{1,z2}},1/4*z,tau))*(Gt({{1,-1/4*z+z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z2},{2,-3/4*z+z1}},1/4*z,tau))+(Gt({{1,-1/2*z+z2}},1/4*z,tau))*(Gt({{1,-1/4*z+z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z2},{2,-3/4*z+z1}},1/4*z,tau))+(Gt({{1,z2},{1,z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z2},{2,-3/4*z+z1}},1/4*z,tau))+(Gt({{1,-1/2*z+z2},{1,-1/2*z+z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z2},{2,-3/4*z+z1}},1/4*z,tau))+(Gt({{1,-1/4*z+z2},{1,-1/4*z+z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z2},{2,-3/4*z+z1}},1/4*z,tau))+(Gt({{1,z2}},1/4*z,tau))*(Gt({{1,-1/4*z+z2}},1/4*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/4*z,tau))+(Gt({{1,z2},{1,z2}},1/4*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/4*z,tau))+(Gt({{1,-1/4*z+z2},{1,-1/4*z+z2}},1/4*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/4*z,tau))+(Gt({{1,z2},{1,z2}},1/4*z,tau))*(Gt({{3,-1/4*z+z2},{2,-1/4*z+z1}},1/4*z,tau))+(Gt({{3,-3/4*z+z2}},1/4*z,tau))*(Gt({{2,z1},{1,z2},{1,z2}},1/4*z,tau))+(Gt({{3,-1/2*z+z2}},1/4*z,tau))*(Gt({{2,z1},{1,z2},{1,z2}},1/4*z,tau))+(Gt({{3,-1/4*z+z2}},1/4*z,tau))*(Gt({{2,z1},{1,z2},{1,z2}},1/4*z,tau))+(Gt({{3,-3/4*z+z2}},1/4*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2},{1,-1/2*z+z2}},1/4*z,tau))+(Gt({{3,-3/4*z+z2}},1/4*z,tau))*(Gt({{2,-1/4*z+z1},{1,-1/4*z+z2},{1,-1/4*z+z2}},1/4*z,tau))+(Gt({{3,-1/2*z+z2}},1/4*z,tau))*(Gt({{2,-1/4*z+z1},{1,-1/4*z+z2},{1,-1/4*z+z2}},1/4*z,tau))+(Gt({{1,z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z2},{2,-3/4*z+z1},{1,-3/4*z+z2}},1/4*z,tau))+(Gt({{1,-1/2*z+z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z2},{2,-3/4*z+z1},{1,-3/4*z+z2}},1/4*z,tau))+(Gt({{1,-1/4*z+z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z2},{2,-3/4*z+z1},{1,-3/4*z+z2}},1/4*z,tau))+(Gt({{1,z2}},1/4*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1},{1,-1/2*z+z2}},1/4*z,tau))+(Gt({{1,-1/4*z+z2}},1/4*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1},{1,-1/2*z+z2}},1/4*z,tau))+(Gt({{1,z2}},1/4*z,tau))*(Gt({{3,-1/4*z+z2},{2,-1/4*z+z1},{1,-1/4*z+z2}},1/4*z,tau))+Gt({{3,z2},{2,z1},{1,z2},{1,z2}},1/4*z,tau)+Gt({{3,-3/4*z+z2},{2,-3/4*z+z1},{1,-3/4*z+z2},{1,-3/4*z+z2}},1/4*z,tau)+Gt({{3,-1/2*z+z2},{2,-1/2*z+z1},{1,-1/2*z+z2},{1,-1/2*z+z2}},1/4*z,tau)+Gt({{3,-1/4*z+z2},{2,-1/4*z+z1},{1,-1/4*z+z2},{1,-1/4*z+z2}},1/4*z,tau)");
	test(35, "Gt({{3,z2},{2,z1},{1,z2},{1,z2}},z,tau)", 5, "(Gt({{1,z2}},1/5*z,tau))*(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{2,-3/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{2,-3/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{2,-3/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{2,-2/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{2,-2/5*z+z1}},1/5*z,tau))*(Gt({{3,-3/5*z+z2}},1/5*z,tau))+(Gt({{2,-3/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{1,z2},{1,z2}},1/5*z,tau))+(Gt({{2,-2/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{1,z2},{1,z2}},1/5*z,tau))+(Gt({{2,-1/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{1,z2},{1,z2}},1/5*z,tau))+(Gt({{2,-2/5*z+z1}},1/5*z,tau))*(Gt({{3,-3/5*z+z2}},1/5*z,tau))*(Gt({{1,z2},{1,z2}},1/5*z,tau))+(Gt({{2,-1/5*z+z1}},1/5*z,tau))*(Gt({{3,-3/5*z+z2}},1/5*z,tau))*(Gt({{1,z2},{1,z2}},1/5*z,tau))+(Gt({{2,-1/5*z+z1}},1/5*z,tau))*(Gt({{3,-2/5*z+z2}},1/5*z,tau))*(Gt({{1,z2},{1,z2}},1/5*z,tau))+(Gt({{2,-3/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{1,-2/5*z+z2},{1,-2/5*z+z2}},1/5*z,tau))+(Gt({{2,-3/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z2},{1,-1/5*z+z2}},1/5*z,tau))+(Gt({{2,-2/5*z+z1}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z2},{1,-1/5*z+z2}},1/5*z,tau))+(Gt({{2,-2/5*z+z1}},1/5*z,tau))*(Gt({{3,-3/5*z+z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z2},{1,-1/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{2,-3/5*z+z1},{1,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{2,-3/5*z+z1},{1,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{2,-3/5*z+z1},{1,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{2,-2/5*z+z1},{1,-2/5*z+z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{2,-2/5*z+z1},{1,-2/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z2}},1/5*z,tau))*(Gt({{2,-2/5*z+z1},{1,-2/5*z+z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z2}},1/5*z,tau))*(Gt({{2,-2/5*z+z1},{1,-2/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{2,-1/5*z+z1},{1,-1/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z2}},1/5*z,tau))*(Gt({{2,-1/5*z+z1},{1,-1/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{3,-2/5*z+z2}},1/5*z,tau))*(Gt({{2,-1/5*z+z1},{1,-1/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{1,-3/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,-3/5*z+z2}},1/5*z,tau))*(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,-3/5*z+z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,z2},{1,z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,-3/5*z+z2},{1,-3/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,-2/5*z+z2},{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,-1/5*z+z2},{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z2},{2,-3/5*z+z1}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z2},{2,-3/5*z+z1}},1/5*z,tau))+(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z2},{2,-3/5*z+z1}},1/5*z,tau))+(Gt({{1,z2},{1,z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z2},{2,-3/5*z+z1}},1/5*z,tau))+(Gt({{1,-2/5*z+z2},{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z2},{2,-3/5*z+z1}},1/5*z,tau))+(Gt({{1,-1/5*z+z2},{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z2},{2,-3/5*z+z1}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-2/5*z+z2},{2,-2/5*z+z1}},1/5*z,tau))+(Gt({{1,z2},{1,z2}},1/5*z,tau))*(Gt({{3,-2/5*z+z2},{2,-2/5*z+z1}},1/5*z,tau))+(Gt({{1,-1/5*z+z2},{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-2/5*z+z2},{2,-2/5*z+z1}},1/5*z,tau))+(Gt({{1,z2},{1,z2}},1/5*z,tau))*(Gt({{3,-1/5*z+z2},{2,-1/5*z+z1}},1/5*z,tau))+(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{2,z1},{1,z2},{1,z2}},1/5*z,tau))+(Gt({{3,-3/5*z+z2}},1/5*z,tau))*(Gt({{2,z1},{1,z2},{1,z2}},1/5*z,tau))+(Gt({{3,-2/5*z+z2}},1/5*z,tau))*(Gt({{2,z1},{1,z2},{1,z2}},1/5*z,tau))+(Gt({{3,-1/5*z+z2}},1/5*z,tau))*(Gt({{2,z1},{1,z2},{1,z2}},1/5*z,tau))+(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{2,-3/5*z+z1},{1,-3/5*z+z2},{1,-3/5*z+z2}},1/5*z,tau))+(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{2,-2/5*z+z1},{1,-2/5*z+z2},{1,-2/5*z+z2}},1/5*z,tau))+(Gt({{3,-3/5*z+z2}},1/5*z,tau))*(Gt({{2,-2/5*z+z1},{1,-2/5*z+z2},{1,-2/5*z+z2}},1/5*z,tau))+(Gt({{3,-4/5*z+z2}},1/5*z,tau))*(Gt({{2,-1/5*z+z1},{1,-1/5*z+z2},{1,-1/5*z+z2}},1/5*z,tau))+(Gt({{3,-3/5*z+z2}},1/5*z,tau))*(Gt({{2,-1/5*z+z1},{1,-1/5*z+z2},{1,-1/5*z+z2}},1/5*z,tau))+(Gt({{3,-2/5*z+z2}},1/5*z,tau))*(Gt({{2,-1/5*z+z1},{1,-1/5*z+z2},{1,-1/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1},{1,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-3/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1},{1,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1},{1,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z2},{2,-4/5*z+z1},{1,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z2},{2,-3/5*z+z1},{1,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z2},{2,-3/5*z+z1},{1,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z2},{2,-3/5*z+z1},{1,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{3,-2/5*z+z2},{2,-2/5*z+z1},{1,-2/5*z+z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-2/5*z+z2},{2,-2/5*z+z1},{1,-2/5*z+z2}},1/5*z,tau))+(Gt({{1,z2}},1/5*z,tau))*(Gt({{3,-1/5*z+z2},{2,-1/5*z+z1},{1,-1/5*z+z2}},1/5*z,tau))+Gt({{3,z2},{2,z1},{1,z2},{1,z2}},1/5*z,tau)+Gt({{3,-4/5*z+z2},{2,-4/5*z+z1},{1,-4/5*z+z2},{1,-4/5*z+z2}},1/5*z,tau)+Gt({{3,-3/5*z+z2},{2,-3/5*z+z1},{1,-3/5*z+z2},{1,-3/5*z+z2}},1/5*z,tau)+Gt({{3,-2/5*z+z2},{2,-2/5*z+z1},{1,-2/5*z+z2},{1,-2/5*z+z2}},1/5*z,tau)+Gt({{3,-1/5*z+z2},{2,-1/5*z+z1},{1,-1/5*z+z2},{1,-1/5*z+z2}},1/5*z,tau)");
	test(36, "Gt({{3,z2},{2,z1},{1,z2},{1,z2}},z,tau)", 6, "(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{3,-1/2*z+z2}},1/6*z,tau))+(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{1,z2},{1,z2}},1/6*z,tau))+(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{1,z2},{1,z2}},1/6*z,tau))+(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{1,z2},{1,z2}},1/6*z,tau))+(Gt({{2,-1/6*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{1,z2},{1,z2}},1/6*z,tau))+(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{1,z2},{1,z2}},1/6*z,tau))+(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{1,z2},{1,z2}},1/6*z,tau))+(Gt({{2,-1/6*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{1,z2},{1,z2}},1/6*z,tau))+(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{3,-1/2*z+z2}},1/6*z,tau))*(Gt({{1,z2},{1,z2}},1/6*z,tau))+(Gt({{2,-1/6*z+z1}},1/6*z,tau))*(Gt({{3,-1/2*z+z2}},1/6*z,tau))*(Gt({{1,z2},{1,z2}},1/6*z,tau))+(Gt({{2,-1/6*z+z1}},1/6*z,tau))*(Gt({{3,-1/3*z+z2}},1/6*z,tau))*(Gt({{1,z2},{1,z2}},1/6*z,tau))+(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{1,-1/2*z+z2},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z2},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z2},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z2},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{2,-2/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{2,-1/2*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{2,-1/3*z+z1}},1/6*z,tau))*(Gt({{3,-1/2*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1},{1,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1},{1,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1},{1,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1},{1,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-2/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-2/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-2/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,z2},{1,z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-2/3*z+z2},{1,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z2},{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z2},{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/6*z+z2},{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,z2},{1,z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z2},{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z2},{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/6*z+z2},{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/6*z,tau))+(Gt({{1,z2},{1,z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z2},{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/6*z,tau))+(Gt({{1,-1/6*z+z2},{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-1/3*z+z2},{2,-1/3*z+z1}},1/6*z,tau))+(Gt({{1,z2},{1,z2}},1/6*z,tau))*(Gt({{3,-1/3*z+z2},{2,-1/3*z+z1}},1/6*z,tau))+(Gt({{1,-1/6*z+z2},{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-1/3*z+z2},{2,-1/3*z+z1}},1/6*z,tau))+(Gt({{1,z2},{1,z2}},1/6*z,tau))*(Gt({{3,-1/6*z+z2},{2,-1/6*z+z1}},1/6*z,tau))+(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,z1},{1,z2},{1,z2}},1/6*z,tau))+(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,z1},{1,z2},{1,z2}},1/6*z,tau))+(Gt({{3,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,z1},{1,z2},{1,z2}},1/6*z,tau))+(Gt({{3,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,z1},{1,z2},{1,z2}},1/6*z,tau))+(Gt({{3,-1/6*z+z2}},1/6*z,tau))*(Gt({{2,z1},{1,z2},{1,z2}},1/6*z,tau))+(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-2/3*z+z1},{1,-2/3*z+z2},{1,-2/3*z+z2}},1/6*z,tau))+(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/2*z+z1},{1,-1/2*z+z2},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{3,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,-1/3*z+z1},{1,-1/3*z+z2},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{3,-5/6*z+z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1},{1,-1/6*z+z2},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{3,-2/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1},{1,-1/6*z+z2},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{3,-1/2*z+z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1},{1,-1/6*z+z2},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{3,-1/3*z+z2}},1/6*z,tau))*(Gt({{2,-1/6*z+z1},{1,-1/6*z+z2},{1,-1/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1},{1,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1},{1,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1},{1,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1},{1,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z2},{2,-5/6*z+z1},{1,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1},{1,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1},{1,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1},{1,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z2},{2,-2/3*z+z1},{1,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z2},{2,-1/2*z+z1},{1,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-1/3*z+z2},{2,-1/3*z+z1},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-1/3*z+z2},{2,-1/3*z+z1},{1,-1/3*z+z2}},1/6*z,tau))+(Gt({{1,z2}},1/6*z,tau))*(Gt({{3,-1/6*z+z2},{2,-1/6*z+z1},{1,-1/6*z+z2}},1/6*z,tau))+Gt({{3,z2},{2,z1},{1,z2},{1,z2}},1/6*z,tau)+Gt({{3,-5/6*z+z2},{2,-5/6*z+z1},{1,-5/6*z+z2},{1,-5/6*z+z2}},1/6*z,tau)+Gt({{3,-2/3*z+z2},{2,-2/3*z+z1},{1,-2/3*z+z2},{1,-2/3*z+z2}},1/6*z,tau)+Gt({{3,-1/2*z+z2},{2,-1/2*z+z1},{1,-1/2*z+z2},{1,-1/2*z+z2}},1/6*z,tau)+Gt({{3,-1/3*z+z2},{2,-1/3*z+z1},{1,-1/3*z+z2},{1,-1/3*z+z2}},1/6*z,tau)+Gt({{3,-1/6*z+z2},{2,-1/6*z+z1},{1,-1/6*z+z2},{1,-1/6*z+z2}},1/6*z,tau)");
	test(37, "Gt({{3,z1},{2,z2},{1,z3},{1,z4},{1,z5}},z,tau)", 1, "Gt({{3,z1},{2,z2},{1,z3},{1,z4},{1,z5}},z,tau)");
	test(38, "Gt({{3,z1},{2,z2},{1,z3},{1,z4},{1,z5}},z,tau)", 2, "(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2}},1/2*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/2*z,tau))+(Gt({{1,z4},{1,z5}},1/2*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3}},1/2*z,tau))+(Gt({{3,-1/2*z+z1}},1/2*z,tau))*(Gt({{2,z2},{1,z3},{1,z4},{1,z5}},1/2*z,tau))+(Gt({{1,z5}},1/2*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4}},1/2*z,tau))+Gt({{3,z1},{2,z2},{1,z3},{1,z4},{1,z5}},1/2*z,tau)+Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4},{1,-1/2*z+z5}},1/2*z,tau)");
	test(39, "Gt({{3,z1},{2,z2},{1,z3},{1,z4},{1,z5}},z,tau)", 3, "(Gt({{3,-2/3*z+z1}},1/3*z,tau))*(Gt({{1,z4},{1,z5}},1/3*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3}},1/3*z,tau))+(Gt({{1,z5}},1/3*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4}},1/3*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/3*z,tau))+(Gt({{1,-1/3*z+z3}},1/3*z,tau))*(Gt({{1,z4},{1,z5}},1/3*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/3*z,tau))+(Gt({{2,-1/3*z+z2}},1/3*z,tau))*(Gt({{3,-2/3*z+z1}},1/3*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/3*z,tau))+(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/3*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/3*z,tau))+(Gt({{3,-1/3*z+z1},{2,-1/3*z+z2}},1/3*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/3*z,tau))+(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/3*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4},{1,-1/3*z+z5}},1/3*z,tau))+(Gt({{1,z5}},1/3*z,tau))*(Gt({{3,-2/3*z+z1}},1/3*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4}},1/3*z,tau))+(Gt({{1,-1/3*z+z4}},1/3*z,tau))*(Gt({{1,z5}},1/3*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3}},1/3*z,tau))+(Gt({{1,z4},{1,z5}},1/3*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3}},1/3*z,tau))+(Gt({{1,-1/3*z+z4},{1,-1/3*z+z5}},1/3*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3}},1/3*z,tau))+(Gt({{1,z4},{1,z5}},1/3*z,tau))*(Gt({{3,-1/3*z+z1},{2,-1/3*z+z2},{1,-1/3*z+z3}},1/3*z,tau))+(Gt({{3,-2/3*z+z1}},1/3*z,tau))*(Gt({{2,z2},{1,z3},{1,z4},{1,z5}},1/3*z,tau))+(Gt({{3,-1/3*z+z1}},1/3*z,tau))*(Gt({{2,z2},{1,z3},{1,z4},{1,z5}},1/3*z,tau))+(Gt({{3,-2/3*z+z1}},1/3*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4},{1,-1/3*z+z5}},1/3*z,tau))+(Gt({{1,z5}},1/3*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3},{1,-2/3*z+z4}},1/3*z,tau))+(Gt({{1,-1/3*z+z5}},1/3*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3},{1,-2/3*z+z4}},1/3*z,tau))+(Gt({{1,z5}},1/3*z,tau))*(Gt({{3,-1/3*z+z1},{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4}},1/3*z,tau))+Gt({{3,z1},{2,z2},{1,z3},{1,z4},{1,z5}},1/3*z,tau)+Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3},{1,-2/3*z+z4},{1,-2/3*z+z5}},1/3*z,tau)+Gt({{3,-1/3*z+z1},{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4},{1,-1/3*z+z5}},1/3*z,tau)");
	test(40, "Gt({{3,z1},{2,z2},{1,z3},{1,z4},{1,z5}},z,tau)", 4, "(Gt({{1,z5}},1/4*z,tau))*(Gt({{2,-1/2*z+z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,-1/4*z+z3},{1,-1/4*z+z4}},1/4*z,tau))+(Gt({{1,-1/4*z+z3}},1/4*z,tau))*(Gt({{2,-1/2*z+z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,z4},{1,z5}},1/4*z,tau))+(Gt({{1,-1/4*z+z4}},1/4*z,tau))*(Gt({{1,z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1}},1/4*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3}},1/4*z,tau))+(Gt({{3,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,z4},{1,z5}},1/4*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3}},1/4*z,tau))+(Gt({{3,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,-1/4*z+z4},{1,-1/4*z+z5}},1/4*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3}},1/4*z,tau))+(Gt({{3,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,z4},{1,z5}},1/4*z,tau))*(Gt({{2,-1/4*z+z2},{1,-1/4*z+z3}},1/4*z,tau))+(Gt({{3,-1/2*z+z1}},1/4*z,tau))*(Gt({{1,z4},{1,z5}},1/4*z,tau))*(Gt({{2,-1/4*z+z2},{1,-1/4*z+z3}},1/4*z,tau))+(Gt({{1,-1/2*z+z3}},1/4*z,tau))*(Gt({{1,-1/4*z+z4}},1/4*z,tau))*(Gt({{1,z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2}},1/4*z,tau))+(Gt({{1,z5}},1/4*z,tau))*(Gt({{1,-1/2*z+z3},{1,-1/2*z+z4}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2}},1/4*z,tau))+(Gt({{1,-1/4*z+z5}},1/4*z,tau))*(Gt({{1,-1/2*z+z3},{1,-1/2*z+z4}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2}},1/4*z,tau))+(Gt({{1,z5}},1/4*z,tau))*(Gt({{1,-1/4*z+z3},{1,-1/4*z+z4}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2}},1/4*z,tau))+(Gt({{1,-1/2*z+z3}},1/4*z,tau))*(Gt({{1,z4},{1,z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2}},1/4*z,tau))+(Gt({{1,-1/4*z+z3}},1/4*z,tau))*(Gt({{1,z4},{1,z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2}},1/4*z,tau))+(Gt({{1,-1/2*z+z3}},1/4*z,tau))*(Gt({{1,-1/4*z+z4},{1,-1/4*z+z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2}},1/4*z,tau))+(Gt({{1,z5}},1/4*z,tau))*(Gt({{1,-1/4*z+z3},{1,-1/4*z+z4}},1/4*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2}},1/4*z,tau))+(Gt({{1,-1/4*z+z3}},1/4*z,tau))*(Gt({{1,z4},{1,z5}},1/4*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2}},1/4*z,tau))+(Gt({{2,-1/2*z+z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/4*z,tau))+(Gt({{2,-1/4*z+z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/4*z,tau))+(Gt({{2,-1/4*z+z2}},1/4*z,tau))*(Gt({{3,-1/2*z+z1}},1/4*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/4*z,tau))+(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2}},1/4*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/4*z,tau))+(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2}},1/4*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/4*z,tau))+(Gt({{3,-1/4*z+z1},{2,-1/4*z+z2}},1/4*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/4*z,tau))+(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2}},1/4*z,tau))*(Gt({{1,-1/2*z+z3},{1,-1/2*z+z4},{1,-1/2*z+z5}},1/4*z,tau))+(Gt({{2,-1/2*z+z2}},1/4*z,tau))*(Gt({{3,-3/4*z+z1}},1/4*z,tau))*(Gt({{1,-1/4*z+z3},{1,-1/4*z+z4},{1,-1/4*z+z5}},1/4*z,tau))+(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2}},1/4*z,tau))*(Gt({{1,-1/4*z+z3},{1,-1/4*z+z4},{1,-1/4*z+z5}},1/4*z,tau))+(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2}},1/4*z,tau))*(Gt({{1,-1/4*z+z3},{1,-1/4*z+z4},{1,-1/4*z+z5}},1/4*z,tau))+(Gt({{1,z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1}},1/4*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4}},1/4*z,tau))+(Gt({{1,-1/4*z+z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1}},1/4*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4}},1/4*z,tau))+(Gt({{1,z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1}},1/4*z,tau))*(Gt({{2,-1/4*z+z2},{1,-1/4*z+z3},{1,-1/4*z+z4}},1/4*z,tau))+(Gt({{1,z5}},1/4*z,tau))*(Gt({{3,-1/2*z+z1}},1/4*z,tau))*(Gt({{2,-1/4*z+z2},{1,-1/4*z+z3},{1,-1/4*z+z4}},1/4*z,tau))+(Gt({{1,-1/2*z+z4}},1/4*z,tau))*(Gt({{1,z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2},{1,-3/4*z+z3}},1/4*z,tau))+(Gt({{1,-1/4*z+z4}},1/4*z,tau))*(Gt({{1,z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2},{1,-3/4*z+z3}},1/4*z,tau))+(Gt({{1,-1/2*z+z4}},1/4*z,tau))*(Gt({{1,-1/4*z+z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2},{1,-3/4*z+z3}},1/4*z,tau))+(Gt({{1,z4},{1,z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2},{1,-3/4*z+z3}},1/4*z,tau))+(Gt({{1,-1/2*z+z4},{1,-1/2*z+z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2},{1,-3/4*z+z3}},1/4*z,tau))+(Gt({{1,-1/4*z+z4},{1,-1/4*z+z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2},{1,-3/4*z+z3}},1/4*z,tau))+(Gt({{1,-1/4*z+z4}},1/4*z,tau))*(Gt({{1,z5}},1/4*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3}},1/4*z,tau))+(Gt({{1,z4},{1,z5}},1/4*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3}},1/4*z,tau))+(Gt({{1,-1/4*z+z4},{1,-1/4*z+z5}},1/4*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3}},1/4*z,tau))+(Gt({{1,z4},{1,z5}},1/4*z,tau))*(Gt({{3,-1/4*z+z1},{2,-1/4*z+z2},{1,-1/4*z+z3}},1/4*z,tau))+(Gt({{3,-3/4*z+z1}},1/4*z,tau))*(Gt({{2,z2},{1,z3},{1,z4},{1,z5}},1/4*z,tau))+(Gt({{3,-1/2*z+z1}},1/4*z,tau))*(Gt({{2,z2},{1,z3},{1,z4},{1,z5}},1/4*z,tau))+(Gt({{3,-1/4*z+z1}},1/4*z,tau))*(Gt({{2,z2},{1,z3},{1,z4},{1,z5}},1/4*z,tau))+(Gt({{3,-3/4*z+z1}},1/4*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4},{1,-1/2*z+z5}},1/4*z,tau))+(Gt({{3,-3/4*z+z1}},1/4*z,tau))*(Gt({{2,-1/4*z+z2},{1,-1/4*z+z3},{1,-1/4*z+z4},{1,-1/4*z+z5}},1/4*z,tau))+(Gt({{3,-1/2*z+z1}},1/4*z,tau))*(Gt({{2,-1/4*z+z2},{1,-1/4*z+z3},{1,-1/4*z+z4},{1,-1/4*z+z5}},1/4*z,tau))+(Gt({{1,z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2},{1,-3/4*z+z3},{1,-3/4*z+z4}},1/4*z,tau))+(Gt({{1,-1/2*z+z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2},{1,-3/4*z+z3},{1,-3/4*z+z4}},1/4*z,tau))+(Gt({{1,-1/4*z+z5}},1/4*z,tau))*(Gt({{3,-3/4*z+z1},{2,-3/4*z+z2},{1,-3/4*z+z3},{1,-3/4*z+z4}},1/4*z,tau))+(Gt({{1,z5}},1/4*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4}},1/4*z,tau))+(Gt({{1,-1/4*z+z5}},1/4*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4}},1/4*z,tau))+(Gt({{1,z5}},1/4*z,tau))*(Gt({{3,-1/4*z+z1},{2,-1/4*z+z2},{1,-1/4*z+z3},{1,-1/4*z+z4}},1/4*z,tau))+Gt({{3,z1},{2,z2},{1,z3},{1,z4},{1,z5}},1/4*z,tau)+Gt({{3,-3/4*z+z1},{2,-3/4*z+z2},{1,-3/4*z+z3},{1,-3/4*z+z4},{1,-3/4*z+z5}},1/4*z,tau)+Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4},{1,-1/2*z+z5}},1/4*z,tau)+Gt({{3,-1/4*z+z1},{2,-1/4*z+z2},{1,-1/4*z+z3},{1,-1/4*z+z4},{1,-1/4*z+z5}},1/4*z,tau)");
	test(41, "Gt({{3,z1},{2,z2},{1,z3},{1,z4},{1,z5}},z,tau)", 5, "(Gt({{1,-2/5*z+z3}},1/5*z,tau))*(Gt({{1,-1/5*z+z4}},1/5*z,tau))*(Gt({{1,z5}},1/5*z,tau))*(Gt({{2,-3/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{2,-3/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-2/5*z+z3},{1,-2/5*z+z4}},1/5*z,tau))+(Gt({{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{2,-3/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-2/5*z+z3},{1,-2/5*z+z4}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{2,-3/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z3},{1,-1/5*z+z4}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{2,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z3},{1,-1/5*z+z4}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{2,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z3},{1,-1/5*z+z4}},1/5*z,tau))+(Gt({{1,-2/5*z+z3}},1/5*z,tau))*(Gt({{2,-3/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))+(Gt({{1,-1/5*z+z3}},1/5*z,tau))*(Gt({{2,-3/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))+(Gt({{1,-1/5*z+z3}},1/5*z,tau))*(Gt({{2,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))+(Gt({{1,-1/5*z+z3}},1/5*z,tau))*(Gt({{2,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))+(Gt({{1,-2/5*z+z3}},1/5*z,tau))*(Gt({{2,-3/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))+(Gt({{1,-2/5*z+z4}},1/5*z,tau))*(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-3/5*z+z2},{1,-3/5*z+z3}},1/5*z,tau))+(Gt({{1,-1/5*z+z4}},1/5*z,tau))*(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-3/5*z+z2},{1,-3/5*z+z3}},1/5*z,tau))+(Gt({{1,-2/5*z+z4}},1/5*z,tau))*(Gt({{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-3/5*z+z2},{1,-3/5*z+z3}},1/5*z,tau))+(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{2,-3/5*z+z2},{1,-3/5*z+z3}},1/5*z,tau))+(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-2/5*z+z4},{1,-2/5*z+z5}},1/5*z,tau))*(Gt({{2,-3/5*z+z2},{1,-3/5*z+z3}},1/5*z,tau))+(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{2,-3/5*z+z2},{1,-3/5*z+z3}},1/5*z,tau))+(Gt({{1,-1/5*z+z4}},1/5*z,tau))*(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-2/5*z+z2},{1,-2/5*z+z3}},1/5*z,tau))+(Gt({{1,-1/5*z+z4}},1/5*z,tau))*(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1}},1/5*z,tau))*(Gt({{2,-2/5*z+z2},{1,-2/5*z+z3}},1/5*z,tau))+(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{2,-2/5*z+z2},{1,-2/5*z+z3}},1/5*z,tau))+(Gt({{3,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{2,-2/5*z+z2},{1,-2/5*z+z3}},1/5*z,tau))+(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{2,-2/5*z+z2},{1,-2/5*z+z3}},1/5*z,tau))+(Gt({{3,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{2,-2/5*z+z2},{1,-2/5*z+z3}},1/5*z,tau))+(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{2,-1/5*z+z2},{1,-1/5*z+z3}},1/5*z,tau))+(Gt({{3,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{2,-1/5*z+z2},{1,-1/5*z+z3}},1/5*z,tau))+(Gt({{3,-2/5*z+z1}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{2,-1/5*z+z2},{1,-1/5*z+z3}},1/5*z,tau))+(Gt({{1,-3/5*z+z3}},1/5*z,tau))*(Gt({{1,-2/5*z+z4}},1/5*z,tau))*(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-3/5*z+z3}},1/5*z,tau))*(Gt({{1,-1/5*z+z4}},1/5*z,tau))*(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-2/5*z+z3}},1/5*z,tau))*(Gt({{1,-1/5*z+z4}},1/5*z,tau))*(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-3/5*z+z3}},1/5*z,tau))*(Gt({{1,-2/5*z+z4}},1/5*z,tau))*(Gt({{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{1,-3/5*z+z3},{1,-3/5*z+z4}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-2/5*z+z5}},1/5*z,tau))*(Gt({{1,-3/5*z+z3},{1,-3/5*z+z4}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{1,-3/5*z+z3},{1,-3/5*z+z4}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{1,-2/5*z+z3},{1,-2/5*z+z4}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{1,-2/5*z+z3},{1,-2/5*z+z4}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{1,-1/5*z+z3},{1,-1/5*z+z4}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-3/5*z+z3}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-2/5*z+z3}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z3}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-3/5*z+z3}},1/5*z,tau))*(Gt({{1,-2/5*z+z4},{1,-2/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-3/5*z+z3}},1/5*z,tau))*(Gt({{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-2/5*z+z3}},1/5*z,tau))*(Gt({{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))+(Gt({{1,-2/5*z+z3}},1/5*z,tau))*(Gt({{1,-1/5*z+z4}},1/5*z,tau))*(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{1,-2/5*z+z3},{1,-2/5*z+z4}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{1,-2/5*z+z3},{1,-2/5*z+z4}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{1,-1/5*z+z3},{1,-1/5*z+z4}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,-2/5*z+z3}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z3}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,-2/5*z+z3}},1/5*z,tau))*(Gt({{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{1,-1/5*z+z3},{1,-1/5*z+z4}},1/5*z,tau))*(Gt({{3,-2/5*z+z1},{2,-2/5*z+z2}},1/5*z,tau))+(Gt({{1,-1/5*z+z3}},1/5*z,tau))*(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{3,-2/5*z+z1},{2,-2/5*z+z2}},1/5*z,tau))+(Gt({{2,-3/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/5*z,tau))+(Gt({{2,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/5*z,tau))+(Gt({{2,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/5*z,tau))+(Gt({{2,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/5*z,tau))+(Gt({{2,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/5*z,tau))+(Gt({{2,-1/5*z+z2}},1/5*z,tau))*(Gt({{3,-2/5*z+z1}},1/5*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/5*z,tau))+(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/5*z,tau))+(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2}},1/5*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/5*z,tau))+(Gt({{3,-2/5*z+z1},{2,-2/5*z+z2}},1/5*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/5*z,tau))+(Gt({{3,-1/5*z+z1},{2,-1/5*z+z2}},1/5*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/5*z,tau))+(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))*(Gt({{1,-3/5*z+z3},{1,-3/5*z+z4},{1,-3/5*z+z5}},1/5*z,tau))+(Gt({{2,-3/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-2/5*z+z3},{1,-2/5*z+z4},{1,-2/5*z+z5}},1/5*z,tau))+(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))*(Gt({{1,-2/5*z+z3},{1,-2/5*z+z4},{1,-2/5*z+z5}},1/5*z,tau))+(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2}},1/5*z,tau))*(Gt({{1,-2/5*z+z3},{1,-2/5*z+z4},{1,-2/5*z+z5}},1/5*z,tau))+(Gt({{2,-3/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z3},{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))+(Gt({{2,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z3},{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))+(Gt({{2,-2/5*z+z2}},1/5*z,tau))*(Gt({{3,-3/5*z+z1}},1/5*z,tau))*(Gt({{1,-1/5*z+z3},{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))+(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z3},{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))+(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z3},{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))+(Gt({{3,-2/5*z+z1},{2,-2/5*z+z2}},1/5*z,tau))*(Gt({{1,-1/5*z+z3},{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-3/5*z+z2},{1,-3/5*z+z3},{1,-3/5*z+z4}},1/5*z,tau))+(Gt({{1,-2/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-3/5*z+z2},{1,-3/5*z+z3},{1,-3/5*z+z4}},1/5*z,tau))+(Gt({{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-3/5*z+z2},{1,-3/5*z+z3},{1,-3/5*z+z4}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-2/5*z+z2},{1,-2/5*z+z3},{1,-2/5*z+z4}},1/5*z,tau))+(Gt({{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-2/5*z+z2},{1,-2/5*z+z3},{1,-2/5*z+z4}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1}},1/5*z,tau))*(Gt({{2,-2/5*z+z2},{1,-2/5*z+z3},{1,-2/5*z+z4}},1/5*z,tau))+(Gt({{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1}},1/5*z,tau))*(Gt({{2,-2/5*z+z2},{1,-2/5*z+z3},{1,-2/5*z+z4}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-1/5*z+z2},{1,-1/5*z+z3},{1,-1/5*z+z4}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1}},1/5*z,tau))*(Gt({{2,-1/5*z+z2},{1,-1/5*z+z3},{1,-1/5*z+z4}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-2/5*z+z1}},1/5*z,tau))*(Gt({{2,-1/5*z+z2},{1,-1/5*z+z3},{1,-1/5*z+z4}},1/5*z,tau))+(Gt({{1,-3/5*z+z4}},1/5*z,tau))*(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2},{1,-4/5*z+z3}},1/5*z,tau))+(Gt({{1,-2/5*z+z4}},1/5*z,tau))*(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2},{1,-4/5*z+z3}},1/5*z,tau))+(Gt({{1,-1/5*z+z4}},1/5*z,tau))*(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2},{1,-4/5*z+z3}},1/5*z,tau))+(Gt({{1,-3/5*z+z4}},1/5*z,tau))*(Gt({{1,-2/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2},{1,-4/5*z+z3}},1/5*z,tau))+(Gt({{1,-3/5*z+z4}},1/5*z,tau))*(Gt({{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2},{1,-4/5*z+z3}},1/5*z,tau))+(Gt({{1,-2/5*z+z4}},1/5*z,tau))*(Gt({{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2},{1,-4/5*z+z3}},1/5*z,tau))+(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2},{1,-4/5*z+z3}},1/5*z,tau))+(Gt({{1,-3/5*z+z4},{1,-3/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2},{1,-4/5*z+z3}},1/5*z,tau))+(Gt({{1,-2/5*z+z4},{1,-2/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2},{1,-4/5*z+z3}},1/5*z,tau))+(Gt({{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2},{1,-4/5*z+z3}},1/5*z,tau))+(Gt({{1,-2/5*z+z4}},1/5*z,tau))*(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2},{1,-3/5*z+z3}},1/5*z,tau))+(Gt({{1,-1/5*z+z4}},1/5*z,tau))*(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2},{1,-3/5*z+z3}},1/5*z,tau))+(Gt({{1,-2/5*z+z4}},1/5*z,tau))*(Gt({{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2},{1,-3/5*z+z3}},1/5*z,tau))+(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2},{1,-3/5*z+z3}},1/5*z,tau))+(Gt({{1,-2/5*z+z4},{1,-2/5*z+z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2},{1,-3/5*z+z3}},1/5*z,tau))+(Gt({{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2},{1,-3/5*z+z3}},1/5*z,tau))+(Gt({{1,-1/5*z+z4}},1/5*z,tau))*(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-2/5*z+z1},{2,-2/5*z+z2},{1,-2/5*z+z3}},1/5*z,tau))+(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{3,-2/5*z+z1},{2,-2/5*z+z2},{1,-2/5*z+z3}},1/5*z,tau))+(Gt({{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-2/5*z+z1},{2,-2/5*z+z2},{1,-2/5*z+z3}},1/5*z,tau))+(Gt({{1,z4},{1,z5}},1/5*z,tau))*(Gt({{3,-1/5*z+z1},{2,-1/5*z+z2},{1,-1/5*z+z3}},1/5*z,tau))+(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,z2},{1,z3},{1,z4},{1,z5}},1/5*z,tau))+(Gt({{3,-3/5*z+z1}},1/5*z,tau))*(Gt({{2,z2},{1,z3},{1,z4},{1,z5}},1/5*z,tau))+(Gt({{3,-2/5*z+z1}},1/5*z,tau))*(Gt({{2,z2},{1,z3},{1,z4},{1,z5}},1/5*z,tau))+(Gt({{3,-1/5*z+z1}},1/5*z,tau))*(Gt({{2,z2},{1,z3},{1,z4},{1,z5}},1/5*z,tau))+(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-3/5*z+z2},{1,-3/5*z+z3},{1,-3/5*z+z4},{1,-3/5*z+z5}},1/5*z,tau))+(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-2/5*z+z2},{1,-2/5*z+z3},{1,-2/5*z+z4},{1,-2/5*z+z5}},1/5*z,tau))+(Gt({{3,-3/5*z+z1}},1/5*z,tau))*(Gt({{2,-2/5*z+z2},{1,-2/5*z+z3},{1,-2/5*z+z4},{1,-2/5*z+z5}},1/5*z,tau))+(Gt({{3,-4/5*z+z1}},1/5*z,tau))*(Gt({{2,-1/5*z+z2},{1,-1/5*z+z3},{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))+(Gt({{3,-3/5*z+z1}},1/5*z,tau))*(Gt({{2,-1/5*z+z2},{1,-1/5*z+z3},{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))+(Gt({{3,-2/5*z+z1}},1/5*z,tau))*(Gt({{2,-1/5*z+z2},{1,-1/5*z+z3},{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2},{1,-4/5*z+z3},{1,-4/5*z+z4}},1/5*z,tau))+(Gt({{1,-3/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2},{1,-4/5*z+z3},{1,-4/5*z+z4}},1/5*z,tau))+(Gt({{1,-2/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2},{1,-4/5*z+z3},{1,-4/5*z+z4}},1/5*z,tau))+(Gt({{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-4/5*z+z1},{2,-4/5*z+z2},{1,-4/5*z+z3},{1,-4/5*z+z4}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2},{1,-3/5*z+z3},{1,-3/5*z+z4}},1/5*z,tau))+(Gt({{1,-2/5*z+z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2},{1,-3/5*z+z3},{1,-3/5*z+z4}},1/5*z,tau))+(Gt({{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-3/5*z+z1},{2,-3/5*z+z2},{1,-3/5*z+z3},{1,-3/5*z+z4}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-2/5*z+z1},{2,-2/5*z+z2},{1,-2/5*z+z3},{1,-2/5*z+z4}},1/5*z,tau))+(Gt({{1,-1/5*z+z5}},1/5*z,tau))*(Gt({{3,-2/5*z+z1},{2,-2/5*z+z2},{1,-2/5*z+z3},{1,-2/5*z+z4}},1/5*z,tau))+(Gt({{1,z5}},1/5*z,tau))*(Gt({{3,-1/5*z+z1},{2,-1/5*z+z2},{1,-1/5*z+z3},{1,-1/5*z+z4}},1/5*z,tau))+Gt({{3,z1},{2,z2},{1,z3},{1,z4},{1,z5}},1/5*z,tau)+Gt({{3,-4/5*z+z1},{2,-4/5*z+z2},{1,-4/5*z+z3},{1,-4/5*z+z4},{1,-4/5*z+z5}},1/5*z,tau)+Gt({{3,-3/5*z+z1},{2,-3/5*z+z2},{1,-3/5*z+z3},{1,-3/5*z+z4},{1,-3/5*z+z5}},1/5*z,tau)+Gt({{3,-2/5*z+z1},{2,-2/5*z+z2},{1,-2/5*z+z3},{1,-2/5*z+z4},{1,-2/5*z+z5}},1/5*z,tau)+Gt({{3,-1/5*z+z1},{2,-1/5*z+z2},{1,-1/5*z+z3},{1,-1/5*z+z4},{1,-1/5*z+z5}},1/5*z,tau)");
	test(42, "Gt({{3,z1},{2,z2},{1,z3},{1,z4},{1,z5}},z,tau)", 6, "(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))+(Gt({{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))+(Gt({{1,-1/6*z+z3}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))+(Gt({{1,-1/6*z+z3}},1/6*z,tau))*(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))+(Gt({{1,-1/6*z+z3}},1/6*z,tau))*(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))+(Gt({{1,-1/6*z+z3}},1/6*z,tau))*(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))+(Gt({{1,-1/6*z+z3}},1/6*z,tau))*(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))+(Gt({{1,-1/6*z+z3}},1/6*z,tau))*(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/2*z+z4},{1,-1/2*z+z5}},1/6*z,tau))*(Gt({{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3}},1/6*z,tau))+(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3}},1/6*z,tau))+(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3}},1/6*z,tau))+(Gt({{3,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3}},1/6*z,tau))+(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3}},1/6*z,tau))+(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3}},1/6*z,tau))+(Gt({{3,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3}},1/6*z,tau))+(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{2,-1/6*z+z2},{1,-1/6*z+z3}},1/6*z,tau))+(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{2,-1/6*z+z2},{1,-1/6*z+z3}},1/6*z,tau))+(Gt({{3,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{2,-1/6*z+z2},{1,-1/6*z+z3}},1/6*z,tau))+(Gt({{3,-1/3*z+z1}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{2,-1/6*z+z2},{1,-1/6*z+z3}},1/6*z,tau))+(Gt({{1,-2/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-2/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-2/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-2/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-2/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-2/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{1,-2/3*z+z3},{1,-2/3*z+z4}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z5}},1/6*z,tau))*(Gt({{1,-2/3*z+z3},{1,-2/3*z+z4}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{1,-2/3*z+z3},{1,-2/3*z+z4}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{1,-2/3*z+z3},{1,-2/3*z+z4}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-2/3*z+z3}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z3}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-2/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/2*z+z4},{1,-1/2*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-2/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-2/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z3}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/2*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z3}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,-1/3*z+z3}},1/6*z,tau))*(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{3,-1/3*z+z1},{2,-1/3*z+z2}},1/6*z,tau))+(Gt({{1,-1/6*z+z3}},1/6*z,tau))*(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{3,-1/3*z+z1},{2,-1/3*z+z2}},1/6*z,tau))+(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{2,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{2,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{2,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{2,-1/6*z+z2}},1/6*z,tau))*(Gt({{3,-1/3*z+z1}},1/6*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{3,-1/3*z+z1},{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{3,-1/6*z+z1},{2,-1/6*z+z2}},1/6*z,tau))*(Gt({{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))*(Gt({{1,-2/3*z+z3},{1,-2/3*z+z4},{1,-2/3*z+z5}},1/6*z,tau))+(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/2*z+z3},{1,-1/2*z+z4},{1,-1/2*z+z5}},1/6*z,tau))+(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))*(Gt({{1,-1/2*z+z3},{1,-1/2*z+z4},{1,-1/2*z+z5}},1/6*z,tau))+(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/2*z+z3},{1,-1/2*z+z4},{1,-1/2*z+z5}},1/6*z,tau))+(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))+(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))+(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))+(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))+(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))+(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{1,-1/3*z+z3},{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))+(Gt({{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{3,-1/2*z+z1}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{3,-1/3*z+z1},{2,-1/3*z+z2}},1/6*z,tau))*(Gt({{1,-1/6*z+z3},{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-2/3*z+z2},{1,-2/3*z+z3},{1,-2/3*z+z4}},1/6*z,tau))+(Gt({{1,-1/2*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-2/3*z+z2},{1,-2/3*z+z3},{1,-2/3*z+z4}},1/6*z,tau))+(Gt({{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-2/3*z+z2},{1,-2/3*z+z3},{1,-2/3*z+z4}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-2/3*z+z2},{1,-2/3*z+z3},{1,-2/3*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))+(Gt({{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))+(Gt({{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z2},{1,-1/6*z+z3},{1,-1/6*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z2},{1,-1/6*z+z3},{1,-1/6*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z2},{1,-1/6*z+z3},{1,-1/6*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-1/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z2},{1,-1/6*z+z3},{1,-1/6*z+z4}},1/6*z,tau))+(Gt({{1,-2/3*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3}},1/6*z,tau))+(Gt({{1,-2/3*z+z4}},1/6*z,tau))*(Gt({{1,-1/2*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3}},1/6*z,tau))+(Gt({{1,-2/3*z+z4}},1/6*z,tau))*(Gt({{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3}},1/6*z,tau))+(Gt({{1,-2/3*z+z4}},1/6*z,tau))*(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3}},1/6*z,tau))+(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3}},1/6*z,tau))+(Gt({{1,-2/3*z+z4},{1,-2/3*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z4},{1,-1/2*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z4}},1/6*z,tau))*(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/2*z+z4},{1,-1/2*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z4}},1/6*z,tau))*(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z4}},1/6*z,tau))*(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-1/3*z+z1},{2,-1/3*z+z2},{1,-1/3*z+z3}},1/6*z,tau))+(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{3,-1/3*z+z1},{2,-1/3*z+z2},{1,-1/3*z+z3}},1/6*z,tau))+(Gt({{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-1/3*z+z1},{2,-1/3*z+z2},{1,-1/3*z+z3}},1/6*z,tau))+(Gt({{1,z4},{1,z5}},1/6*z,tau))*(Gt({{3,-1/6*z+z1},{2,-1/6*z+z2},{1,-1/6*z+z3}},1/6*z,tau))+(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,z2},{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,z2},{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{3,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,z2},{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{3,-1/3*z+z1}},1/6*z,tau))*(Gt({{2,z2},{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{3,-1/6*z+z1}},1/6*z,tau))*(Gt({{2,z2},{1,z3},{1,z4},{1,z5}},1/6*z,tau))+(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-2/3*z+z2},{1,-2/3*z+z3},{1,-2/3*z+z4},{1,-2/3*z+z5}},1/6*z,tau))+(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4},{1,-1/2*z+z5}},1/6*z,tau))+(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4},{1,-1/2*z+z5}},1/6*z,tau))+(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))+(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))+(Gt({{3,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau))+(Gt({{3,-5/6*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z2},{1,-1/6*z+z3},{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{3,-2/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z2},{1,-1/6*z+z3},{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{3,-1/2*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z2},{1,-1/6*z+z3},{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{3,-1/3*z+z1}},1/6*z,tau))*(Gt({{2,-1/6*z+z2},{1,-1/6*z+z3},{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3},{1,-5/6*z+z4}},1/6*z,tau))+(Gt({{1,-2/3*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3},{1,-5/6*z+z4}},1/6*z,tau))+(Gt({{1,-1/2*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3},{1,-5/6*z+z4}},1/6*z,tau))+(Gt({{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3},{1,-5/6*z+z4}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3},{1,-5/6*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3},{1,-2/3*z+z4}},1/6*z,tau))+(Gt({{1,-1/2*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3},{1,-2/3*z+z4}},1/6*z,tau))+(Gt({{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3},{1,-2/3*z+z4}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3},{1,-2/3*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))+(Gt({{1,-1/3*z+z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-1/3*z+z1},{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))+(Gt({{1,-1/6*z+z5}},1/6*z,tau))*(Gt({{3,-1/3*z+z1},{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4}},1/6*z,tau))+(Gt({{1,z5}},1/6*z,tau))*(Gt({{3,-1/6*z+z1},{2,-1/6*z+z2},{1,-1/6*z+z3},{1,-1/6*z+z4}},1/6*z,tau))+Gt({{3,z1},{2,z2},{1,z3},{1,z4},{1,z5}},1/6*z,tau)+Gt({{3,-5/6*z+z1},{2,-5/6*z+z2},{1,-5/6*z+z3},{1,-5/6*z+z4},{1,-5/6*z+z5}},1/6*z,tau)+Gt({{3,-2/3*z+z1},{2,-2/3*z+z2},{1,-2/3*z+z3},{1,-2/3*z+z4},{1,-2/3*z+z5}},1/6*z,tau)+Gt({{3,-1/2*z+z1},{2,-1/2*z+z2},{1,-1/2*z+z3},{1,-1/2*z+z4},{1,-1/2*z+z5}},1/6*z,tau)+Gt({{3,-1/3*z+z1},{2,-1/3*z+z2},{1,-1/3*z+z3},{1,-1/3*z+z4},{1,-1/3*z+z5}},1/6*z,tau)+Gt({{3,-1/6*z+z1},{2,-1/6*z+z2},{1,-1/6*z+z3},{1,-1/6*z+z4},{1,-1/6*z+z5}},1/6*z,tau)");

	return failed;
}

static unsigned exam_qExpand()
{
	unsigned failed = 0;

#define LIST_SYMBOLS(f) \
	f(z1) f(z2) f(z3) f(z4) f(z5) f(z6) f(z7) \
	f(zz1) f(zz2) f(zz3) f(zz4) \
	f(tau1) f(tau2) f(tau3)

	DEFINE_SYMBOLS
	MAKE_PARSER

#undef LIST_SYMBOLS

	Digits = 400; // read these points at high precision
	const ex points = lst{
		z1==numeric("-0.02238648627323588003480882101869990312923420537797390016956947182521353995427-I*0.29064311138445944682217677721965136774656665592055490050508769699393681706083"),
		z2==numeric( "0.06157077181450390866154139476807999487224413032308816760045566223281601862226-I*0.03937963491415113654948346064629243254576707392867715591558093295627818034691"),
		z3==numeric( "0.05669011210979269839031883772631178627238866387091732043126570445132878868741-I*0.46050508957741806608024612622915659188090989616132640437091017662868881805341"),
		z4==numeric(1,6) + I/3,
		z5==numeric(-1,3) + I/4,
		z6==numeric(1,5)*I,
		z7==numeric(1,4),
		zz1==numeric("0.39") + I/3,
		zz2==numeric(-5,11) - I/6,
		zz3==numeric("-0.3721*I"),
		zz4==numeric(1,3),
		tau1==numeric("0.4") + I,
		tau2==numeric(-1,3) + I,
		tau3==I
	};

	auto test = [&](const size_t id, const char* input_raw, const char* expected_result_str){
		const Gt input = ex_to<Gt>(parse(input_raw));
		const numeric expected_result{expected_result_str};
		const numeric output{input.qExpand(&points)};
		const numeric diff = expected_result-output;

		const numeric digits_real = diff.real().is_zero() ? -(long)Digits : log(abs(diff.real()))/log(numeric(10));
		const numeric digits_imag = diff.imag().is_zero() ? -(long)Digits : log(abs(diff.imag()))/log(numeric(10));
		if (digits_real <= -Digits && digits_imag <= -Digits)
			std::printf("  - Test #%lu succeeded\n", id);
		else {
			std::cout << "Test #"<<id<<" failed" << endl;
			std::cout << "  Input:           " << input << endl;
			std::cout << "  Expected result: " << expected_result << endl;
			std::cout << "  Actual result:   " << output << endl;
			std::cout << "  Digits real:     " << digits_real.to_double() << endl;
			std::cout << "  Digits imag:     " << digits_imag.to_double() << endl;
			failed++;
		}
	};

	// The following values were calculated up to order 400
	// They are correct to 200-400+ digits
	Digits = 200;
	test(1,"Gt({{1,z1}},zz1,tau1)", "1.1303022841632950654425418625848975978754556916808189024249719006528120436662084918440101950661332643470674013837079360590065580188869375482303449801140181332367631262507554100623001438619003854501447477256915648984418672502985367366280626491176212942647127662596447352442982388772301710311429434467161078302514327501012894008904454977601717422466406290464406674580363381208775481520961041557935862566033992496517024659998965-1.1924529317523669949418848236459127842756254317136286149535374794102094711611598288212674735928414413744885490636953757607754990233012960777921500915335220505714168170964879318566274731609630106359587679990932779073902800840910107944017171938823417830338513799402615868178810115374490559389665666497592586540269033441316605965948004539663423563600165788150947672049814851158889956048418487686885657165218764654413500888931999*I");
	test(2,"Gt({{1,z1}},zz1,tau2)", "1.236780742352457104733660962497629379215346952515143342837068411049958358707743698007723552891770397083703487807204629463640241911920132485737611701034245613397027114856205574607073845377246916935552207001613547671792976652908954215877998506451921163017687083693266470854603818203531365312747198255553429027600336749887476942601363479957183419174646111433737261930327121778227477458386129799308291507447900208843343885867916-1.3114544613177689737083253238104592473749146552847339523732275773870771708941019702399287827587081970406244517817949453494793227214231890849823814238083896579376654595110640386922600706789929599172602679267856155308265798555330548285288999583104031842793651737194611954361594350604102223817220432190471506663537051716018514726018908542413186219753337208073595453641942643972816292373383888615918823652391582060861718713674771*I");
	test(3,"Gt({{1,z1}},zz1,tau3)", "1.3276245096884728126988751594425972005986496166372718993217000959990712204165452926397715489846309095092435996303749010831501301084981155737593726038791103821968228234383513293590011684953372142439432495426348306134761495175221839928648171713010451218468100499054367148566027892964155986258188015571554393411461238287891540970648969981699529679950317216613768451201501960910723667822392701602524762100883185451470743458750881-1.1645322736150581493917619698461439307613717338545612127527438152118151834931778587373944850160396697599278898966223790657757788405158729897597683743695603846861257377160903796331733436149551480085227210375062263816943551751930448227496424108558780626802512137646259047787093610298750756950092661035063335498367364417995353137323593098294575713774049367656949832738390963168826166564829567152287104176271299610069210474550259*I");
	test(4,"Gt({{1,z1}},zz2,tau1)", "-0.0037153641470299274624716595797685446440909068907472560956089512031942556213672264455732209685416879496715978915553496266765954747052434770312254043554408218208732015065601751245448927011424197379665512352618954748775986078989614478392431757211715149428835466613185171826389399294585331384215282003058883823179414640701391725079029840677916891417179393063034600889582119297110545348404931628862058314104307573024138814930858+1.599703354600037510728474406233912441043111081270069350199452070956298165597070195700060438852255206873314239354219164254645882963195478549155213789393451020411812658270099525112935231098785503370949374308409692895562820886443770964156741779025864538759604135720722071210682437938477807626932479357015928188049339379941748620489040954670727614707369323824759448099684474806257455771293816711273395112663147474494929452368032*I");
	test(5,"Gt({{1,z1}},zz2,tau2)", "-0.0029616161978208463788925191071313423849116080254687790355604664018649277521502070613134934488891741847456576130198453244835293351283653305330111053751500879886910924581298653647192761567840905287830963556024430366430798975460013298822245239566891664192545361619447406542515092714138417680743128322057666287121229797921340549512731769667312768225877965881793276491952272029975960612416042204009648879492034578995351234127743+1.5753324909946553159162249861178012795779966984263733134020500214066654505724266098848266152531399477489416058659599431997195196330281714272472478119741973197521828240587269601058946631377908180352275703651206544980257465590292687765116627820816235248548694810730226460576234055276894331800719235380441676294829231361200476519268613101020242445760025317125903294932722094193490921968983530861229047895743820225879202563625701*I");
	test(6,"Gt({{1,z1}},zz2,tau3)", "0.0240771506330800579466746664607535480806874405410988354751623072526490169597278497128306171081165975299106333972939724494192265874291920751673256416379465719961326890129762074550610308663678250873512357747026056303984183865113934937547027934790366705927389353169871055382741869005373767256467531719373702563107587677748143254373042387761330651032170404588078072883117986078369159286088311030168297642970728350527400166912424+1.5849465067170405115639100183934909824070047412885657545548307354995657491613510702626796237672680045877443026403476474236865994164765590513663116638676878325058620599060627846141753332174463404676998795875788543604529054265854252830732697696047610141436480577787961900818474334394070266294315253327243856713024482801448193318621775439772237828777489011844437795200600688203222269187761305288445543326405925131329010168792859*I");
	test(7,"Gt({{1,z1}},zz3,tau1)", "-1.3681188840820981028020660763975696740821641424265890547625112892921978868270140308634364933612723717235478416873730980289320112567020777363750806139334909282440104279811221576562407386392700337409493070754002401241165190458310258759370077285570184179830414309070564039119493347724228466866501978434932855850436462687473729078998079662715832512339766348212018783687986529972149317956967954192880977479873070699727135807209078-2.7643419409211860826037401425039453907767489248336467764196755851656028284118962687152070060782481729347102927514684486931197215946826108365250560120584754727746536425530906692164559352158039254038785738608268292198378547343629512666581921262687757058828248931993723838700047590129938844389072214559757025515490422276911265586749349367991937638984769963376156070859883666392767969907133715270125167561453278134826074387497693*I");
	test(8,"Gt({{1,z1}},zz3,tau2)", "-1.3685005586038946702262492073486593282825271990623507549500548849771545533153760447491694843344699291660812372828605705677286151722097080877556858203389581058332025184902864176261022992885895861813453086524789839199671300562444224676675715244629093814920295989029594143899412279087910802200302833935349068605218274761709341093869438172384499924269195014129870810198652945099182977742350117231355551158332418170341204461865963-2.7757960788288796493402174020721040867999714143949577935390191923347693315870100726936653117605473920987442913137317819103943762797358824819634821689231254011624413598071832396897045887426994828315307101241268986507417549749938093094251849836146428199958484955686059539423813620839020207821679304616347055790397335866924446023849955382421294219687200699895782299203721871391289281254757857811418039941110915147501150457909016*I");
	test(9,"Gt({{1,z1}},zz3,tau3)", "-1.3554159685399464617117326652587114029916708117983274458826714195977855904631873090099239029652789581314685360252125308461880924922034183616752802715449959938656330578628638926942789458149390918769193909754549197839686574327759501877193960709366936038745138307613556560615110443974259235264535101872210800347927979053891296069555479088100436361102476062636373912793565527894830276934220500108835806932936019860710557370141735-2.772119405999697401581881355719308704002181403326097082918677880895044096769692972657264533364314480729114260478253602235624574055858080699752677482885555003793436476676854252019376695925777387348311417650751994838241152484856488083903120558058116798655998666960890745885508126450574322424078289753574934087452604135887357205062023388862392969285248200399396544569809736487122059124044873558928180017904946258428121043020022*I");
	test(10,"Gt({{1,z1}},zz4,tau1)", "0.2547890861559010099048516139414818542008679227136632082500810001053961414590774886342631404944018116944053218536083648292081347782663252229749460403716135872377263661824527503162437558034649450451688965013333200888939289667573989108516682983402330045176574219105832378087870605154198887925591471784653080302346350782699240663781427254994878204308541260638248678208182348351413994659670513108115145205141906603832198632654607-1.129991296062962933570345117468150577088831315800344303947144691519333433049142124619953852482264993113692762601536035492127985468091154261683655034742861363016724743584447964176449104116304610218575740930466559018491820424177601892972811491632526023391380223975783762670672851727505646906993450409291159833566053350634627111743322692537171974787037346289311750341357607500661484157002326785486803907446634001982910247527674*I");
	test(11,"Gt({{1,z1}},zz4,tau2)", "0.2713575809052185446415637268635386735855732543728075348585412108581904467873267174492252660002971186468520187814466822410568940436238715754764443329668791704994963603957984626824915615360970297439630662714994725232088192625854325773852072992703367089547560423868441960150446579592507109493867719103305685949034217185174312936545132067282858406042423341152513284489688927275751391846106167547867807132467429789639463901771488-1.1552527952394364274656834722046097121353714851338564897157348237391793249709655272003955749669731057276277942363962082814498637410632512152496752268897693080015685364498342352623472954268736552911294347274788674991124361101761532828778523872392599533813176099750707267351849621627030407772166947110951709630223594026400573999539911890647268405355406224073406325985820677558952224945547741499020287401736721636804300398928667*I");
	test(12,"Gt({{1,z1}},zz4,tau3)", "0.2938122291866459805946748378060597759170509304284799064353790163145446590445491510135169330468293934109620562577451691093190802909151479701139829943924934788656209390392412254414748157447990700653586211261337738684834551163422374678283842019193675823799435593197186666266603586137403921582804033828715591294222569659438626320655414800751279669522729772771735666851385914876662212114190889164420149729112000271119899853219694-1.1279393158884975664385388989017168993610633566003065302627351208034683208332792257642958449169001184717248421603623653494635206587636993607263249685334028598980339007468385753717728181976187686443305816146311991329138662313715559897233143624421978583714527391610042352754133504694841282499526059410025537820418699883697564374117528963462552762195846410873961704181374155568005896604792602290460039522537817981188731263526902*I");
	test(13,"Gt({{1,z2}},zz1,tau1)", "1.9799341002260572181935297430296333496905751021414835931754627875776528859543254938022138618583710092868747095047690369828156809868966621231270500245078292613711178134096222970089396734716411906692391622382095387653464793859252223075519487284660669866536812430989245284210089214776552128158777835079215000454670255878681536198714863432908340544913993821758383050275729279061981222779710321413251851471834282432456717398381877-2.1290265281179728931008162298279644296992319889881792924531684749574307346923944914119687237856095757446153168368609598640383664240948242105096648712508896616564630487337067584015511930379952258753213367608733914856511570207664024535943699139113239723579536778733620955222580485204314785730707146335253735905751935429205806022060989452005820027807346215366520334809114787647415213364241359699957762130901940571086215291423476*I");
	test(14,"Gt({{1,z2}},zz1,tau2)", "2.0091764099635221664332083207900892605819369035689570845079039444731422632961962121211873223993386010510531584895835924609928374587075913815301933819772797624643853934624097706802345506369882689793770928349548220635932143982232574647054739854891359277839394491958963992897652688817073639034524906837930044791878940442766491284293480794230656161599017177410293556066605122662239744310081713061668914197786441251262946898510898-2.1420033149891567051793208488681599482306341092455397283240867963962660303029399214151946789971039388707600903836170517637980257995926489626931829801931531000493318363508369828536647603990210826922427383470734513995804015034291111056090350044996044817394675620906514555175263555597439693367209435921026757245357135715182064032936193168355327702019057134640690575143625710837467033545748987317132333686264670877516164963564515*I");
	test(15,"Gt({{1,z2}},zz1,tau3)", "2.0135367293458561657746332438801311909191686042682671120440585073973322974982416989241201872402092149693078073725558813923568241929456297342130902974183325550968584041754780904080050628713300459275047757952298290052747830404326641928667614193802886822149204528802085522846300450820451421960432093762718167414808405744198511274676458272320313670213964861641775120817381220296060205683313825313921797802956592078336214929302958-2.1052263620907415951896703634822389959601945342014423596096767897575874086223578825678735673279387795763121884192018389574520469000927963581533171532809775423037512095423254896153417026539446775021084251949551441926344956925221373748801748835319779590219250081606910368166593222476538911787796499953820389075510598173368644292964396221320762224683408634374564140733570434748545136458850151334862792914261141191147893344093198*I");
	test(16,"Gt({{1,z2}},zz2,tau1)", "1.5449532598027617749473331769349923232221592976960336848051487521492424628576553633377231630730867863739925960140628030988626317043224469085999515173161243680786062323712253886360465215832801043435894040811261799259620664560823088675330448419077161033607252103719669103381547066560364889419481583276454616511970683164802715917290059847285189397268224643882811890820664228114498989198866675016491981378989021330419401467015671+0.5467768674832082901692757162232327090385271272956245980294246376329554739791024207858319307487615950524524569576366630904126620502938937778730648530825026862397007093426213044640624018305210997459709441323304735200007191163291679669136394384933462297641928391078193656737189995141779724244742935033167759731313796181420065976836784454945590964446240408930668960339863194847105286274698525604036107218571614621975929539352336*I");
	test(17,"Gt({{1,z2}},zz2,tau2)", "1.5476018309495026305925994312210714758648198003405596140713898396317542134160610125145952559390927231057788096302093546054651294476555557632632002292726115682670385274235772499766969813469640924544413935159877939783095752134672188135578098660169189970936684278525784967450030087643734931132644542480397698358099165755801951601303673160523482270759208162522929010341004024319412952546399547620033546311037242554091861442450921+0.5343907831549109270744743411652045835338552357968272468259303123730796179633424203113055370116620512275781568188242708964161732069202103444844753401647883586153354084517479315017365730122658163444923093361760275516382038492108915975188439073817753119445437996321264558848342910907561986400176590033810257819461256860277239402356253336150964317431862735911680842360241859855936939215536201504880592251738986077700330172415186*I");
	test(18,"Gt({{1,z2}},zz2,tau3)", "1.5604057045538252078104025214468163338710984062748050056873538363259066106128167740526627501833137554860848640088269400778656430595669385666686692460461968778407761675541898064003281587833007973972725717072227398118498371773754921604207285095155304270144337027127285763545968579627157142734624169367956482437800471101573137444390765063609853332455169739600167284609639857883552528483080521310470182242674467366954787190354347+0.5418029026172179805850922291687352211385224283842418175629840835394259922906941526325310652576832587407435568773159234284174086115695599234399409564917635520667357928274607527688158916262454981894409834468317417714274877755420963343355187356836811978148950644700547376537359919113798187063198009744527849519447629051081905062143700588647322554246568750714311240270732096392927100831296263227514669900340894187291012369427863*I");
	test(19,"Gt({{1,z2}},zz3,tau1)", "1.7121308872031838960322173538661915308423463082457568216354579925680032494972706621013678112579420535746356993418574687611404817468471448430564459264609785567662443660218506277235775904545978083953595042344574032526335861149839585464724797949069300392325345557803306484283854423761891590347585142469162853520786570555999678259870200484369752467997578992445732955403883971769839181087612094978777161458143690153557506377703681+1.8748178845843265649596583888723498286286143561410702878616032775098610573674541816224810200879790187019342201768498045021287944037964139888193379832076086398347688894387613365470194592784700855522440052262174412453729611533417161426661063525953346340196685571296050594194206111273732085774016682689888195334099889606399323502321658488577502545397730984217311519414147065660121324240205937509330147261802514224911509652335804*I");
	test(20,"Gt({{1,z2}},zz3,tau2)", "1.7174278245797235953777364065568922370895185296136585519224293387461821994170853000813480415247097974774976045022433489367426402262764904254492468196807628562726961326956629112810930393289732846911423032456101136654124248118059036588353928868716053026831843168039049712864758344992928498472958752790182252440421358818182910225847826237629354057053636209664523831168954063546072980095688753657031451176414029864973733944786343+1.8918746528251072693644451136092025029210459407956640340681112437755690716689339602090635086896002467476336188473749089506543150853876003131412711265021490689261298649259766775352103178749933784982452423166700105762820610057019202759460110082364598348801025460484381572375347093946295116258497020873955810835154124816456772294498985818512441470596862982361161644955614009235724584900383021003846437778678476730078570396083981*I");
	test(21,"Gt({{1,z2}},zz3,tau3)", "1.6962364747596840426462078639749811175195479109682477486789827880460663724339944938755999077200053271757047756014336554002589261256810313534532268858797067796885932591490928921746929323755032145878688414747048356460512568433818826430042805814690386555406931154341555853436507045799020383716551715716957718277601449021105403512120824913691523862111086196048190448542050935182904428159641582809521504735456410597660658076781336+1.8918318757533884005494246064481114286714542244972758322093708393842874248888345861655614994670063541483355469793496889735578673819865153597911975720354380064285012544073292994136486924667986536695041942767162471897217545811353486019275282375059352794411164881618679958135183458741480479016849450084780364360205683616448435679554296825620186936365226773292643911865063426479349228969591082851439984614046837442061171158399946*I");
	test(22,"Gt({{1,z2}},zz4,tau1)", "1.2016757778776063943405781061924859373059154353439914855796904536403790566971342252735680378902904942869820384582788580459573627813630246146050807472489261836703599437112811840483780352950466562895887570433343806510878175569779109053101962141305556744805044323202151574994381222992334629130988310763385818009102289033502587355576254948187587939354863062242158556140226575845988868888701165659211417657791799697205395138226277-2.4723036296507184203578011284937667836613471331438205510446848687800445745888987097226469305543956134322198173559507100296152808308831690608587587801311804901070960362254102010202349801549368737411952359144969308605557983984759361163277234093201570813647153749518197453271305270421270649903362766098530618904005136642570610936199418916098106147746586842661636977657794503732785797751986765505866401200274866636518017184400514*I");
	test(23,"Gt({{1,z2}},zz4,tau2)", "1.2047790976767878042331604686137667971485449855233921990611755559597862270915673300135902029367664641953704135241975959471150911745409024306384710241356221773651028808307495221947844354879408911799329953376472385355985466734303961951059481912991932682169352948540974918246391489576221898815883183132873058814117551452492346325222362008673055339144838555834580999973190529726207504511503742129119891310738985506701820320376751-2.4778318897020277847745535998976077166017551732101996057515978558503348083335096229630745764312649929503511414287456279785285487812212980689680142780065032489321358082554834443955148938738696918211155441114134476117436359955119608700029942205494519321647442899138935284831761211661806103970503915818077986378047891695339380260201170212952426489761204014351601681672315195544879703736035657420343360717544686335896639205077729*I");
	test(24,"Gt({{1,z2}},zz4,tau3)", "1.2098322696531437404935389297994173751710086341691146442289303330760482090878718684336867381250951546964727072629016048926944440855683507036939122107514752898042555618308330911939452596571232928416102788067976996313033168814616604082427210710115048907337819682353052287568993041193643057075575707951285311733172848143800887609464674694389631551196390496251795869466742976459676798100380364178480013039195846526684897412960375-2.4723817395227544991614024926022069939636948590035354530701762378609861877200658541669270136564456731954299129919873521127305512561580531959906972469291057567733762713688343796568570446785899087723621860572280560804614402175503746615665123521556451514595439003022764520528400462954917819450449202507062727096493922770569191131978861994131529439095578796756557034047889050340349097492647513825484089422783410242159195133232739*I");
	test(25,"Gt({{1,z3}},zz1,tau1)", "0.7952804176214952343136079065810694792691498260520423738535708894659143732758219328880420264790428396864003456876943715172877250600815239574732334351417272555320535855925846882465502069962348918313439780091087402619058875322651938018028344726275387374309150193548460802967340735863242345481819554464104549291584327927214368211465083623049332508302080823834143344607710366810562733037725519395232955752626237933978653673008065-1.3902350781494143910927238301003497673883987628297780469678611285773118384569801174623423933243291179697461549517619680550075113099105886641402038456892858963391115939700459801979526239919645616778754245156552897687014379276993188228127529271791613440077703480045647284222208604920717428335941790478188556298869365895371412847090075639592677738841048869431666362567559671910388230002017998399583949692181549415151648896622767*I");
	test(26,"Gt({{1,z3}},zz1,tau2)", "1.2465373499883309286457305492273190018475250612994660429786538441044850184666045410659906553254411857771046270924063683589686980873219673121544318973596811056969403075436126250898307919924416579122750745624261100303209944882936406185378791646269651385754081930492944728258521798625655750607255898099110598458615054497122658724723354158977234946161076302826023619823288186043260865638288336259933604005610359432912804062038857-1.4901446071790973985714996000837358067176171242054006370032270007080793715201266838951720066709516241943989532920226595764819927711204032465147124972901121156371454930086524487625121325739015992214374397896164571532044988654886922985268163427678604601709067059453031766036573618748084195416764246961411843051995861048463496533731742757836816031853203630712100986593162351404263497565041108996863659867099603129164078855262801*I");
	test(27,"Gt({{1,z3}},zz1,tau3)", "1.2857418864240725578263874027940310887837512042377844047879244761465655592170632634091963882870741955286625553566538846334275930598850475539224630809834608983509883447438814477319520488267596001436035373618982109095047210205410567110064225112115041351292381724462199554087910679233142166848246044944880780944604558886984965441471668023447488624451408574637631103652419499041375175414698115038070492826777034996561968033971594-1.0332754747327466273340665673389993162321418243293385090441718141775598628628865459848855301749473337082961957975711310718453777328484240947469826763839650966839094511330155519409450522462517147399916686430727543554803152738159689263063485464199319197575790929820218743255508596321761877711469865467575118900581000560532037066485000099753882099196331835742374439606823367986198973805887614440055291568312148617782681535853054*I");
	test(28,"Gt({{1,z3}},zz2,tau1)", "-0.3665518000469988285446974247284358442012418073956828421565093405377355259371168481703508254541287000270357278332140435705665981400477734619572111981593129956663389627217627969955735690435906364836102283053162148370570166406388912544093699164288517398764966140753826008296576810111434338443168852218672028469548219097420311226688322231186143370850336938144281067168262545773314261260159884909100803392942946654347474062771976+1.4133186397651014397209241686429240821881336600651295152077989347781594338168900810130868683796877869404074892064319183019540069251104650963452355346111072338271242627335360725733890276339270445255636472173943705745610787130003785538211229648479159802525215422468282825754843715437647211877787963540626470830544982352860661341640357360871682724643632121313799851447234902047918992979395582543316231460235178987423157387292632*I");
	test(29,"Gt({{1,z3}},zz2,tau2)", "-0.3358369372430832621720633139686882756550302814513275007322783106486329587700609582804221375198937037810882125561345172233386964993597983461918767462875807375866170386532269073667033130760525535808697558960951615050173932114689363051337663884196472273639997157064467856820139340966664471671267811411983008702496462525613458933916862913403899990301815900599077217278251408923796521932200215752139860035190316293702867399863641+1.3540177415543076081943415699559299097951944148695333182169285012930717328712972785687647399371315545868237160525926073354311241362500645511892488543665749444671482891807808159626193193685637061940328972882511875482863828787363197094774683995040394168500799211927058105300734129393937755833842550293484143463724445958540544730088041477382516131610464474124461814160387327872418897804229198504492268807521111473858311349064286*I");
	test(30,"Gt({{1,z3}},zz2,tau3)", "-0.2800040635628151860573881055927274569540578571436827171694193292502706329512050770466059767094959784869243630989577336340091353248164917580051340265368452814755633386380826130370020552915154903032622213950225508115085551839466352053831655346845744522571518984951488905608833326077045625276001379411385396995279683489032246287961457741562712876011930570308164095361387366836860422723848230661660605768935363488535626393300937+1.4109425142855728161339687646973255437342734626583319089220331854399553914007040921828506040234295383388802933895310561360953511576195660283723645202325495318127653687241471115078822175768131174781869708049167131609414048162088436792908800817954135029254631512094108718316378207680748522826618032727119174010480271338606614329395093370124984594534791088277128115350816664439860965166739915504755954458485406499724522673813426*I");
	test(31,"Gt({{1,z3}},zz3,tau1)", "-1.8300336566401689093073431715152544151385004106953878145071123401318388152803112264928237675592645115803426177992752612854681416790623375833071597945952538360431729499791274094070762781629260118572763346708914614304095408464201875944679768783509695839975157279346847658275817841170711406224147079873671805976026542626523519172313911606613166687485396358112798958702266999178169560451116705693639226209941806794079283313898605+0.3952312850483137759279399598575176903128339041089023346213849378799393246348955575610607043771719679021225776314666148331739803200468313891919840703439955013517859126775684153852813665002836789884824639237826463515612245131568638762371697441525280455387948225754736847616103057974164330964169950568548124828513108149599865277027343275688998841632180614738475556772752030518291933896964824348442845605545748890158120580618025*I");
	test(32,"Gt({{1,z3}},zz3,tau2)", "-1.806635102730654549071207853002936420327645919490531871622232348156668140165323076364525981532327510223936092828859438139706122971765583746530050725242535787641871155845325386789817455168044639778866213747209309529321597582958983715312172174345367210227724200368519534023637929972180223687383144840782881293349300399691930117436152532682989057537967236422682271664074423456131849026837553272592597073782257155294321159551331+0.3589847970285426446002271133112226328199901075999940974271844717200576962062220708889289796949428412005941826976349878891220885445021252077295918185470886959635842385172174804716631678788453820044117320238586340958314462079423576029706753250421184563976156616629133660835199356523605193971069623244464471037200965900689443892391022605298259985288909396340581257843601236833785408123643013218077819328323821851317285355492705*I");
	test(33,"Gt({{1,z3}},zz3,tau3)", "-1.7736094051392795093993835413130452249959526255455850387271709508457149243972226185660879183959868438544585796096561234535492522956156251607361165186483364079582519645494721918175414777853064410913435132407031279149244612347850363109102195456729098812350769729663985279599288094193621369711667265075848141548345345257043429450227742196647444140842898851341588902867824760467030434859622389644397532262639007791381209408368377+0.3995354554470683882388762034466999585635785582852974449265702124503644896386344370453152823766115144799086702218725211123138602816571723715053626031583748303504719072077733918932754061076047206502033465734927851816623496601895154195862952199029627490524430243233358639458255856016283079912171207110121748326390049110769674300403675648150231049518098553438644234286966326425512713960725075933759646146820551244102545084656805*I");
	test(34,"Gt({{1,z3}},zz4,tau1)", "0.0077472235107919320155553855704400188338572182790984433434102341685339108455183191408204027363341797923999095530733640541417188554910371215520777886090359763794441265924112814086529397508927935241655660370601287543158601314236350989226672917515453711259794650295271629464661155446395139925983108359348981410560377579231921712678557344982890401669837803771539292681670643250403141544907184410988266030440709872898599136755141-1.1368808746031600084083043428933440023716771074603595941658788937198558350449233082020041307096939702930350905156241894665440167095896820753157671185134219638532518759903373309204636209393437421698440148502133602908770714164259773686160056056927424391159879147461884521222941883863030438885200098980100213790696268322269987170162527127595750751617413226106639440538732519564464797087556686627053607279982856021387990536099218*I");
	test(35,"Gt({{1,z3}},zz4,tau2)", "0.0831830720572056086428019928040024620186666539290187896158910781315358798471537858687907581823879847359168994714498293982051499478764649560896515945675692190894945019950545984705852677298078721574509743299842261853271063204207318542400412042735267064578459776669833242166342163297294192047001654158511780849666098546651242076241834837840384715291188083091233257798927098156795462892057359933596975896867362330085998583080632-1.1756250982715055682915837876870664063628599891608432523715264758000883314557271908110610659932463459661353384605187319499295179478795012371917252273228736980504572488363296391152667697780644484830170080826598100003146994276073764808093124414046345598901339793974288423639919432052313238479957350919694073958354999351089017362817231950616167024982149588797572749878104752576181098370728684208762018742414139368738464939958841*I");
	test(36,"Gt({{1,z3}},zz4,tau3)", "0.1021151267281242768999667151971652851898363448560649894091334597952797047850099133987945543356250676017172291121281352864276109170380237548247317336116749225260062594926083147917559269245743510341160374518790589149926893885369086228390650716176243492471400735305852607131201710730660037460955482839260999753670887687832925951306554861476483559583310866443258212612490759758643472186749309847087965557231325536415276142122282-1.0764900916768803315456232209241904136696063702461356305235673964954155933334918109571824123054911497677931588872595668057769236922614127219641938153287812074656972284346877705907439778788335312408417164760235817573023981728587153841807657770350145000143088057155230356372387230604821266193556077487586488233833510718191061699787836975160483700288753597379632598193665561515219156112497158858707572865096532970063192273558898*I");
	test(37,"Gt({{1,z4}},zz1,tau1)", "-0.7329642403823227216553755077296333885922874147609413380086434781537410785714853019678887939654155067247221728745966168108768854991337851642817510776115310949300739151606418485996023935058839057106853443117982501478970409952787250475941364399813681615861168705023726231819296993763057009404398648774850477558062850986318925447216814721872104646506445227580825200782679592787536777117757467047636206488154138269388457860592188+2.2221749567909537329366965775938699608058006306257612697745618471920653439899337004791677060356607177325937800651037166112787417445339107850284298122567031361528699407310002780951653377517644500372230265246560733563997301058607420509477928009297700205683113573071052444224520257975972102798771369459340149521899526067414397417086948236734462994898704670829028196988301729242752393819838961059396556100704694814284456090685465*I");
	test(38,"Gt({{1,z4}},zz1,tau2)", "-0.7493242716597380748199010337541232111306237846600359134159364280202864307023555339297014380739265740879084142897923718877128571828345559712970110535915927780550634038631741491347111935528552725338803239113687615378050498053427988498361026631976488094426229167030809004758316157191504369340187773381418935046221210135330410167694459223047459443880227738578630327945640876173889616166094560603600703038707581314412817454667904+2.2079199487833378419049831688833979526671888148321166265941033877434770575393036916729588356090019864914770361985688078404028596539702295926421054046759013444044980360612072205003871933199261418199368744900696757883917862422131283883781065157715526852733076589245071385787354325883487374506185082199668778518376676630762358573610121672982417667846827118337823425061898754898252114327692209690574936632626291752828576770029845*I");
	test(39,"Gt({{1,z4}},zz1,tau3)", "-0.7276802893473328493221942899047486608038139846352772713752768278540723032349923250830930644692222574656679317615214501112045500985328222837271287222188164414539030876203690342440162531770946322033092111156934129959343302547977238903667516160326014570278724591092471472454835958529801198077608725367920920383635615703683002750062060797641983718064144787038593723903380421204939939395290080735410482581962651973862537981557715+2.1944958561478490509291210384881459178120973920848498734183610513756445126965832776798303629417735955342179876260150846105159602354305061438585444177793042603283862911194449296140248079116549218150892465300445533892309664495031944484619657749831419799764889167530634096256524358684767259307670504098438121308679929761094547556420088637419705647674966818987909142865677762358120511899955044932300285525261776340158513289013896*I");
	test(40,"Gt({{1,z4}},zz2,tau1)", "0.6061880497152828420277584148959363654367765433949890557031688077675342661857176038153091484308643860805728657889618033880379871531543219216960484907823275576571126569736254279813811134551630563149575684631252820377910576256983584413063652270901598120473830425412580041585323190665540557679957003747771973641394145382778418471931742256431064962885958801178021902429864537511592439296358963690985825854434817559005186773881333-1.2282077809550906410503549470043890237043823689019273347284220544328291734781625612735470715552136162126647459849458867227465920237777932846475312627802311909496234950357961130319292005500727073377945948615127055541101437237125848273241672081727926046944942311735102537156022800949067494669316307612248623087624575590480456322072427075157250399596827279386480584790455088081378078538584166807931182145620242410941153849473394*I");
	test(41,"Gt({{1,z4}},zz2,tau2)", "0.5547954053575208179781260056688331989031122025152600547044154717543716905092197931589759653553560750343992071158060753941610022294686577302066680069675058250030403383739522845693181497540135742998675013217530882376953571945874193360471296407182139010711639169786851114564820088619253834760723874862685764309743438237419815328374228070194504680753806716167560337287742031620163391293196781459179632399056510279016833779566058-1.2988052808278785851451877396639760761140758400358348246292460965684626904556583247578076875563030583163125375198817388462327526990508322260591177828278264700302437240475864666236193561484806958083641367370425729470083278859105777000990740116395933964906812113112930337944640599476101695842961188298923998293600805978864530789677611720789433981775582883871973514723581375282283629491920329566502040095997302682513977662238068*I");
	test(42,"Gt({{1,z4}},zz2,tau3)", "0.6509306056763289974381343084138303604543437211107538188708458735509792040405273126802205110093626364571614171200653048180274752583464981554126739559302996221334429870892792010661877973255054180120242878862662353068593696061507463077434878438057240011141888246778624236438373113601412260584801482425396608370641016290420361816674804177816598677335901588094837374710488439683963869873440115045858991049504952261229782796786931-1.3278959069679797315896003306434179220453349647902687743414253096967557004121965086606935321387819049306147154343317440544074412964898436311062524943647503259037504300918194728376632784837726943189753802537738132665125943612119262985110941375936601605807534158580891024527357485552776850834338349909782427091091110747015292024269025188241628107921542760413861560291664549855927143036776299680377268916132150072466372676849928*I");
	test(43,"Gt({{1,z4}},zz3,tau1)", "1.2175750680123950274752650832586412695819602934702845767831542382157546946894614316811258132758441770078148967625000091403172065950630250668968228766324810604830526255544233895434857549966740755911828397340005476726732039783509436219356320503930527719057809796227043386807038067982001003252469872876764932911106304338323658546545849598210586443284393039347928971207825156573460449644270996259673750700352607578217602796765752-0.0395206178787372917627806154595842484862402607067200869764905024289453344695985804348598057372847455752697080213810111377558297008582999270787104635084898781074772273517688717967536767422321387948068323008104336368925967719070958519435168461886960767520392343174769984048954621497694266370690218022656974655368316627270223380521304117013305908704481724918697494381711808669451120697803165913996551837992202394973804519636899*I");
	test(44,"Gt({{1,z4}},zz3,tau2)", "1.3510571550226892724051750634811684615184461682065117987082572529456438020810017092162288455637250574035625031416910910486547461583627051875735747329661884835600603070672542375463874471653395570848578669217342388734134855734817386272505527829827802357331947973930002470971397966557897261418327105990260956566296004334761808962965006880646773357351990335212443890786407807708289421768631104334712156688029153230032949455147215+0.1023941471712772347527813134789050304941642918622987716609302621677622939291507979933804003985494838891544015249410715081823141841794977901848495693318090639802245253745212493630383676880894642317963844363846222992866361293307986971913329447658355597969096145864679223571162663924386374369690325010513690877405579020185608549193105676208590931719929290379331587106353598931858802135815157114306714493479601018970129243204948*I");
	test(45,"Gt({{1,z4}},zz3,tau3)", "1.1566807194124356628284284456882529456538815244823546079812973976818541502845700656653167347137807220127243082952091462454468158500826349822503327668355604391538385049217581882112812052639255672580394259686866182414029833421283381593483497866518362842643862608975834561594303757626820144392904446597932438027665539425514481778953368837921182145794977403176944892560981108005167017394049776141949718769142163485872171847856259+0.2365812236034986651526346873426825456896584593206582371184363002620249748566374153866945950110656347575000095809956410153350415737014090912040844709420607756182423682314004923402658383834266219762577181980899356399667869056469030045059592443068468307635039203504199106643039478098392011352957316287795304104572784747399371698678909018543240042653633808788429643854323829651249801410410830661476510243877616773249945946756651*I");
	test(46,"Gt({{1,z4}},zz4,tau1)", "0.0149693735430378800550646613735900237006721178227006543497365641347619484645329793777282437052856049198886796637644913482198286523047207183087558115918176082612871541840860426761532465768100701976501932296364610584136269336094494099934273851015296692176903740231640496895482781294261230149833053891986568198930405840045699071664287882540522157877519930425137347637313386159089628434349541774963149232645269737371236514192687+1.2943649332987865329534805942648271773259852895587481701302388008269832763650950791104877959355737690380630731536081483795999629360706131967264662179789122294008157208090016047308569410046754945659387090870707349914948073519047979593255018113487730218458848657634083524412509074603646871499213451804908678271633762319411486395514908723736706135780651329316300669981372579431597053655737566191397419946055523915329073190626941*I");
	test(47,"Gt({{1,z4}},zz4,tau2)", "-0.0221876933837416703786768463144933342831173848719292893455164841777159286583234243559025855870799266846439594246301763977574790761708598545913214993431630342304451991575791097339125215177572309000207365620009313372318222610893167357152390093833225890042296580208433078232679990236994161217963222839540488878930325811005339585545586609007813581743215112233101261352672335864435719065135532969112930954218823876258569003444702+1.2865715197945745723864851398392493259745053606955248014797631116190271363064849137817997674939156096986196821539480229688365740802093626214497770402279821137635012732532789086020655897959069245574185103405839978056013120475640208726626117462931550492255957858116078721215132770707108676979529287438004950204404088215505541871406477155331876544391801253322561572884568411624204650355679428961852878471541610260862568246506257*I");
	test(48,"Gt({{1,z4}},zz4,tau3)", "1.247399058705904863395598693696788951427025384794593925861777510443472619106957556731625900541430123086287888738747862573938075861310430055991729427430127403206488479536951338122489971200360888699796528631278131112528486770530740663137148384695082050807329266582666470640850545088740058254273851678413899674729379636630734629763096764654687037819628995904991468561805021922803635858606862034518545487394333268031698588871812*I");
	test(49,"Gt({{1,z5}},zz1,tau1)", "-0.4169657493440353592637468782631426404723429653642337661702574027914424062844625925126876148161138929068761928116881571227562495497843937574042316995780844547324640073893313080024425088394782962375914779746716125365155990651185837580050920059531173341013364611059865667713063377026425466693602172786242015617061720718324801382609704212040193573901686503105151404530060724061576849852158103655022789027660035080586725145832871+0.1421953453286393845716399583333872177099556451732938954108125544887241137533614775856371142836944105215216906992111018090733291580251596049815001261250449274521091097191440153004625905827085610283849413658874050394055850315296147843402576362710349804965249054507961293591911716295169479546689101702314668000043903071569650579983739144836722959514425384593159863863394348638355017275966087510774834910360072218966774206566828*I");
	test(50,"Gt({{1,z5}},zz1,tau2)", "-0.4102451732440653899648147378999370983007566226136448625129318407705670895847112635101926504859619257361022760126851903351549695429717452897593754830527474874284764156045603450810314064512406707333802937733083363238265487990664511488118259836506296098398889578599624833562662382466559319525379311084138397924643711201886587505471874891281992701956723946059075946369481159238133569292538565659638260081903060828142076457135006+0.1496297902535734468202244084248576167157586252789310716932539998935435829864212919736544688189353250452042970499205910622428032326330331849250229602575765491352171059610859108133596921091270390189867278201312236281004463399612278581638096698216316963473866036808142793808628651803159300378409012042791093787018060993564842883012889810798461509315603053103762458832667456191015217871957190369383155498260971991246592561229154*I");
	test(51,"Gt({{1,z5}},zz1,tau3)", "-0.4209613147532515644871413307192009505500481906111644591767139240157301901513791680675039481895318327010211764248001732633188993175871756158288397328979474565378745535075000778480383293378449046622258150022378202841247672880246992646110397663574096093261964288513675948436875192755398258801754174802528837195615021526020440868619982539758981456128471385540255538103781150861860346534027950761366018846420826309884656779231272+0.1543103501452613736170142514079965497754571126058008537151433370775378854545059610249160388493499422450237541994157494095126427018725869116185025461708437188662852833090008339295238148613188491083118815735119815411406705704745431181925313400060719410449722539838400657327336085864930733157149027234986645977493898082057524042672054961494095676357860325403532212586816806440927689600649251657573322812137299690131741712940389*I");
	test(52,"Gt({{1,z5}},zz2,tau1)", "0.3632104188735208431385952336046928259150575539228718811452891725201483715920879089293043009981715382033742108756085292129548135124437667806235127683796426108742050157264690926485235308923079808290769877619495545436485805461414182698488270998364462036569423947220076060077044766222987186017852810881645822482969772585776899100789600166037268671136887111524078684077175071631544024997181142413189237855112079561217000114374777-1.6765034130505729599384097234275226161139481623389796893956999656470187973172359193388498703515238141591303728022534161646308064875725051643206756132761393925964825509672577826094011533497975497156154953928953498842754090485980172140219294460708215309557636830540804839556590209595847820348869023511250270884447898420380328585734011305734090664113832508003088078491639716121999062812905943489454831803543245192726041612656927*I");
	test(53,"Gt({{1,z5}},zz2,tau2)", "0.3918287425160793856438018228372134164832643841142923377647147671491062071839128980391968551961803041123626062095855412397849524686822857424027776870806105624197319473024354874695125617342959877467493110570732661694414704964180439416687057372299747323048687427280417118436337851780353516029530462992885286247148970236250262552368957582843822520747751272880465139197343770891515366511726485546402288292273679736452513635862318-1.6353021516223189885506353841364090023820435441279922583914178392845934075814357311832623293409077020349058056643509382737976234467451732383641704032689850805562865426411196803021627093574241538154045083366917815939394773679256392560431525208096028232419954446932718669691947995867234512688699378512232067654019833660073154584684484137865422942666071351518690058566793548240398344129186549853847036310952181650938160196172549*I");
	test(54,"Gt({{1,z5}},zz2,tau3)", "0.3353727744464695242840129002212835097948958740567058306218541379002394004884819192893824499792972794054862859964325129713552146321378023588715147908487648002967407185000614103824644444011609588293927397958524437264626240967414791717669180808357892813447829779191606112815721561705559083717708391510970995158400747246886793025123443753885900843057277376390518789245505416993253746575274781728181529898651326385052001096936884-1.6173903167373013021149362084005237809594574455195400988686222709192327077462481163283869365885388922465318938192467959730760322091728435968746160376181509967531752451950561770014273971719967457331998665083295576422575217575442717775640431724416642197240840648550889751938217720887727661214431368400465045015561321075771530216842716078534126300400243487760991689037044568132242188976815475524266807025826976555272224515520329*I");
	test(55,"Gt({{1,z5}},zz3,tau1)", "1.0802100643001075158413468926523468179772592550187953903489820143090152273788853813263247967157130046171931113316429917263175935807695993224887796983698221086586866992058858314563655665326406135803901671087453746381742923869326561809171242957745577089244778152018540396998577069722772730892320780317160780979316742865351194198202605152433743888095911169888001614703393633313170965393229308002076992499282102465404808020461316-0.0619404427162807258606066910896718915496809743580162102516584935643522401171072611522760187272205197856474342780020678082422673436420437443326566024356759425435279770724631402932491488488833522231125269980344829318030853296995592663303220651597651052166744515085760412676882103658076794481515137088460678255971475998473166331509275319991342073044137408925960462638140959195741534968203697547565089961782486496461200756548661*I");
	test(56,"Gt({{1,z5}},zz3,tau2)", "0.978347270420931156324785533380361953076503666460646107448765308739312044855981343385780921786379679167222044367863073708902247700094244525477481368651532237291245978284093008896018712796404384441763477954826403452891037587467913902570726812278184263799134516678088746195667154822968331751545066865988154197214650505231769460524857364250615972670726717586421638185044805745018043736838286270694566689810011835304560787338389-0.1440135536294584680918797418006194747173677940086085331098522669389138324116402365572494014005823808944943175767843933209362298572016283838939677522672918004043670859747220458474216613067502620183838666165828909486388766706300528544468089239655621239004411380443098565364155363075446447996277627890311948062604187260846384391733450374091135215229944695673422290464577844143634229185822661339202897841973148284501615681875217*I");
	test(57,"Gt({{1,z5}},zz3,tau3)", "1.1109407036828758292073552850789476505471860818620807648633096657539667451594152003667111358976089125732313970707950938507732071177504528845415867840600575412593700728087908301442566332203704555480914209331264405492936755834891090130349070879481615756918101936275492295680049715970591356793019344921548990450144669415296490607485801227986013664892501494622443766028868798264041000888774233346229741625700408845834938562073967-0.2140129686205106383967892145148035981506590978048790976121599386579826100911360257332384498515329920253977303548550979348579879976993821927009098459920897615170183158035057087587440635573777533262319457556245009885604449181354467928077013905416639732676874705638261957491109152799329091692670979011872722277329114352292519403126559737510540497355752674021417778983241189682320066010601601129800333939992672753078718262751378*I");
	test(58,"Gt({{1,z5}},zz4,tau1)", "-0.0087903855236145109114814388543199943700561450419950098134989737090056487257809564521262298769007788802774744194686296211068787748476230046119492054314035880821459091084949693132870128395017710733123712610293837677815676683481826020507073913224251159885758417711791396307313592203499938135796644214804951224812870425062979830643480882966150550994029573702088962354612949594495224988788613480580483480083031078148886069316095+0.7118294564208344424851163096898208190026501661641382847225892981695741082439009385641630625008414787775106921519524867707897177246018440910199953947555423824427414728177925676020777173411154569969601627991135333531061250893166438697887601252824128259350473899158430953384864448821332761330897289062720721487381533235857055250782720337454910710972105592543667620144387651949934269770996111106829558460879021774348823994727251*I");
	test(59,"Gt({{1,z5}},zz4,tau2)", "0.0129289694191044100342126640286761666707217855166448325383324639373157748776116661668491918622739075867265865485128570540563257595361678459420418849714317739897875803540259887977243322358129289505467346089028352775040361730730607419913043230015633888903073305102860499202269310098625197753336750760640980747535856877360143771752888567219870146386003276985936206698313743119295488029033855876313764371706880489549068997185688+0.7164638124350699612794905811779292772715724185680170722311012527818136827562646680407726020235242431890334028841716491642846789109592255188775078339178597900339537688092870635278595734213084512498450520877589836229779351659683837393210461824681084565259180360066705083485553868858633788656322655520475872172862574274338421659289322677555324315793671842692957844039026898188787162788852358774736873701336311104716534996830588*I");
	test(60,"Gt({{1,z5}},zz4,tau3)", "0.7387315749173066214049799070014942386358416004068768483374260247683387753362218145534807127601697591198885602470921354208491768694926204602942032588681340320907212458639299241776184861894351126398150895674511955808680011757123416850497427517101793260509901155424932467399716792416727650664711531676210591861785214736904633276619847846977848505954917147239497985289176085912042608049089495050701824348032653017533931271135976*I");
	test(61,"Gt({{1,z6}},zz1,tau1)", "0.4239165675194977856832055164690068365265829488484798233177257707602133408190071665854473598951413289154232205418435022589190676692450186292797534803199850390737468002783137625253393931572828951226191983201369287966414308908677476656080243927950987003878669118638764210878150903604553512117582155512621062743372300594759803310471852851856934595963670738717651262299638426658801038297700917199755976462198565786268501113655365+1.7170466145872846370260025068659957325703046672073129480413600566580378862174592325038058581840457866496963424372382809873020669521131563600064350092614746606849116900572658308677320425217738943161991185329566685592536311457439431910486777438624829240709676731096293370138410752659623241633952307313163412645217838597847356105789845620985382484789111406221542695426947186583348131115383770220269630251254497194941338157051438*I");
	test(62,"Gt({{1,z6}},zz1,tau2)", "0.4305550826641644217442082244563756735421175108998278261423992726921229456745514866737250419661484339322216461887086786997435711879198613536288806280043147218023763506213349267397461463530871777214378225463233352631794068032926293915305362755638719565938964278829708510423500593137720964258710474111464963460278600858523200824273009953262090628169446845202058618807503741629526829484161037308962442550389065843017335890040161+1.7017632716490421899266975534741951649174613001817509124863072801730643141886026364570911920010122525454930831341540139799073763998453154904253746616412580797389394139646611182523776422201603695507899140879040582351915187053092876932062788036603643195085510850498201922770703850717577596867753584622830311503413993012322666823361781367079140609003377673555508111710297840365291528158708580456171367801015610741242489803743955*I");
	test(63,"Gt({{1,z6}},zz1,tau3)", "0.44523680578853960666267990195082949131583687788937523788542801437091912027799390503214992339468488160955278772930397797994944556015504148573373863664937584059373826047236158074108003087426283374130785863435875028688992466759824878072558171100766746929428955743657395830773277357402225756027480266282755071281018488302745724931008765190761657815634375801276559710858763033684674783628412987955341097900738116478126854076391+1.7146371591269696577162524122240719979728853357761293183420201861769733238914176178220251443989246748442065968198597913465286909014147351211198321419839229717921475998512231457683227116396594943356319196585019056168671473019614548965061519322729442743806784020139858212186364364357883623151622420046533284529913260454882451772639802263874250603586310550235262809226272247684713358843059516355597454939575202703661600749768709*I");
	test(64,"Gt({{1,z6}},zz2,tau1)", "0.9267982408641200056969979187261690570776259690371739504420713332000707282418812651529345665252889037986069853973514487266751997630900316459079719627816055671812093596554867845692701766228131905295887248973821764127164576257095372323348070200098617088757241778001901332975459286830751892752604466211487991700111764708139548892520071298777325156321703530688360610258001607173971505853348300572165504961657574708584921419510268-1.4430178870161465742734523029145346237064012063358913952023226991148382758387839262336999176349152509920769770881061699253104316554904982488535442951019961024289961635793936291819590288813931545737453051062018632966791644373466501301412817487405973627951000951810505681921710839535450841019080026855633496813038088361645590117841677620090048343855168366698441426805033116406468605818896119974820509973019330042674850892910668*I");
	test(65,"Gt({{1,z6}},zz2,tau2)", "0.9423912618751258245278514860104677416192697025924846509572674815556778395136389190582832001740340005572815892704710921395570851213811385932385990529830761349187447210464032497541307562177523065550726096580963239934582007698979169379979528341578874425790741462912633724057571904763497422174309095138924622341628191085467238698293529708731801382446579802408864553305168111729981255406288378966488104885491578288835411038535832-1.4781342624810078055138080238531979228662989132913530574838996041474742177284832706989452147374522724438668203775182098889167223991749425859097436545846232388898535733756620795855219105432296007380815707453235868629253707086507341336688835623426043118805000865641121557925012263010614045655638437904488769369503922312597632311423090988937059453886534768937284846807150483423019366134296394332075069824329891666581601076289133*I");
	test(66,"Gt({{1,z6}},zz2,tau3)", "0.9755022485526090352418553457912817486653656334674148509381200569329881199879695730194028063240022144562856969795940242800773829418065277634122741938727294238652259698693361651025247166757923421110342508539520541395388842760746119763245804240030741909679369861710547037782609398811028083403074749161642851779226653322446226620291444653956814311854786822012386839735481912955363674182276159868462333261749843398363516564968681-1.4485363344180050219870294760343568012890477415670897308988883390437137409132680375121724961059980996483315679763761309386934498361496400114973761865652530158571886044541722782871021568542941030826104017954658767946983884489607617909192733871650181014927672722076261562451710103570841768944695167388954386052221049771199032306942314644259865080547641152326719769666373347688429021063647719690695421617724361846572492441597495*I");
	test(67,"Gt({{1,z6}},zz4,tau1)", "0.4852147580264679018609996563060287426091178521999701571298093843323297175699890754828950767293033540347034715693237710941378785332136474506255951461552463707186361670476167907203269248259503308576620046112859515595953143918924332991042332976244868380560686962729248046022051041570117460353541909932659997268979769751860146258119346278748530236229646571336154942598747712823970991918228089574798806404805515365179537491782868+1.2701442551188673845015469996161942290934072294100587519938637806764333791938973282026348257041677713900609438900513662892661358433368197483019975962741714286991538159922670083283277904624049815799825423262840176369502976030620138936838063117142124695988479207028541716862959462560586489744059397562718116297055416763155638428041776657999721366238162121754477323409765895958529888449002248295554209140341274950110629950823842*I");
	test(68,"Gt({{1,z6}},zz4,tau2)", "0.4808523100836034263723424798462611870080212087497829447116239241641931070442601707434468597599835386874151926163621212407029166220517535970544923878264484524251394952590780647038643968467819454997702446492001035461955815948951926606785033629593663169817566182791642005659261965628965598373931677256130858144976783822926672652264354638722187065806061654475142090398500973344108782538701807345533923904291688222931867596250975+1.2531509379703085041322598168308660172642390838913502419740142090684036331916872678236639376170408844061197097612137696018528088196951114407164837400650829586409968992870087773121177740878524221960094482591851034895011991188704171398960635196754729376329412164435560248596226603873669808902380813512438163504341792128868576960130179552973950439064442197324205988822584031710499082863083925498791710848059801365853884351111894*I");
	test(69,"Gt({{1,z6}},zz4,tau3)", "0.5014199569890899913401577444158470878948976453904396101244089627846930458895630681005516148359811603539327472703211139446630097828876742513125052754985152990658896571416528994469254955407021673892489817128773588137403312449601179719241796236014535614926043255064559887247783746639851939404410548227889252576294648637498170923033921590133541653256727794468068661593903707731182579163321071073532361460590193420239249327114787+1.2544995935734873496355410400282550991923006339619449765928778074336130935554398633544979742074747938742125902521513896975939127078969117587080352047720145770171295617171106742116876773698622817953561648743479995315505922499418240671834753644969044648832126984019118451375518248694523736169124802395976801880476424543754004967760327692223585314554152661294808052759290600979420554187422098790828637903503704632682915567466025*I");
	test(70,"Gt({{1,z7}},zz1,tau1)", "0.6254520690331401135738723038334332749224820391651922349020206150132853624983050416941375857648620994883495228781631369763330961244748960186832256869242754528957780258878495432616971379190187392890446059804970972863510518999889050547862587631136921985173073448517823311644088364758987109737854960490166134640456067118983113049674425833615136299451054452467157875240356415497959305370195509326456773419278668746458196469345122-2.1282593782061946722127675723934534188711792093506337649141712923548743924835826789247068997323703719445375263467746214457322599180066511194019383741112172661861785286745360884120054712327918238727857261883595332626159439539360332556577516767198256226162262660528666523506952845812265340838388619286954046943412224447512890883286689201286167960113400484714713346476770582729293870870255350060694365269447915810562970400991745*I");
	test(71,"Gt({{1,z7}},zz1,tau2)", "0.6389064615130938250669802126961873663885112541924108001369535500261954012103516549365598230192470584233104116588223071969289899912851068785359071392303585927248954518973349700984630908273787086810938482689500042016307762027790265785300500969760692171517616115360592962856293668757474332996029053419661462329713192409556418087599489136200896733866101001609803369137301724151671616893501652535918977684880866542002487470881672-2.1105470863206534925747292576463473159935241568726852616345155137692301605606940221492578591517684765712398117804388400692133455136212216354107159705853006039976612365025334662660670140182134938746976581248139101669323698931834725127997980864913115021464929783919029258014450470536084175049171091087539130622285699831381667673860770460113578747122455932226231521675617030312203064940084336623179677143932039369769015812362821*I");
	test(72,"Gt({{1,z7}},zz1,tau3)", "0.6143375195428282908657077941171923585512090771630945875030528531920598118142958585529540275679415676554060670615302808671320345417919506149799429315890014959729765787942593147614128618289173282281185210205231492328916481091661049453870997675688222282942825544870957091261659748239202923917142804214820999686330860948014790468311439337010090221015861253582379949073672492022392800025426649754840847317409437715332464255760985-2.1015824201178055772301348955838989908826276621379804551039087911960861815005983826218367750405412389802917488894765046815763145290940106175317760127762781079009378228324472189652857740259019840436779814819279249987439288549694468986345421743381210101167193430195035923832502186226856956732160084488704456804197139027835390426102253404057600076029107375703775027604830443165815391427983483687340159027476540389959502765112638*I");
	test(73,"Gt({{1,z7}},zz2,tau1)", "0.3174448587086263043858561621572428860248041933458205278067650491886507467984839940005313589357198464324360912106354035621975471824228878343824033102216789928196130088602107711758369886903150087413382680954582398340842606871000351770129992831480722217823428940125039541137607108477650545457847156507899997712835034567156826273761670545641579334219129297798186200016165329072690666465582675489695998567442526845177398642817593-0.340669516519054885252578835420988163734402804034469945546721180447487297618992913212063769821472140648199214407356245799652937553020497402221925904450272229516024836659744242246515860915257055064201699692794200445634949274272938384547248177966384133723872697945403362736253371247938137218124470328481479726782147520232375598360766284126673989140338323259256402232451160409323054800628534511816008144937390596702676531713644*I");
	test(74,"Gt({{1,z7}},zz2,tau2)", "0.3114481486392549390679632019378691505374118514498565498549652491228393581275205570730745248582210075589444618116291747503767118419592932283042813099452363286279411443225444877871469361603257522832870322327416415207131958851710607703228343609476079879718181018940056681802087365172053754792173121481797645238902001602667892211282731579512265676551139544367781442046810204342556822456994726988152082213798985444079557919884376-0.3444790469620740977495344781690104619765993052441703219533712866885976452165029129704248281692394703230408719678360706276075088826413218777290031482800093002788218075728183701090985343201911763722246028651672591133239683992711948384287810018868258655582574732558329478265117857503220392372154019179433591105366191685547532142753656561671580536308330387523413306516906599449659590579539087859205104407208451395620608831107847*I");
	test(75,"Gt({{1,z7}},zz2,tau3)", "0.3178682798906829274205674773646962164875685603648550677150867990852088720244657032490029044129722021030714167673542911758737270066312339152777557214521118353916592095285379649500448412248384327597447851344669091776138078382945290978166074397653849116545762871521211100015427986430423862500385370845969505661211895951897993706507004951631075175129271189112712944320533925773003046979420914940906223047972622216128792790880374-0.3497480100044176590576158373652754964451740530745223638582166080245949041179834905178817968776170271938955898003770737773344127666819790254045486960479762752868970421043705868955638339649614952769585388625285967321406957578582369705604861112814414967561218427903165034197738366058645715417694858682582987836521538591213575198206349934015266412725470737868589544426002874783447844332230458369223487265694711812020930192519363*I");
	test(76,"Gt({{1,z7}},zz3,tau1)", "0.8158768851043968063517304441643334749220962070045653743675635972526896173043243457273241216350600503842712648980223589645417804043895769735121480048202929495979220194049482029611568070812412576321218117227171386138250518424107985652821777447957164656325474216530209588621597003052585424482571661442736870344391653541060866503074956011973850057000554952051613269422870856558748200983215152858817655668922420791177512600617276+0.6735025345200028690899426789512783367742512420325339726926816911453411817377604156730634487342063640388305283080952297781329535118191783031466892112165478455304129654765155275589134645499018608352920494082833471893631609148212236881123758110294643931598181200266768033409059544445554818860724033086832747943461383562599507239491971934714389244268342523004347864902711771013655947915041078378454240868347799304657853125067377*I");
	test(77,"Gt({{1,z7}},zz3,tau2)", "0.843525987480033154468477001999202992996138601122805017517887823549024503048211616664649702774705122172016166629437680645050299279539140814187799701552629509126029852997864641121486412670072321923524749478638683624596204531051020466037890860918680211560458508020583779383848342386473242215782086877953146921548100001064097125388382771071241571039892675505850349539151653903253944160713014669811272933970529329587590974261117+0.6797286198438510462649931978881388471631028236503029435403635711779714346544060523652371162230553640389280203204829210651327535936812995223657197153943724063534692549629144759645217762258151384898480868742704933961327827484663428561482826010482571173522280864718397568087026199019249732462194658882039011629891891126629597855363715381093349064338410212729221520869455893353419933952952161319953752848281839688653497929868637*I");
	test(78,"Gt({{1,z7}},zz3,tau3)", "0.8272334709854871229763228048670600780691499204078724645490152478555485335067513419403553633992665577570728669774577554815709228851644382301494991944121462804088364758396638669122359409893454351882727590723154189216396744478957700990555101121069144236320657765166942938628435595346334986359862317155125950097201183374363043586331820288349895179574773524449756772455851010267464469309433894155160720341478516615627329352598857+0.7083734983154946536083258614263135004031101618796547223262795354320948771202624132045017915609400142067755469437292457219946943662198254372203609839818734785898922592847300028411854364228986885787143426432027870868468607017738344920380768044732443975488026953373433169244260552153435923135660675863575230741163575748474480975342727294684214075052537487830607008006147091175523339644992141143507365657424491758568000473154857*I");
	test(79,"Gt({{2,z1}},zz1,tau1)", "-1.1848166349276550878098415084220933968283171074662445192613103357875157715689581371353156273303664712158652752629813123947425947067330249093499338020026465112212931382025342593554257594264489061996723698140045924245696104900586729491113093224587956752382972906491791064819920594712086622846052722112238264675395933942176996792819091006709951463685409457467898719149233475881575876279129568075895234585306025403947263883627806-0.4284924797735984699944056114260859172547543526462510198751049943946504632084285323876948348821904881170859691870829965939330783992926791396403631714760878080316791917568178116601485437001374057325702345748575223686671777127562645519840059566135776029874507377529488305071976229782083053462026946666676594668481120179029642594777789790721877268445079054504645733869478368477307578566476922525225603349979040409216206500404577*I");
	test(80,"Gt({{2,z1}},zz1,tau2)", "-1.9251174834553839655786893772475427959533491866054899178008768466155832756506775285594120632140294056953152649761209151045094715737001265133720307108570020355097898386987411691739934089481186513847944144790036854546004166394934079522771214165284455365651495763261441587720354669587359441371233949294059280478730396585121312610564859320255593962056597187825190606786503141218376516031406413907823709730800584181445203723305619-1.0939510595922121273130804775777210053561997245147558492583788436257071773468895640680774573162418968255666181045763789111636314390036163266220722709271451144983333994617839153481774777158365049432452436200900913989434794668851012127869809392691738259833613974232274119119770340991349162956561399557415212706059161709533045331188804613553193147969379324022728635657421845602938592836123898046489785121662639512672062448797347*I");
	test(81,"Gt({{2,z1}},zz1,tau3)", "-1.0060763640698550318672406049415215199530267325352823293086848176853803028735339982658706372476455627543139280330603547460109100520368924714077054871546951875765773487651823356197157582644777874562262575249728569486391417508335527495993087635116617407731941477694133258847600626754219857441765559119482370205986818671964722103831077995425253158909835998358748766794700384275611172084314898933065641644753403367443261681023344-1.6608597530959570968505091247376739734919611319531017088211275273116164632310569587402385064586242555846683685893807758989541175341048923673250098899713905177234841141170095514728872146800864506741247912557764304525335619373248640411001697352960126753749755034709225043125188423013912409494949027139925208517805728679883231596244352135581411965740049590427192862605426400799995151447076747336029320965815672627203431782096577*I");
	test(82,"Gt({{2,z1}},zz2,tau1)", "1.565620282133778267601931121558090845142532337966015423109977699303023404182299304791509434256078286120446412484588123616989424432203712460748994018080287051000666504700054934266278311902074952275010019310510432743227864733828673881312032502787913331047618912555555431841797615511100279045814085203518774178005343341185932482681124813803425912433750448880757548554238617367858650427455431447296884943610088095232904247941845+0.6062862517748268520507215866326631283075256489602433991425764431178878229518094064818372700080335012653587599417915711115813767066184779515452822729679941702227709144639886193082820439851265706106515786902703867684376671589506861354956267053418975568622529914729147207268295776667660493990981770992837163092626038569519457131817122873150629067714825230029828354265772227315721426142654897796681423954004668981189151598393786*I");
	test(83,"Gt({{2,z1}},zz2,tau2)", "1.4309468450788237991634500660699700189718935277136685193351038942905755247442051548483073715703568944855993810713354250395710249902519083449158481599900382205147534288990092441173648576462675159954378995598260063173377972972474275905527685840953032404499309943755951980010812669004531009671400683819215576452230557323126054429166380293014384535452647686859048240582557917438092054408358563188473079649871455070698933754963893+0.6130049905868522614347194422831627530816116564049487735284713800414976254263737307500708738592458728459727512308283111936188748884517446415914693757097134635509209320700275361463290830331973272552078586525777107369387759816706981653612343022921741149401430872216828083176449530038929384518296797443003223643139421399981069439162849184680679612237011680838986190222536174158128097754116265536566815807467598491562660541624645*I");
	test(84,"Gt({{2,z1}},zz2,tau3)", "1.4719507443511197468064181383848102640656022389789261026971011709191080201106796739090133112121975781630305261784038651604339800687573881332330763614511504310846557118411948031215026728914620665624003689756652459487577316877186094338350101458949466033302944408893973099425774465035878174718616430375489737937368697461374027727062194223785902570742828907341079606503709035290517211352986321224312549018194464694867861300801813+0.4589491699443745947865236056906320709018438247099585038039662868542997497453821283344151556149865146001998865085066173611518251022762297619153802369317206120502191070320670519523627489637387267912918937041270987142109767926154402374920928993944162710337413096817768909871337050745798335221655741790269411254920893802842974625110644636400211759067021849731605960647447720552064315445184450743035566375499708243990162732225447*I");
	test(85,"Gt({{2,z1}},zz3,tau1)", "0.0533248109567116728204667650745146730898411651484428942667523763522732436583090048357623348213086532563844280304687784924198171831828052550671314239067938166669461503507277545329203737364444026414186512751175077780855796014873812501345877914646747239419977360522282143891014451890518193958814630621121863734766981097088592390399337016878813428170518127968190396718958138978828972269877373395137980291560940791939409696540219+1.2869877115916169009246576832033017689614195933005116805332092704708268686997640390886233384729998979520578463180083260863641749712052529451583936919302283106957149880394983590559006691905870814009898592071446005640338775544105057396309830693830666141189213276228238747943094786082898448526937013404194936972753422885664446561844009039413376994649110480233350594261263836154846251024792167536547532435056998903878791028926303*I");
	test(86,"Gt({{2,z1}},zz3,tau2)", "-0.0675331852798735276838279628105443345587024375358518724409686787319788468298716916477773841743460947719799205048824310539887906828070926126649508404616407063743071096453866166725878888876971572777823976403891430246410430488353619135963792835840991048189246603353652384428259335677243307072433620495343286574452357272366679923007400846109412098182382122691986142650820865389000821292123309959191939319519928962509400314271291+1.2716028444355071390087506655812713629556816462122841292185846742546510952414031556439297475276659743584506291325351388124406167459727758938091625244309408934557209163880251382864004445483084856415648925843625333068969703495541090566656978523438874504714534733340250834642438211286801291248872801287976445854758305253673820284315651136871014070806034618934349419302104187390469226243312199073266638964289567433798021173343951*I");
	test(87,"Gt({{2,z1}},zz3,tau3)", "-0.0068843196066094944563945426837241468062386371852059334748388753745791851350191615508926414996230636736651418312108087886567591739710466872352748398604994694378478614482220541431620406981991121105409417302657514981855066181986638186005015770441875303058414181773787633608269102939734576518268340303025659618095291142984983843311127742434656622498692000870240464035680117186868929341161675188982203940507636014990626369756777+1.1407231885361572071929614564722474918517972853917653411084216080540498792286068348344557738693603941623334843862610891226066847130134461966469897459422594675355934641429014481799086628215495685673031245812924309081678502935531000676258917359649307830254200943650803848193940471514193504463935072677724498316206064354498746246011579669292441363456633267106917976876806563438225232844412098700958029567933218973337661872656744*I");
	test(88,"Gt({{2,z1}},zz4,tau1)", "-1.069730976991184642730146369071642995448051902363176191367691516550816511162440176253931799041960552359729238511489608108916728869259239720202657669228366474283822615908871966701194283676421525599316117683387899271678289915224475039409459139410385859876023207625275044918640033058777949847263398813979980899099841127982762613762187760174505166487624210187934634378209660425704101524147045059681636673505964127098843944255212+0.120730791511041510231959679020195955256821616542108623230351344701138123970921538283471418056945538099624258823555887115187833207744391926329165581212801829202182926070506163885182557688422424300870879900179854951750348996021590602984965343515951794017952917034660649507984581374257788615644650754610076577458004434849526678483361601896809438233722831254814616395346282595372356399927720194599887742851262988915749666235868*I");
	test(89,"Gt({{2,z1}},zz4,tau2)", "-1.2185832258941173660379442808726583208445822165531392241980991174493986635317401673023028291326708092936589019098101414031080110269791633053592179266761292925692497218198872190703334213735662432924767330075594521753349300351214273790018742070990300327159522199432005815137538321073926368482879767908465503175457922490101705954615928600912290374599241541588175648933421163118886026501564426347138004996319097561682235323524267+0.0152813530674262825562583158136788687555683925157906395412785960865211041611382888769590505129840934541790049328534476503431648241328020797958717925236592459924124972825447105928027583628362534704313751147842584758557011004155231780977806121829238109042122642675970526071624080981542029784838745918374011795729080437428549126785099083903382452711555048967925740445070360756622119531514240434214543777065060005272083544306499*I");
	test(90,"Gt({{2,z1}},zz4,tau3)", "-1.0485896152400624094844493905599646378349382669702102426167031768015067040361769860821587755581344242924368946728913208849075850024638273022199151845015200768038295752034586155566577968294318735662437126960677064956332233187347888919208915557124284850673323055690779251653390384502396035862622188394945603000455335917222792200369935817053708400288631884458991451351955112229452470983192882759959781715227266481613927900057354-0.114963638024595930573811498287247708427854667302813726607509667460495712934921370522564094123670151507153072956298853832173570951586010406929570402807945105160069021890766261837618348244608329760785441293186433731026569288453533625575087463254973552098625400330496122971106675596636824767165851408547220632470428285091117802733053810500800228730309022044452193292328444855829885974986212990560384863714628674730714576588584*I");
	test(91,"Gt({{2,z2}},zz1,tau1)", "-1.342432835588632766412071487050054510469962860058310128236809388876813342055878849457725044109535549092202352015127469203564594961055492088554742895538376436343069044557221430989128824339754935975179092121819501099808117765606806172229578923518766005843044373265907024329225662210195014635626610753640495176422849432567883501066781630367473428027113302083813195066570850777752888471071987599194665649778684072313923340879504-0.9765801988740096898963353540047697672089582632009760091130450440779247774546406289118610996366086439097657079011817781090066192273902962641599554727752942590779672537933389269106768979423339373718041281904405244148921549097514999955391426339807082917770600084808352715618384177759633294702296832998018661373422685488570250625469468552854506922223734774885654698377724613698368862475948673019726808476094031960234434005409881*I");
	test(92,"Gt({{2,z2}},zz1,tau2)", "-1.3950608053151781317356253337432692630110385678147983959129037707371685031039673641911636952992120062614227725562541134582938613366301811714717170921250187734601733314640729631186652469916742758025401797907904773987281952914268364890325804410468064334716074091014884776499392269635072302857334452489733348826937608912205678629181258300508651783627603087844712678594834611306177162461352318517026434557162854741521796484099099-1.1671772538140873178031185114332638471104129120257971701359077260975456207693652361148797256126729979050962869370193474639276541560380663099338254908441181246265319731883994553140952182388165490917820651834096703724922786267331281005666434154769796618218455126412742338840547327928021081884931476303582859672498992390401897704723057446932213118043976742068205447873501554531810970982363229271920915906480351696805869735228245*I");
	test(93,"Gt({{2,z2}},zz1,tau3)", "-1.1659486040289645061049369041243482706666075183706873998594088470677059763934595540834654157266908013147986529360760441038227668301429912671936443184914602620092681450370649776465060360969422758569337328223488225684664643555007443127758679129313838272830061422782976909290975093662386229587012992211197121598063268860772625604882004380616462807122943679547944368949763577498366089145710708634675822397179531169010376492986594-1.1601337757467122803573523184148779256742948452726940091989632667697613549072017605042509138564237630768250325624241694510478121570567441118724235449361436091499940226247095118383739691524972574762007497015394451332571350226467434563592525693294633094140606375079478382891459612088681326361949278260536875985904132121685067005300505154729780341322954953436877096662344622835709910014703556718447429002615020589032957915258709*I");
	test(94,"Gt({{2,z2}},zz2,tau1)", "1.476468969999101438514211761481368577656365222994303764594777845186420841611921529898930405922650213189488902475448251770859125214533307335120901767706377128780254948542961732049759878555468985864774046936809674167021487936890488775133918172236219101302815871613684141990471348806234689007212181576824632290078926425636299120731245495406083025743769318309359867652739240729805728214202310538339421267021168335506315289952974+0.5431167923808177419721643222629162540409530415703383579681561980055545911080093995684491745596991907657191617308963166434463287543762457648360098697346675806172969743424928774128010806209999465750208649408140739256367780995659819337069718808410069843566800451699462575873682788647255411756582534275917717040901391045910359072363405886068241702635664562222712470413718008866906534190643538488702591075069722944281907774548879*I");
	test(95,"Gt({{2,z2}},zz2,tau2)", "1.5024671028849765009556830655877311678305147488548280126663673108978711314040376151974556315124600750230542743100069081285729461312206564539805690391222201659816130873302426839959071850996356415995838934471018172665362985045533994783733655445748880988349080416906230162955471494794903462983271133678143352162474979255498493469160539359942379454076358124354995659899397839835305706296365708484402473685902659556374759223079997+0.5300594138417117949489443855044333993256429667249200673172301338734041058382998743249115523301042099731761277707197646241055930683490586271369823579318344865448667945549617320496240574726303756587024166068985953624643829487886246560233193150481677835564013140942577495802504814715702631166097455212465719883830329452962287369643933751584030521997810350057802525500099307389742713174950373600972854592221578554846408522609972*I");
	test(96,"Gt({{2,z2}},zz2,tau3)", "1.5077449301584102297248431135229979587171890489669121959423978724216366995885659282310311719279222444480113858787019578119967534447240054270657113366573996208696905663401042489823303131162090269921601634992938006663116732211473307482316843974689539452807065329134414870901978922698394740125673678146901000747941271099519205420010693386585718869829152686880235705554726378082368652878496993218543005289378892267712481435784935+0.5636045394892375011518040659779082141522482868969948060032999698939035679765210947149974923511257713805004846313138758858304965036121650983004180859533634462263953799954385965909311189337562023363778147052307494657876135715398376072287266414761494256151986700494847360869874502365533794520495866611739841057679312120862633225542763823872704915117196945452757201977232528508265137377902687034940776846659864009633683612785352*I");
	test(97,"Gt({{2,z2}},zz3,tau1)", "0.0750997956678031411026720762578377107282489871758164288892890695696707680890480505644835363549450169288174334175067196270150301653142989840832100907280159410180157143631002191246897814916186168941829183427961116348834267832091696094215921989319590200054682069204797133121510008293976823590322653911423567017985306974155354131036152119899342939034955822391462894600899367841659157505944891482613396473587951297413505459777514+1.2828577706554948737249306920891998867288635579740898782183540253520937570152680377866409780098576388183755000906507648346491540505608520758161779687583525657224895355357606988959986987066780546427898529520890320694064080347653095209464075041119277019259853047413746744583202212764736945043854434543573608214213319159346780844003094023489482024486512727635283404795588635007724747426161297873481898007979754344483817243297754*I");
	test(98,"Gt({{2,z2}},zz3,tau2)", "-0.0650159202467863395559735771985887538867773427491206385187634228776940415196108268142542640004818516358705146088426033324055039010314630286852169807781344924889843557188167149572875499695945621393444455617470541222562516325245118595541646084418733991086404858010019285295606510351602578887172079097300713663834045578947649916770807509529575035493358974970428609071887957446960864056094461444240707192073131723552317518786019+1.2936594103360560347712830754058752901558162976818654911441763634410824080850484217503378009006551058983320436526292813163905108620168050096582089761992208438249459625620995425993130114580565558009046004841712498295145214543548895612284619017974665169509155553476712735306394472536205031879901044537449348443645662602133953364857313112813366695114323912080140240893582616417354608475438884244405346702440235208062381430355349*I");
	test(99,"Gt({{2,z2}},zz3,tau3)", "-0.0278238185688139776777454362648345703598303406280794302746627919453018631712604876975431399785333224102110870635791283316334558460043388148392285947953733384267064095414594018042807936227510247067252927112331489028549797040507644601119506917909708671435073599750420860449904665305526668884501359647741024636347757451666841915183400351492713765276253029978999658134330097340146268388849508005803387287193552445601035151171271+1.1313012789255213741392018385061189254898708127070022435157037132748645144266667800577359006971511989766350381066493215866643712019592230908400483597754297022477017078636837283897814480420047145575232132405177763458343709862616917398542990556932684579999999498565135285492827096365261161661456808999590819192414368162461071443317338181539408489715821810854067921464414060731509454039446367314771234109380528211023172284386363*I");
	test(100,"Gt({{2,z2}},zz4,tau1)", "-1.1197280062171594562149925771266074721018598595863365362302275296783003051740528252516976233647400890695187470790536043197905511539729386538568853108421817498031088630678537261226428657839184849422039281528922645365036264788933235404292214966276264460052297503052982527434571040824116619154546078862353421841769489812684701591943572234649422399329479202434344590859263888621679194089141471305437600756165583271649091317860097+0.0243508232670743336498522271204933039232139616991039572320411907659645656298368286017769698348323734716245166321702907064939864488202049798190886694102744474272940064130155690850509740212023458092150996483276766933868734091525164132221147568100675646059290151305724489135315510634465968344345059273552514998908502488367687449537335634660303602905188411643703153102053829851987951400301629279548838059490651825740866978695815*I");
	test(101,"Gt({{2,z2}},zz4,tau2)", "-1.1185945450073621750344401875083289870038925401503268724452887454570568284578043353733924654953219172491674410667499206272182958209661535334868421840425767804130479755196095439450345769424427136020088871229118712081151527719495572586596433620149846917989792819305884774122423891405728735866091586096091309686626534894837822614042253166799194649757637045924909073636462880004707003464949895306207417462152427998905703668597505-0.0254029764758069029665475599213295740330783364198083340879151674388224244122929564771455309174510652373354871121232750026513686360342477814369180617992446087694983381347366745713354760244007224041323210859371764668582567562802004360124458281343257993677370904456394210529676902519178891006415794970085058179538193534990816011065641374890111574226254671555098630065509843859060973896804723353297507620741932040475120725380413*I");
	test(102,"Gt({{2,z2}},zz4,tau3)", "-1.0633989126788630549668097543566165838614297641384212463278317871500536444919962065731187239341418619637733791295491320885058146208367964357374920778888944218138278603839971797621744326439664135900754631961485063609505389034497254800284223207454553445947771663608446163673562996084191544172653745550828660988284678583459092073707793519366302011632096367371320884455269628293191673075947819688573501078004906019442356711275601-0.0062742318590338857657421821420120294125452815236933023099001976226021858033219894189814523256474084912376046953761074675021662987455519520262440111135615824329057917510072410194676536972793355551057253578845422889175559557167655276287072993644483717324859841179051747783263478684365602078164886961334490026821791094868995144395510750770089300480001189793035650774202096538254516884143201306061607079011609548222349077422101*I");
	test(103,"Gt({{2,z3}},zz1,tau1)", "-2.1571007816666768970636628063907052814864703265397657757177640572842243452435217633583266826441895253743754588219238513275615969441016000716579312119571325903175035204089256304564597334547778920387305021720970881622944304383174813230746055087430253874888882809763880196308131708774433570502402270285108154365474762944000722230529848044360045291619605285125516656404645851217795756308565240003109142317269275256488307374393728+0.8372508987849341548780865245982091688844807588225251626728128580865495669116384087025943421168078054803587945662577037879873665535012911386526110075408901959268277141347340754560570305441366304273341927843238883171612608836015989490673162845031582389538250764983225336850808419365143504505924503412560372293265875356696773108435151810088553395293027720925583922126630391567184374904498954672515497402169582568223830560595171*I");
	test(104,"Gt({{2,z3}},zz1,tau2)", "-2.7815594337967571436626599364346867415114884717971146478370581389201363005658890821058269987019752209657646841272405176232038297704008179848619011492176406736097678836125853933737146620394576361625067634890206606351340537759609346249854317567179153154378665388355770769697496438503875602558007829234660162408368325813097191354634396952888460174144474007317237438952619185033818772977398151909940185932528184729612538892184329-1.9916487877345955728272251418559992596031229325622124980687049926445727245960727904519988186144998643008990687791450323577794392741041130552890128168758011313209095685008758352386004951740491282365825624611705172297668765760255383823699595474453616036697423538631174589305664594644208329549060402542729718567304666540911586832744210274900879178884425189921764508302973980851325767175716833465110116400316053121410444034548206*I");
	test(105,"Gt({{2,z3}},zz1,tau3)", "0.0914714182014122485831005399200810206030886979759782446080293487963484529114269346377291656578112355556530055708991836681991527380288590266267565123788875099332632711009932483234618207951947586301885383146688265751751322439158901941174947690827986521856684956605618609769029610601420634345709936275309871159348124738787811962050551807776875749364011064484264560965507609863891946725967491637045160444643361814494734450103388-2.2410097720181619836131833569212661158497939184541851023426396625446249830377464875839850462115681651683770531559535181412063593254827775104881950362307316807608353958248544936143970442491348456202100902077124750512959614376443236576210301703316569780831090696000457721630456492140032517229090623201583086321171334444028411520991457818164110218474278514680062124814393764148091726370244346900491634129571197724509827201731597*I");
	test(106,"Gt({{2,z3}},zz2,tau1)", "1.5876896585793997631616495041042277388892766196691698466035341252766879954894453261707933269536129960467911706176731891068136339715067832237615718865229235362740716732572220934624875102847531320547903803512323134980647681651895269572613878156942513276495755589685936016026939579130768267940118894560267747095397889893258477728769280883291572011661763590389725086832856092042901633670322544043582723636392442705616089997123228+0.8102643897384030892024901621159206398428484559308210962792153002827883271290843984993521921776095258118233334677498500783161513448165656937333051327855055625606644369715864913781798095434459451348670882043147852224346725416621524965545361713105608810480596330209371677193812009980508504432584916512776494131756689584516704221181158452611862927392112528427696399914722384907361191100322322012337962916427997950359552374113256*I");
	test(107,"Gt({{2,z3}},zz2,tau2)", "1.2232328015235805280316415299058303546130210674080683076239867132993708444122521272985684340273030517663909663143097698531131305462516737405850248788477804230304790150789584233596004987744610215470484965758517012074713075246302162858199386604453420794812789907653718763973071221199080923786790786182349675242066461213627440256768817731947391927940044387717671853778924823224267534284687372654750897990309915399130940663424907+0.6184280998703868148207798552500278499181563707871457600523014353350603777939491803664196977364104592080700236677230014394967799771094990079479931372778123458979507564579171089893929724455468048532380542281994812712287003672843189613274397625005477390097496414112633910799013824310659397325124190206618582051969486470226861136755925413265249360246115212918065520702131820888960517790178466518103599807587134808762006938753964*I");
	test(108,"Gt({{2,z3}},zz2,tau3)", "1.577513950211337601543332806770098992094001245040218904079008626766847047634087174340965610002482568017053823038987523567229688577103072068707706732302670679344795407233944182181027533734692946943763440871285374744299268514604525203669039246785484025556520065026707967458272211095620050695336890415451033379581425082704670450948052220830723866312993784599984998697873413170173164236646763981275246665220919969033197924526204+0.2747339374966678563519897947071874285685974784277737492279982928915707618458847089504826080715303070014501348937343396005872249116399532493217572020755274221151035846847094133190475268938125072898346220078914181615632029308098969099572798319755622068672446248289601298996665539868407903310901889554986289924367425721133602174545266384719804590067527496298671432844668311790932144829034578328275465127206538892054030961748859*I");
	test(109,"Gt({{2,z3}},zz3,tau1)", "0.0541524913255113795529041655737409507252914631275006942803135863544380348922571874354898757544740252146229849166797423530698846815968031772751960754604362750938971614007357371534052120178791366629980987013904843573674972264783064806061852215125966949521698848348342898825196582396561559349806472381037227585968625640445631475608299302215550844126741233414817984136936152973096772114219178884161015283034983077235189705284964+1.4084704215261464866015083538408580580787477671291471082023471748990976607198702384704652393903423984247950269470844874208795781865915970869059592053657322794724378287093681526101176121960706907497538412622162894303899990934947951209697682968961413636194291920592836315381358805151811018941448624708101221008928570650645902243323772348826191875064431827455602177067529591888892319987235791478518121369752378833194945772687861*I");
	test(110,"Gt({{2,z3}},zz3,tau2)", "-0.1908901906548752422550960166090700490373854245056159874325795149854641684630395890146835973391479769015732965175659210447770290194242209648661657536832593122808486662243984482451296237481227673269316303815395555362314882671294027828550242365396964202527544921005209018478878714364565147383090633005203258989916690768138429849008763611095383629948280409846988547898410588711354420544454678898661560008153360001414511973458504+1.264593350047810234530349537601806617750488977560412861829257597731577659888321137034096028098400733355330796842671960887727092809766174351131542358248817050210393976606430339766696968213860565259367376169239078366743770057702125485573503100938543279844473705147827909315527636182331676750955199875250007540210796995767968795617116165778183290425279781520992696812893305541013042065744314454716781632482671784561875319761723*I");
	test(111,"Gt({{2,z3}},zz3,tau3)", "0.0672478403352753767532455803087313651081344505725702602502382599405371304019307942481869560450972333824312417295812904231694719899710625242232277246142323075153816281674825579306098799868670271637937799437547485817545845763412901857383700394799412726881273257143637326355097035911222278102984680375006773074415207000067963443668642724959370275755853584135722711664484004658346556648991279618635203812479270021665933452705693+1.0356575225680343317128679773488653089268997955667126712333376493998603002171392283048352192304670723469129837757691183926907463527659909039196637495420101032476231086947586181234553253840293963162490285014635820524139893890136471182970050097907681007927684161163119771266029936284645138801316879969009892781662115163505495489584621655443540843400418131681091261625197898858477018434052037879036536891902155720740839015089572*I");
	test(112,"Gt({{2,z3}},zz4,tau1)", "-1.1949714537531218591702781420214205242952538761109863069293833409261675671365965317713214592000844702510905323682165342850879918595067638297356231587586917505774328284507210598150964198842718010695463605381482488542389146566316109386048545419977019342207968064469986238808897281069402261860968551414110340069922942085589229940833393700258856411838670653815695084551254677438658688698517758120010481093782649621369713379259397+0.3510664838530959473459557621517559150589730613344784878957080855581931490540226427818646679066820419601333866585439227220769089559958244579633919455926363203156597154993496491766590124792017053057511637646756476101199393982131642000659875569452542325338842050821182013383071624437435158377642056707627058390014626767459381392577285310085612991753705488347911049451157797803211365565311167864804397610884992635004475807260376*I");
	test(113,"Gt({{2,z3}},zz4,tau2)", "-1.4354320140758380154798663222787658977739533840933847339200115750036920419395630175846755765913568937749247527847085897280748053742715051464707506442191875035938781612254115301100388484058021659962058329168572368159452805549635657540687000011923967291218941434103668097533690481683783441295049543775816630988184133632100671835841691555157839800110204395072674980073253649375352293979413344048697717235301433111193468852062273-0.123674317553794871256034096025158320415557453472507402285126863309690374661006161653106531645658318482666817223740340241693407379010509303024061571292868722540764705359034600561247586777210663895234922525517822651527794858066125146483805053792223041103639900167340429437229284589658746690258635425917819821089588340357918122195316175054591351536482085999746174874843074261802459266353730985611690029307694655091384349905858*I");
	test(114,"Gt({{2,z3}},zz4,tau3)", "-0.8107639029158812662171861259010357135285119046694040068508871733261599997531139872313416071406022575106628255181626393272417861198158537140726550922088595272809600016854992983203893887067106364890085169142513155298845423602281603351093006746366375227349738221495587727913230874933156250632934462572084727086414712559880562579359109011637901319724927586925810240322145376221285910560083371098027128378983260372062219531059009-0.2400839103846832641191669234288138963761547479724232435588045294838946909974055539286272196793119832853980002204750653520132148821636475015692785556107006618166628501642138674525033365551912844761468042534178839611649255626565523536715610694731638398440900451258437650218242676941512782426551889253167594338830685935481821368617065990442890139273610375770315820404683891655238757942771296783938511024614118382437843934558604*I");
	test(115,"Gt({{2,z4}},zz1,tau1)", "-1.3969667089160137295744729855332707402161206680671186369112963218021999919757403098805082154353524203198092750667826850990082545483597757758261029881822458313424954459824496601928062354856466933673244197303911406665932449521490483347232821124856200760469225701546628321499148835063225347369435882690808932614654524725473838518753138318228766280281961280345730007264297477437898259883627891417841188566271303434121999942563074-1.0727828583850721740155853171345381885638632809417945667832790706780297719752422188776148266624877766688808105725189092885728573818286895173124349180449650072921252938964293470886133079970815508612256155417210273066690277367774415687580207954605059375025131807181015311288135280306042961739801591892740966230777606590966553596752728732275289665223860857029802460468307996142225314884468990719119364834601048456300480256367019*I");
	test(116,"Gt({{2,z4}},zz1,tau2)", "-1.2962231375643070739758618986042689730622163442581564677101909444667105657725348184859669238894012421365552078485027034805813182010596467065413103012592480332382836481446541175725076785832553064622820726313151619730955297959067520734211396395651561844525760133179708870229664561399184052706127486102278760839088601809230987818935667247505142193469827086835363368857941984364251615833843623324919071644545120462660850917954974-1.2116854652821580872670703282830861037377960131046624363398920214075496195365968276197287264430839295703572410704737150264460090036749062572290463586030078174906588084125404670449227841497825372813788240685651255540311044656138384566165655180676450354095726166150547389154679684991467037450904214638786518869460802293820828118355260010758072754740313174366974785116728037194993260212850743364665220633713691070849619255605123*I");
	test(117,"Gt({{2,z4}},zz1,tau3)", "-1.1751088273814939568008141947647028503450582028222458210158523119398220270378348624158544698610269031804891825890500385511192840059649630959837375860139741019935394183193426483763158004537050283542661693302666320806184638585571463326370343223433094745968596143960046860344122644605052024180944506638811106321507149684085658285908356229747560788874598163604496227012578477547862040045639009287348922534567609531433109666577657-1.0498968773104108928648679862076815461519228884692594531605881078829731045368939333180473866576735174276772799912825520483806270235437627345355212230103654873275630527778998840140330118311946751586594158715684710155867798410278931429207733955299625335437887868567980862787671656993316032161646065841075702632317040822327217362442769960409266022333159461000359064776044940943428045836896341810705491803932536095905610211391148*I");
	test(118,"Gt({{2,z4}},zz2,tau1)", "1.1319919216286578948412622449518918816648665470175417455260638148407151167334272152507498361118635546043121821062928338881253743244138581614756172940814178328593465466280743986197169223596332026312305589162476821229165992043834679071882791015634918241690778999302493895803421095371032138054353168656078416237384186750280302409480412015802086143785396293393838087276599054974820254204418786036276185305972997133571987259213557+0.5154291404144511852755668017854817012021810254293380459430993184099724587175896167422554853522975022417274942420872995024942196039078151371695796765609266671964908433404417744330264870596378791820203894009882646745466324901363060105390849120092807307605483835380076283355305436001056453290886928535395150059063448309674748539161583372445228535698376734921110401266484919135695641643872122036368067173833912217846872022346462*I");
	test(119,"Gt({{2,z4}},zz2,tau2)", "1.5721610997075530409012910017291361243401274581742053021604433587407972383061745967411168274777413428356014171063271768693441224664965873009158378328925993564874902058998752170958112719981901774446191679127548052957318275630581282886177864612675205528979161500219750986708689174732131925820296856599247079099634905353437877094604004682524900660885112952448504800056327059775504065348930275807839251368314954962078079566235543+0.1877583896220507250823956182725005400768028666204567720683083272216495509557577308273579121243352430766032949252357465753266090952563370245661117615131121693921155551935002170968083134337331715321348194904598309264719384243973290067970504066315214309410788167931756199311699624060019467722062081479595078360045588978151098581341249003557633976552448572174933351852850916265980957492746305487772112040641974099323006446114525*I");
	test(120,"Gt({{2,z4}},zz2,tau3)", "1.7627930742970906026197626451938343057641924260531159734089202481942695126377724668403323330527426984761213395862224429141703270424427959160845072261683935992723707330993777576773168392140949831532124057865768586136762283828331603322202757121161516139015028817318129273879481545526297774424259547174649070496800855696589752097885368853057871562380746627961057432456546027863798326145345237080705912716834874053063531596057743+0.7915293470351957332876447775175080915062556433202191844073212629388533775321919694989018102500578916428910138599311566477319443620184539592201138452593633651920723399666739614255970977219638211863040724304595903611646215019762222255019720336462296714049193770359716679254501889658424058458446253425254002706918960546805444965989717852174439910857031182415582446591434250848755611128158628760840335255915881291846571985873264*I");
	test(121,"Gt({{2,z4}},zz3,tau1)", "0.8919480097150272613750606266024736844283491571443933817247578963533507036109767153360960302328933770576008049409071820692980996363725493633761666729170179020161581642562369435774460892343238001540180380100810068451538854481149885357172909771496625673792537426362510698462816411311452835256338590191639201871111361170526773688531945405320892795260430369090092708585799806960656218232652619336085318379371569387016438529575677+1.2110729665088323669242234386085710062415304468133553140023962938117279902090145608041729543379385297860746843386711220538638000231432342402465043341258253952995748741683088344176114081654714417032746179839569339516958339031878095423016371661477364698826421061257078083388881129067952076036422923761998217911635997163646181954160516039788683410929471773728630271614547647716129802427460280661428545289090453456705948968588283*I");
	test(122,"Gt({{2,z4}},zz3,tau2)", "0.0017464311462501748095022025636444553862548135847059125002919459963927389003884663618793899330045742289749306434252970336181979226797932160810531853682951479457773208400896594058478362087080383553275040653913368367068825180527748418573424545036882784230090576351804663053374021868090578470516372586271170221731088263543071883478855814851570529495094477775401790523371262672832492918744267759600072543402572004452379781143569+2.0441457563578519058373070843166259165573381787896436659167521374467474633420673027296319654216000376251148699041502827370389227568620678705983605503793776812418740072314807617543294372916577295247094776788409077903582809215253528401199269068720631546956330895611391954061910335406086779154035615437559758575189439154964032206347808808887543187246893357723469963052981467761162705257043453562278742125689179574794113347977126*I");
	test(123,"Gt({{2,z4}},zz3,tau3)", "-0.8397319329115015605756515472516818779049192762932627843332576948664666098443246802606369646056613096711851973529660858520202843213369454010366051891033868950905043302051225798329748722049410630221987832818906873635265054122260837888630581839791478062178518101303456493115582349129117294144750457938050672947299642542437718333083918370647946769356567441951110561202868058122497089661464030255976970470107490649095371022008242+0.8228315956025333265488086009687099873419147868749846342980297285700842354560857617478337264397659659949516571696739450743058937003930018188871755462621244542324249555687905990290853202673333740855015140316975823750264395560137623570147648076732073522104645094201784274901863175587944606602306153680303402237278330288494461675805949523563418112104504122775275290418569948667807979176722459354317913304909637011004356937224632*I");
	test(124,"Gt({{2,z4}},zz4,tau1)", "-1.2315815141896560708859488517019221160254521347576259992972495070346081669968699559182003991296139745899117130174202109679677972489846139046462748262740954794307653791432653365823943320083393826306525933174260952269293681406837843053379209082759061326033786177509737877692382307782115802220318277869866966278043499293864005174814615079994656576693240876440150061146563486273429300025001076509749591168417710960709236767975031+0.0966915475677477146782750573177601843113197898078760773826949881675190053928752818464409421273315942520112324053584639126348683844930833849317695231946184103526952693801976082027152325650521888118672679784995916511378631742285803614128361806928402166451395726853389928408463126439565486492263605096074537516495286603057653995734948485681652073445129398662150797747076477936464327870848569523454262725446810677881604775606418*I");
	test(125,"Gt({{2,z4}},zz4,tau2)", "-1.181327636180236698481982312011612194355752071831229115181032283687479013786966442149270389173902906654783263580584937008047280134427958421239001624458722827843019182151566887333591506905588855059347130842192229111138150172154926598405811484070784865515081167652070643969141844786673802934449975558054088852029176641473306769572382362596109227798668082702110820300755159871205573451007157536302982766078423312852518766969456-0.1434646053303449203964548359220810812810045454581250795829899238241846064925118677101067108618826954758058905409092684512922369823475226078855016938696261419143503953319132570995045255383339272788432320069774663898362235686772619328037903935112513546380042238095876083103336368280079212140439332759984838898851403980433346463316182896986785365042110150748856301561432081326975313204915982704214506586946153355207782326168092*I");
	test(126,"Gt({{2,z4}},zz4,tau3)", "-0.9272076030329728690014733389183044770703800452789599400186951989835445265019742943794441919415209428510830555809982027938549746627168769948038374206806061585713022864074742544042569852835524868860297772026177313284997627498443698837230534065854893648832299651393985623918785353790374861958728886159788245794934884169658298406282403156203015760018373794614179077145892471930931304799776716263749193093511494305247278827362261");
	test(127,"Gt({{2,z5}},zz1,tau1)", "-1.2035849040356986354860856322690116596282832515131421707051636833992995612652326338995632645089423609053214988042607920383328560728493419220760907322727948352657130465078388595874186696923338991621070446377808589791544972743972115390273102298354276869083886100637047628887101371142122714272386662767639306631145069363956003685346820314892092182644020304113263015658078280445129236359835998210956933194426046046366512584172322-1.1185504183418077212895896261331055751684707987752511983878597160409742940392532679002511784768217821273046119140405195569006618459645403038487750467857449258033660261625364111310849262450447943754970466692015183441009230705636492087038759108790048039136358396521331446627894499495822556716376633394872691407303143120486129990971727813232826146398175658432667132528477041965577083680669350917290040259588203026873908224707283*I");
	test(128,"Gt({{2,z5}},zz1,tau2)", "-1.2693471623640611183524371836209991501818469271105829415418330718758764432910383038215488763516729785747946942502607797577817699509503394146754028464328331518358771273243181084005315690087540365113055365970757659027950413480395434214594156868070967309387703064732127126974140062478140625198608925686228177605063616601448865561107867963202581827294395844854709410330132379318347349852052218900278127264207725006640821852897137-1.014956795402099547922922570894206582954909109252661296422105916040437792826461089969158471071285841120129262049980946391039739742879780522052189597579544276682479724713582174597531654755615330425116352153234456146695673074965242509526905096245253847459056920578776273998610783837673462850481330237537313579551166243251983628189344612738839657485546807293393291934890238367003805670175981827431724069196043113657089087554691*I");
	test(129,"Gt({{2,z5}},zz1,tau3)", "-1.3604840858982611874006411955933760377566665542073587460871045600611534886559406627579549850190003584328382828582524768202460597689811013356903364898022650499412429245242936426412281191303518268625914341301514735746431702787159057624014989068690865384644653747238589697125926187153167427714751166436675045553855722125344361003919640966350442590035659259961221893010192244210514979118178239257293065306675305008914930421527193-1.1260290327669156100492716004853615990838222838210441637608990023013146307761822014898553430514732885659203798891398383114243156629766826550095242194948399656371592133154067656888201099637894817432850501168577579295055103500073622952198753548496874320175043982852878526896437102063992889941658647182168614782656529596066509586445749409729555916096959932286753405275748967795785882195571524207634325984437043347740646268755721*I");
	test(130,"Gt({{2,z5}},zz2,tau1)", "1.7093498025633690252029036932604469742502896244363226215537114893800627934717143416584631973814654390807679503139385945612317631802316582312984045122685959538215424470169351231556072172078577608561295379566888887316564991528823321673973987144010766518240535794621794052565737295369406162936766061760685295214430534773470931999201633427386461499524478209121557663739653485694168678016371476778155759451624106677543730559115642+0.5703396672308770287328458732213757191285582131062998814591512472854422562531639749370416813603409296738828403807721597327250976491435451472334455407376862128988782398068588594077310716290975061512857380527984888962335705094802341560348694527541161765420259462482467468908210281931248936163213060571087959953692735831706862572920422888999199701878933949076161526609450910518091808990779654533090434697305534559449898126636753*I");
	test(131,"Gt({{2,z5}},zz2,tau2)", "1.4548740168761542961427756845695733990019555501626880428675242937310164763649618653728570312347754739687961314454497690822864206488340875788774184899082555642856965694811651594091914553895523848137479925881711981468089421512919406198412488976964482625899833941211072847043434476677813057883382707957826126848794113016052245413817169369491991644482418114594832354715500374225856825217297421668243349875564722256579686766895142+0.7587913636554493242885129158530823796553799222433592436988852934141008856230072061313106933119435411342253003615824999081567741619381715691787582160190971976362599938602480822080003028277018433007602907582654674123422956939436231786669990073574096203493314386062407454998278313203412419040392251123954993494749460179920834347476922288990926692104104056131582159383514245695835903242752531485497311789760684769302194993859265*I");
	test(132,"Gt({{2,z5}},zz2,tau3)", "1.3303421050068492806316725163733786891209574103804048084522861246588515652317227817898687067558279615923618783934817654417952581840058968236594690227177020823151529679470716587970414126095753905960914957369912137217352883864458386587846658787660032284424399190538876175492955547405011250578852829125857756265845555455694287574059028028728048322552266940500324024462283443984910831660734937883745029378278014664588673542208917+0.4047855082531259375922073737733189639523943406215505528062893564110459500527928775632361836565520402555007509269974302945109926669244566891712935616134842117074265239216031819876081040989853267608640048136755990650802707935754150296298584578298982610691600899708715439055586744092984749464038157575794240988793683959962473056866861818773354343386389627030474887784205765569887955367772958046731797732329769039934442740743904*I");
	test(133,"Gt({{2,z5}},zz3,tau1)", "-0.51591295622829728731172550671782358266793656869288625117884315056256030841321576394831161068449530674740496816409806899087928190501453018842083951193110251717305762095572528973225563703951677077584025743682001446223808015930672146947374420682897348774208094558798244153084717096771562433562731078149418381797013106308890942881042427064577181020567054425662060821711987293958708177608111400605974531741345041734229752128567+1.3018097625906044308735970751575816050142515430171158067896292310835950046051181039395921441511510874952305035638512758574972709492061572924647232517359957229260151255973699117678838683810667857634434011915731230989089449317636883076719988095503194055794294868649812303116601972644837898630074077458572264384128462260649381816396087951227358652119295488253585625439870729444225119626615732268921503820954511799850084673573395*I");
	test(134,"Gt({{2,z5}},zz3,tau2)", "0.0002041287211730073252918858806420599765687521312780816574564522110498739862677456295486303430013995401375756393258290477728462700805809796802889057406689784820527446411800637328283473680605698881067807225175263245967114484888197736310859236184169657147611325371198046550106227034775030485220746504878716948771598467270886747885266774326495023924874821073768551160112601048227957276512230334140871034507575327870785842992618+0.6691455331941259037160156687786493612040153440008725290965867227526527965181020734277613858600863489245002416873239233793547862957705404896154260695725657252934627597192953591539657527186073941274255378039875027005159228878182770349263811558302022503249110529094505700424023008524348449456496275796730722719269520232736637858163068116397194164201560728529076812220552040244307140314843329617605358750005459698081389257236742*I");
	test(135,"Gt({{2,z5}},zz3,tau3)", "0.4340891704520723937425205946357857543093574155945175947964763888141797942942952425670387016751554428750654210131209356672258816772126077187485711200269304366157545273155805355487616352737942172493529944929472211526903460051764896058752481159100641263574086741455109048352687757396101774471523373016289122299253390996854546201649320498086470191069962370856485834632656662071892565781174950844362667248733398785166795960605681+1.5022731793133394023114814126578347749027589322806881000221361314435803891095955711039041631036349600866122667873161114725262268973420485683616602916061732060259823579542306427199475837025562172719204072209872970662423082934787447313568254311253251477819796166080088479907655604849599350360932822370652876249402650565569143158512714557179552476195232462171223192380391614002362855222554678454952080332057736142362591133910897*I");
	test(136,"Gt({{2,z5}},zz4,tau1)", "-1.0140947121966362012354349820883559174976631267191236583851868977718027636961960896127260794034882620164290377724292569656862288028647199691119967040009756126024777040054808827806768501601102415271076916413918540072298685000335504521126586929043230697917267768011074920215734542293272711816292605237054724731140716293837982607388180744747841047794623082279121096979139699799696382554247111163571683675512274844755817668959267-0.0600042187628478475590907541381203829529177712898305425334011842230049895256189314024329127271872943267295487233456934876989112595011699067856260763452436365329894232126748302430343287815295231413434724652511760591544537467557095751202004631230669570907580121031895495854504471366738721863507006755677805604774306382559835105739973929858078058394405117161258551014671270247177773184109385765613366584334502984355775023741217*I");
	test(137,"Gt({{2,z5}},zz4,tau2)", "-1.0456592510894068838422611815810764861280218844554215355711090750405649531079997048266635076475168608273560213526831422808039529729666109762451953662252491147130530525544405139046801344038925344732988109193731219254433241200490684079993809137892005379210837734219048236287238463190163118622234742502694876045125812634792917311004347415143759000009991576061480435844297331988955162878012381709879461525928068806971781243391061+0.0883753840414915163781222670652967662487665123625864392288470452389184535496661951578033834120055745389602962054929408475230929666342445754657731839599624534361046291936935419063408134483350755955838720546373476151687702013719321203329019528761060075431697175162130202541700765608610214789987383462482363757083230674059242514116639590238986432032030982906101391643092561805509232589838066744896703879432367441600063743582589*I");
	test(138,"Gt({{2,z5}},zz4,tau3)", "-1.1985528325232161779940889172403742721311981505191603501782374600343985469882646701900052252398723307871191437843313177312763626818989248412811450154469002445967812640529518171474885655029405731060981574075880597256141056507755207493162344003285172335326344948560805664718744230182336006312799479759858311370359210227669039458821592862789418741670821897980021300598188079402502070435938802309515266363195556380877719660058297");
	test(139,"Gt({{2,z6}},zz1,tau1)", "-1.3117215185291583311771044381282768315752436421847649416302775634659897315040471877692634608765012418543860004832939167562843540045928794601551213543449456324802999415105701018207615980757369927467577031251451105734524971572030588147325208287890293758520460884747297084247027556881593076784350006773252523616440307659611449633257623745108902775009822969806743503066937348234215160867865238651921086677572408623228178846298139-1.1014187074501810823037116260763551056470333783927285754548925735620576856843075063071023214907158192922646495983360271737686110090905957432907032560034679724595015752108656064476726711050398595801462363624645259132935597865819845002763476749705164135317228118319522108621929169870875483221144468338698113127335590691277187366672004904224152900236828130357481524187047525072366518675816226584666973278741354235547625645083639*I");
	test(140,"Gt({{2,z6}},zz1,tau2)", "-1.2754074755565238397993275426402312189701265412581625425402891392791326433733607042281652253725206361279455341450415515653271619977539632603932426709756793064711399480366808578462833975275528993709007268947332354934313128549780027719390430721631100120255733758642188156275171235646078287767282923618964551944427295518413667358377543816599198195142317659902367390652686376584868803065507644595879095862663926873642920560532584-1.1249429584509034259742573801075887819785173670283185038905177865745693773325945231098851567002087913257771471601470248596345044332212485107779333450951691850542239828737426694649562699928079269316302122070935461401962353691311230976625557258591857922654205893902374606958712978303496664353659551369309136944891016997354997303345816533035219221728583402781106127689410456769124054832505302154716528614603375858375984569136803*I");
	test(141,"Gt({{2,z6}},zz1,tau3)", "-1.2625091706592581883540549738651346858964486591478199232044936436087043017317278065123572104060602329483662735258113814100915439839896463269023055630112133268350172916748772123714824920020070745216028915077274077800201104457273566919919117413165941140437010312977220424137356132884903563083259410717519882810457778210830233991835968034751381460526726303011234395139895908085929332318401738196309281889135690148406820238335505-1.0754948012656548964856496352967607906530404222396762564446394123456609589131940174022181718732154639306878488863563052882726959293577581487450627082120842112937662602462137106167186430659883520133304135246495534157121274260629876842106940833248066228128668504767137170500868134094550304789458662619357136677988739739172636946544990732502217011078166203168465643429024110828442449912343033017411967610081515185822665163602397*I");
	test(142,"Gt({{2,z6}},zz2,tau1)", "1.4344844676653828921620258838073805741236981821264427929750278482047554521718224195019678540996131583566399536444072967471705447555202713219968309284815870170074987636365332649024804622829135958002493497220036603523417829377494607573245091897615748397048712557593679523252151812781598783104687920298636946131529598883035469963314198473920093332210314844223257913770579823148714551535896661174739224083823408498572286529940023+0.407201965530947264921024212097523000940718316746344091991160751183951142163203771145121517240210996993743753981578107471147743161029007007628979245895063257085586702304414779733532430675236421576751060508437245546269405130742080130960637519425892437908303003204887366286994008092088207071933665241999273100730858362116575320296725571567882850508359670536076809839876116652539858676532599682746689884454891696987643158158928*I");
	test(143,"Gt({{2,z6}},zz2,tau2)", "1.6415219910002681924325132078992906146661716254190057694498711950965739143435483778247683923498625886580491603555280087723213491161099223761510774614121795647101165145189760831310465106419612970077131618402968554257131968134199664419535583415779682037151779782223987981147435726270161569619550493955071819311113721926868248095600504371265124068275791114986254173046855050322755559069274191811323178453116678678718993008225174+0.5030513487987487076261349931462751864624422868611573895707784312045733260470306387364450086703876182264783605614027420596404910218855563051954154279531196726898315716590226310825034651753729936557011422859638410121709176566080494636920231467205522955612646390740728503776699822924953913619110377960150912225686263703666289382783200864616601688292681692403688364113573932568278365897852064346790365178285823720343006143274829*I");
	test(144,"Gt({{2,z6}},zz2,tau3)", "1.4624245430517157085421233013790421413515835140582200323229456490556635705738633117017597714060481346557788627132195221635448501287647855230103281546835300901899138776839809992619396042651220501504145667396515171185934299058621921247138966429572050942936171198150723183653047742732089727801547458116264334801104210105716468264613082917964947354247763037551387816879183404204144981381052796120024404379500842908450381417475866+0.6975767008449848602194998043771408541254988376297647375002142059794578219391745265070086296928656612487034167640623804737943878543529948276565845481252339171510790770193773474413091042127730229572024249366063145640204525372215558147155393378717467623156281773099416582195542907530977794672961653701795569601766264205850037516599044473491971083308999058912377803904728741968762077375645493501936029510430698310274392247830388*I");
	test(145,"Gt({{2,z6}},zz4,tau1)", "-1.16085005384171813772971681320840451208897119345083970631035932185542821169004755866903410403615856300045919897269922388734863961054467823512597418034809290946551092539602292305936586561008116577039996333484945510957672893678041283239047432015881779463939835308817445136465599772039815203078162445119675424062306188594093899045290841002490191830473380636951915710799728092985788049114657071085444973903613850680565457436644-0.0231877767891144956239656646761884940511736356525698908191931998280989625778843717738617398796177048883457739237264416068567952229313495599947755303125244574460201169038562036362799581051532446970879132315265245492274327471877167271869628653190244153517125110787168154941164856949406955257571804007818853090705468033232444256368700944531071160952510167524846383462274325777327396991682529409438032620509737097458806481134164*I");
	test(146,"Gt({{2,z6}},zz4,tau2)", "-1.0670746129731723715451454187407031498904834977413672813901209211886307645113159618702962912883955148643386994846810402007517259782952695756462396056734089064689530235318578930223062124040807248307201470091879519204473112195520653105722275115872857134959655501508369943211050336243036065139659605793504417941738622386319344700348599398376310441877504006590534319995642242252760425297188967770361664396372713374643342125148982-0.0620235075285060107037301959015769692433003360325386162650771608788798789420403755639864657696663948161536739284100594464257087334915541817344336352314583497927726000397696076948070753127482036048067254872645283638703690742957784503440193716948754273957814455323454397564787243064193197121198553156986040811281877821753444973359985670295824939377580787856731321293406746741510194936672225928052857638051582199806619239985356*I");
	test(147,"Gt({{2,z6}},zz4,tau3)", "-1.0580006986239256897911284551307792656992098386949782463133294919725537635509723494593360579643165880635030362761445388011056479642836527885860948123992223979241271063093576889495303431811396295589615791267529594540003201837810973436903371327137879222518274943067701272564768163626263309326081723435881772261163380908866538799469801002112985531669313056645073931920156816474101389966506445165258293559958110755356536112502415+0.0572451375078118722580282642191475581560449400088884783241336625889490252590733959375501072636726267505431794122131276866345825552818694466365291817513581808843699275013100597999201445250019451640111588346011680615712534296916575148264075531574888759075686806301548803413172263884810821011928104713265183926218175839177157725400363996640503824383631075139479107421963236732936795283312997432048929647972290767595803525231652*I");
	test(148,"Gt({{2,z7}},zz1,tau1)", "-1.397064135960972895707178592296616296222687649853558065564608640491492143459541349599761875770655531262771933689767073007660963521311149568964349884925551176606593634940856759966656933715335383010215769168555459484371280068238331465307842173080770910853998227996369795814284796513742582530523801621030823294600323459995166762422884765133358950675975924763964151447610967491631052221480216698540131335547174649354014266057577-1.0884058380714539250370771538886403482649479675663188499586438810258374024980006882096496275715967705050224120008114500073704723493384604447875102316722712258098476742087383399377688605238672305264411389616032376876901825883810806310972740668936561308339474106331827844679351850080582058003212347814654099386618432319985044280016159149745230715963986995680610870847435601889062033231743725671459070399529390164606484757140691*I");
	test(149,"Gt({{2,z7}},zz1,tau2)", "-1.2808559866874711479566176142656780530469842932557944700426282297691313275028092370820467834469129962523826192344313167099368351466658533320924433190526904849724990856572132955006218451225757352489511054939532708897307286908820334026515280368725616492865306055345723463649485970460180864698373786334464181578401907349859609251472680974119792752225351689311073646608706294071088957251252621194611275175298455619827935797723432-1.2105152048624773972086211211828381648866784687619948424122863101817903975818971436568402706403998179692790382826177614417786285068646177632038290292004951023740811907873035390689751553074904150663167409266483294481129056970865986723879295096491150980017163927384235256049213992581596929385788834949225349736584132848862363450500651305302700413359173576711254328522879778370075646010747762509431465014155412905820786799900462*I");
	test(150,"Gt({{2,z7}},zz1,tau3)", "-1.1839718111104166764917926499888488197422684129816553458149916369370378591372067817228879325864299247976018343583807525562354569575169824656393879184492998956822242735609291608753670344071336713509024886945393058658408704981846817813665469199351246218530850751033840344337007730638983679833074560580321784897523259497210509582293664818575242728583386517152434485040913930220336858951104631508692077399879697463520180286743983-1.0367264259599910099483359191074460142187777807043738144540526483915018574093204706714052365832629794475944395547827520053117995413933073917642730724464851371769953017570853524000746100839423817516814581727233873424951646343647008440468183501397735909594020353496120573149857904469003026776219132801773721627759622091930204163530594316202093277875901798086975786651792322782621340144655832609630732953078135885953165319606468*I");
	test(151,"Gt({{2,z7}},zz2,tau1)", "1.4425564901747284640080382774838907906099988443585106649994092191767632995848338081522518561521589364168605697430254700009329003714169093685880981837277532611354482836257915928581873089075863695978572706686968073362273242063006962369953326121646449034510402907068603730831350651304846011710076437975277369077802727939420847313265296151636814225890117463484294928320525483573161775275114150629294529621614665082936350239715992+0.5764695650397448187613334658414600735178887244733421888021123250762003449799543813371788537007726357440310890749971558807819176710501399206695991079176356951408443732995417207607088557745461822889464610721340033491873684735123933590450207202065032727823316997739664947155073716675500198600236213359466109457633425149658135994319602897928484613652504561631342105074206129391380786112113308986240481429430562677597137362436322*I");
	test(152,"Gt({{2,z7}},zz2,tau2)", "1.4726213451258792751318287386192415534598302179613655244324176769858859395908840689213757278092052008275322683883680706810454722429455645236945435575806767397878450312355284174502576581853378677452140480937857310005902286369886048790690270598160249119009296773139569986489213929888680888889853112622316707079021326241449145148304705831308444355322942515539264782112600987205803144787491911713828331432471270059614046975194684+0.4927508162319766147899521422066793168706538459963453810993034394368635727029774429136481296553573496948860103828261335798209431181353621898846141904300908012073234253505035878772301026511454611619552049985839602763285182699388446420875396250471644234402143043797738700578757863640913769279745223191711646882738415355319592064551839810120088992782162743189007530198248956454305495122301284958827064721433214289517983710038356*I");
	test(153,"Gt({{2,z7}},zz2,tau3)", "1.5550680690803970118289805731170761437265834054430124796700466740058530856383395845866129745418883306507350705405677986502785714122997877583298809673560495265639290658293779387032926439293629254553243483429581114480893575703142133440675361885291065476270107193016425015627325749436250433875786465219077480598688461989569726047669271534688312516304344364273682613050807109412001412136158269880449685505524316393619191445623295+0.5566758431529399376532508178167957903946200767817494686935944986019014357169249607110167804074305539410259986059612651975052840160601808137595186170309920061195840285031885481190774480996840775651256485719614026474959214527322799726836055804657184204038272504353254265927276343881846805370412613063493864253479906198357278185014881278487054071339123843637119323175139187844954163525075959990692690828156384145006047665424559*I");
	test(154,"Gt({{2,z7}},zz3,tau1)", "0.0812843785756640214449611123614353114277458120293980946132543047042775926836023132584680938008576427514600361315917455414250689418706774025991883454489426173733869122042302392576316927786428367066773906441951239239618460333382644940234024370063249921306332370736033412443752916214491677363561034929755744910366550263437797593579752115814758469990107194663866243307965125672932123228263325376399928831177699701993488190204756+1.166565121722307620601467298871926599831143765998297392244969256033382468459368140041480314312906066446537844852724342169684149824394480479603513174304994966615223193967921392652740909440921486356270311734691640807606356526247596688003400644478230660288092030007467202998789012696270984742778992105472401521141623998957051052961090307701547548384664969873140991843993508815908016209318546290189732712170767966083091222753288*I");
	test(155,"Gt({{2,z7}},zz3,tau2)", "0.0487944600627911453533415279699716520636546479879590294771659254054440808626330770445255504965297935459401953737180708754998688813832404241858443087893443275244798020689545042195920948941548285255437291915178519851850535245252178298564024544816667062030502838657915971737994238519439378358395519086021486495260022529222086919902560190443616554046487346204979473105965672053337086156687357092915845682584890078813274845825599+1.3091843381449389359097949413437244425502961653561194987312339160201621021350011586250827598935932184425361834662939986825703893575587628627897578320816970498780795771103489405747384579781700903985391587625072005864745125782014882404127160943637929000385034143790570676492352351529991330005794252682180271582763730294574680624549593156766268572043208397923821378776610094973255722414888496983781822434484030595843695502225271*I");
	test(156,"Gt({{2,z7}},zz3,tau3)", "-0.0995835787227772752420914333277594044039125218497978288558547572639105705450647592165542628464725548227772660160601694668414799539761901652262153483224184412538510947499198088089123569958877323244607752998815138692129620708488970109584479681761132699011687144441529866943901730261240389475192818962705206947240454564842241908988274759420816399202762831546543358188415126052359053728138245507760131288221822050251793077422203+1.2253355506873237698749710895878786421167859304624494447330315235282007572580692773339304162747416416421649038860652453682304420150563020509012245267679579203884296280955107426632085475505281323945118508711219384556253843286659621722960109661656019724979347272961899544136032569367881797578624908860035830193488849758527985949145844577933806112965236292953672019184836241499226821992511963164321552388119101753817824672301411*I");
	test(157,"Gt({{3,z1}},zz1,tau1)", "2.1045880557917007420344226734451782498284306147606539809468083726659464003065150056154707321784258488342703806969864517743486190967339684392072534405996669791446182919366688472266159108341629332482611526489732648116496832964930525470232397370565626609277261289572510326159001806770452316569353749082385093898371431812833399791180682299895135914802859749801652445630900410759762500027123289562852969111817726429321469452703759-0.3101447547556700613698347425624319962088387226649029022984627417781945811936338806270531579756846003110189067595043958929889667788991515219789217740870503289962895625944267218141526393937023297423345854848146264288415331972561985216421042155270786070623072897622077896260697643604306172276162220641488352963953049401327779532062229692609030032291598872732870968610748423879519923173906293654467486616858257976959700776065008*I");
	test(158,"Gt({{3,z1}},zz1,tau2)", "0.0205418743668228204157384829269116647277216926237639051749738531258905251851242872385855786301100512788960223257972356843356119296050101499629689286373897284395603824634669739683758634975889918185743293964157221675812012835283574068446788231913202125453623904621616867190271172939043371111059196520372955885558688322476606451546047950085686140515248329928629464462166916627773537729927794163914564033288071294978826343417514+2.0246442020064160670664978148506245495139566625127493228434923075552396182322309478805688408949852532086074474032164140533880653703036662154149752781651272288532751449101175183469464128406321429791682445197119647139132651425610694313870533729284745900339737517698161981503930886538942795516704555737013928546006424978633921242998197386927967070191360466638370358113982656979730069992439992782162025882253503238727413011929148*I");
	test(159,"Gt({{3,z1}},zz1,tau3)", "-1.7924795648498891594232480759146220431872743802928351073976797398861879116442716696295615618054909023751860700778074041598816950717591390470261788548018286462807995344002822559126009794223253504548765966584002220263514673929697786120449459645127872480595317671183532054006492511689053207066574606490972959068953923866311604064396704343497365265258418625648257427270568135240377810081025580010923090664294171874903687393911257-0.8745701798289811007716755065436991057461997150648556920016504444450790934787696872173683648255318635413695584790315329042743479680771060347204956406433646750285223523464691652618878560907566344927957877494959753643367602425714948893380613080680799060240283259022448474317700939771356407957307434413613286430712129018480345345065680217949613334724094366397831056240396014585282497641103167254068759291536971814234117644836689*I");
	test(160,"Gt({{3,z1}},zz2,tau1)", "0.2255014589884138061062711250836840557086078504311924922288421261685202363322375500554309003891592466155758466373765581994473010638921748985862013445236518124085109081243096392318895603178825438346939857819470465426938887354543699311944384726446443854887519480074890075059071439397856963922994150153856161734897011107650510247403804456794486094424833204477642906168694943243290191823055799284200881056364627666532474276061765-0.2316432204684353191625206529176648589285961506958489010055957801156305234165409995506774783798475147997823543288872842598149548844662635586804073664830155207011386762067674221302432437358537063624562192155360422975906955995212851791755460553798701909166593866957607777721774170361350669254972829594703493782440443078425792288181370825088431295849602263313730486172876767637395946048737184881785326778954707450288866382589953*I");
	test(161,"Gt({{3,z1}},zz2,tau2)", "0.2114884776989975164594916635353814032116054851005486692639240022939699323576853211673111555866953121614279996896012877153132695591910319074316807972168242350738448820313065198144016038649942217408710764486832591154847812384508314464928773192932607504461994475375612102620445026434512125718784710703448376360643908322743710145671361216033860121995888777017574441675340848187676206029002361261500580961806174016252552469166449+0.2458989770263054323707141235267670527953733510266918061523711275048301920817526958138560202927813814359376063139325998451335995705824477701776445677611430178369860735607778940281521460525868553537172932842927783670135383091861199791154316147044165172584560741623415646823732212354472663562401153171676468565250872789375556228956369923989731319120743001384313855310727895975595422236947295346019359603619001617030350936817926*I");
	test(162,"Gt({{3,z1}},zz2,tau3)", "-0.3246547134744954469960999450567780287034760196438815789165323508122636777753421663159315199969147841487897673032831075577336188132466309214342239343608775588780569395880898688118382299805834732399298092213344553906600163721915726929227292930397374896441799459619173187589385831641009086831800960122680984293428663104732841611359292730791621796832282481283878915930901574409449171529757111541861923765668366377447721618072707+0.0581666911661967290287145128799110365026274117750884222769213608943449485794037524030646949865610152532480249430023306101882003576696532747516594881551163283330319449187565582212116486798819692712816735661963494302639106663552594117098151701338239456232789774823294059931992019803364533267700806812095058788821654379167027868319467054915358030543771531827033477436335900989060862914892218382427777904334925777342167437790366*I");
	test(163,"Gt({{3,z1}},zz3,tau1)", "0.0994557219989644078913443682536086081129628818161300927099417031699290358487715209311897061255946635463817443667457599933969770515787651482119360559237392649576253309550257347390569757526122541773567637772984169125443659268900896712064147126716032796704129958673170155326351815437802834392663940670551483490019846010686975865293503358389956949805333954543842008668774845426000344872468780012753828155190222291281986357643453-0.1154142852214166131789824586750370783978923289350772371482411288223513506487641775145925048544494814137107536782246164934132266005949049127269902825030943817326537358732031209257063116424572340170504981712132192359855495213451331011715331377682331247292932130976430024505880302509944895043959698078474481090112307459747067000512275541754348438808761879024954452915586124987072015454628902184468909399444356828225776011467822*I");
	test(164,"Gt({{3,z1}},zz3,tau2)", "0.1072920804165662089039427827290255938013399732233295822928480120029166350360923623090691629514976165136767622497737517713257862787521923770113477626667215630986489610382802180989869852666545601541073397480495095623462312969536021781400668820305644055795348890832745277808697000169649978951039237044098601649350297877666187152866717217088904175350591096967679651211511949611962036803042893137686632076985901771323010748190532+0.1089947396528167959602128634704310670683157878772073229897641751085906260389107601322494564451438105339219157968257515236087619436628534475540710044855238905696698864972741094080543369156563552734311635397291156667572533645486159663610213297321224468849499027411382532383110796065080768201868348133620344854962827893874305798347276013700531726617519334104165069127006604302006280288038685761614514158290814697464320530247004*I");
	test(165,"Gt({{3,z1}},zz3,tau3)", "-0.1520674838174289705028989077050779605848176888443888180160474537934368350002885483429682670688813832030850838507476481929865723421366695464019955682578811769891478594860187507823245256583226807717714056961479138816163008527585562523163299462252140247695322828085887791816257973375735843259404337401395215186329518517601718826531728018909339349038225662866464163514513791759826069581286966043763839578577785190163406716388456+0.0374558432553703726482494666740802384320799876358138209475781455326287803521262121436138669568323178311476385461640639259299407134331952532665054261333831677825243459186550701396263723693574071640571255674117134566045096547138035998587291622748793106118304885985433801222887890514570851486200446863610354513757933580151323495587522189258801032761741019940032288995296753299563666306238445298631694390597281484874670939388124*I");
	test(166,"Gt({{3,z1}},zz4,tau1)", "0.3892561571050801932210364953273548711100914327561100410948247860592896011810554168330173067009692399312430069756054257410124612914378182284179586724437988304431840716428578853421062639020679654894981372005654256435183022915450516631398735641776389847204654811453919788070886613216764079488075743560783354345376373491400663160043621224216769917261136203845118396057926710911601724163273416493942495573039914577832364322884476-0.1006664874603177810116477729591305763861493804482071128604432868837259923689143698922875448098452523608173972764134893496876696238522880115687711449130240782670978536209246477969480509656271121675767402920101875889681236339089675747916253757923600730267183285197125317303970862485768841335748544309336446108450295830895975346194305659406904707586073602384229315443053794982233611297881588444633984273925933903514921519494001*I");
	test(167,"Gt({{3,z1}},zz4,tau2)", "0.0653751840082177689739825011147473828304474435799324106189888036971986579058008637510752362838719700420237940230535369490293120861406522198487643801570130564168528703562980632501665245388129626543317301355299692839506913075689706401191968103566893480855978143430836331054893586314925348986954884444386430583614784811765590470526844478883177076559311919437910887698206757986812587925799721661628245194106895385930340520179525+0.3947958656145995760073719550185521071678314371710628299377931611658071647802353078317055968997657371642000180280996755363316411198826514573241406571847658912015906313769254706470245801932345881440510343682327644740756500991354459806288920935322828895590013731328048400276772203944802144473649177609137432573507242126077749820384219013848477748414555548827251871241731950094369201761453207411663001294258543672107610470976435*I");
	test(168,"Gt({{3,z1}},zz4,tau3)", "-0.3817327306522403196975478339602172043810795415984701781630668796231376441256980166508808018501701732369174860019387190784213060037914105887971882784812813721639042904814117134103291512936709195378397101644514023403689327515462447104024257772776887861141154850163041575302998506502117087051425302594682116121313041968469788502178031185101063526184826471438527779093770991971573324959696219516434773710205097618103184028954187-0.1437327188030259720102215069286974663903617315492567308734092850607079174298040786355979861428388065663009952605745099440294628890856564221620475578904769406795760650548996377045756268774162762039001431246021292605665390159130307331903433569509539384931860021168841146544106485028478466168362192004529998456042088873215669470807129236964444367605599227406277834625654770475305903624053443915289948653422513271859540471867264*I");
	test(169,"Gt({{3,z2}},zz1,tau1)", "0.4050807217212922656492910159042151431105667120032221226117112456886646063218046800844373640089836580030533424507177502708473485531596020395508306621034480044453991557171047255448785047907845585397565569787895149312285070184322041216200598406727727314343901469947585034425898149810283736042087379996615152864740343182078980764699111408622383801372647312279890549645500099549392280119024048709026648175270521780883564338715009+0.1314022406444407358995256793770970801251366412642688578696257286041157525170493499013418925656290470864233877230897862358646678019582499913428255516346556054500487079641956480646317287323572516501245507787149107211759903399063545940283421908435445660592234423430977794266529088425477756156744584810385776505738251617276279792468740595337139831826382465278401602056903832511242080699319269369225494767652002645863563793295994*I");
	test(170,"Gt({{3,z2}},zz1,tau2)", "-0.1675273999866722697045482697744926530441398599221849124119423077057185763239411690595049089511358793517085778551912929316660289209674760640343215448205002019897300219613316189761912245390496157731491258256392280098296459246537449519258753850434439992323269600089363887615182857144783669880336043608251050920984306945372042798153544025252801890710810338424008174747264348140225247551669985284537814568570422585997457740077633+0.3865223271208552284849396815747634357959644199945548151497508203900586685046954693891731485034969256183151086981072522242108602781997549852073095175866849378200483621543571492285006788704035540626696143893099220116272103364704666145710051472927299797088809795101203231381693843823720667941907376690618916489109880700786662857751767040885309434155016636140792370242575140674701348267408899758533064215808041738959299180408793*I");
	test(171,"Gt({{3,z2}},zz1,tau3)", "-0.257383617424448512326425359739944006692805465129382876283463974633608494002390847237670384876741040177312322120557929450970269853226547631795013730417126545997446729926077090968496081288317756260359479819077369910002768748983277685044913691400154779846294560789841598077422228729708034919355891652950565900216552233097000529194574433440827189651757727300396902003420311312933599718136709409326778902598775089633068397209367-0.3410666621143126079319117362532856532870661758595921108288971475709954230124663533425051807885828664473137612019745624278828489970468465483935725587063851523324958535210395644317918639501721256989029888359210401146472996905745907571131403354425528702586594970247052500508182177638961323202134527233299915403524907378701604436760851814326961078967457102234640618889124547114854992463561140519290246860377065261527572690756408*I");
	test(172,"Gt({{3,z2}},zz2,tau1)", "0.1360704016979236246862150561552350991992417873184715196661891032259523949729079411234048758442358690354263408594664795546636087893026552753290128038725313882654166530081853033993975130068660801815031412896906080234967552192637186602020660652949134474113883998702258806309605042285456140458476771768164781383792550960900943342984670119678480637247423817704666415346181559342814209279325476139172721600237348943209812120072033-0.0976840615407274773525532928869202918001844646415467419748244735016560032672963345040471548691104859907876918957828695918728721711367931471749113122663129103057530479371334212989216924667722953930833730605429145518249828233385614096742009796702912089689450379171003043486566059921644089938390520917487567875365844370310911043418289776583997342074731918216946777881609105917712157772450614926304108387551408663837881333645712*I");
	test(173,"Gt({{3,z2}},zz2,tau2)", "0.0845533727679225383999309749545646474040016374224037413069361211265701557440592642663929828833364046828019926584090034656573250904854184317359179051496197527450145608103154047608589177885371437546465065219192670935990337978747647812604741756680559990606062751066282073843369998754982439949000420678421655934844605085479142149965383014253804745986493861345123283088961204302454546485295208804877771460330172272774752223992112+0.1450985664813367684217585729855430954873546220281371364489812939842401047877787605647635531493550117163985587790399556178753560842724758201809658043258833575429853032962608039420843920814794460375290569653982587456690546567216917556022351521212371602205412143853512526425285085697284448427135571948807036879055234701426565002630745125492560345146412271932560621013415474638121352341008150502244903261024012764206152081476518*I");
	test(174,"Gt({{3,z2}},zz2,tau3)", "-0.1696022378043120255298493949793966750907783463002942200403534248015383120962834115946716085160295839279798536722441662095265414025727331074725061935475877513079086478118650615007157736661140853533307481090487283363745763669760929334070006360440615384424870439940196470861533881524102788748368245455072968101577608936242081560011429998020394447421283624893897428696495827496300154804810171269264260667045170587268031245899378-0.0003963368485734060236401228656650944993212047788813378348925255538870650732864794818075201633142767641232384744357292041055670160636164958010308467123395367080795637733647298458729945605166240257404786639689990955832802493273073820933580964793722350243638717763969777222782858457738372587803379072778796510310142237656834766419571884114718079491323677293451109655948086583214350947963518062313316960734674678250825861641142*I");
	test(175,"Gt({{3,z2}},zz3,tau1)", "-0.1020495887225274318309543892823316350695561941509670656466955556853488333174030983351638716683546347118900676531173733685154388440090837964774557601081377434385413164343397567712827097229832569053109236868904157354323853724641204697280967538356463753506597005093703048478402190855663146258358830430964103266828830891063693319522076953743500985406320323020979367185115145456609185210895020679942285035595081777275043459919553+0.2149506658288825080484051839775528192793367425831436069330823316780744273973348497138434382346196870840081190565660734270681180997879240277758189756960214323680213960969723153326249005847585143687393784295287821401110717001303985505136827822376050614593401033823871830451230707414978330246047171731958945197315340966509252991143453998307341816074332586480009019287710766375940447203247624977033612469552964736845214984452827*I");
	test(176,"Gt({{3,z2}},zz3,tau2)", "-0.2063592670008811696177745379741325766324320965482238069051493687652582448614850826545363364650134627453886683138865572783369196596158386250686541947338238964354298767578121270418064162166905367672145258038694896267481378320209948807360925363664428420826084933571443336887654927752312126915759163172353808284704981134874098528587261869036680199146086745651214263531399543659269896845250101951257310247556893759424182002899785-0.119072967966044322896949774108169840286916905218163349769448083899370475308180759224114382697102577317745617359841294392793765201957565418926543091618795116250457785190902894314100425399613145451525549555509037103363672589327962807812656021913526913670879534441304569545297118308155056838255414393452134712214275224658034456419767199346364385711081658946769530162200960958732133121338284065136616914935450291547084850921966*I");
	test(177,"Gt({{3,z2}},zz3,tau3)", "0.213134760395524564069376912348315734878150338242394450240902606619702443463989949619047529424691660858992915291446374158829058728210366750611794957093618016989047414414491847345904486022993168320191402721529338293242266028049047091330962366071669195225759710922034854559826951973413202242191025273725914746715463794176183770316928581719577422843008776416586961547622231533293214695988560591688028876460170798431771470443729-0.1202368139171355701688559243921495028299117206098092163119057944263895147875191195852937291311290963025833450798358749405913383007972277154668429457485265092271678923401276940719676111895453292484695843839201041341813952062965809026933941177086592126866799448726524516886216064256766979191200642493249364298502508440672440747735783568362289288389646180736325200140756172569237284325153900628522434615724860539383658413273061*I");
	test(178,"Gt({{3,z2}},zz4,tau1)", "0.079712928598597000571868185349414865573132783989358379538172658937364966158840364748088260782603524628779786922438559425418044498532504711308490589418556993202038813523466095057236682936976579167925199807167414130831042129266350397431760071971150902916997478289541389626722549963633371938474241840254615162282527146620355843699949164017529392107841326304461090120861854416270088622689054260386162681799192969325362251204511-0.0264287453324448963254595211827416410353214076888453286381843677143134696439838594903648729367683859515164217513774134151001775484669660308272377966101974788435709420799313973613777622293629909883592365672673149971330394295508648303638170584761320173569147185963132942849620159289350869992265177831658173502255278789345331543922291329073327719408147002382373729523456984416825230623214972839098217216749695281676177668198128*I");
	test(179,"Gt({{3,z2}},zz4,tau2)", "0.0190761643066013948651678707977100117484771300854653034693049330492086988484696623616411770910803886034356723913472856884011206692526173508649996648410254981895592667703664083632610360615723103292402998698036956981589467081267070359162903072206562521364601214459877121340563645161566643022994169406089019024193810378917227003835503105260149642802935737476740891089702341757323488878648947198390495630139164284336556298861501+0.0819887776841076437895343155824742431176999669259888624335538581388526462659086096737777510491480462069230838452878373178566174289415923785712546514557607650807065088247958029584744128085669793751170291739877259535089831694970700218654762138561225039313726590578048774470452149891775769649032127392388901938579381132972638101202570635408275806713858032356257949424677475639230545016724526375265843423647414769549158562255455*I");
	test(180,"Gt({{3,z2}},zz4,tau3)", "-0.0814702237510954929934702349494135933273161991862354076991922912072285470622835954514516707840469378870722455552606334914240916983946784829331512617552499628476930566109757072831886443317815617279416978028606804243664800319664192141494204515197905778492514363905211789613393184544681118402532202971322219257515598591200348613877859066362594473070360937353868785168986691404287318131149213847033511179988109833309165492161138-0.0254130228496213750533764550491321557936765980973010789978836271137257043637444592634250682867930231236576260638999016242502623251569345475195971564123209832549174370821144497796014707233338283137050741353974260154211440310348129973946506716571464063777278447393809031225596324992300892406868780486916062221956100410739706535036189943269044679939823417643258151884427174636339505672139639583898898399474511608121874292519143*I");
	test(181,"Gt({{3,z3}},zz1,tau1)", "6.0564439666124941393833522554329406233298285163152970177961260000392766076826282769657914415273613742091847514117001842298334413194421840717491058107352005716624474679194009830906351217986440484775438098031275430877934150981473828299721202547158084921285721314621795657396297453540224872620343035963944730870730570000279761068184136978383305034581358518352010980815398035055900671896708975981691260988204326593537039254577267+2.7480819504301870999972252374651934208038918019566091498957041850900469263961429965915918146671676759594662495241619925025663505143873133495835286278324435293091957231002928763465554083298239089086189520293077612727025191830096831998360058848228003746683295597788795124524894274233215437730790187951566388939179266564824267720695993601491361512417235962179572269284080073850449810003399259319678048164410007074484319671993829*I");
	test(182,"Gt({{3,z3}},zz1,tau2)", "-2.7861191088162196208866716320210083826740829464081395380896519066222421869034300813196352604573970462685529055606745411284671719268431631453980490760856493279535070786873346954037068899383403534857878090647899695309985665371952357944542485099197946419514563503197985476242781048960719191755640799101364951570758186167022548737038590731479501207234675458763733810703018778706058690963283998622334661988160653533695722931774377+4.708604223038505524763992799884982947575489143032058683646446475548670299295863351290714036120044802163494276430820314713040179588320710491005392293686550990129396641885722196235859582795842405149562632996286904463934506363836862864354582773108752752631670618747354902251893548626844037769278284671042447150876398394031230540165028264368530496595037994611974244945885708459805235162695141988702164610415293709363202918921293*I");
	test(183,"Gt({{3,z3}},zz1,tau3)", "-3.6117918099521430850196495320447442873917627288210371546417553275866473954264682357370037404545122432377593530651749015273960899833672016530413264280907567857684424738592231397776753868660746442571643458664386916526711158298012185868464553086078415442691768960335637955573340494545531805528036576789433626533978160602280092933223581066760875855779935591127502600448335768851361163816038231745417068930646287371097620648859717-4.3347603186603296501895576890478861733670757194523675667334828899600399235780332960137277549279659553552433952410294783782497538805512378705934137652855790228875303744441503692898554203739053968130895112697902245841612409091503877518578766230526811765841279647392262064485846957226180031235528135262961729075425791200467621755610817561663766051456054876704627174972483828729638041229049223102745606299145698950865101275947925*I");
	test(184,"Gt({{3,z3}},zz2,tau1)", "0.8322297217364572601726605560522100212886644655548617515245846867852645027487152930593696556998266023014907628435362120414648251411411063828833959999527097687432208667495581848383976890361983193850815338981740023248367372381698973860674836846381101069750265099321567182792053871479427112191691708277394485684649221782555882050559367077287992056332709746351201149520986827699855833605163340177827837144034512641699228537425131-0.2980189053404269436132919857461646988097529485194275464200925260960106690260046718685715311064178277135872979581430450634584774188002562745604697364335806102892545213658602110804422632408446059507066035850243900036405766922406326092470891560710400896900801603578022467058724606975728584772996338359440341470808921940324478731227449039564456855942296317263294401449468058197247613503493494346331438965477069801021397710796169*I");
	test(185,"Gt({{3,z3}},zz2,tau2)", "0.2323562840059171051504578791590428842267686230053816783552783967073053313463549130155874249471411369130335023418006346282842640609214305896188623592718083033048657517633565027921416600916656878551390428541820173972757686023258242038627679325365567044194911771904505459332777762482381883731048254460325559036796744823846404881757992937021691120466839607500314072768997549365923932929691199057803047529470205273645608253811001+0.8648734012699225663601529709922761422222281855688890927982292316920380961531750900299237303366197409949064422499100067021872999660605200575408568675673228720153966939422125694490986178325584695959081892650549426387200863828606072638313326829465981535586001624448131354209126201145441000809351774321313568016630122276559916461108200487202864013032962827620231360143975623216831042754922481637482629703965885081145575691417957*I");
	test(186,"Gt({{3,z3}},zz2,tau3)", "-0.8782056498826989624424358563964730341368703976899440071039787640860410909699405081157589479515451813060000610569686351228139157609875711684791275536625626347887909929261027621536362152887401416324331952073133231483130892251343337238133203593926705168798862938789878365481348409893690400556787742322817427105452075762427995333896617660285230639481817564574885205708936131308667583511505409068582014194856817516956340417341996-0.2566620119367375147516814133812425599019323790473626505907181295365776577126223881651739997635296747311153731305587023016082877655493935218696185744200443423472709058134381312380563685827543086137347654229653155119652373224712865350964434964807128817363235748005918184197521072844833731090850188727268881133068798231358533819035768673716445370979258358438511700616086143850083182522683090620887312485889620706340224544057367*I");
	test(187,"Gt({{3,z3}},zz3,tau1)", "0.5564707794261445661961890383538205015922043966062555850243881375635977985171772040073328176381830007733289223756312223494446585764720442765463143696133237869616609008893626402204451371374152077436831348385430276361508105764720901397454442235569535864735813780008957491288590376295854598678673207680313739016942290207261366012583624974108119286016477339545683748299265349371994791372916517130948716109759827456294946569891041-0.136774487669058197743435808541048110425052726101732074383428865358427555820887002553679026444646802733760922445209055501532513578933090734798891736269350650944393846292702104510008332310388412143005749412470630429082939041860718613936379871269076424664332570071268181059100580409211646379088145119533825006857484162707587714946921493905411718180646115387557239800591768346075753934217791571084220835528040732491790058970347*I");
	test(188,"Gt({{3,z3}},zz3,tau2)", "0.0992610144172979739574313863217084062146676471511802249094472612855075698455401144455034927082367899715544741012690179785545938870433622665939994192650192243726981391800960295293885769172584373054730432481458969937557354517099584941185676120992316152807823985154422059337462075510425082180697753993730921338110714986203394310375514352084832199361371301706450758651343866565325730822315429156504704722032260808192903160092422+0.5741386721599401553213575469205854236105274122845861727518796189933799530762203776436163600552315890766281938853886342513281505964562305651862263994742606789319996429277395752944771873327629227317864895780990897246054809195542488323897078706955292427650954444025659029676397532871659844099912254503606031170880889986830482565234339452327062830934731727828520640731181789748102808932278278593231090106822784927521114065290262*I");
	test(189,"Gt({{3,z3}},zz3,tau3)", "-0.5582753325074875600126870334966073497118462947725790154724281897081275007183749362184854032316230451064890216605873976016961162781533963059475296332832729922467050607932666969096374006724702345792224673717017866962009907318486267640909297553587336185637269064093737364779534779341112489376647439419394241974445826599924304765637888033634160860977192665816519826871478571256301341331581816094471934928734392305396494927549648-0.2254701868966604899705178881979405907506821482693043066413166211557304514357198466318766930973863935416082477074588511817639280782733645964049352948935512181383726828248544432166976751302606688412081331406344999257465462313834508492519795254127361397040130740568064822803207380233426356091986124154197943180464608381112295932788349942480675578131547354757850144933560127785554200915177901333979395247967230654637064469645297*I");
	test(190,"Gt({{3,z3}},zz4,tau1)", "1.0993816527688186894000191000568098468021253787514782210673046254884677880253044303210012125443269064496664494221297873910366049532787650952332196343286512404745682506968005175530285611277498928515036960376696604117789550652048826670495795360802578097794111619648482765334646083440043725513850210435510982793795886023700142833760875374179737201134866395649807211007165732719111400181852425720016162631967306261636199461617843+0.3035111760190316587162685982171313359448127928112585066490366651298102781603553907133501613841601381775677970345617787747994035284879413386184421534556304797457800394709376676966970043761221250184338857975323391112061885164939542632661120357821860442376204755406315899142115045023580729411973388562703536890052421547238775248525092627395711179526481632010225913613413107576781228055695139671994293631614490942910830966184194*I");
	test(191,"Gt({{3,z3}},zz4,tau2)", "-0.3773447321155680184847518795476305590994446022526991350763837787248763491467932644431660337602630848311599951202039538919510272084435595076461643212767000909861097807609166873125093389503432438847022054592630506740555925144186229363341038881208734382855165228871208708144079209124424251692598679268618398293784366235650326593606701197970465476031990206910211064515896179723254124470858497747623629999846685147978078888823037+1.0648061032393959843060141733681682458933448668175882090212761579341032082607968878375118704560621454323815908115807759634219002802309462339771396573631826330221044234642269971380622632541267610235496461691516070871794388244188746727729542599054510592802312216935756933880537574540300607408985952034707609730644275393835428183850370919446284146196377185151732055919764418215179225828826804289326462649140198611780880425316808*I");
	test(192,"Gt({{3,z3}},zz4,tau3)", "-0.7615383079164580086351328731405359364756964718449615119666349052075023196367496233584955334498399013634304871194389735976927972122881677684348743445898599610212848131843197821504380487149437909240749148166355591512838064481393533862835477599333458168668159634794175664496750200535628945601579447896485432906561297303861771957264521226093538039904981127271214414007864809782921415799381840740847283783431898241719036275010566-0.8959077206284578608272863880889471200517232866863984916118395531059431483319941416378782247800672440346730404802151701829276790993748687519468791641343974574026443349437486102429940606334736416326681497839124413035097141690684762281392802239941783950063525211161544787455004324352815668275090478022700277949899401135956376105177511429972438347899141444022915594495209223887452936545490039900601856408847199360494421981476418*I");
	test(193,"Gt({{3,z4}},zz1,tau1)", "-0.0334002053911531443556018925938626385069434330103344799658564376760739418722887754069963533232029395428181787700822864622105164594392838589616058372517952916766208387377060113603744229829066691821165700285630747515111169179333165011119944594870071076099921131464356572494110396351061160798578455793043752864546090646800959847407026787847091308955605683259644187666099473268827934029208632354391180941804285464682774119008188-0.2874337097056434454366335834148294605270635243730428250558929655401333030619870517451230866117717766855021334034392752395673042613771647004307956175950941951181613192386634272970690527741561634693678955757651334493458567864053294760687821534524903021316867906299566064100997481562670289587372608991223525662574811237440859162456946038398002582199985509867925348650274890813542077569883859090529902994982654053717724815744533*I");
	test(194,"Gt({{3,z4}},zz1,tau2)", "0.2875719258754353410522117123940977164761057086833134311522511365813319270254419502298096274076969322583777979403012820511664097304029359289325667924094385276175320146800329968998949057148741729610593061674760949642656830019058046574429938109689293135192630902619332098640364738547079301605941115896298983235438894797280480629031020452840656736209189676307676581972758539395375554307078861550239658071769801871120059668741188-0.0086211746674462158314276354845379223003001035023954170585599256537750754170645387707216735737119959275833833156437188642583621804908671366530046376371954249340488994387901923310599564377465458205919669594152005338006403656769935353314181286524341328627656049412333871816925554058876616218219920509490455796306174432032244707085773718543907046448414434745595340678310694539855619562735589762644727071338602378996295668473746*I");
	test(195,"Gt({{3,z4}},zz1,tau3)", "-0.1395980303739799206026340042808615692151538851931282568968403345842750008937111502070084629550911096055410679206128698048582942161303811522592453099971657392914105990793943200622187201713208650527308542704875777205668551843204483088072273085533822484372705759596118988516621079497197245643535485547424225537310471735650812908760059048468576766578934339413514519241697187666448361017338989656981652760579773408223545204420207+0.2591828113960984638534721438760425601591112963030831333762686872759178541118997926840846580637003240989812302826276958209987144758170651340007117845890356080440578832111351036097860209487216688016737924663057610627189388771898705717843843520272356229978712498239294280100091177473030027878184494420487930885579224640236605458917066188896956032432695088845862979581962722338809441781884800102356124996627701295273098194666002*I");
	test(196,"Gt({{3,z4}},zz2,tau1)", "0.1167927338277070016590691613948647419478010025900841689868550556696587512944603017471371885608601565136386629782944492972864949230407121284820430676605297639444208710032993454374002341331829123685890735973660662736116069726161647501691742990205792384927956793845288429027908054669810598902984198515449619787560570871800029032333302400562940124867605784843988040147404081905115858528362321012647092978676002475189381471350116-1.1338357226441879368740430029572218851292247426271360083696533201520829817924225246703724953774341723434337182363428877505576846194895619287309134667121550935551816199201997095242231940214638135892835493645339149176738590330070997321192715970853796706092288536995463866340304168975072124127945025775075066329787862803603499053212322088450832363320435873610728052133775857884829791843361187910354289948222122163434370741790379*I");
	test(197,"Gt({{3,z4}},zz2,tau2)", "1.1261621919701407879387691651534501327295768401255797479880331611068667527557927808269891389570209752389308385483100277914619587672530251546877038357026005541303700276322521688560761034639668191897569258949170816201107555053314026312118264157231751875436321773345669576012099788094686268292023643104561328634461166923283839606057590329986382104908433854025163973469287778005453585568975356691929341743113453052727005535074563+0.2480766234406584727975183857684710521432826999227538503944627458388881518142794987097299284212278166382598267283797088180453106085550321214536352999781194384132332255926638964655485302780732956361557619516309978046697866681321672128181222052147837416767875872994127649633088008993094413580865918506326395652835021046930864836427190227133198739330471552182557864235436582381065087003439717637594091268949476026390424429105907*I");
	test(198,"Gt({{3,z4}},zz2,tau3)", "-0.7738557238328303031178303610405326252698775367273695157107584966431282319320085797872350194899716476307403533344193772861840120445426541634969688527750601830346740157596790936151235513188576350472922409362858232066098375164940569358654348188791751272063890232580860943334167967091703546674792168485271769759138789337100717941744293213994383841140619714670417769682710888959248809169282921306349986057715120684482125394990999+0.8331232446076395715042407299126300660992316918392492232173275347570582956677550663894966235180278554882553285828264294463524296362940227889318202755965664280929705566121007750981809788478744305563838005548503758662620436725627330944037753980811780777017401757371015321329477344597328472573380920235490582883226895887868087474007146715018996098228242843450939787014293088144770379164322294823018686422450236849007698001974711*I");
	test(199,"Gt({{3,z4}},zz3,tau1)", "0.0416163930314909297914370210057487727815216445235409776763222761187164189584472132476427608216825278116034802973531286399037425857596037743445133151251980652879316349922084484652025549739444871781029040315890492074982766497120487424562486466002422008039726033684093486263086765476652344301874632754078560383220648764158004143888933788677358067732036153722868754557444282533613524406631252381110444531669421426118658755119825+2.7977193862176598802375298640341203683250217496419978975764809368725380406665461522345910683961022296908714455626862569724740822888008396854420037857497777584452213801472550253958857287708693841292155068219448250995597132566790075106501997284052625222106590604983481183717600747363798934652420354734040277843839115941734040499411753878870132267123649634129670873209690502700018262172982333786477487786630305536513651706494955*I");
	test(200,"Gt({{3,z4}},zz3,tau2)", "-2.5748556711485046796095403404043794267815630965054514088089903370682710772246820991399722461006197939518120601793997810357325267006541948938205618752762037419820694942752435778030420088073304349210171406522085478651049531390030263865777393540041637702550424095365861472845968919609023800184780180461153078725978779563213306062563442572689601897268972425988203685443409428988476370361615061290775683658569644390622712942311777+0.0216041267639843638671162667473249724524318295448667375070342578302172086823530838043509046644650461566453363998743293815069040556772317750782275276296663242350320834590521707260434287552697063424510935674819202594953132528980881651280239713724054304136805190169277071304604342696166929674477026429750696506577543060280194125327514829654059490843170084538153785275116900602847335789752692247308100449483941221757172487486775*I");
	test(201,"Gt({{3,z4}},zz3,tau3)", "1.2619473238304567267864420970783974716308596946530087163330401602517353164336213261588714479155609354235981862809726031264781563817423212326562999518345462696110119331503304949849715736973699005497501322823330887754742576464845106103491035253324793682592909023824450270418526900834880121686113792204464801033255407377655849067432546379090865980241939037887091265681779395027339351759007839198038288309419201312285271947585953-2.6543550496913899487777463440456934694175559864151005595446313621566343138325937784823949382655514515993924987155574728059492455164966016231291760703567680034858928834672619732497830450031023124886092465915369601576270982714304412783567349562184514557686231013160692318397200889554782226175623728660242629131184820092407548892503217965006146538770269020818064179713041720754025789751041711360995473152707631210427056259206658*I");
	test(202,"Gt({{3,z4}},zz4,tau1)", "-0.292776511382145383477465855150789781466479279010588401715050667424019748714940560121443710127742060394086694848178318324668750903596977699121753074361780489300593011979916765219228432947087349009016941425589247910295379827071898749616773107687313200375685214973024455342927991620529153735668316516324046100583240060177958745584630808309029641201810091145757595641613117158348283407797167402856530573230693251521197293975905-0.4106902023294980548278977448795562219684742863745592314089896376178946927615444642877784784090028017418032808668077145132537298207689216363100453741067537336424800143483006840320204918178488694359560866290927617839655097533445580306306518101102188246037805613244574766934294152830698233497935535414346867642726249625964648081680135962759701548852905891754951304153970936982337757599362067431345027726592096919318221603261834*I");
	test(203,"Gt({{3,z4}},zz4,tau2)", "0.4354902563369075093355599666007493491721269340730749550749149937019458934526436328425628034920079363404282910288591368904868489249845733300570551097285092392190522636627282877364178550187198938649850177798603280828095209536111231425746967203243012502616032643675458466035204631247361179359026787664727832227911644274691039755610883858940646299394139168944307009012117091011565479620716539098833618441852198715668449825678484-0.2591587725335112146138861685988571313564909736355752923540124494059497651688612977105258567375237294099035216708336477265143611392799053735989513857811820044340694320978690143628908417193481672426048494649017815414167682316851447058864674603664403018196892328340924904872900973773701552185924841867358013729499441159032840981375824169472271905350067204171776595799047768774082621524245311717231770585362911491668371843709644*I");
	test(204,"Gt({{3,z4}},zz4,tau3)", "0.5183656227921969277069442877520851203182225926061662667525373745518357082237995853681693161274006481979624605652553916419974289516341302680014235691780712160881157664222702072195720418974433376033475849326115221254378777543797411435687687040544712459957424996478588560200182354946060055756368988840975861771158449280473210917834132377793912064865390177691725959163925444677618883563769600204712249993255402590546199525351303*I");
	test(205,"Gt({{3,z5}},zz1,tau1)", "-0.0003895617246478296097706803955307720804861014786491630228897534770748420513967433540953186454731931661604281132829756798665912487266514912653287711735407869528743889762853166549620097117645259388292773375572944671191317401237544990792931050452180363103697843769939580997477525664943876470717627171902927672004246499401284370079320020683613982529489849897369141865089025507814324216984384958154824726994207737675254436744308+0.1318919172980372596992454834610930071115952083457806728934928911669748605042104294679907422324934274667871375383079144480553887585113035950084710282101669094670119987640385512335372303323934888793392707696346051768065395348301641510901924834925492728429825030191792694596808651694511238838884439559933213170625863528946677331598821054909301168712071312166349080130256826115426313103471131942864540137198393705148808501023748*I");
	test(206,"Gt({{3,z5}},zz1,tau2)", "-0.1323150906395816725423692303167171276469192067081211585844173330951453759240047208054306231402502529853221870924663953568381909036456900481558168777760213670476193262099525440888497707146969355514180740310706439018765784089608949255108776003634229090967136896412584993504176492814154427236337966371420355319249809408140966771454884479517227088524504388646024289162802271893886908453264322920403381808956785926542156168652233-0.0135779938487784846194728495162145276913441805271241049614049073177959112258067337026938878096943680683627350902131156566453854249680663715660841725971525525472498226901933938715856091247127605760778477525399954897615378893552446035843926247237080932874031621947898509338165681414421753988516006388337991951906121882373379493170207663063565854568035883066908105277044619914468077715210853388904578011683417880311133774628302*I");
	test(207,"Gt({{3,z5}},zz1,tau3)", "0.0793528864295529483902131996710585737492632364841615871262472459477484604804995925914249696849482906456728178909390228987197664395822687128975669541615204854937893645015026606779076478864455143818697679761366341582815432536785036757081126569029834299843500656711996423412642032333424168610564574442301975165690609602620402995509168179346115561424768199708852231047650664507652709436853126323719694353993426993320200706161066-0.1072493392265805840717909256404764801380779341276740200287241162526401832923630120255828056229617238074630980986638732925087723695106471835780796639537586294613587619168186769328601962450521113004237171786961720690023765604034832524808814832688488320087282386667036921965238809042671712717260870907985576411925661066616022221781020606830988277989261833801208035138647168983038326637859993596054505994137520553544343949033764*I");
	test(208,"Gt({{3,z5}},zz2,tau1)", "-0.0896732832067017266852521755702432047846594211695329437611447594359692236306647451821194414809475043172521503583465601949724501527447449660967348822039119106012203288730396757718898691152725704846495763291588166912178999365891753413184743451426110861806301598141045784701133750475361827038104395706271496659408738263122450691138491661504328276637901241735296338526522606393131768430538860089651396780601750249616572292749589+0.6629901393335120261048372262838483472916738640985768960645796789405882045324356181741490724409263137438325118885411860853264880733398528730473120809850901862846210858788138212419296198697962565658879784166658659098935616634960669873540052036447784721226273624844273234096159489833764747072366504275461632242207643071990998311269571653260812646995723458875188763937098674270103041502419393925564707598166380772793122135837973*I");
	test(209,"Gt({{3,z5}},zz2,tau2)", "-0.6516395589875362535171637549489749788178611322606176932724005532948446453186466157009437197742747709680502694897626902406509539117325551395827354344231694610752686213807292845213266553007935620615368366270584474971478226865594541593843934038613316122224184432011094469587174378013214115022190062471938138684017845711316348849432678834717401923727637039909484532367239613777694755657133331775151396712343800801410404355244957-0.1433896884181642702403170838605678029842500016428976845998656960859214983606859379385749628168735107714192112586235609771562880771044301070087773363210158883206499625485007897701548652911679353388365726682351733375888676058361183917270413598741332160771385045102591931376468369738653162835555258753976638972102426032232539297214507183420040308895455541722359540113316073336525378569687300026945386485618431078607117241605043*I");
	test(210,"Gt({{3,z5}},zz2,tau3)", "0.4642279206033688319729423251931820960916885011270596304535503081424636943329955377313254546549350860969722798292539755463515091962000241077999824302333352539736746941182534609856243370198842876286628302788429230441806434963807288057077321954317060061875938266337241876270224001306699775589717558969278950364563049331087764899453509740792478211259008849184861596382917026073976646184841820661300539999321831847855339862122354-0.5033372432635365569153370292468664089403255628348844951721698109863817840088146242996362280362674655610344228640087456742380774880391792146304590596433872294841164973214261978083963203518082489078034154703791773063740997012899695703411317056642511859339739316622866635398407718748576279545394015438244320880535832094684476236852240327351047116773577752255974239457479267604080515326593929508394624959749439477913676055696276*I");
	test(211,"Gt({{3,z5}},zz3,tau1)", "-0.2524056211381035674432032833077103603597604804942710386219529711967841020238998103706212599512789714911273428376073569249090171623393910650544932119510113004410781880755309493212847219528303329680967200752044016168695930073748472368432399087457073959064594392218434198750102637927376374093778719747503690549677898817319717102036597694221307151630690429241966583891475023412397126369343465311919482476718577330578875739228707-1.6218381420548797834359122803062870901819605972541832210408474872536400895807492278480821811037050362890788357371985599997599399346826572078204178336451615781088013395549086065789166643662029174033408595793302483875808634460883426781008218356103475447481212947936216578536696877660153694469514470931589268419172096088442425838427632805033908456202196103319340582618316454833702867158205105727369501303968921141533587258214955*I");
	test(212,"Gt({{3,z5}},zz3,tau2)", "1.7473593114797997198190811239640137060367238979171429856528674315739969357959175106747624722229514338562719724376413819362094565607036671830160723970500236946804080063144960860605532259418612224970814964619942594543702101893638180390541258926210347729097823801506817229089577458369081901902348367287727691120854418690118260052574873065800077588214850408765890130308820521780443843682048320216487543215678443542389016834424528-0.016685204193713855342434306680104603327000074699884205397693435987168700811486388033560304629205545617489224216133392926415772613674525171750448338835292376955480464529311625997323914110362256505530882783731485399806979520503557590868990691364971860316594966761599569258318821817085830135474368302169776141504727284897056715222505178653434560741819469266877212164883529037323362658368681545731359063338655302269970996263022*I");
	test(213,"Gt({{3,z5}},zz3,tau3)", "-0.8698722177258581569519368250234321514792702621093536769797581462442869018179839484103308576321764340575923875942639402607646734942108034881587042530033961819444807233880754342587003626490649453623258299874169396615513844971719485383255606406522672661700652630662006360975777645430050260513849125900696091432674031354646224123534942742366310587466958466889635302836492006744752644417282228069695286026133437979224693370664973+1.3811857887803016170274062291143952357500210009057967524995677879785314014554483958464000888378706766774817723979768613007838554696316652018998131740229728891929423377868691206299801652037280006597356054815994226913679315413228378876767830303401096984012968849672513101573535453562314626563890028464641795350924795615153416819762601980985774027090800520286834890356796384859583829583487456353928721139099239892376065286354587*I");
	test(214,"Gt({{3,z5}},zz4,tau1)", "0.171957318705404537766914475257692958526686256548681040519604812018479367189740695462990004643027734224776041919724801201107777948252219691614599416359207105420646596540719365930771315505923012579635695665280862297744625667329454452384849868650734397989568648073375412769019142420331607145253726982142919405287653686654012391248285939622941425194166488399690191508622472367599171934102102712276125453513080104687392136269346+0.2374822437196127237624725913226008930221889113279736855367787761553934335334389661108729886401855555853196126704904244905161493142233816402356821754932791256093051274727956586636448518691296464701168588609242664962851921587468747157742723971063503759937894530772365222800734098331556807221147666918552847347463040593888104726559429319180838760013397160378352117740408533409512878370407906156761472827123807100266640523550833*I");
	test(215,"Gt({{3,z5}},zz4,tau2)", "-0.2537819008096608595426859615080743412056642222213392343875474740853362391550667488421543646358690359460796764353309526107032531769619155704976934556092710109701203544434855258159152574396823236969339258765607213394373330400721365378124737672407371339241812533269448713241776459632885772233046989631768285195630300900098675405899186236415818557415557459980445726363766937335622393229903018524023205763946940781715799509240476+0.147329938957066579404067252047167519834464710797666047228174531729712002492973713856183261522694409776816385001647352736896056108299585611846286946183060448925911172666486434630237645135547150023510004408777295784669522993535315801772459245038915274641653653014404541410973562395812386598262960173604891441475330422528027309588124332507948868086055755768443010698745311723300898472083864850486724318500907432806894642094779*I");
	test(216,"Gt({{3,z5}},zz4,tau3)", "-0.2946875472694063059923318173771258908174127642504539706095171416496724496001779886111086070347953377195391534961612539577175537083130934524832645101117111381051519953355857326015554516106880760975883444259029009882696445433351894241301672239538129425283625684255973358296310920973739525825229438437153669794026575110803118536732847534910435762027112490282386569836771844131526363429708894961649030212844134025787247533113165*I");
	test(217,"Gt({{3,z6}},zz1,tau1)", "0.2014890295222349638065326231656415832982295001950927656843500762746367283141435069129803134563109478391708847639509454132960313402748938308288980261153135612273805843772206332870484763999580290979505204287555271849624138094051449477746704930944802087069392649279336254669208052865497839979481392594349341964106932923917268396090754230877420305813941559217794484590487821973237946685130015590650080357558017393924667548344223-0.0905292693535202347939628115985987432239441607583999604434639922033447347952019108746689262713901944704979478831577228390490579010612411948786513771341798197146887054256383398390474724137750474520336052731747610710976898751561397287469957778823163504527757881753192871655422083558377154886185167523158377900219265763860962497063408138647456348610293121196586317354028825041654182864043945028445465763064263305901360080237907*I");
	test(218,"Gt({{3,z6}},zz1,tau2)", "0.0718817273969738388066742371779009553650523033054704039880155803141172096318888814528800444379405476047478497955520670532520196529218695768286076513729021186260474693556743636133606286456876193438262589303021975104684709737522693197724120029196589329625261379753112585256298456693734588826392002329908733176402979750499840934726197023726754393882069968405374380257531700094233845187144383069563706606982235487458247723344891+0.2091382025913743771495700866363766926502147616233367831757043648336361554117061851674900464847266181994558674590578843248929535392320152183510844789672681157702777359892684637187272613963081513293095380840152783033827561860645246538738555786858557592488999425818548902606794464554452550066336641838944685715318812812184716242298548664660848657563093500580751874997750106246782327696347791428197849848163460155270428393847447*I");
	test(219,"Gt({{3,z6}},zz1,tau3)", "-0.2199695216348561013940448369665970527953708780064783305638820008438473991707552224012258186556520241562520631571168454724580810627833187882817218370010229495521197749942440542520288193038996234659173021725492871567521230563559668136740161300212621727811317850028047753461006370374394161804313328274185275266190109501088500042156932931011947029170790666428095117739665765251529621404873619480486137749850623406680546368247812-0.0442956956319782134941091677135986379172948638772829382349976131554538422634460664397967804085098155685292138244117359624624564234437512195463337998336136594541736306000190513839562583587131067577036802873609329096203828710522143382042419476107353192752032171892427127051063539385034947941927608893602341968764617979948352947328367224871428779369363373814895021337283336825302664642400158842314231188958753124874638338149338*I");
	test(220,"Gt({{3,z6}},zz2,tau1)", "0.4638114826793119116781298852575872474616142863815280449847747666960669802518233091211842440773752792553675450720954008745281262838637676502071448871118679371664152484541201874309916423510308132801568042872472135821580522868433412844535858610042911995049550007470252148201448304232708398948016029002019185975870203628165803466938362206632497089144586502042688309247795458030510337079806725520825248526913992734504752769624655-0.207774151676266232804956356841920291672917753310051322326496713860775667978252732004114102581292941504035206017825208488917977995216272604609821209086496249149929316275142286814466262771562612072003066159847359947679149542231894876302174073636646025095874572453158324401537403332430889336415045911116493607744005088832839136364192332857892834247618086606436302891730541841295088396073433575735477960462482978750806900058233*I");
	test(221,"Gt({{3,z6}},zz2,tau2)", "0.1593088471291659094110641909922611762424305058352110891334306910734098597257457286315578243826205624619293337713952833379828663200553438524160253771627125645168072242356411295649974100135118059986080086003902113712442110698908504586517690934177547327408071736009944851719754369485716559873222716293788025325030349031845712787616413298273316749068684849915820443620007240833411420226420648451214245317552743204674903747074518+0.4808069039426649593530691572347806395524429476191342129802437791027109724901274337567305180185232016596499373009361037296774547372695062816701810019189362006432544276658350435243901826656546110456450645167836773668244194116269837107244102611536646572600206744985789653360203850356682434751241082346101400342201525807305392775841799989741338503816151091061622811327030794536025286332030190383508769620870916006442055859089516*I");
	test(222,"Gt({{3,z6}},zz2,tau3)", "-0.4989575019355487976444384319886881177704585155873573944431360145762118581922102125767759675392768949134538412788229977326733234496627343319691876145622683155018369234464977805864367366682077658601211444581402793040843265293716592211117714725488021967642237769895255083933298627324254322599570069832637043124598084121950219837262177299171375990626940078680847548362387628776041817240229398473097202563302003672245638470937587-0.1018833020908546369765065977993546064753471876196414864726974873370065665789353977956538010776318909608788499594639470037009054246403826385138995349853741711816135210023886700354782917109604150117158350688114331801354718451353889275209159793796592200040289851065389701725097102485254898753780944263863402730536206418441250643229983232032184767596278455996061451160533322372724968768883035474076645986442108211115128281815856*I");
	test(223,"Gt({{3,z6}},zz4,tau1)", "0.1090117437657609059125638949268289503123069607238413823260014391672434640312977714598173429725721809849422887828069932694455682674460151912997238685565225548846770204282729888516571078076158668050263866206298490318106796964667092590578058053031629694901379264692876236043045051591472834456802847190184365894808997419443584708955515466569206351431906158200760571144604727932559276581652420567086049925033933900538500506556307-0.2046136457360859154340387647058647670307082577937500282751965938511925879782466966892837487194346288235385474085502223338837071901909318803829632291465974228632310393780864343267157398121312348465502274917869620888692140181934497480845432154712778427711478040001713601277174054929447015568022596837704869636978168390021688488016736424354074657325081071192194762366879738996123225639293907215655068764201870867745641633719271*I");
	test(224,"Gt({{3,z6}},zz4,tau2)", "0.1950242132559016453576993649812097299341700641692364190978009872276445661690605181020657453114121163623391653231728503765121171719712242740644472042731421709123030456017915688729317344391875984264544328571398530378188283796083478327361407393739279548355549624334408363296228767821751587241812043421456948457523081637567269376901796423181444501161631729525428447863940612440029431710203755544614100512224284151241307803758685+0.1282234569334465396669257139372158111714638164516104443952002521720018622214549082228470876446113967391513070796376862665124626777837415953040325355560435050923095489203661647640247569081687597822503543228408558634917101000085178508834361956800018086171391940964789555223095423037549407892527687769292844736341889089084590836506454325058646151921586983710326490468477974596908158080932228866929961964996410981814777782353488*I");
	test(225,"Gt({{3,z6}},zz4,tau3)", "-0.2122296329833972321340255020106670559355390310192769655133747373196120565946939311075956493420154180915337674440054928885742467386138318667810423285993327267408208791646023050597671963212235065464761248755589450758067182179963107469405337852744300258259110557391014830052799887491221324717064166696016952787729884491325156999583456178959155013784790142847654440606119212903101163907166798654758729336666114303275498200138189+0.1034872073640923317967476046597393650276420835323090548691168422583689310637567589931540969237769747305427445187006632311383519755268757919819773288164100792354085200163913865357778561178971781829667092724833547006589671033217810677684026547290686758630898162082826548693051367881905474062793065568768453365862234683158847873750296976852985265021780535703196891062166382981850393387664938553585142145792626977256109983928954*I");
	test(226,"Gt({{3,z7}},zz1,tau1)", "-0.0265757436667135698966618014694788800010951607452372256103582176757225166588339311263775459438402870884848758858872666924511993614718049350533909949409891417176765942822219032233919450445013968700877915303824575167958692128489586662945400526947556827266857877468977791426083076953652190581674060109755613934742565793418782136431891922907875711840797113623978434252039290454729642650861906520057365799141876682579738684564918+0.2953491106900046630391686523715875084436289813494047191724244233176245395703563651147121550132639040467540740762406520945853858208040185151232082895665883351084934257428918500220664218620206312328052653792748505003667532068191671004341243267908409836680146834135265604424626417428063100400069761874217503838214616193299771930085696613901293759393560390239155606739390422298426984581801798286717108346469408109921359247492321*I");
	test(227,"Gt({{3,z7}},zz1,tau2)", "-0.2907227010293737882026511744885536713985663215552273551591498708366236251420772428426278805371656151843104580135729286343770737013153818271577538382364489414001481505362748978292939946765140110826461089348692336475546638367843627697644483887607014418892267649756480230131185357240376843806211769336293461760358943319202097765683832350550092238697517226479481615935700412929324473574897796927560283963738237162631087441626833-0.0512857133191817725604801700688193061409272414127098039930541886572604243323671467817088930082059896720604695343785184317473427575966492371849303984947407449152692966167156142383980918145077586317777128930019073419637648514279196077870631701681401449284987077165262282892940260657419756089845253295084759375031790882183760122577438433695269444941242451690623674860995110943939325240827714160876966988474219387852858350928019*I");
	test(228,"Gt({{3,z7}},zz1,tau3)", "0.1947745228785047003522378344163005539918884824109338937166759060509857512476895960577651179703121283538288229793902716353337595788662160132388870169708029838870399772815980448118818543997853486186293743094500627782688298723479517765399992189650440398075358730564334163038223475477436280533664764956763018991449691673635470614699377442582199485682379018029497674348752219945833618937863244425008986080570180585974858556256532-0.2310897840383609402617801320114105135957243518751612092685419424692047387824732223508978881277020429528227145461588905507794562383367200347221149627078220312734186245222328342999767270993330083399239286182138487954560835978139906584378383205303362082505541162107151091422546260639185071546354443217066100123816574254176882488155392378754501413135095047508203236239474344836871980827797253476738381056149511810770557706152099*I");
	test(229,"Gt({{3,z7}},zz2,tau1)", "-0.0247011130823387342356522068554871906899647532307314853975059041543997977407968109208645656771159568731618708451646645685542092279948615777209876115442175377010705274761802222649270201844102975373983713327604477684113481262232368874350620489282862927497513399548889533418564425793657476894448576373408455794333054467716842543553769305448890763387108056239225928845274803109229040842406461552225807836608829900678880266841694-0.0904794936695333164665775046440118348127752672809778034560030798442447592566726423279201931440507389971608574353056937946038748698999803796317573851422313737448938246860770992847889626993308692441391499774333532296401165248221206069678049298957335149076893938938520598338362087207447001611016926324368482112394652669976608212616606160571126648024613174396593501364299765895544386155923747139552405129843116226905286147184883*I");
	test(230,"Gt({{3,z7}},zz2,tau2)", "0.0929218860338823783036842109204179205900074990100868220923627168754726848179984709567420194297734408017947499579641706438132415825858735862152353322529019313191702653060558272872533777771029491387121199824872403363340718837073577760880689773059649905326143527971365831422660037736214910865600218901854353250853185615064839369814748633380424365016703820298447765997311416414030655320081956003946156840434277148187133758569704-0.0160187099745435143448098120992280470007138325408263127993822776095399235219367981084194932476453072874533406772373525997839247606281257535614665740494839821301792713550988201472044596882160187977385731379940355309229476715267247122435760566081513518896910344219023041631238996171622841899352886142822044140775027032797694195202853891729642928800647086150282740315164194001272514428021723213757461736862379036470799344509477*I");
	test(231,"Gt({{3,z7}},zz2,tau3)", "-0.0336569752343615858951474350970438640421294358637845066228881988485469810059446466949136679855286482391520226632989087270677745258275780007304885573122712988742714333716608599636340296230682768932538986684979438951446504992069557486077030876320687608423106908837994281396392178096593506952483165740700634205470009083911399264637514096093851226557950034161850035090048392325670389422032149467350072246941045190438480053507312+0.0888903981469675930407350050990910646773018449936753921599872256228698892139710942900953810826530638882757143827270122954125060412429380153323513639126794834263907546576051130223307607022888946170464388716389170194458515092904097090041669767897321428712542544548740040992195360845233238554847279701601103555638104692963125036199430443742428029878232708907012752355181231411415322699416984653640274698021176942287112521578434*I");
	test(232,"Gt({{3,z7}},zz3,tau1)", "0.2185391252419198780642568597602761729929269806888069802018697295019587982616782015242956540801418556189779711418435082933421962017442044848827841848421820279544083582845291756348189314757879277150066222855463045530007086826889260356037550484311039348730768820774373492296508324057132979050162446662259466707094295669347297840035564548798666982207352394428416898758878988432631139427790879078455528314695598638070624484633549+0.3086681403306305723632651719512638086302492245918997454216190279837549689428147819169892822411115391066951907334675436718537035816670680284586808022284277872370934548458083555865653458302712230309068136510792950120748505713051748182315346219561760275516954177832668728990328064179987477320176955143519256027100182255751752052025380132827607007839435487460142827661516377842889381586920189198872370366574624387524532681782643*I");
	test(233,"Gt({{3,z7}},zz3,tau2)", "-0.323390894237001148716795467959859151785749633546583425160257812961857220817106196065472993150313460211583935041849466609930316160098330380132117305457891163464826444454479701939050878831531202569321676000248310484486424511422922730371354121267447769747826452764319123833309252319785624471060167584175314873214363079014277144261854002806899753226867378332495411109504112991799928187019720926214646572158732672310426590400758+0.1874776221900670696927953299334809384286533775912029363525099617136809833288030550749357264862006265683714429859152746209475150468015400058140208964772351535508724229858284034163374022646845964834983378093276097142057466939910572404991077256632728004918611243534421868884470364346118488909058068756175406807941209987064086983330531794411081250529262006326098472033841741016033888905714541953434059884776129998804223003010419*I");
	test(234,"Gt({{3,z7}},zz3,tau3)", "-0.0036251422002960209024493234143922803637970261681723041242054280783324204460714752778948356227974325366873455520418981439185606731502477841643433801042144658325213862935201405903103757834315509595737175839484199253698699161716015542255273926861031498065472627447006858349509863711828541840182461624746436424629984111503089062790810310369263022231278415984878286340110569875172731420005615933385430933751824210543011726744131-0.3811290698025519690551455081525401107045931446256698119176761346085505581763949809108151403511182453045400966287575039432152652936414447217391534058521563518406332570744742736880750085108163253226025746330138112650395143409820804590242779276469247533506624812454727845641644018799187402425034518602310596989970971647339495237115799733589965540142907811202884545092880542084568528297700756439139808909340904488726961735751186*I");
	test(235,"Gt({{4,z1}},zz1,tau1)", "-1.4574499955888813733204263327033715603114999761781064933851891957009158318146017684604875471225337404379819885409566800695471243759480071839663102896426661184015288493239958308400964813762778914094918219774953815809158769868501756939700450017068118583525235808817776113639057121139937225212385908473062935006355647933651719991002192077880802366502739351970380384013828433042553855934467611592169407501892228727572945812011428-5.0836686093346150911579129146259220500022933245711430163835412799315273297875685225636674145970112846124730819997718350285330610665767761297118842943864089273260821423572657637669480790182720517157597690484791103776928716896989915272170912115802767654364962242180774510535881637748805157294464042188503023298739527478391946203431181540838989640153835948353556938848983872514707919576335271646105747428706828883889022190422341*I");
	test(236,"Gt({{4,z1}},zz1,tau2)", "3.3546221740510503185840544785848424721908743326738039910222392418590221987431464025517238520362444092861151752933448594952047045312258299268264720411701361815998139800551979762668723953536578644930784682905774954653920734258083604504269466916263614344006539653452465702023114121376928143381539560474196868140989559400920467352026099587300756374361376022388248420889467323653402539705043473570836406865506328118784648295977577-0.7790864173014549441094467442148863364228340841245319124392454820566585288983711379616382033912861501210865674472579673123826631798894802931763281453695026155107669313697176728867271463075320878083972075904365760821585770315791429122863528261253515249231186959663177294132545133917608448019140468844614426400489411103316433655174755264758082967306277021594775359775568526925369648412886685938798034065117437881536562101806397*I");
	test(237,"Gt({{4,z1}},zz1,tau3)", "-2.6888338937439668660978747202558579232847190066055118673343278126986609265818150266320178527424527595566377001667378071683067466514606627739919078041520266275725027819063596217310611224989382981916107739721307005703623684859094273552997942751763163876644422969757265093783718578987882684891861635269451382260113834467205706508154493516081054209130383696720750103746453145897494565654098063422865506168231697062832093262828987+3.0336426288748260202504095312210599920746455349096965773478934780180159341331860160803582636678726044592349737108254172286860936455312002909274608512346283383337071672134300355217183221653001739074247662681632942122901809839751857217362787996196931333902623055777449108169419166001525740083268505141331093054371956136959396733768210819418602569203529904013223779802027882373785212297357770908041259478332517480763515566631781*I");
	test(238,"Gt({{4,z1}},zz2,tau1)", "0.5285677786871855967757985437271680858408990764512148008525827850700061884839920585836578118250492480139260771704674639426413630098305050598645126731979979460681935057045459233461368614524995276816179427150536275342762847330367299034133686912742116004933855434648934197007544312590401524258283542796385033189355898756710843406009348846986503289434077448652765751079702095516817672150075224241120782409545463338814008881199799-0.0202705913674640028385001018462311044816709313705630238048731797744863157564734383659751819647804205885789530928873066963144662703327535701586952079002158960616139429007507817037749980301390982042823185118528264673946232186295151405314974515675854562223013258519801706899652113983032630080107463515925376060565182048114546454185524714527204354925496598025883380653680427685659594154477546508934174931759882250891072298534441*I");
	test(239,"Gt({{4,z1}},zz2,tau2)", "1.4014741452581371667305068733900199641904238065395200045702263643472849408489243759738055772652215629921857103209281147866253056008244700989750641937989207178035907315685956910500362845391340021695073480859650278359432979408552026323933405917583274381463606162741364406960085986973578602811995179219868966663936883403311890147690545145461660275767375176605423571876718671187918199985537685631080088638154820730842360435791351-0.0666257141628101099346407852322299655360104137800985612084789044117891586744357246192202799571085907543618647725150628261977959318118663400772045541470116481808241598709478837173224525609869930674898056938574656269295976698429338628978913757874106966249986256491739699346216238210242756006902334753974123484484858285713209172109575756442025255596849340860209149622555190484152446870230051413003842786405166412293024555765236*I");
	test(240,"Gt({{4,z1}},zz2,tau3)", "1.1399137239681850451639699292584631515445040343618771602660618518446377700168764311231063268525279466295315496109693126321275959829614887111178787753426418177338951473209270243674533544128985038773831603434143599608573737208993885548478823630664176420958971667727273751040920057419705708383472254789514124220376840615488656599949741656276034984039490861640110364866298983665178396914231083912712321132567640396873036104889639+0.9553390409629065075057070751209855686163759411426860751492035415707217512053781577488318186248256591679034443342623569871822942143670504414255000985871526664883448406446376377974888105624038739101603679356596975560459364562476142861424243774633874380917693372464917465974694887545634960486555523901508651550617924488077519784291880828929875418451144241673834810018602922305783908725940215099205953552296567804422129167858433*I");
	test(241,"Gt({{4,z1}},zz3,tau1)", "-0.344953404755198475280661803651803774452417246255817992191998300150391925417970540814674730838013475355370212034866705809538909907073856165293939616536973563961997674763701582388495535516239898374088129138396546760882789123880979209833928387373468004633290099714828142818075317950522487221147198542880277382300261452861320359372111112359446760572865572660240489562775637891163686598236162146820084661500952344715945582593327+0.393502729395513503728816800835670925988492542585996876422677031620154914669106482281958893591490194879533706664542294874413751115789043414576019213313041364448010267765444194607354350043817112411685538699401244734910148697121714082077260522732883858468252144903994761467825327067627007312280396364416105857745410240773303457966294030897662335212764755145689159549602367579416703577402864766162425951908724783644270496171918*I");
	test(242,"Gt({{4,z1}},zz3,tau2)", "0.4388260160152092133874888645480416675501357889006161914163934921090080194963897687766339654016895041975018565925163575727771422792740672279992525287900351058727902757761254815959509003258629518360020526995361556081391525509962422630633323187193097834767653855681985622475108201821438893972430377001739772897522236356528434105138351696759064384110402078758324229500048468436911707949725674939134734596347472912620413334004596+0.4907312249036252349926194689381769853900028340479463891478070095659464753736841325988720403541321319763800056507884364073611074012255330129284691230953974566417197142147758967758829561086016758689151390423083456391614642864972166049809785322879972403451389682323171005413793487687049369522970037764646655725183126502450821486752820183501898562458972262374630165213844650378758037315125236258527722837832733710945427096683178*I");
	test(243,"Gt({{4,z1}},zz3,tau3)", "0.0457983839707989684689377943150379580163653983439479759085015665447723618016945149258538176292977594712849554803285412651348285122052444364873262928853518768792383770315682673945281158648453075422715147320829561260761457042168391974349229432436505936754203768672602400564007678020847374896674775283723582201813250334166763576098868922483350027817699265692061264726207318960143504729886115123166238853577992236080378499884697+1.3605989528972793681563603074236096593346271996762222016665626701275208896550202438853068965421517133222321760668964129069864747700244594729046715816570665057056373808060768626774040713476360417010835146297771669442358404464091139063049342426780201053051150353284995227291606359296433069044359528703163686392921239109282325511432368166095189191433858795088689341658998165603393900179903427363471330663967788052716046603838979*I");
	test(244,"Gt({{4,z1}},zz4,tau1)", "-0.8916639943372868455828921441933989663347876770929805053561277066358697946384083765993105567614358390058490255100658783362322350116060787388458992979558055520300966495678099606833675804434705787694870599423859608587613145634522997533962085971937353077072381691027801904881042253393444621500266133588736831590185206833311828211301021676602610332620269727460832820370919987855350904968579196641033294235437090099242360533715096-0.7883929933810151782237180600483708833757767134262560925363363402480007381901809866926703530775384351275019308258337743450245357192940347799748516495143702129682160307556825657763765760402257232428960797871121300813694715037437323631830795975126983682352328958076191900487980512384483712063234835486760337389262420800187958961925306115731857628072578352543054338450380512891038007877524891579704416699813822893512320687133602*I");
	test(245,"Gt({{4,z1}},zz4,tau2)", "0.0753958622007141177694847390354462926290860593894852936207295693020365972509522962129490057494998100397380580104160029122043919968956499051940636163435025659441547957350336607591665721194028630588407715197607726386080980770143152954195113868696302151731922910517488265632253968422311001856120122240933372256410187407599906264524587278658100655077655466729639999133440080158716290374370854599925106530489818104456858148060628-0.1077768346321884823345028411974557877974137857889225395230437222661956291794893601789732145361468477338514993000152588862608004643024577055045781184895328906320304455509943539593496226300346994459745740040953825896167375044424228061443629376729127750971938551335899939541512796791613045179553993564690177945466666488268864683301614637408661555551896712003725574224718248053018082811236671066695986266611482793604189444723165*I");
	test(246,"Gt({{4,z1}},zz4,tau3)", "-1.0411873519846580556232874801187161149268545638907105543550497266488445178077752068863878783042689041337348651221314687529595549081693142323391242309189396611431037904497457544241837596036800676722431925461221391322734243337650548859521384103468566205818837746015089923084096716214535306829422757511829698775200313098624970743176276253325309021441683610015739738949148298360601988324974824017333797552852676447035935281433719+0.7649002593571044878336451403825110019933749988434695452660120513649873133000356414218985926591701473699858856096855164485349962432529256950943207069752585023175327083931809114156635844001660118368240020302789470631849144460780647077313261744202277478729743071427660362202009661179338605301157241020967678433740307686717420206275918542758967213450871047014836271476995890819954912567432606019575302100462563106780183179772847*I");
	test(247,"Gt({{4,z2}},zz1,tau1)", "-0.4518204658955862930897100817782201150119715562978902916638944379076906346878167064469019002397712907901628313100659765231957707479365327956273878962674844373562660500782186884778011978950578599596925559944291052874038241531555886330351635519450457895139502212464646499609504116225897364702658673913605041849387606856793568617216206290787867683252693756596626069668192889906342878963917244756948547335582259257096484108862959-1.5017733808230826375545698063740542003496190383288965555699776636824800018679809425016043048475599715544662613280342143040385810166872893296893589691747661283930929896601641367476026701513648316112791509949614455716495099801558954202906429845463959242359630079895678501858995682025156478240134842195661373327595397992114814895495722550275668409547369332335571043294271685780793647636455551282162606283714477699685718486017589*I");
	test(248,"Gt({{4,z2}},zz1,tau2)", "-0.1069281920533013680951751992610900575417916567182377350119152268197420265022375776117270940503377203706782801589834528290677395652149439237354952431515896828605206750190202253305987443991012425296015729811656584700659189716813094555161516785942960437424157540756996306824058707133830145295731132019469042807775958080874919558072991947964307445372800518006738367971098172893755789330954620921348790435847068694268393913153454-0.2671030079648021260520105148329821341694535407239631750397596505879132890031549120599087075698938061487797158254191972458425366632111526105555144143526815421646842142975387898772682529518188437826131468308564386673265266386182568050334924482883627388459939166336013915163348007646972281507335443163171939739647333817751563157579842073767127700776371361982278143290431037165773250597354598036797069657828830285161927792691976*I");
	test(249,"Gt({{4,z2}},zz1,tau3)", "-1.6234361888747254126281964816700223599500543581303617468052167059631929690210430684933705276646501155975606165095719492801790158593138460544191002429777916933054176829151123536252705704088839041371864472683663585941168324509823903539845769825157011779920416490240117207169500242250892046329812045597976991726540490622437584729726475901063476054133851175911719002699488195760209404317395128838494927669301323121800733274630354-0.2990107804989495872567195022442451137318907525825686841770425759775039604734504346269057687669393587943590915524148034677680432238331236199976829673673970277798682493117343524803808283475705696129186024699354332616933209215068045416360535431307903805285724280900095270725387473146519578253257270508008219675137890466156598591808747503368866104886730138475652312289786288728404180053844738467026841435032412474701987806003056*I");
	test(250,"Gt({{4,z2}},zz2,tau1)", "1.1071092531023808602720185292196991485871731407125071888530319102421204212855486578657883482032815667803384466710685243268898864438863390268186190999049949625043601501270347838483289773796263122272887733306504192916394334500910521450587006509503632518375989511846374226130607806201391233715664108982459383814878499357980804613589027114850826805909201207194471405552723785591114816713346175186463456670028967382553926761511787+0.3954581243013309270633297244783545328441427159198958103501712768080224974552979372507318617600772238439077636708077290498905794242081834435163815854070055965844888921936900673452347596396586367406762650516064333223266341379121733669671635963694223492178063831334460731115863250948757272847454947762103278337971361292981276034448007615486508717231748479402880385090852063100195871908965322524242486788592509695129016416893554*I");
	test(251,"Gt({{4,z2}},zz2,tau2)", "0.938818126055219325427480298584796730725234636693888996272282814781874415328232910441293855482191907388812986830241300398004429489416050485113358813370122449763990596602780612145243864867571141046544788735735611098031132712381913643985968208751029733823248383697675032387574243380170836757949586099816758340728472740869290436141434199463022567007209265995405042288361569936038912070486712503629706464710750895427693297347433+0.4806375846871368631981142537155187943853273030204672612467689457902684958701697298093436235218406853384928451296387296413739545898096437942423248848912216454021733890326740466559624382270009053773350401303776188190459337483773198544107664064819900645297153713117054761442605456977498329416719452065432258474754622966503585625541543636662095969754639184707415933669761690912062384112275533988410281651662978110093706202831166*I");
	test(252,"Gt({{4,z2}},zz2,tau3)", "0.9017548559197554477856121035299358275583991175179423505302959766723343401996434982074267009255861773582076510338649295777997428459041072904059172703401659212865691355616589864576142123289684491267035718137000793023167578322653802390581711451502056428632563168030394566858205292571787739523787055495759266389607484073342794769078857964718787941323437101527253456579509879576202963626488821637748568021117119971912325439332938+0.2590132510125551615725937448296643388623386867641665915508674432805592236414985489290898182485285587188460179661979049615280379132487659838772057905143915552327761027016670920460990357552830779412166014080549752962717303429553918678593131808205102182754013357081625117969375526752362807896309666866927264833147364290381825894539385415872382040669785700455771053621607298301345432536453190204340080852784953721068202131579007*I");
	test(253,"Gt({{4,z2}},zz3,tau1)", "-0.4871099984820154455380840586983420126740578290795056402539786162500101020599399405028150394965616401098331839950714051121193460948214978220772351046899857823997018227910320635217162609803420092777794909141227177808561925406557142971539309625096030860889714163328551076746376027020866812217359300342715974993271539016487359192462120349234654634302190963998583794957846586481149813761125895965059711493964569543958558224998763+0.4194646270203167476694148895688636432924232017678470812285408818139717712348543990067659790482612510748890368353019738632128463501934756092291000764142297006722637985463187446277720180467611344637729160772714291268554112859314134282795466698836739833691660909822373287017481579663525926209005280244852819498783527061694795509006045107819870866136883746691857487546413219847405124746250691211363769512395623281725294336828852*I");
	test(254,"Gt({{4,z2}},zz3,tau2)", "0.4209257316729205521514055270698580469751175953423240138347032129457808856481414366349668800935791051116400416891781869261007546327355278374610522329926190065824867899587788776215389716071234501860799995498955007762013316859379488084295855433996829449714860087570244156660682181166553678886547854623614805721201238078862792559912699445339944763129719393966386142150640646936134596523215527859222893147921236760308415153996837+0.346574789282458072356816969998924187014693146467295579967789947224549906715601771040638146489431359020139085224498829789686574666513966931914954895543772649317989541322764381932250825462901359031074711023794984161414949409411235758806476694444620283948919235919391269926084798066157099913657974770418995572150294996861806952428937218793725882943619393676141541755740905576511049572520764327008807742709801193200245513394685*I");
	test(255,"Gt({{4,z2}},zz3,tau3)", "0.1850961114931178833269199276683329168598926448704369578764141612041276839918561832707640182759952469098487931780592954419734603260385236221406150431069104989705772966874801141194452251429587797421901221827829044236958570842248620647208878497678176819540273036958662676711047166776310705633758325187359276211496874513797624409683340200817813273640885504214736115104258866114366661357629150469470286420899531038156596447501383+1.4232832441114722991859817012817958644304963120988460323893485752705097346818901262886674919147989366926615800747023797527672029981515851504671269403548829090589535318521684995129947756415972267762904951365903413656810495129790574495451862661504140640586379206913678597653652120887917746591954143646531735643498563617927549748397179617395993446507632948961950483860730014111438907924941976712178221871720781539903072844099295*I");
	test(256,"Gt({{4,z2}},zz4,tau1)", "-0.5698477704579148403584087491129088875490585148743955763332587089840790192628980055461276220009826466942304145657646003667962922761701395729730896388822014944267475153739105424252141455826392349156843493947667270161698168422307907131845647509181404479066382392315841767984092650090598004276003864069809102155830827739668388820295830194514185964916228015469534131791969130444505521784714954037113681472267644793414651175198463-0.1577761846071663051456269583548944080572710820440849108827090555538503814561745270105684250692331715459169972194391066371464026979025758779559485889453157940110167699539403059042640537778846105843316646715491498545108468818680280401308513683661943318418141271869571065355138674859862677527928249723001952181078350347443191050673764794338934734367899276774078255502854340073632410336701274093627025554353522525710884592114734*I");
	test(257,"Gt({{4,z2}},zz4,tau2)", "-0.5761762693958113525860039488005075419572082688941535225286663216131367256050017901402658925213624944880003008673219339256302528214846100281487069272650842534429535472354201038903942283153399185258201631938121833392644583040326054243291557886500837167418649689505829760240008109836960956157200194939965418084368125887970152772471216237511023373251869571590166494142881695490869451003413554692052360382213666255224415962145256+0.1648060233120571356949980574970345297918926944568686525111574356935384775334880581496262789914991054532857289300525476094436721525880647989651183425196050312213374478616220328269136297384290502821983669014120035258737903598706904041078740661779141500802193054364890332055518923086386545300026960412129302853471869672739001103869247743268289640951692709807152111055517743677558300060149017380390410157009039401855659720984844*I");
	test(258,"Gt({{4,z2}},zz4,tau3)", "-0.9426104288505844135809429008671054575689131714163948003432553828312154841477438386178273037700084441450201086848345665071397886961227001773266214322695825410151071050119259496288930963158634023083800771517710137638510991993288099798819599230397083047965829562815338183737653731545910377029533071763418400287688100942864391214624141087272231473439996574223747152326963850200122163278859698109747399319096965224661840482217509+0.0417456832883457891714121182777056384277097202535956230719631935235714731599576555139153370490022925252317724060619939131916025947776366496786637078009119606829809012637434019057642294158077987160675554166133693526206989352866632168924561180189448368652357302846972636178717254143072618550418579758290752383206799488547173441400284899046610728711501135858261166254634335920244666125234848928379245992045598602029502255823593*I");
	test(259,"Gt({{4,z3}},zz1,tau1)", "4.9445304507581303532560915020227721062638407034550208974353563991249999256953523735926190400388642505269149911297108685195970220824770802328721502280395655646710661475784936480562049701380226912944984541225239288608442076739541566397778385378859649776159993427487895903444129029954406310772769157208233206507262723569289401401698161382966651296607350809755010346635522575520850963059689799041844145460105337217347349570125577-13.3128241470312046559470589507516812094914394453909173868944479765057205727160271660660767950124406132052707272400862856253446142816102821560481696921464639585351931437838435369997227961088499521208531184584488843773732320276517322247588545730685055734857116070166752983966687562659434646243459248406088726573030330374774638684354398979156375338170750087514116546721705898243956590524153756243623999244512273914668429978304545*I");
	test(260,"Gt({{4,z3}},zz1,tau2)", "9.004854830746835708774961118561991805118685971529956112421132709593228406159021380993816789637365370003288812826676042422390784342761282246727940464513209487561071018372192291333605093280295795428716631176501153120241689267116107238309327460536124326764194937693629696803967625237661665589896452283216361759242202898638050867720919445379385526567426573863323274177909023103934002263033101941087033708530008852682127772985896+5.029178084371290784243336446265410390618370680989121322177274406273580885565627843692505504592211369516667992934794142108051819274764249183779673874373565612113499217511773786515924260399199429128801588289979308485886211706514879401412160562512226909667302555701087924980525466682168065011846216484477990632737719448833116110525234190283463157983187078663366719158734574017216574166413361360143958216632322721872665457253226*I");
	test(261,"Gt({{4,z3}},zz1,tau3)", "-10.0043445357548751755761233080993343143000376373300641477824846975492260745987391494231609196819349233118208938159941769186720642495161229410810224281970769894812566788745936510045694416068046568863026325541843277548359766795423175306263245298394516751080183743017084432021715065754921843157726992127954875495493957930533681394137032979537085908901287927947216347511556636378971756857434401152160428523966098987585629040404049+6.886807318703513119602392766213150390138665191865951851550367000597862339649700678215780673335277114327972470082351546560948651223737631440803390435179190918070142340540144348896661515522333225917253752110206425178314524863738180514248851857717822786592357841049820530337708882286476785652876115197930809819207612075687834719768127523271261308547918948979420644925434216139161537336505549225489808192518222341069010748809877*I");
	test(262,"Gt({{4,z3}},zz2,tau1)", "0.3936374849084716151677224088491264607131361996153565953362034186812525371863500615957724446179702102050950864452485994100777057240391340652385465185997513363167796950284510995466832551400894272012723918644497527266496788111912170464696204672597892150969970816594508748775534309280419590157317436642612942329407096679614440028979346243341097938954810270177303288671239532351555952646751517715103200451943473504447588785219605-1.3510742073392610190787295306194243180177554020359583601820280870169941643107787392230588004318702653310639702130339307570700444401778706942938387376205391940876695998036577651140174679151105875453527791247763448975472561254187214021054893887906703314121765973076296589796565240604631083817002485038219999898394706347674608874830129095157906075472622094540163296482173179037890843447295684148446409488957939005887585837910152*I");
	test(263,"Gt({{4,z3}},zz2,tau2)", "2.7603152309378920635475656423542754787633681101170835021528487220656260432680546186946572640808385260783675686969245654145382737337358581887480648077410827149413208386695040659204892142085060079162939237119616130236049523530922615526415094208332334685564075093520452146458477316337874409463901449359314146256082869736681813725154052899568568024681517679031931178177487359509120543761638655689111205130570606209795929591364946-0.115462870719421916410014276915329906841109115849203894967781070099393147284517000831710602868092404975267401938138938762709363764254959677611458730048923201752629834170566201635998895955266740869287875456137603530257881812878215056757165753554899029162831302218355215560155335590352265877259622550282490978239124723076981697483818418386597407866208098749658012713877177354768489978945199644594221008329896048845596919008189*I");
	test(264,"Gt({{4,z3}},zz2,tau3)", "0.437703044154628020802886680782073792588903647111081659988261011510536190955052584470624246184111619740036007450239451479305146124402210797290760248764548135553565579814689321986750405275214456460756390436282613188417091394173152498231361108955867300340622626787348394237646237649202171838429731320064564057358115755332781305025454208153847838572376241181014690933313680412260311092427907917688748622488003966443054543885234+2.1809157161634424720439560131778817017186619055276329608507916136154528754052983898672440459019717483146882089703371369850470582299712073159563839798088349228150195082073719149786008186834724788414709412870261434046285598899355263712240331511709404464774774682613530960618179358981716018609449140605875591392362018254716032454083920063500258504574792075564496804217334541957972597247592416775155713540148304853918605976100527*I");
	test(265,"Gt({{4,z3}},zz3,tau1)", "-0.344910749602285436229348689148336355624420273509020657916372856421606408893473868336415034517411246888826143450125446002967105145375574385876985681305116903767471487256453206448054008996131842122391032223611346720391690474852742482822419859514990541825984551359695209631135270962187075477127899483239707606770137408610579876941152269129547740248259021795879805992533172403274311074443447126002126883509991270883471925021175-0.3985044444814693412990724005983780844479225229463069729183295550973663304126601696322121851562025416443436047777972392105635745649070352149068880275251390203057264265394448514925274599214067693950293811988008511075680306650710127694470702013136512622846389167613567839743785513510588892530778161085562170967645015140341950981844329862169647205687535919954871421706798920049090263325223813393018032389548127297795827827854697*I");
	test(266,"Gt({{4,z3}},zz3,tau2)", "1.246537266478095128977300917228334175622971175068723637907295137326595288116651990538708314266196506219057694349135829083298203679668599312401519110522587894948280522655873743610204470819017342217453976978734984590334592739168739729202463497702309000319082132567082828108291298471297872720726778588478932031666147726296396681287304570689196575156114880114068371162524719039261514014766015298192336298278483301702146055987394+0.5285769420821044990978254088472166361400247893764090326242864554139112556377531358725544934411756841736824220723696416411507200230761793109965145070613531177183788026488215608543643053895685383071673167552060587037857482186730898386374846900985815278123282453703807022910330485113446197722523752354778235532947558714990364835703407398371624694623665262599828253406539516084970393839020625429067954261344250976120645103763361*I");
	test(267,"Gt({{4,z3}},zz3,tau3)", "-0.4472733725570919191761123449653755983141799133223699714572681217273699509498151688561845103434569985026199926499165540921287789468078952640275770621091474306219318988726856067220699464208174115812565521150863657461337106304248448663233860959746500067643402525900563046262329612959852636569805981861351377932709419241084202930421493280607766629111080811674303606946954094303421653008550536832662350464006428345635748490815463+2.0595093718686383998629961845016808471639441408205755218045307585063719193385162658860249316419392239486249138672197676208428887004526860834012338636797070895138260825211124447547785088690692038656669805299451013174842989105348098331685209843340598899534613151564671775726271431497767421589324185536567698692353558398678080558186159716256611602977861578024408370243930794665670228253099159201207145337304059145483442288497462*I");
	test(268,"Gt({{4,z3}},zz4,tau1)", "-0.0647241047104843774554680485994898264838930299315617562907357563075461715925435159875316609306478747121213358742444598652149603219948078105194573929660295534817634376767250244011197086294192850642270117491550312444630995911266856902846106622355923633925897869747118211019212956188564493099338681394710903049774133865569545248431552329124049023678532948259439188155263395435398716357261753148363847379189177922450607952794333-2.2848361379795953573481987797699533274459276493809521139871817408290841282863378189610298429096362977566432151383778337333600926149583274220042009966256096012417246987612867776447403939021367969684191989994720272524208872526537803461216656921862011872896243635318517287804250806818454110985414923617159520281401729070528002410351224068284354410248276126812166704076624445734767027687511460976982921572010517171747167602576115*I");
	test(269,"Gt({{4,z3}},zz4,tau2)", "1.5031428028343233619446759263084794176574431864748057819506433633007620679279067841591147364500645581225688617701566112523316148378550150399725486016924512213078563889742239059727281174899264022498232918669901219161416374262861650858205530320039795296947315995467499181921376886378267824101258837532660948028313397196328852616932742204190114240625330337463191957278897397577821423650822690775586251501146078082082146942605536+0.7871700657596345211381180756591765874657592156092863542447094133930337779624414802338059624194736456534922962537682761445639360859763155763824516425938901426779796591471171581957262311756624550203871739381341534803670342866976987434356150275428779817692375603149618703475956923420285829604282637085998293386311132407576634943992686586802770924851360421703964308355618850152721629551867199601892528849723019313503092476143568*I");
	test(270,"Gt({{4,z3}},zz4,tau3)", "-2.6234918878541410937637904351588548557166285604418276525754680859826578525883302617506523697007601608431338283806009038233069720062530749238648780443213123985588660857817070151988775225835751492625576451132491319661906728528044576482343138834161602056882837510929665928046794015165797355001446726384973065901144069853448864200699175976946361691302495952415803515368684083788064879013152412207027259737510203920585118515833385+1.5970062731396380802399557474076537208394099071186402926525562855179949491868109917780369462284087477107465345827107698578342002504217437160730140093303881222514740292177101662914036333581411645081027353099411684907783022560055119341165825855542215012106750197241487755323110409715581873855952760440985854998173189121768713526980673234935080772368272195340582288636500595458683128038865768169236937327736585585422988157076806*I");
	test(271,"Gt({{4,z4}},zz1,tau1)", "-0.1003794396092247135622822847526216485683246634735136506342288971562054410957964810943645572045640603550529732608670757245930038852954560902441062725290703063013118391597347204863692016035690214538571367889449901429072569552163909447266627266688787303232020705641322060312455546032950139783060076197894548392640462953406078742322691499224542132382534542492651259246757977724367543916881276768719361981588335723355710383976789-0.8720455818443024323013392911268205045169883114770598372089574106481635327837209037183967720405179737441425202053573745332528238943817000536219861920531511009875157997333840624891168755077657103620596269035408831908317371754755091488652884965856515768505378297379712795815453590042257437551291478147900364409108929020849119556394413231192838103560055322962657100721465913438682379178945904248980705201964818816889927029187982*I");
	test(272,"Gt({{4,z4}},zz1,tau2)", "-0.7506493494622933174157129502560003437719241176941715667547767011566181569455443155167906952194411107656916046212507037066987700472358887447455354648150775730309923604670669258572586787599650627131192878785306109653364507672229374079777470685407887479427945275000775373726438009235234976947155688492583325916780079626490766120336986273101478630637967575172452629469976481827388201912857090412277608392253323039837245446825631+0.030412160318594368964249592526313834176533542457572715688673097544701794448440906770711232667701045931331553202245960420673571282468069213220522463630723275239864799236790980968398624408341609573644318916346718187388476092589361782595560152027181773367626592755105673993062097327146244378933263934786074510139658308731509075124560816067262825316603516739956467588645262981799378456986803916765183333210563871001496357822408*I");
	test(273,"Gt({{4,z4}},zz1,tau3)", "-1.5623679466630948083636906733199876030801349411781299540975584796031661978313585907442344247577356273480124815922561626735927170612715188509670699882879808455297440742261598663920186476815239007342830873522818493554465047575173417556278762232737623491722924791167205602906341123235860923341641992457045970833294126657454686359654422494961795194410438683555618804570106366796500095548016229716441044813709363383745976604268186-1.0324774079645407692370449454345641736008847149404601094481344001278648960802285125772738046931140728017663748772322523792574945135671360960757139905351294001833683994194438411202815166328696933292216187149738579863209563492258375091833460833371744301437007808214258259476776960947689177919826303690336496377552003097303914040191268869180781833625671547698765230759761680640636722036182533958076882854166575146617837271427533*I");
	test(274,"Gt({{4,z4}},zz2,tau1)", "3.3518774813838410494316467093729525784586311241255555880107679577037570601001352128287449482093025720438745197016467211886905262931679986435342859946619129513247489753302901498325410719686900214157067046562188712503155127523112423942479701153547832228491307033273466329116348499408956671382587379945656034879167125416962813968490343560530076810990583607753498835452034383150497830134040382317943000798496822575935579599174963+0.5904457568719540454525448766892393368516121978891438014364721769890976794170606131450139233006503443929780953779722264529881107725764601416748533295520860195222866043028417149645388501848096373447353044219483636695966365459307333758666364703882972318260099841022023250978730997220307468486318687789957466298558853997940930554437416015621892609492792337167853713117062237630350192575051594501739091956093931704104996142145448*I");
	test(275,"Gt({{4,z4}},zz2,tau2)", "0.504203125299609658385333020313485731657241325446736786697560100882846441999207582260700038222995054872556292658533660480265986979571497906055108776224990248830059203959979752093117670530991525148651932122895234492230322177343259858019111062653515171609222027140005247226375408450002787945874921006814220681598165071867547846015370066023703728449890741325583810092448674749793649627310893183240537027628588095719230147205936+2.7246102538534838829170660507803689984122529445514703135534522803159797161895727281601972721028699364351492398014132345566693717497002628141643095076090131148553050652386082899673885437201046773638659228368102950488280901661101700205135051358663579257094191621572757351253675945487606722805272118921229722767407732031187691613782615214462880635616873689511539661644202389176269560436860374400493153155011584984705670056728833*I");
	test(276,"Gt({{4,z4}},zz2,tau3)", "-0.7956458404059230759573088311581878809174129221629675299740883939524168480427428216942305856346348441462762560467340422599215337098767554233392993338659534913525609289271293490018420608038772519576171223770092372711649059407852886204351156364356804869411321018472840154432860509900386136640949433363162213741731531757317557699346795102950013480212393325254099154978092468896893776720298074082545520855572728836557784428206488-1.2575895474876265255880441582669101863742335324530270965146564018036325259593568715574931842338461612968167786396267388295221054343684600421136132836269841068311662502576007750335119039804290678058092915935068530727796079344866701830144824466516339989472707745777562525053876378836225627182549209900675045648453595220355160226969743688863668005488138175166281377145515494394751933226768966229370203554535823675582559813487382*I");
	test(277,"Gt({{4,z4}},zz3,tau1)", "-5.8200140049139894992113659099912958546881556801187329079275512928778670317385680332979772666867680999259568998533111992405625605529704029325326403501037572641610096582595777462887003930258191828240597119240218378502998664932450330318580598406018650727895930981307679219594686751257948183477254570602398784581381014608633649059808945028748199841744906416366603111689195852731225105088998464572344639284152656194125601962524916+0.8481484040651768488440530355127268135669603533083688178360632950775658695041609315685109190975383281004761149745712027467682497920887703793462252239259786936711639026396366635349447350917850116444571308533514843515477128829680185932341728218675557182808163172873851598365857681749485535748551885427987110858202589918037213873600848866928645616868720711414326957552218188342887917788392295206550352865086352583674561721334516*I");
	test(278,"Gt({{4,z4}},zz3,tau2)", "-0.0680177972526426662482392507859682917373195122749259093555373261262036224773410462441406341026030088471105250607081832438514626161575317817991407698235015843945148591574218366593807846503509387962042362885012884880154603401813396289221298544011350283292104398424565401943527850840330901402935236477070165694986194455381017872641759577776641444355334028074163706830821634370048101370483568955481509582650078730776311642202407-4.5568077475186644715416667696411492779619821811391867849934897415242587975956413049162271597362304613050406480259474811965690146623233547759298519732923467175837406573333544720779536116104882821361549762360382432808015467066380974046965579396916603289118329318115920150705159679706466451043536151010353802164893375063379776365965965967718008349722966190620453569482449234957579477067179131410788382005952738425760069623586484*I");
	test(279,"Gt({{4,z4}},zz3,tau3)", "5.5825094790315004186193509777325808905693526927553876377789420594416857531837355887797651711499018044337261416153202667091545618866920472871576410083225163517077183343345659143623720422988790085402205823834938272050551739259808171714859302974795340992670889778470698518326379828886569492876633354316683137837707313222484848071837814117670028199029753804494993034707341435859691715469823424394240341773120811608054522549145812+3.4792651331202868724306640092915201465821131216705816998246419047316224974746889281755146530492588265276564855823974028233228566054939680572274287583279576193428328581160099878054668900358223548313824166652120437127473313282282328878987169996773507774152869917733743792615122431998793285986617376085315751294263853069650801691110281973121173314240091176062880125006143157777061164630787349245105664065812364921202781284200126*I");
	test(280,"Gt({{4,z4}},zz4,tau1)", "0.1625996970007612971476071118096821310698004365288222437013907941073661282500951359268105773079860885353850867381829684264742317441009026772628981801374555459100828376667106846571090876462390073726128780798632022696241620490385597327897479202652458824753886761210559181995468329160776205674911218819694660642140069993143247743121805811501721366550555108623546135187130345351947112244162139779387769151595069447656741640690576-0.6245454125864514108805571562075094120915806339385905712596847803770664557422320432145369765866791958795468578610019822056911156537652560178180955659581862446988462579363936695581055321377369500307270858221449434695221935966247196223978083193541524082120116635803277306028456599996140670653378536404057596041257317020631659744498935418354011103071570299836171538084379367479105180697735066525431973081486449540396048054547507*I");
	test(281,"Gt({{4,z4}},zz4,tau2)", "-0.1581290248191641804099642353506525089568031002611271996760573683804927491829803194511680269407072325909640532007685511639102243615561500038945515067316856791194609170582300844375540053096206913232666557628915777296382421028107523935026474323131499992709609873726140170543729714666708154101929776555698102641715835697752737395640752408372545290449930151443353780858680795713006950583525412960663593608998180221568680426458237+0.9332631777307388487480298754373641531709383743618161258462361520189314508083642688755282974978796760128868468854443920809902092998394560083451830552166310299121029805818486186762269805637207348943517720342409363903935021779709405033235069593967287558425990147359382686432813476672101191406283880163402282273901652896706499602483270304616450789195588405054737986339766202774724999187970603091850773160156936423967409589378962*I");
	test(282,"Gt({{4,z4}},zz4,tau3)", "-1.8486998020190928143132750824487721275722836160740728270184835839304452453578026884218920153907965530238437952244073199775258051885941995091672033411384283616610906219700547608622860718608482902650044573421643041280640849396194678949073166402187800277281746828945649829044499874531490919380652332224127866473477593733574341966359588058725194784462859399876853735850045849904397077923067350604482867084201492149814854861074979");
	test(283,"Gt({{4,z5}},zz1,tau1)", "-1.363257872400126103278243740824833838129640392168334605813576766465666982198283041345386170539860307607194062813460073121972155194249078971199709426939437859708925520293188265305918310214467986158545497585327862081500950303653493005544179121818586041211174204120499157133367055830169560482523476632643960334608663703196020871414289647752835487472257588961995020151886845093551067124462290479466010476584430319635594040945821-0.5819863113125787516691557679255434413431900392459887589554201359201102799773111214879198459697791392212764885460251989202614440828705578073750209995319857968846326033049106065831503166860174637065721291453837133871318448202683105440930611780693167112243110144428229149473040566499584281919122401891577487848490186510339598371085790112670983159510781422218670811587478918027705231994913299044707758147257069364678518065115278*I");
	test(284,"Gt({{4,z5}},zz1,tau2)", "-0.9390343360015161465018369050280805475000921599876182250397464133668308664450384515902411475009207599524914061961623059339894593443253346571367346362896487275231580694056859305257072924221005778584939676749927240237044096002860546629299335327160268582321189421553509491319647857934517092108189174534651274622689427325078470768073848878211729539130660803643344547937300935538970384976107833341628679173658699505013004780765295-1.2550072653454671213097049805978631432816418622712504209593820050448147723415823295201354318077341661544226447393665627202652798195438895861815901668005641124391571112841237622943489893632223276545963603481284516211486451926453082040862469179650581724303187828329333834419505327364040482134654981874891782663315473989453703698359816546688272035193026158860549264813225879260181562727382610728456955629244404196861299656110339*I");
	test(285,"Gt({{4,z5}},zz1,tau3)", "-0.3289595479510990346490665124059796417413379353508897908781904531075942419334028504350813135737608405616575081939758383519052081331608699991466887496253666865176390759101644424133560063583143489313535848655749290441027330549630584726837696810460141856008219837514824439659982587231592154359195978401853739010101426999391151413489734061613792952365550546460640801400734856933863145800143296995656357119208328979772454520936149-0.525898060952581584477127671182571237309069704997919710404304667781249028623201046005879060454594131863492846801117127645748462449829233951833044295005925196342186484559385471281003547377121328925115838872558469078448152133925610645845497573740312572372389375263886320572027398277167044040669319924357284436446280811769442442749173300113925727883154235305438360565433223508829386363578753118727807097460961987538182196517611*I");
	test(286,"Gt({{4,z5}},zz2,tau1)", "-0.4101701927582027021133780351085950512164200283694225725455998279303569233792373284584475528135146403820048232183994217948012608152940156442827549739624006235041987086070151309984200624615344506590655250982547755786871919803619476544260788183267764454872282644266463266198344044762470174635706284607793769759008160864613031860395918682680363867301003310956774291511250847445957025977642330615303591591893529019211618506009895+0.207796338957872921048707379921788418588328272280924651888329557444585562051693460061144348790308136818077365845319152932721037359917906906311560358581602683411299116935201400300212621033778851954339343452746644642890613922892294368372717659926855216516078644579423587496450649348126483094699576671889421689321787863330149625141131899182751185087413555214387890317104919275705105672944656339516465875338237232027733459980612*I");
	test(287,"Gt({{4,z5}},zz2,tau2)", "1.2355247638729934926480088825145954957633989954930283502506937325290636402910380562948805805071124808239306025518202799385933138747770189341661043021723607978767062194971877894737749608338499956345878785284635188264293328992964068159639521403301552760099410155594446512030982441158015916984085870897838770401596879961692235945626006879197013401606104210264332778971634853259132628366853394680759781690924475358880807120335599-1.0189340373857826983121151265131614685061220307515422120510337438850690491725357870488675514249484902180714523615397530888542911683886960703476833591733187629158006588400485089968443249262064784401738569578203309443037411503900329374067546607984743230542675979161655261443788016967592350826229807660607017885722754787727657744543673536301596374401332944425288007731754143523677104287824269660369937228596940126829030125587467*I");
	test(288,"Gt({{4,z5}},zz2,tau3)", "2.0820083744868764053442057520362633923952080310542987969087720356425716850470122417108944809644057505042540204122131551028883037968283671211097766285048048435075065192138939726594530665935969975231269332642198046884760369925965612918410435595258890570711163830263530209641981326994271767745801339000491296378464193360565646094649165227532566142351149247713804383698687257803607933502722626042478011665652493540525519925747321+1.3157428036197032402101343470930715678990913043449578664145394473551809190621685578027675969478137354327581327118407375642662396368461134669812851237883212540103367444772845782999261678126403688940866252477645235733193110557971445758666630818249735833147418191501953547679343990950552256776407443876903056041839043307604521852690193622537836754016578220432389691060962724707562935728208069523196832547386091359347081256501221*I");
	test(289,"Gt({{4,z5}},zz3,tau1)", "3.3656501067843628466314769028816316533183307724641242517425831406343185020868213304401955784571359925385870044078759291835190726008114109500059162714652335615625712624318017507586139794836704403992998951460796034494889908171214067237877531235288776359046902161846051667636147768879770912211103070531667944704380965540820625267365553299716490105505658008540913804784787498764986334898838197918067377722551077491621834021986421+0.3201703915194644945825758686961559081531350215872191754692305086522718528546173882689422062952041361243494776937398477528818536460641456233906582668945616449183141580421047858218634847583359338675879444473726967535287480778137125065034746725028398370686219880484790627834243620949092646075086735580459040787643024884012635204081354427003478818244378752851714763157111814536940522753655044004256718043105452706582471276442357*I");
	test(290,"Gt({{4,z5}},zz3,tau2)", "0.0321345105639264857926480335969270223411429907644194437720560924415015196830666774565993691883769457478340931929506828653571419455930149324561057562278808982549317680851345352684719758081547439792393610983731516624272515725337063425383814579612620629204063623733602161062327567724989853407233191015674634422820050275477713531506484189361305084842696461010508656829376951433072996743243498972725088147209264800563793351101821+4.4378541675672908315254173589006357414531508010694006034810889701624871846272009111012647624402665686969148477610018830462339730618408876636822055057023343829903042316914980710980374544828037322314755793005525861721038853680382745763685976794413037447655508273232940336011683193309932025017269944808518122541157256394558861658202717658708720989762424594872315715005691026034083674280128057798930894556866826641980672233973104*I");
	test(291,"Gt({{4,z5}},zz3,tau3)", "-2.8900321932506447792417558071177424721656103345711263822966348726321769227399946937123982780588516175057592913906538275094030305722074376988969941892657738922701587673451431830968395751690620279554542647964907267826926046694836885248641287527339180771716545433338203325963601119445435829434932946493943138405783274623192130675554921884192786819826249142285318245367920114397624427887828474761634036473138958904611715345366153-1.0441517579163018773603377257981091713504466076905701801713353849257923501442507213728489968523197321227537261478434139561158973894300788665158601849517359582501507272510555834186944942875099482367936844630654517917310271660288461567050200220885323488671069950777328906200114537215513862174709331820028524476993128283287594301726492177735807827555366397779801194739731005746879583991026471343554502452426364024281320027665288*I");
	test(292,"Gt({{4,z5}},zz4,tau1)", "-1.2622206182149416119883623579292720100514384806592486440194366139150778944198927200704400030540237529635037572882644033608458525840427975914042288030492336785756062657232676698888821585561450091521578126202452944425221749584258643733287217636915777609596154260061532524935343926551394725015899245567565047994325892663804778946105978009143539905486985362274826627341624703802489411770182851898843763532549156146000302035272981+0.3877142989750358162426591606326909142280419921320695203589078322690830137291274864990129392674220549770019364297858043922745715473552283462279855944917414180444097169816363419882079059742576997679646002209236418114236304659581220326781609601520960440088503030078484413790324042249286065951016851891978904758605379002734742948672543927115431608079776711097096341433547187401690081958603678592838547377650310288169154183487425*I");
	test(293,"Gt({{4,z5}},zz4,tau2)", "-1.0605783538824998740881480157398609908685140037398901395070447311499287409500670765340550904107946672340215354196430168105581705988242905734077363950827997879960047904326974693919756271095640629483116496136816278948169791900789326235580520952363746286550733467129149923388078842307939409308142527564835632837770668772282306675271534871118377614980047138972493992681806146810788755726808975550286385045497791110422480720183586-0.5749753040397486621655414889485641946132232776478857839988595674519316054328365071491838333933721752954489460250356696679601143511588056088680351970368655937848352613600978420866973179891851324278020856335486101951052710587749240735659823290836349759529478091075828216281672764204402453974611804115225043612697959346124591848298257849540210417453884613937329591978550611390025799227909891452798504508275599505674698944405598*I");
	test(294,"Gt({{4,z5}},zz4,tau3)", "-0.0433002036974815039262655399144819865951420130775807003351341341092276437772062977603328225141784026059567629311438438495781225262110956507023279438191776541502835648408259370302070354878780783952917556072174418417276432369300153063779428541792772359524061024424611133079085122139764268878078869039130107873192081281215127805072949994191361923024018483965262249705316757583841421336064360596059357131633649136774176261542232");
	test(295,"Gt({{4,z6}},zz1,tau1)", "-0.6574774672717839878285360876773525661348098117510141941603520225432984375490876571266060193404498712870032418090406397694059453492008768456649286062679997468337061246202071984893508306441496073976219899380430668035862037991182631601721892654534430341934324564166947592339997359864436025904563629846786001260737026921448820818431761098413877700316891477933075943077026287817305353271630559582018779578752740742208724394550548-0.689043983953127317165569365288669656373528244867321635014949532149705884354880481653133360774953389103864043962580670785528710924621644015466445387424016007301440955504070248433162791638929177865746791899547953315207946282937648829204021546338054610757718748703086987768990481047079273227704942225213047707086934234736095714838216232189614181601101753830114710638910984518385262909929959809783357844747695116463149869707689*I");
	test(296,"Gt({{4,z6}},zz1,tau2)", "-0.8924087419388749579424268612303239316142737343385359744902110855424807682647601459331500133814368552746541356936321151567257635961643647060583181201599401542923408302172946815463337254852561213566094308394810418919119597901257008481682744349500420210302116356929889298662515090021117309314270731276582170040443650975930822875062360938712762607069423421529507265215911692576286993452319075453137567423438104214880389323142767-0.5357601940212916444299889993748754119475200982314969801099933098387059780291673772584004790128153222414824078396214102284708273927118369468666824868251378279041796407622248849843468049282620797563198311694593113414169028757725568936079840876808172181966050655348604345367174572678130522776738063011440755094851508580859014287774470363294611123113035135542278429083147697508294892825264563898423350683920213426298721204946159*I");
	test(297,"Gt({{4,z6}},zz1,tau3)", "-0.9808802626695968432606370517990775306048801035673937580958749253050056467922425128265276529101403949942890061730322715808048708270825092847057224709977928147396671461866932989999158732957686375218389316261049401669527089118594805342574015130069560759360924091862921606082720595545488455058375295411499886388830634925384908162259365387474292397968730764522340257873592876712437371516239754467974585988888348322302517387953427-0.862117603154461712691045664697897181078292298543285443349016510189428855490747664284807275011098288659762121821557207191901076756471381174833813265250808815834880125994544220605196219919950348701885333692577394694237851782710280632587359126501098241528830383001942672692531329092176691732462919845048566407891840572245220961834252010942918653529460167797993427440748371035347802124424426759552882236735022484922027560598603*I");
	test(298,"Gt({{4,z6}},zz2,tau1)", "1.3750670068510821540328593314797676218680925876164609133059192171161062398137345214159478923783237018894588962685609918741142483583568307708670753533556169820702517551820662334774681213480823965024472515478350990877390282575037490503468781140926859642505928017012137652634563722853120220968004595989924841010764008914117619280015228705041254067829753353961492660840610489145897528151677543891094607464215808724419556042789931+1.2834798562114060386328736551131897582902027742887476180625368086866945566668313881897486212150640402907633833255580337196380300047593788891992023636440175523257640695695057772942394292300678508814813912897524375643832413187609576094381708580438076651586255971217197494138371877779940382375352156676067644012989633853673150645358441375086211715755398622376581777254203311349501119031432029359095194045613231991319872286030204*I");
	test(299,"Gt({{4,z6}},zz2,tau2)", "0.030780629267757702596718209826499405052809866725359663892833678295635145253374392024307347116331902543598436914410046037251220893119291547891663031927679997340388741297809573008894578678453619450826516171179721310716714661886057576102844873475872043044883995190138128534667690317386026259872185907161828754232971819388596547734068082762159125417405081809395676727452767169905360450809642916444140926471511519237940233174006+0.6661357426575318585169817903197298592419564289256277753499579776255324455700540190810210702143909872700274794210794218853398110278596231398487816050917241185368212954179298522440953753193064042654694136063874803822746199934309927041235012370556274597174666583465489607594893858223609667325059232211409557868653922637986395494449943817225806047183621890424441677145756358042872219832797742734048929354613175824775054144454355*I");
	test(300,"Gt({{4,z6}},zz2,tau3)", "1.2033474220716735690238900093366205848466044019921355136406579014698583073335758524476823104912706796431032883173183233579185846210443158354165161233918985504321942077038669185091120060204416247926112443715249210302132274061665621347087781847332243304918529699424204203069056374997672958264232582314535999395701781125105143357943002681302390814725398044722384322302625610032836043020767864860500412681788754432039600253980009-0.6324554147105614799863481036263136768014230251497311594959380173515288017848213966893298958365472160784268086338962899875500426817492555703212324471614712887166353844548164742535654789717099081952054802801882248912308657252547843088529205025625464372868833514232567505934408742252243934023362913414187395057660326924373815611264744996340392863999704763699337198233547143814923034552938100566064661314577356677986906911147785*I");
	test(301,"Gt({{4,z6}},zz4,tau1)", "-0.3037906688686965398890171148687517070523398468000267021586404246468948792830131957360871690175635203217056084233572541437788371768730585504043283053101784920816953120451176659609518131196539199689871549061258887788266574434654771453208894012713300910675740019217468640546871483514909370974926339181848652378620684094869015305561323231483583119362883190326292365502848824159812216579108312983984141742388001777890857104769198+0.1539511775856699939452401542518742511427509422830370914929410340133494526529649710099987230607150742061506466708090700948278472539700800365191473875766974743357073262638339123146105791496316257269235298132636285444983540516896174225897064363328557696822668155091549075666190355115970315367312071404336888996673801361682623114247674070478783291239410311369730981410359883237152969197353152508762985515439690529450389112778456*I");
	test(302,"Gt({{4,z6}},zz4,tau2)", "-0.9109377350436966343831811456298324147370172814541850671813146952729322918387354956861281265870916898422264454902801070088831993744902013393055003923834323251532036046497948577037823215255167822997708219951744321213204428883422681661465725088935208321521450310104408002363229206173460802647339336658267281570232318887650926671437225310758540728424771076224750522922976339182362885434411134027211338547589438898712296218547278+0.4076978877859275485318502737873428804663954134216085168754165943868587689636175650450347994218319622945023576246925629383167190822547003897444665056541733966752439982090238162846172103951520365093396699421499693384030146498339114453930192989428709649283193755710247446943042008632750045890564493108308231269848453690792705223602669763085478093430358447278479475037188515190635597039796768576445907124323791378330930591828387*I");
	test(303,"Gt({{4,z6}},zz4,tau3)", "-0.9785382092950784022234313314575674530064718154924144737827426269283960567583978156696268950365328799170660004576023845764898908027238771731679418698374848524227434593649481295620575934255186756922337077710365145229501048285420113656371811754370768097952187730278411117762843874403493873502714679243867684052594308398167143233848459712157171251587903013688209287531470285259229910872660302856558545291914992141578795257567977-0.3808791493348087850988334906094918126359695059093455277627513932559091156566244240734140075820989769960510936190865084905165699767422242288837938744499719633975022373606280900711412041643963310169197044475684253832815198888804612158354761570956748721116786594142072949208758277650513466132835541084239681382330593096860400057206435835609214954045323689466668279833858278978054148599295723418418491691340878072319258973872431*I");
	test(304,"Gt({{4,z7}},zz1,tau1)", "-0.1004313586039567525448377895788631041607729952552936230136133461108053699066983313723858466864521072411025054624647932021906852308404048072107785988926578554879352125814378075366663687383028124235560091117957422053592569298092036212807136284997814865662645225363242731070503333563990307582719010920097050070690184117899589941468454513606136551742437029323874851277250221470621893294869467149203684937938262278679939712583481-0.7701664888824144547756587313728447237789664594364430207409670441960586830912235233811080110859248095496497748101630982660183031445678746412936379590377395762120058784595122948041832555136930864100191324855148552762525885593139053330105211983247498824242201144015016958743485394461189778217481086753942004273464806397032091875488974154069561393504275459893836600625041179795721198767807297103775290009591086552140032234038979*I");
	test(305,"Gt({{4,z7}},zz1,tau2)", "-0.8512614101242336383584946027565718625398123984155629316777054427957399139682704733911800484599450638036624585046136647511918394956370732816842983247728895596595229681771579696697487630582496425896649896228303706719461895773273225556815315074177390689176590797344276768050368451706477133096841017614821134974103080375771548523791212345796028728710845321319420010572129272932936114028219288451353395780435041053128045540532942+0.0237492982816035030895062510197354235109279625866388346841977228533976919458458142535874177041811587063230589488023031795999130111605714604784394782160991853619803007160362636998824668349800940889974013102639355196954684755167488636946296037581389604262145831768857553250631419712351922827833375696574369443793866702590583510699411411363390814017284688857977503538606171515178123687672788466804649309366545670036929328968852*I");
	test(306,"Gt({{4,z7}},zz1,tau3)", "-1.5033900556965854125627335328064591785404149691949612670413142220427429309838435308143299540728142766369558011563270519118333937050290371581632649930780917960545778543688589025811060363397920568804973541896500616083054072618093883530631925313065252643080026500185812481246536524327265395644816219709540048790144423726736164693488028228471861556336596796703617196311485554542757007486188137258472865205504269229049034078115901-1.1200933727357225674018761018343307788444034359268435211861440794319854693289617372979609379487128031130110027666255235738772320410266963799626190577731967652488005538057438687281389029681104491959971686723989103199486091033083566993648471749352942113378082676938685390135663928067587360837991288500606299272214095931426826531664526679026452546856841711797417663960599797561970375869521672204383567156718989947924090193689715*I");
	test(307,"Gt({{4,z7}},zz2,tau1)", "1.3296507083690453248943231618713807690138551966693074934355918282035818132428150268379807325620795207498019267341960422772721059771230436135646671693845938449560568986615500555053933464591423513339360257644739749734008076256998927414153625508350600867420402530618789299731496161144204444898296574809708815589127475674007297220383363107191319216270669434594707141175584723420907330205923159161294885175910583250515767596691076+0.1794520253194755069077720745338686793139115693772197869185721539146966579256550018021298621252903071714972138563650609445727592875970299004327796446364034418031552588460752777333426016132102650582581387335751562584159204399216205669394269757445672278498158254397717035115556076043430781565941994967832923735885217764319201332013344506307906845570902054974603124141297731529632530570895163590805927420151241589166963020734244*I");
	test(308,"Gt({{4,z7}},zz2,tau2)", "1.1364601898770830174250430599008269963951654406065742176288870313077063101768225242172930138030023197433516322692215048418903191202059017363777865048089529453432497783313434201480509265294394037998645741927865797176330118119497944637194313673984438665492926033187056337831510318174793180679750773250356044178133252936115642702504425871969135856946952510978695458852408901193474846715037227607351308493684932705404983254298805+0.7228493310453494636489045248342520731528671550163290431681492087802198087130675820605631338814948922718746692486238819352996120352344982874569382115103143158116888633635543111838329564703756539796732040199096113462280980529110867876425265622150069473221793502414551926387997931085879245749523423755456010371853416740136968452770529743783346285768746476171259748406875736975044669448256605091852788670576627548876123984974655*I");
	test(309,"Gt({{4,z7}},zz2,tau3)", "0.5868751161339692892397843035357885507836573847031029537849956152174831867817511222744936273197376994081203627158142964819554074203755653907236046663636261143128907698374370991905743481927522813152373955385348791646963289699947224394668264409223281462034693281292794693504730678181002217553901609485544038048261646294049558451628770531851190249480129170209870605004939592735878248649967826542337187493905348253559372421721745+0.3051247454400128674566810438049661324143933469345041319890459349511655049711735649371048345126092043212394562442953642758224715865059574997844918083694554108554975675825739876055548486119631237990815797095679741746893214346118551299499209800409609269818113460115675554459259373208289183982839614431989030330139402741633110285829221041923920986249247905277433585458544455742126798148533271481879282122067836785041889285051031*I");
	test(310,"Gt({{4,z7}},zz3,tau1)", "-0.5325526098734396322782585377883420027557384560055157318713679429073390038915036328465270188410103710804197539996194094796754280712376868714972353446993689804851516455226227066690830510351097067393176644488927500648653890090987146319943981617259544856668488629720521595728062493669449772733975875332280496536226692627497735331277775736133679236263211447140887976945121432931220487423994806141499309445184717859265558256731187+1.1775182220021669704122860208299067680007629614068861198741953835350111695149916877626792476602305241856697530457654466116019171393875746234348399642662450414074624819693615191661980269321806027149369197112630917928589643391047917386749270675057053995139519029041255716344748159683979901548069545138575100695392390726377929426021641928304630839232835621284152137732287902819220783765967307256609966470333692439103611425061412*I");
	test(311,"Gt({{4,z7}},zz3,tau2)", "-0.3246579951798685632229919392332835297209271321170401941128818645379797241805376666581449071194324596812315753818040714572771271932732271134811895171829648321489577887619055921571348416185124083271357544025198707609413551365669874523808496381703893735520658849080766521244613694625299300027376666819969454828960405568330843843676537603701871119301441443907962998750528277958057445005446698067049427176724414182374940729607826+0.2523619558053342275671247941482633905090234277538094165224816636970242865630542489967048408758817039048995037909878477939153233561357293011041407242038444741070570654402167318091907996513318026055670161333033842983825133015066942658719066723888678102536100446891761139282247949724271561839884950946079113310096048879964643450552023171645244411403530911474018540198051312024937772676043268999630950671877330491803325077882732*I");
	test(312,"Gt({{4,z7}},zz3,tau3)", "0.6626039702411455260698966150933587567123358524970613339330330633576884201436576905370999702628586181520196888740834952176046804153660745436924802930211871007559070770291690940828802414221704401582229795124340366045572337732700065176449801219543191757763410630700167277847499290372414123988145575982053559445433235192483748674689761425063014634207571754987083797847691871476338561757478519760895669802212079239950062696951917+0.7977295313385133193006317118906101649470982672478441967992495954373862963683097667377401639742766672324345988105447138801709330920066050980835394343078787728830235584269894745248809738482747581610041413283542316522223191105607215653394659943228677903265504061766834777097783215693692337470433961883661386470774481700536583953716771972710634152648618381513367254466736157142873550891000696921205856435322336903667820406354402*I");
	test(313,"Gt({{0,0},{1,z1}},zz1,tau1)", "0.4608210362633173575626018349719282117078253862539967342293114578686959627999905441154294289820885453052271239292543385147756641378541961507134022662868944016365523487912339049180538638735579110098707587591461911304324860155464190329512522125148820851196638482909151859663765334167470416158048567173149496622806359049407494228078216387916041300824854732563283680553055040469724094491610659114632925230952376439473741281602801-0.0415259216611757294287038079786470488391075675220332703490477317356809165132032012955832648729248125186489181998832787506025052238573018854539284124578100968768163394381312217979419346905516992379671589063540943043150842009545256218233723631980937620039316461018751801928429649521036203466010362926336772578767998527102807384935247894446642150949114090208037601429823196103655962109813784851541111918869714244789451838380528*I");
	test(314,"Gt({{0,0},{1,z1}},zz1,tau2)", "0.4859305664009664215194527075524041235013617447693277774512097479578477512011559007274108882490297332070060405506057925694374075416387913146719264038260210966321657808523704876307662073536220991641193812841794237017593304580730266715339411042672376308271189593461566155833113463385419748438524216164790854317374620791050746926313507192333087553283878053397016569406124670905538433981119883794718752757311237871589306425618959-0.0317143712795745712865864292691746378517866800605282318386071838730019169446659510957392586979448283070768001035682102376800767272186066618236486063123051307367728644096810886190135826235610446552502949422191075794615467751381131299428401284800738968889253101922491371540076109348155216647312145148560035716575143892092995025187848098682990019670465381148244203017801303351799953782195484802647289210320269858644719961045993*I");
	test(315,"Gt({{0,0},{1,z1}},zz1,tau3)", "0.4672036562416786308626394471436140095006005900698637323277658539883374997916720332641764303867073608367814486203319571499427685265486564109001486211615571845922283053641103856721857544847228120358549519167371974070785263100467912516998516539751414547166968384595599824473034032314173472460661829160274490798498464837224669463152819549695344649337218803532411207133471824403740168220919451470965868009577190558964140898171404-0.0075105946988267832160784020680448815173040493423128400500560923929416752915640533892946821768609073957119031729623917468910734548536166669013965842801321875071836481699796082246923051286201731362960922278592187474847750071403389169304601347038023477316357598049169328525744066559671139803401627766450167018207098515209137325614740909953209055590299996090445951392318018691522397057421930103458285096197646547490449326607937*I");
	test(316,"Gt({{0,0},{1,z1}},zz2,tau1)", "0.1962519462147707820249087922715164595647843799759709270079726840672201057650048933812927656772046635646301826667130970946157801822115341406423386924946135932138390000299266779628634921211615542717251979553199469941556218283830298488435223462110075537839348869946000141883860346343425653943149064669922271693189032005635147710432127084934664723832400272749201254581838217374963485597604190577262247318333048670901171092620461-0.4136619704220640749252245654165125053589987047569750177890818319193607918458673653803001436621872195792468910790183468827075112722103266011646732485115936724105148348372127435076585222365971284153688069439907203884234203868084940130530112775540471817832344686646489938086617091610517177471202371882140932981882979030300637211124819426718286917429822268561183636827629914487473010664514211295307087660356048825196604501319129*I");
	test(317,"Gt({{0,0},{1,z1}},zz2,tau2)", "0.1961800154873456670697227443622207868090705212824638778192905257713126239869411944472699320473126793278206916342276941666476613617777273020216271541997307305537179445909841940648765685127168194002983656040809290255005229860772912535396197945943492281045030712502641587758497898759751053011627523309411912304046532417649887594264093306887445194512454211142086636104254259795845724105584932911859090048660259701869737960971903-0.4061188221591280287507125653460713610872069975166957401008104644121197557728966134512238826848127267649912255986033031465239080261632495999131868325375124379264271200235399894871774828905112734485128628562813204408879394636174197871851265888703963516261005456958304043049096855072065390605079689720217848188213046857475955965310373494643316984531525157701393622097389123169789754688862207000921715916748155372622861770281633*I");
	test(318,"Gt({{0,0},{1,z1}},zz2,tau3)", "0.1877111854290228874184175272695670151686831303374926323850934196777947169785262891391640959717135286654616030253059472821111473136191417752016031270375910903239472653259096454799865500092293865699124189463853265592020581712475082829781294441263565839930681190492529669641370452843504465271190560635869842181176901749380345428790670921033274507784215738657827271610085228075755227945415994605305642510751219864674705124653678-0.4089288568814658910463087397168273715149997997554574498741879730578146403599119260320341669526989336995676365964650598469184884420615815618759058535908893426108447240587280903585402711278548524292623768619388599452990254200134749236053523556331715886281989466737488871740039857378427994618571821473031621666491526173438217746318072869943723959177119243335889376770406256019772962663756580472484553832306771948957677656455686*I");
	test(319,"Gt({{0,0},{1,z1}},zz3,tau1)", "-0.2490914682849701052554094569383921337865879635843728392413799446566780250405284448184077602894171013128767155704693821348181749627462885070899275743368346003305678096889161603973231852353240286143559310450641880525084663858906296346379558777365827839167423490759045736697626004132593935744429076685734813128311336809337802903518667748960358126545302102973845567003839562399219928350983386102888832687245212759459037908094685+0.4488209233457778989190844440783081001576184104921794700383308950060843936573643466590025879648395763696505694880737487355946436822420800279339903506269242256585391415779260484329070936846705047959008184571569114294245806322156874300732293076376996352784362073959633705737973860864457565274306731928796180412869742582007837017534699972611224074222442148036914606346071051299156982792803112842071651961129286686790218251717315*I");
	test(320,"Gt({{0,0},{1,z1}},zz3,tau2)", "-0.2525016182569632917544554656258089070126987715609943562035979245450367791518513333101394736645226229755406258043352532773675165773773222067269980136499263126545393212776118690376721003575316146500942132742696058840418067458194920488371214276759042327572403175240965947918132698475956122550808640809337080697457995697761377951008898624874322489807212368779728413874009217739479020096621357113806591985358785230209008187919111+0.448728706695753433843611273008530570275096265288901715656247204588211045813635241796592840684570793658933825052301493136951886570502744726596256338464033777340049448599651418173096971071356751997613772376642576266993193006827439177318611161705499091502827708511886865155507724896092459874204538286537454441534595368133512803596662254952863531841428073422300590396252616515625841332842707533105923563763285252422450215741037*I");
	test(321,"Gt({{0,0},{1,z1}},zz3,tau3)", "-0.2511763155722086070260365806232891488464505167985318273093332445776908983647290484258736988235373231937987002556163442684847742037828503317350337381061698511822273752761701780506133965262135940743687851149928982840816635829984532432059915032671063186783210590776769104239475227820268159568677503289662494232381381142269968166118598117131124184959969580576574426551020742958678098170210962815989070205369359248731788946813128+0.4449045481972981054690403325382836226345942620959828243974769521675815642729041100693333157854435259866696995262385215495525592003583076298391843522431943050434570059249946779758015756632954090763663869117121350613079108749638558294789488884983596256996530898413814072045587066835790646112836019071171401764701524009445165463058172021272145942586879210110751575456287055430389855083083370416013523640961701566544184539694041*I");
	test(322,"Gt({{0,0},{1,z1}},zz4,tau1)", "0.0389205452338607473960398194107636554131596772487174063003469318875209705444229461702421895900673215677162000566279789582964847104739248495712761495530328956959444375336245126155750989617730699932789937277099204856549548602837423165713529286060208298772237976317557383109768671335509015880389519678743886322579109336715569401457438953674257660576644830769877089740217999472973159206467689942289224290721191281546616256950166-0.2083602977664218277510580123639613060552361573372262091094872401219350433268224558227572090509743499257824704613185228434456046922814528812011363763565569898748697049862996056648279880315993375595646819716130876087637146823879884541911551667749713495997742645103381520557033817798059990403242575027060097329664430113885222182049573166217244757727754397806783029630275483590503299558939629705030857256519161389392501043881989*I");
	test(323,"Gt({{0,0},{1,z1}},zz4,tau2)", "0.0431517677385136298409238043614780406150838882101023052181276470403748442101577940804110628794264510697653909450495238228128757206606003541409495433108706253048358433371711550011233162299172788944439015531206658951311597678046870403138545894022433088214639634776782362471974261621558982862016551409863724160348944787774173043058810984071563412613979351942876858228134748141906742880493488131704455848310840219035018198442904-0.2115013845988075605870905964343518934618753470236183309177635325997795072849177936341649911367038636554511895469121247279032534641114525164745490760055545028257950316666108061345685696884648815745777680726583019885833958245131301521003152485726472518621295165164738331524588455013072051123084248283546730458176342968163222469711482170623786378257468996805400094040663785028362494449506812721340662179723923678672929216157204*I");
	test(324,"Gt({{0,0},{1,z1}},zz4,tau3)", "0.0452132405542942431240333301168443308233336447646669967592103115806178018879371468372891702589390971734862454845618042756035057074117460912581095674152238669329789825544642306439991838494474578749830158813262324855072979197000176268282377426311191330829388789976387086937258514728113226002535617358675549746582186727484780521706458204662986853342682082137694183205421654750891632429946929429311039355433109220089857291087258-0.2056431917399327217535815035841831759649944240747360348487763342196647662295629974190913486349245472276303547033785341054593812353165809862787752065086050061185746719390831181763951592014223798396143279629759756842981938436572308880175655536454663095624919485447621232867799786586871123926637900095039473502851507046972460130350831045925771581809991538937402434178938649100097655079370386224583931951814566759761657813414102*I");
	test(325,"Gt({{0,0},{1,z2}},zz1,tau1)", "0.9425965028657310573665656703407626243660605757966066463578108374776867000611226060710708916286553824724990604168219074774950064799715105072111852272493792051541233006011170517644367156427867181011083924228049408108910791347639833243580683079177420480650936952839649385412061797191867201291139467784410334828512593367237638425913925931061162102215330748835620364590231542805764415249257027418384989900424062153151017697312459-0.2250755282685619461406685797993502353659851630694263922374407648348527607925231702257766744816230953572535944591850654350525668098150866904828259820740371592155372414129733396683001705816604710062763693009229898744527722293028597328055631774151122990372808022902429146628629879714509222758589612150202214507023650388324075624880564642953441169458342906639197435240127004220243949958845319942803438760397225679278277564760317*I");
	test(326,"Gt({{0,0},{1,z2}},zz1,tau2)", "0.9462671995687690629062016465670648280909322049697884149816851850666038967585559600932368724573185114004885253252708868261300362111050431133281746397339635983766187670389206778691605789243811955194762369720799044685439469822569901839328382175162031587268980695367702198525849366280891620078295025120052222952338416481172006394780093427286805687514469067550120725082777756516384370168704775406424185232329314492284550258293937-0.2216892140336315262820151413597149653785775098603844435705629036071422659792843753517409805980635519733069349726874448887162319803633818806320185152743927663062464403952084569117568724989774925527002806866994394625822949840031842911083204158802006009970647332970694860118385192796851381809051490309486805286956103308174755894186476202886892549010409879569937789305747864384069745112519064093644424457931349817639806830558426*I");
	test(327,"Gt({{0,0},{1,z2}},zz1,tau3)", "0.9412131663054041786921266734577317078767158120582597141847601691264587230756327462106755060111568752334823877254327786058051967480007439573407360584855635971630331413576820760925227682489251469843602222690333526031926923208452486110644428340543002773127726982583323317769655572235185358554919153314406382248146088390424985440582960176397787651705479479119634777707451611205621826079136437026383603289503239171494194819309206-0.2187844295092983738418774001743415815248378902350933070383798739899458729587871374251516732535788288981732783114100486491098460019277160047468907367775941213721379132928622361679908581795799021932092447270871100617500766896964939966222174230210504681840664387985569739694597928264806292482520767108219490651446027303205681637496250153341887314431571572340235935030272215285231398197156034216035405483410546652883065922806567*I");
	test(328,"Gt({{0,0},{1,z2}},zz2,tau1)", "-0.4040080696807030133910675912364605983360115116652638710337752470493712606774955692689182635047325588431379496066400611562290814874233885307786155448828925983110985407233128991889904675065932018266182197864721383401132149287123716395337175183353425302057316458606284598300185480502081529035075710680812531466183732926264119106511763053613854917975158837305212947010465871924639209425028777826037578200920359275376726872761762-0.4417140981423739949818320153932175937848717165171705319923967262233921958283981304821390306281061676291643513430455339738980059888087680284229594711059871371982495807871980038157388354113586267958314887102770134625208629776024246308363199005562102838563686105218621142386139769537326318161885205921994227511822551610176501655049907171969564335569923569271737361941356762735621644737697177050907332448150590659439644126603166*I");
	test(329,"Gt({{0,0},{1,z2}},zz2,tau2)", "-0.4062387106586456596259249877758678346208864208889000862263034594481708014403391497512606226084300060278149263914369903239617504162668592578287898193422097758161672081623470776974801198147428129105447948166174013026611251985436917178704296028644299779507225943550609430074399789556700389149833404269455486928453674290089383252185465842049788943520310774163363721968183999275638439175854334173478842180625331865398111833672795-0.4393498094125808022408855755841254273670965227344800889174113800101763383884049138990418571760878968422212572210807093886359075547148505652695977116509569457455251878671821957209684857989611156613557702455912959329030616995201589053225396204054120525135994285360743437994605121789230954996118274806483874801029290423463380458626095509834744981864161222977043949714523479086950969388319156829566685044581549295163899757941699*I");
	test(330,"Gt({{0,0},{1,z2}},zz2,tau3)", "-0.4080690965950649722278342083284510507596798884244608358264024177532811197192360704895755755101690731852560081977324625692703071405911500734628525953784963605625147002613193673056601585664039438957851426404230051536911104515337351776964589555167159634856540184959683904219645297168333741808129421757720874303585850499471695063852343268171489000856009083906986625325584641633192942732794161948590539134016110471230790839662763-0.4426844480298174856045823283552436738977746780040351512463206269817515358019181283503036614605231806878305347974391276492573250161677939035811075289218189012997078819492997731865135415955546791074541513784744713137574258440364978838395068905723290120878365395449070051230927144195647079495754126773646199060657665443840444064551876456317292406400267331890252575597821824562878412815331726315884881144865619011482699347251514*I");
	test(331,"Gt({{0,0},{1,z2}},zz3,tau1)", "0.5418080372630146271414812262951990307709255099413235820041472418461938970622921015597099934535483502026589875667364974320883284467414456101379919081639056309659603567022076992219644695529918998440503368623078194233520632112164196051401753477176030106401275810066319006599133413662580564739871532954022186728357234525315237659048204814826026674222165405601230221089610870737093071994420027525804204800049424982696961646902909-0.2775987741065266129050175329765966012434150007122729891531194196555811485147302011682494714190040416932706517024534378367422303353513414672918499237945242698069804377040570877872466244815977868069081792364748771468003367858611569370063325302089356807036522627128346056910352331224022285045752764338851391477451625470597438716294295557112292771728270714754980352691133126707935022522437259417831472511728395147886456795568839*I");
	test(332,"Gt({{0,0},{1,z2}},zz3,tau2)", "0.5435005888986276100844040447108877489257189582517780655368562917120336848579169759876379176503776635410368546677608893124955047470226047546103221433459280449748985039372822703893475714599501389934090288596034195577881813348799969808015554650635152349672488110925567916631052428055399550158882452662431097765495571224910073177399551012738585294856435348767258069187570841743002773871794174789289561152929098336526047269928754-0.2784732948233769031762362856361467302737606201230476551494891649106261235400309345937660494185062374388071234889348507500748791334724055763048912067327297773573070393209780310496351125923874978045689147223916193544664004566658483796854203400628170592246659356588561945633908809080302913874053031352628805585186130508581951214539486800180446121057865949437978364503153220168831066731171487367625113656183469466477175103994356*I");
	test(333,"Gt({{0,0},{1,z2}},zz3,tau3)", "0.5438814139296450700198443397269323548237132081698201573489114550730175752190585676493420241123508012192653631181888308567035076968463642820033771651164016410363732304303558338802416364218130859254594266515523693170076664063990880130217933665791873243031434441617043889446845320945141173038137553446894561945626794218528340396913365029899006621748583959642681306391376183056217161947919920134665444880373037138058616129628002-0.2762530771576631012474932467215200946337033314821766487168147046008950663843435481956998921283089751014022731310365958026707111410016803652091466692509012727124391545510795821836028751525671498501088836851935169058263807338768220509229521414476646640274806770105864369800005847897229034664445021305230622824696779587866145589991499843792037241552228256609827063743183303241483719005061055367074565051615134958354941156814678*I");
	test(334,"Gt({{0,0},{1,z2}},zz4,tau1)", "0.1123987294927283575742317530847412330880048339302040279005415827476510283887631802201280135730531024123418012209700894275798862755266730866472493256458002478485461380444105416368331605890717840794977670791353918297379180259043240884132026053725045809146250059451220110245368192946039553493077931363785677319748551827122713731979908838484360285113623051766130182494257371299631683669647620023424169463127323455028052595340498-0.615232189960137735394594246424706403472996656864816863381036815871977160694524083275092342350785538705203525484618394622612836577426620566094588625132320072529291261974426738241879783492732395835642711930918332549047674230796171399221161076147129782267393026666244162441266212125679937070573616855943586291736943887119705905396989773381838268688695926135224573970710985872967654201278708961964777862722498858421899021808202*I");
	test(335,"Gt({{0,0},{1,z2}},zz4,tau2)", "0.1129038460591906812136125587606102023711832596169077071319794514268077035473059991699880134379731869490881570595522105914549714378744612683879900631084317350004993881905118595077196068959680878534124268663344412890166028926711886103279336006593055603765666598268227985248185811002258215010421366197417550981351717571353581281691252043370413134413390284095622439597258186948246369134906732494006332463457088517288308147626799-0.6156546562005317684058456618035683781324057841832265983767363444777146486137541558207306256364152327349004152701803800907937119782045044221816451094494738513219170021822585819760233771155119153299174354157806140824677964658995290788029756648896882145018144302108207011348196376043869848823263754832147168091309392745654483522959080644651160023418424580666728873627329393522099057108753566907041511925068184554939483914207956*I");
	test(336,"Gt({{0,0},{1,z2}},zz4,tau3)", "0.1131940611947422178002747314455488458747905295186849779084651170372546966640359109179063740708319122207049363693114018446486785290403437827469863988483468893616908544454092270297214893965920422456945900097910917863864865314779163274508491362762254680443950309922589234140184353386423832878253304299640308807581878518023790174659483861416248286662686577578685405824515383536172031531970093347326440707397484926164452405382499-0.6149379195833190107531635032719424437466953903185819966742269938484816743351911163255100092947152219454685424590108718315481421412108631692458326755121783222818041484468512910882545467256880266199562778564221651421125068244298425070320112847812990890985381551427308914297839852576118693416323452026139064491987590745307568863126967011262275424400112917238259842634311462171130289156536441146700681215312145213860480629207296*I");
	test(337,"Gt({{0,0},{1,z3}},zz1,tau1)", "0.4353637392568520140604242091937083796993330337817173362952617468471286444536224800105438630717792908659783045370727603930036322683364508389596410606458793460716062894106892879600871715476124745565745898516988665305382024327321070937460612882543857535262001519833529978049005382703965363093012687696053967858334491126231590514855917786251801653227024325465158313199183502235662084834868090869604042412221374726054146825174986-0.1177008007898321258997755780870554373004070682167268253713999315978155124353905122050067474746661039068822071566055687971482033094136426314449784990858794428040954848752847774677282722754799230765151337055626264647576569676668841220480529361771323717393047716792114988311943241825389269746723207304463047756423897340437107493264987734110065646945101854494719030038181124957031324128914730246335023570207404226925959359850996*I");
	test(338,"Gt({{0,0},{1,z3}},zz1,tau2)", "0.4808370523935701412896391386906921864007622940733152408255921690641290382693177492711146921473888770524964650966568957675479363281198521673109646154197607012580010057349923705081847596747668150406206939081935040750017429519319839072474884530367598754425847272401638485904689394083958886623109224852359982266323628295108814196047310750005557671998411403529556610272820768129486701752286050053496120422319075992994866774363972-0.0565774653231365949869329936975787011219661900951224257990654750902760455099397100172906637056865224300247262589317365478138964650873830807480224117666475192992652245884954234349700745117252706285331183872666069083544648476654659217275001246227785286612139073792703728661229953624672692977294276652046633465621672732651824356801065756370498265303195459783675466363088405925639130894983213911722256580761934294244740323000068*I");
	test(339,"Gt({{0,0},{1,z3}},zz1,tau3)", "0.4031405095863051972633217177994421475872812346634149349886558360243009850759206075229610843195161173321893477519277776965838194511792079629855276977145745024257280506248512377764602985703862432098564311328242766909430313571884162245855182432844519123941331758536500142476634694886721238237383818208041178563627044806760686974891792219095278573896183836763981896950031483866299445570686512437901247530380747167071836515372679-0.0200271602965183379711016696946383045373859561992292481087026914526683701776424355074686037235285180694476056037400199463379118896428013918289009440595663429187330216272000886083902690785753309857541725995060405851718790567499343822333970917054043914869441781098608024663196400417853966499157073846864745117987351848732937687998547352721691092670451982400679114325830430913679513019992565605801194697627477330401230159980127*I");
	test(340,"Gt({{0,0},{1,z3}},zz2,tau1)", "0.2193237832858858629115499301279752899195924391007955080368465596989362947993370485054005975187636598924957207907570882006386967517655387321452158168567428524291271607090480384923174736946142960173474769137124154471760369729767762520398192481861331742427157949461477054365968447635766266485971335365141728459641039233783015669268912413723960660723497253376578935644864796624369826452807680867833611017647816958853639952624278-0.318344604025180916801836861538031829959408423659950364667222139345549178744227791086250559323952295680631805123953070511870305732027305711943389927630330289766214639990791043074412000389116883455012337077022172031151570076573688363273888545466404993517574368057455025351581870111955814704857342855709794259017687897203545809109982466031868100812101554761121370771225246318132568143460640630769621211843350659014023745102641*I");
	test(341,"Gt({{0,0},{1,z3}},zz2,tau2)", "0.2099234345387511815453299353300796420091360150420330678043561778064404820314239269579967430025026664978916390400221422138763555624370616934055049420012209155569719299926174012528780623476611197597872168430641720074778455758924293240069181246764214873911518269603314611644218020537693366940388235980094995490169257804234584865140460947883080841997390354193375389201802418354515468771194316303129293321507390803395469576195339-0.2986445559476723523312933715732271107232634911892399553864393906673110486316192630966666413572233485622961705306436123063974258009902552161996421855916427405570850239141347450721049769561057318503720983643103685571047610588302718398891447011470647640345964946735088509991658309161886349836834737029671546287849502714453956860129876648095074464196007263656467536133548528275284105552380065008500113922258985774732593404861205*I");
	test(342,"Gt({{0,0},{1,z3}},zz2,tau3)", "0.1910092515909557284282645043593910744907806852655749063940327119081121162337093716886599400157392083961537387474868093170876064903057123527131949917917803355165541205396592359000328530962404874769568279794330730427884444666928745838632597777548096326535582311848498807421791937195796108810778234939287473242006218559808156659856611889314936368956259578389163229078569967685572719343730144078282561422152014359819059692800007-0.3169863008294651188862532593950393236276050847808943906074878672043571911774093827030189882019313555358064507975434983534685106241455494918747394578414714775304074180352562133519952432029143717910112022867226040944482206770118087533714430983042443075652861829363829556065113273347158727446504236085113034896091565124277397109768728952482975369934064240606784179885457118964037596397707269288999821882605471584264264228221362*I");
	test(343,"Gt({{0,0},{1,z3}},zz3,tau1)", "0.0361081003255715334180320119466535151384139726553593618439758346091442665094769542087094808643573341852395939844841000086941063851144770069115557563626721082304481168355967909070811815189335351562601022979498626323725438137848502147670319078037367694929710891828136727892886384213962955631181778847143939127836861142453845084108493617970641483887144659225611260237915616136885287200053225270842762063002158348039892121788293+0.2935805786579375871647172045984848060899943699806953552940374008766970440969290993321718677481636264424439105611834322019984801080143044182382867498015999157929134612954728542581484696134320876413465308808112246643017133210402719760652943783340014895167694352736775787371335172318988392545413497206816834175427019689002261646192560409538843818663512430999183710949106801064454791698059068237487122086388678941491654250705027*I");
	test(344,"Gt({{0,0},{1,z3}},zz3,tau2)", "0.0267803798609900900401319895080236369720003975211542438989779057427256408179574778984275228748909333296480134206223852726225054767668969610840143393726753937698797911451844546305252023897045032035971654418874324350969814052946832225660469744198942740542079707177429530071837042001763956111175268680303563753020183714291586361517848040804209485051376382552366329166895143587599077927260157823651775983523574678503964039918704+0.2877387336317581240319243031072224860047395307342591223737496507288421080001422720670151810443440616173731871808487882910250911858342552347780737135431213409304349481961020964553388427143011709468854540778135775211759844906206067707885897358105471137978615280808844719440469249276477374134964549608796475174965802659571965365257434709380876994899060131106273978029468371733111390626741420996176729154486709752343917040580244*I");
	test(345,"Gt({{0,0},{1,z3}},zz3,tau3)", "0.0370721967799024088689818646827807596528880170847267053906035449795255212466116792864018785083478735076221800172088454468939917956448374402810541485372628645863109833699649697262729947810089811578507132905004352027821516057770411380701688805919461205730512851300522862757350906954614015077001007929241686050674748407553809763663946765394950569109908764830063106386482452027668937042835590305835914069847323235600802198664084+0.2791309880844000915536791698637031622193737848016434346980412987981159251347152825298688419983509010393111157778770830813453573766274628897043197433785208527307581051539436453398714994724569224123931013084951639820918175518829440679560011407697483743547674215859557400610440270814346323026392350274795632063985497408615911707157827379704073539478870784171474165747563558171656068461824924639578748043878881390986419153395267*I");
	test(346,"Gt({{0,0},{1,z3}},zz4,tau1)", "-0.002362288026730490469589918965448926603568127656495017971099617323453921004201379079945859125806104738909716423701254117613447858054119854194194893141034692692531687762714554114776759356948949787565077988602115607835481609830578477576377504165756537197560126609833915071240204248287514688738402269167261492841436911661611078558429443708229461098942080263388109942610448466904369723300900132925143930939497457363645147220444-0.1972688016301039376675833760362750078407291984335563628327080779836474369290514513582295652967139183928822880573229839520331098302881619088334966681279834724958269565479095520349663738639584674909415824819386474835682839167478035389695943423394988306041627955362377603139248248952366618561228705833364649517685076871457250206255929295465637524069162791484778333337616234561935629448158127036735752522035001426125833445252412*I");
	test(347,"Gt({{0,0},{1,z3}},zz4,tau2)", "0.0124458691553406469740236661197580141776260800623257648489647834511036573369161568384085632622257738348564579050564852164801687959671488090146245989181610912412483688460806261844139381263055578963624043692449844812786019337641340725453667563299306013534429416562041454884796845587353396511711199768451750512080894365042070448271209800468057510657157122484814388022773280147376757475111316821512688456817299123202790998386854-0.1990838025835667486322914201788080445191129193963061745474448437901853119207836453297584892971672573118028245903341512863538478372924944629515134699541653323157614407026972237017311406657326444026472564990856407188942330076907140446830703989893479406079776108672319462792229700207778003188206788398215708048904693812262264889692471230844286693866503286045975261347750015954697144570129788017035789453918078352254843631591876*I");
	test(348,"Gt({{0,0},{1,z3}},zz4,tau3)", "0.0094122798577633575306745123882042716721397071389576027432777201894921074223351223060758985710213834050833201395430078699990124993791040403065394878495216427030737030294436594677922313006081407350648497559106522501447378352428599395031659992146292279482520237102202140110217296693662508762846735749330312478760006896643428166340232101552762833319188861304685268047192445995832533891093696713317806142288092617013045870387556-0.1812246418469619162852385199196330908759299655977081946490612871427670602984128582957921282672700021550377882495296308754966210363102021121260591617207321296342264296609617605853738574376889740558032848896032727140727475431379549267187232917164408244099473902126057618573727859585660766348597053396451175493647235348382991580291725243503057305388061693405173682291252897890068617002164952787818756302507539522213925867302602*I");
	test(349,"Gt({{0,0},{1,z4}},zz1,tau1)", "-0.5968023755665244322189879095746376787106643635075355985076628407461442491821805030733886847357369985313612794775207318845549743278607448874602168280571986431931925598931955791321521014179766834355561583969932312799450037676504041214045402423445527565412017872483632433781430893579065197130338061386470595232115813091377943588216358083350171291588356932611372269305215051075807562995610494854230914419372269338286856836427417+0.1533326798600413946303840560346817265548919968004724891617314175303183104536315865874152596688481522486760503606898432280360186892757154903090651766288141797068480238218509715689653266471328195330824082310005267640389728482794514477683885876332964842894662083098492991304438341290560467414006801844023682368944816396185721474082845424065718365914335708643527837096859515534841464977100089533905419948734864303814458663115038*I");
	test(350,"Gt({{0,0},{1,z4}},zz1,tau2)", "-0.6006105100652938811701668357991067021439102286622110119729137107938472614532667985045297291492422775265824433898383211673956318669402668449059192735074298898765034472149732076962842860172041266033197716610432239327934005084306952744405510194838735682189223684930037594142990852642882732320088181286438321952179545826400866153030019267701955754496582245755457753928701255015019585033686012260368171628485479585627939692670543+0.1455739609174352496914408687238198680263355670995290006963130197469880346931841394282546786989122490726326589604469773601033879590502706115611034170147649127343003317396569542793770263708371152555688071817725377587006498434367333099486593460497587854034614992267218255779965696347293392372454753492501689166592911961872312233952260797083068183369283620958931992624510752341485870700692073132754564688148640743453954486111066*I");
	test(351,"Gt({{0,0},{1,z4}},zz1,tau3)", "-0.5905622412237103568693224183277268596516191382219856493529136100340613287542539476014771725595025090404965444089930630845468558242679057653902079456222903795805256330585484225986142043310373626234171653582738167286746986653797234840450783384502767742318470690935019670834502988767569733834223179921294421737383556524675669608846203025500715354619469161832804763937615574256627789874080911532614430207083611668974964570699795+0.143961673090845919495522663236375870389981408977022750576566226849087113937845748317785241879675040797348725151156123272846464252220835000157042815471727738255381031539878287971088813977361018424056631216051706295358819284987775517753212105489557438093888349921780637485286299327946665987444843164012061195985272049186347741317138631262574553707334842870768388892850904885733636487648820549527395659055237164673197527140074*I");
	test(352,"Gt({{0,0},{1,z4}},zz2,tau1)", "-0.2561488594253259617056665509131483030864484708072523613773568232893057627813817394139236901933509024304999585233004965409915519678388619259476468962694847354810836058745607759258356835289707127609993797181259150380432597593800327568087974702073615757884038007023111146158125129065575704378925632519499812866704068889059580469060265715061013498022044002460075577049672528383596087432184822678995520683289354836125755717571213+0.2282560493742400481663450578621041317038659431184319116514828954745139571532106553845190972855174509295174027344875776951007325150377130505659225722536909295437332316848322397227634827265333575396273933283910604042399209663822351010273703710202344104764639776906883394550067608399555562019737627914321135242405447661836628333726322152877047467890588531359518538135162577742017606969485372004676530317629620485871125889236034*I");
	test(353,"Gt({{0,0},{1,z4}},zz2,tau2)", "-0.2624409820335071112088366973589673677193985631736279351961294990729592018536901847838199558258695341053983225123225264445195929268613298226876473347887169709125533207683065750540382120849623158659793317794795470124619921359196894769627320240777641638021972505258720985967213206903533150769414642418622919089540953534381780352730551296006336941940259723626714929192763794941163447655891513043842761849639519812121309045098344+0.2460399895204251159350071508835835672304774211128020010383193144815633366661658581019265848470351113464908273643689289245851526537062018625612997939684693173349773208945544310533851435568422971535120586416738072341165429249785649772578228054028039327926853829461952855624943095160971585854978640063260522841642474110834734303165270953163934949313018583934056296782466591489835216694678268999256708378860846374398816530256473*I");
	test(354,"Gt({{0,0},{1,z4}},zz2,tau3)", "-0.2797556408680819142378611274111346409127633415300165070028922027852761545323714221929353365059097475154152707822143369010608602399917843562757535186660017949990858533622036962104437007205340119308109661685807052920693348934543677843184681748905266287760377279763071124451735336703981416993639133714059706273495976415086528553680142920830046973181690079819679170435376926469762016436676363468156222357730188311641088043521322+0.2328911260994295134239193585639985958837906780515353920464937618485919348321054144810934963463415849731240964881230117883284518443133549522558962526653388498622040855853339727231390779849793807288936669740183787897250469151327550293814610739672216637896525930133322043979732972813621213980101858826031417789518377004608712839403695198353818882822321951394000448677970063394033153430894407065008270970133881535390290661737037*I");
	test(355,"Gt({{0,0},{1,z4}},zz3,tau1)", "0.0092755473257357450117395721778333980686011388460488875844110514790460255599535504508942750805060601981327422737172662357924805123653892293272724382831318712527297610240883919184805388389851145793984696347172088850946464348727133778433715778748807246373356021484415158546765306411556808896659786523619988263867798707385733916188678613763983609453647373293332798458442286659925526855855705190918934184053449492674161444871669-0.2293553368067662753304637261490568231846975826562776900869605976859919037148342822650794725993647063057613715362768529922901352339990747428313929976829159387018330290136145251986836547530199913468010628843378500732510184288786661994218570770714118572048307403779745173263178058852001598436429334104827509816103024887008794782870943098014982802502043764540420825024529314267272838769261245394591940057431670951626305529693461*I");
	test(356,"Gt({{0,0},{1,z4}},zz3,tau2)", "0.0261828286976968968953074094422277495872037360596560263797460291794954111725401738124516380233241970831904571869370570200199297239225520019105396479068049048966648056388268487280542903855655706261745887280474907703696243095832609936916310572823849081745790045440486521803704824693291279099918549385299468308578441728032788904552801895020823311632993003233257671544205181789564679122159490362275124297176088437831793908984344-0.2463196082604866336942946796889173235969177847822682573207350857417005514480855483557536662430485796033777524970777928847869590272054316330945077108240303196203258472014451079380381205780595826321282752905614140094193336951253239099121270828059884579258666849438334332564421957778400982419473734830389574346929199097492923304130435555093956531942676641524358112657736260795502342350593410381272099575068276943008185285973762*I");
	test(357,"Gt({{0,0},{1,z4}},zz3,tau3)", "0.0418061902969535788881043033568665440348185986103881881434375788661940107063629810245732171547267585337364861940680670191334156628794316765750386077436656719199754496821614896899153016481591934891403420742634702670142724713710582380957886794399127186092869798136974723601649250826687759551831287740990865847960817912255548151055947173055172551068295318110302479763648267030785541111469280277809145288420559934827717083492219-0.2223006430167888201157438245515153674192415271058858837221950102512717991324879244062526849706944843006799363531166844315966168145635137764387935501207298705225859405143451399958463474306890408342320241777764512354859144247458341615386260221493460049123429633555252666791565762367451037079044245108236443097548190545179753764945560802477443557713896243878043082656756956494267361729741463938659422713691572894763238280125801*I");
	test(358,"Gt({{0,0},{1,z4}},zz4,tau1)", "-0.0121024665579558495208700659851143977606506770592575811343184773364731237079488330092233496480632413646822463800265943475957704723872803011743798503489108450342426713946817632164171021178134745430291723623060012712972567885177656268886355468836351151593665418156057967970078622949825309203568811485880884363084500779658584616400099956956681298380598619098399543918805316140121199497438901912402789896245899661325940168959756+0.214777112635236219080217458522040682049407531098924366477834970052598607251781410622881247107001435741052056673601180600901348435817534332897024639203555386224596560486671502627113774989404275437771102082418226454013681828372354574724523263064809720440328976088044241396948118842618232836456869511410414757674263586244450015591487792304608044418846907187751448561226937857379771219211192001702549135883344755107216589098712*I");
	test(359,"Gt({{0,0},{1,z4}},zz4,tau2)", "-0.0187832393118566682982430641392073302063467745985178062659281162014652659637510913632172766011062815265382265999822939396068175522635465734529870939628763778987762096656152757719307397300467162011053495012518408717300731129216159646373279602910361377060105674321188609608211482408493957290549245243389636424667188571423835264237052233354867274906044119396858560015378397667162048808157010335850643618892462938386344863969845+0.2158487153566994311449285222247705901770763536072572094580746072793000776844846456905194024400253666549742975941960260813310647732644310350141288523098568275146312736562346947125589237308923602912142762648231656313928137468891202916033288651371612938758788086763002499543654481604862186661537164147698538614364987401135658660742398503877732721187763245886879122234629964707777021164923173130050086581296705560014619429266864*I");
	test(360,"Gt({{0,0},{1,z4}},zz4,tau3)", "-0.0176594865855604769394481138140605414872595826873329870072739124724864361016580062148908212640980267596023673098083004534376868002607106648895422080907242094656765927351664336776705192954966049212405398917432190866083437332132098248663739358495594933275593629753078149793692770446238404388611274827242349878915827275720896757361580692998925029117060366766636693182767991467402357788836260242643550624226053048717702148908336+0.2078998431176508105659331156161314919045042307990989876436295850739121031844929261219376500902383538477146481231246437623230126435517383426652882379050212338677480799228252230204149952000601481166327547718796885187547477950884567771895247307825136751345548777637777451068084241814566763757123086130689832791215632727717891049605161274424478396366048326508319114269675036538006059764344770057530909144263512451039321579887594*I");
	test(361,"Gt({{0,0},{1,z5}},zz1,tau1)", "-0.1673708199036879295113971461806855832431647990435292106148811803269000867756175354283854586242263744262655814181475264300263313965027443197408003264274371724683037559009804019017585030301939427034655743480955727145284432959866240729375025046907586475374496462815118828191181754091071399319907757156695735158066251755019754924115188188117660611713172797774984524596928910386204639745441904561198779425174901805245277665364926+0.0440517830256238629384943106916269977728931347816179304875286996929276621255442292198749587389136577695995797961840472495164531261758661381219315098092209579541400185328947407213341686390562190972376454184603866154850798949380721772004819987726689612613319214599339422743459406642313995574792321490464269197762776617720377984556216640236921990434300836745947582910473660559834671726791196656668917746215619095473184900727432*I");
	test(362,"Gt({{0,0},{1,z5}},zz1,tau2)", "-0.1650441758740981434755014192463423129610681385488125829233435575053678284865076751434250732940585235458309755267859387282692656221074182260103675333805267210043979918007743698802537751777551031739523794054683907677408311853137075676272176512351351731935439471809004819543862062296951812809557927998774606770134744812286275912110882969720178175149982381525321038656753581996336149056520771497036642827160830838599005239150517+0.0481140470912366311156875928540393216491117875081267008196013810720738633992973205523919833417197040053955390170691921631110526086754575651488678051070278696748722677990302734567158053031398488228207577145805825766685017415893003405398069236526157333031848260617625972407977933087762798164061284082669097802017232552285450552315525696856771210447466386044174904168547060827874209280625454998017342482497534204983032012673125*I");
	test(363,"Gt({{0,0},{1,z5}},zz1,tau3)", "-0.1703858590087815655728694073159030472088864573647830407154821566402765586052880023331469471325782007066368768868664343173785938532250472032405431438714883908417768550547797798405227565896815126583130289990651903196399822274046765624104514373367526116931838790922679336074776512073525302336346235018246188932471370501026573454131042741476447768760108383747716424345764180255681861365095773168981887341087543187905789527363605+0.0491804083272073321214271796621763004731317528436469705988735560613167920044378637485792164599525657954334332317678989654554037796316895315980920826733318800125469530923686388616091295104866305256664369579053786356298981087738060767801742546664022495418521598480661499312987826663294652288619463688269738598326531148997625953324938254843428427804366961380097938827857672673063645696510142942676939253953483786688120821403787*I");
	test(364,"Gt({{0,0},{1,z5}},zz2,tau1)", "-0.1852419447675687020202231504145257040784315877929791168912117936971006576795632491365203858887503577584789199076872617414311289403251072158827276327295983044075608695152452054700169362789119626674246101191053380769652274032018806630595776498057787465339358004589247028549609861252696160977156927131813846122869597675118636658016634819892215656553469849355293098744905616599109181881526303187732943720533922909375009292659511+0.3509477233051966836325167968313436141476266129923970621058293453852327234753602991689101275567638255830110745343205465159770694561909390146217645506432668436533528007299816409626975380414617671352102510798244849574207363573507301588489402343485413884447779488860409657268882875907479079413069631715395729066915813175020997013158326650430139100192925116291171991477746221523089302815876218855632700565399790125353490826177479*I");
	test(365,"Gt({{0,0},{1,z5}},zz2,tau2)", "-0.1815294897261514357044311735040846416986886744900685721047897527948167208014162625580716783853169869630648260903463099939882897666804761429774314627878572531757685000616832675174445507269069193155098479047449450649860709250099700872979369964014252776780467480561507828219810677603159792667358195904472927634453541192269575138737353512210035202226090748539102439524021713670887564126262177732766545163750195619247986545376379+0.3404281968943977263279568846620204752061161461991549027110083597533156757339849937379192726142331546758771833325515122286171671021056076596859145550352789151708944921165581271883373065921894803925180866517612676488710114845305690678116504084639072244972204041017271733561713364113016596059972055333143478444769639294473204976896613538393503761230962346984679885531483177056846515983274295066512699170129067273306836073827191*I");
	test(366,"Gt({{0,0},{1,z5}},zz2,tau3)", "-0.1710933340116483346184255199918074565654590906638664489886999189098786213585009085407807963353230688326831981558813834135642216180839752872142728335745770024958507604508986775353096576763941705594291328011166110816154370482525069621688471140388991129952074193430387020243961730563163221733083793806015833065871350550301457663642305916963069387780987697985570323916691008411565525830225038497043366413054984712975319529909172+0.3484360653240202687654883569587328191963021312803326984212501384638102553345522542369759400254827448417787667961867992984177428705711521235657416470725588835287830913568922610894147343450279655028350355500485991213657678621029923468412030574743148873761802548641394750303321178532268711966620144771933508804755908127138961082593833008271640854770771490125323922789332435678149630826395951383826607254005801614522676019955736*I");
	test(367,"Gt({{0,0},{1,z5}},zz3,tau1)", "-0.0258002457164687648032714831327558065986713218683599847915186467022843805507244047626769541210279310323904493716810379835811397633422744274031576525449313745133383948855471458950843908590173941499109982350706713876471042908713011565668857202564079882337212995229106883716568482121720985085868927065038804123983703243987549303004543383642450245483010098158851184967210185435616585256492203594659592728098772471384319544490358-0.1928924220059984419264281635240646624651300230143767156434104354269776432772745331674778825367675477776750100317116727518638825708160508855484157770936148979922397402832644133167560547908275570507466289480027346462260672600734743543535295909545778940846584203756321640921017651397033234661518618091932990326701687541831475434764543635356436668891988998611679413773575698719946260226571951211622284205266245235074592578925439*I");
	test(368,"Gt({{0,0},{1,z5}},zz3,tau2)", "-0.035633439883904136068206386053138196091377530973223130908420350314784256799455176040552330933378709177955907159235273491322378394743624558489127765463081824421654223849637931004581139729014318319485925747151491100213480195842850768470665306830465256657917496134561347195879721737418907630256987965325631681341398474847509616037700110119308220799668506293413444902689572190833812902200982416156818936738513008761339946261532-0.1810443168364355172867643590753294326185396238311054159080513582233168191401570210595521938335217143308557900673736322555049046894326293173350659932699224517773149646133062131636242033897764822445454185397938759282671006475483759395627026123454702283119885266864897525795211709376008486653136162636710019744362504294048325286447977230541465654943863518615429125305009266679423121208791405074093618199598377734815506338597412*I");
	test(369,"Gt({{0,0},{1,z5}},zz3,tau3)", "-0.0442323015291142019225370471617178435049586973921870533857364418029951055781803595557521861824064770703667748996415385139714827710326046396656591147344067609219383487285582416404044774887411378974713694531230102813263561232804382570509976632350236634114782048735573952833663576931531580837291362665932805243729045034952845338093730109512941232045412381302944667707071961667280545178158328163771485499959092672555281799356444-0.1965948051204169471662786942352664579512869189479019453192971926595471278573000009897902830011844536270852317895822694226755386033797659738022441380516288907388000856646750531202329193821104764354034318037988332164999215255830206749283167725029178268589586892252765650459850842243537814281910256464528120370576035477980182285457193953334745849866485993232839282995217689817086456363507117245113243663820306426611699294051179*I");
	test(370,"Gt({{0,0},{1,z5}},zz4,tau1)", "0.014723878592669267742838433440129977499142990192057348036567756540829530353419917773754195153494111343695972149559907955508544730077375100931283114712700595627490503121928217473686175063166819597154759262190897885877394238085321791728557866359598193421673313396778142842371013510573539202524667990310933659698759750235679335578409320649390207641141363345067319339139301053402070824810605821346344666998243289065581417188173+0.1192486218318732417654531580346033406339297265575781652150853433213696510914109639998414051873118000815045042742644492526782367471749169648528704075676535002147036958514185709359661515760353717071201474823293447610784525751402194024401573591759492474206996689046681688720502913332562938858527147542282730333695687816176165626267096425416232401931268619473006536690477277909288637734248838821924557649646484988733859327501842*I");
	test(371,"Gt({{0,0},{1,z5}},zz4,tau2)", "0.0186678966635623133167718605753608166222971920659971853685027565103713902390536062305456713961313740962343232068897614574028277543392322760394206612408733493467350312700606773565624684512329483719537837943927450719016939913905331796658916638196986364476937386465384898776933071654044020818599677860137676045518737906687977124019067370075662142306800890719037035062837334450965318426078088144013557424075529504523255208960026+0.1185170065307414188849632694409822638096420302803355619121486203588045349773878214580522389805192817306621843609694240905547780860184287050051972353478485589699230371969558935970482340693115929623652858536826714373690271429367578061404769441418529837093038312313981467151542595842489010847330835543865610918044585252971608643639902642441886572282514567581059824360834933671092801115286103560876473797667055387577206101410967*I");
	test(372,"Gt({{0,0},{1,z5}},zz4,tau3)", "0.0180296734527843104790465994571481887787304069640609954697219650843736478624043831446987545564308163893412735524664305845917084514727015650130852901658018197323879051434613583812378445071862030884466009800482520016444167033996391610162018880498137997099455414056827288952868054770411495731945330066627412956924048087843300096108612960149304821596763974128645147447508107349465669821209545190295573821214805528469112752404604+0.123121929152884436900829984500249039772640266734479474722904337461389795889370302425580118793361626519981426707848689236808196144915436743382367209811355672015120207643988320696269747698239185439969181594575199263478000195952056947508290458618363221008498352590415541123328613206945460844411858861270176531029753578948410554610330797449630808432581952453991633088152934765200710134151491584178363739182901184224209102025041*I");
	test(373,"Gt({{0,0},{1,z6}},zz1,tau1)", "-0.4105146668831925734682787216677806537950828760318770973269175302287923385191311065599330120255802307120902899974414387240637384906938861693314242059541141027337164026354917836356216176403199586575878154303477071055616195506968228492653124572699975255334820250375565450460927031345746674571299919936197935177099954428455942530612626533347308626391091879374640499733929194348273160556832542871859977912202317315145300001547905+0.4532763578852703978831493762219089773329711027058782022789699238738741056382554893240520127063282143483752725922998243505833328447865808334575130760675681554494219210126258968887296750000906631468268160695896772309457507664289645652438501877661484915851549214366623472255991600942007900824630228630337927503837945932029593840395763604608547817837462734164962823314113563909265676519579232833217737389444377830017255578192238*I");
	test(374,"Gt({{0,0},{1,z6}},zz1,tau2)", "-0.4071798301330173422755760256887498131919340554674898820558217914592665776289484186918881275574205055758210602283802188202896459250798888991802480627340237511929387605515516880727091314705113802665178369329688249989229111186904891570813306252257599233528176747275932821225793724626202093617177508434361790045450864800142791905915702057662299548841828422910437298611305941067019894436376408120803561108051797829552519179694587+0.4505946851263866757370555688522233087816237877869554660177966711370611866093481418792027077099732941717501858912049927973389791562721943045814308181891326716898736841892331126758924223431315543275528813382283883312845100859896701440280255177187041616734637376369293040234593121273644543492729836724092876783097490938881703029630548822125188791090730306671163418240668598932474810577536347356863319992517151970606795166821376*I");
	test(375,"Gt({{0,0},{1,z6}},zz1,tau3)", "-0.4053714672002618933816731005939111158604693799645477888063159689791804667877179572388657123696123991468305790737914133698658479962052322066288901999592802838177861149576822667592600612969073110498799351746982427099782884498499127769203071921596781814065123213568550676153293288704999276617473823596588465496584757771249536531059208089993075412490324636578829137522284070350516306303746814826216882753977821057624215623372343+0.4552939490452586152453475182246176802829363949285400013486192369610852172106210573497516966610623771417496109334700778497758315833229485510667859797758451524143252769845359990299333787279762379868176270258982931710460122021816083183629983363275436430962850168883424535016988435937617348107292740962128986951693281829097561581286371132638617512594536598802050661046600199654479234815115286749109506714604703774143036245399277*I");
	test(376,"Gt({{0,0},{1,z6}},zz2,tau1)", "-0.3811282729267603141962942439592088206806268563323808797741697158375413034819841738206615629298898168619133468431313483349670722276751057094524621912538622913345171354956764841346058890051885891054740684534862847388581569308963947810035591587978644935559901381264101894501059068905053803626439852218014768335131686648527382341297305865024084093013202487649849107444911127599164827113133580220652730214186173442343251877808212+0.2916693858125906155764782396792366384300106898777312322040191794443425141493285567728389679214555920983002712062480570025012732846547438601186384285453706652112460752628896761205932571723075266682648328511799197386930325117358602706122073947379651259301421209369026371518224738895625160305510252307633509501689049039567297364963037951820233055946907375276844369044869934908347647729897514660367349164862249266148804365393395*I");
	test(377,"Gt({{0,0},{1,z6}},zz2,tau2)", "-0.3890635810011428914465773074666051004137934807559307322151303656212341056753053503643864791275662476427217738876987361626060008721777570216253288049823832917463068773309466761701138070654879411425363238341255664943823900433652738195332839238449203092283212657468630193517388808341822372701152117999925619133486851439879267384255374068612917383995498754187016196576999021547650268806822009176966123215439183327042386065841846+0.2935527186489801482973423951445722229331543287121599763418836741113004119428837721510118034490330703557637797414583044401846375619613619406629109833060996408598424959272654560778235491027447240368939571615802391435239069451491277541548022583643093172505039178010127162911537824657144605682931109448184487855129302816421452304621698750299859594042559255385914259372443607181284955595109587498602302039566960993044730879829628*I");
	test(378,"Gt({{0,0},{1,z6}},zz2,tau3)", "-0.3883617557498310316565145005377881644421877197180344959389154899184039473664623454878116251370106447956982446752879161379552510309011921144949514767268652658391432658415955363062453921453791465485880659774044843267043629683330991371177321418139064876060107255115650231378733687642693764904422344629582392108800975516692775467250901775693358153497581861093858524357119905837482122643783681766760081003727663156634394644452242+0.2840554750426983876771729579407082792744277301056337645115851214628689397258127607595711321920406589449524923587251814202752560047841491114740162175641529749730819376452965603871287827032139282790315053977043868173135354628107284740483551097713434310715190479629642042919612157900798784567837616119531514301455152845948564339198381381840597496056149436175235557810514563303858177893473610548149526208064183379556947630964506*I");
	test(379,"Gt({{0,0},{1,z6}},zz4,tau1)", "0.0748813489249904428150678434340454888987969910835444498439692824224448829070888475307338703822917992389854490239133775565326824286916083787132163738158715915563302044831498751586202397204177165395652573120468891602192805880146610298989534119166723254091082439008410346153058888553354467593845096738411717693281271225890999180550737891209617180713923506229693905621673729619420345179974866935479251306416083541279456078809334+0.24939691565331117753938455343878021997130248402302348313115643586676755635428111545235394440346408949133551861634670770174318341662972717585243327803925641430917192654680130443317592699824313660508406701654192732543806995444411498953169647163858038892798371953232656299797857720940357344933638623266024748608100550498552763196559857781531608318209269418763813247629911333231699914341006983670266233201629150303308230590798*I");
	test(380,"Gt({{0,0},{1,z6}},zz4,tau2)", "0.0732174340058077802846177703212238401016854228560397840182576876922899218544832073715637366478094694797008002632145402266772339576788983417489992045521233458215533565775723473474390310389264005333980002579688939027779649424981758083577348889956717460467823651425168949268987597609207651884626014718446336557654001252216490526138145161952684187348451346795984459467922451988074184662213764431455233718183868083130916874357348+0.2469608592709847058597791639944372350232378677583338761801059179595626894394081779972433468996690986736812100045673225853230290482996895967186724836825574938319178834069824601630658673022826326341283987267538884280640068736594702962728628092044773554931886214874548662176407571513091797180347795695082857339094859796761667254087898701704038901313341808232456152738976948766230275371858110122728110130958592054664279878341071*I");
	test(381,"Gt({{0,0},{1,z6}},zz4,tau3)", "0.0765565391885550911550583044017566039912919233796505247260154836976398702395533045991574123969779301066974320574263991737903655968135289047140573183662290323907880013631897164462310875685506204586115297701785673819546596499946860771108540823982541674765177060936929182279539460142599429277021201158612334250449567428638227634309939445336320872948834339755184854386546525948519593114900546940467762939146838387919960856442761+0.2459847918594327478373596265294213924309055364445387503632768876386718631822249191534761804755494025185252966538562000103667803673154036977301167441127435835982360321331687496084152693514463149415597453504105692600919524470467115788875073809983589283288851583618785337832646664969454724527466356565139689247663104701070367807103545505891698655214472393784004377187147924644362316629746360142117195748975022825079429728735324*I");
	test(382,"Gt({{0,0},{1,z7}},zz1,tau1)", "0.3764452555087417969086494581925510598605051646028393889068710764278921399129628721374915001869491894706879579004548928511621002367634920842837507918246248791465817932676283534140050907021119138325727787562652678980904671464186283109596917385403875074415280810559980792794950389916496350012837354088955096670562666872095214420383783192891260148475307235941919797217908727473915777180951216159738238208986267384092849990800219-0.4413611024090348489196349426212069318873547737310197616087819040293612996499363141215785686818962191868961686946209633792970759700455545154429624027562386969214752561045274415831043143171018248125298877669799150444588604594957632933903489659659302403884365089294399569332749884705523568192944404962480696628017028168979417319161554423305609990567100585884100729003221737001209068773560283178602420547507714726298298294101587*I");
	test(383,"Gt({{0,0},{1,z7}},zz1,tau2)", "0.3734917909705657237183819104776077073291024809059984135055554420115742517752917406331322377452378198820238793728500776785719964276880723066258063926439416073311506942305873793705342666055825393495352606049851399410668899598729979517680392500148063700136496539535335311106194270526307264538977436793976472889494765532664241765991868487175429656869104818338707398785894775613198312735521961303459457038691876797950975430643449-0.4382458506685224008048482898669142540092998545058166343971039687788369617538188762547331925693568677397287981999546649351185423551159649188729916747607431006875354934345420549487751039784706625219960663338451748276529427467814938221856300024449292796998937062338678781287708914001793850868092003550690832018279538370022687477578143709294531174278586170316011039425171807681954015286217069658268272532050141247551144913521275*I");
	test(384,"Gt({{0,0},{1,z7}},zz1,tau3)", "0.371056044038360635859891808847277985717565260909934592806613997926021017818144431944764688421449010920172593102771597287027701140453730930581465009427634339910352626735389042796228153902606571153252106847395540232992410136455170543660028500293361781087231823831210243270816050169195838176895787050114806636682652419722274174106243850713308897047986421119787763898439303016698999763965544944885056777296536350281065209110474-0.4426586674832472832987495565444516719615349084843610768061662025198412915349546168658250241217619779117530637457534203727878441874447246218410667032630457574479718191191083781231065879990802603779267913011452935530336012989249606204164124049698653145341679419430584459244503138414836559640969751490324635356604225801899946528051977741610263825524919862864017675765575359326703019955371600496072905085110557347840647981220019*I");
	test(385,"Gt({{0,0},{1,z7}},zz2,tau1)", "-0.1445792304537847871399140979067853556282328118113820059268309415841994887893090585105949150994656613560501649393407539737862057209090111646734040628316843663220218517249399477904813903204274920903863783067043158619430769319990666367421645294133104770784696902649453581370247083408731586799887745183378587585053628372597927898097771696294480808651861285496091837710061118413993635141934869682461621226775834987119438851710299-0.032519085097623795099482025194826428224063077550518724628803650776457539839065388353897201619679917924612377258442379254759823525677446475372481367167709767788028114272957496044189459332179691418462163079413160318009062319861285228214685036877198014989336926814824178876729823144429277292929986940454534667336225555205750103217328594235988511477524689697312315732389174916050699785378496481778997680440547321227941661311542*I");
	test(386,"Gt({{0,0},{1,z7}},zz2,tau2)", "-0.1453473015219073464923075096398641199436971355287727721042938051583134383119291727354470012772295388708281067279230362871696528376472172146649499010941712323067971339418716654183251087768630548442080916374214421716479949184864226741352387228648961640891077299179258435640653683095156514764273882418376037664611424862261057288284148205382683347255190878855118384193523211614441900775162868602104114084545521669158116290680447-0.0303931431757567337198110865138058059009994364151746449840952199274858870157992320602487101822596252161304802520227856305590414854322490362931645117216116023837727148595291160831405970463173259384757666013309595735892660468161552373413931711794573269703516534812811261035618939156753762739419039684345141601947272781384631786003003957184858498084725874185802213596226242614459747609647434222779451028337263697020777398219248*I");
	test(387,"Gt({{0,0},{1,z7}},zz2,tau3)", "-0.1474283793129044709655480029506443015976750129723817847199131597003880686759569514680441354411300256901851299414314353730293569740887716956438106730720118536712801023193497124201856931680007881993628585938378817672112281714445285923161219934825544683829519816161520949679735883997234518706781614256584800453026592339465922489320402149013451638487760046059589927542289201940666093571743787317633586223338957219155620534270373-0.0320127994025520688878894284556013044639148838785593920543639170064975589913345382632824417705754268642724067227867371020383926454249275333143129442264139698744032072620225622881760006334767645135453545575653626146578492945952815861527580821712634228012042225508430515042030652648022152810058040876594714495407218760687633457669389798413253752695729948661756350491801463445882970048712543756148764382983351905670181797101724*I");
	test(388,"Gt({{0,0},{1,z7}},zz3,tau1)", "0.1597675390679870618327623222721138084273977911625594659679781655521481661030482690012048261500883452934831313229114307586404417188487127746788997224103016768148534253565620908630226875026794024348043789564844462785518983220594298459417348612753187699587874431544010938096777854910556995801322597193702419611692026634595261560788679506718672846356837754908224929071224713297147992647177974345919722255344252487729568688076551-0.1194164221655833901638311029489546887927337538329936175153971704572601915652376884678528168976578659271823570016930512229168019733721501198106446838187294872451239822595430508338816477149384434378263807799937422149616483811523075406532355152306786630211835311508748425623686623044121783475399087783608918865051245761206396825002456255713292810452627905340819776603456412770088971392412880518437945049302400348761369512542734*I");
	test(389,"Gt({{0,0},{1,z7}},zz3,tau2)", "0.1605650611280787112748964422029969157757587772421024094152935072532777690495257418253247373469242410975687159658942212736156031769304128138731578739783618887794582971763400608337399575109267796594139914358614088205380944630110201849155284187987308138585107918634023327297641721962911577297637127606045639651242286497599580600134892071429292528378940785754357687436317443853272110787489359828589290597643079987074653239034014-0.1230508635162334442559513258934294870959700700490314974386378668353674284821042784495071122533163471233819339956806513308644162338208899337895398199618727631898583767439907858602593857635280225432273275512202852439027575850538141415591865629943048067012045380754433069522525295336481287625814184913807997424883964309564717949234691284654549500063926587730123895710241516198806798019770211068166443384066669880659675681368507*I");
	test(390,"Gt({{0,0},{1,z7}},zz3,tau3)", "0.1643305041667501352394245490003478571020041357100464313163521114214630540996509941019801589479561895135171344121327432528747780280001155745758868371257137242184273093994390203835003767531266804019042226931133563408685916784087880473858835034368226931798894255650541241617418592322643325252043518476373083432509280245307909083757843508273205283299697036970835307737659479880353307487980294984787989851942059802850640359237833-0.1208986054984817253148977082124251209550250041219964144654328755146030779468116252415019784553423511132825257840267761726241026395862927933280313799396689484594339340996488148680912392614614035787977764569164903014680354344390373701738919795623817454858267633780164594255470830104988047947194738682202776491704814578827821079709562960557279988169004185645865717536080855910520732261049095845240168853698216388245217136586722*I");
	test(391,"Gt({{0,0},{2,z1}},zz1,tau1)", "-0.1696611366817842031014082363170637884458972461420991526830736142506664027186237556953608224483926365503284053479819362820307189761796970812284476560051019436871395740029837621183966442579860542766174405010644218231822784453847395791849556807713442466499238124526234002181083224792320374832658741428708347781969475901487050141375367348159206081053825273971004525374647199373454996298949133525246330239168364733540550232780851-0.3759440134449710977044536339911913335798823434761886481702098023598676451676744439548505086011623278729524704609166058830884459765752861048927706875454194383102769443851995647684696232898517967835072967610483157586497025808738747127409819234283750472708446202937213041488048603951700906888130514876637826668980007874622189884799485058191702063530832885416498629030650179518426107558035040341107191446978030107860324831741086*I");
	test(392,"Gt({{0,0},{2,z1}},zz1,tau2)", "-0.106556130639832991124416371855627400615291563814425355157310085326741856395226744062614617271996982071891044174269927864569425194898842661418130408715697696944103523486736569623896891822451902495748294011156248214387457609946490355353211029885106999079276854062297399318334389975526375877701515381365001054376924269438914076146407530649609244104866044490076003519560993588271792965041802073390367149880786757974659181248868-0.5317984864376851179669377399612110588893785485512717947349018298580945038571650946604840967568528177877121655212826104245073923242013935293982147005451218735657030571783051943079536322111665038046872603523521218993993343292991779964601467808091384195686477012385183170863826583227710145423654516578654975402236661679648196381183156415447042710508736823876808961717372272515345391256996589863340781958991623588669086085017346*I");
	test(393,"Gt({{0,0},{2,z1}},zz1,tau3)", "0.0435165990316253087026597228702488345733942874588946082458924528741607118284705381248774172133450257741572454523110217095514576475476450687351140391260790292591470304985086687688604408514381403985424226109972251719710503355617896277578921677456669418308673403874220290419590929743298903481040598458777811713344169775009100986054814117394868408236797408688517188536922010874735361588201608575140080026222058072740474661779395-0.4129580147054714923189745997897892498015640798890910516509570252434986821756365712197610787162918317114418782184317190225807044866271617128680114460393365357600507397401963204382359424441675413147924340972457443055235830411418966136624310592535159146417075002675828092006323710950490869170029122792774781158052297129817214111198875197634292880463022430505685337665208063986932264538365435142821363454179270235220290980407404*I");
	test(394,"Gt({{0,0},{2,z1}},zz2,tau1)", "-0.3194870910274995818138854502084594485012593146512223982250303823552830766377705616818518811255402889929820534866929427678029643636436892368076377655221997735476889771736514785012757449154651398847736691701178586645488421697342572599030105986825460328499415846062737155563640208665652727206258987779011746276040885771365271274013180907874340335498943296288427244911087485081526980128876852560008434991225305098801240087394177-0.2686059051834950599107954207282401788906090125390905663947796644351654603225733063069112352278669942120092009130271337975176196942853331709714980003809779970778847502424727351478454971079496433579628511760499779995182914418451398526565761762188171784589786548024705497050506105711742249779259845175010632403230045654407562832580812644052056943687141506641783862793510377823185541743410681297383266231270165304029168419208747*I");
	test(395,"Gt({{0,0},{2,z1}},zz2,tau2)", "-0.2724692281542972642052921920256888378260402173139585438193361763677069278421373681118236954530856653435648468147435645473029850546062219870523396988450768416191979320142880181632129496641582451526823204236817642376734427839431791001643412381871204022681821509388679935840992658406003999239835990476192231930393342536418932286623360648292814991382582017857344945577722347138334351711302066988178687065744178609944116592455031-0.272801263700025731975146903159314209444997654837290909001744829466014687915497322205228521121758037274372071761580983958829546362210669381017882534466290555088150561012369406180828589117539903569301442975063822214933025214574767465592678281654901945943704760853127001310218551990464279159307098382246224514647360772151354500294862444259329657207706995698174468171347063770041676777376969530813570973497382113101053922415083*I");
	test(396,"Gt({{0,0},{2,z1}},zz2,tau3)", "-0.2848023790909080419021053291375361812039672203595160845927970853823649574383564299032665728120910792447912438450745361204111838372368970735712036904855963381303269172388465864489927028895153458818988790174084418362456691390262929751646074374688476524400328836300617826879874078286621006857786489069247351089403397075958839294739463192194398700958866848002549923074812622926364945408868810313001234956896201903469532446749976-0.2181402326241599308939099160991263498581699568440046879306920719899803671403486456761976081858352386582730340997956166253612732499006834484545721403634429599621479148288815228498017385111332854266930010652121129422730180974992107953570237900519890593937945904296433092623969462518116053473877914376020886160016762579320331527231117324461838342653418458256184422463048018218760221447378927395660795442754868002067863459535733*I");
	test(397,"Gt({{0,0},{2,z1}},zz3,tau1)", "0.2415549266634169540441203172634498129112311087890143557144321746728812109416983004997391394185492187369249256862060846681152058231505216387769791885434595322465841631151834441598687569211350256315228528959902977863665348222133089638463446418277039735044471554371147653184367925803263111911409734202061050372938326942402505991414138583339391124840191315632605788379537104625444361738728084752842820543054701507614051112150286-0.0123648220659753690137963291384072763353979582538890905190413882299773427270287398496503993675232040534726716290306578625697362406605626771306388420082389404122885288414831614527078317634054614782659957660724511343383915750101512013690502705114730688866021502878846211720880629209835861489631717146151191738560492619383298292504122367909467171610277377728028501394349197350279156950266255319975866098507967084768911238345325*I");
	test(398,"Gt({{0,0},{2,z1}},zz3,tau2)", "0.2388577659972335073277419380579005637597661925407577080089450923166819565379126456940004514213292713175601952946448349929435053300489915087224810240343561216688850460697439055873612203537176255434889797403713220707516336563724508495988195844951050712804686203015319455751591233022761688994699303302199050860641044953658814497043481853566070584831045993502683157298562454781124417140784928689463304830174351569277786445154548+0.0148719680696887094116139447418424383429096827657320860812087910803699661651855593701357824882818088106090027628309322816515607972375704425223043124656538625978276400744606898747793315024032886753206248493319200403059253228428885425884482631096234137274536974367666251672749143279577769627716230533516321853815988286443778607327806413542813459957729537326665686387267930515908620253790846428569112609820400364202592955672635*I");
	test(399,"Gt({{0,0},{2,z1}},zz3,tau3)", "0.2090010338613237124778952488245669696390623792241763373319825669902024192380638102146573002797259547708066701616267572100330891848277027219772813158900505546465872118848327802464717644198592549068238019808702066812061410522615022363181298002748321708680643576921575448588975775791305188883539006007929736319581857243041644705724026938978551387097139823872861178299308463362138008446787820253179849051661527819642830967203529+0.0020780063688792561036688919384974348589233808546709064125962042079558956748323103044842568365202913432675804231181200786558473394772379692662030712317820958275166313729713552334617638951303508726551438052749848812102392375981988321355408567402986005530085148444437532721027765418874483106361010874038036397870483744473438459644860887003809312137164013252454124336616685280293024484299890740993000678408031735759659131808808*I");
	test(400,"Gt({{0,0},{2,z1}},zz4,tau1)", "-0.1861428165194969156674932261882980251306703969645419540382862372257722360178034075651046235257887698833603823960147860083395940184860628788807530555698834548230535455899932414369903337348007751331627622894162592477550046148714674530279196477984047996705144619696557662809459831316171672438223149413018255787663008199027283272796103753297280344701882293203680225418494648353641709180280168933971042885494133617248840032025704+0.0220880083604078378915761537366379687071338399235422734359674345274966011792590348302851887912083790186430649456794259090865917913060159377346639553525136179361408234860954959982375174594104384528619561559791053441101475208607490347079096012948003786882863616431316204389591994114581832464236753251089584019766888093740076095257436394202974331672564244098107520378456477843738076103997944465155271765572416125402710015446419*I");
	test(401,"Gt({{0,0},{2,z1}},zz4,tau2)", "-0.20436100803587126413671049650793713881580738755831354020873631497853338396245217491128625895902664159168202534692986911130292443128354306996303721443005801820161225619583604416310157720901674027135161533487598905374289640836400660351302805892919156000776417161574701669790286330531775166960155537794994059538454902399819714951302294856403557704439829767589955243577714368929502174012746806638479947015518660277340523086426-0.00530207614980249546046577404117516545527816414508858624539797586550001221790885626629157472945857659967581450461555249039851480209826171232347099904015943431079577193008922922644722751623516379270416752033864448751696174302481198458920406054681984545817295620814881493212946672424844386631742267667507508435639227697289298018534221789532235292718301369607191975828937493908319977752831929919117708581753330297296582716646*I");
	test(402,"Gt({{0,0},{2,z1}},zz4,tau3)", "-0.1670899117160028796565886840121952262958233434506761527860087436298554065304360557772717450094802092963763538068992254622726393541020264081525033323561472934453005322296040355593990795500146195537144761705609667711960707778956538809666719819443589289948025412033842341474272509466955087399780282092280627276569842849130947630100907413459745182510425858284039391891167024085597502053634607125482956156741623389778094807579018-0.0163528121503714690537037579113760852544008223048764290164529578837170238705567532738042294581715036199151521513567135789529263500364830620031287129298158726039246436453319525636639312587656268095716483573575970530818332241790746411519947895277237340369123720487101118512948460322941623155989197843815291246790580648177918572984305866811063057287582057378267912740641659138326563127970355005874855831388411356634094800338163*I");
	test(403,"Gt({{0,0},{2,z2}},zz1,tau1)", "-0.0928359935696905668974566434160710746664460813982593148515299694339750199259860748300472502834185245412685142822683837589359729640842715564851270970984572984270236037716690674789305695383880730282286935201393388545655300363765555078747175235034881305714977335871282907014715254836999500077022008417292910555409147142971252131674659931037760875176138426150437192628719141247010479548279275873232313414953633180449461004233832-0.4318546781604461684951707822827703528147283457677689246222916152940548759376575730053637028698005043688343971516926971615174610248940865247636926824792960621769757057626065785038023517752067828147683709655039388329543158081372057101863147802700343720101143264405578137892656214923053464080038474199209011964616599107454106535310863652180841613074597857475696063375933036988178604126586729232141374775282216439603799856586563*I");
	test(404,"Gt({{0,0},{2,z2}},zz1,tau2)", "-0.0608016509275372657184043648021270076368144639579439047177211644468581751237931208764814703480885190016518598534375916264207413982138136278190420811971589037564322592654405206249800180093595467076965487936828439228070114270496036627233706368761417321922320482446342242058372281105462925494444157073151708639575262447439528794899183521781439245934935519869046990219185943447589751134935070029357380925170475969814026401743357-0.452448739310170304245668599366074320981933970705932205150482359043470279115266967813842043229187595389554364971172629864024786780341563169186867076765499455594250478156601316388460436972594127369442930981361400885178961981957417442973192667674917785451204226793710662476124792334630068258225507068141544186793982713905800658944693792459852024909296018174738601119285080740996263069093686574234322375508263398387751364903481*I");
	test(405,"Gt({{0,0},{2,z2}},zz1,tau3)", "-0.0490854248761097216473323587367476916057954391794407268892412416470647199511993847884250468204646928586171920466591840880560392091391510926159130938919173534186383313153297280553016946309281206173773984933150576801678570559341619665360950851946872657421092917825774765283804797785193363176930193316459686223813800566540569479274477292653626676318878750449370009940770011549445956604519753972085489193036454361718834634318658-0.4094616444503702696114109725989416821599174754362740856134287111932318351943943552540534559631445884294113808070226940747555589841198022066270715751214827822185847434030629376119428542063573808655460019498086551653857490781953501236125130390916226308252167452532075914571917995927740972416482543319248340664825134048518824262899367413489617611691414436970024328587019976023651579400128279961434195266462410255301286561190202*I");
	test(406,"Gt({{0,0},{2,z2}},zz2,tau1)", "-0.2975471061018602465753560870500135630733798591130667926344319001403242155117131008449755998215581335711437141351108757277312951748944758292409692396348509443185175957488592243854680780052310820905664711839922687868028639559167872440945019892392906154251700473669235476483466158137838491543482616892616138898872993020188638339151619520961376565045385012291232808408471664797523861029022899992343396563907647059774138823461955-0.2479665182528868187116577042176519880885569288057680259982201204758481790738962258640397782816666688262414616566137043126195154163351077116809808978169575020043822747001200832221529626653577555040468339692118681301270285611572824534485879260806785949700857496694921609265018396569088573975265795906592198503967589183870379174315564654318120768482574744252491377660505317386856178710616489505537809308418604809837331831565877*I");
	test(407,"Gt({{0,0},{2,z2}},zz2,tau2)", "-0.2950795575035282887824827317017635144171151998992580897350392287020080527678373129930909551703006863324506021717579761438058784049909164581486265004667180719466587019268069260961267050481110967419404503589423598790379894761738625306809119626021367178780931383698766417165467385892871350483231617419513693921736811941724943071486685472825638235882061441011247130087804842428783408438885427716937895340155060006019029687144324-0.2527314699266071457534700218791185823572767807546223616483894107210970557603303055962515262464552502004191447899966188422753449557663419846505250714255870819440098301134990037378397438835467022965406645467919456342937158959176777793546910348307993503899900906390434436542056111844058374753590894506245758764320255106917800811646743402281129038307466967782747254555872158033640128602978324462465883434828347316635275806530531*I");
	test(408,"Gt({{0,0},{2,z2}},zz2,tau3)", "-0.2906884555567198006963123611437733318034562299187910357310300772266786739914332090770054759176660132436300067388482094833874674475029062787940563077838061736841600968713090571528121470788675052340257818633698249858362240784538297409334267249213313120958344494217593992964889285252645743591898176870055420978885792820626514377007361174542037691193751869324945910936489740763330432518209765784254491783694103833887112050632481-0.2482612908633820164049914288217177083067290581910693795825701367638968178926189679033795192630112593611175944091187903077661770057209652193328036089917206263970689303220270916207060747510995980508496957232921334899616404008502672535460970134097313090293355531252411530766468316651536197663789024525679848777413829744950363763520000274492067449376200714703548498112886691977166163508649046881353674786060247017742381733472516*I");
	test(409,"Gt({{0,0},{2,z2}},zz3,tau1)", "0.2365005192886622554317835183862701865374460347628375538508393095510261060551231270977848176546703262906940733587442565398713761963652446199888091815064452225360508582748507777667156183261597189870709022816347258887540887705099159606032237673121303921503173441962872437117909181926002197429049927055962585906199517916417912793895974686028735174482147412176601466864124677292335336411096056725865390523765777834565749560257877-0.0094207061035298481871638183178633724221427431171219726836181645989514755562882264371093243410505390472334207792783767014067788579366236564157494193998416808753565817700725679773432387104628818570170174439567927127997026060868629987396158281076637036758314623892260048399979156724601266540577883166678138029377317994931292076714770158265328476795271790222528788274200369713980764310567051383862141240107477083464429184944931*I");
	test(410,"Gt({{0,0},{2,z2}},zz3,tau2)", "0.2363161452668607077666435845442368642998371843039615805650103382031653958535287740498911949567048604642502032305163332250461142237122599013294970499036215957547568983956370035565172534082979514231106089025232449539937188707448449515018366552025960783046601106045099579008105868328983831583417495910698658497130970500993657270599611856802014286022398266470843802086373393064901096553422534201790404075017622646826158902786835+0.009580240006577336618574806851116730497405148245627782608715027291061026729342936459454592599623282580788791761070752996308622026145461771853580211953616616616315905237670012370182091543528075883045887864106068424500412305299006916082029759477413261985281863912048605596466775334232861780788754882746685401028417921478043304069251398843157310823110843135978867754786581208376331690135759146363461955405300365319778111102562*I");
	test(411,"Gt({{0,0},{2,z2}},zz3,tau3)", "0.2150051412369260703219913921406894069883310299545840401367899459170851427221438964888946791761036208127223391527659853127407743217279165312547768660348177138878208457222375959491128819112835038514214343164643286721312918231009788265576018944870826124049006683716728490694147619105632045249285019523994227786306039445099767119286455867571625198579571466779263101283872553702097400006318015472238460746899689796006581523186966+0.0026127066219011946589374456838793742229371203761803540268079974232530335799037204244367857718612770189314601454296048262381643903746164516426321974447669824290211119346695933357118636506577069399604502976890905626016749334311454170795837714158793693168139010303306843081174238946846183043549156177499696743182481929456710805099259538767280448279967384282137752150625828439526758048886614060572964266837514469749265575389185*I");
	test(412,"Gt({{0,0},{2,z2}},zz4,tau1)", "-0.1882197725934374328874213066100755630854986566699213585340936202193129489077426059477809504276323581756037097469378042114986658973264972911949485589499873206746663687228230714183438038396643163749844016241943925942468753185819670882436899399731166556603940864449281068762499067634666931961876290357721289361631712152143518959499831384178872136623778535009968418780447243305775784935563560107528589436629433172024635396874514+0.0045887341744923693616684956344741500156289518407487957343504746082226104785257618440116128689580137158336283156789734835511817386997405417406693082885653650027683395040952301421546411339285393271091381681970852218973072855561186399593958394840144356130155122556673477853476006375710533274784202176852960276360472531832530430349659597514725954938346369785345197974772915397392819145405021948281346492985558299253102040619527*I");
	test(413,"Gt({{0,0},{2,z2}},zz4,tau2)", "-0.1868127202609897536071465444199401906979335599186741642248075953973882122025429421225748857337538120931629185622487531939954507587632521504133509954165553089260351097435013361655576178006390529045975388408264140834237455538817910486858974074155238803536073798928214251004923163280746639154927691546816480798131973746314004697616106855693386804100153977796074130211369302255828883144545180447343740313902495478077012978837961-0.0058766946704696342786209874395212007393019596651066235279985406059429093993425281903495919319977424245195950146730217669175838600491234126942174401751588093375552928424883773153920878876941301295501568619023096802350964401080448257204122125743673603283259986510400008236597004995796223851583898873920179607388366696592864630891026344195864315783020600508323767188455698338701391744721217083414200791247510291074094956192382*I");
	test(414,"Gt({{0,0},{2,z2}},zz4,tau3)", "-0.1756025992687528828973347013350467931893859877842362867329329238064944352883794712290489697847387785995780905631943918880604936261299540041329540202617915420590825919720448331777140050604627762519017815654719880907984014813464634167005737396726603463009458329301782509315253704850021475952607838625616474399637941675122690300588139367410571091567719596372762678831077052318014104549937944341173171689436776533616757608858237-0.0005380048836837665712983626767494882175318718793247188568164013179903348266144987562764164728552462425533914104093606791686659244719693663799920101684190024397108705941880528627815542780590018900275578068644193645279262610850087676121287699969224650577874418165402728006551646621259670827740969308271860068495210919823948266795672721296201116192386376045045644882537692174824275814123215818716194516290091734698606162384186*I");
	test(415,"Gt({{0,0},{2,z3}},zz1,tau1)", "-0.4075915165691095854517405161936934350893720038141976507471135642924748630060565388743449073126293773029623915619141815863460375872897308814401555435172203480340474023636384320710743737434950075970788804538666646450141923341939436185376383720909566556150432734195240265609457220371519508099260244897970931275931604669610305409120309459672651757881390205169864585827139504558696373990886541883645946516183110718459372470739824-0.450810922090023982298326274835935774663698341862789808807566749110423699760626412264648683735647647217796397986766886028856483131808332286110581036538762285929892179085234154779328866153463362413487312106361360888550754950329678009066942026849368626806131445567677196806672305707527878965171376104814662731144122269857374231089686269653001287427740297372290925849493804173506677981364634878804005612822543671279719751314933*I");
	test(416,"Gt({{0,0},{2,z3}},zz1,tau2)", "-0.0238394757557477511182862370824491013426849745013096001658322021859169728887554627364730595803534523731428083424170742717100791032548382844336397579101781185175660325751709967968953923331356495133276663442317663147366647059148772069478989805030579183914382707805010464169203350779953783823161272114621963981835523892661902879192795507573196439929775642294959316566158238403222389401655882183747152652007565250062992393689188-0.7357352220609388358033506115809064018751924653276682122946950899527317455348920649506047823761473775586972037590874694783491596488475472989405090051614256076720747536075226343322942417863280671273531462061457302467935790593086659779175677932575422234248349103685173331265040583894322922418850403711798194649620159612340693639261002236449954330645888879027111295058479550825947186779958476809586085898025823314045374172493903*I");
	test(417,"Gt({{0,0},{2,z3}},zz1,tau3)", "0.2063746164720446496801271244488755237656080577544386921643976250859703122014093434983531990302828007440179932980766809057426297593983689571660544977348942376652182678322105722816985751442563229640843244590722436119843020352321126648574248368328389387964532325678463437519152426904415513792212051074217814359050011730546214079331651987215618239852045746099538438730728929415456172646945673532763638972592392013883185632455733-0.2466496784253123881189506333428022618960279269707509410234493690639199329342757211708803349075807453579865076710204366570323382109859817042999143829786286319192339959604434422520085191959583400009610881339193887392591584686619796874575953412853259384618888750154106484600317846804116697959656378444850906461716245586010026935943045802859601878236975449857039050559570048279271844938684211321684673191344828723773061335157042*I");
	test(418,"Gt({{0,0},{2,z3}},zz2,tau1)", "-0.3278227326020084558154597421222311026625305830064642348932032504496619900034316111730497692443519384558725284887889294157822782447721160761510132416771977301921998283073297802228880029989443325860400364001400434813808887668050289551212575099152921953916197343547985996895691614591691808034426932068289654916224788322149655184412360186070284664877101709849052471111807642266671317571711812903999993875089811305860064801059472-0.3338852725883532804507807696753386932318527233664794121278762532541352082665399100434822504355541285811107692707958729395066375458008806110393895198630639998539554849441044536014887182226804165773323314265985887441392151676995377928516566409117913513920885938256477921619846385696731028846406810146244098963710633796891966926946449672365911645057769410449929549085585532944929737280250409315003287887309871613154450034792821*I");
	test(419,"Gt({{0,0},{2,z3}},zz2,tau2)", "-0.2051525542986813673434309066314217490100547119680713949372098576388648125623088940329173144522870966409212757161054312507406328907799627697510152603711493701959594355230389537583040466523583079641481297180374000532049269112169118222092502486844912420595476869547728555926386001488535755198555853512089639286015213892547823838358672898688484304196242502011119960612627733425121808816847550119512659007698813879925309654575819-0.2763926516795107817230773934130725645581187259314712315374070920111323287522290669981798126794410167121911236589285532446676229011248247325293562246777659290762996839941773828975874149207643919983001308268326964317509938607678586671687679170970271249679075386333050913578062396446171717122943738842669497939037740117480507741763623316536325773432888788855460402058042280880585660038712488309394103730212102721371852598731166*I");
	test(420,"Gt({{0,0},{2,z3}},zz2,tau3)", "-0.3185027907650699502817151286816035937345376820936409031057739436895839030259088745882990299452882975511135219605486174872035459899209613452952758420808010751572129626938954557149465735416022248512729942589264335888736002669293713341158867079670116950091834898306320701740042972378161269347780395199526671903789316047221093210300182963719424178940459298013171472637942648742092592497699815655593761232481660787438951167312441-0.1577050076434232641887049429903315807184270417937927903168877632741032516425408121846548498509287995347474706722970577507960196035894628688276754317520583537954557143188805780207237390031639328647844245625927166504863964094768291950488420001710558125100124863675033336606376356100264417976002502973371193387428082941348326181069345390214288324198263464299538216633069753150811969485562077048692578545056062916808057647948621*I");
	test(421,"Gt({{0,0},{2,z3}},zz3,tau1)", "0.2737533983657542119651881892795625049538820136329731240676095718335807659392927641480897010119044947286875197960692887039117668177571838276117144221084260257560971698557614043581457077314076096756717405062629086659124121465995227424520579212516678787655131515459509592454476329538492653477796119022012430990758650555993742904599935207507124948579856748099053694800745472696350374444333824304660969311894192484456024727006539-0.0129368159679757787784954116029965868687193013145511592120071540283539488772905719239706346668383785652443751702575589833760171465926531157935872840452922946983987222073962364732759525035711462365798317762420749016953346395273414823110417860465213276182929357221108722569844966497680200041345179066043452743951849155154725914925110821060441610481691510788137045544355042374728777967118466948658477103539994889634884963430466*I");
	test(422,"Gt({{0,0},{2,z3}},zz3,tau2)", "0.2374892878381901939126506420434750060424978357349180662461357268603850605538666904620451450656274966349086177382670220463873364604757197819863182321192364860140044208780923043369953625461285078015076545231807240598680324129851130958587368800705578966382006516503258758792697633152792075102454018871965221453583608556423072473065687573627383074192560665549367415173395558048495401278285001156399933281440776535474669630262839+0.0476695478274666284159231345988675052598920791631462588962420660704035937139327668406752125888965594047664961834384465079957376910744352584631920085611066862985785337685282881744668354450690275391699784518083904707307769348085076993665698335384836839971676311035555190145694444435206999671630635515035002033084454398559646318450800356216814647958770136678468035703773547746572732073325504596514378851749147545194852019167961*I");
	test(423,"Gt({{0,0},{2,z3}},zz3,tau3)", "0.180765345714423621877965190677261117698255718552315579531562400240216034212373806215454749782015371263689743071115356897929448983925861450899777100484903749857800422455428793565893502647923818244049534209228905570378385224142985627830435693026086969162922742381024696988232723106139027765756768249008103248968917284820767032598169438961297037144835941884341013022961124302745622967458138337677714221837220212658711195046988-0.0173826879570888304834799849018727368621121068733458142731486771943470350637376275426595401183135329351968807395335047266457655207339954828389566624298411062085450320436766207833946972445730511830700763339769753743793151557585416996171373888109064365205849117788787266952585027237225907118096043661550734569302033424048812513462398887371047779170007839041580430184513469964425603558304268706644344277937429817941534268223317*I");
	test(424,"Gt({{0,0},{2,z3}},zz4,tau1)", "-0.2209079407547918054477829227563424757732001219846577064398244610216211080016743312813721884125700543325761381790624494175723731301044287193042914339459271267255272379772235877000093676126676144165530568054052808129939439145053686668060412678899254225990526301114213039667014038196176665876070175726158438559455314049589160308643262164887421753496832146577821634973161356926790660023025510442746591761338534517355325300818569+0.0518697176490816872761143312164667534674288090270479080725388537262807289237019696997193830091551203830059561737139169863448298837257074702917347207270521095032009031720380508712915141673615230248748106345614754449414179560476992402412844618605591547052610603796938557069497242092159688290004144612496439082281887896162946777604766980342557166694274929990669110917216387567687803172641223974887959043908807874373273052305334*I");
	test(425,"Gt({{0,0},{2,z3}},zz4,tau2)", "-0.2320287274523292054662796472710500984047366039281189909536596797783130976460908720848818957891695970767429365941767687233783168400579897038202078632850610642171060110582096987175616596604902188021162534086291405212814007843390925189154162830651189047627009655900566342779990332208882011717499144448289375955411726598844923302613113531558310162970593857906426301645150385253370577477847307982924616125883906408914312588884118-0.0414411172477131594721164143335064859292510956767254255155360376869561686200451572011493967194769567426409921906766368750436114345987511253597836391750710537487681223921471450907869770432930439811869320162052870069293392724146529750171586345978900931647774850369128199600026228333928357723847779416803399991815739393622825795063418544305511339940486672269697919104856390104099229430983507351675212790052027360166356364913166*I");
	test(426,"Gt({{0,0},{2,z3}},zz4,tau3)", "-0.1192187747264416355570040175306136559500921323542366202762677498156502694741225285393820398822108457691202767024935613762670477821425841656998746960986435819965468941611969398713057599951698281040679423452197821314127848195922285156855909677767677333327575345964905293175213828589732328501921729785080338808303758753614328895941882735778815407964666400346140660169743160095130676011229135271579481500580572634163953347800359-0.0220167269162622635407059845850712772623502774623771327433142061810024097252680026789204242213396233307734411149873169209276281992524094635853183643141040515981782357970242587033245923718706651271869895019952209187834133298147883876241157852226630181496251968727269466876982915765702592233131775158883123586177955464408460959762023196997247437306500281586106407983281181154724047036861199144857737452249286108707681840337678*I");
	test(427,"Gt({{0,0},{2,z4}},zz1,tau1)", "-0.1062201425739329098722248527940526484311324689689010352226018454231689313648438392784920996578907550440705752935088961820068234520565482167672551484681892519089223071765290562248826875550837188007936801179801214026967874286056928852060416653274907318822619923203410150645477427691082892302836216330968098109992031490323377058425206155493434934979421080324090543088428233626908857026725271340228686624514506343809284044747531-0.4408055210244848091204244592910046790779898436165141106571682975066007014538360796074209132086993823452723021359034701265975380689947444298313823783360123570653392953854361108540076841942256286177227747051770024434791606889186309380445351235584293289181191613423173790269014022017008511206537662181259578341851726794830819612053426168312788910725786593210375902148651220072969167935331150267727469316904225901154432061720059*I");
	test(428,"Gt({{0,0},{2,z4}},zz1,tau2)", "-0.0509833688596226223936338692822400839238490402198338965364201117356025050969234039304051025853757413640758580565568567429618983194486874729428897104827235491285038567413259283682733458342943757754318815972819021321034205320610937823012200796876955058109518425774763238264024766709796327134097822356285857616372819394875789052790082885241645588466465996023924522731516162122722286232651714136238376742414340702135315663908785-0.4649050043611912754226481063165488487200595898396185494207352258707634822082101554267144989333457371768127968896079963075866207358645134097721890327364751098995678799419549425281771762214239316059399708623344501908274715045375319984884320507446057195861003823336191758877609293210254885555948759376069847551611725472180491567634077035145343511807603607967970899845842024783145296617448540916117466367655593542862888730901481*I");
	test(429,"Gt({{0,0},{2,z4}},zz1,tau3)", "-0.0428899188265458689370357874708729329277552361369079480591189047600214482060832741586361277294955578261089664827671867402175052545165562623953032956544462320284833199909078013526813605883727455805265446624997196734930169548694830279082345957161780377766026666080844195628952830695438336717960312815597294967230685122163808428997545763373520725733331850781231896508007552730054726375508354891159875994657311541343470049657674-0.3942559376076217084753480679088636452169978550154541881035023213008503664333209392884076249410533304502814124500354720514254254767189613918646682414241563591969548585313527900608712467820357060688174960220063489480036671169126644920591836792861788149432443079730052095923752029124474445049538283866272751636754752259602305331679982122348253718535576958427461691527546792908689650533500333951482981972942150942402802021651514*I");
	test(430,"Gt({{0,0},{2,z4}},zz2,tau1)", "-0.2553632465476421617351443819244521653248967407339779668938112885466714647300427459428531931092690400567384307528477762303911409581457694662460544480100013513362334375621584347625025779152605693839232878042867654595287787491818467019212556110532311409592058999126511501734125379315670122932564620508557231607669994298827003850530337446669215412667743231578650317449214927662088989325411926578812283233295372826105174482190019-0.1805882795360530718115794485463258180636152060623264338633950751580561512832920728067210851278135045768584851225293609378151125679532888243607973137069248479529440710322609729146883161544309588967383627605139348068953059721394649652316650112514971139522411585693719485011380488573384756904335586812658055666547648427439837558303378195773526174354635708383861537265005248697227101410009375016764605331081557844417851472652289*I");
	test(431,"Gt({{0,0},{2,z4}},zz2,tau2)", "-0.3660821997274797371264849237793113687712846080898891757733511650147920468043402479287112235298439397618290371972199487420591707380734717514030064130054275145364828878133370885715386992658935878335797652724675072670415995403528739671022144908883858590287038682077343373199548746433416539801353369954805227391498654974805748228920968106806750122008789950226132446837423576968237608887638385254476216974062331885455420215014575-0.2182490976623247293325582569878100479011086099224940687821546040189896023230489281845204866805203055815003109410570068022986499804992777012981920335862688205124953488313010232429364457350705297183780714358252972978938586211285075732496951606266614781588238345603257363730268775156022145311254733217610010275125780423937078905831900022366688249411222880154427854811390913344033209368559177515557935494951427162628878557331552*I");
	test(432,"Gt({{0,0},{2,z4}},zz2,tau3)", "-0.2857653405373365873925993938765985154908708498393117455289273303119611869907722883492902636050318846186809140907560835987062544404133829876146899277523502735258091386625715011258269261370844746933788679009062029477033896088236591909635253498199318822502336556173347514648144307579275659889467206127141750140550955127002646237967838327492830829852086544507931392419529438520650074874412658327226795192886408918384007093171782-0.3270386122574516435227467381342943598207253947464349578293141571546696415162482511184348911628010585081970730406582498316936192792253638881341123384778422711931650279591048517243186530311969941059591130755003234999820965419427038277945784263659618991025469801134659956522560150222086358958759576976536350035873461486573441312962314518670165315379248269327474933477276278085327565778009125980885682060339972948620330606028682*I");
	test(433,"Gt({{0,0},{2,z4}},zz3,tau1)", "0.2230015752703334229494934928003808647102016367754787178710029032373275033153617396709264433220285063305765852218525515510764091178057189315213283560347759331356934665449348635411395655067976763277148346461853218205461728790388455481582592860664950588438908589336476785565629202594893746974969825255820345070196976661217220635776328831032273280233358901202585017353031386442802514982728388859808528364061356084710733833928213-0.1062729588343645698765181569890739305460618941525985492449865757123504951799071118015622121422703726807976653884715094196370160327180420395921874412854486535635718942983270677522316854391496446410356912399298348794699008849586227103858364686770060875563638355809216082894905816114371423178931312949599166935568404679252318656952280853654278715237740996587085894433151382361311944120323638384651359010963007463785661587415832*I");
	test(434,"Gt({{0,0},{2,z4}},zz3,tau2)", "0.3284302522462558220548340949979149412819354709107751113740236319926987522138592997006646062457323450587145430855864233847876066670465039588319724251389639110660979233944138774595345234015998046534044762975288792711026134473965089985961508243682911422837902104015756517107857480405437215194415893261994676650932649407811146010748128072669247800234833182097290504665556797130943690245569842027638154819477112424496424701880411-0.0000577839518883811385462745119885117457866938121729142678996538077707506535093167001482697276282801134512008555589936875327612447304309277862820916120412974718117168421238208756524424164601179194371972337713911867393885420373819398524141224426756919526292775947990811226103018901397169084465000594339957074045267537327847084537612759242574928268124650410086956880372946485003617263913329531276035160545496030399621759413246*I");
	test(435,"Gt({{0,0},{2,z4}},zz3,tau3)", "0.1775118674277674393567770578912751293405576228218202606156793142534710414492510643645206007807497571198840380684229793899576346460859256435346318492157656810108856180406882241288937084698441948661162954974563089670692127956131642910851611042547002004292655825487091461552605904448751754004150579234805418135002346869449764893861218802160693399944959882433132765404067542656348549367545664646999509285851348769864630922039072+0.097306646356427920600668752596552165302203955259146084929228501416004488038090665986536734897253179886186201362324547185726202467769272313093112131880892020931689509087488772288155703877485446828114211836670781724621892806224593791843766275037522665473381515858277840467862641090984101853108272507076804758388399604567332126425430426775003833714342032882995220179780157512790038250291499447246880471725242357977331691327902*I");
	test(436,"Gt({{0,0},{2,z4}},zz4,tau1)", "-0.1994947830422321664794147741548712705368806158195757875639881928680845924115450022702855020794433474129470024304318733637111231302282889515619479045495569465398687137544325843881464301449460061513510384175986543519343433147573846175934617640364949471080935750899304516135507601937663681529767334253251271542265941136328155507142595272976112313856081510738738151040197497662697192635406109918406616593179721747004625036910435+0.0243093752672640469185166521867868537969046480792131405597526410413976451733574943175482809134278381140953538363506887944707539517783119305546207142715598548192360973844151472187254044432374215855505841424227392460872061380592305036030393376474756540740332438045592421700889273331980491258328901117280119573941027429036701412217543517109903359054795732779325295919242029906176501012452948810801838187815662654357060689716337*I");
	test(437,"Gt({{0,0},{2,z4}},zz4,tau2)", "-0.2055221423761170113538971055357427366917693630179364133645740619781443413802173382433666804003435550806605393723413724011760706113213907976510788132030764391779564343179897220592399034227700002479538561960792441503556053107132791336558872444657558978808781458172271987433244648709595713936636321089036716546856455099634504838037271624346046711297865218120529873506687437152683404317454706815358376148705810680384870061776009-0.018670212145406654526186723706662094730380294573281512658430347014037283636105899075071899258335493801980480973018573692711312346916083719358365401727120993324453204944450502762604338413122278363509197330830464769504235748324107540393814864042452828325555540774551970522311230453169914938177241611927069050535674320593591899421776039203359282267808086944346116896459520657831799363300872348187929800466922670193443693426171*I");
	test(438,"Gt({{0,0},{2,z4}},zz4,tau3)", "-0.1545346005054954781669122231530507461783966742131599900031158664972574210836623823965740319902534904751805092634997004656424957771194794991339729034467676930952170477345790424007094975472587478143382962004362885547499604583073949806205089010975815608138716608565664270653130892298395810326454814359964707632489147361609716401047067192700502626669728965769029846190982078655155217466629452710624865516200613029809329647697016-0.0104887356722689284705639890055280407960101380883191737890435547527912373395044902349393531945055088325059546436320032566225126620160830973688578630101474145304974580194188804607220865182869616047571938871728703995453183215514134190394041063738112432758529694368410865162748933528476108413965029603602274166096264895345687439374725943646030027089551426070036641739829354885671005659928063139601113244150386602333908376138251*I");
	test(439,"Gt({{0,0},{2,z5}},zz1,tau1)", "-0.042460374967454294519667220809700399948888281162709666545661994366796017908525808651885184944935174353833047038854615224171067991685245637401107662103378848793024180445785959101184139635348233522601727158424265776251938873782670317447602700574790971141571977702517877035932782031084626331332757721798428772023249748487596285085905179903279631601264048092189748484133249478097953973355474811556877826607442513216192911171016-0.4186227178510597196793593128903737207874776673488118675425974601052607007297024699872212820028915649130706519659962637956388331681725028104231754325021588846405424600406057684113528529251672237689009109564144245936458494078925880134263760119739413281558262966174665034569859888026801330962498219999149154042357900876216717217524150174234980329287863860983932223869329884895818059196643940533951684714762455672798243614265129*I");
	test(440,"Gt({{0,0},{2,z5}},zz1,tau2)", "-0.0791022872114348035556951058988243507311404888030057788523599278190439898647550541441817048702045405135413323859650387806087975953088565832982595188017465640455399616735738071916655552430937954827054371476833515830372480898166229719852598711037486770486971746539965956796053360937552716007057630489113333390704823010923328004917892324941295495692385621728508074569492421968713431480177031074906835347326987192711551147822686-0.4036658808451089879069871237112579368360261244838067179849599261985151451712075524598229248000360816257460976949052212722949310215165029413524462758318423830481760504451099384706369922069237304854914682963116381329946092384630800155114646626940195843826893772207772638728255364493656317935232133873207311560560827814722868536268433979405731407776235518903936281292101782907987380700879091897521123523707317310180068321867032*I");
	test(441,"Gt({{0,0},{2,z5}},zz1,tau3)", "-0.0822220742740720307310070068417584543330953237379351272890423877185780448022748559658953409636162406368777464809099524230940390610956051833949780794935996353380265613225298461286607661220851469783402166659178804162118855031888618808339166897570586264274072117484052114829556634449473832980181669551407325572860575075786925607333223225048395332877998167430875233961101628727008881637104554975073850048827126562150954968979296-0.449860639840537021326365353187004776895963117937372074880632300978323657927225385060906389405537601517303421537146010155207958047817971458985130464452398061655861842712701897134996418182195733072753047494174925259921693585954921692304582082586541057062258049350987810739222273917805852369413415669096063018678882063244938536418609860976007964295406959522798260240085802241743529041189584858290177593174560474860580512848872*I");
	test(442,"Gt({{0,0},{2,z5}},zz2,tau1)", "-0.3170627655219242828590922959260672576849953582260974149083176188467426088743509539792857418663023339898098629433698948394254464521396651790154760023006978917663655530987276395345774034793089836822342675947326236899098457377174802405449037559196615018284479356857227795971944464212030232904816691526206257093096077942921247014134348580920301683506602806717626422807577228588533821392333905897774214080566667034955083549433255-0.2887437170422882098637704571138428419221504679580092772079877254609883546799565036303349050580506806441076849822788734536223244817781369499070475941423187892442403781758970790266271364460770436413076788725165523779128546567146108835404347216806120790729505979849055487053125932971826879593784515754612452121490526965141732579748265041870094750073899525240326109884043339524062317623946502512581019300161286556984363632447584*I");
	test(443,"Gt({{0,0},{2,z5}},zz2,tau2)", "-0.2524131626907622162584862669923662124993943599152713484892440665401565215348109294835372892106396064231747515048687289662684801202754680032089579661227409141585334921226961242199937742361054319032278247292785803343556004230176370746308956810319545440465159755008654742427667346542575983684027536485007521936573903589898063833116903396946249398257595590560343705986351890617042848074412868681927707566570158062315442557974178-0.268417541707690676653096687875118637025441621629994079512304495031105167736913256962864156119793837816723753564062036437913342094790046968828928127791102494885183262868307875521987059251262902288207475464592266742321095829459510235602912908471644382802700408431766275417509661374887194842026530016496715896588575379200598435473847264679504496543677253115679474858093771368837053688921536996805499067942739628182487883844943*I");
	test(444,"Gt({{0,0},{2,z5}},zz2,tau3)", "-0.2989572167964991600065336333822312300501681926955828242116489964757993251566367969026538124103093343148188034653632759448369141135055442607801849787445377664339021669637975207581257295175324220954156637414860740747647620490324099351516138998503563041812939558516879170604610972989908249743256959997397388006774181989649173532307059766350522303337043664994682133104493275842390937657373132221302297633447670784739971101635383-0.2031053270755403330575471035545389923026053518292936122990546041702967877979865053643153227050199973923308819908551500992368079071186124794421483038451519588803166851734675572969685502546784799775286109611130380628943339383656459243362710413755376297127332215195791187037257364717755214980586194693516255580584517059866638103023712025227077810049380841829539406513603132964286392360743547807607961691834100396952403818309033*I");
	test(445,"Gt({{0,0},{2,z5}},zz3,tau1)", "0.2356596269484443862395438506629037556830011700194250072364383675027843652982588218979896857317099377185073760949877901301858384358388186352069164564919112445621794571949772599911144899837729536308088421758317968047971133259109271351941491307243377396161170224774383438243047190453671791623645480214053909672084248510433451042889931240477651944769306934488960592574880861963681102271191729182893414655586482353712652317090909+0.0618264388168075975282499043756262979501129883708453828067134996013574700860065697682711787880314020942711604449088950601438527962216759291236472925358977584629642181535856431644048297831610849701662307371929648271963996570987749339681957260017691420428892768291525837753571074830131782027281838733000035783745462871172644928220507514899440443249095853294283130641552275191490631724819243904927545544575237592897680306964537*I");
	test(446,"Gt({{0,0},{2,z5}},zz3,tau2)", "0.1629218083436841985632804455650447882151068673084199556284517496485495026737125292468493989035652362469608876005251691667836994646426967378127496645445721670481199542632720591989367643690183383570975978879075585960353023710526610435023059982667031660406049706816369942435260394335865197944563878826340856215375850375594399518837808903941927820772725030685920412813318670224619382510327004841204769331930792115944091900689321-0.0002786855975269777981039626739857131759488090249996105204884939042185159875316471380312333477031784738806217838822556577627528681580725131447846333372760379211646828490811349148310782784127736901327939244248021063593046636096984479038258922787941833951169019337391437800845066622637609493541104250978115181452983974293690589082724489055426945525128191570752253047258415359347955135313815234988086924442165082650078421916181*I");
	test(447,"Gt({{0,0},{2,z5}},zz3,tau3)", "0.2606330894037450702238227126141909369580509511671307157444093180936312744122818420731547059870327928437149055263734077007997666732582335909206308964055854913405509676834781880880568428597606102959113823067761608472289708797674698434036428161516365987821335600173380909584164513601169068313673698169348840251833022096427362461221301355431348688842505234750905969826681297600514401124140350578037125839106152450836583146357987-0.052579661143071841128074825949483525860763390668502432011124234044940350426297928598141221073654905459018913437159107794572337908466008538507932840172036435091862176735483907134606709422596989094130267635845389265208068891193381776139114085677486501689828471975585539686168870872602636525382930789332257539205156026088544199042898826707026579825888632980045977401145081862786654173326148022506883525550616841735278347753559*I");
	test(448,"Gt({{0,0},{2,z5}},zz4,tau1)", "-0.1725286235258200037870666138120714899044983852561068842798052597154937554210871999404357309044919819711777249409370630958555957485441373988150102856797693838707696936000837521077617649302300516692729660027431772025459140374050202102726180108161419769399239743968910445770838185916123933919181927371747993644610241976510421914179212099032240556719323721671131861917098155517138199562411713376656474028063081838719979103094911-0.0147797203677745263180125959061995759955571404463968966319306320553284094039821918674491588687820496419674132387752315876460380905559264804984235115520566109138974919320920413408530381071436495839099981989389855679286300636726399212460888518443222824713914468292206880079477303587395679220706567671444467505136110901976926124158518807856504701218687600232458588949186689744471287458565130798171694269910571726363852244772771*I");
	test(449,"Gt({{0,0},{2,z5}},zz4,tau2)", "-0.1691311025725607633896067871158416415896650918761324254651300609540265915842630955389389034946642102002928441378674474095580204589429502698331830874651188494761087621080253952941281267330938002774069871350926096301583435222378251035033871406376487939316349654291775244787630232965531665062929262248100134921515057322520657309078839753344956436702295840677412672291230225864766592883736607719823118204890560479302822034293045+0.0118141465795220974172161307703245065352642695228552884337723276238640397256320783130139637175975385304773358728162937965278341102261227069015115129213211315817366830812208254876354490176819732981305440610281271619640048122606493930319154003839674913799019963524729331494053843854855587574862588291074307025803704807172769292085345841332753269102473939537213095094421747283811239802149453226286857552805610662912674978581019*I");
	test(450,"Gt({{0,0},{2,z5}},zz4,tau3)", "-0.1997588054205360296656814862067290453551996917531933916963729100057330911647107783650008708733120551311865239640552196218793937803164874735468575025744833740994635440088253028579147609171567621843496929012646766209356842751292534582193724000547528722554390824760134277453124038363722667718799913293309718561726535037944839909803598810464903123611803649663336883433031346567083678405989800384919211061328037361359538565356175+0.0058318711423374314622651186008358058087106403132701935695766052862236043384854355565050152786267107890862283889227925102758521076125977247823275347212467730241982852687699088166906213636892369213518489414652240295304283348102212414479476599532632368559823931121100327252076372558800432235692438743047077482650911914426976754089230212155384218679763498721984348991340123382430619414951925196368522346365179905516209110262001*I");
	test(451,"Gt({{0,0},{2,z6}},zz1,tau1)", "-0.0760457075067825601129515880640211461990057125648776517289803740099687716531203233005772669376816860600526708791856756614669993235168442942928299360220154630949917506165250742211910550157114454954498055914847607795983385416355362244409413074210068107541604218740735485232816180131553203275103968799573887311006716518084386425981974970252906362645169411168679523306300895150436979588147406652818687323914439366044004452429986-0.4423061236577700151371700563978772534104190054843548633370492498417148620460440259567904236556735008756809441455419871364764425027391937301391567294770500973333131176516899085648932528480317481344133012574479243757398156987948582465251537582101234899914385916691031651176248754235118081936066532780161446865760172897923430119355067866208522358762483182207064365709458158441720619257499577385229051525586436506347752218207715*I");
	test(452,"Gt({{0,0},{2,z6}},zz1,tau2)", "-0.0519734962135696834547772632423636879221663453023921092997648145824514353132602034297737929823651394460963588527467624597515768026001508920464803987649417088355356226652200125137807614628419973134106600883768247400153294362215984368547821411298063880112833211902205598538172881098709300175496520811872708644568697389970568248970698191490862232506551016506002912878883631047590290505001536108802313800039246649819025732535595-0.4349042883244111473969194972873825270286062767523833371976940449992132950430440632695487957532386847349063120225922652538275863198854647321453066570969287684864838023547021477633759840872516106856128996224911601950921127635337714866253843051437266645054494705601307146435463375080605638876521561673062741312029509881615622492820582920525303135049985629392100837964835950595474481470546672931641463655367405596630332284835903*I");
	test(453,"Gt({{0,0},{2,z6}},zz1,tau3)", "-0.0690530263912008099461712151715069187250928981163864657624811423407258196469612237423488290217623430209257251039917424804425224819251540038150451528670981533441254614697462088698198497710025906287279086565320349477804593670750338432898427667167768326972569141903723348783499650920397923143199103743458682047724582285639053030980472163515213788528273881426642417755041613077879623733290519584150353525880823011071067347346362-0.410558844613197614407490511399342286201701988223563134808260428761053223325229399719569664269480194556721241175042439515146386800048099184034061466310113619526323737905738362476846005326409452587944443308671287110289446508636667427887477490653933188744953374661990286093042992485995800756609805448142966098707460238870114084873621963885267610287433572004428304554718871969745147348970587803080940744572072483741177162222152*I");
	test(454,"Gt({{0,0},{2,z6}},zz2,tau1)", "-0.3127400245030195444751847848216111787923164457962520310801691530100108850213465416288510925450614959275102910551432327233145091779846336488678153393358682421789418703250996547170544530549748444502854263727367017251642429811314472452383585650002444656601213503638493094050102993508317947002084049475984585313466959915822478170125037453959837785131206328443292759679683617207797563952368150266753993972135727931422369423050269-0.2211082693018574222779835023474513065161614779473255173150840348090122121688695794041806217818545250299022130620985117554478093965254102387747011160523597120501205536192831979643628925010504038732415949135517072666801122050618172291406402570264806196596467406033540940232578678521413953811087764903434660903668916960336166734051623014755617817157280179692885783730163489218023097217703972716748082185731932732668666877114587*I");
	test(455,"Gt({{0,0},{2,z6}},zz2,tau2)", "-0.3202360158573475347038415275209363068649535532790636336243525697646818291522137300871928424875728680733702660373809618319295660952513103595526450637416025048996060286738403718906371486403132429552296241830495529335875927212325959047797107375722579011826841605019681487441447865035713252667454024334625910460344557602933185738549219654554150637536346638854571661775132077474899724971230919620651745165435584320963334843671936-0.2703885108474626452971005264070357376628897851393654813173671494899044222862623333346261255256281066670867452071102878370633495596739888248126867360071262699743107096345970878909754247938019436868470803049459724648028843521948689706151885490130944165079172473710842253446758794745536801566351782257336846742459003273842680241357716716050865286881637277004830049844729158804342671014420193174033605635936495509655843314889121*I");
	test(456,"Gt({{0,0},{2,z6}},zz2,tau3)", "-0.2625646432266146933726847491859096539477071648885974691386841955213123960459010582980859286496304276317190321273703544962159204816267198822870185256683299799372500166796078202734004125795056213892612698659870975601658669777284150805676752104847415743622413173871156699460539312358504135768215188538075638096944653714252083906656530984702935844281797123205605541428930084738577478527244514036185316759257811548941173507198651-0.2613916620639385724350700324424213319598485492050256547906194901847998253204760846555603956778475308809331050731128356414393755477953921496122671048240503028869061939827264163879109039730328998445915001455667252715814731076629191631787786636357815411563524226336526092624751779369416304265890196459354916922708715919408029549236655418540124567661972289216722200577490161758157849692473084162659916868734551789159242927691303*I");
	test(457,"Gt({{0,0},{2,z6}},zz4,tau1)", "-0.1956368074882044970702197743076193657931568817785855806270253892398717487148758579564995913876691525367944136995597839746224880827243853226193152648790470271888642606495196293110418501289518459236213953035623286380041729613419681364239689476035100670500818086961106042592067491755636121255470832072832985350404611149445742681943459043153979950069019435309108211697835541772675947773191997017207624795836859247749694594175689+0.0002334829385938610387799649937134487433816407569374368890275949348198607683331844139597729096396784365236932892320134302154478646920186533727472059011521822236029850259990541282578454922474555118736077429498304357945426954088874209796716760922713953198500271855869434078811238651381511031240752472940927244127191308402419029530067753600452394018756690644974086775235834833291345639513784164749929393041058169035772798569194*I");
	test(458,"Gt({{0,0},{2,z6}},zz4,tau2)", "-0.1817741488818097136121464063104632222769144022242188258215071918114041933943663090800908141565646384330009433549232554576149024661245702120889072780359691033936339449119870036755233934015350870139602766787929424748314854501868756023286894214006722430290174865369364360746656307130237641931081370871477627094203901738741214739049669387810886358916830650404869272836210177556961420571954927676650896679701955492850246887820322-0.0129095496976557321455547547761875137551072503757589657219459573212121038410975222188055530533392051742035230111742612128956605931452152420571181806379432316514163806945643410663484760189268423955111708800622107704152251531622528575285876248995683573433246305836217206332719152999001207322522045622752781290555525845644964779795058411528093239685044173930971905850791990455588341380736050711475044931397797885613677476429577*I");
	test(459,"Gt({{0,0},{2,z6}},zz4,tau3)", "-0.1720749847190366635545268797853226662880266544186013748472486596234737402735528272590983441253794673604415507868563559217470269604573497182247635999232440116119289417869070119916735747241646393796029053126928831210852205076983692438343013907637207661375256892442640012238570513770362771232589889149301136118852184966166630376255538449161337449985420096921298075785827949039597762452983865638120819761762845004586633738485476+0.0074889692994124724955496404210001655354429285301575741579070202246820775289527028772757709104769650751328923580393881659238711454889467014579286970831522026576577871318359722627815217107829756743495055094380133211937296834346748914277830341123276742050923524953680712402067372772640765665949598400037748581934018790447013866886365916639691614662607018107052808133645671429907224438755010506390721716836072555681712097399638*I");
	test(460,"Gt({{0,0},{2,z7}},zz1,tau1)", "-0.0781270324435261756386021982626890180564319300785367293496981212067214697387741141591957407147852783842352097553980779129626616466225356650029123401992616913733822817322914475863382226550661687071794456107278116062390371509505500385113881990345199915944161437560023757967471879849487959779527478471325985392573941990322021618525900543098326957915962679050388552020146473667143833192289172544172899597391196855529513713426396-0.4436753400117266621457936002604662524526045508820964874961540472151172421868209117749518874529537201057325425944578699035859526748218093347284923221869617374269577132514808253071399723942696127789388321561334647371838692977920064679326146285892696025089022249025816697154754798032868438521547295346636439598645880619852573332756848157704774346006160075168445590023589223713404436982247537409173843357248666335630689757576971*I");
	test(461,"Gt({{0,0},{2,z7}},zz1,tau2)", "-0.050468031474880641986045925063769607925859557684146444653106105391289164948311036856452175935916735069733337394227381508094432202751393646369887578032252244787073756333629679498945388373669261087116500037962368067397650183759048346426609748649695424152840596172448636931411992896084173680555771153573871764888467971857520798436623443550638562581433787898781878489062094169646783886601254393230590321425729174987922577537336-0.4368337278399705274380963657240319045750213441388269195033635963736457161387030931512710073693509372162378393083446281511030360211943019929958053250065346635715101314539002736900696433549511035783131195336375413488438219795999105112874937035953063360484978727890331286795272637772607385105677354561187310980726437744209479988021364461745248659256905095168901569270458106674006340457715138258424875143992321956219505376412526*I");
	test(462,"Gt({{0,0},{2,z7}},zz1,tau3)", "-0.0680778877250294632125456012850145034840614050259858091491503160538376338411861517222486846984948050912548803896957434994088981847982495090071512213068791075871087431816334457253282644748017155655350104746554305060534031769731232597357783023348084256413120048702028198336630370336065955680773449806932419967605978707956400994547541956986755768221663504905294422500933065507595660774506967851538815027482206730467659094594823-0.4082241490618933221101794307772336193452487889824895651993677454413476813613080806619565800859814680125477421125487770061928351252870478336829907366033983963304957311700165495447421658646982996800924867316901269560307451400724464972358662464052714758881560677997998830064012487616173176142276795953349488257434650797597542161523945665566115873414298407470949457594405842075194672613142716965669250725426092029128834441823506*I");
	test(463,"Gt({{0,0},{2,z7}},zz2,tau1)", "-0.2820070969882731288027513575667436850748167569841381940380061359231248945716381660940668995760646468049044001885838814129619181079054848447505661344202989516692465743502593677748198571202649504880150060983105966943039479255826523235297121311483198770001315424383794239589820558024574369502273208301899736674120180829242481493249749437451365631743115997707471486176296309531999337008484196246444425066468836217085055567707998-0.2477069264464995545885582256343742128526623168418130366112431688560224592528826220714140334641259608194638809852082087728233961392820659004272890904248732940480736236183192191292155452094854343785648707044276171143815453598894076393523814512802564090064660727554466759775195546951546120071422574344738828501663265933941995368256692923975007027511235861378078350898950225217200169437259279343039082759115642382169014869687741*I");
	test(464,"Gt({{0,0},{2,z7}},zz2,tau2)", "-0.2969044972615402929925094578999672238312108945473637731610874907512120509276196288059471469644845247909788296611751529391673940494721899248319509355090176540410442834259387780112780374548795990539238915984002217533839248751751083541912464603226852060369197246977883455398659935449712731860818462107486431721841217824093701236976549586062392665247255477718602644179780571309406321607699509295694201862791026458674316331677458-0.2372704200761076492126874220678245647131101490557065313480917608036152827305766827650776968823695369731232451752790907727596340075086101657074767730462450501417167891673874633672883121872123996434726750173584022280116228463972429159600103516446974339490433123882050161345148959397601997487642397796400892649295220396513898732500430426976214679515340037006798593861740674605728323912572788700382173544301854125959643769884727*I");
	test(465,"Gt({{0,0},{2,z7}},zz2,tau3)", "-0.3031820636547518451620454214654647252547664838076919786686016683809192155703945539537277813394791249549052372732129341000002500881992483440997314456946960073432777829979856696993054723172076630985249754469025663840320363339425182653875363594365379240251242336966105522988568930374952776726184858505671021176555529730956570666173061319256726571059259641425426893387755282392314382876350513189388906157154483725691080585825483-0.2576484571132795943480630300647728408099043851733771278753817807253777498260919919218789771449828676715541104776177775765567259844249554301361520526473217958984191172051169010713621065414120567145446966156676937719295712052375809944193551055850373572167702122786356095635507261789932098128069004777646400706931883915895604577476558512703182196476085977725847085186565423341890046325779955150967623088275746872339159197363389*I");
	test(466,"Gt({{0,0},{2,z7}},zz3,tau1)", "0.2216876387017367176473689844470017868522948503495388046432542102781822463411446034476805325015213733760212657657617451529862919487187182830650423318466663439762753945880439628446476770616449992756082252152788112756302330441467832950035312524151269552534305322523276512755806405670681767990867426200040216973805152367552093391896048520691737691919743276748780735450867342940181490510152495460373580101353019940616337785430581-0.0085174412129769849070818070671208615223880948359051620519750308261282126911285070590049781869629205449588318964114416934809424255596838258298474262380568936188997814155254052766234659291280876045898748020155092400895432078248600996050042535690275064952424064898528324144721168484715676042168033238246218207663983875058049051906935000814432265829323265194869592432999590997477291156571604891504577096134190234287413057511613*I");
	test(467,"Gt({{0,0},{2,z7}},zz3,tau2)", "0.2367120887603929645005098707576309716214042905023434104447251655847753592516410726237333340674244249811831341273275401594989733805018721716265165470596004146409095332578458271778678137707948999498368252685460776070870067450491614968428092598451985803483644178773013270345540243060075459052137472302120295555648402309888471886931538091125893028390044757140567511051058715723516568882200141871616172033976817176239470238458682-0.0051580270852897138040929824964802672503420468837825987344343777896375550875992704984119411349637392438518045285571787209552828054477449348107482468985335670071090632718723632635670519640421362460596944471691600581478067454698260288520569886102485544768555753912330642429684543167303184782581955041982243823273168556257683754906135260904677122693670780313609832487581732597417007016510064581178941721464676124395059924497938*I");
	test(468,"Gt({{0,0},{2,z7}},zz3,tau3)", "0.2278467894870666848933922511311636910624995764325777045185478241681131579438659063974024965197673551899524668905827951899045310055660693445199799897389363052223119079221651157032272652965939218058427248885370401929434527767425655522669450190526801146837198607055960324498040812579116541898754108513670296002370425049742213860348129732762694755601535801550227047420817594034706114512418331191864767808517465709561263364618702+0.010473087511784022244311354710478237606078415287423528611176316954269350031380380487259845934433624954841873136068281964365538921862951519944701243044006070684331201648336826131837774603692286405551583534405314520491787506896950193674031700869098036681794053342305258685217911152195532650448869571228485752456756725543781957453905685693650576759503486495654478669023921353879459271412827300625815556828168850683417925662501*I");
	test(469,"Gt({{1,z1},{0,0}},zz1,tau1)", "0.3774811651444900496072844326514861128888108107393988427009400765226372246168840439774903716246505749149586789649568818018287264972788081856938156398020800205162575428120566822917856832862375761943386155202379451490899355734336805591342646186017708139881912571971001230515267835911890770659964134434907519428597239823093069757377189233236436348470432316768150813216558295254661661984370309655257280222023546056528991095254607-0.0467632936678157101177839857149597377365677869526755886745079183500300960175796346633743181395722615017124821403886057764310867222678910688082284632022574584337818438125806016053760652215904130933418447047284291134198359484081230091172764593768124352910328033216119269180312165884247877925482768852003976737765482014673037605482988883488187231366148470349363099142816802112052626908749078160365673526167621959072169185451433*I");
	test(470,"Gt({{1,z1},{0,0}},zz1,tau2)", "0.4335654102224148405627835090918244168509284518064894437129894581473283989928981315755775582983301206691798036214689947045420610444844567164265359675134645785718966137785710326967459492365018517464994020887117317338487903884058170821681116326858126841762340612007043731953706211143053317553596667762024687778112376438849918058504779893637656521415020850552423903336131985353820825564380651961223192002722366276679413472800956-0.0674926211835362935484401261843613422193143846621368953079287676246387268014525846285917822806945695108655734889302752940701118301630595860736095819615516313932409598475653616019432298154972374006304738820227665869631604842167935145574313531103429573074647463272519054478932960042340099598913162553879088456869850444195969242621647966334207875451182425221333817468132588536840142048803799093200080329135899996517618202269028*I");
	test(471,"Gt({{1,z1},{0,0}},zz1,tau3)", "0.4387473270751784825538425216543802089866633383701927126586117885219053373351732504444659687226452504584661852010550472944110418292708996594525961524744826593599078416822100926555491491001104175221238224173924619927419570601845421131006411797842254970303094855917689712730081382708097614496729050590986167798093872901051482561174809775932330497101421366832608433081244261407717451084756891204781157266061612036923544773352343-0.0041154887815549574804170463577521846134143879818753998162806302066425061319268806383653173178509273075787740099287023943781035901148686078517888804309595684547977817265118633792449093827665965057133524625565993368839736720108872329869613484296417227647255302614749317255879810468789659987738504846706402689002427511845203407725137071148498912798140185091174999641820469974078585295397367218425431505566859604059336030060445*I");
	test(472,"Gt({{1,z1},{0,0}},zz2,tau1)", "0.0720540814369157398218695146370607099927451518527136265535764267905845531775525148924627109353871230125910986915152558690418346183385805313522690925658589201067085043059505952186309673200644022315996452939279846133825446562813690002610451513786841943167037780473317782176241928687939782124866559411343161873186873615798702289661288070127722311930114991078292341140173234255364340088675624378888286195411582635314832410001943-0.3128566906746601087076094335477286952502184537627804471947342842238787256704518221969802765637778961594355271606705331437460332531549260992491590883655833211095575065496179205380697067520543637666137561721367028940497772178343022444995732442314535681927716685830352250899965453681951059541727563043179229659932600559620715473281892060459913061458387183313580814992369163233875788313602315454155577374419005525480413626868359*I");
	test(473,"Gt({{1,z1},{0,0}},zz2,tau2)", "0.0677215888592578763612658983727754911743432957395689982184271140397368879655012134401921523655845123506751779554110139383738628203004673285194343472089158324301418105361353896034025160209077520277470308305311996196142589696151632379466684830056285562285454193052144976826563122292824555779739225764021245693783140899181104196604644529495010107762343023173651766002086818946997896500949917172895591668425295511234603413267267-0.3094477983206239434813924630381042998386395670971079391820734069784779610740601444104480874614966295446458341322388180265407399481463431907164239583730825412355407997449809239092608179641720529945206682201125092793802198984715506471579857399011964501168431593709769780364832775813560478478759744368730878473704277582810097815498995347239513817321749113345233238608219783398013155801030663629271697706737295726151333463704846*I");
	test(474,"Gt({{1,z1},{0,0}},zz2,tau3)", "0.0655024063118111108967759607380965358018688232676779469459409571845651729000044120957534396070205641310212504159820064781612132312045913658591492828625023793378785223798086031124997793756323557410597265418311769112983874059812473731001263689233596438523555260425219561381674420007610078843104012621985784287763123712490651978385817839643311804118621640018634449006779438407341289970988930050267873358830805135995636123293404-0.3155142318833082905344597736600365451682987469810434565935651700447022029945962323728374315504423194892406368365752904508725642087871742618790931157128689493154177210370995129813223248729547883281900470948655990357308195049742233631143419446154986204752793869906563228771541818240984420967043336841065748478734503349268984910602483637003878647257404162812625663498265236297783534726846610133352076074030255303018252349774223*I");
	test(475,"Gt({{1,z1},{0,0}},zz3,tau1)", "-0.7795201679318032360814422500873259461214403113462271262643813405834427874115381567705207666722990438361289843623520276238916734426351109851810457677501241230888808107050888776181200682584766120284272862885494751001931993607658245316855574124480286562422567936835818903682661704154756308252744694351950776066002649319900879021310765150869441868920930800398422106963123149865529033251461069349124742163777834761158931682515293+0.0602561134211708051335643429492275755683548669047543172387995557395424400309675742252821312148898731486815824037977810409709577063767630977711771458177277487410571386738495064309800851630018747591064187055995179207591761047380372983629312681583669180530535090445523173218389613823727847246718654246842335249077665184001137572760485469885337203619184910132777583064228736503479778418984662913099359759827401077934099909678871*I");
	test(476,"Gt({{1,z1},{0,0}},zz3,tau2)", "-0.780372102675262825765039429685221023685570591735369438772271116922730889131675114739173388841577061624402124993504342771490230836312399664811613701406368649118005108706641014450866977113626862911518363962918013103899200280375704395199989904727104360563214907676981680670146834983824329677963822843840565876214885297832120841446566977292464108933839501165149217965969569060521972145827404177782206067814069580007202034966094+0.0604903511607557729475760570459055657788321054821990002606682181117881634750161844545731244362854669837650033406509251712999311350764876528576343552840925338404852085305841578255756944939274330204648169729448536496265760871011104229004922025471494893503565052399043329389894060087687010756687301641968844012655766357496917785062195394415637103406286730534719024512392595715147572689501403290728164949078752594053665616543546*I");
	test(477,"Gt({{1,z1},{0,0}},zz3,tau3)", "-0.7803293154002787961025814718398656199127611833791088972447067949033550100432737066998944340413240950855047160683418211233911298024019414966429375532755451657293103376952872891257966720277681717579378933928519189952278692566166459728143596563863189421015760448984705361200500510702318894171317812883389835507029758847366887993917190912825840053750438973109580115793241286509903083830360011696782687642665184940916362986249537+0.0594457336964159729338953922044828904186064469741748182154650830647544539384478876132593685079367743340497427287430611783140300159905843425401874367986987042739450549057769764957396200744434270110353184702546405903068265557720752353714383894971840643020535065849190324159295529367031215329097492335478237044762476996507785804423421747410026427379352132796243157494198677499276490964140077674484280119473247544306429945192301*I");
	test(478,"Gt({{1,z1},{0,0}},zz4,tau1)", "0.046009150151439589238910718569730295987129630322503663116346734814277743275269550041178857241399948997085573894574809318106226882281516891420372530570838300049964351193859737489839486306048578355110638439401186210643021461968723987045869837507390171628662009671772007625285486371589061342814097091614047377820300759085084415313637013132403507419286892277620580299584278331083150568008914776041582411133092041510331001614191-0.1683034675878991501057236934587555529743742812628885588728943237178427676895582523838940751097806477787817837391934889872637237970822652060267486352243967977973718762085163823939883800071688658466272316718757653974002254590045455101331153304358706581973524768149231021678542354626958832620068926337243768782222414388230201523761502475573328491562370089824256138174249874745034980964401459579925155769802847977755111905200219*I");
	test(479,"Gt({{1,z1},{0,0}},zz4,tau2)", "0.047300759229892551706264104593034850580107196580833539734719423245688638052284445069330692454005921812518615315432703590872755627214023504351198567678089098194996276794761665893040537615448397686877120537379158279271779986390457152147881177021202260830121383984603162424484126490927672030260602162457150448932912760728393126912289970502272272273349509510796090326842822761667705440154190105091814652953843753395712549860123-0.17358288048100458190147056096718467724991514802100049898748140864661360103873738209930020051895383825375807519855327803258003444957629788860867599962436859984139448048333393895288052878715967018913204350316798717778741621221225427552563554717377273259830968680854974242593614188626047514676380674201038394185648550406369688634684884595919697568609997445524020146212764408246215805323424344449994336223884096029970943742421*I");
	test(480,"Gt({{1,z1},{0,0}},zz4,tau3)", "0.0527241691745877504075249491518422611490166653781596387192493605242304177935792368338831407566707006301677732680199187608361877228933032321132180973822739593555613304586161778364924213988188988134698578273850254706538537857473948624478903246753367277103756407756008468484942680651021414525065727250896314018158669825661428251845346728920773036498227842119551039078373650207995771608116700292162343887994163151916992206499228-0.1703365802228998003925981293830557904886933614586994752388020393814913407148634111690072663373754922629445926834089210110284589842713188006299997830025292805141032949765297402808624468644505430418291992419010906933397615667999544418902059005019329765613256311755726218050244714978075970239870786374969039103954726247593394661021678608561746005588623931353918133881519402755904310455560481205569414557195211630981717410857985*I");

	// The following values were calculated up to order 200
	// They are correct to 100-200+ digits
	Digits = 100;
	test(481,"Gt({{1,z1},{2,z2}},zz1,tau1)", "-1.2589056353179175400227678514752135031656581661206602775512350560712707585538630577518425069256088351025038497315970497971396167354370832741963706842022227066593978620762649766226689440930174502485990032647650319779785807267854723773438332710974311182663586387133507486278431017453303576608891813424503421450482449676390372872293969217797879016911843432664085530258653603250220604344459901949563983712056641176466956774651817+0.223986735781265900372796559348739934057599352543118981631042770087214332383001146339330107466977372755809523091779094759885135498652603602536446427868818746674793096115569999595967892068043426672807179859095161895544486739369694461067030597695826202563837197694756153434657992471276849612142652335692336963460035728000755732164846961201500192235468180533023034518823165175110666232557973703118482741714569184908019463557552*I");
	test(482,"Gt({{1,z1},{2,z2}},zz1,tau2)", "-1.5051852941041022015029291931520173946502000184455884974244633460294206515748173668733266497476752844266050773652753946918984720275541578059374479881461981523114543398013433532142873900646088280350665502894706450640395721182195364590923247691567623141504829968806941592168026780559509649581922438562443816896255889085081072008260795169014308763614287293228472433803849374192482596809416360306580553975149351885857623315569728+0.200501912535595593333788369007648354357173421481095744627557000743444710578868882498327033465471301662602733672735482701937966679887243743215254157687186540396014180625350644257847561458899887749232638803423253711238105938643135750317804233294037853461743816654813726514535850007741358039991831072360110827310457739886708833107944259851989978212938014367699971420501936288395313783044563248819253428778336484264840114508275*I");
	test(483,"Gt({{1,z1},{2,z2}},zz1,tau3)", "-1.37532251332709559448385952796356978042079481899815566133278967109239659172571967959757092502039176207835022248211152832997770182987901339409357058050523524749799346535885767854164867445656059789045902206710294779020936156244924661537516041099734434258755681936587435933374664694008320458241975699430949476085745339626039791436833724241738173750104969817248060546198601738617252408410273465090484760415138016436565283566463-0.0378193914331775322038286102114772241781967313082008533608515340134585788051534980287058863665482672828948726101903041786088889586759314658680737414489022834234530292319513166893874838119479370749848683852513644115192399721074182125759350206403821495888922410492296181050647341005853412983269842044313372481091925058375787343600448683248614557760488164079130982752970286425039387326746081985956819547775805889929446495339807*I");
	test(484,"Gt({{1,z1},{2,z2}},zz2,tau1)", "-0.2307326497859305743540106664524538927447728729705671292294070377077556009444683058828200198185331070834568195754712763839124483546634443338870923448578935870112345441292848457449321747774825991354937903556826876188672594406986527589171950115995566397921747923197238815019897242631170749649587683803876578952089554220418491428406549882365624251204186693386616710585751605281104905536305232057581812538616786069622530998348247+1.0413069577513343163097835488127340226618988195281734232926191602974142905089296514068847585966189693177633248736575775305942262056959916420285488669715737985762870159781907272587723802384900783719411183326839618130015782968054593523991545650933775364873172125144422757519883246854481234890129376002226541338086895309399975235577846355307606328978701522425598341126532727835262917393376473826986552134342062733532776986541047*I");
	test(485,"Gt({{1,z1},{2,z2}},zz2,tau2)", "-0.2351613710092800955716260620851556259760912416695816096874490948647821521837176132849203140675156281614818452420442405343539131179085626039100329036095622881112609092178715602674418339042297707937763467362828798982983623455814485076250824005612875036727354106756842988315157929604659692356619187194375200662837640007243603730230173231020691357984289052588284749397437672833654056734078901399346818752611272435200614239817417+1.0229854710407722969582039353567318893392641085491908648178168243849351710208433097825847191733019847851373974865123076224430227646371305342453544901155434597662160061649727025580084455553859770104766018994665287270211387460901817946146304163253790522221872954657063363149757997498578648119815672690818913227716759914248931940998311441254080531582386601010005357004849462191539575577050458920950182239574777737031636437317476*I");
	test(486,"Gt({{1,z1},{2,z2}},zz2,tau3)", "-0.2138806845227611884980138406371926697954611151001666020415421176066687360406366292724855681365542329820230838588448383455576411352937379989832072286937367130534727533054630771867416828791356928317795091288424582350857930646626572422869910937883677587159345627848338879751800893735427340721393685870120284690148598385401216294907336180259615124328235027066295867071572389851817689763263665756069043326912409886328347126126836+1.0246361814273288898809178609856700814585510213360012434003633126534910346014909685971290423436313274947680110452571840252413751255585301713118926579643278772465827588915177366787204910255418516779786849560313531789573665382316394404057570005682474602602017828759001611820959865260397569342008160532233776431195492692554728440681719605365329523470150075124917161157907669660677268687546861506016169692096410386074597578625513*I");
	test(487,"Gt({{1,z1},{2,z2}},zz3,tau1)", "2.6825069980512041610446292741623877199590139591138827428490386108532077708419643604407148384366642861902791486599895236318565577835077357184735173368814851804680824218372991951396986737136349170672268066573619741305061889414738834040495237653656490526191167070481685714962091083440997631480905811989510839884688662225108970809052837009274632559503807435891274736323069103841234169877780912382526315430782082837526833755122537-0.3111949202239205840619179407062379398986463474788837930817510253674493008592070323394569968468546637657267307384751800599589049600999710440803278969906457610174870034160145106408182173619926458889203828721083273494208272910395580024242684078037904027710544705186134898200220265132017361735329004137525689377546818042726053375216981203447727876816260497760244315012704127883429334899986923683513165994209687399411014780991712*I");
	test(488,"Gt({{1,z1},{2,z2}},zz3,tau2)", "2.6692170797199886095036956056545756535545282550539392137089355037735662719170539039729873758712483679176641473975832684632434432165210776516918899664946736572424938562306270920213024157898574069077899141061919463843165532568336977262786206328028356181101829132724314439576382759178110435745207672903676619085447214183317873955118575641665718601565187615760496361061860081606249263506891398322586720546807424675124328691544764-0.0708898429900338103174450742665607137464703396691494361850184052468671178463034365422354166109626902948974552317792552772508349601099568103437418888366693629637950021328681873173447053510879055420101937847832554349117349487756762257796872921675437259821496163220051506788394447880165025841550612280964281279985980496064144435380059124394856362256777586168427863812487562990049644516617383156242763653444731931277804435033386*I");
	test(489,"Gt({{1,z1},{2,z2}},zz3,tau3)", "2.4028881710552143011814795545680182999642909414987061296205902996330812984938442619771043493922563040687064376900187791550770878676590429455334720310513593440437896162227941596996131338730162633467892932595462601700357679948698458919565819171623494272580822692696691948606072635074565287296737654054566510980826054638501120947377203651791603858544743059002277720868320557688491396871227402985105670658090055448314129409950022-0.1722072137327815272229130315778740411710964807890668078372651579948481610912231823731454382066733867963603782882460071821453643652410251145437239584621388667512444477241625504932340894669540758998875283599395517668135900936253509475991914185669870928206398048116019114741057840866818067673016998844170958677231612586832329951824450135983165845205331237526558327413606069927618813830640435746744765956406817264953270603778468*I");
	test(490,"Gt({{1,z1},{2,z2}},zz4,tau1)", "-0.1419572726990846841373104115495610355828243509178565426368147380679276449507824444363122449923386051426245544032819506934958058338002921214967953226156178551872747722766648068631879064339892654707946067307588331452518279771545180410567537237606262978813628468181519157478172833397975975687379679507022156696992542813266438982518158242106219482960704179299503938542422878467756603863737130508219603318693311524699822683463287+0.574477136135301326453128717046976826330228554911736456082763334517696207717682406879881003257168290357594153391462831828950572883825594406503513251281625643958038774336124482268164706509448647434511859012478147632676224239940725048071359006084164121848909824674431293316857079412811086798965725421972621382403968267389870243630890760207588834319775950996880003923739310540203688285948445819962419735543944370109053179578123*I");
	test(491,"Gt({{1,z1},{2,z2}},zz4,tau2)", "-0.1778782248795150846529944846655655326261333639620117802529082802907544272823558350699078131371182923806082439626141631051860441319846125355078071567602530500796015581128207726943519513567293302043259945202433252426704603223601783891648184544866964233910523303345797317230141257670007234801834434305261709843295978520362685361971353375007405585993089246837176115657282058469178503088347963948100440946217557912040842833208515+0.5786262510816107607747186994192976861709988851298559200448459723673309316681778505167465041091424825229390597605581799335127537465531644017795488602630095363163556116641861252245787479332042198811948614290686213631202062543341502055587749119239469600733069959234337897685357286612730020217226913724409578912804140814117575259125008392309445265350265874868798011154859384907192749722029126447572552764143087309079660200177312*I");
	test(492,"Gt({{1,z1},{2,z2}},zz4,tau3)", "-0.1680520086510781501143679049531207244504390918008349635438453511241209763412703237893654609446308709489408816200432015323782795751857889222191818273920921653680357932877352270679402941135370322239299233908734765054365033174871405446086077055646261705470797756149663618458738022907291452309411722547430594508383867907005810719671104802124159395078317319027954045295396905530287141523090509114963071251140032482352713698194647+0.5375498131313511190537352771970275041963479139305826803178957030979364824516893149563947471031393648168823840965408334984002064245236758692080258565100626376448550510280761789263276543646575208627252582262743945021481924438141153565173850017356073019868649269389881833835153931134070735230716655015375784744452664276262434950868475758417223535173157851789920695801426071723677302328484628668974261146737983294388775942586132*I");
	test(493,"Gt({{1,z2},{0,0}},zz1,tau1)", "0.5392533055950888887625163393834491919130077103679650524650091413300748370251958336491154224246458363975871819023250035671492312462831291243115859060589707606661003295398712302029001546904850882183353264370545764694777668326689875267853150008211320675158265488157363260504066491639093057267686270008234760650726118235183429365605200625108364052903346398943108553600107639479151996534709919690999752232785532932640656994369728+0.0547332157112382572291934978429885576801430547445309992391926554606724029139311498425134928246923639124785240610656367490161642333836592894264066904554666116265559235433684695613417631207227298046474353769188801168309807865123702117544084871450516120355726156192732068828519895410343832376539770105858257655337147530494323342515066567640618206918143273327337576204741635062462229456721612644158810546263557710248840967373255*I");
	test(494,"Gt({{1,z2},{0,0}},zz1,tau2)", "0.5513127053133901503958565481637899662795678905039514240844262854100102626946072031057577429441248559663422366135390648215898456643218005130329951059682265423342688151950314603140191492903845902798950753492436266027842071341931172625053083049910946803553275030131798610429323034223580330267572833853749413598280082196567613217073095138581592782849013112044729654668885021364172387623743236484023277490714425138045074411303354+0.0560333911757011334064827838978290057626091751109429776934703678369792685932032100402108299223058828307282192196046588544992810714247792456917416136581563114418021553725186238255724661556886932958513103429924007712770098637402501148226214259550668290466522004803475514565916635719541114407348112579596384891893133860175914682772521131971866799089448449623746122232072649280487820079496311684080920986134617391539933991630711*I");
	test(495,"Gt({{1,z2},{0,0}},zz1,tau3)", "0.5458082788363935910232037461495991219018247883401785793823149120110300091561341438923558894551709784633183866229312947896979803206123170917201415419345786134259920394515295423717131071555417967614027820227579954397426382957642028824469857473480049617583546144185126825529266011076970333272248196570993832735466055907601982054195657290247388754572620584251816592066127864169616770374261105340780345864500797644127643928524609+0.0689250580758612069761170397096454367400848899859531574719587617835975494288147961983877110757524431865141272854399585864891557752067353364714604794707903982392944096298479920206759484349848266098885508327978801617142177164237484847078696915703419582371558365759569870391726721772439924205424163380468928050266362596991414121465621717800227936876489718397131014667703842236097230808398631996662515001536657607406771348029764*I");
	test(496,"Gt({{1,z2},{0,0}},zz2,tau1)", "-0.2071139341339570177689015364846637940464275872836624920848731097766100070794670712081657312521799569999347906037823751918215195516135898889394879723228074243210450830398375145985354298776896926143212004707725614941118469410277559118290296246010919330278385826389623627114016368744151345448140580424169182750705187330530740768844254619327875101860266841759135509972586123531676116740794198974725486373132045520400049361793229-0.0643130515898476448590609608640838732848186788392705054885029010891885816080758334616475256789665975479188647915573558261000062878028036281041831392417772360753214773395007599557518281088188601762686896058184741857458689088639456199253873178647785650814156695891411733967083224196270735034426695093703250875618227181572465835787883615432023729632162542101459601834752363850327559344552659211251682882115376989642725319339424*I");
	test(497,"Gt({{1,z2},{0,0}},zz2,tau2)", "-0.2081515365198552300704199392516305571529345854209739850927338398374163289367072707337923587710321019974881730615813984382106128890898734559373734857542398469602337968336845927992139277197117142689071203770446822773580719842244409614330099880341767111617027244271203280170587037857373339389218016700847817840166040061592569999529555493072087732707352177058748112035868429075377958475980335706026004402476035499839011030320103-0.0614887305130533909220359998459946293075803693511407165305158564616673720127609004494386568493648530212774035440258213318580563866763225215550911175451396907604973281208751935941188473098961496012749056447291064960446874341838737442941456181951844372798349153327462189390252731109374254616978508046526464843257505472326599082056146987896879783307159562857844601566662885506861919315869950275990690365200745278233604121727605*I");
	test(498,"Gt({{1,z2},{0,0}},zz2,tau3)", "-0.210905739887137428497560657164706564143489891515198106255836827259499371238140710459243224302935424063749549296424098712902083723950410499904128114621147688869139713458735570748171204397388835525947074530812343859487567575592048233438861486800638879460848487143596081948441679796140768583343947178544409733914278909879032262778617560202813299727948843018615875187609933830899459886339003113946164417353503754391250809250091-0.0636572160400100402055266809019841489315852510952089334423220200753356230687153427641694023235215022298852259664483578614857709232613443075787953559457731020071195669585776579266392840923797711509280878970467770358361020683482491433528200135970841793001274402117847595664928188369696468716015663459434964460542850520549884174124024049124041885787670996246583444687450922838741145340013328702912271911985952755488510343033773*I");
	test(499,"Gt({{1,z2},{0,0}},zz3,tau1)", "0.155811697590813287680007660204202340461781891978768672109155337715225402384137599422015194121188642656330735761069314823153795950911200035101683755387645543916557147057955394107181471244526818989939657482367690464051215633942032971545882826083121006678591089101294141950053068034237514437664007467488521075546133439722595061616568430877366202292033029362603139528439325548551069682764826200681475725083684467279037420546352-0.3594851290217781148085705443970132673830220605859731241774344993789728606232042121996694911500761964418512920226517262892781429226504811288094536054416058511657390908926735307886965969265580576970050922891667226035046206075243740381360772014759329868947738454930264285891669899857777575722583667173924106317633057433290041564203406043121692121613628428334076880014652117725403050211625202307512352761213963346273541118203715*I");
	test(500,"Gt({{1,z2},{0,0}},zz3,tau2)", "0.160465969417594804846105982063096502411202236318288521539887902096855566710093350606154613933022588273757614905347314308042965896250121321909544842825521623572514418801673651321504187821334897145788025806429491377646373565341687553877955231101271469591637346292067046644981422560201686260090428880476785944626527861929349179338352161032989417635265736696933017890041311803794722701131222211253145858137643532653454544499665-0.3605815987027382466638194312436728711472492247461946920208467920368282728630665055665035568328382782025697351463498993892870572947250765110047735348704820814617631916550781382380596073419234614290051363152999039404335628158071283717672293531421072739037469486238768452523067771091565780407734920560598010547894657107663909680498489342841436523571792084178190953074814573459143885105525946409955063945836545728587264305561045*I");
	test(501,"Gt({{1,z2},{0,0}},zz3,tau3)", "0.1600692270381907538245965563324099077849349087656161798161954342618757755820767818628634098393222631593302939128271884103573747559908180833749274514379848411556720863346113984315770420450826931049630840388137462622877984732413752017554398905967711931768960010833266922975256444052563713204032126929652211632805740655152122519448788818914264937272916922699511493213613904493061799117352245611300917488250295486419918373347352-0.3549165151004153310211606994635703791953204461891083385666347908310462307983458029754108335343050071406774738702568673717656352703642314014107990549849376200096863971782978829946003649843575962980371122275441524380692919375455764805389406629169646196992112312424628563263718423844586450116483872113049344146398719592887175056868659106592578787539306916939704602159313836016353896345284351262406682886736663743309605581875613*I");
	test(502,"Gt({{1,z2},{0,0}},zz4,tau1)", "0.2881598631331404405392942823127540793473003111844598006260219017991419905102815615377279990570437290166522115984561965877392346515943351182211109234371751467082405098593498530459595178426104346836984852686427350539580211597549795466901961326710139772455431381616163748086092214718071989550584838890676262016618877850711482053212176144244835694671331302314589336219151487315697939293252768529646303091065003777960016299863884-0.2088690199234350713913394630732158577474523875164566536338581403880376975017754866324566345006796657722030803006985087205922570328677691208583309682447400908397407501007099954315318765589132287447557000405806444044709252353624739728880800602929225781875120983176957526677772968883624179262051420140074343383965606676326477924763241904880986029028569686201633252845488309181252057237875165548974355109481552355834400937374367*I");
	test(503,"Gt({{1,z2},{0,0}},zz4,tau2)", "0.2886891864997385868641075974439787300116650688908896925550790672264543721498831108345420542076156344493686474485136547242500589536391728751581669449367756574545349054197379812238752049333455425398985715795479715561829126651389434547073824631070921956957451051245430320833944685523149084594873028180206801956687466246143867493382868626187271978634889234515904560393805322960489465702261181549033631308388419744383251207237189-0.2102893070334774931856722048289675274015126068868399368737962741390636208307490518336275665073397649152166318727348292353824709488692616008076929832193605649887949339029025661558149208424446486104544126213572017881134155326044578778646890752934624295531003330938104750262390694510065519166904217107212160701373237819458643230441309426332982139835310090783805020263442338326194177469924985566406274984068967377235118304832033*I");
	test(504,"Gt({{1,z2},{0,0}},zz4,tau3)", "0.2900833620229723623642382451542569458488790152043532368345116606547613730319213785599892053041998060114526327183224664529161361661491064518176510047354782072397276661648684700349269304891157220348421695924748080907146190956759704752967245540609428288668656250861761528382813327011457186146938598350788128436809070863243172361828741036713628897069443587838579883997732275283720234501490028045500230307273684174882663763720893-0.2091893269242658223006373275954598875745362293492631543491650854385137215714975017301323285907666691196747618716515788726953749441751545627510664067975235966426546086760935021306978015005086096374977841626538535513746399147536157134901594992705826280546431449580279259211626968408853913067159615476215177873510383511548828180865986986781567721965080015013925835381651554608986076674346063461794015262230680604711872716629253*I");
	test(505,"Gt({{1,z3},{0,0}},zz1,tau1)", "0.3382073163320025910194574844063586396784349859885052051635846095703485739429414463032399916964904559349665485983820065211694824450825816171835502455673814233737805944455391315179962245842222832362061135134483190935200753769331399390621991839980563725711832621078123360293100624399733881113176341356440976600888479012971754101323981093774112173170477500834202258849146552532003001018093149180763653730835456326612927846142891-0.1593974071479410751885174134583911455580185072695391549948756120586979801375577559759594437621413055391853447120168153293511263286711927784778522716287275383741546909330230954733894694205475521595566683844280539326548533688549593324393788014991175161754655324129074418139381772696336460633927824724320258575704915665339976421984498314103839490939646145428944781617399816461191306741721021715577368304558108558891595735275342*I");
	test(506,"Gt({{1,z3},{0,0}},zz1,tau2)", "0.5020273831615780537393623088692074932256448545686100616038251637059799094393669163763456656531837267987073239000558186404233753644751270304251287270601801211560471861451074997653577697050681363935480968482021535144716010017853405243716613474608204272212105632659274054826714755735179839213268701545175913037949085285355201291107083399911791079659606610201179277315638308876829286362996790449123900118385172355660220345734388-0.1090664814806010809073750005926385962153961345784951416393084404843799032273748496798295337875042157467559898274882963124731007010296734631949150175795487193755314172521023535020353697639725041654236724730045553788048842522777392194677924475785633292798279596608270073692367191740559317000313602354490497548229055523548935386967716376229909063430194093973348723968421655451821989265112216261880947626070800676911562520175993*I");
	test(507,"Gt({{1,z3},{0,0}},zz1,tau3)", "0.442723984363331976066991558403229749115762343099100485893358647732046203973029580534920683837425263493421114103024296931728079243537806442224947378386775156795494855489508459151225195139159207249786589131847467833348138371301706098944179835656768692629302998429472911353382705876253232037579216463729124485269413818461765702669296938619453510437979447592647838324623570568779642364178319046445071363946880017389951363525646+0.0456303539587713392529448426971056008014343794567153645107838426389418767334711037097620430509904564327662743615719163923693142049265326397687538581694056775028841555943812614910678160593207041640532881003752277245442406093788517857130454590698949246419530486816650073763339213519702865842304172849898391016216324217394048577713047866296159568755091672606101201954173683701963966152805798644527613201315162632885959781886652*I");
	test(508,"Gt({{1,z3},{0,0}},zz2,tau1)", "0.1828437779084486930471029274011950165971761136656728242658969024024549105051977444681521345388185621856187370314727967757020914357745873272262073168629778966934264115897668207224474713853995916234026286703909863927009685582682374104497784806955733102886877715231945541576464661048432968119342803506177700307496726910988870625174038595692789919291059244888657169641004515470176956980296443683360808678752747934806301869016495-0.2629809928299108419502245534206792030622635448340098503708440015245976732133239831640334856971873695908050383613136426666507189378634283003115758732572571960289974704950679176869571143008575761578239796678771121208060841407442442551523329676354148590420344730421758817414226246636557589822014170409771645172715318707630460608917600874174005674000391750852429181907355441423754414152875352003704669726790971573275118661897512*I");
	test(509,"Gt({{1,z3},{0,0}},zz2,tau2)", "0.1683993423459439935349678932847063773453798487019774098070765924008743334941230247491104731324254882579827132821377698677888453539196755861223441456754419104541968933495855714227266634907597495668863369001724629649717303030495040996031598154316671551281037882565347128702937368134022534034010285165174627375245022543290457323105627356431147313106338640096333511864577040179599058801938562977565875718981241973082302901628664-0.2608452309758929864859422442604444993017440747216903027719362702668887913633541886605803084214905286500180892187548502785451206019573834917937945026179612021786977223015914442891805251229902241561645622991818867443103928962595962196237738096484664936093186684024041743553803677674247339354549059686582031109700698251930427054910514832599688180151042422935782651479949078294989154301824301260127558522765314200599933130700184*I");
	test(510,"Gt({{1,z3},{0,0}},zz2,tau3)", "0.1714224084700405830746945801777665723228366450912755862305876656274580094321072544539087864318760079119585964078703668422657711230020146026239266827334227446987082918104634401256757234505838913314965556676087507316905268741661608196775682061889294900721183084842105178483437760786016050422143579338589079386085198488646102016372334254210328985474080031856733656920164708091862002305235991966048141262845479278273454724987474-0.2776838617368412059226527069595025594865398766004608727076239955448198509371037524177818053874388018401062887115079654966642173479090803795360857135990010657749223446420999169787539070864450695879240202708872949343646283754556506878332778346098782092973687208944608074052980357616098148413534033737437914340938796600967797513491790808060415918751896568694363351397185497724971632093997251679743312382304445412161745211737995*I");
	test(511,"Gt({{1,z3},{0,0}},zz3,tau1)", "0.1109574608409060226047544471163288174269915230635631968686415007759811561871676827597612072343883550711402171521846273707299316919749489530067815162123286178225514212717264164577820149558220217953542225280896600750433878275608188335808189539954189162520144642975200853105065563658223591920585859759413276687650611653813293329027034886372821760604957435499262654556921085669176458766745632313243767558446334172247078294275485+0.3873749449778692639885451895223413617830416328390584504840591008863601790688747080458078561606386983166015775219268925223242154107647913965103074097672940365987511933917604547822246134909926813707459932502274881339536768279126798278362398181003942926887061670908186226273096646380633321710591631214176581485834790362573288058348008084230181924536809245085292919306310345602406141241957710898153256526341189202334055563098923*I");
	test(512,"Gt({{1,z3},{0,0}},zz3,tau2)", "0.1067978631133306280156125193550823047003179215168035597536774361843078279403777546793429504695972978810930819611675937209198236706423438287121667763086963099981699040070721698529806623780138634402444400441903653119618997286806680414993413140282780035713448169870271105124940638560669536565459738128961351828126533903783789413503995174133551556687907521852924426517151927985844219193338958557296930224144429493028737331192468+0.3845101880943184336774721389951701559991775159081677870568830060202541069553744446482249366838350049369534329607698086407595571719597184773057581613196262256511053088939434799691522323537282395148306640575230065546845819699984310696790695302633640251278746468762416466661487488150005238205788132343756248844040868782122599862887487570007953311352574743532458461684976986801288099033866348176955178577081800618390206204836931*I");
	test(513,"Gt({{1,z3},{0,0}},zz3,tau3)", "0.1115949461919517383947039706197362949286195644532324738665732310732551053479241947381599380639892710303518361723499196589979956151597963991560912760979684097870996133020475093972147838316307353960899519694962301633144087027794775495578915707339463183493627642206609886985066097069044918958317898236434308160368322742356630382439301642591012473690954421994393721467695964813543257917134740582803075046681295373130675778853521+0.3808290715679258138938314458588809660016201871638687582123390120115745982334912538385724724367958035589329216948759604557203194025711112326055892132105251246705074508549149572354356844114556043176958199683704699150515744735805679433336915521751413924528047200548411521928454829035100188643319039059927055401946650226873294990954983093951711200277848892974068964204930189690788914297734029323278051080243401230580609426514838*I");
	test(514,"Gt({{1,z3},{0,0}},zz4,tau1)", "0.0049446958636611344747750474889289328815205337495278324189030287129652246193741521268859933712508313363763529413923754689940208098844655613782208226773800181523463966268516479176610726072465476289536000009554918592741016536384568438839332680829383275728866149530096360533955760965006860196045058811519533516195606218107785269213610744243773128093319979633161279467454774014191999095297634634110355738764637781606910955818028-0.1816914899042827318018514049281729929498298373865635018892515532563045080859229847091051449398507383714627421145517458701482290729083987829384257047098238487885903354488695582718548331158227798990064224681324726133907398887275222505690741928914153157678331760458250570601732379001976861067171327160205238955763704207677114395538549321249367992292724824598976009303000924080271649901090186159989516814860488663571348378030963*I");
	test(515,"Gt({{1,z3},{0,0}},zz4,tau2)", "0.0152818215303945559069103314815761398285961379140138316896655759260749692788017717845216894652368877437825085854267912495882145199916728430152592659376953151219164651522709066391144844502970661561212537407497575804971001730427765455346469784279116341325057175661236292503983875511744667503956018284223035411577181019705349128898983682747577930002303923073090916436563803061711924610612985308086483797192519936695585063230496-0.1927912301736017741315698423835474242685070769906415762430639814765107985644587516072618660339148580102422882298387593636226581453340059494457282724867925670343909755760793226700244492602888384250250795284676292812106668015117447822533670814788635793554003822652443345087743443809659742971778995241561181262248391149291330895871487406135099347573964570269715224059890531672710255342748917285473795046389965125679191988988737*I");
	test(516,"Gt({{1,z3},{0,0}},zz4,tau3)", "0.0246260957182780681026477260108508233911390744797307270597667664089344608393348488268556195408536391288224228978330372254768578063002372113017044233543699981389283834680924454627930776742499762763071627280490340548528252942694429347765223579912455551341280007999748728933516606883224170390805091863597897661023603711716493894394710181308121083523217931061871939732795860376964843913554263370061553139456214487742016535946956-0.1776053887119981942299692203884303803472721578176703488587945116890381374794177453566020091678937144342265980462235580597623535277769354618620054433888616061876726464839341629448741351885888696911439539357379212050280518478149502013415319672952306755948222116925685833550401217282612989049254972432612103662690740292698689140270936195362829682819905875853478670443608066794190975685952469992722194379282770842086198969890992*I");
	test(517,"Gt({{1,z4},{0,0}},zz1,tau1)", "-0.4297786637795660068721740643045426631089279384578186132405620647978365527906766648538105135893286216688116279652731875421132582649794015882858093629635328625471595805964548345200812779665728564706854680594934440632014189224618793308064372362247708333334875780165971611369601686647184404136967868122167807522828540815757903141893849070424770518848724929754825624662797233242754377763334521973432041907233511093097898951361804+0.4689941398276563206631357733170497619619411108899272933807998768233401475119474559434972143632543587587947995731684006467370620913812479944247720909474563450827466146763585206882816905740940140778727906763492584623245748945799962695698717914021911032034695975391308718003358894732382982876047815986835483425241715107999585036175459480970020653260677370452851426134351627868906409262880147689080605577065397109282746830529379*I");
	test(518,"Gt({{1,z4},{0,0}},zz1,tau2)", "-0.4275992721431165819779222903261340010860959856325752031240026253818901323670864236190401102359230818648608502493703064822800023190219531814472835056185917416998039589787337839663821439083848102915391794944138183297478309963907057397882311910870603625548031219960335046308537223956299763224646744398871322221972285003298780003574193720280692651232315611736149215657398271658881403380321602938561081416733294745788204411896832+0.4657400618881538170448688892226642631369934757983178458700751587995392075130257890142987878306230009630739137270640670684161082450532669391369806729449723522317661010364891453375369145063156558762464658988983819528371301892454538115734346407019304920389208488528089916424330438350098560571561504104896156106833588545681537553857533581030921559130903042333275700501082803430518878112435464137782150630395568528957706357451054*I");
	test(519,"Gt({{1,z4},{0,0}},zz1,tau3)", "-0.4247316903376991380093737008978404239992341131473891109562313699542417070729208184075392435640853698825199448533386639956615721259700636398828870619694827860292920014865438106278937370452478512075698424868614654694833456171591203160186100501301517873345194955434588902175057821287308486518601057671674110981794310186792217315818030231386226844650536244774883169339928710798182610698017655724132717645040760811371583131973076+0.4693316143577042605927697784724181506221319123909762762648356405693901449354909549496509115112092423057736461024826096881865768734194549680387466000559287761213885927899155564970421100491528570160584378921011985277964808787192290203580373414096674484209848412421650431868902853764325505229673489835629948555987246880402395217145094656753945092831840368351402768821311804815610543859722464948061254796433840962237075514641903*I");
	test(520,"Gt({{1,z4},{0,0}},zz2,tau1)", "-0.2240924599983783157547373409645239730325136861589729773060932802226207356100018407143837679889715424294469835903852436952410560148306135859009030827313087467940563661648713465407196893453579458778561899390013307533043661456470761271269115616502977122882769238302093841058177089891784607478610801567429324915588696115924191020094784639272634026842870920328696079135533755675169730534059920502084203676333102988591066281077907+0.228988873077041890761008061020871181801087527937824761516968691609455713699849696073632592319284067850689276899903282371777750849032533273688159313879662594611728368684319331234246877402184636409907524743593834507996483243143455837841644761512978016922227135449485441843844949733052448729279819585758316143172071523470911039622214511732926291490890095783870459939157312893478770856595834712020081007439501280615106245395679*I");
	test(521,"Gt({{1,z4},{0,0}},zz2,tau2)", "-0.2062062944791033278205700497072255535586044719150384998349943072131656513325042904753491884738276763871988612178726461181075335363753199711434183594556961490938692718944512987329217486764573035717181007017334976776584976001809004439842937842492349936755665499860184576977266327727802409262442833422069969422292611836479497232350708681199102429700619524861193986088759907214807311258752481648069172882489090558641991754091062+0.2518601463265995591649359722612969948223716845448826070999960295753426045166876573220960667859978420491301551708217579852816891106478849821280969546405038648752328003071746427109494176122527264366755411700230597021804405210344004516278407446167028699790969390624903930407272009685599692413792391317958453729313232883768787125955629569232012264378980274605626792915934295062452012127204412924941879218336520048052164076810239*I");
	test(522,"Gt({{1,z4},{0,0}},zz2,tau3)", "-0.2374378916612763731959211890963276008467669046217345700256086246873554943426282895597047268851126774535484319962645771768074141844291735922780191497570473299849223942694385800501612990232032937944838492765632190281945988362100803750137541103776852166756888222475542941653599902361358246585000991999964506524412229650135198189004810562299059191253748096882149595727697138452051350305727419353108138039144040974786194917286117+0.2622094882429912892407553766897953995156376251528550810241646424217398798333505676452455997606352051312647804014108362212764119595182182177993789187848007551930177332748556177829492944988482136868183669784708001439101767995445113277420913176380334999249917312940342564126306727689492095921318866995555559099824630154969605422842213430841209467092387754608265142801681106138610318446464358396555138582281255526753432088452356*I");
	test(523,"Gt({{1,z4},{0,0}},zz3,tau1)", "-0.0239811692384138912766702391903446969303311398550194319483631674328565845160911822307056087953497140266906006284731404801514247440547626321932606017546409548965220396227803037211868121700862213203468197553353132309298289366343032333945737298851483540970717667430496537682367490320371688960060245689240552035026668105852682741002498492805649046749824340213139576450000841032309095764228903949852027176895213708728761651422452-0.2237043460006459143930824113314835932267498425440152009340510943540904181791143164634674425205769119588465515490494004088218973400238768845609147947120302639039108574678050795233763880780797765406485065622280644424590372266331405052730294497898107223339972735650363688139027603241026298929259707832702668512255807639789330802297678115246286978493500518352678449219406122530204469569389396677130675313959107046269361057588691*I");
	test(524,"Gt({{1,z4},{0,0}},zz3,tau2)", "0.0119180334647353621562025173032728122596747969423053465552861213731289383984968381208852089649760658719638956204935156881747093840106391258172428768415612478103767402530325081599322862311725190144768459207312271871949329941407292015332639314649825036258510630435760617287124802552972890803043220551112676066904174225378276036601952727096393374059992685716891560228318570013337004258422793255993347836733453359529866694566534-0.2564087591234560445676709614324254609340960344073747829786074380793735073062551876436050871912135142564878549219454620944174720183213309672016194473126884151123725886957501619390614168652749047607795963966396614916717621396147918796951762809658546283194384728078241863034416204521024065063746507982335955788883792841808614739113337103381306384785853401395162107630874382876366282289242798761169930268433572652078107569102071*I");
	test(525,"Gt({{1,z4},{0,0}},zz3,tau3)", "0.046225683005908274415191063803345631216303314102828741888332568461305482437791801240815841648890764159529267371020411002672753306694862646262001223893875142687572536323478833046423889370108101273786363598504381551638846263529932044811060901969153471191333024526263982036002077621136288941423932572718938199160333538817795009720485075918263520189259790147276058581173239000756315530492245792464005190257235563186939922328952-0.2081002526765784900227144000890835536585677881539982659076457514261461301884005970278116720163033223602547787635306388863341433632522347004565552724187821688865573618103085340029597404529271736203706124266883157520704754553413468561358310085310555152793656171076623572024082066568318137988582657884037576647863114996656045728486694426833397971118585620033534553175643790775645386835017387197638359887720347535091004901625496*I");
	test(526,"Gt({{1,z4},{0,0}},zz4,tau1)", "0.0170922577389684762058916197763110723275413830001577992508973320480604398627931594684660975498251096713118062679480914636690466898221872072772984542128500477880050561227104441084681843100834979422459034388514882907684657663875820968864446752508116715652633331566604800268572883381245719253513162783209740429394636059673817640288195917803522017673105262573445326464576474162802997842188434603157430487370037556570095945638469+0.2166778651310259585709427395662350437259208987539916902322446302230624848699169490806146848715231539383023010442682021922986392095393367326784641001227486902423420131163290322831718720121542227508751342799386852098179206222625780783839773407181146201749659791664252094168021836441699962135169122154198745180468618244025995309256758318199488267738414704561252404381521459673019538826256062123392588461951050001735555815542361*I");
	test(527,"Gt({{1,z4},{0,0}},zz4,tau2)", "0.0113873415172761115053507820343762187786409796412080431507559548088932897443099499112497480720796392983235734584389018070209911935399266219225465941818220331552944766130889058606265658907943059010984373139181970926527990258918437193989149571632619413712673480918377583530651485662829236884561504296876140131690413301088722069055190030352262747658305748652491472897820915221459255158513279176893068639637543885830210676692297+0.2130084579081587596505665243883125184810920999579177243685130965937089677510103255700805200579465032445655964571199815749477932534720231721357968277661372104065358174281916081547962728677432812279252271820381669704742902689655533326175417169605570558659864532609023740861389775297507372331639264998303111453769708670699521963059760547899559460276170505220641402060226172987596356447544340128197708444803310266868830921558654*I");
	test(528,"Gt({{1,z4},{0,0}},zz4,tau3)", "0.017659486585560476939448113814060541487259582687332987007273912472486436101658006214890821264098026759602367309808300453437686800260710664889542208090724209465676592735166433677670519295496604921240539891743219086608343733213209824866373935849559493327559362975307814979369277044623840438861127482724234987891582727572089675736158069299892502911706036676663669318276795431331384310371977769587939250000023302138576524856963+0.2078998431176508105659331156161314919045042307990989876436295850739121031844929261219376500902383538477146481231246437623230126435517383426652882379050212338677480799228252230204149952000601481166327547718796885187547477950884567771895247307825136751345548777637777451068084241814566763757123086130689832791215632727717891049605161274424478396366048326508319114269675036048757306324137932314105203760883818320587872590417523*I");
	test(529,"Gt({{1,z5},{0,0}},zz1,tau1)", "-0.0426442707833656554586774557864024524443675058396199233284567249246704895931100348467750825819561806479233640114812221175397156472548891139740167451163640740280602435539068133193482722781054465053234158506554911878478353499194951874645692563879687729275798753667549213414216868380957692539493457464043540265935787015650134475097176169523589201946626733300411144457592888210595337455555118367851719932462908348182363407141199-0.1275841814621329560434703530293201963567914216187779000007309377061393931898874504657056891069774686351648513607211032515632712711408518113138903604798149211586388015388723440883012612582926457086980109439881694956221014210143836639761455226113377635681326953694523074146968296296006387449450962755308887216766227992583148052565793111097331227524237104456332370847103439278068498273217123745807661805294712929654926681694673*I");
	test(530,"Gt({{1,z5},{0,0}},zz1,tau2)", "-0.0448280384422785075508511313429190276148131526968192706877845270263345307803368149501015500017784025063170111348014825231887735771959068319707965584292369821377801789386994683058017040412711047517281777728322682745850049596426176664639977056624876067598082809447563130148452484132726088598143007338303066349482322887637774176027444841479286148385007641805639400038233222170207682258221665900016663770894554966295651228704601-0.1265068199736981168440716528683238838968847978538918703635429347037808958961634378527309573309889024831332885051618917605546825289391563862812340116241555113216297350090602165998493275309938605165426984558321841363181772686932385254598631469723892416743337032462325227343500219706717110858274873080693370533294759165424024329764456967739368789133289177353399527346967139326089464074741672640503998530390824558343369580059012*I");
	test(531,"Gt({{1,z5},{0,0}},zz1,tau3)", "-0.0452258371267403357827871288005841734307847078421713829351507194183708440385851935548182722777891947951026333186108830584866577812581469241388718673489923568300893152494788618300534634725176165293923327096448867715489005384164705235186978515446611162923568128210454502924716841722723596315387568899735606232983120588140573165521435436194253557915948898414160639710316070727419570316724878180475300946419693292177818662227923-0.1293198100216392512398387318527912962440528757977727907088722959396537467276402616380299440385500325535478945689291477835184395650971058413431560006326853151673205437710250062451076181605205809275000748116483125959599590156303006822221002875165040623097117904114910559100951817427771085957916068000801225732308918072335338532889497666447058266067625229652805555286926168448388628930266251716678682970894045024133601584263303*I");
	test(532,"Gt({{1,z5},{0,0}},zz2,tau1)", "-0.2592709356833695987900853945224975315989192668042168379225969153592880986575917869332747293205209771180613895937224875646231631314417194542117089551102018720286127446125412852596666790789215596317281183987785329084360895046590396013299077577983488739239683261358497441714539158326757499695163430217778997122857978994540311860184656203215318282121992417684972645642990042611997070697585002511178122659384106118931103603045699+0.3505642128146284606648732657925057403728705351442362710649709284282359406459141035878042481942335608493646052298598877809384039679044203845109873879341288087448020683007240175092460340900312435064354761384393846872478679676551018934589504517715152431983515079272973805243999152265590247318289304430655847890974633100249669006702428142151260878405407501075200303432451273441204101411536290870126402409872130662863642245418844*I");
	test(533,"Gt({{1,z5},{0,0}},zz2,tau2)", "-0.269124842748513571013311916050716896493741787765032715398953266093118335242722828111500916745522313730190356464129849978819625869299303825114829219763311728016824081576580082594815550105828615902246650934888503156477237649834323095528363758838042495425104951541837730692718270888093392127902978975342269860204172058573516078001564227538981359272316840757767073967022094660604136012937526386962002176207310443760491624307707+0.3375861725145825223756376831089328958566929159005807342030927423663454446966215222782429163716645379879286576011619301134478969319375749764427908622402186034665531572608176310980300131297115612627530168099500584110430513272447444825359073569112194821983297045466015717770994567802637141884515009249359913690108487026791064202216798211692870031149920862187175587021183888231832523741122691388869680156760913652634077115565463*I");
	test(534,"Gt({{1,z5},{0,0}},zz2,tau3)", "-0.2509138889808425752571000148421965871375849717969837935296405222585871634879110135545787461169927826351416414184321036295341540374468633541761244412627351940675908948846689021387757165800723895609796660695985168859407972280388773515616379968994642939337079450657612535146826781269136426522825248583896975266298300195760521371377007395078102882557652920191371187554118111285025001884835602927106586068620516476419026212197494+0.3308458889670081296636623150046246780921807285827345565669055071524677420444499030004240773666975080966395312131563254971485238767268703303532252988245337997944155367090321297565809175753831227528115683513985197919166077085359755688782696134962342716682123085986166239350824191890014317056685139249176648827432446608579087405659158483315608125512156101863392056509214335577665622766506727549720515374451433204702931814676779*I");
	test(535,"Gt({{1,z5},{0,0}},zz3,tau1)", "0.002752206981740706710539733378288895753035031309742152956876521246988912003148792887915047552629175620151039076836468552134192084773069950136976130778616356292891634616883611391966382572347898787690826939102040932914105990371547893852632078727437583760158526375312861260465811093194975039509624973285215894652423840034481714663920100116618238753670609541267451023834347778879438271076664327799399326797661131622261570288774-0.2090537429200715647181370152318735885042081457781170491054457720974069228304087172240475743211492612403825466947926844694988940005883170223496591486697959086396575804912457045681575725159680152625165522331614185210966368461086450038998802338122773311087519380284369143752821491813801193102427944676679719996012936825947544992597533429701287575860764465166321942880433595587856231618841582703442070992480358810557788917052535*I");
	test(536,"Gt({{1,z5},{0,0}},zz3,tau2)", "-0.0179540034216173599087820658708723104509550251773801042617556782131855802409161559824001713277779947528854284110861992633979927351211013631578176351555774545088107688415561422552444604432274541775547110208790026217750458132985918986689922937771204096454366513317263504213204993226184546996845025684728759060681033331285843471787015783006229205590377358325945985254973693897508167657476006794593952222611334878141961236004078-0.1829987024871929659816883378955032501212273904589010006736342131585811927507536368142968871631901642872675326419082174715776216797724390705951048240053126937187576639062047954465843596417655892062347716071970298890471410987421091422902250793560311500774476477280712078762264235661155314213725382618802077123311949798467609470435922847483017988680587553722241568808747021447074518058905159410485319405962193471633411455651386*I");
	test(537,"Gt({{1,z5},{0,0}},zz3,tau3)", "-0.0354019240945778066249082195592405753669015529010084588357482713716402236367313556195858410073489492622837205654000434275891745629113354742383494389592498393385441665819262325887241885609591241152195375625448664643172021773560672081580111680248337620489632347832942296668602566181084641434074473886658228573409540272013029306655314152497285447571785793211174771800174304514487439023207052083486644757121922578566521593351344-0.2167862307199811488817782073426099628173210221129783072863403339675038980165183950666629306663158227414141710604605849991971717651351775445356803042971185203638115184274760147764449738391893700740413859254175162538051872433329042053845046617090368750617993091928666032686783443855416560888822914017402124014643471108168040089546239152408124672130700596424426379783972449342525690817396608459026852149114425090226705043263062*I");
	test(538,"Gt({{1,z5},{0,0}},zz4,tau1)", "-0.0176540071005407713799989130582366422891617052060556846410674144438314132620135699244629384457943709704551302893827844958775043216932494358019328498565017916548724728247598739114485126763340766215922163492006924751379167942013826590787936634670732320845319273205045227192814665840235371403845561308044320338591887644044453299331920167482618926742756824684702847509597327065519116577702262706990274496688458925959682402668862+0.118027863641738239062918945195336932366953662163801263025777756068488384989889348854879615646302026177665726443053046337585002494359031065487128057350860627266210128421178951598059754204336447291866573450708499689956922454631995220822762682584855027890982794400612862907445190294121464825177194881195751016209815659577618612399381035373540450172609991137488267002431860607402278552274986488035196183825799787531005367958162*I");
	test(539,"Gt({{1,z5},{0,0}},zz4,tau2)", "-0.0143582401905275099720343058991354277320565968937822411890586018645994652798497175082626074420400715673254610240521424393840525011605096607254066995837294246834725044853853477573210243726286387217715389247584666460670152670328462656687902228191775068175912951431098065709509968287835621567487427606590682463006785614234595866768104514335705426844799798390391632830066086744533489083066802851908969300156067795254090435240675+0.1203042642809485682082002576183274952808821092423367954982184639018000259413670678888719616939887993323489499337544589642067815509679798012873053759581047043747282190728064609122382904044578907875830648422369897702902845790527034402998717833475165017993355141041586894010308693777055585371443382962959679806242939505141198576123204916743221532982042713316592790318840699058502919814331349364035817437062679249048134208079518*I");
	test(540,"Gt({{1,z5},{0,0}},zz4,tau3)", "-0.0180296734527843104790465994571481887787304069640609954697219650843736478624043831446987545564308163893412735524664305845917084514727015650130852901658018197323879051434613583812378445071862030884466009800482520016444167033996391610162018880498137997099455414056827288952868054770411495731945330066627412956924048087843300096108612960149304821596763974128645147447508107349465669821209545190295573821214809666195279577017751+0.1231219291528844369008299845002490397726402667344794747229043374613897958893703024255801187933616265199814267078486892368081961449154367433823672098113556720151202076439883206962697476982391854399691815945751992634780001959520569475082904586183632210084983525904155411233286132069454608444118588612701765310297535789484105546103307974496308084325819524539916330881529347652007101341514915841783637391829065979005187176753742*I");
	test(541,"Gt({{1,z6},{0,0}},zz1,tau1)", "0.0034932566867018308760613708020280758503486703470132457403772286059295793660574906936555296566700867725398985296809776092748192309950579814150497268584167144108404247249455407312599667977356563166762632642155531498339005495539300418362160558392583773277608962945919035990602299531648130419172858148399012098609205461129813786433367271909718957220552998734010260221805785217422855121141310506341598655753065297009243334545828+0.3576773443102698724513934402788316372116420337878004552297357551428317835928893335475813919504967520834807078055042726541041624896192230233715810709844019752422759048691456979914656193024954540106972395983090664393769756773670891678011422632719194155984781083640519012390705560463429667451805223059336157675587777986050809477686227138194729549884847627167322582002275314411173774047987409952606077230149584734254760214859244*I");
	test(542,"Gt({{1,z6},{0,0}},zz1,tau2)", "0.007841888489027403446918048735337937567539451324505763422588414418173088379155953008943829923880977294223141197258598853220513221753529663620386620441953799449552399305651936750417581141495256394281949696733573006499040203538185388709480171475548546588253586585345183270072433904405407072215673179688968862715485479752595128626158215040813469082678680135407078937613312018077151521562968585257512444269535527658982269933297+0.3566113517047945855824258851548380300502252895838699985659962590278082111490573818969712044924709622983993319273512988880727547023074325212274255091857295533424381374640963656891169069070933823710677926714953061348333178101784951868326019742301285751328366614934908546453814911552117707371930885982632599189993499955431870639571549228789706255810402601082837618262182139353164495232413678124030994300833684301551591949311181*I");
	test(543,"Gt({{1,z6},{0,0}},zz1,tau3)", "0.0074681017488024540747007916133772848160173170826943588009594991915144823989963742607291343605646113598206340149333676665365681308607866790251042209245625377186281699248289013255070361246499847637793601559308534495763099695594115025679000820283537363043924480857903042824662984191058206718671413966104818433240058663429135879155282511141363199437961776083527363170351746102647503250901749237942700599314539705567518460648087+0.3618268116104394201515508898597135626984345119872755120999113071049074191996964486115547507860798732506752244027099001020201730549471453080811947678143434201158583737815615547334058224362452427178481078523703667818288168014494420181829281542614604470102760790426366695428136244742031456789422678398777496391006839358398118774410110589964395483325270375832253824913604744732048134587958357561418203975976033883116396678278597*I");
	test(544,"Gt({{1,z6},{0,0}},zz2,tau1)", "-0.280646635908106541681249890796017733457239694285952815081704370317994194722304631378713226308636469120981445730955186982891575364796052322587358204648715347182986479489746750078722211242988992958145024017569560543641305759741472922050960687208838873974582988858396784020201059533452977264307551568738838645194061506696182914463997584382910206565434241767641505666174573268457582329122415065026261275540859175761905678781729+0.2097814620810317657195307297366448628326885605869328344809337949229964292522293806563840214311092501437850692978779909030424502345986591604967954693707236049990051664215262972914794537911901936557788516838935947819205114161064820680022710180212688450731917714817250231435127882785060512577728409470285232857249030340127136964090439992363871089448794267616803089434907246256599813321655964953559137953215303626927259658765637*I");
	test(545,"Gt({{1,z6},{0,0}},zz2,tau2)", "-0.2856517939010519660759595537864132238302577180922423459824593327468348576644292488695058446199033737754749337224047952459515218555852509517408098585013006427287041974951802991035932187300892528691466393165024513131316266368179925988045388672233413681664927545704572061920524707355475766199473573986697337431867525499553931348690987933512259820037066006781387005064863204796484419790701943653742386339114297165638565494805078+0.2212612812571387925452011559081670577968002268972833655367079293328173198329415310509160941298941139955378434305169119709119341901606949237562968963109436876642395536145440384717252235927342555091158673250962463407142887031939773623920011612499538265077564911038277015469023947129521299252701513136762364195282933053848083205400481596246533866710223648272649395329098864559674579536394530017689107900199833226122698754209946*I");
	test(546,"Gt({{1,z6},{0,0}},zz2,tau3)", "-0.296471079631628457723985266282156945772062191816214482364590168225088397325735163651824460421868832928850515281196298539740952551702472628366251006127675126439430275750464766939722262789030026136862569558484701566141830989557912059698168160896812070658512752964427837953713135938170171934533355409962645334500555701598262080525074671984547435016404927427054316029307934700596202973887823298150479197188808772866946066014132+0.2117864234188387528069251553521190662309118195742318198315562352802908012973747516972734740748671990918778769218164801112391725213464174786378969561382344009844661300071651143474968661179240615581947869427577905812625846346438491318305512076667887953547796083756294767049215110587139763173178032075780004062107548769037230757545005711102902427974556408284501161264014720737665845589456392585821688965825304587072821055846267*I");
	test(547,"Gt({{1,z6},{0,0}},zz4,tau1)", "0.0868569037504988578052653753346307586375756263164456025326338456883316896162408442968978218608093187725823748325278794748466104157129407714953153415692105320165485178660557217481554018882323937463220775583817613596458242092828167364691243539581566206095813215234672335854291458636684685857335539905808281396378652024729049572155710868373226231362625350882357741911242174655236652126101162922787017495829778242727236387856692+0.1739845027196446172944644464332845230598332591136627675334648243587102367103513272818576641645918343053514626803370810613455285311492127402482325873854673952572126787839543650096000031558918572549101137588860785535453625799098896416962389655994904342716322540352914942307867382092826428754655936860970230571541750537863269823024606441180079623591793765375111116373597498663006638048900051064824779728303407056347332590301393*I");
	test(548,"Gt({{1,z6},{0,0}},zz4,tau2)", "0.087066669355393361839496389627529888900988313393887864218950287029107780493603516209585216605518376749437597275572833520223738249671686190602498258056692804986826475175453674220515767910000914633192081291764473945953895589133555078535099565324117026280469840950537838595076639093378088090668454436693061615733826002209240035794997305095471150125356920469572957066491120579329540951735350468372274091721826096451148294998776+0.17075612005245146218430744161585143739817516020544953781123215172990518829115424461064463230601119612835869324917060061529457389159868088352015542967247015904841441635535379894097339072700150809787475069297447940176972616596400208369249169735401362371779178399373047540223346297781314724537791421423965304956857375795278583992888278159539445783748055908756125102018843951372694189158365317102024601533913499684442852969069*I");
	test(549,"Gt({{1,z6},{0,0}},zz4,tau3)", "0.0905834464744749059583276104035257586403406250838293453154541705639244783903010514343597925483491233446134836993473054744306376641490291790567777734666094006311752176840279167027440776116834353378047974674472188892921174316586865801972057921355636863543504024084590780136388455404017883857782314917350749941648648783861162673368034418041526344803408258400838032811421376628541266606206476750709690881715425533147910029260834+0.1721817393317297020411540534799969739665280082094429085010157148391991680029217019646898109269421954395455667635275965554978572019835668885058949908112612754074738217725348084621472897718411123235589762743720972504249116362672297768403177405006092266321857411054254145959192751262053187528908577566852578045829036813514300515483230391516163116303578493314264973732615609015444534766061006121492350220524677048477109813811137*I");
	test(550,"Gt({{1,z7},{0,0}},zz1,tau1)", "0.5769008441495810714560832644336390569829892337884635043429740608789139489559036537647911248388038199779475314709536043848518938911172677361366866841129149695448311463868116645953919501639026691144165263055817777977917577458890557456261997379805609908522972054538192473248561687610263736397722286931194378153685940786812496630084805949314032328348404162091842889618087095481999007826702615451872736235385330122329307239531023-0.1801760320803343687187203093344288098315777715273299950737380283179445925858925834276779269587741927086902922879000933257608066898187407482293850008390442192592083614492584518103454401573474067348416767863146038325110070492096546580540879335835710195927892865472505937620265603574930878154742169729376003466428383659222905775428779087324283414060107452332251517709266611647316684340400443517818740095077363321810812220220547*I");
	test(551,"Gt({{1,z7},{0,0}},zz1,tau2)", "0.5791974244597586989159834583560211375602582938532702190926949470883853415503100788418787129495246917601471184342369021513014248401201932544734360485176651118307789441768844143898886768231663216609905595948433317532132360569386465847919463166362977920577017006761639030412577083801135783679197591068871207626521946117523317179458855762647446318379493216364526422402858628590877194132824382039567332493260319202154658650882636-0.1718986924921678529436360497164320770986708152770603509280392315824310004614012407379574314934171519819512578414757136262316657314346092260918851740240712706299205715023343401955033345461730305287714042451822487369070894439670515982962745522946591337533180181602878308385828879921454199735741704166895608147840820427697356886027727058082798959143804642715618130982551602544362733593245846935040134914910510084945067531124326*I");
	test(552,"Gt({{1,z7},{0,0}},zz1,tau3)", "0.5690630619559442566544458627197266977449488332296657813542128785509110359562970804314996406768952801255330226810507138116792305072098000151046380714841689461527541799385211629491514535526385148702734435112511296340833089111094260179191211338378525579864448200919249474161387528863616410802781184638866942445574891248448156683613107302320046241059357469600508883689282779052969351409877985997021247601502016720744597806208813-0.172179303181754128199100454694204814965620187361753104850007274982612381978846799505706642288068582738758695967298989497316240298304622647336344964523368872642401738854126265686217243261382403989734781136632214118845981584791088688255359120499061136646591750338849385396295279813390334551219174672199643623492237143628425891535742130696883679712114492913299460197575102671859752345353387315462175708959265179920670835406672*I");
	test(553,"Gt({{1,z7},{0,0}},zz2,tau1)", "-0.0564921398942211654260563877651564077327150161394329218794854896367381275404400606613542096900462922818782304364165310362457447117563237816283123652835181232122912311396586855426517025701373240906191783521514932278222603805767421138701037804665742520796043475801236663101208736161006465589947200746469332132084057449224285799364870843450693719105882570450329226624099298513373880343468449423457179575894186281608635133880987+0.1344613888821201164530115294205835974022333501903683088792160730847827021691633196017164159583351589653575201206196206912055069739193124736823501659717961005829523870355333867178450495119410028695429212875008175179199958148626149947188434190645120994152522162727720180712182067521978911799012329661477830661141935488890646342731642142904503661225111665185196982620217953751375469203287251532307347398510591928554722240316726*I");
	test(554,"Gt({{1,z7},{0,0}},zz2,tau2)", "-0.0536332132623420966483557527540023891148626811557964708826159164668343623124215659746576783531987096188958848780629034009361646315623788081251633402609377398432827783759667695031065876221047558920507204610496047354423009746516828763556949415166693534911252286068367696404481428839647075536618357815376975355480819935027724831545733151643912958743685190747775398126026466025299919983697309823590107353386204106661452973729707+0.1350665033853085307309995580858323695886729029511850633204061420533449236384107663225051506616043680425371026627373492150148208220638767759980584517065006841028227397025073242288124111651378110301512322285258037655873553989563700355127909300609889648348626394940224304189112495341663163776551103912878984564569450553158558300525725918935047795769383095968481051974897238610242515342363049963786727207674834982088480776379398*I");
	test(555,"Gt({{1,z7},{0,0}},zz2,tau3)", "-0.0553485374563240452806762172305514098496578869483060642784654260697151147487671317617556660443692676768602335559136638213174665203421190125044121648081562386000175302413202179487922978980557183284990729443715400416063761995445089775424169522304364984951484560392587359057202928420308189544827545908377291002096041040235428092126264484669884509904032587121256637886529728298849326080334432232194369100481616928589995957178222+0.1380111209379311229803475022426711001001568146091997734313418480792331579499766894723342896158150722079554689890051463503023530382660759832246333676425663648637768429941013500520611788982589175128114686393338762663316218175423617231350596200655666178691332547483606710886916154633245016068036929985258730750290783946528986869405957124966528986867580540416572167844138024961040573278913811736856888086232321144677005847734351*I");
	test(556,"Gt({{1,z7},{0,0}},zz3,tau1)", "0.090842754026906005755605348565656860686301095997746425270968691723033287621572381670742083123909842765365708260530804241802830282899203471921983333083375776507013239097249336941649012656339079982007792628337787210610133854345547488404880178008744930735980879307525344713473320157763395229675281551790804589806995418904801508302628325018855139143547474574082643937571727528127412915962471563883643241379815586561474136401684-0.1841713667817626614796477953245937972257782447934051582867732440804656150337014005772844887627479788208049806668610685477891945151012114720332255887749015193002628011610381754879648001999914285270861453620293050632426534094087506054882628236078074338406873644462142562302409621791745252974565827439233470590096888521422351600791734876342176795757322336756496533823527614153891767793501023975357501282143936853508259359070735*I");
	test(557,"Gt({{1,z7},{0,0}},zz3,tau2)", "0.0923619583158182630403075267311795492536317834381753158760757775820454017853787502597799935996746598613164003953574736547202944352783987383991264321198840836246676125953604156726585954226990333726584816900546417721629139976933059918572475370513256595082532791127692407787540726692151248151545504963961076576240486190619292761845946421875542658461307664972252828935178085979861233677981853819116369773368849258497275329180601-0.1908251564250868925217689665504739465978931034287642495797681923072245891021352641114090421491514288368252816071331096371588001280956243631697404489858606771559373315565146471010457083910058884445162317297811689328094901209502705738535126263535361000204420727590159173564774386683585646659110960359055662270196515794394787454335481006501540385775514758352387680529177545723983723375232016173508755054135265293028449569951119*I");
	test(558,"Gt({{1,z7},{0,0}},zz3,tau3)", "0.0992552745564454253682335040363833963979931555253730908612565037128194496767986498514149576918695897728240466056289090802794477456702814706138094850139413971648716002804090136737047241398339216182353842044224007341471251887212557671014848755076715471480200573699713240658370759133650181746735819012463259926277686290699445287167185318078790774027276960374804111301050257673299670186115290076520743125607761431155124532210803-0.186914969055218033144592007478607934094505681261772929593255698212446531371050549094504252265524735028124288018285254642068437765983394672110597270301090682480694118560290110009951754380674032854758517193892077079274087427622978683684663333152601111547664912063845487320817005492338320047731002953121958953946374575477266743876450736873771600815076759505008310879915077625514974644800842642588759947164359367549192826457624*I");
	test(559,"Gt({{2,z1},{0,0}},zz1,tau1)", "-0.1495865243488017911462947481588573305655616415543192032038024185749145937904604069915139939163867911848303956088864433539412001596819562535379054696172342596786050193103986618101698873516165505637313722014449318479514938126790238608281034219018980982820152195560995998146647374566233232406782512562400348747822851955409869375383392294147149784777027507296359944881144623909685370436526452191565422529094800952828338402031726-0.1861069319759506681966450572723466397589108898779307560015179240498846260065987293881223527796807862313328159430395335867929861680598670626836155499978036438958423198008041348974635618953514268520858846611475754540982273154493655953050231492184308950626653977237136148578085042578591120567836388085969724175656832158673013840033035278276997024945459960332739992893129818580731985481121292430668400205572307998786103704081334*I");
	test(560,"Gt({{2,z1},{0,0}},zz1,tau2)", "-0.2795893347103627130135789927450072880211147107901304296989056036447665619929076373861969349427271872074263697982251027224680582391763345699229374782094847254052700471185111812379013784287022032299064464289584009366955197931165128612517574190431248289773946172385090197694581816964911042710369465486915764645784540607521622967304066317398613448317643015868538763060945923619984488021115754975306537406968722608024256604199737-0.5365482546217389335447267720432810651839890728779796257428248680277923871916810115125367993344913238726972371982091490523495811282433923424417370554021320662585769148447042557358333870807159502514431894858175827213691609751170697182002881116146775174489753876024235258358852844479803486849743140108338084252636794497992391265933730543260466999019312216053478482169852287723494952975859980198043833690100946794849997663806968*I");
	test(561,"Gt({{2,z1},{0,0}},zz1,tau3)", "0.117733536679783594485952816115115763808912330836712519597429643999079791127870188798179203079614623279883112144589031905822326610192930989857550784207386686827149174855406737497079484985444339384904267372855587172609277760804690426481226671116327917937563863198332423910729202615437587651399242226729740557698919179448560085019431737416511074614784871232557382059847219121985539041925882732596258347586150208511909615309204-0.5701360770252367860751374938384107731779763390843793912255109829698251727756536421108888178851201488846834282091135684933483713690193771341913109067677374953443005143538315165094286573692254369335249870008313675333296714839764178460395229426860503449675946245305886685329924676907402455099766351523042756567284493961029418757680479706198867158023335134087561758454313151893206347727644326794779680785726646808911936153372988*I");
	test(562,"Gt({{2,z1},{0,0}},zz2,tau1)", "-0.2911107831314436402391447951518955657850011095975319245436209828688674699228214939612552864137624393357216740160030636910195748665579488897904185305347801606579097361884359944623189046788660160475465310377963644846030003039991165120501573001792195248158733316128869060687693596031102095215611104344237094017636638139105421398022411403282793815792906054225196280382065404462179981093356868022195456306821726783394041269902773-0.2679154684031165538034907904250763839244459145280832461580543959568026023222687322864481871517976449589858496261276615530024556080297452474314929448905823463719706889335918535821169688083628050260471120530973911403095347224069207043025322887951824177136182812929620468413836485292058439838452313948810277481002816839774805153016723957658484304784938891506370394311632269600088614791433930719331589211862138553953410775023711*I");
	test(563,"Gt({{2,z1},{0,0}},zz2,tau2)", "-0.2757936575412987827510955067164976816475821707308235633206627879998808276523481410881526914356870957664697163458769101202322138231905667900684069930775640449926577136554085640475647142149759196966487483886883171473842448390726564740418630072317459909008535440888493556361786663407749138039569399564768856455854886013490005123137246438963913195420028620272088374498472192367777050666811235659580347012746150746576644995912679-0.2443285092313777473254520967689365905722626254202669230973503862679437898871915963680064380153829024941245301885635465145620827521275629074945176326728279953995754341389931357840333188385943766368508699754728049305045361448167878140575866197665156160737730202799340809252548196159262096921085250197256967736385767620210980480319794629582137037878831179088241929696896367095990134212827761679291674810094811714395863279401503*I");
	test(564,"Gt({{2,z1},{0,0}},zz2,tau3)", "-0.3077746127446294105454823146343934421603931599065178478173454293475735782604494307875187699848949159717044687876306633317153211012128471175791464343217610285361467369715035123717510842035564117267041548996303353244575309505614106672995938122813011220832652803120953915153285412514478440931914719287900052157822831590858556325497638864065824750140339619772976177349268379083374582934957866634818866157216280792679782593778617-0.2357981504999848427192159580365081203202685184600398915206276474909163103381201461879175598714945916114138201913807627322043105959268644655306588760595005719081215706440753618912196534694461772388397393871535032223128054228548484606573383097461458007220459874527305867827903728963226521958765312712440772248448123574625176711420450486957120461137214473148361857703076693974537032919869300721892311849465582691373291438096768*I");
	test(565,"Gt({{2,z1},{0,0}},zz3,tau1)", "0.2373332008198236947899448066564987753193131218781060406119749948693134669014838984451376048272540432910357989287248134686209036836349529821164591042237784221632913839343138952448318820846824273577854737149882080835104710157828402218703441582897351136092034705713379984925257644098183400785463528485639885674623221713353234574248017180226326454868742694062223967745079168807773928267597080787508228304501475155497690482973039-0.0074773400910170444426993541458196335213319392978465104376171710107035312382280408497367654194857458232279740411067746144596777332017591582798407608274790387694821337040226360089918393039255007446058843733987735098872525947033033618060298466925323958922152072971494974020965848338625958482443206907968253757146301046843366935963470936071139305011972417688935145224775126163743103631355115320356194500715033342212484232858392*I");
	test(566,"Gt({{2,z1},{0,0}},zz3,tau2)", "0.2343056524172186990974141846048905103960429480148332164732902649734737160014134685211058076337152377412192838055714901591656481611274784013639083513063969847859887069182402483690083850627079619637373167902699765727447290106966331303864865863620554490399592171260587879818860025397057071479006266057056984641914520431233214030750371934463633750915879488202788261623750513346869181944351540585697107989014493981903328580307917+0.0102571301729522302395384402199611085463834942413583956540756542757993627402096970920021821629923730540447256570358203135376682158349487186503238952701226442440520354245876701890906219527088235477422053126568800791630067956287496254607644683120198631756681686740227800573006155525924464933936319652800915080537733854603862992023247441294498781775934850527022357293102513495338585349008237207246222714963151069474097328279321*I");
	test(567,"Gt({{2,z1},{0,0}},zz3,tau3)", "0.2154620645929803843186057091287563220789913906700995460944611133667095408229007930272436931770630478969976193785009940524888581968846006077950635685750641932234071161227408486212722490160393395570696906758287068597231160419696062988454645146777185734956944594210888663323989473659126214127491234535451549503878419303267338772416881855965166044245073414817623000896551258893225600694617921673444490824073762809046388765270936+0.0004836489567401367835555173941163201676780160419442214333913413189250191139083197086028950654894506497032188522754218716033327491573885030540426966803097567503065578719120711132088314486695387436771406125569012512645877750335239747657057800778435794737950768593588845744609167785000752816086638552717811546022774089831274028451209745956126417094599280271362352331059886324940904123546368596827276638897515020398443820308023*I");
	test(568,"Gt({{2,z1},{0,0}},zz4,tau1)", "-0.1704341758108979652425555635022496400186802371565167764176109349578332677030099845195393094881980809032160304411484166946326489379336836945201328341729053699382206597129640807967410941573397333999426102717130405094710920235366908934419000653383904869548266072387692486919340278879754827052654846633581680542669795560915258773077855447284403543590198407422768555842204219732038629233543314598827413244074888161585443112281006+0.0181555888099393321857437392700940163784733655904939343074830137062161068110481445975386172277734670145650213288392031293093526112754480377083912383850869917979201518707398919634900017700637029807616704774141796398066354778131144996204121798771835526510312773684219293970356610466277462921245415930944004571759793355758346166353768945453057129106511860084607867606031130807503111895761122849445354758029638323639274194421345*I");
	test(569,"Gt({{2,z1},{0,0}},zz4,tau2)", "-0.2018334005955011912092709304496156347990533512927328678572967241712661705481278808561480174185302948395376086230068446897330792443761780318233687611286517459881376510774596955270095632488386741594739623343104950047020802700098025228209300101038184508975532350319865104733484140638131272798277702189989095104640483923385263823075080047997074354422430870437063025286702284146678458099246794785209228967986429643657990996725142+0.0103958605056112563125518793124014550404676283170187994258241745610070469382882858919445915671199410844021494822333683738462364101425290722554282632147125163082666043576041327573814803038472482828479592252667306461355287764966530439551309312744611157595770442973478324678502694236331781924787142072875421442140282915538446177448455206921017680175681819950027777731250536309706037619121273137537903513043542184728992708686683*I");
	test(570,"Gt({{2,z1},{0,0}},zz4,tau3)", "-0.1824399600306845901715611128411263196491560788727272614195589819706468281482896062501145135098979321344359444173978814993632223133859160259208017291443593988226426595048821696261535193931293383017000947281282687273483369950159424163402918699597838993609748939863084075743524285367176924554427114039367907056915269123276649770022404525558157617585784769868957758559484679990886654940763020461164114778020462501836634863418286-0.0219684005244938411375667415177064842215507334627281465193835979364482137744170369003838019163852135491358721674095710317715973004921870736400614213394991624494316969849234680488755181561038164440234987403712141905936898719721032340397010315572674499959627613947885958057407124999181126067896973518008777528110846968792474102792540168191604371813448016103239398233786490381106390121983688295107606332721951927455375233020003*I");
	test(571,"Gt({{2,z2},{0,0}},zz1,tau1)", "-0.1051860793518729820378061185318602620138533462908229654564773442020072576575931334878451503737641249681018337032374031941186793282639382699979041412364114253874509056745343149699373726400050395448231096772234251027289175889595989007817707120085945144482695691596303582663136768529549958767822829189232133508165737147853409980662632243110549853388353604589436445221018623472923186952049641899541812692157543607521412184088457-0.3964892112632951993684238347957746932200863302333817616773990815219402346216118887561704740249887164533750129348106860355166521274732930477769375837491941776444545380692027466544045798688884314186282697360410327221229971952014813454838104881553638637306205346222702835629265821770520236305945993182153249892427746341647902878844834517990660846681596390629220703449731424317518329397923926128902043753289702758205970331430746*I");
	test(572,"Gt({{2,z2},{0,0}},zz1,tau2)", "-0.0942129785406864330574500115466600559006862734812284129763420641081223341636324057864457956143798308052709257978283968010046464710592682590769857544502257096841913489427481165532676889044545711580334329302555521385328918610291532011437879299731195562544130037578548707829541594742874911991605787296650010845430740898186719968907154733106197243664342687844092326362822293078434087218694299692565351810394384548896411964822965-0.4677706581157164602760893980073216670614732542557281565068229110469620140194412623346822815261588758805741111196829534663382854527233430821778972287053795374968214352082321255569251138044357520438654677037651963596690917796107813459253250780433730938165179928368824479553029624423984973638646355906892989679980649530600626391788807246541260128154521297987676002298921336152975256784441316082270658911207137528498294831224791*I");
	test(573,"Gt({{2,z2},{0,0}},zz1,tau3)", "-0.0189232721129490089478089277334554920627498778942293559896071197860868258731825878027094940274701319618792717442690832954189024689310341309654670086743709457149782377075557759473776697293802811414265074068878347437816857018288804966600758777387313237602428912033093435042522284713576824241355114225728232537462830248599065504862669364210200371350626624910393296152483226944018632451131049545762880994057342021286724552186095-0.4316400627671690217629353996243101324085933536769723779273032452028770853505675160370928723497576129421499328710147467124273433671799917527343883802340999793530023070995954312205250057954306415024835347079080026260733550324704612619595514339239960022739356174676579624519409617340973488060348672606093414835698567099195893737461831057060484323800010059073975386693719261980118481726468122755417265735515973275385535881211688*I");
	test(574,"Gt({{2,z2},{0,0}},zz2,tau1)", "-0.2830556875918374502993794477916376268545060989559542831260774510343958563999466340294936616560602800237313812470647010911757372514367441198564693128516637598726548548347988106138044139619366476612212847819976616954492584533481955132273291392127623574712086746261236557797304962570503889863202937592418630791638561922021600849081352961994293874259743552986920572216844609769229875155274285433809601590712373278946814156147621-0.2449829126776381522401795540275380418424825969525273096015865800880498056377919380138649141719716352352123683306108452143932154926217673433403860042559239500426133868793854527616559931337355360378189306448688687426963003826726392819708401999773307935909957040100368283085926028099145943350352628362017514573846405334270585605250180513510915351379312556970401948918477360587171489914861697068377555414189472077805005622428549*I");
	test(575,"Gt({{2,z2},{0,0}},zz2,tau2)", "-0.2995167990766302761301249005267693286485419187624194805604378599999417772003419269819036186136789491370447133407009607197096800826572660982288017606911065587723542962520218840149543695093151928905109166219238517726132338980674573653030949142420571813026163585343938922756299157773407057799663261413928391626022517659523081822585025277641741277456042042019268658040692775961701460409092104781870783599040963386601586366459199-0.2386158110883336930189670279178214907624952077173580953038370201285803021274486337510114815799112213821396559756347285537469307941412183154691072038516774093005924092392513217991860252115275905421941131217698399839764473164118090985666210930747885676990406348370866422042570113674350974152453137718504975634196936641557230237596952741762614744340627727211783473078921106510005995101696433133861583964886304658917452374569968*I");
	test(576,"Gt({{2,z2},{0,0}},zz2,tau3)", "-0.3007160591458664777745277703703622501032246839166699189998129001038692311587977879694485050213534390026553909189791556623059740949514323080342276490985421705219060820719228050740009905759045036154385960642252625272802372449928628161489147730761076981564990447730726085178909929216005324045143577852336878578747020810526321942982796394169656430451481679257277451864907736946974765590850540869865835690401821795545058926567373-0.2592134124157034011939085140281341700335213682565954436820567676823927386948031237934578695512450714016572507970196613968835075851931107298601565013390717859719986095204623725691964557381577559946820654835132345600846749412530019350510291625969804510698119008676739748112592338206772226230109255140503579100733645190210094666576977634123448535800111223467137999166188550661530947142874400640649300420785805348151352603782988*I");
	test(577,"Gt({{2,z2},{0,0}},zz3,tau1)", "0.2408508571722473870812631921401210913143640951593212898342102232824879809302581097626242902627977011136234502249868930551015740258484484374223906406685377671692874978980057782924854974625951851455112020018376029442720356592262557121409344649679179057363417876979782726541500361443756419821768308037701153710309258142775024358157576600111701086829283973776487488060313970270351705217949711448208758639336767769066123130626146-0.0185239278644597006171404612576780397398387050109993205060862981879230172496465531779349995366245017519795461953758736718055138665768269955616130553600530507774470655444370235589538289825684054893084464713976404265404204999452690129261586291149182476682032574058844964834534717361587509517381176353762571258015014730151915195443782045549217030819635271289334554806794265641888424385655619791333299411492168553978391902606808*I");
	test(578,"Gt({{2,z2},{0,0}},zz3,tau2)", "0.2450545213191857427717508478142893311671420600634605686897376866332613681949177436834095007584289044405191502126270223527827948680441932427643225101401084802325054942737202362446871181552448929904059929376368771075686345624206094542312740184562412126527755675403585229799403514901738060779093682761686244058749580553260386776463794352475839461229641661214176381550128614199239544804104797798423597722748541733437098205114535+0.0146121839172518603302029612244781448238647009913200069841168423617289261201042521981294190349560144129186267248795797036794659754283456211201890265939272280388351735253016872654246058001580606890041803294200104143911389271633639468580748913238078298230432608545042120093827429159502701796029181804638741544028469145145987493337903485864381762475970443226707807887783614851100147398280311848869700305928140539339553711734381*I");
	test(579,"Gt({{2,z2},{0,0}},zz3,tau3)", "0.2059520646512604329952056119674374451864498994536914946754034057924919430960188123705888494733063403264835585267182272496570382025211103808468051286376196783185489597738391193847247949051464504354329533303323359061536776208869966698421827841363825808168993129699358349037733343451881633004943059104753516035191346948151997564771925669779188700443685829039535572293035832067414351949895677559206128236456042191810061127658461+0.0077405362675544864349516311502655694079557493715280019783940274595937897061223070478190166141509722499080853509281888259626445299235980213590447626785914367995563430557074500756610196563679493534120311201607641441506630144461440385280730809995408903472851876163824759092235287013340290448373799747424738524002518618308521071540483732023158343779326368173048020641158316311565745481737711102002001472051709989410209031013187*I");
	test(580,"Gt({{2,z2},{0,0}},zz4,tau1)", "-0.1850228961456157191842428857654602609484546298588574868759822230067871528169416691361182573606143381809025392794133972284315178206644822600906798779974065959263699189664615039558704847549751786057502410934363622512543335077158074252327172255694254930080158303235046440382357945973371941089639069263063184585624784452084714904481359360704268663152714532468146444839307386234783946427483596994283944150245358255430825722455418+0.0035282069145324085216155800723569512921090353922858566763299223137655780647531810232473770759861107747078772283777900852801470775736611181990269148481927841396629959669099595528623502064722426092958950479121403425649838508280534977813090794526747525889608261211901351858295830502444789506664150914331211389942361630956698719496118947372041912696716434095889186392578361219936497988028854478234932860210313595010497381752968*I");
	test(581,"Gt({{2,z2},{0,0}},zz4,tau2)", "-0.1860521280747976380710001847495028049700306201314347932569553197549640639500585030018892694313534936565595617933345536817439811815587990274155963992643036178783142154297018451494539078468418516294054235334775429859479720367680613708673170465894710169127190474173747340369218133854496272800436170485213955764076871218631936173731310866573011412485725037512228894334118324412406784677104784654725398841633073061681742817808318-0.0025909641547993333768948658675886572717241524748294878346398485403312320714217906353655850404859459879255673560347365672995390186289591811180885804245893935856108198690905142083864041204394440051606168334100824753843224786520219862837363968037412394609196981641731395273295295843930073150554699449441506452457697815070740706130854114100839542292397623343375776166714249614318932887547024034351635082366851549979275450398721*I");
	test(582,"Gt({{2,z2},{0,0}},zz4,tau3)", "-0.1788637049575348020916018834504920680977572669285707953763443385768567795422859309619906048599751753883463691466553188081081112474823114744462100057011732652121933614892875600763441391541926949447900394999108473628517781531367784099755670339091581018973132225234366211909267293844709038771610076557993079263123617852697007057314458472378196245642979192751010949320679490446383119808711328888351328671309689190812157760101314-0.0015534057359941953506156980372545215866498886285730485798169978895437271078261643833840676356938899211924768213826751433320561751098812842954226602027681917045910599894810274770409969543674432950076839790970947317779257241539130749307736631245603255197078862227614521254536179606862196531647326345506303273778719445132383448002830862293828650634280687219299572042196340004593896480591184616637674510055439348834632644363326*I");
	test(583,"Gt({{2,z3},{0,0}},zz1,tau1)", "-0.7127614212092057893624501531647513476518450098104860560070853707438891539427964184029339462908738060864969689007220662469322100293853509291257398474157604556331527398653297364696453402804487896204233845489686981044020461626502729719380374122693246985933836093512627613608943905872363021610223483259775428160459669044801088148772288884861677563811867290890986502959109613615462892560184627640765465883721836914512463222389884+0.0583051787272560036795590839656689233664890622903193636773757446694365824416581372058849162798061828970111749269662784449860354678714879460119789851702462765023330336944422359090822192485367082358244452399948676865314486325361131758671331364757469946729941861121021460409157981113123436805377559772434217283824950366907973092399910951517343879442250935870200453734661582507693468670800441456934498260349842754925801897175614*I");
	test(584,"Gt({{2,z3},{0,0}},zz1,tau2)", "-0.3970857741801223439677427575084123079790878853121609464677188077780786094665836824674665303752502623698723955408249563938736634039443182712192130424648794723749142840852169721051046251050802094004161012772460621900104220032372373731371047038161271940002624562706016387911030772365140550643033631035021527050483154418898336667282413593191698673314476907604220353847253483735442461159756852342549970379112341280917888462168586-0.968194283087805818820153839221162223207188302303966211231119236818697050579539350694283756450832643173908327773859334100776104331845142318065551714722425554194129473386026459504747081388460373387798324096455307084165662479977138410012384929760676243829617928210105008942278260380425783724831912108355779739020585796282385744533185249144949292707370275821414818688280077093175033972954200934259103246389780659320696962596891*I");
	test(585,"Gt({{2,z3},{0,0}},zz1,tau3)", "0.5763024939658934551383432050937114462195278406075878573469470417928139121132959902050228576467863028454790299265585655379815239417882257379584555004787372561937016899781392266276059942650548548452438156535012884974257473666496140045721632323706414062702364371926032332279971517382915272685743051149334810191036072561751236521049379960581212918537719935509652483635408625872918081621218661424062277379538825455946451085199007-0.5968536599279667026291573625497978497510288019010551673541703163963676594799697641076307778956604272057962078695018076239273422165506369234231699825442055751805533845168575386094503340231743733046965907132549480866452015956437119384219835954218282831089071248502614790604415716513313859560777216827658991514568538655682762663950899438509615757849649356909851423763153155726400359440338585868926899733286547319682028983644491*I");
	test(586,"Gt({{2,z3},{0,0}},zz2,tau1)", "-0.2588103199776818307545719751482492780709082893395367247891400140229132867581355312759036503716281387785771450018920906197166816696554789856941503967326074955662068594355370591515763516005206456739625914830343014806670149755798518214808900141212437157895713384607695700553948616683118108472317503616065410339052886641149766221467608831317166048334579680372105521632115467926209382794432045232958209991900416012134452883472985-0.2990316658438511145621411910613904631782608598802706060693257829053680911918908557574161186829008521896377592265814086138938853940638403931026778246114409361435402955858506641171894166172538683718904993308710931520994609848299981378530608001784145733895344385877256116139406578998022093582666755543853328387395887398572016555394638853316368873230366014263657826496961753774642463801709507604246753638453401191851658107285977*I");
	test(587,"Gt({{2,z3},{0,0}},zz2,tau2)", "-0.2477910055063665247462761765416783159792015296013747545801278031231876295079050883144529332101155776273053716942633578365158722098161845504023608283466911886833877179819862659675246240496540525362433293844680051322590658417948909050634213032223911405969844898245795533473659219247148704544282896384847365355560963429658470031546870136015511659828706833174468154023318473839547314699443334140624438833785065959547594382878304-0.2085831636670193736249749169879118202644255599034245894540004368728811655258504604454087283265081854949664723939365605669861321795123477126960235902564758138066365563028840705363125041684094774352555523425665937412648454390724738780409672062350212847985554634993766112598819393894883920474153599857395254158275958085046802894354113520357896762556505457451096210748701032466758953986650272118911631033103429690585089874444711*I");
	test(588,"Gt({{2,z3},{0,0}},zz2,tau3)", "-0.3527600152027600743611348177020919827588784556109205496345635954194787189243620864991806611348275155321541023930657457159848052719737761746849040480138855723742676855710518460868913539767136967869803751964031973588806849330589148795286754019092509790992394052857721358085992518381437643867085458429725285530824975477094098458315631411905374146708607632732431723748057041039183738336820180216015088901838343939303348478634074-0.2300930465569517253194216436109822339799356559073529736484007773498731177415728270011799979092714649839055005434740475106758489069761794074668319942721113452385724223495234886999236652073814858484949164952994298197285739175678175403917311312517804069920944751501116896580137422938681683172696203577372359694374708544661651691371197549622420547370272579284945510145181906188942554813419857069887521397563058935805748026824*I");
	test(589,"Gt({{2,z3},{0,0}},zz3,tau1)", "0.2503384454841248956992330691846207784572200305157825148944838119463734736145709515867704145652419117251787097309408490653975242254735494484259929982081629554355969462069944852280790557667502943523116638274077726311357065160898905220607928620233863226372764508193084800498927281858496226670316914231872891991577103367813245390474290913376630155352356185569761703844033974120899634850805040172560269800984006652061054732003485-0.007213326054247005553140228406992420896161652115191849129697531454132443906118327520775148101401406217116837517238973146201286943429517346470513175633536043264040411549817531321506126888281680515721760750545324327681111078445236359122519734878315902573409478424930967008301068181208035619271780930694338090095302981746234398415805552202831840796596227616519701174459029959806070620970577546764203593647826914875182555814022*I");
	test(590,"Gt({{2,z3},{0,0}},zz3,tau2)", "0.2330658977145999943560924208981572364224591128153115596405310252555349866905776046283419869897874162466099717668912145999359147740382736940697286793851483383692831778171604250901925793262490085315029461493931370003973244254858477973231636237886740577919280140351808891770380701081664094087850279863837686123939278912870635472773836711385362422962441888871801247844226568043573191593402012358778484977453956691315576299273011+0.0233606921152124492271980931813674599869190372953934500274207714556876233711642642316885539810004028003089274507478327127657948070533173625635082683844341038011252549335703744175458975516074541831812812131624781443009598493903430761337846848779373539788823154070483085630296325179847691669617389026203104134019782389142014563870877647325718580130195979525651244899332138391877003647068701187002479245815886565952143174065105*I");
	test(591,"Gt({{2,z3},{0,0}},zz3,tau3)", "0.2046028184331419529523929836942516637534436953780582054343625391014719834984237006367744352936414263565965781918483320559907777339383637644487297807196782095606401362898908882378442239274735201252267292961656933113248602275089924648878798711170578411420663852558549897005762508230126178490402328546384997614698248708286192140174167377257307841688768220856643668927105092043653642260431335991480601877407311632056834468475848-0.0076402334316671372064026955310062040946247221847075795659649793295268311588208209970908262260671476064057843080436934398155950067342368824245063738991147354179284717974436390225852390985401696245775891830941665728915657650980523784961101028795797110466672661194360182184146579825339902564024555905987348326748542484693699786054462134165864544788264032367101280602484735956472213698522148044305994375283332306378939418601986*I");
	test(592,"Gt({{2,z3},{0,0}},zz4,tau1)", "-0.1774158771629154809423097912507976989918845033856710625366366526204347477105245126424016313207914357511207059436763953441236241563978258906075829523069701234669503715063500989050227723487563192732957300406441354717523609710385016460622435794426418888078796387042449039935951722160290754744252674746488107305470335155353529389835205929423592297799795084877668236639139070311134988195831348221937647516764773427146572515362036+0.0651524436352836285058709228341185515522288780844449212260305081264503207609722445609021729597388936037051727124673905876808064349395673490293959278038266639353523353277451655209281566590390454103755772869970737584318951766900221597807113904545255894727003413143455447391526632720318697835876540956310929115822281993888174416365125636978156905190595893961394774488925842164224231234353125364721240630607781065698051123013213*I");
	test(593,"Gt({{2,z3},{0,0}},zz4,tau2)", "-0.2464486105729501330270091268218718675199145241030092536863441785562509163337634671100099630746160341815653143340594278526466182846991786783367090181213347703141867093502608113191179564747771698632856908969899384173670260673154293991074837173323466716112637488800656356397906495019045802047517370142888385241927824396524347527217600936475691101988393284586246110048806948340436402509566726390208473760466695828923002631960368+0.0002163447297815357201050489917870457907319445192229580871604165837260437330431033167805528375908505817520531160965234611458089749285813576850964487441148129018465539391356115703711147842228226827752911743660127897534076530592779261892236166671490794635641849811326768142595279701732535422985661331431523763656115589022236275627894738888070967660711499471162048794359611362268350224304429659820367684824033887216404812816454*I");
	test(594,"Gt({{2,z3},{0,0}},zz4,tau3)", "-0.1510358595788521198487246911030649152260785025355647153406946412930697304435821338710651624979899067344339984702273183994802142577960337389910103346376429270971064397339694929021573695737337173922682299595306563785487293004838249293508425904354447742455670727866957282795863129721319755042389757738117160134635827259784591846151802764472118700582591945400361331191187621245062284022817504179936091825516479682450588516861052-0.0580112432119654911656829898912000215297013051950972817762873036469624872738671819639553156717643710976925589585043715297434434281354730369377744875561295023407093809243803637808431864798597630315286119158107404016048951910707290635997379046017249284650714848358876416529097976548135001909052187924508368926035374739840264510243885411923690226321893435626510737400813560101865894852357553096662903263468770708242025025494896*I");
	test(595,"Gt({{2,z4},{0,0}},zz1,tau1)", "-0.0810025877749550533232911725190768773985334979300103775783773631870124748472808086157011621410674299910281051250300479104154433611942011633674467109078982865506091521244831955541073082852913079920543050629654143550515688571399554424166980933883991186085334164672769790976478857881560672591309913954200396866130697622289235431637608211190687359255923533334143339588545116742332298349175002868459164284680018356679995861306847-0.4432353633843645819371448095692221278672903919731586493587426473252309069417551557150182743347883239955276390096397995289486278260383697405291682357622726062260880512283212212414869177531850740072633552662245784283195084458075873853455203574997080120568351691892964954966408348990090026328463586220512376026369714684137415796931184143350829485472172901146490393188754724552464568246982193935569872375211882636742819784611269*I");
	test(596,"Gt({{2,z4},{0,0}},zz1,tau2)", "-0.0506484996964044407012621617457294476578166630059596470905903494705647423754995660725457555837960998790615926475346259389828127777396059901985391874740472446708738298975756884700303874299143479843985187060759691860601345998045933741275025403868337276556952637448474091407317850572063107604323825677340686154384801212784020157143074004366002282073328843052875596084171866590012708432603708563255062094564012560768378105156627-0.4397267062869527366034632879152777227584196366905852899885529773004177246685742135069686790092574094011449297440443203795877621759219155993942424808717806166678836047204872121435119357914096934212251280681776694329431691690207823573990750478234945723744916092522424679635937304402811943285279713033816481753840854692153294004836363451552018927031060892313603722635662571177652613476844456771074993557885166430305048161107905*I");
	test(597,"Gt({{2,z4},{0,0}},zz1,tau3)", "-0.0654368980820998099269924197514673299895098334740147710835341276021847741597076777441979866304137619383893880632014776119254731666285251001931806218875485053064760355610025368427374643115073237580841227524481098327525906696251729941799509914877251461349096540540913792309364448369426608658726022826513802287384756313853859181692456512758272641150041612691068610434883073753688691073261569118071597043507748180346335949984702-0.406906787003936525342221844967033107897271472428305605634411144753449820015345982177582345809114975740009120942914736097816113731118093773265430897554544214858641204825142380830040227983365226694481732611327565441614664973940595944558929385751609731337786523699814272734481413197126948222048518402468380783035127688913386220164215023879388029313255461923084375273930355990126909515479335701109491209339915435853895465070432*I");
	test(598,"Gt({{2,z4},{0,0}},zz2,tau1)", "-0.1732736792750968353437439897257971943224063673084998188699435893732036330897955975928693030192254162079640393460283559228865077401789241448964174062365492796731453067120441779621673662231179064938750831665701974536149336559697392238320540709892335058090714452235518464283969671603177848472875992594851679795622890249770273360883899612978310911476688097347188499011405690008624194786134267604465948295948260534898758627250384-0.242363013651049540332676441575420571851217563332720544607509190229020213043910773708823047566117164633736345641589429332551640548498027749750099361016156912309897403408982536446034119568979671473021452847188677974748354118047009690756874647498151855573157453633248841429917700584113016694032710369426532870232636199022512340964229322690722590481321883506408150308999029789562343047870915501413984835724974536161425266356362*I");
	test(599,"Gt({{2,z4},{0,0}},zz2,tau2)", "-0.3172434170207631909209753527491216280371849708858855904700110950274166212058400984823882884544674634385559062726622041935731167459731632752219618538147807707864176435482955496831498872822070248404367805303747051372123929478800537538639376207037183962225933668821796193600935781056669925681852616392120246696020442754501776067274896279343910706466429117869508680662955203078733503260624929046920033407259775624594953146671145-0.1291224748444117131278367364545462183117018149037174688816958014779536726775670186181010658344071499259196047548016499071344351708515188297178317392502759977166843317854207025291570905526883657036651926816610327937305088928951179021852618586595259310852586223190226679196194404393646320341536631796242407360013309840826516405592238314556075428952718518408358162511751868186579453896864409314261350335338356982568276516396809*I");
	test(600,"Gt({{2,z4},{0,0}},zz2,tau3)", "-0.3835826808494143976442004061707111839993861305405529237708768144380493921952741107525589800136293903239227076535989249512946307609666304052809762008415711046720442894184575769747478784308132141179397287182188010589250348603165887709468773622009169364102356368759788769360718193227576838760456196407986696014150503510994641540317992452099835621182227145298777868106313955943495991332713954511342573223176090528079208338045558-0.3265463002929100023476582985726362478852410595897900609542882764559752973471041158544546538535914631967130050085293497363038341802810660187375997046378127371062023699665523327789904100447720884739569268724713909785842240227611057240157578439774404932469092776157626746360914908474050971667229438427027712166077765137706370378730396461305786549792472343809379294337867168233985506726231293875537929341770800727102786791009447*I");
	test(601,"Gt({{2,z4},{0,0}},zz3,tau1)", "0.2276386755676031007830100487058684067122718424837707944692887576900164818414125784043063129871184206028218048205669729651663108708058785292743959066934436964552783724867121244167733561581092867920196208733538734141358134932531989574079831810818195543409792413484190609697511873971253425070479470144631980011662320275113435554675657650882310209118815991478306009771721959176768570977630972303715505401853528854181233325581254-0.2256208955805970740811419021697065274297268272208302280947958375207313016337373239749991207073892529223355941300400530283488068419761835785200841777069737077766405730796483329423142882441756458581729706119984859389901994449926553959067918316787442700928684584169000108790974275899003028704531616027335383603221425347493470659079843250990219030713091682997540286967958859974804001049723052353135747562629660256603404704513689*I");
	test(602,"Gt({{2,z4},{0,0}},zz3,tau2)", "0.4321963836945008721072278710763015622690500654168512967135998383512359788957239436450314480876450289415907000057478968216645764907818714958176775356572025241240033672858355153380719217164455086674721122255722357306006996359045515158310711743904184688613756035216403649369044931364693259453666126628722044441724773866936118627238389815953511423412860386682136259296830103826262054699778783634862894820497703906382815041287592-0.0005920630776313089080694950619435901034387223226961557734589792974869874913252316331070512664427219571503708368595593386765702022987201279174777986635013270788120242424735413892635374368001431525801670289607252501992424429300555788027030048781467164485723927512515703896057454635719335164414141645011545365460870405536529963304869489463694465757000004770140700182571332891341036108237158472924645907777162671378451473651417*I");
	test(603,"Gt({{2,z4},{0,0}},zz3,tau3)", "0.1286637692959352114520346225291818569493688693743615218066175477474573025639584475818483288274871588268374735644126955721915883998303103332732861715483708284089996742691155187247256366012777247365746875473789077083572747430090104121513531775201270795015780647088727734877739598885585741284669969099249298442180850827554326207705562758400385756774899499832327275802150955975546411978632779007224257798573672304722926915038231+0.2151576058799418100895311881357986614662165074495769971211766868438077374849825475384462796325133934424618105727141333598105453282002050706326086589844782427314871571250466692219035139563710648037793873143589714992712816678786760540880368606628126424720857739229182884161592621543696407998369018843570339651839628065688390819192807791853645324258423864125022301934533449849466319909940163238077750398659549321486417345510347*I");
	test(604,"Gt({{2,z4},{0,0}},zz4,tau1)", "-0.2110323883543198571492348430791027681382700957662995455350949761434514632540783163691146309637613107836902352420415302922781426194332490166534770375418082132703864126266558611393183471911671213921998260215433773903754460654705434841858452053888070970930326308270608109761953167323041585877005425036704383883748558628293179551128943087022106545041665448074645202675323528939628021588376146442617419060014749569335798346127459+0.0079211405886518579742417002524665409735352818567455519011456883477753566242675996312653664623493599699083902987687991764075355097193825310893024601266462819649956590756507221821796730784466413517385051837437913042920815866836296168679060559168044181410132804238870887768598435481208004239092300581411392931557401438649183253027439311450647332093580733441391636663116980188731355678052159201987530761040250677071881850871384*I");
	test(605,"Gt({{2,z4},{0,0}},zz4,tau2)", "-0.1882537363506285548067636651347946614268146609258066250291033659176819965487714758063901159909574138042672151545202732681730227668212620094285883949498311701030499597325325737186239322124262847718285207513181655533571114133383630658127165835578390572908155767334630159130561500579316962511530264104476912959907467038609851060537336250974317381364361724219839527495829765479830467141742083960088198065945329752513738242352371-0.0291513229647083189392982216006982656966212205794268472025662942606909185280647234949636710289587380232881492072845157910527666471997571499368018295627543873136635934995205829372305034329890307294385466714953573604411721079016464372074486004612976232204458671619772322477999818228327254665040694800724255794260391454208529826887633906962002299002622514139490931555882387331826784497906068181577446932480554470081197917530003*I");
	test(606,"Gt({{2,z4},{0,0}},zz4,tau3)", "-0.1545346005054954781669122231530507461783966742131599900031158664972574210836623823965740319902534904751805092634997004656424957771194794991339729034467676930952170477345790424007094975472587478143382962004362885547499604583073949806205089010975815608138716608565664270653130892298395810326454814359964707632489147361609716401047067192700502626669728965769029846190982075581114638295193686789784417778447983007585784675842801+0.0104887356722689284705639890055280407960101380883191737890435547527912373395044902349393531945055088325059546436320032566225126620160830973688578630101474145304974580194188804607220865182869616047571938871728703995453183215514134190394041063738112432758529694368410865162748933528476108413965029603602274166096264895345687439374725943646030027089551426070036641739829121439647948540578588695689869573616413071570866855626948*I");
	test(607,"Gt({{2,z5},{0,0}},zz1,tau1)", "-0.0540875981591989328900429670641789555833185873356653805667319368119393796384971626021940953879450856901408001901269204851451919280711509442589760412210961616928152323047590590942141661296470556921210046939097631109180073730443591132050896521013575587814876336716159322697343547935974076347444343463104144730146365193571501922768965519364177586419135551538052713755825687270761834553224840880941078932474271337195871061213856-0.4188069133141448367856091853912080067376536946779168233970557236171524610340171824603977657727163837516853129485664695109967103093701152154365437465018799148446746389987297184589098248745115457582452854571677872202716763477596390243105723365473464410066212175347670105910719423823943704247684221280730964483538681395424874707403763777889852595355431413842562296369468921619133415558762439060777075381710127438835608499108363*I");
	test(608,"Gt({{2,z5},{0,0}},zz1,tau2)", "-0.0776241743098491832941145387486297901881434430192344695415863315324018920765961876898361998831859740905855110216430165279128461046018489477410843921138769396096255430783828635520312050851151686149982714074313784034875936775970514592082772146692677655310709380109642709395158650636677251652128749736724610610099433319468450803283027471578245892000606066786850955676291744742096016061536728204762554060685056259378165442040581-0.4152829901497302085670984068111490139103374038282588348002724050158810751281250405686651710349897226027025459213409410728044907951900579338063752827017909354699500846346264790892442094840175938840724212424750550651483837101198790370238335537773283264392662572960163872191040046632650395581181362615264270601376592734442822800705965998009937198848862311158534027364014273162776578030157976862888642764255932473560391708871104*I");
	test(609,"Gt({{2,z5},{0,0}},zz1,tau3)", "-0.0730237083039446290054858592778710006973972044625867290979620566048336055148146020130886555099694696299557238707619007663271523611481301191877450783643370789266717748034757526051448303508222382503087592725552749680637807888412206014293760989719893129012328182988038358084075544518930500514685069631506403932257646687741871988713520281918425305203588966525083502554290357249829999487127783599392336765468763720004816228239457-0.4427853782379804623930643028667449259989497574886213976818198299395735442941325611061221893908704338343516209057026857263295449837366352216986961444180112082563778919884046223640527977471327737613252667614500915241331788767865848569650022747611992209127237908882274417139363126344621179288029771188976811429732466618464673709167623982151261361002297865017591856660081489759759530017103125610506632967590292230221211900461282*I");
	test(610,"Gt({{2,z5},{0,0}},zz2,tau1)", "-0.364857806256279102595844161534209656210376587060575008585026032081469799994840357012326946413700892464892065317685621520831778112502315886429739670728746506305734640426008818665167668161837383863519414528295910917531452731709601415599617417591505492304269063937529764977020410790673410937257176584559058073936295158822045613561912015270680391086226173041527719638764310961302844500681655136111674014991446627965075119735615-0.2553932200050355801243707068635236473295151723751504997748208169896776819228279964659430587602273151150579614552436921850943774039317811555669799188438197240739916538148915897149427353442162980852983371745672725150451545245901266698295269365168928976895549741595092371211562229255460027636530916010541745544714290572121695022654304162615020460197293338071091775645306786051068757220132692694529732833504436316427216778236241*I");
	test(611,"Gt({{2,z5},{0,0}},zz2,tau2)", "-0.2824279813406724249098414369879867540740827212999360697734527604958341534514887173913307912531768370099677581827992342682598244809927553317206816448020711730319519192708224496889113822878620115528945476844216591030459603634274896470342630742704991234361636608470825612213574666413436972786535591944860339836480328660138879914858683195834209262645301625977367814054783504427113476892275171270000612614526041012208940327215521-0.3189666264029332170775990697893931626515721468410717656772307479001707081525533597498744521065824719966628594169595159431450496021087425833379559611416812496031570716786657490195449876292845451659445948568902666210905289354271570701283553960038589429392656596518610351391965941395951024427143447126771283461072110580330374584419834780557293365347515661353480841363336195665593165193041383592687592880455547026218009224877206*I");
	test(612,"Gt({{2,z5},{0,0}},zz2,tau3)", "-0.2382794886492749779546162814616301649521406643434336632534934401794134249398929273473532055056416718209440464074773487523484923438279084835970853470703521750914447614494527029092614406824436858366031365488064232272985057216349475563273487263140560591749550832685702881443528606360508496821306148191117703464417274678702970867447967809980771318703551788349160769134641081714760095963742678331250755061734625406787461326983727-0.2026117699345978520138865160410175910140970986681453191731246089142296624919034480688004543254446811710176511930573094264461863357874263953365330006745336359180087446002579534059965809828198397100914890012683456172798523655459703807474725707657196936954734725519195193297441322316254879569239954513427116367720204588792624201839321269167116912371697007420302250969681477655456118939700894730375291930249569269999121436636975*I");
	test(613,"Gt({{2,z5},{0,0}},zz3,tau1)", "0.2487437857115195224885216210032323595428018291372437844699826693834213359153056245779325511069333819384678942811212696163888960843607924933192070654790527639385907710398040841777150974408219973517684474075525576787605314572345769614155656943734076293230448693370614221722460670191015890623759558006383085280668916071432159172958505338342740148368260055748715637922432622214552359874294044350908182309861460555562035221797036+0.1301447721957418230804431566740758571606262088397775912569340367229712206745510159968955715476693015464382282089519964113623280006342307539877470898536654881771305226040397371449674927592431054355239290550477666017733747980181127611605615831314450638097628328602442519641827471122124302829118699049487719278441919004590115310981714775397720398488955709061478657431366971099737745794392555238739556167519818102184586045141523*I");
	test(614,"Gt({{2,z5},{0,0}},zz3,tau2)", "0.0860672445578500502094489847874906390889072421943047124483881698877126029106732522756206127749728941878456523313280627226742165160135213783731503759433795393335775386282777439422538922175754729977174447289561980231656961069743353665238782793137069831243355998038935715379615948982525355583137079452604499403101928836458827751719957343175529352856028494323619967163240053766981535030686086233499099092413715155964563838570938+0.0002027293003785017723628519377988026586675763569510363357489480364868578772414189892761879970723577049954298884891146690864767710610883306057491315111731110279928565680980331998456502227574356347682608175760305609768683336270086101356988201003812304526542845166768644679550539542997820649990464476512744604815072184622193630194616722328538147122682270649202975160580516509302332239745467589637076540446086284787986324750027*I");
	test(615,"Gt({{2,z5},{0,0}},zz3,tau3)", "0.2983627606187485213762795210357893827832656475345133262738275364165249883753986699346080331038297758045135189451869173781272423552427426813667428981010715586217170677112910340680356530359605581509702012201532183093332875507511004838405809443680114656623446459108102750781610631200016914222528338439661206200144371235556513195255407259948722824516953057548539808317341947697722894211606206850214007673888999709126915046299675-0.1089449191821442965835170873144923533177485036742175650126446302328159510306093311610538798196704348347929297218231923672024126636248027936384104735899843803728600828786436101430874950627818391443539816149802712720633915380085019877780758393683194039961891680551000954681406402523022298286977778351276440769647423485229715067912677949989625345012165256395143993281307524946863961200515878630943393798358854423884109243633975*I");
	test(616,"Gt({{2,z5},{0,0}},zz4,tau1)", "-0.1655029472063920632914117135507138159280559903169343351819237062084404991443114965971396288966707720342986209832060225593731471857441025908889886156538891536633895410684098754857971851231400288397629312110541074665307087959394966070982682201519657129906516178701447860967739994848300303352915607707270247932436663454768905621616848149217039792545550639088575170409281744416093927955670657011200753865120364832723523032951703-0.0052216858865080895350176554731738849887487833168799508792030960190065871045574519333618120402803818002757696690066662415869323292777968217634518472296912679304323158054662354068250714866995247965378259561447397851228545185792632704606446358633667032255278905385091618538690853534850561400462434580448134363121991225543018911088139168762854651579447438821294261389037067004587970269471331123699427924947431756339470075410743*I");
	test(617,"Gt({{2,z5},{0,0}},zz4,tau2)", "-0.1794219811239081978911469400778505204530088696090080863919062973928283927850701394032822657211747434088258296463602666840432971987125867222485487012766308554282422554101214426740985847348703778803592831713650976783227645177785310324964064972920847187087262923781240833974782588097856041144482318586131490426860213555743648461255942718369629896634368018009747472990202218131551794742267519516803368971806744626208067298445264+0.0176443147676417413754912915847744155476579012646735246425100207891087781242566534062538307530709863158427628623480198193131968786519588182537462150653330195636315266500103551478114887984297185670640799571843220430922519215299946470790519172414011778011545761529314069353179744681347817355133206196419814226557372084180311545953534022080242208241536388098154035453275773318025171061129902355345377073789180419658811558971696*I");
	test(618,"Gt({{2,z5},{0,0}},zz4,tau3)", "-0.199758805420536029665681486206729045355199691753193391696372910005733091164710778365000870873312055131186523964055219621879393780316487473546857502574483374099463544008825302857914760917156762184349692901264676620935684275129253458219372400054752872255439082476013427745312403836372266771879991329330971856172653503794483990980359881046490312361180364966333688343303134656708367840598980038491921106132803737009485229749966-0.0058318711423374314622651186008358058087106403132701935695766052862236043384854355565050152786267107890862283889227925102758521076125977247823275347212467730241982852687699088166906213636892369213518489414652240295304283348102212414479476599532632368559823931121100327252076372558800432235692438743047077482650911914426976754089230212155384218679763498721984348991340123382430619414951925196368522346365179907416608357025013*I");
	test(619,"Gt({{2,z5},{2,z7}},zz1,tau1)", "0.2113579031021469313406947058693944228963331549840598630487062602742614216154606556880632604485619102325427019799933118459353783914635595039801892012236871109316926412767796177997149108580292935606485386620327053114168376799708320518758000315493309255036989563786552536777549226614646372555473651450814709161941771665459501925406383077615719289642828196648464621470928515547703598514913804388824168673381614163787547405444014+1.4299869061596419808323688072827893601970460951110847916295026948840481193003409002834226242722672114044295695857688215705706732997374366729393712385117602206726310378243850011207490096306954189096625508505727298898130233951713103654895499539501967113823567534809830939793974009992875701575170378843248093296599308150007529901963927220456273674082130498299108502024623363275333292905448680714948922805839019813629984165123585*I");
	test(620,"Gt({{2,z5},{2,z7}},zz1,tau2)", "0.201268362727031797278069768634580985949268911380447439642375024056615138652541290094472408786540808873593191710525293771483683113388055028928573497759327871718630001704798731656554037921025718027867889650943451066210609505120081867147040001196563711422399305164074880297013415265299163496318229066069786301914215533194397032425373602914282618097200317329799240205501988159907757004196261557436707842119837408525287322445807+1.3971220739735851954249869645046022040802284247262876405679477320322572198980114385019813134621290075564951591544398342193497918309896013489438694960034378476112917068513791874828492408602558146982776850186425503054729994334241471071947713866977095033205784899496098867207911067016639910120482391546529321656881775470972609495757084379369816063712501670291801865458514293560412966839148189202468461701402509395719017224057285*I");
	test(621,"Gt({{2,z5},{2,z7}},zz1,tau3)", "0.2422480930721479852088212943327651809766775898928072492406760955312853552129861270758977613949027477981931770089958074176074485739389702901140830247318087308864455337146562316306718931458349147659303234597683296676599754577675771757112762029404564625354294036149401402703303986844945470405894611434412088164655032658162059370904495815402710082377593889940402847296098599438627592260782592052008400364590648613412124612701609+1.3908152947606845378482688816470810353668145601674915512673102399950188012870578391981019767985870440595999868796491603775995844872638083215807288187477757052554175985520790660196677948784261899098482728572808022645853191606773117476183738622553365599669575638747974116600346798640097333569336869306739121334117869721683649932196081118659197951247407630758622469599447317117325326980302480765113397762924096652339280273311596*I");
	test(622,"Gt({{2,z5},{2,z7}},zz2,tau1)", "1.1542087155876965187271152894149422197936692343529252541234712887463242203988772581256010954332527901193419444417853811434816863092003202041363282590992031826298551638859029722277574859730900850329975278902057301230959685076004835589951157145601520362895792765821797903141931780896920834167111586270833819572273285750038243809186544196343261755951695904667281084271082807112968952300369350999301786950432240107664733606089176+0.8396825141656496786059244969265405212520347454646367262833846021953860743220594904747102065994123581248512299900764841587172819127551639457611985501908251735622686480423838258131269817610160288619606333166604518645459103308647904650180882591973164507118870720169781231563768781696477362713716540998404733678600367950955994724542654289012388543817410673953972488024994945605401397597210073386166961194326062639907463502035815*I");
	test(623,"Gt({{2,z5},{2,z7}},zz2,tau2)", "0.9449420401563741010771359058413473231887949226172931106656895197463626216297369440461351216696492826974143425096091565948492219358456674610622078541900172774294259176952635986435934104503652657994751086189766268073657925812379278357387496161145299651305761977747596279389809232517457494152681082414013652723013994983083469722672567465770018966483304413671317074492082877789417694860075247547302585243195702961879578822394493+1.0066821298725759993122500499320546672145497181434726399459998059772651405672802329825005558558645541561056446052583420905434113151900478924933376010930427856965563348101384924832763928649539902146327232920964193544248692659763506418530421091939997008108880611894695857533212453028822550338373823297108423449726499183187128243881508355964438061258776559521461436380183880663285630496265293242478175722027816354670785768496583*I");
	test(624,"Gt({{2,z5},{2,z7}},zz2,tau3)", "0.8070930180808029099502778662882093634537612900999263169475260198669413341986230508904206083317911875333436422297356527474504518763003252715241460835964459956017192905491114992168121886575764349415511041996622678120038201090691081010479282284392043553808285590448776723791102161085032277482600443843226131547735252711325743865845613821825734323581517199954198748042533807478425336691428483714059473685865750198338941331352689+0.6898322385965889451457317716423446022939902381706089000839842965975393909865580512964380163117915527239827651981541591710464274718153307194721390302869507330617930628708711224975236914435023701965426314248409769891060656763624540696607526110278532006992908893629491789058580237974279846460746897545379682524445906454731662200634749864884681996332077262420790387118434763241654086018912126891222870815887781032872606268105902*I");
	test(625,"Gt({{2,z5},{2,z7}},zz3,tau1)", "-0.8154060932621650826608269776722118285373634656278220657428104914012706187126404708743276364551154450609763660011101348316527562997292604375590497117010841954350504580977132663474316249592910934122696348917796122861103869244672379861124396270526773697472574642587524865140118423711074206409628439844752797434107311194300071898009353107695230092488655085393530924253146008508624557007689085152045460567616561544469358411307767-0.383087214737172874058731170643727330162203862117123402893011751671146129823604218437373903894304127061511357769617166722944509845908527710470324878999685693761469225917273855342102205677814102827130932934394448118268315809589648578880151995227666072971658683759011535167785530103785994047340470702211198159258863799153000044668312546564757394846843916235731375900072604997240636361901566907520764470884341770169624277083814*I");
	test(626,"Gt({{2,z5},{2,z7}},zz3,tau2)", "-0.2902867047906497572019108798865736225900017276046901340597874189159302869305927519517153620746386806196709962393871859442421010453719748469282830719208039247906695250517517850122245702043295451401170618429309596373879280444087709430812011549334829161487557612097785483972358391428541188902999126388660891981476730807566556040863020477027169279329136740079526006638261098715024303189966381603202520006645054216372840849257747+0.0033909440965580574950474409477031220326960627986948392536775222727316547812580975566895049982731002322689073511574864455393796932206284649246111294007978793909954392544077055871651428774518006527266035743332699467902095621683619275193951756596382760993017420156324272391765775576567203712122619477706289212792402655759575225381250833222598053870630371354761559680287674890211335998979227873218268456538752151189237524140624*I");
	test(627,"Gt({{2,z5},{2,z7}},zz3,tau3)", "-1.0018200673935699182265058763293648091478716359200611821359300100368999389458587882749952855259674022025119130523883755069992185174605724905037132116507135932445780997091207138584502198711953102618202333796039309470473437840160420795850542505515059775857060479431845715142745252298725740918829008575583187083290158159990528353900994922193113404021369873147148591835832183309845647792994201120805387369680659790340170416068576+0.3111546534979533793649064657933310476292682231976705461805520655040083666033881290027856761079816195304490345939463200158685634719040533531917592708460553556056634297634022334684204213219276320179542725882327002350756185337082995733878397399198834231956475110904815744154640135319131446972709821587112100936128108964155056816520626606762545658872980873697033557833548977464747870022929759303342305178633326702375365457793548*I");
	test(628,"Gt({{2,z6},{0,0}},zz1,tau1)", "-0.068386115569528828278215267447221782899661515089604483688530351221081328405355577827068042307248525165736319443186942815561028401910846780737432973504690676085824701502308730006348411198812695149070286506500323706283615420812995213212625924149875841317563282020420300808288151042864293559707771106242855919029313990540501647543116332226484808619638550360345660149412216230345142692504796089320821206010106073383623685333788-0.4244876784241867173536456238146601816503385594837309283004513750028175458721849640927339686845394159330976026922403690467214337923370984297959246584792842894193258105174043785566196215748461280174296319656282775882621716703731353134934621112913877032366153344369347665935312807641721048781629812129681658459380810024878489498066221961475198197333154111901267929745802825747907376648690497200098096942308042357984296177358419*I");
	test(629,"Gt({{2,z6},{0,0}},zz1,tau2)", "-0.0704544331031734720755413516847968934833438834455184476607753542115538364914856301825822593461816448685433504104371010308478482321168119425809064612171834923368009627778379655579510069086343244640305526648712883890241374541761149449805927483940779192568834322667456246756239574702095677269190635609753754298794275529759096253014605052637751989689425069928884993579193903564133857746378011232018357387577402963234580664421081-0.4289592906569491353328167285013208375997243434747150601663709381912463099077880354860281571506829559247952867515455916298722577416554766071891682378154136491750435336449511126100514270524611139413564587698530679640615233154200341121093601186623924651532556845901348335703491765006450822810499971233956006077953751920157382241610133466224831759804934450993319682051596253072773500935966276440657681122046951586032001742651611*I");
	test(630,"Gt({{2,z6},{0,0}},zz1,tau3)", "-0.064827283077358251350027012870308678556841938204704552139058241218115205057347948330064425745529326518707838608955927840068947695378222014095166447303246973690275862201384667082818774421117384430253747856598469614523341231404306038516771451288425997508830871190168022646311319620653103152945251222992169335569103797019385924365389179253708597738442730902491711520650975413281866590010514730660627721336324369833273853764628-0.4297205181001605246732645043214393174517999961658505796066468232566559845614922025714144928964272473590357777325736466839771459403979752696106148442296703318232508674819107221208350961366616962044553818018511746485116533361036833329519971156149394321666317074565020776944027358398551132329543960845959583254160265513186265557694982758406982351621724533528496687103429319554284192748908484244751020548731513447825448095237027*I");
	test(631,"Gt({{2,z6},{0,0}},zz2,tau1)", "-0.2714313756351784381116259754988231274707599781368312837281348952190679180295539296205837397783638644326111834226576721286929327598076248750107627841429486469465658445801645175557570186277495076203694043252526938742492423475703851983854029952900953581907090229320185321192248726087413275955914957074609177457222336245666019006644449536117975645329246067259207707269167314325957323660619625332369177362200075897098287486730866-0.203064277823106665016153029240531668386599575170571353631281554066303609630920719821202589010601151511542515870262450189299285863195698772904609756525660816732456529246388155155837986368179023416232263604556741676711429707627674774789491965248581446613121045904216640282608605433046557248787688200391061693975658449948751153391010205620628948294607437897619361059909419849794671276189437091348617528711229396807101968643484*I");
	test(632,"Gt({{2,z6},{0,0}},zz2,tau2)", "-0.3420684522217707984641571287270893202395658953133218481668227803814834866627728503467792283476030994910268982730797972667612683326662131544986087867868985639142628826491904395036637180616524537117201074845762563336474953117963502945838422265703628694276405109655646177602179009751110293374611137197048249310123059928365575771411385218584805475752354498184123558839373289236429105270553475237029439615900521959713069977784005-0.231858191652013284180803944521455964688642888882510051274631882058573044974191474516370883201040333060684339349751687288463159090837766255392227126328139872336389726872666788577458144483815693839757147707511309960166398900015147919873445332183484660881550887972378991179449556399265099653044165368675887112409916115502912900917412531459177737069130421608271633619069498116525675073621114367229369270106889659128380617897088*I");
	test(633,"Gt({{2,z6},{0,0}},zz2,tau3)", "-0.2859110019590919398674546628326163285548234140177538771823160045680189838917501471485155594649138415491541238876584254991569164496620774296233968472275538021390461421583661063327175871419058672771510078292686002179186469505508068857890818587899090081125557075226238953652194934597888835332449138019017979757808337147977365112370787171820348075582353490070288488698856143661806918322680129752544803657113001174168844475854675-0.2994260803439162548459655803830891101406421141818434737026960904060491736869137975251913675987054890989861372415430152375428212256440394501272957155589170987285699418521995141442195319860509371307211207973780645045570315147784564097502977486169708665208996021733783793527518105418194617643289677030535609908581196767537156456955999828465535332277351196976603419590887988753296987778163548444592721164383314215335904775639885*I");
	test(634,"Gt({{2,z6},{0,0}},zz4,tau1)", "-0.1913132104590348821730191634285154715698335160383609881430943847119376551818066615998451099577170351300253192913399573211603917874571740890893427952369839426329727144824880117087467717410752093331785924747208230651880700175848361410395224924494291978297176423332808795290119167312357718847134582764489528785005595137024053952899568990262359777613426585922622311995488727993516987197296572018973874335824448145253977152440886-0.0079627418682986929134351865524429467604395193077940671620919948775195149609613083385803528695122467326389512638074939658343796056691351733710057160053270013722763573272844553403511648606318704109029121534586719522036869444714596633753259645319462004370875308784925485725866190967850496117098020475547211607695680652813233781652968068444142781002926746486589547929327276759067144636741293967895940266575081421363943121934593*I");
	test(635,"Gt({{2,z6},{0,0}},zz4,tau2)", "-0.1739173887759144102362353999364378276865800970229036013085331152514727281094056782100079496062338665217786231399704246093023395266405196464598392571885005320960170629319656273319120107331584879296131056576030414986509516229971461678620530824617563281363043635133425620323693804950774379782138497726357178886375639056698566827733197078314550455042337351791975500495670569860625387860441394913469658120505324839755977137906819-0.0077649528118462714223553105243381426593261949684205730330797629717478558062492696358566022032162597645143682982957586025795756513519694851876930311058762182795078193186921948319205490853225588060910709490259653508748978716030066259194188323320567851219358512604934259522209928022396525051210805429575898979871766761606183544658270145237181740107482755354605201247010258458248390264821357931209240948035054431385979166005673*I");
	test(636,"Gt({{2,z6},{0,0}},zz4,tau3)", "-0.1805919148222718997091826052582704222783766251463913739238611710340441809101046225606803418627260619940594613051918236786215223609705345446372680042098301210294467603162122176581698730028819038067176210628914366969148862202286632040624776534742085412797501421913260411949685540771724998542770685329326121301535608670122215890234395218209657727237684255293726568187557656451769367535851616083631944759627523818542998459111358+0.0115927432031914849237931143187156871832387181394719186168042006383009308907384291019075981774139105083815007793649877296209897062716764474209143635006338576371321887019340476705251931308843393803208807687623760326633547931292109468476861502735019510974305410480168888735656715188962841338026436504383979393472039822612038708247088748907142993465270006939440227673675407481071707322349321970958921499230673534111505913039763*I");
	test(637,"Gt({{2,z7},{0,0}},zz1,tau1)", "-0.1039260345574352786748384014367778880487669308422446329009512882430146653777801161148281816452381219732383737166739305575682900102426591852973473710309461922332400111583965754880683613192922204913243247111410716967690345462020390225929122261690952866939940282848542497481788209860580759421777898579142858994161175396997447328251296991006696044399014428968868015009724180193962958933081683481138813151918975378370992985297749-0.4464909821564646671877260205216422487782877064199538858425866132154567259405798064933987592235208641454835093157809532685088527151905734283934199631404077661744140913368793805908088611151504014631118017617436177891390619344227254666312696485261799254676700079098162148317808411877700505868118192371181403477469719518258816477859069794473728802139714602360206487765613634680248257690547444612661574186052124553429408392239528*I");
	test(638,"Gt({{2,z7},{0,0}},zz1,tau2)", "-0.0455607350457406399808279041055654444669048270982817845260901341580752535838188510532659793949127274789362046796615781948547240021600832320782223066647986768941738234769156593899720794549718039712021841291299644302263654399894117898115096825142319197350008757398934029492774935324763157298211786818293863254497353198249251806874560709299433409767158087606415161112481315037198207453334417294230769837395108322488660338488121-0.462219197618886040125471742959167664079778023096949225718304141186962981419173205235578959329375990542575192033353404381169574145371516712017835769399221921345214561505552538380371282255828736713867444592406397432497120805891207438494308150792035768601015055557142828428041614282094237225423821984716530260934200985013337817483144987202940208602828982951884550105636745057836488223903699968804163995576713787583438297924637*I");
	test(639,"Gt({{2,z7},{0,0}},zz1,tau3)", "-0.0480956432880360373031415591748211981424973491354018375673455395544398454192176695925425301491252057639450218251451659957523301815022711886707857094061861393366269562547671428827066755826662890107564740586404363341261481058640023536482355463932989798949238293035796011571516676456803337195385917885468502266508218465312296354703455883523557803300556637421829764447759325980719430992877041841541664564997310717560518583746518-0.3907564274326420639336024610042865994474973498194346710427099997436839960740624968075207730436344022378813674332767471279572523483620028709848717344671641057292738610355564581830759435370837863200307781871850960628226592333913474257312418834609484318703724176876768308243434672006399230911473519366116191476566021650658740656350874390611114747818432132833782594213431494142389517973851538371306175927780366201720939302358913*I");
	test(640,"Gt({{2,z7},{0,0}},zz2,tau1)", "-0.2776221377057368850437104636493269650706861121302338646462217580081583659248090467825783776035757940939058287881705941224833230551392991238899998553484677935961194923539647058218138073298499447052775249360444060821469410589686288910212689361648287760744981548902899662019796239789287420902267621884527443148608821618278517347059996815153044914113338149665014645547938494769359217703942744966134220279715508223256002630521967-0.2547507757218998040012663959954227705147111531906700388290124548292867323354780344405577245464335448306330726424159555670814437730980886552478177771286775654476121690918316769959560619302090434130233688337494280549233276776057356845612440084168071079849186877141663383464153978269943457001576322905443510037500866820957601605614615327628924713312497910550217215218503777828954423159553674088922345555815385406072430584503918*I");
	test(641,"Gt({{2,z7},{0,0}},zz2,tau2)", "-0.2903437053933784265721176420137263234145120483750781441249760316089558110116799196121004652789582809390548360576120387434588755412683850997454058922590020941461495538196720258805128506723557942729687476111323292614356684299814197565528122657221926530416489566846540391393917055408020225784914809461614675803318133968860220607554828247694158118678267026691986153262602627102362373805625387929574771333659012433564126259972232-0.2321441145200131154865350080990244745926133021180113807994188093471521796419846512583249833534180039352015315483092847558180703594074818866740142094733029524929195457433082976671016472002585152058608201188410197310239538673154842496821939878915329711133302570337153336666125118450926833666611252727793549537847007623418349124589068943447960437030375871578082292641078712559317122243964326111995456568867398473955538324066052*I");
	test(642,"Gt({{2,z7},{0,0}},zz2,tau3)", "-0.3108877514623628675755857633455584659188195967785372673082447064002122507366359455883510770207468421082881889624152483596330683692603220164842340729922823824991835452042304535096176699371008658475863626742666444390016847437752290167716518870899816790713639757618848318581427019630913862019255371992418250204844089837303455415568066741005270106280133822941272588984936872396557534779845131303275805448560630742189197174146477-0.2545640285605471520112750129472227848390504597254959226879265279431287200758698906808974794032548331676711279182168245609921044097735762328095183890473191885832587703588045196757370226436411933909452623075350532788864981410567333383874792199875645312076530517422406073299627186092706673595355684757424329811097968627073870150929629748451106588668179224033608800249995998459331778511341556491543396085621888471577531574538462*I");
	test(643,"Gt({{2,z7},{0,0}},zz3,tau1)", "0.2123912430911339479784369974632421009448737449784276550110988498918393701725862814617542924543109739487354663039369825683531802009384679033954249203122222831012491558874195873614372153413218857975599577811999482688800922192699474326025341273952226734397685121134508949602687510572142566237013203424422589086362830532567093576172168514265720735619869930565013386285316731156794123654987362960813359712982936909556868578425265-0.0217284760550275974725882228425692178598761218202338689536168959543334795464399137044709995163362083228594475481538468224833257277103952356773105571034946543057374886156686667511412869538049119339647822566894963720166597011803081186211037932410260230765662210252349708625599291638696677104813027859115894473483409477967155432664090761480239360853604507231307575614842797513821255756267247322430728521304969105148634458203303*I");
	test(644,"Gt({{2,z7},{0,0}},zz3,tau2)", "0.2504354034633388135515248269163688934515609126266686550331669745663269589527928585006599608889816116012845797404804567502854684994457434896175523422579990576187238773849150136099923664428821906874595957069828517311401593852996122774147623988675687577559627026131458078377264066944234314843018569120918983500297981732722766773463365522506835507267237489256796940966152895113887411914917983582979066974237103603604694977891645-0.0129983915040748713818854000611461844825438476325369561340190630537281874013864974698560162047949969345925421700033154518182184053149588270288044204019814572647498710779856077565431665460728754482951271849946326655395516710060075256375103647023796269012994352352279890654023112985780207904577017609926351301613085826865854787989607385959392597067492050508303464446842485129102127890058490592944287941234947434668003426770587*I");
	test(645,"Gt({{2,z7},{0,0}},zz3,tau3)", "0.2281005689236864898770844913044859516691564682924997338666132057367303438318616716985530113760640096650970938454220826116140164682363806486203656566714208369542227566921744316417526352469575962581551348206074331063947527319540389720444006614575403792827616513213162495874976906482672274980252220073149036412626775945406049711329039034686474499032819526564960490653769463202850828282582722517304523568973712181114025488019834+0.026581962130961401873270867630781036772617434092886243506087238223631773268438216417219995270738812694713547548507707094246175769011588840535973488066765831306226790708108334725958513434477538792380270954680596790242355679665924384103606788089233711048430825302364067663764672230825222241923055222373774998050060588813997863979548018104398001454784067733875510592444616307018025957758826448878986174383520588829315810436112*I");
	test(646,"Gt({{3,z4},{1,z2}},zz1,tau1)", "-0.2015270633318657106878437535214018410437256631680312122493757781430799500348580747882879043419714935211873649324212584905717502985099173623994074737182871251488270033940410239074372811919187584849161081636203566646890398547198718869233501811380097516968093459433935891584134861473799481033298960616381484642114927733057069575928554016830100069587994711711331699983098853340219279760694934908048137927022463895168572791720236-0.279069176780699859399700823855253046317977290055159438939111906597665777769671862994686918460329161339695401669653198562156153905524215359867261600699746689841296211068515344555320471299826033008048809178741623527064893247973059925005660675647232921668796955139454444466409795372273143927717344449924713257366502389918749760253728721857597997674013767427102547916402367009660050979871288582600641557752558878227315114403979*I");
	test(647,"Gt({{3,z4},{1,z2}},zz1,tau2)", "0.2961968507063106723031058106629196074595644075117501468235770631599362880848181289206307366630699263269257076199145422273573095893661493530066066563519159126143014437057969542096246571346466116429537778005359454255278386971107926656520010728751059409386445081082352471407201489595710434017209040583092659847121014310324810122617084419861546482816457007167930785737936582383696388560378526017788431711447023978388837386502279-0.1730245131812748313412957634530055053998653053824220653411591732398012190746190463455836941994700725441133931089593706539447321609784518239119173293332891290667762905280855017235371866117793528777351304193983914528048928433109659867928392186224121126794861528067483901277490644229861227275336833350689575832804795834288844548302920351275848700319339190012437390478509548821726654960666678812746424959898702181451291043911794*I");
	test(648,"Gt({{3,z4},{1,z2}},zz1,tau3)", "0.0020951760917471833673863965039553013172523417198902776726740606125968629651532403584770369848242824312521175684290625207845092007175487103921438837185749949379629913954206067677311015950769131875231845208753643363585734547074984202927571220317732809546676058457170615136968591736886039180471362183703583524811256875284157737213424542842743219285317762062361079786485190603199728136609618715446305077402448416404470307384827+0.3464188545513544644633491900771497442766466915158241186716142075184597394147339858764138314845737009256027867547245062327123590145827402211765947500443825062096904291548273221208219976241496363992330474104010014814928128767800858585877054932611381716974791180098858109016874902234197883742621439188458944007186381764079606165816967946331640955177969002401768573834594874198943537419322801213719001402059306283719699193524539*I");
	test(649,"Gt({{3,z4},{1,z2}},zz2,tau1)", "0.5156828190947177782378529646655296143738628624277720907546120745143837733933453050809611176142040863585156912237227045106094332135367386932125037522578419065848009404038391275572904413789715875705942826447113746156077362480963819572196273664934356609117181741796388628095222825671079506253371777149051020135821821264273121852673645517556361243893022368325385012914410550161069486670921608816617192005308547228145821173212246-1.5057209749120672038284909552239074148855479067590589250065864973636285609784868533113580053915116238197543669087688199130017922858575539785338567037989773196828408844564375841338546040845457432871957339771283089934505886429747296125735405521110977096605361198660193802900732227216608881341877803668966403559224887274400057145774922112468199502553710467150836395926321617147530930412589297804109004761064116379359904199421466*I");
	test(650,"Gt({{3,z4},{1,z2}},zz2,tau2)", "1.4511730350072342589578932025404403867386598497056012167382320350204997060957640506676987833089980609843163981802434366419197071640210839461314182571245642706024450309775721969049484991151786732521091099832994597026109300462443866013497047676984433708365396768718506134726101233232296879643200641956061907551787193710812918744611314299501960159667411127975878142171597023105076808898368872946230276555305694292984254135020927+0.691371323696688331611395957364596860378843939844399943559482790621519564201550420216003134879857661898708981803545122308593046048249804061336991918160968993124802741149296046770782813514712579315651600132622572147810591752079909091292936657107982201215311767536982259580541220876213248987769731543335376337311194841359060312778608661967955125962156540930923994435019303888852737138787614263261833407751553251696096468437402*I");
	test(651,"Gt({{3,z4},{1,z2}},zz2,tau3)", "-1.3116485122242038469885137411932126808103919708911834062339150369577062644647456676890218947390477274106970855425193874058199941034144675202740237844044105589058436991717080213982644362728038169285170914334555358351453129072023229815167066615843776480491339071372541488070853857622856820915358333114983279017775495319936997889775126901574385067127804782889533622392823881156361860886715789040741129630409606591102113447843473+0.8905441571680563982684589121801414348815697756129229378577671379409329739869277508913871358777779515458308603562071140698525619510354691028724931968590572454088536093060834239722103768035386584415372734251214549898883757785785038085113828149270112251565528781707990720689471278519805244325461880512988383137328403522791643179536585529819312199894378351956320280831153842585787331973080595131242863735167724247098615471037428*I");
	test(652,"Gt({{3,z4},{1,z2}},zz3,tau1)", "-4.7548137523071319550945438947409564521185729880807989680380316381441504889596775845678025711523766974522725029414620605385542652456627381849336492010894624966674814706910410546085318200376052746926946056946112663296311287335680045559516206078677093034754754584084007059444707344140792144875814001758486753648714614440184799484063552742684818751771299993591777949086105100761291591789879100729459141991794799895698681780391206+3.1889042837446766979496052716235011067824634929471707876607363734028777685283540160450733461892381148548669672276160861406579663239634699132134676691591360157234520496336539999728842886782897825119292568264848513615881855139405562512307381390554001085897109825923539095618638625479395711963790470187439183288942629090060772731910359995177708574510859451138642372552423288770510528647561728161225906447052354250975160980377365*I");
	test(653,"Gt({{3,z4},{1,z2}},zz3,tau2)", "-2.8870028063233606922789798219353085211560578238383564346047145882943187923534936612218853996295194407442745303528264449893674335138475150158317728230270575319594452432132685758798337654069889599714540273075509254891764442913311761668829442523169002867171172193726772818692748428812896650895533327729129388116065002404738489100210042285383142952452885290900327775469339838677179192739412749687398127847848349221191836097689109-4.3368358700649805908419886098347676597742656336756542364505281656979357307526990314141274492824294220697138858384776689381861254248908677974886589725476782884769520189963105592600231488927868474604255993344823390038460389362779291547079036857708764850448574152838098538044446956385507869662982033716556153704038252666812297156295100158402796570877720839681768715818207116905184091426716688427544725378868237418823824547960581*I");
	test(654,"Gt({{3,z4},{1,z2}},zz3,tau3)", "5.915392235793901217324848438255029026238150492621366003462557717046443203626422522128429769750285321424037636671351401583197587918983934370647733118744665329350913159158511018911768878644252669040371660781997464252599309890797473351628512771850555889167066738634066165624420625091486852222368756170198739417243527680354251722801379646465557384359748387460771737229289812933975313226382188637403351486029550159268604327694249-0.8781711496201556500878062381186935606440477843665863904600795276365769950184221432401660060742994012223900661605670987230321568092373654800554306786486009967293584422483409369189188692638847597342394127706759560793916164788565926884280970392864678993745773474670601697343637027455356362312202263944131016825435422713934252829058332231473015654366022286594545805905510614921726360982606329725910367650221011380105052081560172*I");
	test(655,"Gt({{3,z4},{1,z2}},zz4,tau1)", "-0.849369701298178365357680256306389870021242338638849992757578391534929674002509584784828698010786642873220311982420742999670886398989452249364999557648645303772852754919099938893129456640846344143987336911327874157320442831359437024464643876206450335385086121206294770868603020502690202350293917206447008112179318238970461214712685804206940207776422242581282119750092189202970740662211431532056784288686772623265675011189339+0.1258648687984186901858257142369378873415302821125217662999042502611450124237872317110590474130168366611807080473100736040011615676344336037959541886410895197769797278767918364513804668461451106381918066014379116003403039094738394349754428435772650806556757372955400283878215750002906491151269857457886514922180080465264140257967508129657364973757350793971258874795671857732703376430026633167490576037226121513750702686180805*I");
	test(656,"Gt({{3,z4},{1,z2}},zz4,tau2)", "-0.0538655774390964070352130184004084976912042959045757285390537627017905636595417098649210173898406012897823084762948234272418999483484510949948784211276628893412581193644386015969513062311044514989251178060729817900648041784255242157706909991819370720542041415981981409943103534643600054739584904231911700902917219838829434450590809862553032192177472336930713171256570771542873043035520456536649383191206211879268334343285286-0.8664868261052130203789834162143482909571224474098384758728599760390574595719306200715306257544063710280846827987525497219610519793517615538112281119525429120054618396392030625589354506086644350756301181384214936330462926424411926656777939875129033796878036624890120672145842433108258469086041494190029308987592701220001439066616147665170842552700760036626654710022004168619803477968039643052769233321211566360421840823938075*I");
	test(657,"Gt({{3,z4},{1,z2}},zz4,tau3)", "0.7833946435000721194683579276867132139439083190572560786187399218823773917088777511884775524010755470971527596763529238312272281040295630321778775746837135992839785604467427535098220125533978754347966460645472584488993464495799476791128322778564997116522418886511513960906530488090321451685301753647634479175689620595562179052602757477356735482877492572024330333772012245290157570429778704061553233334511844047727656671742617+0.4086328404562549095998254977398568649489888139257236502694929507758618000307499503876656843408745666751911913829985462259152140503222174858052554904604085726701572950121749456201935722234421742294738514909854776024157515809226353464649082830045623771455067506084235349150394630171953937978326372453324858398571212113363293653286227088072770121694876429425880294130641278635212657978383424002391532894453896678281213175297617*I");

	return failed;
}

static unsigned exam_qExpand_sing()
{
	unsigned failed = 0;

#define LIST_SYMBOLS(f) f(tau1) f(tau2)

	DEFINE_SYMBOLS
	MAKE_PARSER

#undef LIST_SYMBOLS

	const ex points = lst{
		tau1==numeric(1,4) + 2*I,
		tau2==I
	};

	auto test = [&](const size_t id, const char* input_raw, const char* expected_result_str){
		const Gt input = ex_to<Gt>(parse(input_raw));
		const numeric expected_result{expected_result_str};
		const numeric output{input.qExpand(&points)};
		const numeric diff = expected_result-output;

		const numeric digits_real = diff.real().is_zero() ? -(long)Digits : log(abs(diff.real()))/log(numeric(10));
		const numeric digits_imag = diff.imag().is_zero() ? -(long)Digits : log(abs(diff.imag()))/log(numeric(10));
		if (digits_real <= -Digits && digits_imag <= -Digits)
			std::printf("  - Test #%lu succeeded\n", id);
		else {
			std::cout << "Test #"<<id<<" failed" << endl;
			std::cout << "  Input:           " << input << endl;
			std::cout << "  Expected result: " << expected_result << endl;
			std::cout << "  Actual result:   " << output << endl;
			std::cout << "  Digits real:     " << digits_real.to_double() << endl;
			std::cout << "  Digits imag:     " << digits_imag.to_double() << endl;
			failed++;
		}
	};

	Digits = 90;
	test(1,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-8)*pow(21,-1)),tau1)","1.2632462597846006137408457618334161827709162282116698345621016506129293852121086629489810930419040796434693328812222+1.8866914108225177364372781685451126373455366079155434561083360539804650254308248437511796987706902619144031207630645*I");
	test(2,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-8)*pow(21,-1)),tau2)","1.2821469220144560318638376375830875902984252471351152770648807515733926646025872594991034711056145369302825654747392+1.9002717468142343454873245407932140461668819439530827159752074149617007400835515881723997497959404260502101251795906*I");
	test(3,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-1)*pow(3,-1)),tau1)","1.1365476290326253754066131004160842889173032157620671809257400655477394325042123707326461993045416419796904016082532+1.8681495026739343229603284179426928098120970134650978779579627730977630614300431245421334070807093275770268962380372*I");
	test(4,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-1)*pow(3,-1)),tau2)","1.1516088005813127986019773345887325568495454005190119536050723165767443241581829158510032569037845454931186826782805+1.8781848747789590353213167129229384248206066781303306019416817696804967992994194538615420236881907126843963415246534*I");
	test(5,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-2)*pow(7,-1)),tau1)","1.0175253160914387108420463660143148106128593013633671850074459044730606692758907837812248772222146003193669112182823+1.844445632800951032435698389449392445077773606951075823420531859647091361641031620570997613417609827454724694646555*I");
	test(6,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-2)*pow(7,-1)),tau2)","1.0297646353257974148847673551135240730929137829844618005184509852279913749648560611546047284796522169816330027509228+1.8518105087293284341273179505398795126406339421133730823962844523912675402935585340745353538792655689397923135205941*I");
	test(7,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-5)*pow(21,-1)),tau1)","0.9085338911798923180775800835442606320497854538578210718953823319762297915309231741697904102529245617304069661914952+1.8146720379057830353809757152564262232775998755448289895091387028780149067981447570865546305668519039101148399935617*I");
	test(8,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-5)*pow(21,-1)),tau2)","0.9187158889284009848749912680977300508840725149341185535253004945388630145698822153759521992851955688182615048724429+1.8200140167681216404384141930441018288548982336998210423013012299709834966819746123729393772171493585389072304245737*I");
	test(9,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-4)*pow(21,-1)),tau1)","0.8124284128646704491289025197967846949347735590165132185553409116099704439612947095912969227981521706691750913500326+1.7781196293770327773633519869023906077499107761389619955756696749124263918467498142826917348654265166130473189262072*I");
	test(10,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-4)*pow(21,-1)),tau2)","0.8211327928162059104380774846327328012348751777370526002364286815456978653156560110001221078585269701673861018449771+1.7819116178308979383086152225745066360836203610906686453572826494111740577929853763894380857993031802942597802030117*I");
	test(11,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-1)*pow(7,-1)),tau1)","0.7324731876873205491143999601787594375392322676586985144991359335753509721150944051665309778850326623105271453420737+1.7345275365771486989313169665302269492065216053956091754500000939113851226061698722435775009855508495904351399080599*I");
	test(12,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-1)*pow(7,-1)),tau2)","0.7401467652051187069398426480570277888108057238153701990843620226649903225002303303994904882839953953125025132242866+1.7371069469511053030081153619342846281871680518786636808404809988682427608657642478804175587522726323257933857099832*I");
	test(13,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-2)*pow(21,-1)),tau1)","0.6720814643761557322236945777312199428213726807461334498585976520642088890479532539549720194652219948028480021311746+1.6843628835869895953570299763089658110790475455152692998242816786213555595475324172028025905349555793310664448886294*I");
	test(14,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-2)*pow(21,-1)),tau2)","0.6790782851040650662227102166096993284784884701339637536427280992000361124496755112541036996283309429852991174797944+1.6859601640739432617471676632014531045929040066997070017831973831604045383039665123483487671891087177876365003870992*I");
	test(15,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-1)*pow(21,-1)),tau1)","0.6343580245304013257517167961004575945117848699379849596491622783753103842019373625829635061570649412259065131999799+1.6290222150105052902957246561716116204050542523474918093470808331121412792570650135103324438812579067939797974254433*I");
	test(16,"Gt({{1,0}},(-8)*pow(21,-1)+I*((-1)*pow(21,-1)),tau2)","0.6409711998257291775784500756192321241622266112826500986436383304206118250784702181713274478961877531205653231178936+1.6297806037053405676380086251472933811884252157408143246729683520169752061469637412020081042998298637811097339151972*I");
	test(17,"Gt({{1,0}},(-8)*pow(21,-1),tau1)","0.62151556127522070253470523243449490605521719507537645857987718974191050957247019418292841657225393325374690634615311145733518098055452+1.5708084142852927404562595603141440757239343970426262181746801122793263355670950415575606840359992571257194016534019*I");
	test(18,"Gt({{1,0}},(-8)*pow(21,-1),tau2)","0.6280036500683374494988674414813647297467745787271588776375931157919687275989755149041475518417209252105144848833928+1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432559506*I");
	test(19,"Gt({{1,0}},(-8)*pow(21,-1)+I*((1)*pow(21,-1)),tau1)","0.6343609058725608362300400742719480235921039433656234698957367204852658461492378139710241093791003650001733761609856+1.5125950746842283342071277893301138318246695016356639066477400027203852859655034292934873925856824235617937154515937*I");
	test(20,"Gt({{1,0}},(-8)*pow(21,-1)+I*((1)*pow(21,-1)),tau2)","0.6409711998257291775784500756192321241622266112826500986436383304206118250784702181713274478961877531205653230917446+1.5118120498844526708246347581322095030087441836342914963019762402908412001392452574260267210422872042010383525946958*I");
	test(21,"Gt({{1,0}},(-8)*pow(21,-1)+I*((2)*pow(21,-1)),tau1)","0.6720874869288387715164683316392460679326339813849485791278471704151556484417985853090445518679557194808137682453815+1.4572558310689937469626347716029872623792231097224856464025769964390113346740234810535197509195591970895774349171885*I");
	test(22,"Gt({{1,0}},(-8)*pow(21,-1)+I*((2)*pow(21,-1)),tau2)","0.679078285104065066222710216609699328478488470133963753642728099200036112449675511254103699628330942985299117478963+1.4556324895158499767154757200780497796042653926753988191917472091474118679822424862796860581530083501945115861235996*I");
	test(23,"Gt({{1,0}},(-8)*pow(21,-1)+I*((1)*pow(7,-1)),tau1)","0.7324828946247932182920272884057177753627422550474204661642202589168211007709914229591708734162305004943773846628076+1.4070936953944978108624381891222626756212369094953162066803371720641495503276071214000090517511584453916454080742699*I");
	test(24,"Gt({{1,0}},(-8)*pow(21,-1)+I*((1)*pow(7,-1)),tau2)","0.7401467652051187069398426480570277888108057238153701990843620226649903225002303303994904882839953953125025132672389+1.4044857066386879354545280213452182560100013474964421401344635934395736454204447507476172665898444356563547007993165*I");
	test(25,"Gt({{1,0}},(-8)*pow(21,-1)+I*((4)*pow(21,-1)),tau1)","0.8124426796560095258424746388279793403629996078733569823130231957758940173582891577126569633583201286786613379292881+1.3635054393014903257683836365552886672221866556959549048154946320756025719134238803557355082886644070001740438520428*I");
	test(26,"Gt({{1,0}},(-8)*pow(21,-1)+I*((4)*pow(21,-1)),tau2)","0.8211327928162059104380774846327328012348751777370526002364286815456978653156560110001221078585269701673861018485494+1.3596810357588953001540281607049962481135490382844371756176619428966423484932236222385967395428138876878883063066162*I");
	test(27,"Gt({{1,0}},(-8)*pow(21,-1)+I*((5)*pow(21,-1)),tau1)","0.9085540045475759218340011254220166359678474946528156251336817365958462960735726675127675072990190226971502696842867+1.3269585329035762919146278130997679288049521100524111442656072420811548870269908815982890159981676679582645827093487*I");
	test(28,"Gt({{1,0}},(-8)*pow(21,-1)+I*((5)*pow(21,-1)),tau2)","0.9187158889284009848749912680977300508840725149341185535253004945388630145698822153759521992851955688182615048787417+1.3215786368216715980242291902354010553422711656752847786736433623368329096042343862550954481249677094432408560858614*I");
	test(29,"Gt({{1,0}},(-8)*pow(21,-1)+I*((2)*pow(7,-1)),tau1)","1.0175530900608972887041399041272243095762492480007765384603174725870352240551079914644688351719468685256974818133604+1.2971926018006039879808424281197650100927863061354745299562415254684736762785033023641423575833188018296876922009388*I");
	test(30,"Gt({{1,0}},(-8)*pow(21,-1)+I*((2)*pow(7,-1)),tau2)","1.0297646353257974148847673551135240730929137829844618005184509852279913749648560611546047284796522169816330027469062+1.2897821448604648043353254327396233715565354572617327385786601399165488659926504645534994714628514990423557729895607*I");
	test(31,"Gt({{1,0}},(-8)*pow(21,-1)+I*((1)*pow(3,-1)),tau1)","1.136585568539266172735655318116561290928480727695973519194472366900672164659949703719368120349272516490881976513148+1.2734992485789007908174684157809172687630170749241736831991764866401126530158926062249766491012427718200894831716701*I");
	test(32,"Gt({{1,0}},(-8)*pow(21,-1)+I*((1)*pow(3,-1)),tau2)","1.1516088005813127986019773345887325568495454005190119536050723165767443241581829158510032569037845454931186818465412+1.2634077788108342031413266703565644593765627212447752190332628226273196069867895447664928016539263552977517449860647*I");
	test(33,"Gt({{1,0}},(-8)*pow(21,-1)+I*((8)*pow(21,-1)),tau1)","1.2632977865940285547388357581029209603683383857249697555945837592371847419536811365964449128631056962279654836012476+1.2549716584378837324527091332465003861855608510133553501711419337491207826705649744396580356056494244182207521198696*I");
	test(34,"Gt({{1,0}},(-8)*pow(21,-1)+I*((8)*pow(21,-1)),tau2)","1.2821469220144560318638376375830875902984252471351152770648807515733926646025872594991034711056145369302825647344302+1.2413209067755588929753188424862888380302874554220231049997371773461156662026574104556350755461766419319379613309914*I");
	test(35,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-8)*pow(21,-1)),tau1)","1.2442543601317745973908267831837085513444047231303774925759372591143276285349366268125073261070419018760623510822462+2.0189479988271343052784776402321976482365312313482014148089454327446915996609050892930930009250318426414340514619179*I");
	test(36,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-8)*pow(21,-1)),tau2)","1.2584656935605574665768971775574231311419661283889135121416350578117040658685981879131813121875300239938376526211716+2.0363393046940373014932468678300929467584059861703964924477333175601950720254600126829140682063519831501241437616918*I");
	test(37,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-1)*pow(3,-1)),tau1)","1.111945457082408379760346740752562981217215342384012359042340151529925733711882395855758794701535234722134592343996+1.9942915449906676288407105453683223775274338578239244132226340465946998429037726565294150606201414966755590544707558*I");
	test(38,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-1)*pow(3,-1)),tau2)","1.1234830183906169845029487267618755712096132195929052438454683489426649984396172193901824609703044271007694311542619+2.0071286397934790134622443837619046204593026718229842745117701561966604817939668602345778050796859045509290954248688*I");
	test(39,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-2)*pow(7,-1)),tau1)","0.9861017327290881633789747908041365107403723209230531057875151223721377817491470470068356771887764432190494509499047+1.9623729351000645512102438786322264287442619673067069587218060470078819664112415450728699831200229823999161053860398*I");
	test(40,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-2)*pow(7,-1)),tau2)","0.995683441263544877209191483641843559093179760036268280170779220617568526165421610345848839518163136649752733546686+1.9717864963838165422593015975148112592703997961182888861828592527943072984534612123891568641574884883305310102154842*I");
	test(41,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-5)*pow(21,-1)),tau1)","0.8691496421559356933640437809728310461145496997438635897517888420481919729180177535075498316197163972233840369798219+1.9216800021868261622037750927412750130753209729674473320912594720609206853197288221685035971597039623952509044541784*I");
	test(42,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-5)*pow(21,-1)),tau2)","0.877310330291295093601840978493300792328764533866206797160789805394446891483851341077902911608693073943202273490297+1.9285044296595658253726491269683417847390204045498681573522271713232816873141190430334290814580033387215057119347585*I");
	test(43,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-4)*pow(21,-1)),tau1)","0.7642849316678706372742464812977728173169783218646553818744703029073276324279591801069677960375523670107932999328269+1.8708743921168128926742431566683876436925609789596721394613055425170198650196775306973543565332286345984903677502363*I");
	test(44,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-4)*pow(21,-1)),tau2)","0.7714275233938483304523218056921665341637172251900390781931597265321806163634908785727395401238263632729707159882997+1.8757174470365792732430442144269121328545882430076240822878996424805790568806393751929488045961795182931860825066512*I");
	test(45,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-1)*pow(7,-1)),tau1)","0.6754892311200082723508136800821610017216374968885058377937415252288837345907451951469643478694988261568587681562768+1.8092004263544544123769840543199064303269183235276259157809419827611400769486784731469246490289776782119457532252232*I");
	test(46,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-1)*pow(7,-1)),tau2)","0.6819227455774074804666797706313270799531266855239808835090607741412614943016946845500741829951481161956325763155206+1.8124949623477236800987363460135864045329052116884425326852726847110595246276327500076211471801617699577291177635275*I");
	test(47,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-2)*pow(21,-1)),tau1)","0.607288475880861403674461541480823412917117867230192058870691599494437174296998716526105878127137085653584300234934+1.7370257961097272781110923043890588192856066442031825436869832496703366779170392791259583384852724665575702577149586*I");
	test(48,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-2)*pow(21,-1)),tau2)","0.61325683591971019562966515088262417248352853462794161909312979298174326609226019914347000645092635340013081490042+1.7390671127258782464820731094058693218674443345591527816595036881790871181447624184599905385162039693862288462159958*I");
	test(49,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-1)*pow(21,-1)),tau1)","0.5641144609052458942802288352493018671074388041359528416857358991345131613255677611157366091634465789353801563374704+1.6563306014842209329615124730133039923733550800466961956621473016944152485832404606536625657161316614468127224054518*I");
	test(50,"Gt({{1,0}},(-1)*pow(3,-1)+I*((-1)*pow(21,-1)),tau2)","0.5698190257957779337563173166344555739768015577136480332968434466673955965138043836874707699640953370803668554320546+1.6573021685587459438523368378351005847973190832415993434619416120876711205054188970585621566250254209575860826366532*I");
	test(51,"Gt({{1,0}},(-1)*pow(3,-1),tau1)","0.549306144279327840506057223846309718339154392996227358502548773550557343600806854112691572234937156993417089024574557905204624610204798+1.570806788821965118983261528813064768451932882060761612449164154310934217838388598521730759964893978109100396647906*I");
	test(52,"Gt({{1,0}},(-1)*pow(3,-1),tau2)","0.5549241851709482843473695151794916211832085133066407393312846847550531446161866679746976448017189000689537580032693+1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432562196*I");
	test(53,"Gt({{1,0}},(-1)*pow(3,-1)+I*((1)*pow(21,-1)),tau1)","0.5641181295549411879731572419997513308949941546424679815549716146493244187383718948909931749418924893299945455018164+1.4852832906832552397982159479543589671022959057414118787926207806130522134602474804659691246198264895875873230711104*I");
	test(54,"Gt({{1,0}},(-1)*pow(3,-1)+I*((1)*pow(21,-1)),tau2)","0.5698190257957779337563173166344555739768015577136480332968434466673955965138043836874707699640953370803668554495826+1.48429048503104729461030654544440229939985031613350647751300298022014528578079010156947266871709164702456200387516*I");
	test(55,"Gt({{1,0}},(-1)*pow(3,-1)+I*((2)*pow(21,-1)),tau1)","0.6072961440559140340089040980724470640654922957686104580684951039950268353536038354852989751034933074517974251528298+1.40458906799527784474251995102144372315862362828828861404118817607159738403671081338843384797127797456824039202135*I");
	test(56,"Gt({{1,0}},(-1)*pow(3,-1)+I*((2)*pow(21,-1)),tau2)","0.6132568359197101956296651508826241724835285346279416190931297929817432660922601991434700064509263534001308149078376+1.4025255408639149919805702738736335623297250648159530393154409041287292881414465801680442868259130985959192402952259*I");
	test(57,"Gt({{1,0}},(-1)*pow(3,-1)+I*((1)*pow(7,-1)),tau1)","0.6755015904134314548248628986090269484624861012068795883431657388366808208509919914288628956855137655147330668879865+1.3324161547611471707132485322714395512057582486363543944064430932661085489052596396171577538363819440137059224024997*I");
	test(58,"Gt({{1,0}},(-1)*pow(3,-1)+I*((1)*pow(7,-1)),tau2)","0.6819227455774074804666797706313270799531266855239808835090607741412614943016946845500741829951481161956325763238502+1.3290976912420695583639070372659164796642641876866632882896719075967568816585762486204136781619552980244189687462862*I");
	test(59,"Gt({{1,0}},(-1)*pow(3,-1)+I*((4)*pow(21,-1)),tau1)","0.7643030967648301495341298865343336216708197052087361670074484025808514874237735388288549161631228998922912535061823+1.270744805939706886917398319735229884840049628292276184191821454303872432423900997272690923952347116976030121350205*I");
	test(60,"Gt({{1,0}},(-1)*pow(3,-1)+I*((4)*pow(21,-1)),tau2)","0.7714275233938483304523218056921665341637172251900390781931597265321806163634908785727395401238263632729707174767364+1.2658752065532139652195991688525907513425811563674817386870449498272373494055696234350860207459375496889620040036837*I");
	test(61,"Gt({{1,0}},(-1)*pow(3,-1)+I*((5)*pow(21,-1)),tau1)","0.8691752513672493448520738997735439264270328032371433952297389685382978471463829679060573896528408536543710477023319+1.219942948762939592528431602011960242293720601444727100010005780859647683274832016722066313311585514455327120438541*I");
	test(62,"Gt({{1,0}},(-1)*pow(3,-1)+I*((5)*pow(21,-1)),tau2)","0.8773103302912950936018409784933007923287645338662067971607898053944468914838513410779029116086930739432022734702165+1.2130882239302274130899942563111610994581489948252376636227174209845347189720899555946057438841137292606423745757633*I");
	test(63,"Gt({{1,0}},(-1)*pow(3,-1)+I*((2)*pow(7,-1)),tau1)","0.9861370957505436637646693009913888594271553213434411696510398779062989899227296702027230865016881844785327375017044+1.1792552431688673378222602503611506946352996011083299986719610461257134361219700373345531158695769855303470204947163*I");
	test(64,"Gt({{1,0}},(-1)*pow(3,-1)+I*((2)*pow(7,-1)),tau2)","0.9956834412635448772091914836418435590931797600362682801707792206175685261654216103458488395181631366497527335459967+1.1698061572059766962033417857646916249267696032568169347920853395135091078327477862388779611846285796516170762948805*I");
	test(65,"Gt({{1,0}},(-1)*pow(3,-1)+I*((1)*pow(3,-1)),tau1)","1.1119937633063183660188299478635504165273003527368428797422862236290186190405981581320608895259988692284567432440011+1.1473438064754770305811272436181653697535879723098950032690279955605576082025446177122525949191006468700769844205207*I");
	test(66,"Gt({{1,0}},(-1)*pow(3,-1)+I*((1)*pow(3,-1)),tau2)","1.1234830183906169845029487267618755712096132195929052438454683489426649984396172193901824609703044271007694311294191+1.1344640137963142250003989995175982637378667275521215464631744361111559244922421383934570202624311634312189910862937*I");
	test(67,"Gt({{1,0}},(-1)*pow(3,-1)+I*((8)*pow(21,-1)),tau1)","1.2443199662983962589106378903978166133057717962256569693952270444459479970028885832333883667789188937553868961957462+1.1226971186652188138382082823243809379191419555906207591032852048546011459299621200044584611571302833146004604847062*I");
	test(68,"Gt({{1,0}},(-1)*pow(3,-1)+I*((8)*pow(21,-1)),tau2)","1.2584656935605574665768971775574231311419661283889135121416350578117040658685981879131813121875300239938376517599846+1.1052533488957559369693965154494099374387634132047093285272112747476213342607489859451207571357650848320239427494438*I");
	test(69,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-8)*pow(21,-1)),tau1)","1.2206639659765416462315543848043211368009151152260301488967344138895534182037261256776839622121548735502295154122052+2.156991182565104776423036141790338492052782637286392502169698968568820426468240539921616402183611595068322769457861*I");
	test(70,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-8)*pow(21,-1)),tau2)","1.2292308355537207951046177286550798117720991416749447097867014355332495594909645690040037650093104435163084142446811+2.1766943565881955485271601293797551381836274985819156744321308921667147221890972226695773303179389706288284066929032*I");
	test(71,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-1)*pow(3,-1)),tau1)","1.0809852788857733022400264949575427622756786095305745813208579876053149816252025075404318068620903063286606233641584+2.1276805791051430599434955074215114747680695986383389873242817217651166760794183707731340354795924259164625012055612*I");
	test(72,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-1)*pow(3,-1)),tau2)","1.0882927485122235894541220142147087234799443908568496309024669296878669689811148461330698207640758846837091742648939+2.1422038833171031747751123567797758509877982268729211174748062847516427573975267620046982642667124340386648861915989*I");
	test(73,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-2)*pow(7,-1)),tau1)","0.9459174905843525911200454653559635125549976956538301689903336662326277515338366236076081149432736649174837177126395+2.089098612309144703056303870688608740767279696533925591566104856814057591809558100729205605001136063311625539136032*I");
	test(74,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-2)*pow(7,-1)),tau2)","0.9523151351022059475825597522732810878100953758198629981561551547656185236784107830171890156811913852786646455513515+2.0997379470421577257071749143347697848460673026435656252434578652669589246096600960217987863255986525848855363017899*I");
	test(75,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-5)*pow(21,-1)),tau1)","0.8178160786702915765680005185446376318606814626243761581713847442785886540959618193822448805786179524635981376222784+2.0388933030814749809748835031865277321164309291135226043598336588833186801901967254346376861470666358760962533934545*I");
	test(76,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-5)*pow(21,-1)),tau2)","0.8235587590455172353624229294509635325378785153260023993574711302510588505399978827434026255241279922790570086366222+2.0466009412799696505105296501365986579004198774337642733163522821541253417636139510857725513740833835068928606698563*I");
	test(77,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-4)*pow(21,-1)),tau1)","0.7001750081185130834687814351566830489454540428838731552509196999974518769463822431522031547857365459179338870352701+1.9746851715713250620161696461782176897801451945084446788023724820842624339883335196089167767755944273157895567480137*I");
	test(78,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-4)*pow(21,-1)),tau2)","0.7054515145576066101408341116613283541512628775671684885798672356764081456591468871939490979671066205828622712949223+1.9801525887571336339617637387075728407108993359793459246823101509163162690592780444486574521280721441741698423177939*I");
	test(79,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-1)*pow(7,-1)),tau1)","0.5979065992543500431405121982323816600937740116772475824236326446936174695334541871058645095423609205387076545765025+1.894653568055051301762182153533225561376642992145380741954307591195564320680906171273914618702781070563866987446951*I");
	test(80,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-1)*pow(7,-1)),tau2)","0.6028598228803196589276542684105006288508342291648944097040986376383268004725405386522831152155575915947409472851803+1.8983721626805028074321400417802710863580129056263495336906594150788807430795658838564191410014431878944350803642836*I");
	test(81,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-2)*pow(21,-1)),tau1)","0.5172878392655983226092321111967770627793115071205471062337132990062854453714689244031258080213210362170845010001645+1.7985155799754207518215358998095950160662194733622155026819738607798297938679840458635583104465120827050703674802615*I");
	test(82,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-2)*pow(21,-1)),tau2)","0.5220294101520072058863826414464297423639310040519126346545333288945347339447690113932449695839905951568402505498696+1.8008201077070908139519527678965982174724534175423072201855172471232916955700865928171052363932469746491194206733225*I");
	test(83,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-1)*pow(21,-1)),tau1)","0.4651404925503764557343574351419771710231464825619738236450453497561710476516356262084260115462189726235086537489424+1.6886756360757266674966289622377344871615203976705582442220112221119180692361923502398355782300776624694969628261566*I");
	test(84,"Gt({{1,0}},(-2)*pow(7,-1)+I*((-1)*pow(21,-1)),tau2)","0.4697618334137523171205256785761978540231313734156882685758995194820386319533735296152380433888969039482984935933457+1.6897739296725107088194644071688397863192612231639027720079377998598266600597441799505255262508404825399741895944677*I");
	test(85,"Gt({{1,0}},(-2)*pow(7,-1),tau1)","0.447031123282580211864514090407909478664958406509312502243335934543688000344075747379287640864119480216558544990231685853953061713198917+1.5708048534929651809583008266933174637179352979873394990423381966242053459168642503210313076930714079904265609081957*I");
	test(86,"Gt({{1,0}},(-2)*pow(7,-1),tau2)","0.4516122726407059762099996107866697390398649351293904483262041331475527961406919346187820133238052437678128553294278+1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432558742*I");
	test(87,"Gt({{1,0}},(-2)*pow(7,-1)+I*((1)*pow(21,-1)),tau1)","0.4651446225319825017848547434327323189624590452980488527220468167554445330554776072303983664400310425921389223379253+1.4529342108863500840892822226372770430310086011722904424784080310288906645068261159475178998404730038589597944060337*I");
	test(88,"Gt({{1,0}},(-2)*pow(7,-1)+I*((1)*pow(21,-1)),tau2)","0.4697618334137523171205256785761978540231313734156882685758995194820386319533735296152380433888969039482984929601997+1.4518187239172825296431789761106630978779081762112030489670067924479897462264648186775092990912765854421738969166877*I");
	test(89,"Gt({{1,0}},(-2)*pow(7,-1)+I*((2)*pow(21,-1)),tau1)","0.5172964717120158858285914881744838350273128876219190350281267315024540966971636559796943205289678542891101141564742+1.3430946995395493948776937552607156974294971294270441699518545968593935867893461176235088839307650305077994538957179*I");
	test(90,"Gt({{1,0}},(-2)*pow(7,-1)+I*((2)*pow(21,-1)),tau2)","0.5220294101520072058863826414464297423639310040519126346545333288945347339447690113932449695839905951568402505644676+1.3407725458827024245106906153829046667247159818327986007894273451845247107161224058109295889488700933330286658368023*I");
	test(91,"Gt({{1,0}},(-2)*pow(7,-1)+I*((1)*pow(7,-1)),tau1)","0.5979205127262652233083704628528638186754142255262942581374874295816403365225122621611477127742341370586993556549706+1.2469574756015219630405721989391050993894787393517776884525938757313736847390725033299927248893795185505696867682009*I");
	test(92,"Gt({{1,0}},(-2)*pow(7,-1)+I*((1)*pow(7,-1)),tau2)","0.6028598228803196589276542684105006288508342291648944097040986376383268004725405386522831152155575915947409472905916+1.2432204909092904310305033414992317978391564937487562872842851772289356632066431147716156843406738800877130061462374*I");
	test(93,"Gt({{1,0}},(-2)*pow(7,-1)+I*((4)*pow(21,-1)),tau1)","0.7001954574725119636036974699320952682281468467228624921486814444872783512555497294491311298856913561786564863451134+1.1669270367335227620991476490034062448285564589287822679383580851470101363771059026382456788996298075348410921650205*I");
	test(94,"Gt({{1,0}},(-2)*pow(7,-1)+I*((4)*pow(21,-1)),tau2)","0.7054515145576066101408341116613283541512628775671684885798672356764081456591468871939490979671066205828622713003353+1.1614400648326596045008796445719300434862700633957598962926344413915001372269309541793773732140449238079782441902975*I");
	test(95,"Gt({{1,0}},(-2)*pow(7,-1)+I*((5)*pow(21,-1)),tau1)","0.8178449082344258850742964321723475838821585828404789739280194258676620843998111170833774027441531501545653297548646+1.1027205754179933743916190217030665503971181431900837350595753149090421383833482513361645762115179346926311696322973*I");
	test(96,"Gt({{1,0}},(-2)*pow(7,-1)+I*((5)*pow(21,-1)),tau2)","0.823558759045517235362422929450963532537878515326002399357471130251058850539997882743402625524127992279057008652624+1.0949917123098235879521137331429042262967495219413415476585923101536910645225950475422622739680336844752552258388078*I");
	test(97,"Gt({{1,0}},(-2)*pow(7,-1)+I*((2)*pow(7,-1)),tau1)","0.9459573004981606292536306872542865691845324443143649879831414447977920698191248681837991648911867014308867008060859+1.0525175925662101765325690346655826010928475578740814305028290354907126820138082190365377671655296807701481384141613*I");
	test(98,"Gt({{1,0}},(-2)*pow(7,-1)+I*((2)*pow(7,-1)),tau2)","0.9523151351022059475825597522732810878100953758198629981561551547656185236784107830171890156811913852786646468711984+1.0418547065476355127554684689447330993511020967315401957314867270408574816765489026062360390165184153972625502074812*I");
	test(99,"Gt({{1,0}},(-2)*pow(7,-1)+I*((1)*pow(3,-1)),tau1)","1.0810396596069697031645837888462358228596525541883192656642885625808607189632967845605324261566316255743866612390122+1.0139388181432998973297177434620050970806138563043675324758989023450618642910131740590919170851275509777027424408994*I");
	test(100,"Gt({{1,0}},(-2)*pow(7,-1)+I*((1)*pow(3,-1)),tau2)","1.088292748512223589454122014214708723479944390856849630902466929687866968981114846133069820764075884683709173142953+0.9993887702726900636875310264997270332093711725021847035001383075561736488886822366233365610754046339434832003187597*I");
	test(101,"Gt({{1,0}},(-2)*pow(7,-1)+I*((8)*pow(21,-1)),tau1)","1.2207378221045447735379167204151650838119181842012146061066107498229815968879932826704268101989129056715434623019376+0.98463256097388800507922281195315341023119171251727890330098550016949829851067793525367867706195733875527680940438*I");
	test(102,"Gt({{1,0}},(-2)*pow(7,-1)+I*((8)*pow(21,-1)),tau2)","1.229230835553720795104617728655079811772099141674944709786701435533249559490964569004003765009310443516308414257861+0.9648982970015976899354832538997477460135419007931901465428137001411016840971117759584574950241780973533196798170933*I");
	test(103,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-8)*pow(21,-1)),tau1)","1.194097357450232160029812343002418418727804416193129289162960529420020232961262678508209075283450451362003927179689+2.3021829103486903800868993450490106129624241872443060360905496851404084622403953275509297837007770719996490433964027*I");
	test(104,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-8)*pow(21,-1)),tau2)","1.1965438106744371216097828598163255440443664073958626057203636217497583430169619227376151430229338129201308272318089+2.3224715343939299880559783860421361012146119833900293769661143719042891569584574345708298992827170247638137247143297*I");
	test(105,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-1)*pow(3,-1)),tau1)","1.0455468022171755878823812559329544160775615475488975005298235372425448963752775149311589212384180678655489014701693+2.2702871224200583218079556813358829641366063151841657391996631282044302558848549427042894680435930438649861008777736*I");
	test(106,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-1)*pow(3,-1)),tau2)","1.0482813590580865623617127272432057111966376984580492908192901913703824782205157763773878000626858395443526258223843+2.2852190501553155817716900140970231948587702977865833009395635913752974607999231852851255652310266969150278900336843*I");
	test(107,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-2)*pow(7,-1)),tau1)","0.8989571090451981335119439239871647550399066673789651610027263201030216333750315141448575681293947274137454502935028+2.2274339997605247500079037110179611875116550015017832917031350798308693376416391655313772727160636427278244950263449*I");
	test(108,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-2)*pow(7,-1)),tau2)","0.9019204805636299918773641476280032870189870490972635422842838071566959944609690822556665327322080098462509198560381+2.2383604671877415352837560529078933712256513101492882239998136413677603008240384586075674655041103044634533881271085*I");
	test(109,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-5)*pow(21,-1)),tau1)","0.7562667428039101114902895409406703459168552505056659415151210707225617671245262130049780122695132934399598390291572+2.1701903082130803899538048190309219764461628337215589709874738240553389502711804241689007003556255009586003828005037*I");
	test(110,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-5)*pow(21,-1)),tau2)","0.7594059085818843782850265354908093284344584045667422819005058127235561578883702985109812023419858100158295484260645+2.178099668560270482222116132511809539327175836414826768170270542172596068530631295534062693533678670152079447813769*I");
	test(111,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-4)*pow(21,-1)),tau1)","0.6209417268227190142425354593681929143431293291116064330646228398752196840762151805137862709556550561756075219350836+2.0945678767458012778840707437813930808368921996506886741076561482645297497658189264504769762303090084348355771987827*I");
	test(112,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-4)*pow(21,-1)),tau2)","0.6242116442348698751680052439359559641117981698380312468185827291761393914555784955980796250694477029854599015524808+2.1001753985063021314436095003880384514766311935069959485635358612420189919042568572031172653260707261941638851999373*I");
	test(113,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-1)*pow(7,-1)),tau1)","0.498797214240956600581764851718198895338870688164605139723994564030710974080477565638846460195015871796987084429157+1.996674763777104899622036102836737577977792874425737066731491040752799454843175505810152346676442445223349797596008*I");
	test(114,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-1)*pow(7,-1)),tau2)","0.5021603228108033055929250613327917078213104108658235086943340234250911237152746831852501760031686100219895979075941+2.0004875460448350413300992409966578897228140288680447834261638628039009737175861046784012201965671432564345756643904*I");
	test(115,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-2)*pow(21,-1)),tau1)","0.3986217378196694141823498336711205722682174762303097019834151833709479677094533523861196759821772087834606462587094+1.8742982166756782969247989302411494799810403409313338754600849774950321120656150697799197803488945172398528045050659*I");
	test(116,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-2)*pow(21,-1)),tau2)","0.4020465402034643231158251452929979214876524126807808392582168845093430785131508842234280256957486756427912560698487+1.8766611634850571664349950513911694836206543338572700285663057467150251174013099508759760134263601674328712481360474*I");
	test(117,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-1)*pow(21,-1)),tau1)","0.3315299452884577055873763111793744682785205295004778495144003638147480350274789526091711177950740324170054676060331+1.7294950402818178736948454053224092171676258731716585275473028758938764650731885229994557624710823261833848409014376*I");
	test(118,"Gt({{1,0}},(-5)*pow(21,-1)+I*((-1)*pow(21,-1)),tau2)","0.3349892677550318015665425201494741910801340295920861745154020168855272056539257745679917185508299364983980123102645+1.7306221032011611301750920390676608352982592925023429128444255638424855055485275293739980301666798464453069000318657*I");
	test(119,"Gt({{1,0}},(-5)*pow(21,-1),tau1)","0.30773869357579759981546895289576434461571334604882160597307374095158629257189814988171888242948908934707703134539085722646877011457591+1.5708027802607676293731375759198555424076392554166582427246356811405785313947278365575531731216587746816156724957574*I");
	test(120,"Gt({{1,0}},(-5)*pow(21,-1),tau2)","0.3112078773517370084384907643172623435667210127740447092394380259396560408586983295023740296185416457626622000215133+1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432560567*I");
	test(121,"Gt({{1,0}},(-5)*pow(21,-1)+I*((1)*pow(21,-1)),tau1)","0.3315341696349846510054077000284788928657724099182831464528030627969173354388132584890110263967897638288397374341003+1.4121104732309693375901003457405756063197401696598523225851592630040425747354465113214150594118622763440491312088835*I");
	test(122,"Gt({{1,0}},(-5)*pow(21,-1)+I*((1)*pow(21,-1)),tau2)","0.3349892677550318015665425201494741910801340295920861745154020168855272056539257745679917185508299364983980114086459+1.4109705503886321082875513442118420488989101068727629081305190284653309007376814692540367951754372215368411864790678*I");
	test(123,"Gt({{1,0}},(-5)*pow(21,-1)+I*((2)*pow(21,-1)),tau1)","0.398630567506704756700939413952633783076016740157702339907181467255171840970547130273846622553616859874666204237426+1.2673071515711439102936178583466871394801297243591786159730242756415287455018402047539062766448981821972848813898484*I");
	test(124,"Gt({{1,0}},(-5)*pow(21,-1)+I*((2)*pow(21,-1)),tau2)","0.402046540203464323115825145292997921487652412680780839258216884509343078513150884223428025695748675642791255257825+1.2649314901047360720276483318883334005765150655178357924086388455927912888848990477520588119157569005492768383742944*I");
	test(125,"Gt({{1,0}},(-5)*pow(21,-1)+I*((1)*pow(7,-1)),tau1)","0.4988114456183256329716317719739173131983030449540439142055105287585809545398395394853476740225073466214528041051768+1.1449303478449921650085496339677783708463445463019304230227792659430761396677873297101796904667444773811911014464692*I");
	test(126,"Gt({{1,0}},(-5)*pow(21,-1)+I*((1)*pow(7,-1)),tau2)","0.5021603228108033055929250613327917078213104108658235086943340234250911237152746831852501760031686100219895979001691+1.1411051075449581971325441422828449944743553705070610375487807295039154325686228939496336051455499247257135108458715*I");
	test(127,"Gt({{1,0}},(-5)*pow(21,-1)+I*((4)*pow(21,-1)),tau1)","0.6209626434189134180492396482839394644800399019114738383748829615879881414645622513468552582650587363679143748366876+1.0470368437478162885315764438433520621491326297316228691652672557397697926181199222032438437154628384833222238135597*I");
	test(128,"Gt({{1,0}},(-5)*pow(21,-1)+I*((4)*pow(21,-1)),tau2)","0.6242116442348698751680052439359559641117981698380312468185827291761393914555784955980796250694477029854599015462349+1.0414172550834911070190338828914644327205382058681098724114087310657974143819521414249175600160463417879842013111227*I");
	test(129,"Gt({{1,0}},(-5)*pow(21,-1)+I*((5)*pow(21,-1)),tau1)","0.7562962310875739796646555455293105383760260612179341538089631750391540097447058524580068231387969898776997963237295+0.9714138513724100844793880098250084699607487553289958601589428715169207793268092211550350212328072907202053107526598*I");
	test(130,"Gt({{1,0}},(-5)*pow(21,-1)+I*((5)*pow(21,-1)),tau2)","0.7594059085818843782850265354908093284344584045667422819005058127235561578883702985109812023419858100158295484270119+0.9634929850295227562405272507676933448699935629602790528046740501352203377555777030939721318084383978300686386967778*I");
	test(131,"Gt({{1,0}},(-5)*pow(21,-1)+I*((2)*pow(7,-1)),tau1)","0.898997828565810731892304645714893471321639153552158091169049079914893014227245067558195530104766915013156059737192+0.9141693785488644746960243543801233416647731542687882533431631803725594227793233263414102505538924765565383124420156*I");
	test(132,"Gt({{1,0}},(-5)*pow(21,-1)+I*((2)*pow(7,-1)),tau2)","0.9019204805636299918773641476280032870189870490972635422842838071566959944609690822556665327322080098462509214270975+0.9032321864020517031788873303716095129715180892258175969751309509400561054621705400204673598380067635186946983828631*I");
	test(133,"Gt({{1,0}},(-5)*pow(21,-1)+I*((1)*pow(3,-1)),tau1)","1.0456024254699263769905833161084904698147611685657152795976959386804844940919424857234064760929755817819069139676587+0.8713151837819321991195108043755512350883036424596657914705628385545970310317791144170245868603303061388056687320206*I");
	test(134,"Gt({{1,0}},(-5)*pow(21,-1)+I*((1)*pow(3,-1)),tau2)","1.0482813590580865623617127272432057111966376984580492908192901913703824782205157763773878000626858395443526258340519+0.8563736034344776566909533691824796893383991015885225200353810009325189454862858133429092601110903710671201964772138*I");
	test(135,"Gt({{1,0}},(-5)*pow(21,-1)+I*((8)*pow(21,-1)),tau1)","1.1941729010985121048679490207042458541688998973347545532179534608261670136231629018275052638999042878348167153231593+0.8394179362211934814437504418346276767678054474863468781723990378717999485806108555967419744808604532439032151779398*I");
	test(136,"Gt({{1,0}},(-5)*pow(21,-1)+I*((8)*pow(21,-1)),tau2)","1.1965438106744371216097828598163255440443664073958626057203636217497583430169619227376151430229338129201308272603637+0.8191211191958632504066649972373667829825574159850764440088302204035272493277515640572049260594000432183343617976127*I");
	test(137,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-8)*pow(21,-1)),tau1)","1.1666872784329227661503661701564843048805202912433818043384142738467844973598871213979792151442012885535494509967976+2.4554901009730804613231244529648797655915272498815160582069674465192429400055240272423885776240500614774028761544537*I");
	test(138,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-8)*pow(21,-1)),tau2)","1.1630748503497888479578209821325426389706215069207375875534294534975690387889848815990918847665125013696868062889462+2.4745546191756813360256918556783345732311386323937434363491332097813889870052229607539877830032250467425614098591353*I");
	test(139,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-1)*pow(3,-1)),tau1)","1.0082908031060025535958519363787785037899455610557112127626292913267514220039193777657862060298964847975434097969443+2.4237296435232488030911751435450448984987619206079051243331836246898473236296592123667442546534816652406727727951992*I");
	test(140,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-1)*pow(3,-1)),tau2)","1.006512687948186599997145460223285797607042826503822506973597757249092048126404477109531261748066415974552450953282+2.4377391328922638240112201134092307160827951386740688306380163453706291834964571070503053603576796306106852250228838*I");
	test(141,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-2)*pow(7,-1)),tau1)","0.8483450342274030015000859507535275852541574543669389277471820323304136773808155593637961214062190982656795587468582+2.3800543450730794789991410724401334228461110518564095656236045707373032773836018296402758094286978966191653874591571*I");
	test(142,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-2)*pow(7,-1)),tau2)","0.8479273718632661674407272356195824532078238480603705666286175573230211346063672499435198383454339252504875662042346+2.3902943340939732862124578387118716478491084338996278194631059655935371668967065828944250089771179265967515826713128*I");
	test(143,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-5)*pow(21,-1)),tau1)","0.6877452015997336104841915260928743089620246671472651945067742906819346734390660348515119561187156333082525003305368+2.3198553965427254147060579126477788424979313528341298720864915281586534877393786190569266393036773369843012273346114*I");
	test(144,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-5)*pow(21,-1)),tau2)","0.6883257800224166759529571497171703086872506065599721819405203739171152364818276588353864582396949900858912845310386+2.3272618672670686116505693266147076767060009645692586268013803839264796396306159152397934919304502664438076572539382*I");
	test(145,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-4)*pow(21,-1)),tau1)","0.5290552585162450448547051082349128632980233225409273238285126463570041150411125908329419931172971787218834608865229+2.2369657297692820171353350590607783906123453871411386607154942641734007279365113944720042957683422071274841523441936*I");
	test(146,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-4)*pow(21,-1)),tau2)","0.5303558956369578307920033572145029335727796084525094678768971330011270825627067766072310695822311425693826276332387+2.2422138043478172830070680382930238839026282731367677032922081869135719135568804587713695267039259572525108299574978*I");
	test(147,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-1)*pow(7,-1)),tau1)","0.3781875214518648347082095695250385324461438774201624845868674570434495668297759638168915243238482068050906287649566+2.1238886288054797323780921120521277944582276546722875764703825591209163053485473172666475429657237960710496864574491*I");
	test(148,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-1)*pow(7,-1)),tau2)","0.3799917162103439629429341364955237474517263037373981390663046880805160296634555402021959535233252338076150073087128+2.1274558236924119880725419030978813452049467505309234837951210041538067058438793829511704110548366569747226216881755*I");
	test(149,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-2)*pow(21,-1)),tau1)","0.2467986883364861885818844130280805534205563908933693649603398545267825981537712253203801494478137055032404105371586+1.9736887045634930747365408811847519466367542055688001788717125690554990747098866200481334494694603562366936559040833*I");
	test(150,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-2)*pow(21,-1)),tau2)","0.2489335928686257966248921571837798043446694538790236277879572840490887450214425251445220644613635250042966193738084+1.9758992986029331542777489485604778004502476953788781183683901842443467528478538692704883917570704011759636818245512*I");
	test(151,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-1)*pow(21,-1)),tau1)","0.1535594597930188229829651331598142319798714479017023010018999054277805873119295170153870860941718468737892524997772+1.7852515545365161790119687816245627288654018958365090714116585746604016887662470750187977076976291348197256228376703*I");
	test(152,"Gt({{1,0}},(-4)*pow(21,-1)+I*((-1)*pow(21,-1)),tau2)","0.1558811225649336148312329128863694568642357014660756271045769270448673369713544593167612164909640718557559259824359+1.7863065400991397231468020783610203008817505517716690096151964088185134074863232167359332787712704736231280390387375*I");
	test(153,"Gt({{1,0}},(-4)*pow(21,-1),tau1)","0.119239854945097020277419318089873887826036646260608268379616841005056260646621419997476342686373840786172328664420309814633895606607511+1.5708007533411514622159209120872942263758193890859230938652249995099038316773868286177464723601653205935864114828065*I");
	test(154,"Gt({{1,0}},(-4)*pow(21,-1),tau2)","0.1216207204171367447405275832259101693329370167420756025660845656066614046087734548615972486507899482801855455781318+1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432559506*I");
	test(155,"Gt({{1,0}},(-4)*pow(21,-1)+I*((1)*pow(21,-1)),tau1)","0.1535634031527394718155357081609676167951304674772156521081969697188403469316713094052920969192452073261376320815365+1.3563497223290778229366821883466886676373849784935040407939448842984537435870419038726325644732809590227881581908659*I");
	test(156,"Gt({{1,0}},(-4)*pow(21,-1)+I*((1)*pow(21,-1)),tau2)","0.1558811225649336148312329128863694568642357014660756271045769270448673369713544593167612164909640718557559259816045+1.3552861134906535153158413049184825833154188476034368113597481834893029987998857818921015465708465943590200474719612*I");
	test(157,"Gt({{1,0}},(-4)*pow(21,-1)+I*((2)*pow(21,-1)),tau1)","0.2468069307076976696996064539871299603542504248803919050982998707417966255538545676242777118929324582611466609402966+1.1679118621247960176606529787654062856631289445072481422460048445991176350802389527940834940415730512128304703725617*I");
	test(158,"Gt({{1,0}},(-4)*pow(21,-1)+I*((2)*pow(21,-1)),tau2)","0.2489335928686257966248921571837798043446694538790236277879572840490887450214425251445220644613635250042966193773809+1.1656933549868600841848944347190250837469217039962277026065544080634696534383551293575464335850466668061844046850767*I");
	test(159,"Gt({{1,0}},(-4)*pow(21,-1)+I*((1)*pow(7,-1)),tau1)","0.3782008062143548406498402869223228002346803591880605756795561213092600501093584088504154827551006709024885747593694+1.0177106832939890008775926635319186149717498309970210453083055820738427776032318599171752046445253270354531483134835*I");
	test(160,"Gt({{1,0}},(-4)*pow(21,-1)+I*((1)*pow(7,-1)),tau2)","0.3799917162103439629429341364955237474517263037373981390663046880805160296634555402021959535233252338076150073046962+1.0141368298973812503901014801816215389922226488441823371798235881540097004423296156768644142872804110074254648219793*I");
	test(161,"Gt({{1,0}},(-4)*pow(21,-1)+I*((4)*pow(21,-1)),tau1)","0.5290747838232714659087090111804138904872130814057160142047088719849521396523538798524378289467130337165918900500292+0.9046316701784350924265842180230831521599176373305178822055469915404598920641277674576983296644086486096356295774194*I");
	test(162,"Gt({{1,0}},(-4)*pow(21,-1)+I*((4)*pow(21,-1)),tau2)","0.5303558956369578307920033572145029335727796084525094678768971330011270825627067766072310695822311425693826268929297+0.8993788492419759554555753449864790002945411262383381176827364053942444927293285398566652986381911107296372565530842*I");
	test(163,"Gt({{1,0}},(-4)*pow(21,-1)+I*((5)*pow(21,-1)),tau1)","0.6877727284395146740815456787209787262465542135428964214186838613340488197399166766468560190728913685716947026251349+0.821739261233265796659162582007069481204356473364611568138681026674189479786978340429520811054497604118900742171831*I");
	test(164,"Gt({{1,0}},(-4)*pow(21,-1)+I*((5)*pow(21,-1)),tau2)","0.6883257800224166759529571497171703086872506065599721819405203739171152364818276588353864582396949900858912839645934+0.814330786322724626812074056664795207491168434805847194173564208381336766655593083388241333411666801538340429257039*I");
	test(165,"Gt({{1,0}},(-4)*pow(21,-1)+I*((2)*pow(7,-1)),tau1)","0.848383045246715062390365664497618398909816210043733018559986282153184159982778298471255644536068848746893793612271+0.761536493194633418681291201779177141673647323441011695702419743455613394080831737133280847030773460562197688735825*I");
	test(166,"Gt({{1,0}},(-4)*pow(21,-1)+I*((2)*pow(7,-1)),tau2)","0.8479273718632661674407272356195824532078238480603705666286175573230211346063672499435198383454339252504875662079222+0.7512983194958199522501855445676312363480609654754780015118386267142792393895024157336098163649991413853965038386932*I");
	test(167,"Gt({{1,0}},(-4)*pow(21,-1)+I*((1)*pow(3,-1)),tau1)","1.0083427265201923953450702894996138089599646914689200424451588063043770018182306552960684001312700483827136956971808+0.7178559534180271757000683885535586629201881545781230301967874306695429869916809086929386761668282791157269857048589*I");
	test(168,"Gt({{1,0}},(-4)*pow(21,-1)+I*((1)*pow(3,-1)),tau2)","1.0065126879481865999971454602232857976070428265038225069735977572490920481264044771095312617480664159745524495211202+0.7038535206975294144514232698702721681143742607010369903369282469371872227897518915777294649844374373714628614874291*I");
	test(169,"Gt({{1,0}},(-4)*pow(21,-1)+I*((8)*pow(21,-1)),tau1)","1.1667577972167879597658495539906107027436611857981900380306784490152771481164258409555004963760452662904006424857011+0.6860883601081537940044482512013787645602447283967927848494040349224570544474962156353101778855269385984923326152453*I");
	test(170,"Gt({{1,0}},(-4)*pow(21,-1)+I*((8)*pow(21,-1)),tau2)","1.1630748503497888479578209821325426389706215069207375875534294534975690387889848815990918847665125013696867989386601+0.6670380344141119024369515276011683109660307669813623846258113825264274192809860378740470423388920212395866218054213*I");
	test(171,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-8)*pow(21,-1)),tau1)","1.1410142783029488199103988639570711709504841889934159090930231024861373459944915537537223316839478804867967424206169+2.6172300756853310819408577423527426302829215470328055214261838608882417221804640183786667294296258287678518242944435*I");
	test(172,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-8)*pow(21,-1)),tau2)","1.1319553727956824895145487389422265612163483926716759137184254071931760421050863279608100243243954831484720699150601+2.6333399759000016171661062154851598603728397180003100633049351176416106750902203794626649023529174384451280392807598*I");
	test(173,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-1)*pow(3,-1)),tau1)","0.9726969639440296935552065268111171130869542768792004722338537407822079311803615760688355415725807839009897030248496+2.588872965860677216768847173140438211464736163445345841071333481903485255932306349246709768540784346387117645588466*I");
	test(174,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-1)*pow(3,-1)),tau2)","0.9668744601887265950664660185478352780662486265103850396065614174344874721961354100492156637518027255308831637925312+2.6006947176950811072654531685933343137440708493384918348747685282927778080125179861603325360385264922134798953279583*I");
	test(175,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-2)*pow(7,-1)),tau1)","0.7986433002932507510103395980133484392632026850213620726776092889548302766304541208332417326703130982221996348942163+2.5489282222403813328189820479351369454078221714261934080011995178906657805153019386576227650149485866210685996696427*I");
	test(176,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-2)*pow(7,-1)),tau2)","0.7952022302429600612825099857541436625771915393307172150699115718612543355512880283642036147262994528960350939075028+2.5575602745539998471220174100141258616359314465485454572840068944189404862447509611364057643307432656770615208229581*I");
	test(177,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-5)*pow(21,-1)),tau1)","0.6178445324050452598565070183272597881246829315700722982064473134673588616776153121997457146805457574763016783061643+2.4919488247211084710269816492058161462416415931570427546918309246476640291300784611186164377661799794481041959065764*I");
	test(178,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-5)*pow(21,-1)),tau2)","0.616141008655330951432839560499394987066559187260385171133334280742381514873050973384906768305830453756089832819396+2.498187648381365480414026147985215287988098614454810830963127543187904152290896849139294907249117501715349613618789*I");
	test(179,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-4)*pow(21,-1)),tau1)","0.4303133832626843113254625703493770417189835138217647828676298014436257268582849213012133534070243820196340820419365+2.4095437437370592560294911128762171177287787771697302008997561344265033289017082043130023637461669137389167408155143*I");
	test(180,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-4)*pow(21,-1)),tau2)","0.4298582662673248079978967298809126821215226434100188663833673006568305757949826671645457126557397904412838604621368+2.4139621279254864008615514230714680622229341839398927540087954983319823168487696796913969385043461793973491556921817*I");
	test(181,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-1)*pow(7,-1)),tau1)","0.239435793194325304596605266362688047598403788470258202536653381649907210924456947082090213903105732121996030435314+2.2890781142934757430062301207468706704698662578887324414193248440999519223832084047004579951509808481692700720606046*I");
	test(182,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-1)*pow(7,-1)),tau2)","0.2398514908248074928176646037955752934360897240799585298007668890444126221272410491258200480110104488453733599481352+2.2920803544047981280620306316784128185724246128093884306092707994898864596884075857337192858363405138861796827393628*I");
	test(183,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-2)*pow(21,-1)),tau1)","0.0575149722827858294464150163743482842910464383066456379050546309947933263000943670819025925434193414371311081553744+2.1137101974221655230301552486768815250586699487612856958015832551367500961881903803187771895545969768603340725197725*I");
	test(184,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-2)*pow(21,-1)),tau2)","0.0585018732728225391420972592674303042762364035616894776390933718396480273879419244522313149950996587121688829251279+2.115570479082942678626715271652572763490318004689293983850234990253162623305604720157709839524204005881282889392286*I");
	test(185,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-1)*pow(21,-1)),tau1)","-0.0852710182726299123954472069406040878673476906017473668411428553904282508496116600979861103902457878166124015987797+1.8698983903205963855999319941246316129311596198570901103777527981280308931266023970720753162345278143853382689557056*I");
	test(186,"Gt({{1,0}},(-1)*pow(7,-1)+I*((-1)*pow(21,-1)),tau2)","-0.0839612999587680637968331841250241057707792731170431346505788253185656699770255428656799795337276441328013544830251+1.8707865505257337702730760137266618865535197591942325139434330703930777097723489269098106279802932813841513826312794*I");
	test(187,"Gt({{1,0}},(-1)*pow(7,-1),tau1)","-0.141831482450270921865731300462543003584046028991388574570934051930208658116944854846536603976251899856970594103499290516256891253422995+1.570798952834819578905874830454478105763372539204537260509714261285988396488812959423186308796107843883724396372869*I");
	test(188,"Gt({{1,0}},(-1)*pow(7,-1),tau2)","-0.1404183573785678315066136137186492654420959533317161432222159194816338836589995218445567924012351000041858141140928+1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432558742*I");
	test(189,"Gt({{1,0}},(-1)*pow(7,-1)+I*((1)*pow(21,-1)),tau1)","-0.0852677062845319055710992638931252292291486677921192809701959700167097080551368266597302321064675301299569066247054+1.271699123144596111056636056650380950184696547991863593617363518872577777810991494495874833013106334868487019757398*I");
	test(190,"Gt({{1,0}},(-1)*pow(7,-1)+I*((1)*pow(21,-1)),tau2)","-0.083961299958768063796833184125024105770779273117043134650578825318565669977025542865679979533727644132801354468427+1.2708061030640594681895673695528409976436496401808733070315115219147386965138600717182241973618237865979967038788456*I");
	test(191,"Gt({{1,0}},(-1)*pow(7,-1)+I*((2)*pow(21,-1)),tau1)","0.0575218949673202110730378337325357509531396862210934317557147076140206277539767706358566918178287822477375635241832+1.0278861040567523178382787488764964238487328161223033807892137245313517498300639058361452158163570853652552451669189*I");
	test(192,"Gt({{1,0}},(-1)*pow(7,-1)+I*((2)*pow(21,-1)),tau2)","0.0585018732728225391420972592674303042762364035616894776390933718396480273879419244522313149950996587121688829305411+1.0260221745068505598359281116269301207068513946858118371247096020546537829806042784703249858179130621008651971158055*I");
	test(193,"Gt({{1,0}},(-1)*pow(7,-1)+I*((1)*pow(7,-1)),tau1)","0.2394469509324897073011930839599071780325848278501326444948405600934166489582654248957859560263974757918491706790733+0.8525160461082490298469877325755179688456035672277579540656074812503735683509106661128053035143726960733941121955521*I");
	test(194,"Gt({{1,0}},(-1)*pow(7,-1)+I*((1)*pow(7,-1)),tau2)","0.2398514908248074928176646037955752934360897240799585298007668890444126221272410491258200480110104488453733612679822+0.8495122991849951104006127516010900656247447865657173903656737928179299465978014128943155395057765540959684037699084*I");
	test(195,"Gt({{1,0}},(-1)*pow(7,-1)+I*((4)*pow(21,-1)),tau1)","0.4303297823713277584281530638302039063912073455992061851353296136875447717637900086838769911552003428779118352453361+0.7320471533927166018321427080589435648094891910707847318707520576895568546038462191561419885635224723207108452507046*I");
	test(196,"Gt({{1,0}},(-1)*pow(7,-1)+I*((4)*pow(21,-1)),tau2)","0.4298582662673248079978967298809126821215226434100188663833673006568305757949826671645457126557397904412838604753167+0.7276305256643068376010919602080348219742352154352130669661490939758340894374393189366378868377708885847989308178149*I");
	test(197,"Gt({{1,0}},(-1)*pow(7,-1)+I*((5)*pow(21,-1)),tau1)","0.6178676519204287000800897825735126392766540850356238541664319637838456718921143622231109723863891053964349024736242+0.6496373926273352421543818054667984960613793015595898625614937698225734371677062672315843611378098750324853798659681*I");
	test(198,"Gt({{1,0}},(-1)*pow(7,-1)+I*((5)*pow(21,-1)),tau2)","0.6161410086553309514328395604993949870665591872603851711333342807423815148730509733849067683058304537560898321222998+0.6434050052084277580486172352942875962090707849202949900118170491199122539953121494887399180929995662667984728917851*I");
	test(199,"Gt({{1,0}},(-1)*pow(7,-1)+I*((2)*pow(7,-1)),tau1)","0.798675225365386008908458943323393697812288793892209159731286752275010192541929553932056030672737132575170962294328+0.5926514767476813543571787024774566128347576831429120160999560461805454928099797028641515439168572988301963953923604*I");
	test(200,"Gt({{1,0}},(-1)*pow(7,-1)+I*((2)*pow(7,-1)),tau2)","0.7952022302429600612825099857541436625771915393307172150699115718612543355512880283642036147262994528960350938965181+0.5840323790357933913406259732653770225612379528265603636909376978888759200414580374916290610113738023050865656874214*I");
	test(201,"Gt({{1,0}},(-1)*pow(7,-1)+I*((1)*pow(3,-1)),tau1)","0.9727405738964413073490539365436578557339318341216171444733408352575522162233650844150854717930362447016992168341895+0.5526977882967708064829503994164902449512777206286273636273206410166343131165371157296774485321573391698631964297816*I");
	test(202,"Gt({{1,0}},(-1)*pow(7,-1)+I*((1)*pow(3,-1)),tau2)","0.9668744601887265950664660185478352780662486265103850396065614174344874721961354100492156637518027255308831621602937+0.5408979358947121311971902146861685704530985500366139861001760640150385982736910124677022893035905757686681911823265*I");
	test(203,"Gt({{1,0}},(-1)*pow(7,-1)+I*((8)*pow(21,-1)),tau1)","1.141073506318975548316862628015759145755157296236142472561734578902104026847207474386852993561515198879636696737695+0.5243285004366332375383505795949236557458832864695320690861453629611675215735490025883626501588420992248311477022955*I");
	test(204,"Gt({{1,0}},(-1)*pow(7,-1)+I*((8)*pow(21,-1)),tau2)","1.1319553727956824895145487389422265612163483926716759137184254071931760421050863279608100243243954831484720712106965+0.5082526776897916212965371677943430238243296813747957576700094746662057311959886191653699229891996295370200472293545*I");
	test(205,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-8)*pow(21,-1)),tau1)","1.1198873188894057741478709545836301342829362774902180978227220539828718318101171342667324179309975815038432426762661+2.7867977633113074642255864607565256889953933891964390833961823164170890129034898036524511764076767281522316554594528*I");
	test(206,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-8)*pow(21,-1)),tau2)","1.1065058570210437381223895884844616256481713816135699877889790641398750792913365995028159207633128569881749990085967+2.7984674446216913042669684133845946632123525395738284048012189906667392888097944051943902862187431967173851595304018*I");
	test(207,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-1)*pow(3,-1)),tau1)","0.9428515896857254759176141084691701226543339815942034984286194814346788777726249780065766949685031800027355752675435+2.7653020055439359471638462824047287515805957589697486351005376554935837205517233937019848701613253603523618498262392*I");
	test(208,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-1)*pow(3,-1)),tau2)","0.9338275148574441995544246797886958953607907182376462947028180166866224912235060153300041723021627889022877246964652+2.7738556674308882309920415919718292657491244058130230405346627499773793067584425716326626629673134060523178724347691*I");
	test(209,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-2)*pow(7,-1)),tau1)","0.7558206418682741123832363442059310156205454175285122501205776671585050161397876154579099208123828332960802409435678+2.7343574053792485725313797488156817429392033826622051768345774651431129956010342133661656487792356587469007108181839*I");
	test(210,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-2)*pow(7,-1)),tau2)","0.7499907818410624975662116397796363008544285568410359595325338435291155162603215917925581541893142826250407203727423+2.7405978439108075749821405582727511899554275169285090176183992553624973271225624698254621131659358617229523567060766*I");
	test(211,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-5)*pow(21,-1)),tau1)","0.5551517803025987774999898197364773960955005307265536338332015979948268034134565002593608887548226680866720614170472+2.6887445442333403286324228321157892999935291022568249139908044738186493870708386752058536055802597092221401420595929*I");
	test(212,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-5)*pow(21,-1)),tau2)","0.5516461903744068582807129254491647254970414316986430212094047222382867169077322651104773030929923455437978152295521+2.6932519595939731339590768995698307436185387947121866248473608201000865030807831843165054842084881298727255879737199*I");
	test(213,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-4)*pow(21,-1)),tau1)","0.3362579030589583360942703312594221074049265937116002352963964162738067304775165695473214893629425189090023213389232+2.6193183132522405759365892356080522912837013796889092583300621660805893551854476859306873721230719307296322683347614*I");
	test(214,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-4)*pow(21,-1)),tau2)","0.3344191072520215310048280315587322637786060361749448746172074770370565383897747427942645086883599078048717952343471+2.6225089959372290575763016271921732466002576272892189506550264469000874654236833384719072539534433570631582130155031*I");
	test(215,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-1)*pow(7,-1)),tau1)","0.0947984218715238079295011476303268550236451722120583719482096756478002781935532155634311900417610887167622298196633+2.509165667071184417993106132464523223774902790133575356218974407094244040655292428724202804022154183048987143132492*I");
	test(216,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-1)*pow(7,-1)),tau2)","0.0941208375992374284670705402569938290403766461849899971512809522513012746247470151396240000896814848790752075913445+2.5113329869183514446873139288917291033714995120774435007829317052205941927553765601345999542619752132913640641802513*I");
	test(217,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-2)*pow(21,-1)),tau1)","-0.1667490355997393127726926891867809634021500640864276165356575221367477557882492826863528134307033624038984717229741+2.3263565894229574805037599521718090934039755752156055415171628906955369054507940048219497807540135008199952458842408*I");
	test(218,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-2)*pow(21,-1)),tau2)","-0.1666654070806824340065424734345747724587635460175571716537499734921496058569292417171958207845981086439457483378062+2.3276993187312130173870039126679763433365648554326889501214583799015397069252126764211119709309223400530860061519747*I");
	test(219,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-1)*pow(21,-1)),tau1)","-0.4130976916710142305053399310021039551522405690030053132244249358029878624172435913304651704981764970997161263942228+2.0194578969989158509795490775875284906537903369458351419031354931343736777079144984922145637857912739680116079557061*I");
	test(220,"Gt({{1,0}},(-2)*pow(21,-1)+I*((-1)*pow(21,-1)),tau2)","-0.4125837414061463618322583178991174970895275639944983148088667954827277890115267401156606956354752571726658440123723+2.0200991356696462475191813787643460603126185037470475372943896552145923101934155799237633791544364134321065481308571*I");
	test(221,"Gt({{1,0}},(-2)*pow(21,-1),tau1)","-0.528463003917468159418347564854353497899708185734367199813762265415285842575300156154577666138996113279841740949407056299179579263529427+1.5707975387246609591693302996637881414016413309363019091711046503217934885540054630561551782562912923706510578330397*I");
	test(222,"Gt({{1,0}},(-2)*pow(21,-1),tau2)","-0.5278105928332934756928365470283920291448847701352504702796895219512801066686373866104230342036334197565769998907272+1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432559507*I");
	test(223,"Gt({{1,0}},(-2)*pow(21,-1)+I*((1)*pow(21,-1)),tau1)","-0.4130953053392157772965664259026252461871935080301477434392486479476107858610941517537326794246552544612568484732661+1.1221366607073079288122587098157999584502187761622433197510678172746682518181501670354976674957200114740092534046712*I");
	test(224,"Gt({{1,0}},(-2)*pow(21,-1)+I*((1)*pow(21,-1)),tau2)","-0.4125837414061463618322583178991174970895275639944983148088667954827277890115267401156606956354752571726658440087999+1.121493517920146990943462004515156823884550895628058283680554937093224096092793418704271446187680654550041538378771*I");
	test(225,"Gt({{1,0}},(-2)*pow(21,-1)+I*((2)*pow(21,-1)),tau1)","-0.1667440477127825634400355357744716888388713892299626169588540560581759468745595582506669498627480936637460800034037+0.8152363621783211708568603468096492401709475905464876313871679000772228938999000049185092275109403211922044950679534*I");
	test(226,"Gt({{1,0}},(-2)*pow(21,-1)+I*((2)*pow(21,-1)),tau2)","-0.1666654070806824340065424734345747724587635460175571716537499734921496058569292417171958207845981086439457490781152+0.8138933348585802210756394706115265408606045439424168708534862124062766993609963222069228544111947279290620803586074*I");
	test(227,"Gt({{1,0}},(-2)*pow(21,-1)+I*((1)*pow(7,-1)),tau1)","0.0948064611713618925955347263264169663909605026603202208931505529980471192199580111526784938587958625448819018957804+0.6324244472086331694889448332838942935681946946245433184261174408863262552545507313097132595600323946505934547818121*I");
	test(228,"Gt({{1,0}},(-2)*pow(21,-1)+I*((1)*pow(7,-1)),tau2)","0.0941208375992374284670705402569938290403766461849899971512809522513012746247470151396240000896814848790752075950319+0.6302596666714417937753294543877737808256698872976623201920128870872222135308324384934348710801418546907840223297549*I");
	test(229,"Gt({{1,0}},(-2)*pow(21,-1)+I*((4)*pow(21,-1)),tau1)","0.3362697188364498311607382752258876940849705145335123405665836561455658261601855097576240078032182529592364630731067+0.5222674765914644620937433919951914355429179983992336701610446141377006324617915179804335172871432464601005968149246*I");
	test(230,"Gt({{1,0}},(-2)*pow(21,-1)+I*((4)*pow(21,-1)),tau2)","0.3344191072520215310048280315587322637786060361749448746172074770370565383897747427942645086883599078048717952203167+0.5190836576525641808863417560873296375969117720858868703199181454077289408625256601561275713886737109189898734946547*I");
	test(231,"Gt({{1,0}},(-2)*pow(21,-1)+I*((5)*pow(21,-1)),tau1)","0.555168438223185178135426023144671864325834924728825188654196543559662336913330138140225227169736252556120708413113+0.4528350440385354526685559806908299136428691198275599451157234180348589913181420090915286144211778427849612915760408*I");
	test(232,"Gt({{1,0}},(-2)*pow(21,-1)+I*((5)*pow(21,-1)),tau2)","0.5516461903744068582807129254491647254970414316986430212094047222382867169077322651104773030929923455437978144180478+0.4483406939958201045035664837096721405786306046629191961275837722077299032054258143115293411336289381094224985375067*I");
	test(233,"Gt({{1,0}},(-2)*pow(21,-1)+I*((2)*pow(7,-1)),tau1)","0.7558436443104742562835603060804913792812581002347478710612257245219821816622688390297788134690424672394761505873261+0.4072135448650708654445245827272191564838663453112440797595322767943365872076767409174107415516403836276794083109902*I");
	test(234,"Gt({{1,0}},(-2)*pow(21,-1)+I*((2)*pow(7,-1)),tau2)","0.7499907818410624975662116397796363008544285568410359595325338435291155162603215917925581541893142826250407197867851+0.4009948096789856634805028250067516942417418824465968033565453369453190791636465288025727121761812062591957298054661*I");
	test(235,"Gt({{1,0}},(-2)*pow(21,-1)+I*((1)*pow(3,-1)),tau1)","0.942883011240712328988758884110691718648524352988492929502192927328247203603368198881431949062017068161957642140176+0.3762570911530503049657608576170725691961092980296144847456998963192940982391687107255522106262202147097960647333124*I");
	test(236,"Gt({{1,0}},(-2)*pow(21,-1)+I*((1)*pow(3,-1)),tau2)","0.9338275148574441995544246797886958953607907182376462947028180166866224912235060153300041723021627889022877247028242+0.367736986158905007470601791307673618448044993562082780440281842330437099527766426995372162374803661929830214075141*I");
	test(237,"Gt({{1,0}},(-2)*pow(21,-1)+I*((8)*pow(21,-1)),tau1)","1.1199299934684629010232604077883978255099442646802135829180861091133960469234451261623281566897077146963780183657366+0.3547451952466614300021434533297215939335530486883932064541520326121826082104701148004539816265001666512782293743267*I");
	test(238,"Gt({{1,0}},(-2)*pow(21,-1)+I*((8)*pow(21,-1)),tau2)","1.1065058570210437381223895884844616256481713816135699877889790641398750792913365995028159207633128569881749990132747+0.3431252089681019341956749698949082209848168598012774161737256016410771174764145934336445391233738712647629269794703*I");
	test(239,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-8)*pow(21,-1)),tau1)","1.1059330663437346158243412888649956630077171694786974485247842384528986449170871250466255824884042049335263504897657+2.9624879968236186936578168267971957901211561763109865985506224911569080920833837652607241784234408315925713172329231*I");
	test(240,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-8)*pow(21,-1)),tau2)","1.0897701503385567830731134160877325361903936683362366444972018436074427402622143608324752800879187628235849982043711+2.9686276908348216442160419506053576658620801069272827443504482791274590121245734665403875414106838944580121609267284*I");
	test(241,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-1)*pow(3,-1)),tau1)","0.9228414233038929896209541689283123856869995703371456331925604216662913657922436445401859721279800897990057982880094+2.9508572334900661539435686491553653817034204602304256493081051771217471421782664507820966265115424388076117262475677*I");
	test(242,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-1)*pow(3,-1)),tau2)","0.9117610833293608712760608141342654117125478868973330206138442186192667699871332542864405468720808682826675599245639+2.9553537117119822237317030503133758721588423364758179333663500512214243966608403469587905631233210677482983415722158*I");
	test(243,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-2)*pow(7,-1)),tau1)","0.7264565288869455898944255748601260239213754699765800564518699275711197620920540618649134668146547922773013437662813+2.9338513391312098420498209049722906642794982334138112732718092618928322249590380478886109113918057885696588705725235*I");
	test(244,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-2)*pow(7,-1)),tau2)","0.7190945836986565423953386455031425273721130717787919051612125823636113626110091644269868390509696871166427573564578+2.9371295363287834144945647933754620805425562936028347126252288956139301560012371003417842675353387366326727639807244*I");
	test(245,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-5)*pow(21,-1)),tau1)","0.5106204114628985853949700712730235270552180960881730796046232229786497185767073962298676274933561641440608495195264+2.9081625355359343171607232658797203662940619038262083127556165331597136146055213155381060685636038663753639005041746*I");
	test(246,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-5)*pow(21,-1)),tau2)","0.5059602965218232703264142688993791099480071941839286681739024028614627923905902098674766070048227020995483640673919+2.9105289485306989488689264738724664414692143213137147795936161637959326094071757871932688376573321031941025829089798*I");
	test(247,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-4)*pow(21,-1)),tau1)","0.2654625050680414216964402317025360754511341971638557465937479389162060857286614580478998577694781118553603585556034+2.8674178535488605137528520830337071568139523048233579461196527244726512515482423245451933964928830107265075824666143*I");
	test(248,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-4)*pow(21,-1)),tau2)","0.2627380155441782828061527558550600311566564293359771856411396682798270786817697313789392717747095503564748967819502+2.8690920973632863116272057469749323857913791590937088906277203509845660063641021361062906630905742064830439732677171*I");
	test(249,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-1)*pow(7,-1)),tau1)","-0.0255941039136874809583086036539448345807919003498717861734715482770162873842771557027351425803302535865959917587716+2.7977229475300004304878442225933562797750112838535385572050917684681256305797523409661912025015520485601589096563143*I");
	test(250,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-1)*pow(7,-1)),tau2)","-0.0269709570365042408768759345417224661481548849197623034310786966599204743981373977658516833823967222657512370491059+2.7988596726672582202309211240051162100252660487204655941043587215746630029727984622073703879193841291887831590398349*I");
	test(251,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-2)*pow(21,-1)),tau1)","-0.3907186733296491856989752859576306753091432571911079888733688731855704597789155603344880145054852345227854108675906+2.6630906943290325680745884385206268188311905034540262942711555441038973700577198699815658252934714856702845876702446*I");
	test(252,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-2)*pow(21,-1)),tau2)","-0.3912123628102974282372743493053793822310434184205180031015542472357358406694436857032250845787261614725220502976912+2.6637946732539116987148050713632574461718234012169707785582771689178097014464538454683747107830737276075905229036741*I");
	test(253,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-1)*pow(21,-1)),tau1)","-0.8600612750852700306704884372159733466605188100122522492465023709977240827136671352641041464611560635338700383549495+2.3487345126887479563316151714773802570383531166816608676458129652981335535653635579391638486316429947708682961152054*I");
	test(254,"Gt({{1,0}},(-1)*pow(21,-1)+I*((-1)*pow(21,-1)),tau2)","-0.8600555780721843020100416473773141609843634191556685091921793367077621356686585365363072465605662281083201716975026+2.349070697327061950981987881261714458195049001006669261828770928148531655267295730833547581987671435753229800201451*I");
	test(255,"Gt({{1,0}},(-1)*pow(21,-1),tau1)","-1.210378167093500188729032537738612189647498776250138059357070212307774127005934197648568952819351929448205382119218322437874368059822627+1.5707966366605689038465509169694184912455931297642831280064273950590059133006287914890767755465465007157812938951119*I");
	test(256,"Gt({{1,0}},(-1)*pow(21,-1),tau2)","-1.2102113185575225611042969536237089680118656945425300160175857209671296088055426438497123105184788531392173182311694+1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432558625*I");
	test(257,"Gt({{1,0}},(-1)*pow(21,-1)+I*((1)*pow(21,-1)),tau1)","-0.8600600264458236307937798460649354874950174940855839050860302147492566458734346126644648105505985785562415047694376+0.7928581595320953636501319353992766584690407692244665565073636847820506253750370134797393025781981869708051245847241*I");
	test(258,"Gt({{1,0}},(-1)*pow(21,-1)+I*((1)*pow(21,-1)),tau2)","-0.860055578072184302010041647377314160984363419155668509192179336707762135668658536536307246560566228108320173625716+0.7925219562627312874806555020177884260021203983684365591461736641592847510189132677944872433544456322289182863077197*I");
	test(259,"Gt({{1,0}},(-1)*pow(21,-1)+I*((2)*pow(21,-1)),tau1)","-0.3907160634359150500309649250910565586199781333933889523869905376795409834888151791734631080039227888391794902171197+0.4785001203776678148503301845554223273335645477001557116439128134284817283184813022752295182843215973019983916443255*I");
	test(260,"Gt({{1,0}},(-1)*pow(21,-1)+I*((2)*pow(21,-1)),tau2)","-0.3912123628102974282372743493053793822310434184205180031015542472357358406694436857032250845787261614725220503117216+0.4777979803358815397478383119162454380253459981581350424166674233900067048397551531596601145590433403745575636064837*I");
	test(261,"Gt({{1,0}},(-1)*pow(21,-1)+I*((1)*pow(7,-1)),tau1)","-0.025589897379245945177245459653293901503151878867154571375687866139564520020527439256698858524188775848651295380247+0.3438645857194341245285598415848009468948304796951708249164927541597274044515695521073323876817251563409342630355182*I");
	test(262,"Gt({{1,0}},(-1)*pow(21,-1)+I*((1)*pow(7,-1)),tau2)","-0.0269709570365042408768759345417224661481548849197623034310786966599204743981373977658516833823967222657512376350631+0.3427329809225350182317222592743866741719033506546402268705858707331534033134105364206644374227329387933649274717078*I");
	test(263,"Gt({{1,0}},(-1)*pow(21,-1)+I*((4)*pow(21,-1)),tau1)","0.2654686876306988230596686871348824227978009097092532484441932434248820041165548385476424563174682199313489644720936+0.2741646783454240222451665032675632351124165960135200680506639532395035155586396271220034439233957470022198476879267*I");
	test(264,"Gt({{1,0}},(-1)*pow(21,-1)+I*((4)*pow(21,-1)),tau2)","0.2627380155441782828061527558550600311566564293359771856411396682798270786817697313789392717747095503564748967866282+0.2725005562265069268354376363045704984057902402813969303472242413232503999221068625217441622515428614991041132421551*I");
	test(265,"Gt({{1,0}},(-1)*pow(21,-1)+I*((5)*pow(21,-1)),tau1)","0.5106291276593433924665614955823139851431332243451091839075949470517755225792736714467272250241554972012376816091775+0.2334128240328962493575250764334353286412262174509721098253982022828471288341213373219011157696480825320119082680885*I");
	test(266,"Gt({{1,0}},(-1)*pow(21,-1)+I*((5)*pow(21,-1)),tau2)","0.5059602965218232703264142688993791099480071941839286681739024028614627923905902098674766070048227020995483640513403+0.2310637050590942895937169094070364427279550780613910413813284285118837968790332114347659876847849647880455036013849*I");
	test(267,"Gt({{1,0}},(-1)*pow(21,-1)+I*((2)*pow(7,-1)),tau1)","0.7264685648312805774953136813003888401376904879473916723423876271149437047113112808149183457586671646001650868736186+0.2077140302695453447487455761126928995869468505344865135736778852301944219729324936367138081825993398327956831453052*I");
	test(268,"Gt({{1,0}},(-1)*pow(21,-1)+I*((2)*pow(7,-1)),tau2)","0.7190945836986565423953386455031425273721130717787919051612125823636113626110091644269868390509696871166427573699261+0.2044631172610098239680785899040408036546131057722711083497156966938862502849718982862505578067783313494753225301096*I");
	test(269,"Gt({{1,0}},(-1)*pow(21,-1)+I*((1)*pow(3,-1)),tau1)","0.9228578645184130809740113732386698118328501116116277112831033974406455613356158869641439646459995013280071056795419+0.1906944268863376400209273733414213199242532736236330485298609944219659483845750759947878962520995116318833205579729*I");
	test(270,"Gt({{1,0}},(-1)*pow(21,-1)+I*((1)*pow(3,-1)),tau2)","0.9117610833293608712760608141342654117125478868973330206138442186192667699871332542864405468720808682826675600950779+0.1862389418778110147309403329661270120383270628992878876085945410863920096253686516692442622187960002338497449382653*I");
	test(271,"Gt({{1,0}},(-1)*pow(21,-1)+I*((8)*pow(21,-1)),tau1)","1.105955395662222665419271423608991028363219090245392061437461804142840574956855990689046985808773926909750664885385+0.1790449992547100899902444251263278626897600582525104967895650715188014027237789767245883414811616779454103473735638*I");
	test(272,"Gt({{1,0}},(-1)*pow(21,-1)+I*((8)*pow(21,-1)),tau2)","1.0897701503385567830731134160877325361903936683362366444972018436074427402622143608324752800879187628235849982119722+0.1729649627549715942466014326741452183350892924478230766244963131803573941616355320876472839314331735241359255832058*I");
	test(273,"Gt({{1,0}},I*((-8)*pow(21,-1)),tau1)","1.1010559162698336239681503070182260201618437782114469108070227345832282024598229475573669309455928131524333954260254+3.1415611137414259985738756967051365392665512456822203435425212590958905417712436458432579082076827875958549896419163*I");
	test(274,"Gt({{1,0}},I*((-8)*pow(21,-1)),tau2)","1.0839263934464480328585254381477778467277696843054055273935941450611302922300806757027739210523683001469154250104532+3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582*I");
	test(275,"Gt({{1,0}},I*((-1)*pow(3,-1)),tau1)","0.9157842445148774051687856582319637627186905344411648981560145398830014661726553400696908891932922837163170943604621+3.1415708797676014432741893020439617088590181871753231957074672892875339396907104278245207961772393319315356306126365*I");
	test(276,"Gt({{1,0}},I*((-1)*pow(3,-1)),tau2)","0.9039900355698141997321076688772141638717828910149858305507229727892549675368765567111459587286207965046357266801108+3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582*I");
	test(277,"Gt({{1,0}},I*((-2)*pow(7,-1)),tau1)","0.7159621963186451397849156560785464551871161569554755190366850836774402634282343475998592719823912750773933568629443+3.1415780529648007585825610246683285073556359776852413288498315909502399906541523957722886008399174161371378122122624*I");
	test(278,"Gt({{1,0}},I*((-2)*pow(7,-1)),tau2)","0.7080688157908931793498072339115271173416746820171541888139547121114561762456885875680093460047943921695383837522654+3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582*I");
	test(279,"Gt({{1,0}},I*((-5)*pow(21,-1)),tau1)","0.4943629505384189581104431308520803241510041696586089416527413011871879721620704762021888621832501586107215895695235+3.1415832802839614095344496924119740566369682190755065361673710494017489136682557692445514090161264702696953757555719*I");
	test(280,"Gt({{1,0}},I*((-5)*pow(21,-1)),tau2)","0.4893026989838114550633571692650799673874222258905473068003759279979222770427237829393374550914507765971938174909914+3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582*I");
	test(281,"Gt({{1,0}},I*((-4)*pow(21,-1)),tau1)","0.238632566051946914420473129122337684879084764024255023104726227807285605151052694176003652893733255404255194785608+3.1415870331772051505283820506475583904965401472387123926886301979805565146162520783233219446699036223256754115751701*I");
	test(282,"Gt({{1,0}},I*((-4)*pow(21,-1)),tau2)","0.235601406261664390580442082392513448578185793136637265005084822923951795766549745061253840953451792014285780927285+3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582*I");
	test(283,"Gt({{1,0}},I*((-1)*pow(7,-1)),tau1)","-0.0746855501297263028219358274928892853739875913385876698394206564634955682927914766411999598421582458154716982442257+3.1415896501181224379273849724076286170453867571603246213687310182733804067262103753387319782878654978754554455134997*I");
	test(284,"Gt({{1,0}},I*((-1)*pow(7,-1)),tau2)","-0.0763042215910210307216617079374349208666435166097752081002782747406580386327503586072499592309878129037544321288279+3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582*I");
	test(285,"Gt({{1,0}},I*((-2)*pow(21,-1)),tau1)","-0.4986224239869255672866953563024636129296266243443607483277032111941614891787867747092869221017619274465313586000225+3.1415913671287185311418830636961638935343063832605311643957466317257426600236490951488588338795277576030585069382149*I");
	test(286,"Gt({{1,0}},I*((-2)*pow(21,-1)),tau2)","-0.4993154213721615642328579222367412093335725760366100546699394576219575088791933140605088861279679588230519407360508+3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582*I");
	test(287,"Gt({{1,0}},I*((-1)*pow(21,-1)),tau1)","-1.2029181407353002184181047539611807136783306234455670380988813797286286923067463724167417113922493530779922143599146+3.141592339066247340196329072134909577927417566393421089526511896315652920889405617962062442405520448631964559519046520497584306773002165*I");
	test(288,"Gt({{1,0}},I*((-1)*pow(21,-1)),tau2)","-1.2030875256922395832393022974257962630590376460178699121151482048847989593581816257122337734995624879058360539955758+3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582*I");
	test(289,"Gt({{1,0}},I*((1)*pow(21,-1)),tau1)","-1.2029181407353002184181047539611807136783306234455670380988813797286286923067463724167417113922493532838873643261977-3.14523545898266314311144593306269751832981684731448432695992163485396803380665972382936596512533316809237016900666237299812369261e-7*I");
	test(290,"Gt({{1,0}},I*((1)*pow(21,-1)),tau2)","-1.2030875256922395832393022974257962630590376460178699121151482048847989593581816257122337734995624879058360540048743");
	test(291,"Gt({{1,0}},I*((2)*pow(21,-1)),tau1)","-0.498622423986925567286695356302463612929626624344360748327703211194161489178786774709286922101761927429070479056138-1.286461074707320760319583338990662863016114574656579197960582073746262559903479175991462588907943832640473311117385560319283471408e-6*I");
	test(292,"Gt({{1,0}},I*((2)*pow(21,-1)),tau2)","-0.4993154213721615642328579222367412093335725760366100546699394576219575088791933140605088861279679588230519407282644");
	test(293,"Gt({{1,0}},I*((1)*pow(7,-1)),tau1)","-0.0746855501297263028219358274928892853739875913385876698394206564634955682927914766411999598421582457996709111504087-3.0034716708005352584108718742671517826422147811996062135740344359995599986232893028470542512877083605561820367e-6*I");
	test(294,"Gt({{1,0}},I*((1)*pow(7,-1)),tau2)","-0.076304221591021030721661707937434920866643516609775208100278274740658038632750358607249959230987812903754432133854");
	test(295,"Gt({{1,0}},I*((4)*pow(21,-1)),tau1)","0.2386325660519469144204731291223376848790847640242550231047262278072856051510526941760036528937332554273944187178259-5.6204125880879342613326319444937006292521363934282863143943272598916699569203047128806722132672524509885460477e-6*I");
	test(296,"Gt({{1,0}},I*((4)*pow(21,-1)),tau2)","0.2356014062616643905804420823925134485781857931366372650050848229239517957665497450612538409534517920142857809350714");
	test(297,"Gt({{1,0}},I*((5)*pow(21,-1)),tau1)","0.4943629505384189581104431308520803241510041696586089416527413011871879721620704762021888621832501585755694196473387-9.3733058318289281936908675288275602011802995992848075735429060674926179532293834834163259903848074144393851082e-6*I");
	test(298,"Gt({{1,0}},I*((5)*pow(21,-1)),tau2)","0.4893026989838114550633571692650799673874222258905473068003759279979222770427237829393374550914507765971938174928853");
	test(299,"Gt({{1,0}},I*((2)*pow(7,-1)),tau1)","0.7159621963186451397849156560785464551871161569554755190366850836774402634282343475998592719823912750354203313058139-0.0000146006249924798800823586111743768415334216898644921251130013575764156320566028557462245021994162522894850645579*I");
	test(300,"Gt({{1,0}},I*((2)*pow(7,-1)),tau2)","0.7080688157908931793498072339115271173416746820171541888139547121114561762456885875680093460047943921695383837472394");
	test(301,"Gt({{1,0}},I*((1)*pow(3,-1)),tau1)","0.9157842445148774051687856582319637627186905344411648981560145398830014661726553400696908891932922836932665409743967-0.0000217738221917951884540812355411753381512121997826252674773030202824665954985708035140291648777360506124559006458*I");
	test(302,"Gt({{1,0}},I*((1)*pow(3,-1)),tau2)","0.9039900355698141997321076688772141638717828910149858305507229727892549675368765567111459587286207965046357266818188");
	test(303,"Gt({{1,0}},I*((8)*pow(21,-1)),tau1)","1.101055916269833623968150307018226020161843778211446910807022734583228202459822947557366930945592813177436806120376-0.000031539848367239888767686574366344930618153692885477432423333211925864514965352784776917134434280386293096871366*I");
	test(304,"Gt({{1,0}},I*((8)*pow(21,-1)),tau2)","1.0839263934464480328585254381477778467277696843054055273935941450611302922300806757027739210523683001469154250182396");
	test(305,"Gt({{1,0}},(1)*pow(21,-1)+I*((-8)*pow(21,-1)),tau1)","1.1059553956622226654192714236089910283632190902453920614374618041428405749568559906890469858087739269097506648687252-2.9625476543350831484723989581531750215074093411225953241853795207890150035624300219034464838609553900367377391394241*I");
	test(306,"Gt({{1,0}},(1)*pow(21,-1)+I*((-8)*pow(21,-1)),tau2)","1.0897701503385567830731134160877325361903936683362366444972018436074427402622143608324752800879187628235849982047094-2.9686276908348216442160419506053576658620801069272827443504482791274590121245734665403875414106838944580121609294382*I");
	test(307,"Gt({{1,0}},(1)*pow(21,-1)+I*((-1)*pow(3,-1)),tau1)","0.9228578645184130809740113732386698118328501116116277112831033974406455613356158869641439646459995013280071056789291-2.9508982267034555984417160099380815642729161257514727724450835978858504579016339226332469290900175563502647659550192*I");
	test(308,"Gt({{1,0}},(1)*pow(21,-1)+I*((-1)*pow(3,-1)),tau2)","0.9117610833293608712760608141342654117125478868973330206138442186192667699871332542864405468720808682826675600908023-2.9553537117119822237317030503133758721588423364758179333663500512214243966608403469587905631233210677482983415752545*I");
	test(309,"Gt({{1,0}},(1)*pow(21,-1)+I*((-2)*pow(7,-1)),tau1)","0.7264685648312805774953136813003888401376904879473916723423876271149437047113112808149183457586671646001650868671783-2.9338786233202478937138978071668099846102225488406193074012667070776219843132765049913210171595177281493524033679444*I");
	test(310,"Gt({{1,0}},(1)*pow(21,-1)+I*((-2)*pow(7,-1)),tau2)","0.7190945836986565423953386455031425273721130717787919051612125823636113626110091644269868390509696871166427573567623-2.9371295363287834144945647933754620805425562936028347126252288956139301560012371003417842675353387366326727639825564*I");
	test(311,"Gt({{1,0}},(1)*pow(21,-1)+I*((-5)*pow(21,-1)),tau1)","0.5106291276593433924665614955823139851431332243451091839075949470517755225792736714467272250241554972012376816060322-2.9081798295568969891051183068460675555559431819241337111495463900249692774520876613061337095724689854501361782450424*I");
	test(312,"Gt({{1,0}},(1)*pow(21,-1)+I*((-5)*pow(21,-1)),tau2)","0.5059602965218232703264142688993791099480071941839286681739024028614627923905902098674766070048227020995483640678241-2.9105289485306989488689264738724664414692143213137147795936161637959326094071757871932688376573321031941025829111455*I");
	test(313,"Gt({{1,0}},(1)*pow(21,-1)+I*((-4)*pow(21,-1)),tau1)","0.265468687630698823059668687134882422797800909709253248444193243424882004116554838547642456317468219931348964470924-2.8674279752443692162174768800119396490847528033615857529242806390683128907275693715060313814187213209799282388248488*I");
	test(314,"Gt({{1,0}},(1)*pow(21,-1)+I*((-4)*pow(21,-1)),tau2)","0.2627380155441782828061527558550600311566564293359771856411396682798270786817697313789392717747095503564748967825552-2.8690920973632863116272057469749323857913791590937088906277203509845660063641021361062906630905742064830439732702794*I");
	test(315,"Gt({{1,0}},(1)*pow(21,-1)+I*((-1)*pow(7,-1)),tau1)","-0.0255898973792459451772454596532939015031518788671545713756878661395645200205274392566988585241887758486512953850959-2.7977280678703591139340835416947019373023389196799349960584518381480890018346394465207024376603919116412138234775273*I");
	test(316,"Gt({{1,0}},(1)*pow(21,-1)+I*((-1)*pow(7,-1)),tau2)","-0.026970957036504240876875934541722466148154884919762303431078696659920474398137397765851683382396722265751237639912-2.7988596726672582202309211240051162100252660487204655941043587215746630029727984622073703879193841291887831590413377*I");
	test(317,"Gt({{1,0}},(1)*pow(21,-1)+I*((-2)*pow(21,-1)),tau1)","-0.3907160634359150500309649250910565586199781333933889523869905376795409834888151791734631080039227888391794902019828-2.6630925332121254236123131987240805568636048516749501093310317788793346779677276963528053070577954706801496948680582*I");
	test(318,"Gt({{1,0}},(1)*pow(21,-1)+I*((-2)*pow(21,-1)),tau2)","-0.3912123628102974282372743493053793822310434184205180031015542472357358406694436857032250845787261614725220502965846-2.6637946732539116987148050713632574461718234012169707785582771689178097014464538454683747107830737276075905229058999*I");
	test(319,"Gt({{1,0}},(1)*pow(21,-1)+I*((-1)*pow(21,-1)),tau1)","-0.8600600264458236307937798460649354874950174940855839050860302147492566458734346126644648105505985785562415047737062-2.3487344940576978748125114478802262257281286301506392644675809075257657809111719851482955227639188810113429619259731*I");
	test(320,"Gt({{1,0}},(1)*pow(21,-1)+I*((-1)*pow(21,-1)),tau2)","-0.8600555780721843020100416473773141609843634191556685091921793367077621356686585365363072465605662281083201736299846-2.3490706973270619509819878812617144581950490010066692618287709281485316552672957308335475819876714357532298002029774*I");
	test(321,"Gt({{1,0}},(1)*pow(21,-1),tau1)","-1.210378167093500188729032537738612189647498776250138059357070212307774127005934197648568952819351929448205382119218322437874368059822627-1.570796016929224334616092466310084392951576269610822692968517197248810492985580207138958049795570567266366792616613*I");
	test(322,"Gt({{1,0}},(1)*pow(21,-1),tau2)","-1.2102113185575225611042969536237089680118656945425300160175857209671296088055426438497123105184788531392173182311694-1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432558625*I");
	test(323,"Gt({{1,0}},(1)*pow(21,-1)+I*((1)*pow(21,-1)),tau1)","-0.8600612750852700306704884372159733466605188100122522492465023709977240827136671352641041464611560635338700383506809-0.7928581409010452821310282118021226271588162826934449533291316270096828527208454406888709767104740732112797903954918*I");
	test(324,"Gt({{1,0}},(1)*pow(21,-1)+I*((1)*pow(21,-1)),tau2)","-0.860055578072184302010041647377314160984363419155668509192179336707762135668658536536307246560566228108320171693234-0.7925219562627312874806555020177884260021203983684365591461736641592847510189132677944872433544456322289182863092462*I");
	test(325,"Gt({{1,0}},(1)*pow(21,-1)+I*((2)*pow(21,-1)),tau1)","-0.3907186733296491856989752859576306753091432571911079888733688731855704597789155603344880145054852345227854108827276-0.4785019592607606703880549447588760653659788959210795267037890482039190362284891286464690000486455823118634988421391*I");
	test(326,"Gt({{1,0}},(1)*pow(21,-1)+I*((2)*pow(21,-1)),tau2)","-0.3912123628102974282372743493053793822310434184205180031015542472357358406694436857032250845787261614725220503128281-0.4777979803358815397478383119162454380253459981581350424166674233900067048397551531596601145590433403745575636087096*I");
	test(327,"Gt({{1,0}},(1)*pow(21,-1)+I*((1)*pow(7,-1)),tau1)","-0.0255941039136874809583086036539448345807919003498717861734715482770162873842771557027351425803302535865959917539226-0.3438697060597928079747991606861466044221581155215672637698528238396907757064566576618436228405650194219891768567312*I");
	test(328,"Gt({{1,0}},(1)*pow(21,-1)+I*((1)*pow(7,-1)),tau2)","-0.0269709570365042408768759345417224661481548849197623034310786966599204743981373977658516833823967222657512370442569-0.3427329809225350182317222592743866741719033506546402268705858707331534033134105364206644374227329387933649274732106*I");
	test(329,"Gt({{1,0}},(1)*pow(21,-1)+I*((4)*pow(21,-1)),tau1)","0.265462505068041421696440231702536075451134197163855746593747938916206085728661458047899857769478111855360358556773-0.2741748000409327247097913002457957273832170945517478748552918678351651547379666740828414288492340572556405040461612*I");
	test(330,"Gt({{1,0}},(1)*pow(21,-1)+I*((4)*pow(21,-1)),tau2)","0.2627380155441782828061527558550600311566564293359771856411396682798270786817697313789392717747095503564748967860232-0.2725005562265069268354376363045704984057902402813969303472242413232503999221068625217441622515428614991041132447173*I");
	test(331,"Gt({{1,0}},(1)*pow(21,-1)+I*((5)*pow(21,-1)),tau1)","0.5106204114628985853949700712730235270552180960881730796046232229786497185767073962298676274933561641440608495226717-0.2334301180538589213019201173997825179031074955488975082193280591481027916806876830899287567785132016067841860089562*I");
	test(332,"Gt({{1,0}},(1)*pow(21,-1)+I*((5)*pow(21,-1)),tau2)","0.5059602965218232703264142688993791099480071941839286681739024028614627923905902098674766070048227020995483640509081-0.2310637050590942895937169094070364427279550780613910413813284285118837968790332114347659876847849647880455036035505*I");
	test(333,"Gt({{1,0}},(1)*pow(21,-1)+I*((2)*pow(7,-1)),tau1)","0.7264565288869455898944255748601260239213754699765800564518699275711197620920540618649134668146547922773013437727216-0.2077413144585833964128224783072122199176711659612945477031353304149841813271709507394239139503112794124892159407261*I");
	test(334,"Gt({{1,0}},(1)*pow(21,-1)+I*((2)*pow(7,-1)),tau2)","0.7190945836986565423953386455031425273721130717787919051612125823636113626110091644269868390509696871166427573696216-0.2044631172610098239680785899040408036546131057722711083497156966938862502849718982862505578067783313494753225319416*I");
	test(335,"Gt({{1,0}},(1)*pow(21,-1)+I*((1)*pow(3,-1)),tau1)","0.9228414233038929896209541689283123856869995703371456331925604216662913657922436445401859721279800897990057982886223-0.1907354200997270845190747341241375024937489391446801716668394151860692641079425478459381988305746291745363602654244*I");
	test(336,"Gt({{1,0}},(1)*pow(21,-1)+I*((1)*pow(3,-1)),tau2)","0.9117610833293608712760608141342654117125478868973330206138442186192667699871332542864405468720808682826675599288395-0.1862389418778110147309403329661270120383270628992878876085945410863920096253686516692442622187960002338497449413041*I");
	test(337,"Gt({{1,0}},(1)*pow(21,-1)+I*((8)*pow(21,-1)),tau1)","1.1059330663437346158243412888649956630077171694786974485247842384528986449170871250466255824884042049335263505064255-0.1791046567661745448048265564823070940760132230641192224243221011509083142028252333673106469186762363895767692800648*I");
	test(338,"Gt({{1,0}},(1)*pow(21,-1)+I*((8)*pow(21,-1)),tau2)","1.089770150338556783073113416087732536190393668336236644497201843607442740262214360832475280087918762823584998211634-0.1729649627549715942466014326741452183350892924478230766244963131803573941616355320876472839314331735241359255859156*I");
	test(339,"Gt({{1,0}},(2)*pow(21,-1)+I*((-8)*pow(21,-1)),tau1)","1.119929993468462901023260407788397825509944264680213582918086109113396046923445126162328156689707714696378018364567-2.7868474583431318084604999299497812902636163506867126145207925596956337980757388838275808437156169013308698571384488*I");
	test(340,"Gt({{1,0}},(2)*pow(21,-1)+I*((-8)*pow(21,-1)),tau2)","1.1065058570210437381223895884844616256481713816135699877889790641398750792913365995028159207633128569881749990092017-2.7984674446216913042669684133845946632123525395738284048012189906667392888097944051943902862187431967173851595329641*I");
	test(341,"Gt({{1,0}},(2)*pow(21,-1)+I*((-1)*pow(3,-1)),tau1)","0.9428830112407123289887588841106917186485243529884929295021929273282472036033681988814319490620170681619576421282349-2.7653355624367429334968825256624303150010601013454913362292446959885223080470402879024826147158968532723520217794304*I");
	test(342,"Gt({{1,0}},(2)*pow(21,-1)+I*((-1)*pow(3,-1)),tau2)","0.9338275148574441995544246797886958953607907182376462947028180166866224912235060153300041723021627889022877246973132-2.7738556674308882309920415919718292657491244058130230405346627499773793067584425716326626629673134060523178724377411*I");
	test(343,"Gt({{1,0}},(2)*pow(21,-1)+I*((-2)*pow(7,-1)),tau1)","0.7558436443104742562835603060804913792812581002347478710612257245219821816622688390297788134690424672394761505866335-2.734379108724722373018118800552283727713303054063861741215412315513479819078532257710624083790476684354468678202115*I");
	test(344,"Gt({{1,0}},(2)*pow(21,-1)+I*((-2)*pow(7,-1)),tau2)","0.7499907818410624975662116397796363008544285568410359595325338435291155162603215917925581541893142826250407197819361-2.7405978439108075749821405582727511899554275169285090176183992553624973271225624698254621131659358617229523567075794*I");
	test(345,"Gt({{1,0}},(2)*pow(21,-1)+I*((-5)*pow(21,-1)),tau1)","0.5551684382231851781354260231446718643258349247288251886541965435596623369133301381402252271697362525561207084156684-2.6887576095512577857940874025886729705543002795475458758592211742729574149680669895365062109209392251971867949368941*I");
	test(346,"Gt({{1,0}},(2)*pow(21,-1)+I*((-5)*pow(21,-1)),tau2)","0.5516461903744068582807129254491647254970414316986430212094047222382867169077322651104773030929923455437978144146518-2.6932519595939731339590768995698307436185387947121866248473608201000865030807831843165054842084881298727255879754382*I");
	test(347,"Gt({{1,0}},(2)*pow(21,-1)+I*((-4)*pow(21,-1)),tau1)","0.3362697188364498311607382752258876940849705145335123405665836561455658261601855097576240078032182529592364630862271-2.6193251769983287763688999912843114486542514009758721508138999781701157738244174806476013080549738215220474896975374*I");
	test(348,"Gt({{1,0}},(2)*pow(21,-1)+I*((-4)*pow(21,-1)),tau2)","0.3344191072520215310048280315587322637786060361749448746172074770370565383897747427942645086883599078048717952354536-2.622508995937229057576301627192173246600257627289218950655026446900087465423683338471907253953443357063158213017729*I");
	test(349,"Gt({{1,0}},(2)*pow(21,-1)+I*((-1)*pow(7,-1)),tau1)","0.0948064611713618925955347263264169663909605026603202208931505529980471192199580111526784938587958625448819018933122-2.5091682063811600689736985499956085906289747047505625025488271514214901510316582673183215657820846733315546317307945*I");
	test(350,"Gt({{1,0}},(2)*pow(21,-1)+I*((-1)*pow(7,-1)),tau2)","0.0941208375992374284670705402569938290403766461849899971512809522513012746247470151396240000896814848790752075922822-2.5113329869183514446873139288917291033714995120774435007829317052205941927553765601345999542619752132913640641816817*I");
	test(351,"Gt({{1,0}},(2)*pow(21,-1)+I*((-2)*pow(21,-1)),tau1)","-0.1667440477127825634400355357744716888388713892299626169588540560581759468745595582506669498627480936637460800008841-2.326356291411472067605783036469853644026221808828618189587776692230593512386308993709525597831176746789943591444029*I");
	test(352,"Gt({{1,0}},(2)*pow(21,-1)+I*((-2)*pow(21,-1)),tau2)","-0.1666654070806824340065424734345747724587635460175571716537499734921496058569292417171958207845981086439457490755955-2.327699318731213017387003912667976343336564855432688950121458379901539706925212676421111970930922340053086006153375*I");
	test(353,"Gt({{1,0}},(2)*pow(21,-1)+I*((-1)*pow(21,-1)),tau1)","-0.4130953053392157772965664259026252461871935080301477434392486479476107858610941517537326794246552544612568484749493-2.0194559928824853096503846734637029257469506232128625012238767750331481544680588315925371578463970565081388331059086*I");
	test(354,"Gt({{1,0}},(2)*pow(21,-1)+I*((-1)*pow(21,-1)),tau2)","-0.4125837414061463618322583178991174970895275639944983148088667954827277890115267401156606956354752571726658440104832-2.0200991356696462475191813787643460603126185037470475372943896552145923101934155799237633791544364134321065481318087*I");
	test(355,"Gt({{1,0}},(2)*pow(21,-1),tau1)","-0.528463003917468159418347564854353497899708185734367199813762265415285842575300156154577666138996113279841740949407056299179579263529427-1.5707951148651322792933130836157147427955280684388039118038399419860229177322035355718796470858257756114970286788617*I");
	test(356,"Gt({{1,0}},(2)*pow(21,-1),tau2)","-0.5278105928332934756928365470283920291448847701352504702796895219512801066686373866104230342036334197565769998907272-1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432559507*I");
	test(357,"Gt({{1,0}},(2)*pow(21,-1)+I*((1)*pow(21,-1)),tau1)","-0.4130976916710142305053399310021039551522405690030053132244249358029878624172435913304651704981764970997161263925394-1.1221347565908773874830943056919743935433790624292706790718090991734427285782945001358202615563257940141364785548736*I");
	test(358,"Gt({{1,0}},(2)*pow(21,-1)+I*((1)*pow(21,-1)),tau2)","-0.412583741406146361832258317899117497089527563994498314808866795482727789011526740115660695635475257172665844010689-1.1214935179201469909434620045151568238845508956280582836805549370932240960927934187042714461876806545500415383797227*I");
	test(359,"Gt({{1,0}},(2)*pow(21,-1)+I*((2)*pow(21,-1)),tau1)","-0.1667490355997393127726926891867809634021500640864276165356575221367477557882492826863528134307033624038984717254937-0.8152360641668357579588834311076937907931938241595002794577817016122795008354149938060850445881035671621528406277417*I");
	test(360,"Gt({{1,0}},(2)*pow(21,-1)+I*((2)*pow(21,-1)),tau2)","-0.1666654070806824340065424734345747724587635460175571716537499734921496058569292417171958207845981086439457483403258-0.8138933348585802210756394706115265408606045439424168708534862124062766993609963222069228544111947279290620803600077*I");
	test(361,"Gt({{1,0}},(2)*pow(21,-1)+I*((1)*pow(7,-1)),tau1)","0.0947984218715238079295011476303268550236451722120583719482096756478002781935532155634311900417610887167622298221314-0.6324269865186088204695372508149796604222666092415304647559701852135723656309165699038320213199628849331609433801146*I");
	test(362,"Gt({{1,0}},(2)*pow(21,-1)+I*((1)*pow(7,-1)),tau2)","0.0941208375992374284670705402569938290403766461849899971512809522513012746247470151396240000896814848790752075940943-0.6302596666714417937753294543877737808256698872976623201920128870872222135308324384934348710801418546907840223311853*I");
	test(363,"Gt({{1,0}},(2)*pow(21,-1)+I*((4)*pow(21,-1)),tau1)","0.3362579030589583360942703312594221074049265937116002352963964162738067304775165695473214893629425189090023213258027-0.5222743403375526625260541476714505929134680196861965626448824262272270511007613126973474532190451372525158181777006*I");
	test(364,"Gt({{1,0}},(2)*pow(21,-1)+I*((4)*pow(21,-1)),tau2)","0.3344191072520215310048280315587322637786060361749448746172074770370565383897747427942645086883599078048717952192102-0.5190836576525641808863417560873296375969117720858868703199181454077289408625256601561275713886737109189898734968806*I");
	test(365,"Gt({{1,0}},(2)*pow(21,-1)+I*((5)*pow(21,-1)),tau1)","0.5551517803025987774999898197364773960955005307265536338332015979948268034134565002593608887548226680866720614144918-0.452848109356452909830220551163713584203640297118280906984140118489167019215370323422181219761857358760007944453342*I");
	test(366,"Gt({{1,0}},(2)*pow(21,-1)+I*((5)*pow(21,-1)),tau2)","0.5516461903744068582807129254491647254970414316986430212094047222382867169077322651104773030929923455437978152329482-0.448340693995820104503566483709672140578630604662919196127583772207729903205425814311529341133628938109422498539225*I");
	test(367,"Gt({{1,0}},(2)*pow(21,-1)+I*((2)*pow(7,-1)),tau1)","0.7558206418682741123832363442059310156205454175285122501205776671585050161397876154579099208123828332960802409442603-0.4072352482105446659312636344638211412579660167129006441403671271647034106851747852618691765628814092352473756949213*I");
	test(368,"Gt({{1,0}},(2)*pow(21,-1)+I*((2)*pow(7,-1)),tau2)","0.7499907818410624975662116397796363008544285568410359595325338435291155162603215917925581541893142826250407203775912-0.400994809678985663480502825006751694241741882446596803356545336945319079163646528802572712176181206259195729806969*I");
	test(369,"Gt({{1,0}},(2)*pow(21,-1)+I*((1)*pow(3,-1)),tau1)","0.9428515896857254759176141084691701226543339815942034984286194814346788777726249780065766949685031800027355752794845-0.3762906480458572912987971008747741326165736404053571858744069368142326857344856049260499551807917076297862366865037*I");
	test(370,"Gt({{1,0}},(2)*pow(21,-1)+I*((1)*pow(3,-1)),tau2)","0.9338275148574441995544246797886958953607907182376462947028180166866224912235060153300041723021627889022877247019762-0.367736986158905007470601791307673618448044993562082780440281842330437099527766426995372162374803661929830214078113*I");
	test(371,"Gt({{1,0}},(2)*pow(21,-1)+I*((8)*pow(21,-1)),tau1)","1.1198873188894057741478709545836301342829362774902180978227220539828718318101171342667324179309975815038432426774358-0.3547948902784857742370569225229771952017760101786667375787622758907273933827191949755836489344403398299164310533228*I");
	test(372,"Gt({{1,0}},(2)*pow(21,-1)+I*((8)*pow(21,-1)),tau2)","1.1065058570210437381223895884844616256481713816135699877889790641398750792913365995028159207633128569881749990126697-0.3431252089681019341956749698949082209848168598012774161737256016410771174764145934336445391233738712647629269820326*I");
	test(373,"Gt({{1,0}},(1)*pow(7,-1)+I*((-8)*pow(21,-1)),tau1)","1.1410735063189755483168626280157591457551572962361424725617345789021040268472074743868529935615151988796366967284989-2.6172641531531600009242928036845792284512861129055737518887992293466488847126599960396721751832749687573169388104138*I");
	test(374,"Gt({{1,0}},(1)*pow(7,-1)+I*((-8)*pow(21,-1)),tau2)","1.1319553727956824895145487389422265612163483926716759137184254071931760421050863279608100243243954831484720712119857-2.6333399759000016171661062154851598603728397180003100633049351176416106750902203794626649023529174384451280392832335*I");
	test(375,"Gt({{1,0}},(1)*pow(7,-1)+I*((-1)*pow(3,-1)),tau1)","0.972740573896441307349053936543657855733931834121617144473340835257552216223365084415085471793036244701699216845813-2.5888948652930224319796929838630126392458916787464784573476239512911820931696718828983573768099597288122848900829769*I");
	test(376,"Gt({{1,0}},(1)*pow(7,-1)+I*((-1)*pow(3,-1)),tau2)","0.9668744601887265950664660185478352780662486265103850396065614174344874721961354100492156637518027255308831621623875-2.6006947176950811072654531685933343137440708493384918348747685282927778080125179861603325360385264922134798953303674*I");
	test(377,"Gt({{1,0}},(1)*pow(7,-1)+I*((-2)*pow(7,-1)),tau1)","0.7986752253653860089084589433233936978122887938922091597312867522750101925419295539320560306727371325751709622965804-2.5489411768421118841054646808020462713624117162321938048749885461272709134762292957638832814252597691519516911204617*I");
	test(378,"Gt({{1,0}},(1)*pow(7,-1)+I*((-2)*pow(7,-1)),tau2)","0.7952022302429600612825099857541436625771915393307172150699115718612543355512880283642036147262994528960350939085985-2.5575602745539998471220174100141258616359314465485454572840068944189404862447509611364057643307432656770615208251789*I");
	test(379,"Gt({{1,0}},(1)*pow(7,-1)+I*((-5)*pow(21,-1)),tau1)","0.6178676519204287000800897825735126392766540850356238541664319637838456718921143622231109723863891053964349024756662-2.4919552609624579963082615778127043881357900978155159584134508224852429691185027313964504642043071929496627066466503*I");
	test(380,"Gt({{1,0}},(1)*pow(7,-1)+I*((-5)*pow(21,-1)),tau2)","0.6161410086553309514328395604993949870665591872603851711333342807423815148730509733849067683058304537560898321251006-2.4981876483813654804140261479852152879880986144548108309631275431879041522908968491392949072491175017153496136210513*I");
	test(381,"Gt({{1,0}},(1)*pow(7,-1)+I*((-4)*pow(21,-1)),tau1)","0.4303297823713277584281530638302039063912073455992061851353296136875447717637900086838769911552003428779118352444195-2.4095455001970766366305006752205593193876802083043210891041925346182595516823627794718928367785945956614372412612352*I");
	test(382,"Gt({{1,0}},(1)*pow(7,-1)+I*((-4)*pow(21,-1)),tau2)","0.4298582662673248079978967298809126821215226434100188663833673006568305757949826671645457126557397904412838604639284-2.4139621279254864008615514230714680622229341839398927540087954983319823168487696796913969385043461793973491556945769*I");
	test(383,"Gt({{1,0}},(1)*pow(7,-1)+I*((-1)*pow(7,-1)),tau1)","0.2394469509324897073011930839599071780325848278501326444948405600934166489582654248957859560263974757918491706701743-2.2890766074815442086156556507039849153515658321473478669093371110574428379352983325152295218277443719087539743166132*I");
	test(384,"Gt({{1,0}},(1)*pow(7,-1)+I*((-1)*pow(7,-1)),tau2)","0.2398514908248074928176646037955752934360897240799585298007668890444126221272410491258200480110104488453733612646705-2.2920803544047981280620306316784128185724246128093884306092707994898864596884075857337192858363405138861796827419444*I");
	test(385,"Gt({{1,0}},(1)*pow(7,-1)+I*((-2)*pow(21,-1)),tau1)","0.0575218949673202110730378337325357509531396862210934317557147076140206277539767706358566918178287822477375635248702-2.1137065495330409206243646344030064603484365832528024401857308677764646564561450927918896095257599826168928413449461*I");
	test(386,"Gt({{1,0}},(1)*pow(7,-1)+I*((-2)*pow(21,-1)),tau2)","0.0585018732728225391420972592674303042762364035616894776390933718396480273879419244522313149950996587121688829282782-2.1155704790829426786267152716525727634903180046892939838502349902531626233056047201577098395242040058812828893946498*I");
	test(387,"Gt({{1,0}},(1)*pow(7,-1)+I*((-1)*pow(21,-1)),tau1)","-0.0852677062845319055710992638931252292291486677921192809701959700167097080551368266597302321064675301299569066376329-1.8698935304451971274060073266291219340124728513832422273575810734352386284752175041321599923290107331136610667532884*I");
	test(388,"Gt({{1,0}},(1)*pow(7,-1)+I*((-1)*pow(21,-1)),tau2)","-0.0839612999587680637968331841250241057707792731170431346505788253185656699770255428656799795337276441328013544813545-1.8707865505257337702730760137266618865535197591942325139434330703930777097723489269098106279802932813841513826318408*I");
	test(389,"Gt({{1,0}},(1)*pow(7,-1),tau1)","-0.141831482450270921865731300462543003584046028991388574570934051930208658116944854846536603976251899856970594103499290516256891253422995-1.5707937007549736595567685528250247784337968601705685604652303310218280097973960392048485165460092240984236901388793*I");
	test(390,"Gt({{1,0}},(1)*pow(7,-1),tau2)","-0.1404183573785678315066136137186492654420959533317161432222159194816338836589995218445567924012351000041858141140928-1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432558742*I");
	test(391,"Gt({{1,0}},(1)*pow(7,-1)+I*((1)*pow(21,-1)),tau1)","-0.0852710182726299123954472069406040878673476906017473668411428553904282508496116600979861103902457878166124015858522-1.2716942632691968528627113891548712712660097795180157105971917941797855131596066015559595091075892535968098175549808*I");
	test(392,"Gt({{1,0}},(1)*pow(7,-1)+I*((1)*pow(21,-1)),tau2)","-0.0839612999587680637968331841250241057707792731170431346505788253185656699770255428656799795337276441328013544700976-1.270806103064059468189567369552840997643649640180873307031511521914738696513860071718224197361823786597996703879407*I");
	test(393,"Gt({{1,0}},(1)*pow(7,-1)+I*((2)*pow(21,-1)),tau1)","0.0575149722827858294464150163743482842910464383066456379050546309947933263000943670819025925434193414371311081546875-1.0278824561676277154324881346026213591384994506138201251733613371710663100980186183092576357875200911218140139920925*I");
	test(394,"Gt({{1,0}},(1)*pow(7,-1)+I*((2)*pow(21,-1)),tau2)","0.0585018732728225391420972592674303042762364035616894776390933718396480273879419244522313149950996587121688829273908-1.0260221745068505598359281116269301207068513946858118371247096020546537829806042784703249858179130621008651971181694*I");
	test(395,"Gt({{1,0}},(1)*pow(7,-1)+I*((1)*pow(7,-1)),tau1)","0.239435793194325304596605266362688047598403788470258202536653381649907210924456947082090213903105732121996030444213-0.8525145392963174954564132625326322137273031414863733795556197482078644839030005939275768301911362198128780144515606*I");
	test(396,"Gt({{1,0}},(1)*pow(7,-1)+I*((1)*pow(7,-1)),tau2)","0.2398514908248074928176646037955752934360897240799585298007668890444126221272410491258200480110104488453733599514468-0.8495122991849951104006127516010900656247447865657173903656737928179299465978014128943155395057765540959684037724899*I");
	test(397,"Gt({{1,0}},(1)*pow(7,-1)+I*((4)*pow(21,-1)),tau1)","0.4303133832626843113254625703493770417189835138217647828676298014436257268582849213012133534070243820196340820428532-0.7320489098527339824331522704032857664683906222053756200751884578813130773845007943150324615959501542432313456964255*I");
	test(398,"Gt({{1,0}},(1)*pow(7,-1)+I*((4)*pow(21,-1)),tau2)","0.4298582662673248079978967298809126821215226434100188663833673006568305757949826671645457126557397904412838604735251-0.7276305256643068376010919602080348219742352154352130669661490939758340894374393189366378868377708885847989308202102*I");
	test(399,"Gt({{1,0}},(1)*pow(7,-1)+I*((5)*pow(21,-1)),tau1)","0.6178445324050452598565070183272597881246829315700722982064473134673588616776153121997457146805457574763016783041223-0.649643828868684767435661734073686737955527806218063066283113667660152377156130537509418387575937088534043890606042*I");
	test(400,"Gt({{1,0}},(1)*pow(7,-1)+I*((5)*pow(21,-1)),tau2)","0.6161410086553309514328395604993949870665591872603851711333342807423815148730509733849067683058304537560898328165952-0.6434050052084277580486172352942875962090707849202949900118170491199122539953121494887399180929995662667984728940473*I");
	test(401,"Gt({{1,0}},(1)*pow(7,-1)+I*((2)*pow(7,-1)),tau1)","0.7986433002932507510103395980133484392632026850213620726776092889548302766304541208332417326703130982221996348919639-0.5926644313494119056436613353443659387893472279489124129737450744171506257709070599704120603271684813610794868431794*I");
	test(402,"Gt({{1,0}},(1)*pow(7,-1)+I*((2)*pow(7,-1)),tau2)","0.7952022302429600612825099857541436625771915393307172150699115718612543355512880283642036147262994528960350938954224-0.5840323790357933913406259732653770225612379528265603636909376978888759200414580374916290610113738023050865656896423*I");
	test(403,"Gt({{1,0}},(1)*pow(7,-1)+I*((1)*pow(3,-1)),tau1)","0.9726969639440296935552065268111171130869542768792004722338537407822079311803615760688355415725807839009897030132261-0.5527196877291160216937962101390646727324332359297599799036111104043311503539026493813250568013327215950304409242925*I");
	test(404,"Gt({{1,0}},(1)*pow(7,-1)+I*((1)*pow(3,-1)),tau2)","0.9668744601887265950664660185478352780662486265103850396065614174344874721961354100492156637518027255308831637904373-0.5408979358947121311971902146861685704530985500366139861001760640150385982736910124677022893035905757686681911847357*I");
	test(405,"Gt({{1,0}},(1)*pow(7,-1)+I*((8)*pow(21,-1)),tau1)","1.1410142783029488199103988639570711709504841889934159090930231024861373459944915537537223316839478804867967424298131-0.5243625779044621565217856409267602539142478523423002995487607314195746841057449802493680959124912392142962622182659*I");
	test(406,"Gt({{1,0}},(1)*pow(7,-1)+I*((8)*pow(21,-1)),tau2)","1.1319553727956824895145487389422265612163483926716759137184254071931760421050863279608100243243954831484720699137709-0.5082526776897916212965371677943430238243296813747957576700094746662057311959886191653699229891996295370200472318282*I");
	test(407,"Gt({{1,0}},(4)*pow(21,-1)+I*((-8)*pow(21,-1)),tau1)","1.1667577972167879597658495539906107027436611857981900380306784490152771481164258409555004963760452662904006424988215-2.4555042934816394444581951320781241196369246709783130361255405573853593518387127829927246474565901293836557538972166*I");
	test(408,"Gt({{1,0}},(4)*pow(21,-1)+I*((-8)*pow(21,-1)),tau2)","1.1630748503497888479578209821325426389706215069207375875534294534975690387889848815990918847665125013696867989537969-2.4745546191756813360256918556783345732311386323937434363491332097813889870052229607539877830032250467425614647069623*I");
	test(409,"Gt({{1,0}},(4)*pow(21,-1)+I*((-1)*pow(3,-1)),tau1)","1.0083427265201923953450702894996138089599646914689200424451588063043770018182306552960684001312700483827136956843637-2.4237367001717660627625749947259442212769812447969827907781571616382734192945280899350961491752887888664211008076547*I");
	test(410,"Gt({{1,0}},(4)*pow(21,-1)+I*((-1)*pow(3,-1)),tau2)","1.0065126879481865999971454602232857976070428265038225069735977572490920481264044771095312617480664159745524494989701-2.4377391328922638240112201134092307160827951386740688306380163453706291834964571070503053603576796306106852250250219*I");
	test(411,"Gt({{1,0}},(4)*pow(21,-1)+I*((-2)*pow(7,-1)),tau1)","0.8483830452467150623903656644976183989098162100437330185599862821531841599827782984712556445360688487468937936098028-2.3800561603951598197813521815003257425235220759340941252725248488522030122053772614947539783113436074199503977767816*I");
	test(412,"Gt({{1,0}},(4)*pow(21,-1)+I*((-2)*pow(7,-1)),tau2)","0.8479273718632661674407272356195824532078238480603705666286175573230211346063672499435198383454339252504875662051722-2.3902943340939732862124578387118716478491084338996278194631059655935371668967065828944250089771179265967515826727433*I");
	test(413,"Gt({{1,0}},(4)*pow(21,-1)+I*((-5)*pow(21,-1)),tau1)","0.687772728439514674081545678720978726246554213542896421418683861334048819739916676646856019072891368571694702622444-2.3198533923565274418034808012724334029928129260104942528362635656336269264992306581985140142876194638632473443409712*I");
	test(414,"Gt({{1,0}},(4)*pow(21,-1)+I*((-5)*pow(21,-1)),tau2)","0.6883257800224166759529571497171703086872506065599721819405203739171152364818276588353864582396949900858912839478404-2.3272618672670686116505693266147076767060009645692586268013803839264796396306159152397934919304502664438076572552846*I");
	test(415,"Gt({{1,0}},(4)*pow(21,-1)+I*((-4)*pow(21,-1)),tau1)","0.5290747838232714659087090111804138904872130814057160142047088719849521396523538798524378289467130337165918900478438-2.2369609834113581460360591652564197320372517620445879387693976007673565142220812311703364956777084193725124569346914*I");
	test(416,"Gt({{1,0}},(4)*pow(21,-1)+I*((-4)*pow(21,-1)),tau2)","0.5303558956369578307920033572145029335727796084525094678768971330011270825627067766072310695822311425693826268954493-2.2422138043478172830070680382930238839026282731367677032922081869135719135568804587713695267039259572525108299588982*I");
	test(417,"Gt({{1,0}},(4)*pow(21,-1)+I*((-1)*pow(7,-1)),tau1)","0.3782008062143548406498402869223228002346803591880605756795561213092600501093584088504154827551006709024885747648445-2.1238819702958042375850507197475842692254195683780847756666390102339736286829771387108596206975917409466949381990464*I");
	test(418,"Gt({{1,0}},(4)*pow(21,-1)+I*((-1)*pow(7,-1)),tau2)","0.379991716210343962942934136495523747451726303737398139066304688080516029663455540202195953523325233807615007310481-2.1274558236924119880725419030978813452049467505309234837951210041538067058438793829511704110548366569747226216895097*I");
	test(419,"Gt({{1,0}},(4)*pow(21,-1)+I*((-2)*pow(21,-1)),tau1)","0.2468069307076976696996064539871299603542504248803919050982998707417966255538545676242777118929324582611466609380548-1.9736807914649972208019904045140965985340404548678576787289397477086987712059700458339513313005440167693176161395481*I");
	test(420,"Gt({{1,0}},(4)*pow(21,-1)+I*((-2)*pow(21,-1)),tau2)","0.2489335928686257966248921571837798043446694538790236277879572840490887450214425251445220644613635250042966193756975-1.9758992986029331542777489485604778004502476953788781183683901842443467528478538692704883917570704011759636818255029*I");
	test(421,"Gt({{1,0}},(4)*pow(21,-1)+I*((-1)*pow(21,-1)),tau1)","0.1535634031527394718155357081609676167951304674772156521081969697188403469316713094052920969192452073261376320717529-1.7852429312607154155259611949328142165597844208816017801809997080093626626991670947554022608688361089593599283207213*I");
	test(422,"Gt({{1,0}},(4)*pow(21,-1)+I*((-1)*pow(21,-1)),tau2)","0.155881122564933614831232912886369456864235701466075627104576927044867336971354459316761216490964071855755925983861-1.7863065400991397231468020783610203008817505517716690096151964088185134074863232167359332787712704736231280390390962*I");
	test(423,"Gt({{1,0}},(4)*pow(21,-1),tau1)","0.119239854945097020277419318089873887826036646260608268379616841005056260646621419997476342686373840786172328664420309814633895606607511-1.5707919002486417762467224711922086578213500102891827271097195927979125746088221700102883529819517473885616750297528*I");
	test(424,"Gt({{1,0}},(4)*pow(21,-1),tau2)","0.1216207204171367447405275832259101693329370167420756025660845656066614046087734548615972486507899482801855455781318-1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432559506*I");
	test(425,"Gt({{1,0}},(4)*pow(21,-1)+I*((1)*pow(21,-1)),tau1)","0.1535594597930188229829651331598142319798714479017023010018999054277805873119295170153870860941718468737892525095607-1.356341099053277059450674601654940155331767503538596749563286017647414717519961923609237117644487933162422463673917*I");
	test(426,"Gt({{1,0}},(4)*pow(21,-1)+I*((1)*pow(21,-1)),tau2)","0.1558811225649336148312329128863694568642357014660756271045769270448673369713544593167612164909640718557559259801794-1.3552861134906535153158413049184825833154188476034368113597481834893029987998857818921015465708465943590200474723199*I");
	test(427,"Gt({{1,0}},(4)*pow(21,-1)+I*((2)*pow(21,-1)),tau1)","0.2467986883364861885818844130280805534205563908933693649603398545267825981537712253203801494478137055032404105394004-1.1679039490263001637261025020947509375604151938063056421032320232523173315763223785799013758726567117454544306080265*I");
	test(428,"Gt({{1,0}},(4)*pow(21,-1)+I*((2)*pow(21,-1)),tau2)","0.2489335928686257966248921571837798043446694538790236277879572840490887450214425251445220644613635250042966193754918-1.1656933549868600841848944347190250837469217039962277026065544080634696534383551293575464335850466668061844046860284*I");
	test(429,"Gt({{1,0}},(4)*pow(21,-1)+I*((1)*pow(7,-1)),tau1)","0.3781875214518648347082095695250385324461438774201624845868674570434495668297759638168915243238482068050906287594814-1.0177040247843135060845512712273750897389417447028182445045620331869001009376616813613872823763932719110984000550808*I");
	test(430,"Gt({{1,0}},(4)*pow(21,-1)+I*((1)*pow(7,-1)),tau2)","0.3799917162103439629429341364955237474517263037373981390663046880805160296634555402021959535233252338076150073029281-1.0141368298973812503901014801816215389922226488441823371798235881540097004423296156768644142872804110074254648233135*I");
	test(431,"Gt({{1,0}},(4)*pow(21,-1)+I*((4)*pow(21,-1)),tau1)","0.5290552585162450448547051082349128632980233225409273238285126463570041150411125908329419931172971787218834608887083-0.9046269238205112213273083242187244935848240122339671602594503281344156783496976041560305295737748608546639341679172*I");
	test(432,"Gt({{1,0}},(4)*pow(21,-1)+I*((4)*pow(21,-1)),tau2)","0.5303558956369578307920033572145029335727796084525094678768971330011270825627067766072310695822311425693826276307191-0.8993788492419759554555753449864790002945411262383381176827364053942444927293285398566652986381911107296372565544845*I");
	test(433,"Gt({{1,0}},(4)*pow(21,-1)+I*((5)*pow(21,-1)),tau1)","0.6877452015997336104841915260928743089620246671472651945067742906819346734390660348515119561187156333082525003332278-0.8217372570470678237565854706317240416992380465409759488884530641491629185468303795711081860384397309978468591781907*I");
	test(434,"Gt({{1,0}},(4)*pow(21,-1)+I*((5)*pow(21,-1)),tau2)","0.6883257800224166759529571497171703086872506065599721819405203739171152364818276588353864582396949900858912845477915-0.8143307863227246268120740566647952074911684348058471941735642083813367666555930833882413334116668015383404292583854*I");
	test(435,"Gt({{1,0}},(4)*pow(21,-1)+I*((2)*pow(7,-1)),tau1)","0.8483450342274030015000859507535275852541574543669389277471820323304136773808155593637961214062190982656795587493263-0.7615383085167137594635023108393694613510583475186962553513400215705131289026071689877590159134191713629826990534494*I");
	test(436,"Gt({{1,0}},(4)*pow(21,-1)+I*((2)*pow(7,-1)),tau2)","0.8479273718632661674407272356195824532078238480603705666286175573230211346063672499435198383454339252504875662069846-0.7512983194958199522501855445676312363480609654754780015118386267142792393895024157336098163649991413853965038401238*I");
	test(437,"Gt({{1,0}},(4)*pow(21,-1)+I*((1)*pow(3,-1)),tau1)","1.0082908031060025535958519363787785037899455610557112127626292913267514220039193777657862060298964847975434098097613-0.7178630100665444353714682397344579856984074787672006966417609676179690826565497862612905706886354027414753137173145*I");
	test(438,"Gt({{1,0}},(4)*pow(21,-1)+I*((1)*pow(3,-1)),tau2)","1.0065126879481865999971454602232857976070428265038225069735977572490920481264044771095312617480664159745524509754321-0.7038535206975294144514232698702721681143742607010369903369282469371872227897518915777294649844374373714628614895672*I");
	test(439,"Gt({{1,0}},(4)*pow(21,-1)+I*((8)*pow(21,-1)),tau1)","1.1666872784329227661503661701564843048805202912433818043384142738467844973598871213979792151442012885535494509836772-0.6861025526167127771395189303146231186056421494935897627679771457885734662806849713856462477180670065047452103580082*I");
	test(440,"Gt({{1,0}},(4)*pow(21,-1)+I*((8)*pow(21,-1)),tau2)","1.1630748503497888479578209821325426389706215069207375875534294534975690387889848815990918847665125013696868062738094-0.6670380344141119024369515276011683109660307669813623846258113825264274192809860378740470423388920212395866766532483*I");
	test(441,"Gt({{1,0}},(5)*pow(21,-1)+I*((-8)*pow(21,-1)),tau1)","1.1941729010985121048679490207042458541688998973347545532179534608261670136231629018275052638999042878348167153279245-2.3021747173685997570188929414448752074293639518887589428025455544360164577055981430312928508612566147382448713343367*I");
	test(442,"Gt({{1,0}},(5)*pow(21,-1)+I*((-8)*pow(21,-1)),tau2)","1.1965438106744371216097828598163255440443664073958626057203636217497583430169619227376151430229338129201308272323576-2.3224715343939299880559783860421361012146119833900293769661143719042891569584574345708298992827170247638137247152108*I");
	test(443,"Gt({{1,0}},(5)*pow(21,-1)+I*((-1)*pow(3,-1)),tau1)","1.0456024254699263769905833161084904698147611685657152795976959386804844940919424857234064760929755817819069139702015-2.2702774698078610393431325789039516491088657569154400295043817537532193752544298842110102384817867618433424177802864*I");
	test(444,"Gt({{1,0}},(5)*pow(21,-1)+I*((-1)*pow(3,-1)),tau2)","1.0482813590580865623617127272432057111966376984580492908192901913703824782205157763773878000626858395443526258237267-2.2852190501553155817716900140970231948587702977865833009395635913752974607999231852851255652310266969150278900355784*I");
	test(445,"Gt({{1,0}},(5)*pow(21,-1)+I*((-2)*pow(7,-1)),tau1)","0.8989978285658107318923046457148934713216391535521580911690490799148930142272450675581955301047669150131560597271932-2.2274232750409287637666190288993795425323962451063175676317814119352569835068856722866245747882245914256097740701606*I");
	test(446,"Gt({{1,0}},(5)*pow(21,-1)+I*((-2)*pow(7,-1)),tau2)","0.9019204805636299918773641476280032870189870490972635422842838071566959944609690822556665327322080098462509214085876-2.2383604671877415352837560529078933712256513101492882239998136413677603008240384586075674655041103044634533881290321*I");
	test(447,"Gt({{1,0}},(5)*pow(21,-1)+I*((-5)*pow(21,-1)),tau1)","0.7562962310875739796646555455293105383760260612179341538089631750391540097447058524580068231387969898776997963226238-2.1701788022173831539832553734544944142364206440461099608160017207908956269593997774729998041093097772619427757601612*I");
	test(448,"Gt({{1,0}},(5)*pow(21,-1)+I*((-5)*pow(21,-1)),tau2)","0.7594059085818843782850265354908093284344584045667422819005058127235561578883702985109812023419858100158295484280163-2.1780996685602704822221161325118095393271758364148267681702705421725960685306312955340626935336786701520794478157481*I");
	test(449,"Gt({{1,0}},(5)*pow(21,-1)+I*((-4)*pow(21,-1)),tau1)","0.6209626434189134180492396482839394644800399019114738383748829615879881414645622513468552582650587363679143748348771-2.0945558098419769499310669394361508220480367696434829518096773365680466136680890764247909816266542294988258626985669*I");
	test(450,"Gt({{1,0}},(5)*pow(21,-1)+I*((-4)*pow(21,-1)),tau2)","0.6242116442348698751680052439359559641117981698380312468185827291761393914555784955980796250694477029854599015538422-2.1001753985063021314436095003880384514766311935069959485635358612420189919042568572031172653260707261941638852010324*I");
	test(451,"Gt({{1,0}},(5)*pow(21,-1)+I*((-1)*pow(7,-1)),tau1)","0.4988114456183256329716317719739173131983030449540439142055105287585809545398395394853476740225073466214528041037577-1.9966623057448010734540937493117245133508248530731753979521653263647402666184216689178551348753725906009569850660287*I");
	test(452,"Gt({{1,0}},(5)*pow(21,-1)+I*((-1)*pow(7,-1)),tau2)","0.5021603228108033055929250613327917078213104108658235086943340234250911237152746831852501760031686100219895979090374-2.0004875460448350413300992409966578897228140288680447834261638628039009737175861046784012201965671432564345756652645*I");
	test(453,"Gt({{1,0}},(5)*pow(21,-1)+I*((-2)*pow(21,-1)),tau1)","0.3986305675067047567009394139526337830760167401577023399071814672551718409705471302738466225536168598746662042353534-1.8742855020186493281690255249328157447170396750159272050019203166662876607843687938741285486972188857848632051224669*I");
	test(454,"Gt({{1,0}},(5)*pow(21,-1)+I*((-2)*pow(21,-1)),tau2)","0.4020465402034643231158251452929979214876524126807808392582168845093430785131508842234280256957486756427912552561345-1.8766611634850571664349950513911694836206543338572700285663057467150251174013099508759760134263601674328712481368193*I");
	test(455,"Gt({{1,0}},(5)*pow(21,-1)+I*((-1)*pow(21,-1)),tau1)","0.3315341696349846510054077000284788928657724099182831464528030627969173354388132584890110263967897638288397374299903-1.7294821803588239008725430375389272778774292297152534983897853293037738315507624873066197659302547916380989553029951*I");
	test(456,"Gt({{1,0}},(5)*pow(21,-1)+I*((-1)*pow(21,-1)),tau2)","0.3349892677550318015665425201494741910801340295920861745154020168855272056539257745679917185508299364983980114026286-1.7306221032011611301750920390676608352982592925023429128444255638424855055485275293739980301666798464453069000322961*I");
	test(457,"Gt({{1,0}},(5)*pow(21,-1),tau1)","0.30773869357579759981546895289576434461571334604882160597307374095158629257189814988171888242948908934707703134539085722646877011457591-1.5707898733290256090895058073596473417895301439584475782503089111672378748914811620704816522204582933005324140168922*I");
	test(458,"Gt({{1,0}},(5)*pow(21,-1),tau2)","0.3112078773517370084384907643172623435667210127740447092394380259396560408586983295023740296185416457626622000215133-1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432560567*I");
	test(459,"Gt({{1,0}},(5)*pow(21,-1)+I*((1)*pow(21,-1)),tau1)","0.3315299452884577055873763111793744682785205295004778495144003638147480350274789526091711177950740324170054676101431-1.412097613307975364767797977957093667029543526203447293427641716413939941213020475628579062871034741798763245610441*I");
	test(460,"Gt({{1,0}},(5)*pow(21,-1)+I*((1)*pow(21,-1)),tau2)","0.3349892677550318015665425201494741910801340295920861745154020168855272056539257745679917185508299364983980123162818-1.4109705503886321082875513442118420488989101068727629081305190284653309007376814692540367951754372215368411864794982*I");
	test(461,"Gt({{1,0}},(5)*pow(21,-1)+I*((2)*pow(21,-1)),tau1)","0.398621737819669414182349833671120572268217476230309701983415183370947967709453352386119675982177208783460646260782-1.2672944369141149415378444530383534042161290584437719455148596148127842942205939288481150449932225507422952820072494*I");
	test(462,"Gt({{1,0}},(5)*pow(21,-1)+I*((2)*pow(21,-1)),tau2)","0.4020465402034643231158251452929979214876524126807808392582168845093430785131508842234280256957486756427912560715391-1.2649314901047360720276483318883334005765150655178357924086388455927912888848990477520588119157569005492768383750663*I");
	test(463,"Gt({{1,0}},(5)*pow(21,-1)+I*((1)*pow(7,-1)),tau1)","0.4987972142409566005817648517181988953388706881646051397239945640307109740804775656388464601950158717969870844305761-1.1449178898126883388406072804427653062193765249493687542434535515550169514430334928178824786656746227587982889164899*I");
	test(464,"Gt({{1,0}},(5)*pow(21,-1)+I*((1)*pow(7,-1)),tau2)","0.5021603228108033055929250613327917078213104108658235086943340234250911237152746831852501760031686100219895978987258-1.1411051075449581971325441422828449944743553705070610375487807295039154325686228939496336051455499247257135108467457*I");
	test(465,"Gt({{1,0}},(5)*pow(21,-1)+I*((4)*pow(21,-1)),tau1)","0.6209417268227190142425354593681929143431293291116064330646228398752196840762151805137862709556550561756075219368942-1.0470247768439919605785726394981098033602771997244171468672884440432866565203900721775578491118080595473125093133439*I");
	test(466,"Gt({{1,0}},(5)*pow(21,-1)+I*((4)*pow(21,-1)),tau2)","0.6242116442348698751680052439359559641117981698380312468185827291761393914555784955980796250694477029854599015448734-1.0414172550834911070190338828914644327205382058681098724114087310657974143819521414249175600160463417879842013122179*I");
	test(467,"Gt({{1,0}},(5)*pow(21,-1)+I*((5)*pow(21,-1)),tau1)","0.7562667428039101114902895409406703459168552505056659415151210707225617671245262130049780122695132934399598390233944-0.9714023453767128485088385642485809077510065656535468499874707682524774560150285744591341249864915670235477037116064*I");
	test(468,"Gt({{1,0}},(5)*pow(21,-1)+I*((5)*pow(21,-1)),tau2)","0.7594059085818843782850265354908093284344584045667422819005058127235561578883702985109812023419858100158295484250601-0.9634929850295227562405272507676933448699935629602790528046740501352203377555777030939721318084383978300686386987569*I");
	test(469,"Gt({{1,0}},(5)*pow(21,-1)+I*((2)*pow(7,-1)),tau1)","0.8989571090451981335119439239871647550399066673789651610027263201030216333750315141448575681293947274137454503035016-0.9141586538292684884547396722615416966855143978733225292718095124769470686445698330966575526260534252543235914858313*I");
	test(470,"Gt({{1,0}},(5)*pow(21,-1)+I*((2)*pow(7,-1)),tau2)","0.9019204805636299918773641476280032870189870490972635422842838071566959944609690822556665327322080098462509198745479-0.9032321864020517031788873303716095129715180892258175969751309509400561054621705400204673598380067635186946983847866*I");
	test(471,"Gt({{1,0}},(5)*pow(21,-1)+I*((1)*pow(3,-1)),tau1)","1.0455468022171755878823812559329544160775615475488975005298235372425448963752775149311589212384180678655489014676264-0.8713055311697349166546877019436199200605630841909400817752814641033861504013540559237453572985240241171619856345333*I");
	test(472,"Gt({{1,0}},(5)*pow(21,-1)+I*((1)*pow(3,-1)),tau2)","1.0482813590580865623617127272432057111966376984580492908192901913703824782205157763773878000626858395443526258327095-0.856373603434477656690953369182479689338399101588522520035381000932518945486285813342909260111090371067120196479108*I");
	test(473,"Gt({{1,0}},(5)*pow(21,-1)+I*((8)*pow(21,-1)),tau1)","1.1940973574502321600298123430024184187278044161931292891629605294200202329612626785082090752834504513620039271749238-0.8394097432411028583757440382304922712347452121307997848843949071674079440458136710771050416413399959824990431158738*I");
	test(474,"Gt({{1,0}},(5)*pow(21,-1)+I*((8)*pow(21,-1)),tau2)","1.1965438106744371216097828598163255440443664073958626057203636217497583430169619227376151430229338129201308272598149-0.8191211191958632504066649972373667829825574159850764440088302204035272493277515640572049260594000432183343617984937*I");
	test(475,"Gt({{1,0}},(2)*pow(7,-1)+I*((-8)*pow(21,-1)),tau1)","1.2207378221045447735379167204151650838119181842012146061066107498229815968879932826704268101989129056715434623010209-2.1569600926159052333834205713263494739659776868578269176739590921383181077755310633743561482801597292268712771075598*I");
	test(476,"Gt({{1,0}},(2)*pow(7,-1)+I*((-8)*pow(21,-1)),tau2)","1.2292308355537207951046177286550798117720991416749447097867014355332495594909645690040037650093104435163084142464727-2.1766943565881955485271601293797551381836274985819156744321308921667147221890972226695773303179389706288284066952986*I");
	test(477,"Gt({{1,0}},(2)*pow(7,-1)+I*((-1)*pow(3,-1)),tau1)","1.0810396596069697031645837888462358228596525541883192656642885625808607189632967845605324261566316255743866612381085-2.1276538354464933411329256398174977871165555430707382884990456899627545419951958245689429082569895170044453440712273*I");
	test(478,"Gt({{1,0}},(2)*pow(7,-1)+I*((-1)*pow(3,-1)),tau2)","1.088292748512223589454122014214708723479944390856849630902466929687866968981114846133069820764075884683709173153188-2.1422038833171031747751123567797758509877982268729211174748062847516427573975267620046982642667124340386648861935082*I");
	test(479,"Gt({{1,0}},(2)*pow(7,-1)+I*((-2)*pow(7,-1)),tau1)","0.9459573004981606292536306872542865691845324443143649879831414447977920698191248681837991648911867014308867007971869-2.0890750610235830619300743486139202831043218415010243904721155568171037242724007795914970581765873872119999480980039*I");
	test(480,"Gt({{1,0}},(2)*pow(7,-1)+I*((-2)*pow(7,-1)),tau2)","0.9523151351022059475825597522732810878100953758198629981561551547656185236784107830171890156811913852786646468678868-2.0997379470421577257071749143347697848460673026435656252434578652669589246096600960217987863255986525848855363043715*I");
	test(481,"Gt({{1,0}},(2)*pow(7,-1)+I*((-5)*pow(21,-1)),tau1)","0.8178449082344258850742964321723475838821585828404789739280194258676620843998111170833774027441531501545653297499201-2.0388720781717998640710243615764363338000512561850220859153692773987742679028607472918702491305991332895169168793852*I");
	test(482,"Gt({{1,0}},(2)*pow(7,-1)+I*((-5)*pow(21,-1)),tau2)","0.8235587590455172353624229294509635325378785153260023993574711302510588505399978827434026255241279922790570086398887-2.0466009412799696505105296501365986579004198774337642733163522821541253417636139510857725513740833835068928606725841*I");
	test(483,"Gt({{1,0}},(2)*pow(7,-1)+I*((-4)*pow(21,-1)),tau1)","0.7001954574725119636036974699320952682281468467228624921486814444872783512555497294491311298856913561786564863457339-1.9746656168562704763634957342760966393686129404463235530365865071608062699091030959897891464424872604473069943473233*I");
	test(484,"Gt({{1,0}},(2)*pow(7,-1)+I*((-4)*pow(21,-1)),tau2)","0.7054515145576066101408341116613283541512628775671684885798672356764081456591468871939490979671066205828622712980725-1.9801525887571336339617637387075728407108993359793459246823101509163162690592780444486574521280721441741698423201578*I");
	test(485,"Gt({{1,0}},(2)*pow(7,-1)+I*((-1)*pow(7,-1)),tau1)","0.5979205127262652233083704628528638186754142255262942581374874295816403365225122621611477127742341370586993556609274-1.8946351779882712754220711843403977848076906600233281325223507165764427215471364952980421004527375494315783997443387*I");
	test(486,"Gt({{1,0}},(2)*pow(7,-1)+I*((-1)*pow(7,-1)),tau2)","0.602859822880319658927654268410500628850834229164894409704098637638326800472540538652283115215557591594740947286489-1.8983721626805028074321400417802710863580129056263495336906594150788807430795658838564191410014431878944350803649407*I");
	test(487,"Gt({{1,0}},(2)*pow(7,-1)+I*((-2)*pow(21,-1)),tau1)","0.5172964717120158858285914881744838350273128876219190350281267315024540966971636559796943205289678542891101141549875-1.7984979540502438435849496280187871867676722699480616510230899954484228194968628810045259414113520374743486326168149*I");
	test(488,"Gt({{1,0}},(2)*pow(7,-1)+I*((-2)*pow(21,-1)),tau2)","0.5220294101520072058863826414464297423639310040519126346545333288945347339447690113932449695839905951568402505515401-1.8008201077070908139519527678965982174724534175423072201855172471232916955700865928171052363932469746491194206738839*I");
	test(489,"Gt({{1,0}},(2)*pow(7,-1)+I*((-1)*pow(21,-1)),tau1)","0.4651446225319825017848547434327323189624590452980488527220468167554445330554776072303983664400310425921389223393495-1.6886584427034431543733611606422258411661607982028153784965365612789257417793828826805169255016440641231882921061766*I");
	test(490,"Gt({{1,0}},(2)*pow(7,-1)+I*((-1)*pow(21,-1)),tau2)","0.4697618334137523171205256785761978540231313734156882685758995194820386319533735296152380433888969039482984929558646-1.6897739296725107088194644071688397863192612231639027720079377998598266600597441799505255262508404825399741895947707*I");
	test(491,"Gt({{1,0}},(2)*pow(7,-1),tau1)","0.447031123282580211864514090407909478664958406509312502243335934543688000344075747379287640864119480216558544990231685853953061713198917-1.5707878000968280575043425565861854204792341013877663219326063956836110603693447483070035176490456599917215256046483*I");
	test(492,"Gt({{1,0}},(2)*pow(7,-1),tau2)","0.4516122726407059762099996107866697390398649351293904483262041331475527961406919346187820133238052437678128553294278-1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432558742*I");
	test(493,"Gt({{1,0}},(2)*pow(7,-1)+I*((1)*pow(21,-1)),tau1)","0.4651404925503764557343574351419771710231464825619738236450453497561710476516356262084260115462189726235086537475181-1.4529170175140665709660144210417683970356490017045475767529333701958983370500166483881992471120394055126511236860538*I");
	test(494,"Gt({{1,0}},(2)*pow(7,-1)+I*((1)*pow(21,-1)),tau2)","0.4697618334137523171205256785761978540231313734156882685758995194820386319533735296152380433888969039482984935976808-1.4518187239172825296431789761106630978779081762112030489670067924479897462264648186775092990912765854421738969169907*I");
	test(495,"Gt({{1,0}},(2)*pow(7,-1)+I*((2)*pow(21,-1)),tau1)","0.5172878392655983226092321111967770627793115071205471062337132990062854453714689244031258080213210362170845010016512-1.3430770736143724866411074834699078681309499260128903182929707315279866124182249527644765148956049852770777190322712*I");
	test(496,"Gt({{1,0}},(2)*pow(7,-1)+I*((2)*pow(21,-1)),tau2)","0.5220294101520072058863826414464297423639310040519126346545333288945347339447690113932449695839905951568402505627971-1.3407725458827024245106906153829046667247159818327986007894273451845247107161224058109295889488700933330286658373638*I");
	test(497,"Gt({{1,0}},(2)*pow(7,-1)+I*((1)*pow(7,-1)),tau1)","0.5979065992543500431405121982323816600937740116772475824236326446936174695334541871058645095423609205387076545705457-1.2469390855347419367004612297462773228205264072297250790206370011122520856053028273541202066393359974182810990655886*I");
	test(498,"Gt({{1,0}},(2)*pow(7,-1)+I*((1)*pow(7,-1)),tau2)","0.6028598228803196589276542684105006288508342291648944097040986376383268004725405386522831152155575915947409472892829-1.2432204909092904310305033414992317978391564937487562872842851772289356632066431147716156843406738800877130061468946*I");
	test(499,"Gt({{1,0}},(2)*pow(7,-1)+I*((4)*pow(21,-1)),tau1)","0.7001750081185130834687814351566830489454540428838731552509196999974518769463822431522031547857365459179338870345832-1.1669074820184681764464737371012851944170242048666611421725721102235539722978754790191180485665226406663585297638513*I");
	test(500,"Gt({{1,0}},(2)*pow(7,-1)+I*((4)*pow(21,-1)),tau2)","0.7054515145576066101408341116613283541512628775671684885798672356764081456591468871939490979671066205828622712971851-1.1614400648326596045008796445719300434862700633957598962926344413915001372269309541793773732140449238079782441926614*I");
	test(501,"Gt({{1,0}},(2)*pow(7,-1)+I*((5)*pow(21,-1)),tau1)","0.8178160786702915765680005185446376318606814626243761581713847442785886540959618193822448805786179524635981376272228-1.1026993505083182574877598800929751520807384702615832166151109334244977260960122731933971391950504321060518331182281*I");
	test(502,"Gt({{1,0}},(2)*pow(7,-1)+I*((5)*pow(21,-1)),tau2)","0.8235587590455172353624229294509635325378785153260023993574711302510588505399978827434026255241279922790570086493575-1.0949917123098235879521137331429042262967495219413415476585923101536910645225950475422622739680336844752552258415356*I");
	test(503,"Gt({{1,0}},(2)*pow(7,-1)+I*((2)*pow(7,-1)),tau1)","0.9459174905843525911200454653559635125549976956538301689903336662326277515338366236076081149432736649174837177215385-1.0524940412806485354063395125908941434298897028411802294088397354937588144766508978988292203409810046705225473761332*I");
	test(504,"Gt({{1,0}},(2)*pow(7,-1)+I*((2)*pow(7,-1)),tau2)","0.9523151351022059475825597522732810878100953758198629981561551547656185236784107830171890156811913852786646455546632-1.0418547065476355127554684689447330993511020967315401957314867270408574816765489026062360390165184153972625502100628*I");
	test(505,"Gt({{1,0}},(2)*pow(7,-1)+I*((1)*pow(3,-1)),tau1)","1.0809852788857733022400264949575427622756786095305745813208579876053149816252025075404318068620903063286606233650621-1.0139120744846501785191478758579914094290998007367668336506628705426997302067906278549007898625246420656855853065655*I");
	test(506,"Gt({{1,0}},(2)*pow(7,-1)+I*((1)*pow(3,-1)),tau2)","1.0882927485122235894541220142147087234799443908568496309024669296878669689811148461330698207640758846837091742546589-0.999388770272690063687531026499727033209371172502184703500138307556173648888682236623336561075404633943483200320669*I");
	test(507,"Gt({{1,0}},(2)*pow(7,-1)+I*((8)*pow(21,-1)),tau1)","1.2206639659765416462315543848043211368009151152260301488967344138895534182037261256776839622121548735502295154131219-0.9846014710246884620396072414891643921443867620887133188052456237389959798179684587064184231585054729138253170540787*I");
	test(508,"Gt({{1,0}},(2)*pow(7,-1)+I*((8)*pow(21,-1)),tau2)","1.2292308355537207951046177286550798117720991416749447097867014355332495594909645690040037650093104435163084142560694-0.9648982970015976899354832538997477460135419007931901465428137001411016840971117759584574950241780973533196798194886*I");
	test(509,"Gt({{1,0}},(1)*pow(3,-1)+I*((-8)*pow(21,-1)),tau1)","1.2443199662983962589106378903978166133057717962256569693952270444459479970028885832333883667789188937553868961894005-2.0188955349245744246244351009551219462780274437844850618716593874532152603562468786235763641849867846675476260274246*I");
	test(510,"Gt({{1,0}},(1)*pow(3,-1)+I*((-8)*pow(21,-1)),tau2)","1.2584656935605574665768971775574231311419661283889135121416350578117040658685981879131813121875300239938376517851461-2.0363393046940373014932468678300929467584059861703964924477333175601950720254600126829140682063519831501241437630961*I");
	test(511,"Gt({{1,0}},(1)*pow(3,-1)+I*((-1)*pow(3,-1)),tau1)","1.1119937633063183660188299478635504165273003527368428797422862236290186190405981581320608895259988692284567432331672-1.9942488471143162078815161396613375144435814270652108177059165967472587980836643809157822304230164211120711020917621*I");
	test(512,"Gt({{1,0}},(1)*pow(3,-1)+I*((-1)*pow(3,-1)),tau2)","1.1234830183906169845029487267618755712096132195929052438454683489426649984396172193901824609703044271007694311556457-2.0071286397934790134622443837619046204593026718229842745117701561966604817939668602345778050796859045509290954262821*I");
	test(513,"Gt({{1,0}},(1)*pow(3,-1)+I*((-2)*pow(7,-1)),tau1)","0.9861370957505436637646693009913888594271553213434411696510398779062989899227296702027230865016881844785327374949623-1.9623374104209259006403831329183521895618697982667758223029835461821029701642389612934817094725400824518010660176843*I");
	test(514,"Gt({{1,0}},(1)*pow(3,-1)+I*((-2)*pow(7,-1)),tau2)","0.9956834412635448772091914836418435590931797600362682801707792206175685261654216103458488395181631366497527335482996-1.9717864963838165422593015975148112592703997961182888861828592527943072984534612123891568641574884883305310102168858*I");
	test(515,"Gt({{1,0}},(1)*pow(3,-1)+I*((-5)*pow(21,-1)),tau1)","0.8691752513672493448520738997735439264270328032371433952297389685382978471463829679060573896528408536543710476936267-1.9216497048268536459342117812675426419034487979303787209649388114481687230113769819059685120305315535268209660736793*I");
	test(516,"Gt({{1,0}},(1)*pow(3,-1)+I*((-5)*pow(21,-1)),tau2)","0.8773103302912950936018409784933007923287645338662067971607898053944468914838513410779029116086930739432022734920714-1.9285044296595658253726491269683417847390204045498681573522271713232816873141190430334290814580033387215057119361004*I");
	test(517,"Gt({{1,0}},(1)*pow(3,-1)+I*((-4)*pow(21,-1)),tau1)","0.7643030967648301495341298865343336216708197052087361670074484025808514874237735388288549161631228998922912534966009-1.8708478476500863515452450635442729993571197710828296367831231380039439738623080013553439013897699510061179651617871*I");
	test(518,"Gt({{1,0}},(1)*pow(3,-1)+I*((-4)*pow(21,-1)),tau2)","0.7714275233938483304523218056921665341637172251900390781931597265321806163634908785727395401238263632729707174459528-1.8757174470365792732430442144269121328545882430076240822878996424805790568806393751929488045961795182931860825078815*I");
	test(519,"Gt({{1,0}},(1)*pow(3,-1)+I*((-1)*pow(7,-1)),tau1)","0.6755015904134314548248628986090269484624861012068795883431657388366808208509919914288628956855137655147330668755986-1.8091764988286460677493948510080633329914111507387514265685014990417078573809493590108770715057351239684421641092968*I");
	test(520,"Gt({{1,0}},(1)*pow(3,-1)+I*((-1)*pow(7,-1)),tau2)","0.6819227455774074804666797706313270799531266855239808835090607741412614943016946845500741829951481161956325763181105-1.8124949623477236800987363460135864045329052116884425326852726847110595246276327500076211471801617699577291177646833*I");
	test(521,"Gt({{1,0}},(1)*pow(3,-1)+I*((-2)*pow(21,-1)),tau1)","0.6072961440559140340089040980724470640654922957686104580684951039950268353536038354852989751034933074517974251524556-1.737003585594515393720123432258059161038545771086817206933756416236219022249498185239600977370839093413907694491449*I");
	test(522,"Gt({{1,0}},(1)*pow(3,-1)+I*((-2)*pow(21,-1)),tau2)","0.6132568359197101956296651508826241724835285346279416190931297929817432660922601991434700064509263534001308149017903-1.7390671127258782464820731094058693218674443345591527816595036881790871181447624184599905385162039693862288462164181*I");
	test(523,"Gt({{1,0}},(1)*pow(3,-1)+I*((-1)*pow(21,-1)),tau1)","0.5641181295549411879731572419997513308949941546424679815549716146493244187383718948909931749418924893299945455028366-1.6563093629065379986644274353251439170948734936336939421823238116947641928259615181620657007222905783945607634415065*I");
	test(524,"Gt({{1,0}},(1)*pow(3,-1)+I*((-1)*pow(21,-1)),tau2)","0.5698190257957779337563173166344555739768015577136480332968434466673955965138043836874707699640953370803668554333613-1.6573021685587459438523368378351005847973190832415993434619416120876711205054188970585621566250254209575860826368427*I");
	test(525,"Gt({{1,0}},(1)*pow(3,-1),tau1)","0.549306144279327840506057223846309718339154392996227358502548773550557343600806854112691572234937156993417089024574557905204624610204798-1.5707858647678281194793818544664381157452365173143442085257804379968821884478204001063040653772230898730476898651926*I");
	test(526,"Gt({{1,0}},(1)*pow(3,-1),tau2)","0.5549241851709482843473695151794916211832085133066407393312846847550531446161866679746976448017189000689537580032693-1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432562196*I");
	test(527,"Gt({{1,0}},(1)*pow(3,-1)+I*((1)*pow(21,-1)),tau1)","0.5641144609052458942802288352493018671074388041359528416857358991345131613255677611157366091634465789353801563364502-1.4852620521055723055011309102661988918238143193284096253127972906134011577029685379743722596259854065353353641071652*I");
	test(528,"Gt({{1,0}},(1)*pow(3,-1)+I*((1)*pow(21,-1)),tau2)","0.569819025795777933756317316634455573976801557713648033296843446667395596513804383687470769964095337080366855448276-1.4842904850310472946103065454444022993998503161335064775130029802201452857807901015694726687170916470245620038753495*I");
	test(529,"Gt({{1,0}},(1)*pow(3,-1)+I*((2)*pow(21,-1)),tau1)","0.6072884758808614036744615414808234129171178672301920588706915994944371742969987165261058781271370856535843002409813-1.4045668574800659603515510788904440649115627551719232772879613426374797283691697195020764868568446014245778287966854*I");
	test(530,"Gt({{1,0}},(1)*pow(3,-1)+I*((2)*pow(21,-1)),tau2)","0.6132568359197101956296651508826241724835285346279416190931297929817432660922601991434700064509263534001308149064673-1.4025255408639149919805702738736335623297250648159530393154409041287292881414465801680442868259130985959192402956482*I");
	test(531,"Gt({{1,0}},(1)*pow(3,-1)+I*((1)*pow(7,-1)),tau1)","0.6754892311200082723508136800821610017216374968885058377937415252288837345907451951469643478694988261568587681686648-1.3323922272353388260856593289595964538702510758474799051940026095466763293375305254811101763131393897702023332865733*I");
	test(532,"Gt({{1,0}},(1)*pow(3,-1)+I*((1)*pow(7,-1)),tau2)","0.6819227455774074804666797706313270799531266855239808835090607741412614943016946845500741829951481161956325763212603-1.3290976912420695583639070372659164796642641876866632882896719075967568816585762486204136781619552980244189687474421*I");
	test(533,"Gt({{1,0}},(1)*pow(3,-1)+I*((4)*pow(21,-1)),tau1)","0.7642849316678706372742464812977728173169783218646553818744703029073276324279591801069677960375523670107932999424082-1.2707182614729803457884002266111152405046084204154336815136390497907965412665314679306804688088884333836577187617558*I");
	test(534,"Gt({{1,0}},(1)*pow(3,-1)+I*((4)*pow(21,-1)),tau2)","0.7714275233938483304523218056921665341637172251900390781931597265321806163634908785727395401238263632729707160190834-1.265875206553213965219599168852590751342581156367481738687044949827237349405569623435086020745937549688962004004914*I");
	test(535,"Gt({{1,0}},(1)*pow(3,-1)+I*((5)*pow(21,-1)),tau1)","0.869149642155935693364043780972831046114549699743863589751788842048191972918017753507549831619716397223384036988527-1.2199126514029670762588682905382278711218484264076584888836851202468957209664801764595312281824131055868971820580419*I");
	test(536,"Gt({{1,0}},(1)*pow(3,-1)+I*((5)*pow(21,-1)),tau2)","0.8773103302912950936018409784933007923287645338662067971607898053944468914838513410779029116086930739432022734684421-1.2130882239302274130899942563111610994581489948252376636227174209845347189720899555946057438841137292606423745771053*I");
	test(537,"Gt({{1,0}},(1)*pow(3,-1)+I*((2)*pow(7,-1)),tau1)","0.9861017327290881633789747908041365107403723209230531057875151223721377817491470470068356771887764432190494509566468-1.1792197184897286872523995046472764554529074320683988622531385452999344398749674535551648422220940855822319811263608*I");
	test(538,"Gt({{1,0}},(1)*pow(3,-1)+I*((2)*pow(7,-1)),tau2)","0.9956834412635448772091914836418435590931797600362682801707792206175685261654216103458488395181631366497527335443831-1.169806157205976696203341785764691624926769603256816934792085339513509107832747786238877961184628579651617076296282*I");
	test(539,"Gt({{1,0}},(1)*pow(3,-1)+I*((1)*pow(3,-1)),tau1)","1.11194545708240837976034674075256298121721534238401235904234015152992573371188239585575879470153523472213459235483-1.147301108599125609621932837911180506669735541551181407752310545713116563382436342098619764721975571306589032041527*I");
	test(540,"Gt({{1,0}},(1)*pow(3,-1)+I*((1)*pow(3,-1)),tau2)","1.1234830183906169845029487267618755712096132195929052438454683489426649984396172193901824609703044271007694311280353-1.134464013796314225000398999517598263737866727552121546463174436111155924492242138393457020262431163431218991087707*I");
	test(541,"Gt({{1,0}},(1)*pow(3,-1)+I*((8)*pow(21,-1)),tau1)","1.244254360131774597390826783183708551344404723130377492575937259114327628534936626812507326107041901876062351088592-1.1226446547626589331841657430473052359606381680269044061659991595631248066253039093349418244170852253407140350502129*I");
	test(542,"Gt({{1,0}},(1)*pow(3,-1)+I*((8)*pow(21,-1)),tau2)","1.2584656935605574665768971775574231311419661283889135121416350578117040658685981879131813121875300239938376525960101-1.1052533488957559369693965154494099374387634132047093285272112747476213342607489859451207571357650848320239427508482*I");
	test(543,"Gt({{1,0}},(8)*pow(21,-1)+I*((-8)*pow(21,-1)),tau1)","1.2632977865940285547388357581029209603683383857249697555945837592371847419536811365964449128631056962279654835990621-1.8866209951519095060099342500330024980116085483617504708038026585586956236156440241883767897364676435639273343922411*I");
	test(544,"Gt({{1,0}},(8)*pow(21,-1)+I*((-8)*pow(21,-1)),tau2)","1.2821469220144560318638376375830875902984252471351152770648807515733926646025872594991034711056145369302825647369498-1.900271746814234345487324540793214046166881943953082715975207414961700740083551588172399749795940426050210125180991*I");
	test(545,"Gt({{1,0}},(8)*pow(21,-1)+I*((-1)*pow(3,-1)),tau1)","1.1365855685392661727356553181165612909284807276959735191944723669006721646599497037193681203492725164908819765172815-1.8680934050108924476451749674985856154341523244509321377757681056677037532703163924030581762408742961620586033405286*I");
	test(546,"Gt({{1,0}},(8)*pow(21,-1)+I*((-1)*pow(3,-1)),tau2)","1.151608800581312798601977334588732556849545400519011953605072316576744324158182915851003256903784545493118681842611-1.8781848747789590353213167129229384248206066781303306019416817696804967992994194538615420236881907126843963415260534*I");
	test(547,"Gt({{1,0}},(8)*pow(21,-1)+I*((-2)*pow(7,-1)),tau1)","1.0175530900608972887041399041272243095762492480007765384603174725870352240551079914644688351719468685256974818188356-1.8444000517891892504818009551597378741043830932396312910187030668393427300077056962638924677587982661524603943115911*I");
	test(548,"Gt({{1,0}},(8)*pow(21,-1)+I*((-2)*pow(7,-1)),tau2)","1.0297646353257974148847673551135240730929137829844618005184509852279913749648560611546047284796522169816330027526909-1.8518105087293284341273179505398795126406339421133730823962844523912675402935585340745353538792655689397923135219284*I");
	test(549,"Gt({{1,0}},(8)*pow(21,-1)+I*((-5)*pow(21,-1)),tau1)","0.9085540045475759218340011254220166359678474946528156251336817365958462960735726675127675072990190226971502696892039-1.8146341206862169465480155701797349553922172893226946767093373502266615192592181170297458093439494000238835038029551*I");
	test(550,"Gt({{1,0}},(8)*pow(21,-1)+I*((-5)*pow(21,-1)),tau2)","0.9187158889284009848749912680977300508840725149341185535253004945388630145698822153759521992851955688182615048742195-1.8200140167681216404384141930441018288548982336998210423013012299709834966819746123729393772171493585389072304256899*I");
	test(551,"Gt({{1,0}},(8)*pow(21,-1)+I*((-4)*pow(21,-1)),tau1)","0.8124426796560095258424746388279793403629996078733569823130231957758940173582891577126569633583201286786613379270462-1.778087214288302912694259746724214216974982743679150916159449960232213834372785118272299317053452660981974042660067*I");
	test(552,"Gt({{1,0}},(8)*pow(21,-1)+I*((-4)*pow(21,-1)),tau2)","0.8211327928162059104380774846327328012348751777370526002364286815456978653156560110001221078585269701673861018468662-1.7819116178308979383086152225745066360836203610906686453572826494111740577929853763894380857993031802942597802039634*I");
	test(553,"Gt({{1,0}},(8)*pow(21,-1)+I*((-1)*pow(7,-1)),tau1)","0.7324828946247932182920272884057177753627422550474204661642202589168211007709914229591708734162305004943773846608186-1.7344989581952954276002051941572402085759324898797896142946074202436668559586018772280257735909586225905026784375453*I");
	test(554,"Gt({{1,0}},(8)*pow(21,-1)+I*((-1)*pow(7,-1)),tau2)","0.7401467652051187069398426480570277888108057238153701990843620226649903225002303303994904882839953953125025132264145-1.7371069469511053030081153619342846281871680518786636808404809988682427608657642478804175587522726323257933857107979*I");
	test(555,"Gt({{1,0}},(8)*pow(21,-1)+I*((-2)*pow(21,-1)),tau1)","0.6720874869288387715164683316392460679326339813849485791278471704151556484417985853090445518679557194808137682355979-1.6843368225207994915000086116765156218179462896526201745723675958688050716121855175745150744225578708925706515943988*I");
	test(556,"Gt({{1,0}},(8)*pow(21,-1)+I*((-2)*pow(21,-1)),tau2)","0.6790782851040650662227102166096993284784884701339637536427280992000361124496755112541036996283309429852991174812195-1.6859601640739432617471676632014531045929040066997070017831973831604045383039665123483487671891087177876365003874578*I");
	test(557,"Gt({{1,0}},(8)*pow(21,-1)+I*((-1)*pow(21,-1)),tau1)","0.6343609058725608362300400742719480235921039433656234698957367204852658461492378139710241093791003650001733761680839-1.6289975789055649042555155939493890523724998977394419143272045895874311203207055693345474327564346444203543710604501*I");
	test(558,"Gt({{1,0}},(8)*pow(21,-1)+I*((-1)*pow(21,-1)),tau2)","0.640971199825729177578450075619232124162226611282650098643638330420611825078470218171327447896187753120565323120317-1.6297806037053405676380086251472933811884252157408143246729683520169752061469637412020081042998298637811097339155017*I");
	test(559,"Gt({{1,0}},(8)*pow(21,-1),tau1)","0.62151556127522070253470523243449490605521719507537645857987718974191050957247019418292841657225393325374690634615311145733518098055452-1.5707842393045004980063838229653588084732350023324796028002644800284900707191139570704741413061178108564286848591574*I");
	test(560,"Gt({{1,0}},(8)*pow(21,-1),tau2)","0.6280036500683374494988674414813647297467745787271588776375931157919687275989755149041475518417209252105144848833928-1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432559506*I");
	test(561,"Gt({{1,0}},(8)*pow(21,-1)+I*((1)*pow(21,-1)),tau1)","0.6343580245304013257517167961004575945117848699379849596491622783753103842019373625829635061570649412259065131928815-1.5125704385792879481669187271078912637921151470276140116278637591956751270291439851177023814608591611881682890866005*I");
	test(562,"Gt({{1,0}},(8)*pow(21,-1)+I*((1)*pow(21,-1)),tau2)","0.6409711998257291775784500756192321241622266112826500986436383304206118250784702181713274478961877531205653230893211-1.5118120498844526708246347581322095030087441836342914963019762402908412001392452574260267210422872042010383525950003*I");
	test(563,"Gt({{1,0}},(8)*pow(21,-1)+I*((2)*pow(21,-1)),tau1)","0.6720814643761557322236945777312199428213726807461334498585976520642088890479532539549720194652219948028480021409582-1.4572297700028036431056134069705370731181218538598365211506629136864608467386765814252322348071614886510816416229579*I");
	test(564,"Gt({{1,0}},(8)*pow(21,-1)+I*((2)*pow(21,-1)),tau2)","0.6790782851040650662227102166096993284784884701339637536427280992000361124496755112541036996283309429852991174775379-1.4556324895158499767154757200780497796042653926753988191917472091474118679822424862796860581530083501945115861239582*I");
	test(565,"Gt({{1,0}},(8)*pow(21,-1)+I*((1)*pow(7,-1)),tau1)","0.7324731876873205491143999601787594375392322676586985144991359335753509721150944051665309778850326623105271453440627-1.4070651170126445395313264167492759349906477939794966455249444983964312836800391263844573243565662183917129466037555*I");
	test(566,"Gt({{1,0}},(8)*pow(21,-1)+I*((1)*pow(7,-1)),tau2)","0.740146765205118706939842648057027788810805723815370199084362022664990322500230330399490488283995395312502513265111-1.4044857066386879354545280213452182560100013474964421401344635934395736454204447507476172665898444356563547008001312*I");
	test(567,"Gt({{1,0}},(8)*pow(21,-1)+I*((4)*pow(21,-1)),tau1)","0.8124284128646704491289025197967846949347735590165132185553409116099704439612947095912969227981521706691750913522745-1.3634730242127604610992913963771122764472586232361438253992749173953900144394591843453430904766905513691007675859026*I");
	test(568,"Gt({{1,0}},(8)*pow(21,-1)+I*((4)*pow(21,-1)),tau2)","0.8211327928162059104380774846327328012348751777370526002364286815456978653156560110001221078585269701673861018466605-1.3596810357588953001540281607049962481135490382844371756176619428966423484932236222385967395428138876878883063075679*I");
	test(569,"Gt({{1,0}},(8)*pow(21,-1)+I*((5)*pow(21,-1)),tau1)","0.9085338911798923180775800835442606320497854538578210718953823319762297915309231741697904102529245617304069661865779-1.326920615684010203081667668023076660919569523830276831465805889429801499488064241541480194775265164072033246518742*I");
	test(570,"Gt({{1,0}},(8)*pow(21,-1)+I*((5)*pow(21,-1)),tau2)","0.9187158889284009848749912680977300508840725149341185535253004945388630145698822153759521992851955688182615048769651-1.3215786368216715980242291902354010553422711656752847786736433623368329096042343862550954481249677094432408560869776*I");
	test(571,"Gt({{1,0}},(8)*pow(21,-1)+I*((2)*pow(7,-1)),tau1)","1.0175253160914387108420463660143148106128593013633671850074459044730606692758907837812248772222146003193669112128072-1.2971470207888422060269449938301104391193957924240299975544127326607250446451773780570372119245072405274233918659748*I");
	test(572,"Gt({{1,0}},(8)*pow(21,-1)+I*((2)*pow(7,-1)),tau2)","1.0297646353257974148847673551135240730929137829844618005184509852279913749648560611546047284796522169816330027451381-1.2897821448604648043353254327396233715565354572617327385786601399165488659926504645534994714628514990423557729908949*I");
	test(573,"Gt({{1,0}},(8)*pow(21,-1)+I*((1)*pow(3,-1)),tau1)","1.1365476290326253754066131004160842889173032157620671809257400655477394325042123707326461993045416419796904016041197-1.2734431509158589155023149653368100743850723859100079430169818192100533448561658740859014182614077404051211902741614*I");
	test(574,"Gt({{1,0}},(8)*pow(21,-1)+I*((1)*pow(3,-1)),tau2)","1.1516088005813127986019773345887325568495454005190119536050723165767443241581829158510032569037845454931186826822107-1.2634077788108342031413266703565644593765627212447752190332628226273196069867895447664928016539263552977517449874646*I");
	test(575,"Gt({{1,0}},(8)*pow(21,-1)+I*((8)*pow(21,-1)),tau1)","1.2632462597846006137408457618334161827709162282116698345621016506129293852121086629489810930419040796434693328834077-1.2549012427672755020253652147343902468516327914595623648666085383273513808553841548768551265714268060677449657490462*I");
	test(576,"Gt({{1,0}},(8)*pow(21,-1)+I*((8)*pow(21,-1)),tau2)","1.2821469220144560318638376375830875902984252471351152770648807515733926646025872594991034711056145369302825654722196-1.2413209067755588929753188424862888380302874554220231049997371773461156662026574104556350755461766419319379613323919*I");

	return failed;
}

static unsigned exam_full(const bool abbreviated)
{
	unsigned failed = 0;

	const symbol tau{"tau"}, z1{"z1"}, z2{"z2"}, z3{"z3"}, zz1{"zz1"}, zz2{"zz2"}, zz3{"zz3"};

	const ex points = lst{
		z1==numeric(1,6) + numeric(1,3)*tau,
		z2==numeric(5,6) + numeric(5,4)*tau,
		z3==tau,
		zz1==numeric(9,7) + numeric(1,3)*tau,
		zz2==numeric(1,4) - numeric(11,5)*tau,
		zz3==numeric(-6,7) + numeric(23,9)*tau,
		tau==numeric(5,4) + I/2
	};

	const size_t digits = 4; // the reference numbers are not particularly precise
	auto test = [&](const size_t id, const Gt&& input, const numeric expected_result, const bool check_without_tautofundamental){
		const auto t0 = chrono::system_clock::now();
		const numeric output{input.evaluate(&points)};
		const auto t1 = chrono::system_clock::now();
		Gt::enable_tauToFundamental = false;
		const numeric output0{check_without_tautofundamental ? input.evaluate(&points) : output};
		Gt::enable_tauToFundamental = true;
		const auto t2 = chrono::system_clock::now();
		const numeric diff = expected_result-output;
		const numeric diff0 = output-output0;

		const numeric digits_real = diff.real().is_zero() ? -(long)Digits : log(abs(diff.real()))/log(numeric(10));
		const numeric digits_imag = diff.imag().is_zero() ? -(long)Digits : log(abs(diff.imag()))/log(numeric(10));
		const numeric digits_real0 = diff0.real().is_zero() ? -(long)Digits : log(abs(diff0.real()))/log(numeric(10));
		const numeric digits_imag0 = diff0.imag().is_zero() ? -(long)Digits : log(abs(diff0.imag()))/log(numeric(10));
		if (digits_real <= -(long)digits && digits_imag <= -(long)digits && digits_real0 <= -(long)digits && digits_imag0 <= -(long)digits) {
			std::printf("  - Test #%lu succeeded (%.2fs, %.2fs)\n", id,
				std::chrono::duration_cast<std::chrono::milliseconds>(t1-t0).count()/1000.,
				std::chrono::duration_cast<std::chrono::milliseconds>(t2-t1).count()/1000.);
		}
		else {
			std::printf("Test #%lu failed (%.2fs, %.2fs)\n", id,
				std::chrono::duration_cast<std::chrono::milliseconds>(t1-t0).count()/1000.,
				std::chrono::duration_cast<std::chrono::milliseconds>(t2-t1).count()/1000.);
			std::cout << "  Input:           " << input << endl;
			std::cout << "  Expected result: " << expected_result << endl;
			std::cout << "  Actual result:   " << output << endl;
			if (check_without_tautofundamental)
				std::cout << "  Without tauToF.: " << output0 << endl;
			std::cout << "  Digits real:     " << max(digits_real,digits_real0).to_double() << endl;
			std::cout << "  Digits imag:     " << max(digits_imag,digits_imag0).to_double() << endl;
			failed++;
		}
	};

	Digits = 17;
	test(1,Gt(lst{lst{1,z1}},zz1,tau),"-1.1608128050606474+6.232476788321447*I", true);
	test(2,Gt(lst{lst{1,z1}},zz2,tau),"9.887401508646514-23.018376509455713*I", true);
	test(3,Gt(lst{lst{1,z1}},zz3,tau),"7.229473152876459-12.287306232956503*I", true);
	test(4,Gt(lst{lst{2,z1}},zz1,tau),"-5.06381847332505-0.5824168875553761*I", true);
	test(5,Gt(lst{lst{2,z1}},zz2,tau),"142.46799295960096+58.57602019782604*I", true);
	test(6,Gt(lst{lst{2,z1}},zz3,tau),"-33.10276570486072-37.44143825302238*I", true);
	test(7,Gt(lst{lst{3,z1}},zz1,tau),"0.48866975540410673+3.612202656613558*I", true);
	test(8,Gt(lst{lst{3,z1}},zz2,tau),"-258.09481901276445+665.8030214261587*I", true);
	test(9,Gt(lst{lst{3,z1}},zz3,tau),"-124.0523476703351+41.647950860988885*I", true);
	test(10,Gt(lst{lst{1,z2}},zz1,tau),"-0.47709712033560303+16.46760932909897*I", true);
	test(11,Gt(lst{lst{1,z2}},zz2,tau),"16.341300797846063-39.35077658633666*I", true);
	test(12,Gt(lst{lst{1,z2}},zz3,tau),"0.791708735393355+2.032298611397106*I", true);
	test(13,Gt(lst{lst{2,z2}},zz1,tau),"-75.68865501360236+0.10560108537099566*I", true);
	test(14,Gt(lst{lst{2,z2}},zz2,tau),"336.94308281060285+127.28021485465216*I", true);
	test(15,Gt(lst{lst{2,z2}},zz3,tau),"-42.04585729146242-15.255716799558037*I", true);
	test(16,Gt(lst{lst{3,z2}},zz1,tau),"-4.033502059274273-224.1689128336466*I", true);
	test(17,Gt(lst{lst{3,z2}},zz2,tau),"-690.6133395005055+2039.3345442330633*I", true);
	test(18,Gt(lst{lst{3,z2}},zz3,tau),"-24.24098308342476-80.63162110218724*I", true);
	test(19,Gt(lst{lst{1,z3}},zz1,tau),"-0.2913425300498754+5.770810148036546*I", true);
	test(20,Gt(lst{lst{1,z3}},zz2,tau),"14.005440982107267-34.57498838292574*I", true);
	test(21,Gt(lst{lst{1,z3}},zz3,tau),"2.726399887031464-4.602214907631874*I", true);
	test(22,Gt(lst{lst{2,z3}},zz1,tau),"-8.010054246902234+0.13874660127265015*I", true);
	test(23,Gt(lst{lst{2,z3}},zz2,tau),"286.347249027193+92.05664505542944*I", true);
	test(24,Gt(lst{lst{2,z3}},zz3,tau),"-21.682332376387976-17.26168181669165*I", true);
	test(25,Gt(lst{lst{3,z3}},zz1,tau),"-1.1503728825299593-9.54014421396245*I", true);
	test(26,Gt(lst{lst{3,z3}},zz2,tau),"-412.60700745408025+1698.2006589150985*I", true);
	test(27,Gt(lst{lst{3,z3}},zz3,tau),"-48.835027710107056+41.95596168119579*I", true);
	test(28,Gt(lst{lst{1,z1},lst{0,0}},zz1,tau),"-1.4651004390265583+6.52832890977206*I", true);
	test(29,Gt(lst{lst{1,z1},lst{0,0}},zz2,tau),"-33.7283839614701+31.12120935265859*I", true);
	test(30,Gt(lst{lst{1,z1},lst{0,0}},zz3,tau),"23.07250955403196-10.993292385496407*I", true);
	test(31,Gt(lst{lst{2,z1},lst{0,0}},zz1,tau),"-4.817082716608615-0.13228176771093514*I", true);
	test(32,Gt(lst{lst{2,z1},lst{0,0}},zz2,tau),"-219.44442962016757-231.58947986510515*I", true);
	test(33,Gt(lst{lst{2,z1},lst{0,0}},zz3,tau),"-15.133447205593418-98.56018414990137*I", true);
	test(34,Gt(lst{lst{1,z2},lst{0,0}},zz1,tau),"-1.2392814181099254+13.27511306046899*I", true);
	test(35,Gt(lst{lst{1,z2},lst{0,0}},zz2,tau),"-48.54663861540919+48.006399182445115*I", true);
	test(36,Gt(lst{lst{1,z2},lst{0,0}},zz3,tau),"9.122619894406174-4.65814222056869*I", true);
	test(37,Gt(lst{lst{2,z2},lst{0,0}},zz1,tau),"-59.58476049597118-3.5747221997188254*I", true);
	test(38,Gt(lst{lst{2,z2},lst{0,0}},zz2,tau),"-471.5914498557986-443.85883834876523*I", true);
	test(39,Gt(lst{lst{2,z2},lst{0,0}},zz3,tau),"-15.116363030433874-37.402239797777916*I", true);
	test(40,Gt(lst{lst{1,z3},lst{0,0}},zz1,tau),"-0.5871538501367708+5.382412784529942*I", true);
	test(41,Gt(lst{lst{1,z3},lst{0,0}},zz2,tau),"-43.77181158993266+46.46287810853461*I", true);
	test(42,Gt(lst{lst{1,z3},lst{0,0}},zz3,tau),"13.371818244112252-7.66403903553481*I", true);
	test(43,Gt(lst{lst{2,z3},lst{0,0}},zz1,tau),"-9.751470928075987-0.42909908976424305*I", true);
	test(44,Gt(lst{lst{2,z3},lst{0,0}},zz2,tau),"-445.0729077647269-360.89032538653896*I", true);
	test(45,Gt(lst{lst{2,z3},lst{0,0}},zz3,tau),"-24.169297304574485-53.47663288881819*I", true);
	test(46,Gt(lst{lst{0,0},lst{1,z1}},zz1,tau),"-1.5497913009755027+3.888252059979357*I", true);
	test(47,Gt(lst{lst{0,0},lst{1,z1}},zz2,tau),"-16.310333970547532+15.548590261469577*I", true);
	test(48,Gt(lst{lst{0,0},lst{1,z1}},zz3,tau),"9.525396374762629-8.488187836666956*I", true);
	test(49,Gt(lst{lst{0,0},lst{2,z1}},zz1,tau),"-3.7063959174355237-1.7031833935625131*I", true);
	test(50,Gt(lst{lst{0,0},lst{2,z1}},zz2,tau),"-72.29193056122648-71.56536288502133*I", true);
	test(51,Gt(lst{lst{0,0},lst{2,z1}},zz3,tau),"-14.395861852699328-31.24972731039773*I", true);
	test(52,Gt(lst{lst{0,0},lst{1,z2}},zz1,tau),"-2.3175211868350405+14.679515205917403*I", true);
	test(53,Gt(lst{lst{0,0},lst{1,z2}},zz2,tau),"-35.592467624176386+32.39511140576595*I", true);
	test(54,Gt(lst{lst{0,0},lst{1,z2}},zz3,tau),"-9.868983813942611+10.41986481941614*I", true);
	test(55,Gt(lst{lst{0,0},lst{2,z2}},zz1,tau),"-69.28376429141386-8.860280359595242*I", true);
	test(56,Gt(lst{lst{0,0},lst{2,z2}},zz2,tau),"-230.75802083059193-244.9790898795291*I", true);
	test(57,Gt(lst{lst{0,0},lst{2,z2}},zz3,tau),"-63.66407004581097-51.9802333878991*I", true);
	test(58,Gt(lst{lst{0,0},lst{1,z3}},zz1,tau),"-0.8706238149780468+4.393147402952584*I", true);
	test(59,Gt(lst{lst{0,0},lst{1,z3}},zz2,tau),"-29.27427808655399+24.56860776846187*I", true);
	test(60,Gt(lst{lst{0,0},lst{1,z3}},zz3,tau),"-1.1187915230051964+0.39100801580849875*I", true);
	test(61,Gt(lst{lst{0,0},lst{2,z3}},zz1,tau),"-3.9078172829341327-0.6697103801720043*I", true);
	test(62,Gt(lst{lst{0,0},lst{2,z3}},zz2,tau),"-169.53290524228402-184.23326118194788*I", true);
	test(63,Gt(lst{lst{0,0},lst{2,z3}},zz3,tau),"-4.452259142718589-14.574325901770635*I", true);
	if (abbreviated)
		return failed;
	// More complicated tests take very long without tauToFundamental
	test(64,Gt(lst{lst{1,z1},lst{2,z2}},zz1,tau),"45.91188088826212-309.5443887719172*I", false);
	test(65,Gt(lst{lst{1,z1},lst{2,z2}},zz2,tau),"3307.023498849635-3157.6961469085927*I", false);
	test(66,Gt(lst{lst{1,z1},lst{2,z2}},zz3,tau),"-336.48367023736574+356.11316916100174*I", false);
	test(67,Gt(lst{lst{2,z3},lst{2,z1}},zz1,tau),"34.22648637361343+13.627704409014449*I", false);
	test(68,Gt(lst{lst{2,z3},lst{2,z1}},zz2,tau),"17070.684163949052+11051.490167762053*I", false);
	test(69,Gt(lst{lst{2,z3},lst{2,z1}},zz3,tau),"282.19292784169306+836.824483557357*I", false);
	test(70,Gt(lst{lst{3,z1},lst{1,z3}},zz1,tau),"-6.05356605129921-3.321695438613892*I", false);
	test(71,Gt(lst{lst{3,z1},lst{1,z3}},zz2,tau),"12229.48185867546+13594.549047923676*I", false);
	test(72,Gt(lst{lst{3,z1},lst{1,z3}},zz3,tau),"-274.3116628437325+284.7094195200641*I", false);

	return failed;
}

static unsigned exam_banana(const bool abbreviated)
{
	unsigned failed = 0;

#define LIST_SYMBOLS(f) f(mTau1) f(mTau2) f(mTau3) f(mTau4) f(mTau5) f(mTau6)

	DEFINE_SYMBOLS
	MAKE_PARSER

#undef LIST_SYMBOLS

	const ex points = lst{
		mTau1==I,
		mTau2==numeric(1,3) + 3*I/2,
		mTau3==numeric(13,12)*I,
		mTau4==numeric(1,4) + I/10,
		mTau5==numeric(10,3) + I,
		mTau6==numeric(-7,4) + I/4
	};

	Digits = 100;
	const size_t digits = (long)Digits;
	auto test = [&](const size_t id, const char* input_raw, const numeric expected_result){
		const ex input = parse(input_raw);
		const auto t0 = chrono::system_clock::now();
		const numeric output{Gt::ex_evaluate(input, &points)};
		const auto t1 = chrono::system_clock::now();
		const numeric diff = expected_result-output;

		const numeric digits_real = diff.real().is_zero() ? -(long)Digits : log(abs(diff.real()))/log(numeric(10));
		const numeric digits_imag = diff.imag().is_zero() ? -(long)Digits : log(abs(diff.imag()))/log(numeric(10));
		if (digits_real <= -(long)digits && digits_imag <= -(long)digits)
			std::printf("  - Test #%lu succeeded (%.2fs)\n", id, std::chrono::duration_cast<std::chrono::milliseconds>(t1-t0).count()/1000.);
		else {
			std::printf("Test #%lu failed (%.2fs)\n", id, std::chrono::duration_cast<std::chrono::milliseconds>(t1-t0).count()/1000.);
			std::cout << "  Input:           " << input << endl;
			std::cout << "  Expected result: " << expected_result << endl;
			std::cout << "  Actual result:   " << output << endl;
			std::cout << "  Digits real:     " << digits_real.to_double() << endl;
			std::cout << "  Digits imag:     " << digits_imag.to_double() << endl;
			failed++;
		}
	};

	test(1,"2197/25*(-(Gt({{0,0},{0,0},{1,-1/6}},5/6+5/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/6}},5/6+5/6*mTau1,mTau1))-20818/15*(-(Gt({{0,0},{0,0},{1,-1/6}},1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/6}},1/2*mTau1,mTau1))-6591/10*(-(Gt({{0,0},{0,0},{1,-1/3}},1/6,mTau1))+Gt({{0,0},{0,0},{1,1/3}},1/6,mTau1))-386267/30*(-(Gt({{0,0},{0,0},{1,-1/3}},2/3+1/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/3}},2/3+1/3*mTau1,mTau1))+386267/30*(-(Gt({{0,0},{0,0},{1,-1/3}},2/3+2/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/3}},2/3+2/3*mTau1,mTau1))+18621/10*(-(Gt({{0,0},{0,0},{1,-1/3}},1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/3}},1/2*mTau1,mTau1))-386267/60*(-(Gt({{0,0},{0,0},{1,-1/3}},2/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/3}},2/3*mTau1,mTau1))+1330*(-(Gt({{0,0},{0,0},{1,-1/6-1/6*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,1/6+1/6*mTau1}},1/2,mTau1))-893/10*(-(Gt({{0,0},{0,0},{1,-1/6-1/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/6+1/6*mTau1}},1/2*mTau1,mTau1))+1271/2*(-(Gt({{0,0},{0,0},{1,-2/3-1/6*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,2/3+1/6*mTau1}},1/2,mTau1))-1271/2*(-(Gt({{0,0},{0,0},{1,-2/3-1/6*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,2/3+1/6*mTau1}},1/2+1/2*mTau1,mTau1))+8141/10*(-(Gt({{0,0},{0,0},{1,-2/3-1/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,2/3+1/6*mTau1}},1/2*mTau1,mTau1))+840*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau1}},2/3,mTau1))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau1}},2/3,mTau1))+2016/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau1}},5/6,mTau1))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau1}},5/6,mTau1))-462*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau1}},1/2+1/2*mTau1,mTau1))+576/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau1}},1/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau1}},1/6*mTau1,mTau1))+1557/10*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau1}},1/2*mTau1,mTau1))-560*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau1}},1/2,mTau1))+168/5*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau1}},1/2+1/2*mTau1,mTau1))-57739/24*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau1}},1/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau1}},1/6*mTau1,mTau1))+36031/120*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau1}},5/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau1}},5/6*mTau1,mTau1))+386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau1}},1/3,mTau1))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau1}},1/3,mTau1))-1330*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau1}},1/2,mTau1))-386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau1}},2/3,mTau1))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau1}},2/3,mTau1))+386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau1}},1/3+1/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau1}},1/3+1/3*mTau1,mTau1))-386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau1}},1/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau1}},1/3*mTau1,mTau1))+1127/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau1}},1/2,mTau1))+6216/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau1}},1/2+1/2*mTau1,mTau1))-1045553/60*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau1}},1/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau1}},1/3*mTau1,mTau1))+462814/45*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau1}},1/2*mTau1,mTau1))+22867/30*(-(Gt({{0,0},{0,0},{1,-1/6-1/2*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,1/6+1/2*mTau1}},1/2,mTau1))-6207/10*(-(Gt({{0,0},{0,0},{1,-1/6-1/2*mTau1}},2/3+1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/6+1/2*mTau1}},2/3+1/2*mTau1,mTau1))-72913/30*(-(Gt({{0,0},{0,0},{1,-1/3-1/2*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,1/3+1/2*mTau1}},1/2,mTau1))+4443/5*(-(Gt({{0,0},{0,0},{1,-1/3-1/2*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/3+1/2*mTau1}},1/2+1/2*mTau1,mTau1))-188113/900*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau1}},1/3,mTau1))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau1}},1/3,mTau1))+1582769/900*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau1}},2/3,mTau1))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau1}},2/3,mTau1))-4665/2*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau1}},5/6,mTau1))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau1}},5/6,mTau1))+630*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau1}},1/2+1/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau1}},1/2+1/6*mTau1,mTau1))+1052*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau1}},1/2+1/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau1}},1/2+1/3*mTau1,mTau1))-2734/5*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau1}},5/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau1}},5/6*mTau1,mTau1))-31277/30*(-(Gt({{0,0},{0,0},{1,-2/3-1/2*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,2/3+1/2*mTau1}},1/2,mTau1))-1764/5*(-(Gt({{0,0},{0,0},{1,-2/3-1/2*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,2/3+1/2*mTau1}},1/2+1/2*mTau1,mTau1))-9379/30*(-(Gt({{0,0},{0,0},{1,-5/6-1/2*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,5/6+1/2*mTau1}},1/2,mTau1))-1512/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/2*mTau1}},2/3,mTau1))+Gt({{0,0},{0,0},{1,5/6+1/2*mTau1}},2/3,mTau1))+1512/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/2*mTau1}},5/6,mTau1))+Gt({{0,0},{0,0},{1,5/6+1/2*mTau1}},5/6,mTau1))+6207/10*(-(Gt({{0,0},{0,0},{1,-5/6-1/2*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,5/6+1/2*mTau1}},1/2*mTau1,mTau1))+7343/5*(-(Gt({{0,0},{0,0},{1,-1/6-2/3*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,1/6+2/3*mTau1}},1/2,mTau1))-4281/10*(-(Gt({{0,0},{0,0},{1,-1/6-2/3*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/6+2/3*mTau1}},1/2*mTau1,mTau1))+2541/10*(-(Gt({{0,0},{0,0},{1,-1/6-2/3*mTau1}},2/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/6+2/3*mTau1}},2/3*mTau1,mTau1))+665*(-(Gt({{0,0},{0,0},{1,-1/3-2/3*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,1/3+2/3*mTau1}},1/2,mTau1))+462/5*(-(Gt({{0,0},{0,0},{1,-1/2-2/3*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,1/2+2/3*mTau1}},1/2,mTau1))-386267/30*(-(Gt({{0,0},{0,0},{1,-2/3-2/3*mTau1}},1/3,mTau1))+Gt({{0,0},{0,0},{1,2/3+2/3*mTau1}},1/3,mTau1))+386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-2/3*mTau1}},2/3,mTau1))+Gt({{0,0},{0,0},{1,2/3+2/3*mTau1}},2/3,mTau1))+1338/5*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau1}},1/2,mTau1))+288/5*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau1}},2/3,mTau1))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau1}},2/3,mTau1))+6216/5*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau1}},1/6+1/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau1}},1/6+1/6*mTau1,mTau1))+168/5*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau1}},2/3+5/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau1}},2/3+5/6*mTau1,mTau1))-64911/40*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau1}},1/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau1}},1/6*mTau1,mTau1))-4737/40*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau1}},5/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau1}},5/6*mTau1,mTau1))-1579/10*(-(Gt({{0,0},{0,0},{1,-1/6-5/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/6+5/6*mTau1}},1/2*mTau1,mTau1))-165/2*(-(Gt({{0,0},{0,0},{1,-1/6-5/6*mTau1}},2/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/6+5/6*mTau1}},2/3*mTau1,mTau1))+893/5*(-(Gt({{0,0},{0,0},{1,-1/3-5/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/3+5/6*mTau1}},1/2*mTau1,mTau1))-126*(-(Gt({{0,0},{0,0},{1,-1/2-5/6*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/2+5/6*mTau1}},1/2+1/2*mTau1,mTau1))+168/5*(-(Gt({{0,0},{0,0},{1,-1/2-5/6*mTau1}},1/2+2/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/2+5/6*mTau1}},1/2+2/3*mTau1,mTau1))+406/5*(-(Gt({{0,0},{0,0},{1,-1/2-5/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/2+5/6*mTau1}},1/2*mTau1,mTau1))-6666/5*(-(Gt({{0,0},{0,0},{1,-2/3-5/6*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,2/3+5/6*mTau1}},1/2,mTau1))-41*(-(Gt({{0,0},{0,0},{1,-2/3-5/6*mTau1}},1/2+1/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,2/3+5/6*mTau1}},1/2+1/3*mTau1,mTau1))+221/5*(-(Gt({{0,0},{0,0},{1,-2/3-5/6*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,2/3+5/6*mTau1}},1/2+1/2*mTau1,mTau1))-253/15*(-(Gt({{0,0},{0,0},{1,-2/3-5/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,2/3+5/6*mTau1}},1/2*mTau1,mTau1))-37841/225*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau1}},1/3,mTau1))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau1}},1/3,mTau1))+321817/225*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau1}},1/2,mTau1))-136121/225*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau1}},2/3,mTau1))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau1}},2/3,mTau1))+12277/25*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau1}},5/6,mTau1))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau1}},5/6,mTau1))-208783/12*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau1}},1/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau1}},1/6*mTau1,mTau1))+1078601/180*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau1}},1/2*mTau1,mTau1))-846/5*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau1}},5/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau1}},5/6*mTau1,mTau1))-13319/8*(-(Gt({{0,0},{0,0},{1,-1/6*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,1/6*mTau1}},1/2,mTau1))+8037/40*(-(Gt({{0,0},{0,0},{1,-1/6*mTau1}},5/6+2/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/6*mTau1}},5/6+2/3*mTau1,mTau1))-462/5*(-(Gt({{0,0},{0,0},{1,-1/3*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,1/3*mTau1}},1/2,mTau1))-755827/600*(-(Gt({{0,0},{0,0},{1,-1/2*mTau1}},5/6,mTau1))+Gt({{0,0},{0,0},{1,1/2*mTau1}},5/6,mTau1))-22841/10*(-(Gt({{0,0},{0,0},{1,-1/2*mTau1}},2/3+1/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/2*mTau1}},2/3+1/6*mTau1,mTau1))-1371547/180*(-(Gt({{0,0},{0,0},{1,-1/2*mTau1}},5/6+1/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/2*mTau1}},5/6+1/3*mTau1,mTau1))+969431/60*(-(Gt({{0,0},{0,0},{1,-1/2*mTau1}},5/6+1/2*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/2*mTau1}},5/6+1/2*mTau1,mTau1))+20261/120*(-(Gt({{0,0},{0,0},{1,-1/2*mTau1}},1/6+2/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/2*mTau1}},1/6+2/3*mTau1,mTau1))+577/5*(-(Gt({{0,0},{0,0},{1,-1/2*mTau1}},1/3+5/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/2*mTau1}},1/3+5/6*mTau1,mTau1))+1639/15*(-(Gt({{0,0},{0,0},{1,-1/2*mTau1}},2/3+5/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/2*mTau1}},2/3+5/6*mTau1,mTau1))-1011209/180*(-(Gt({{0,0},{0,0},{1,-1/2*mTau1}},5/6+5/6*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/2*mTau1}},5/6+5/6*mTau1,mTau1))-11644/5*(-(Gt({{0,0},{0,0},{1,-1/2*mTau1}},1/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,1/2*mTau1}},1/3*mTau1,mTau1))+231/5*(-(Gt({{0,0},{0,0},{1,-2/3*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,2/3*mTau1}},1/2,mTau1))-70291/40*(-(Gt({{0,0},{0,0},{1,-5/6*mTau1}},1/2,mTau1))+Gt({{0,0},{0,0},{1,5/6*mTau1}},1/2,mTau1))+8037/40*(-(Gt({{0,0},{0,0},{1,-5/6*mTau1}},5/6+2/3*mTau1,mTau1))+Gt({{0,0},{0,0},{1,5/6*mTau1}},5/6+2/3*mTau1,mTau1))",
		"25.6176497080888367938616255837404512389096682806130938344226300380162829339297258844440649248504598923158978105147913");
	test(2,"2197/50*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6}},5/6+5/6*mTau1,mTau1))+Gt({{0,0},{2,1/6}},5/6+5/6*mTau1,mTau1))-10409/15*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6}},1/2*mTau1,mTau1))+Gt({{0,0},{2,1/6}},1/2*mTau1,mTau1))-6591/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3}},1/6,mTau1))+Gt({{0,0},{2,1/3}},1/6,mTau1))-386267/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3}},2/3+1/3*mTau1,mTau1))+Gt({{0,0},{2,1/3}},2/3+1/3*mTau1,mTau1))+386267/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3}},2/3+2/3*mTau1,mTau1))+Gt({{0,0},{2,1/3}},2/3+2/3*mTau1,mTau1))+18621/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3}},1/2*mTau1,mTau1))+Gt({{0,0},{2,1/3}},1/2*mTau1,mTau1))-386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3}},2/3*mTau1,mTau1))+Gt({{0,0},{2,1/3}},2/3*mTau1,mTau1))+665*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/6*mTau1}},1/2,mTau1))+Gt({{0,0},{2,1/6+1/6*mTau1}},1/2,mTau1))-893/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{2,1/6+1/6*mTau1}},1/2*mTau1,mTau1))+1271/4*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/6*mTau1}},1/2,mTau1))+Gt({{0,0},{2,2/3+1/6*mTau1}},1/2,mTau1))-1271/4*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/6*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{2,2/3+1/6*mTau1}},1/2+1/2*mTau1,mTau1))+8141/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{2,2/3+1/6*mTau1}},1/2*mTau1,mTau1))+420*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/6*mTau1}},2/3,mTau1))+Gt({{0,0},{2,5/6+1/6*mTau1}},2/3,mTau1))+1008/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/6*mTau1}},5/6,mTau1))+Gt({{0,0},{2,5/6+1/6*mTau1}},5/6,mTau1))-231*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/6*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{2,5/6+1/6*mTau1}},1/2+1/2*mTau1,mTau1))+288/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/6*mTau1}},1/6*mTau1,mTau1))+Gt({{0,0},{2,5/6+1/6*mTau1}},1/6*mTau1,mTau1))+1557/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{2,5/6+1/6*mTau1}},1/2*mTau1,mTau1))-280*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/3*mTau1}},1/2,mTau1))+Gt({{0,0},{2,1/6+1/3*mTau1}},1/2,mTau1))+84/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/3*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{2,1/6+1/3*mTau1}},1/2+1/2*mTau1,mTau1))-57739/48*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/3*mTau1}},1/6*mTau1,mTau1))+Gt({{0,0},{2,1/6+1/3*mTau1}},1/6*mTau1,mTau1))+36031/240*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/3*mTau1}},5/6*mTau1,mTau1))+Gt({{0,0},{2,1/6+1/3*mTau1}},5/6*mTau1,mTau1))+386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/3*mTau1}},1/3,mTau1))+Gt({{0,0},{2,2/3+1/3*mTau1}},1/3,mTau1))-665*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/3*mTau1}},1/2,mTau1))+Gt({{0,0},{2,2/3+1/3*mTau1}},1/2,mTau1))-386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/3*mTau1}},2/3,mTau1))+Gt({{0,0},{2,2/3+1/3*mTau1}},2/3,mTau1))+386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/3*mTau1}},1/3+1/3*mTau1,mTau1))+Gt({{0,0},{2,2/3+1/3*mTau1}},1/3+1/3*mTau1,mTau1))-386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/3*mTau1}},1/3*mTau1,mTau1))+Gt({{0,0},{2,2/3+1/3*mTau1}},1/3*mTau1,mTau1))+1127/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/3*mTau1}},1/2,mTau1))+Gt({{0,0},{2,5/6+1/3*mTau1}},1/2,mTau1))+3108/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/3*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{2,5/6+1/3*mTau1}},1/2+1/2*mTau1,mTau1))-1045553/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/3*mTau1}},1/3*mTau1,mTau1))+Gt({{0,0},{2,5/6+1/3*mTau1}},1/3*mTau1,mTau1))+231407/45*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/3*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{2,5/6+1/3*mTau1}},1/2*mTau1,mTau1))+22867/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/2*mTau1}},1/2,mTau1))+Gt({{0,0},{2,1/6+1/2*mTau1}},1/2,mTau1))-6207/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/2*mTau1}},2/3+1/2*mTau1,mTau1))+Gt({{0,0},{2,1/6+1/2*mTau1}},2/3+1/2*mTau1,mTau1))-72913/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3-1/2*mTau1}},1/2,mTau1))+Gt({{0,0},{2,1/3+1/2*mTau1}},1/2,mTau1))+4443/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3-1/2*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{2,1/3+1/2*mTau1}},1/2+1/2*mTau1,mTau1))-188113/1800*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau1}},1/3,mTau1))+Gt({{0,0},{2,1/2+1/2*mTau1}},1/3,mTau1))+1582769/1800*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau1}},2/3,mTau1))+Gt({{0,0},{2,1/2+1/2*mTau1}},2/3,mTau1))-4665/4*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau1}},5/6,mTau1))+Gt({{0,0},{2,1/2+1/2*mTau1}},5/6,mTau1))+315*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau1}},1/2+1/6*mTau1,mTau1))+Gt({{0,0},{2,1/2+1/2*mTau1}},1/2+1/6*mTau1,mTau1))+526*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau1}},1/2+1/3*mTau1,mTau1))+Gt({{0,0},{2,1/2+1/2*mTau1}},1/2+1/3*mTau1,mTau1))-1367/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau1}},5/6*mTau1,mTau1))+Gt({{0,0},{2,1/2+1/2*mTau1}},5/6*mTau1,mTau1))-31277/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/2*mTau1}},1/2,mTau1))+Gt({{0,0},{2,2/3+1/2*mTau1}},1/2,mTau1))-882/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/2*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{2,2/3+1/2*mTau1}},1/2+1/2*mTau1,mTau1))-9379/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/2*mTau1}},1/2,mTau1))+Gt({{0,0},{2,5/6+1/2*mTau1}},1/2,mTau1))-756/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/2*mTau1}},2/3,mTau1))+Gt({{0,0},{2,5/6+1/2*mTau1}},2/3,mTau1))+756/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/2*mTau1}},5/6,mTau1))+Gt({{0,0},{2,5/6+1/2*mTau1}},5/6,mTau1))+6207/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/2*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{2,5/6+1/2*mTau1}},1/2*mTau1,mTau1))+7343/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-2/3*mTau1}},1/2,mTau1))+Gt({{0,0},{2,1/6+2/3*mTau1}},1/2,mTau1))-4281/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-2/3*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{2,1/6+2/3*mTau1}},1/2*mTau1,mTau1))+2541/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-2/3*mTau1}},2/3*mTau1,mTau1))+Gt({{0,0},{2,1/6+2/3*mTau1}},2/3*mTau1,mTau1))+665/2*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3-2/3*mTau1}},1/2,mTau1))+Gt({{0,0},{2,1/3+2/3*mTau1}},1/2,mTau1))+231/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-2/3*mTau1}},1/2,mTau1))+Gt({{0,0},{2,1/2+2/3*mTau1}},1/2,mTau1))-386267/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-2/3*mTau1}},1/3,mTau1))+Gt({{0,0},{2,2/3+2/3*mTau1}},1/3,mTau1))+386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-2/3*mTau1}},2/3,mTau1))+Gt({{0,0},{2,2/3+2/3*mTau1}},2/3,mTau1))+669/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau1}},1/2,mTau1))+Gt({{0,0},{2,5/6+2/3*mTau1}},1/2,mTau1))+144/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau1}},2/3,mTau1))+Gt({{0,0},{2,5/6+2/3*mTau1}},2/3,mTau1))+3108/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau1}},1/6+1/6*mTau1,mTau1))+Gt({{0,0},{2,5/6+2/3*mTau1}},1/6+1/6*mTau1,mTau1))+84/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau1}},2/3+5/6*mTau1,mTau1))+Gt({{0,0},{2,5/6+2/3*mTau1}},2/3+5/6*mTau1,mTau1))-64911/80*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau1}},1/6*mTau1,mTau1))+Gt({{0,0},{2,5/6+2/3*mTau1}},1/6*mTau1,mTau1))-4737/80*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau1}},5/6*mTau1,mTau1))+Gt({{0,0},{2,5/6+2/3*mTau1}},5/6*mTau1,mTau1))-1579/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-5/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{2,1/6+5/6*mTau1}},1/2*mTau1,mTau1))-165/4*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-5/6*mTau1}},2/3*mTau1,mTau1))+Gt({{0,0},{2,1/6+5/6*mTau1}},2/3*mTau1,mTau1))+893/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3-5/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{2,1/3+5/6*mTau1}},1/2*mTau1,mTau1))-63*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-5/6*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{2,1/2+5/6*mTau1}},1/2+1/2*mTau1,mTau1))+84/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-5/6*mTau1}},1/2+2/3*mTau1,mTau1))+Gt({{0,0},{2,1/2+5/6*mTau1}},1/2+2/3*mTau1,mTau1))+203/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-5/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{2,1/2+5/6*mTau1}},1/2*mTau1,mTau1))-3333/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-5/6*mTau1}},1/2,mTau1))+Gt({{0,0},{2,2/3+5/6*mTau1}},1/2,mTau1))-41/2*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-5/6*mTau1}},1/2+1/3*mTau1,mTau1))+Gt({{0,0},{2,2/3+5/6*mTau1}},1/2+1/3*mTau1,mTau1))+221/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-5/6*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{0,0},{2,2/3+5/6*mTau1}},1/2+1/2*mTau1,mTau1))-253/30*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-5/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{2,2/3+5/6*mTau1}},1/2*mTau1,mTau1))-37841/450*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau1}},1/3,mTau1))+Gt({{0,0},{2,5/6+5/6*mTau1}},1/3,mTau1))+321817/450*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau1}},1/2,mTau1))+Gt({{0,0},{2,5/6+5/6*mTau1}},1/2,mTau1))-136121/450*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau1}},2/3,mTau1))+Gt({{0,0},{2,5/6+5/6*mTau1}},2/3,mTau1))+12277/50*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau1}},5/6,mTau1))+Gt({{0,0},{2,5/6+5/6*mTau1}},5/6,mTau1))-208783/24*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau1}},1/6*mTau1,mTau1))+Gt({{0,0},{2,5/6+5/6*mTau1}},1/6*mTau1,mTau1))+1078601/360*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau1}},1/2*mTau1,mTau1))+Gt({{0,0},{2,5/6+5/6*mTau1}},1/2*mTau1,mTau1))-423/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau1}},5/6*mTau1,mTau1))+Gt({{0,0},{2,5/6+5/6*mTau1}},5/6*mTau1,mTau1))-13319/16*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6*mTau1}},1/2,mTau1))+Gt({{0,0},{2,1/6*mTau1}},1/2,mTau1))+8037/80*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6*mTau1}},5/6+2/3*mTau1,mTau1))+Gt({{0,0},{2,1/6*mTau1}},5/6+2/3*mTau1,mTau1))-231/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3*mTau1}},1/2,mTau1))+Gt({{0,0},{2,1/3*mTau1}},1/2,mTau1))-755827/1200*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau1}},5/6,mTau1))+Gt({{0,0},{2,1/2*mTau1}},5/6,mTau1))-22841/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau1}},2/3+1/6*mTau1,mTau1))+Gt({{0,0},{2,1/2*mTau1}},2/3+1/6*mTau1,mTau1))-1371547/360*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau1}},5/6+1/3*mTau1,mTau1))+Gt({{0,0},{2,1/2*mTau1}},5/6+1/3*mTau1,mTau1))+969431/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau1}},5/6+1/2*mTau1,mTau1))+Gt({{0,0},{2,1/2*mTau1}},5/6+1/2*mTau1,mTau1))+20261/240*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau1}},1/6+2/3*mTau1,mTau1))+Gt({{0,0},{2,1/2*mTau1}},1/6+2/3*mTau1,mTau1))+577/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau1}},1/3+5/6*mTau1,mTau1))+Gt({{0,0},{2,1/2*mTau1}},1/3+5/6*mTau1,mTau1))+1639/30*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau1}},2/3+5/6*mTau1,mTau1))+Gt({{0,0},{2,1/2*mTau1}},2/3+5/6*mTau1,mTau1))-1011209/360*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau1}},5/6+5/6*mTau1,mTau1))+Gt({{0,0},{2,1/2*mTau1}},5/6+5/6*mTau1,mTau1))-5822/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau1}},1/3*mTau1,mTau1))+Gt({{0,0},{2,1/2*mTau1}},1/3*mTau1,mTau1))+231/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3*mTau1}},1/2,mTau1))+Gt({{0,0},{2,2/3*mTau1}},1/2,mTau1))-70291/80*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6*mTau1}},1/2,mTau1))+Gt({{0,0},{2,5/6*mTau1}},1/2,mTau1))+8037/80*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6*mTau1}},5/6+2/3*mTau1,mTau1))+Gt({{0,0},{2,5/6*mTau1}},5/6+2/3*mTau1,mTau1))",
		"75.4124433800420659075521880745465899659418510435675780598576614472786636537594933918719388027054148821286973188342351");
	test(3,"2197/300*(pow(Pi,-2))*(-(Gt({{3,-1/6}},5/6+5/6*mTau1,mTau1))+Gt({{3,1/6}},5/6+5/6*mTau1,mTau1))-10409/90*(pow(Pi,-2))*(-(Gt({{3,-1/6}},1/2*mTau1,mTau1))+Gt({{3,1/6}},1/2*mTau1,mTau1))-2197/40*(pow(Pi,-2))*(-(Gt({{3,-1/3}},1/6,mTau1))+Gt({{3,1/3}},1/6,mTau1))-386267/360*(pow(Pi,-2))*(-(Gt({{3,-1/3}},2/3+1/3*mTau1,mTau1))+Gt({{3,1/3}},2/3+1/3*mTau1,mTau1))+386267/360*(pow(Pi,-2))*(-(Gt({{3,-1/3}},2/3+2/3*mTau1,mTau1))+Gt({{3,1/3}},2/3+2/3*mTau1,mTau1))+6207/40*(pow(Pi,-2))*(-(Gt({{3,-1/3}},1/2*mTau1,mTau1))+Gt({{3,1/3}},1/2*mTau1,mTau1))-386267/720*(pow(Pi,-2))*(-(Gt({{3,-1/3}},2/3*mTau1,mTau1))+Gt({{3,1/3}},2/3*mTau1,mTau1))+665/6*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/6*mTau1}},1/2,mTau1))+Gt({{3,1/6+1/6*mTau1}},1/2,mTau1))-893/120*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/6*mTau1}},1/2*mTau1,mTau1))+Gt({{3,1/6+1/6*mTau1}},1/2*mTau1,mTau1))+1271/24*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/6*mTau1}},1/2,mTau1))+Gt({{3,2/3+1/6*mTau1}},1/2,mTau1))-1271/24*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/6*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{3,2/3+1/6*mTau1}},1/2+1/2*mTau1,mTau1))+8141/120*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/6*mTau1}},1/2*mTau1,mTau1))+Gt({{3,2/3+1/6*mTau1}},1/2*mTau1,mTau1))+70*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/6*mTau1}},2/3,mTau1))+Gt({{3,5/6+1/6*mTau1}},2/3,mTau1))+168/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/6*mTau1}},5/6,mTau1))+Gt({{3,5/6+1/6*mTau1}},5/6,mTau1))-77/2*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/6*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{3,5/6+1/6*mTau1}},1/2+1/2*mTau1,mTau1))+48/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/6*mTau1}},1/6*mTau1,mTau1))+Gt({{3,5/6+1/6*mTau1}},1/6*mTau1,mTau1))+519/40*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/6*mTau1}},1/2*mTau1,mTau1))+Gt({{3,5/6+1/6*mTau1}},1/2*mTau1,mTau1))-140/3*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/3*mTau1}},1/2,mTau1))+Gt({{3,1/6+1/3*mTau1}},1/2,mTau1))+14/5*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/3*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{3,1/6+1/3*mTau1}},1/2+1/2*mTau1,mTau1))-57739/288*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/3*mTau1}},1/6*mTau1,mTau1))+Gt({{3,1/6+1/3*mTau1}},1/6*mTau1,mTau1))+36031/1440*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/3*mTau1}},5/6*mTau1,mTau1))+Gt({{3,1/6+1/3*mTau1}},5/6*mTau1,mTau1))+386267/720*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/3*mTau1}},1/3,mTau1))+Gt({{3,2/3+1/3*mTau1}},1/3,mTau1))-665/6*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/3*mTau1}},1/2,mTau1))+Gt({{3,2/3+1/3*mTau1}},1/2,mTau1))-386267/720*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/3*mTau1}},2/3,mTau1))+Gt({{3,2/3+1/3*mTau1}},2/3,mTau1))+386267/720*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/3*mTau1}},1/3+1/3*mTau1,mTau1))+Gt({{3,2/3+1/3*mTau1}},1/3+1/3*mTau1,mTau1))-386267/720*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/3*mTau1}},1/3*mTau1,mTau1))+Gt({{3,2/3+1/3*mTau1}},1/3*mTau1,mTau1))+1127/60*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/3*mTau1}},1/2,mTau1))+Gt({{3,5/6+1/3*mTau1}},1/2,mTau1))+518/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/3*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{3,5/6+1/3*mTau1}},1/2+1/2*mTau1,mTau1))-1045553/720*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/3*mTau1}},1/3*mTau1,mTau1))+Gt({{3,5/6+1/3*mTau1}},1/3*mTau1,mTau1))+231407/270*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/3*mTau1}},1/2*mTau1,mTau1))+Gt({{3,5/6+1/3*mTau1}},1/2*mTau1,mTau1))+22867/360*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/2*mTau1}},1/2,mTau1))+Gt({{3,1/6+1/2*mTau1}},1/2,mTau1))-2069/40*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/2*mTau1}},2/3+1/2*mTau1,mTau1))+Gt({{3,1/6+1/2*mTau1}},2/3+1/2*mTau1,mTau1))-72913/360*(pow(Pi,-2))*(-(Gt({{3,-1/3-1/2*mTau1}},1/2,mTau1))+Gt({{3,1/3+1/2*mTau1}},1/2,mTau1))+1481/20*(pow(Pi,-2))*(-(Gt({{3,-1/3-1/2*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{3,1/3+1/2*mTau1}},1/2+1/2*mTau1,mTau1))-188113/10800*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau1}},1/3,mTau1))+Gt({{3,1/2+1/2*mTau1}},1/3,mTau1))+1582769/10800*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau1}},2/3,mTau1))+Gt({{3,1/2+1/2*mTau1}},2/3,mTau1))-1555/8*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau1}},5/6,mTau1))+Gt({{3,1/2+1/2*mTau1}},5/6,mTau1))+105/2*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau1}},1/2+1/6*mTau1,mTau1))+Gt({{3,1/2+1/2*mTau1}},1/2+1/6*mTau1,mTau1))+263/3*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau1}},1/2+1/3*mTau1,mTau1))+Gt({{3,1/2+1/2*mTau1}},1/2+1/3*mTau1,mTau1))-1367/30*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau1}},5/6*mTau1,mTau1))+Gt({{3,1/2+1/2*mTau1}},5/6*mTau1,mTau1))-31277/360*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/2*mTau1}},1/2,mTau1))+Gt({{3,2/3+1/2*mTau1}},1/2,mTau1))-147/5*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/2*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{3,2/3+1/2*mTau1}},1/2+1/2*mTau1,mTau1))-9379/360*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/2*mTau1}},1/2,mTau1))+Gt({{3,5/6+1/2*mTau1}},1/2,mTau1))-126/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/2*mTau1}},2/3,mTau1))+Gt({{3,5/6+1/2*mTau1}},2/3,mTau1))+126/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/2*mTau1}},5/6,mTau1))+Gt({{3,5/6+1/2*mTau1}},5/6,mTau1))+2069/40*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/2*mTau1}},1/2*mTau1,mTau1))+Gt({{3,5/6+1/2*mTau1}},1/2*mTau1,mTau1))+7343/60*(pow(Pi,-2))*(-(Gt({{3,-1/6-2/3*mTau1}},1/2,mTau1))+Gt({{3,1/6+2/3*mTau1}},1/2,mTau1))-1427/40*(pow(Pi,-2))*(-(Gt({{3,-1/6-2/3*mTau1}},1/2*mTau1,mTau1))+Gt({{3,1/6+2/3*mTau1}},1/2*mTau1,mTau1))+847/40*(pow(Pi,-2))*(-(Gt({{3,-1/6-2/3*mTau1}},2/3*mTau1,mTau1))+Gt({{3,1/6+2/3*mTau1}},2/3*mTau1,mTau1))+665/12*(pow(Pi,-2))*(-(Gt({{3,-1/3-2/3*mTau1}},1/2,mTau1))+Gt({{3,1/3+2/3*mTau1}},1/2,mTau1))+77/10*(pow(Pi,-2))*(-(Gt({{3,-1/2-2/3*mTau1}},1/2,mTau1))+Gt({{3,1/2+2/3*mTau1}},1/2,mTau1))-386267/360*(pow(Pi,-2))*(-(Gt({{3,-2/3-2/3*mTau1}},1/3,mTau1))+Gt({{3,2/3+2/3*mTau1}},1/3,mTau1))+386267/720*(pow(Pi,-2))*(-(Gt({{3,-2/3-2/3*mTau1}},2/3,mTau1))+Gt({{3,2/3+2/3*mTau1}},2/3,mTau1))+223/10*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau1}},1/2,mTau1))+Gt({{3,5/6+2/3*mTau1}},1/2,mTau1))+24/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau1}},2/3,mTau1))+Gt({{3,5/6+2/3*mTau1}},2/3,mTau1))+518/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau1}},1/6+1/6*mTau1,mTau1))+Gt({{3,5/6+2/3*mTau1}},1/6+1/6*mTau1,mTau1))+14/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau1}},2/3+5/6*mTau1,mTau1))+Gt({{3,5/6+2/3*mTau1}},2/3+5/6*mTau1,mTau1))-21637/160*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau1}},1/6*mTau1,mTau1))+Gt({{3,5/6+2/3*mTau1}},1/6*mTau1,mTau1))-1579/160*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau1}},5/6*mTau1,mTau1))+Gt({{3,5/6+2/3*mTau1}},5/6*mTau1,mTau1))-1579/120*(pow(Pi,-2))*(-(Gt({{3,-1/6-5/6*mTau1}},1/2*mTau1,mTau1))+Gt({{3,1/6+5/6*mTau1}},1/2*mTau1,mTau1))-55/8*(pow(Pi,-2))*(-(Gt({{3,-1/6-5/6*mTau1}},2/3*mTau1,mTau1))+Gt({{3,1/6+5/6*mTau1}},2/3*mTau1,mTau1))+893/60*(pow(Pi,-2))*(-(Gt({{3,-1/3-5/6*mTau1}},1/2*mTau1,mTau1))+Gt({{3,1/3+5/6*mTau1}},1/2*mTau1,mTau1))-21/2*(pow(Pi,-2))*(-(Gt({{3,-1/2-5/6*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{3,1/2+5/6*mTau1}},1/2+1/2*mTau1,mTau1))+14/5*(pow(Pi,-2))*(-(Gt({{3,-1/2-5/6*mTau1}},1/2+2/3*mTau1,mTau1))+Gt({{3,1/2+5/6*mTau1}},1/2+2/3*mTau1,mTau1))+203/30*(pow(Pi,-2))*(-(Gt({{3,-1/2-5/6*mTau1}},1/2*mTau1,mTau1))+Gt({{3,1/2+5/6*mTau1}},1/2*mTau1,mTau1))-1111/10*(pow(Pi,-2))*(-(Gt({{3,-2/3-5/6*mTau1}},1/2,mTau1))+Gt({{3,2/3+5/6*mTau1}},1/2,mTau1))-41/12*(pow(Pi,-2))*(-(Gt({{3,-2/3-5/6*mTau1}},1/2+1/3*mTau1,mTau1))+Gt({{3,2/3+5/6*mTau1}},1/2+1/3*mTau1,mTau1))+221/60*(pow(Pi,-2))*(-(Gt({{3,-2/3-5/6*mTau1}},1/2+1/2*mTau1,mTau1))+Gt({{3,2/3+5/6*mTau1}},1/2+1/2*mTau1,mTau1))-253/180*(pow(Pi,-2))*(-(Gt({{3,-2/3-5/6*mTau1}},1/2*mTau1,mTau1))+Gt({{3,2/3+5/6*mTau1}},1/2*mTau1,mTau1))-37841/2700*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau1}},1/3,mTau1))+Gt({{3,5/6+5/6*mTau1}},1/3,mTau1))+321817/2700*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau1}},1/2,mTau1))+Gt({{3,5/6+5/6*mTau1}},1/2,mTau1))-136121/2700*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau1}},2/3,mTau1))+Gt({{3,5/6+5/6*mTau1}},2/3,mTau1))+12277/300*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau1}},5/6,mTau1))+Gt({{3,5/6+5/6*mTau1}},5/6,mTau1))-208783/144*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau1}},1/6*mTau1,mTau1))+Gt({{3,5/6+5/6*mTau1}},1/6*mTau1,mTau1))+1078601/2160*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau1}},1/2*mTau1,mTau1))+Gt({{3,5/6+5/6*mTau1}},1/2*mTau1,mTau1))-141/10*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau1}},5/6*mTau1,mTau1))+Gt({{3,5/6+5/6*mTau1}},5/6*mTau1,mTau1))-13319/96*(pow(Pi,-2))*(-(Gt({{3,-1/6*mTau1}},1/2,mTau1))+Gt({{3,1/6*mTau1}},1/2,mTau1))+2679/160*(pow(Pi,-2))*(-(Gt({{3,-1/6*mTau1}},5/6+2/3*mTau1,mTau1))+Gt({{3,1/6*mTau1}},5/6+2/3*mTau1,mTau1))-77/10*(pow(Pi,-2))*(-(Gt({{3,-1/3*mTau1}},1/2,mTau1))+Gt({{3,1/3*mTau1}},1/2,mTau1))-755827/7200*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau1}},5/6,mTau1))+Gt({{3,1/2*mTau1}},5/6,mTau1))-22841/120*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau1}},2/3+1/6*mTau1,mTau1))+Gt({{3,1/2*mTau1}},2/3+1/6*mTau1,mTau1))-1371547/2160*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau1}},5/6+1/3*mTau1,mTau1))+Gt({{3,1/2*mTau1}},5/6+1/3*mTau1,mTau1))+969431/720*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau1}},5/6+1/2*mTau1,mTau1))+Gt({{3,1/2*mTau1}},5/6+1/2*mTau1,mTau1))+20261/1440*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau1}},1/6+2/3*mTau1,mTau1))+Gt({{3,1/2*mTau1}},1/6+2/3*mTau1,mTau1))+577/60*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau1}},1/3+5/6*mTau1,mTau1))+Gt({{3,1/2*mTau1}},1/3+5/6*mTau1,mTau1))+1639/180*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau1}},2/3+5/6*mTau1,mTau1))+Gt({{3,1/2*mTau1}},2/3+5/6*mTau1,mTau1))-1011209/2160*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau1}},5/6+5/6*mTau1,mTau1))+Gt({{3,1/2*mTau1}},5/6+5/6*mTau1,mTau1))-2911/15*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau1}},1/3*mTau1,mTau1))+Gt({{3,1/2*mTau1}},1/3*mTau1,mTau1))+77/20*(pow(Pi,-2))*(-(Gt({{3,-2/3*mTau1}},1/2,mTau1))+Gt({{3,2/3*mTau1}},1/2,mTau1))-70291/480*(pow(Pi,-2))*(-(Gt({{3,-5/6*mTau1}},1/2,mTau1))+Gt({{3,5/6*mTau1}},1/2,mTau1))+2679/160*(pow(Pi,-2))*(-(Gt({{3,-5/6*mTau1}},5/6+2/3*mTau1,mTau1))+Gt({{3,5/6*mTau1}},5/6+2/3*mTau1,mTau1))",
		"25.1178990042108109572504191508733325549504277323742010664667512106819492532100588794494869807697402431701852966043169");
	if (abbreviated)
		return failed;
	test(4,"2197/25*(-(Gt({{0,0},{0,0},{1,-1/6}},5/6+5/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/6}},5/6+5/6*mTau2,mTau2))-20818/15*(-(Gt({{0,0},{0,0},{1,-1/6}},1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/6}},1/2*mTau2,mTau2))-6591/10*(-(Gt({{0,0},{0,0},{1,-1/3}},1/6,mTau2))+Gt({{0,0},{0,0},{1,1/3}},1/6,mTau2))-386267/30*(-(Gt({{0,0},{0,0},{1,-1/3}},2/3+1/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/3}},2/3+1/3*mTau2,mTau2))+386267/30*(-(Gt({{0,0},{0,0},{1,-1/3}},2/3+2/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/3}},2/3+2/3*mTau2,mTau2))+18621/10*(-(Gt({{0,0},{0,0},{1,-1/3}},1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/3}},1/2*mTau2,mTau2))-386267/60*(-(Gt({{0,0},{0,0},{1,-1/3}},2/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/3}},2/3*mTau2,mTau2))+1330*(-(Gt({{0,0},{0,0},{1,-1/6-1/6*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,1/6+1/6*mTau2}},1/2,mTau2))-893/10*(-(Gt({{0,0},{0,0},{1,-1/6-1/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/6+1/6*mTau2}},1/2*mTau2,mTau2))+1271/2*(-(Gt({{0,0},{0,0},{1,-2/3-1/6*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,2/3+1/6*mTau2}},1/2,mTau2))-1271/2*(-(Gt({{0,0},{0,0},{1,-2/3-1/6*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,2/3+1/6*mTau2}},1/2+1/2*mTau2,mTau2))+8141/10*(-(Gt({{0,0},{0,0},{1,-2/3-1/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,2/3+1/6*mTau2}},1/2*mTau2,mTau2))+840*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau2}},2/3,mTau2))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau2}},2/3,mTau2))+2016/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau2}},5/6,mTau2))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau2}},5/6,mTau2))-462*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau2}},1/2+1/2*mTau2,mTau2))+576/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau2}},1/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau2}},1/6*mTau2,mTau2))+1557/10*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau2}},1/2*mTau2,mTau2))-560*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau2}},1/2,mTau2))+168/5*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau2}},1/2+1/2*mTau2,mTau2))-57739/24*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau2}},1/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau2}},1/6*mTau2,mTau2))+36031/120*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau2}},5/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau2}},5/6*mTau2,mTau2))+386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau2}},1/3,mTau2))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau2}},1/3,mTau2))-1330*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau2}},1/2,mTau2))-386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau2}},2/3,mTau2))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau2}},2/3,mTau2))+386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau2}},1/3+1/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau2}},1/3+1/3*mTau2,mTau2))-386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau2}},1/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau2}},1/3*mTau2,mTau2))+1127/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau2}},1/2,mTau2))+6216/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau2}},1/2+1/2*mTau2,mTau2))-1045553/60*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau2}},1/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau2}},1/3*mTau2,mTau2))+462814/45*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau2}},1/2*mTau2,mTau2))+22867/30*(-(Gt({{0,0},{0,0},{1,-1/6-1/2*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,1/6+1/2*mTau2}},1/2,mTau2))-6207/10*(-(Gt({{0,0},{0,0},{1,-1/6-1/2*mTau2}},2/3+1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/6+1/2*mTau2}},2/3+1/2*mTau2,mTau2))-72913/30*(-(Gt({{0,0},{0,0},{1,-1/3-1/2*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,1/3+1/2*mTau2}},1/2,mTau2))+4443/5*(-(Gt({{0,0},{0,0},{1,-1/3-1/2*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/3+1/2*mTau2}},1/2+1/2*mTau2,mTau2))-188113/900*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau2}},1/3,mTau2))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau2}},1/3,mTau2))+1582769/900*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau2}},2/3,mTau2))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau2}},2/3,mTau2))-4665/2*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau2}},5/6,mTau2))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau2}},5/6,mTau2))+630*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau2}},1/2+1/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau2}},1/2+1/6*mTau2,mTau2))+1052*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau2}},1/2+1/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau2}},1/2+1/3*mTau2,mTau2))-2734/5*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau2}},5/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau2}},5/6*mTau2,mTau2))-31277/30*(-(Gt({{0,0},{0,0},{1,-2/3-1/2*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,2/3+1/2*mTau2}},1/2,mTau2))-1764/5*(-(Gt({{0,0},{0,0},{1,-2/3-1/2*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,2/3+1/2*mTau2}},1/2+1/2*mTau2,mTau2))-9379/30*(-(Gt({{0,0},{0,0},{1,-5/6-1/2*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,5/6+1/2*mTau2}},1/2,mTau2))-1512/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/2*mTau2}},2/3,mTau2))+Gt({{0,0},{0,0},{1,5/6+1/2*mTau2}},2/3,mTau2))+1512/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/2*mTau2}},5/6,mTau2))+Gt({{0,0},{0,0},{1,5/6+1/2*mTau2}},5/6,mTau2))+6207/10*(-(Gt({{0,0},{0,0},{1,-5/6-1/2*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,5/6+1/2*mTau2}},1/2*mTau2,mTau2))+7343/5*(-(Gt({{0,0},{0,0},{1,-1/6-2/3*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,1/6+2/3*mTau2}},1/2,mTau2))-4281/10*(-(Gt({{0,0},{0,0},{1,-1/6-2/3*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/6+2/3*mTau2}},1/2*mTau2,mTau2))+2541/10*(-(Gt({{0,0},{0,0},{1,-1/6-2/3*mTau2}},2/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/6+2/3*mTau2}},2/3*mTau2,mTau2))+665*(-(Gt({{0,0},{0,0},{1,-1/3-2/3*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,1/3+2/3*mTau2}},1/2,mTau2))+462/5*(-(Gt({{0,0},{0,0},{1,-1/2-2/3*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,1/2+2/3*mTau2}},1/2,mTau2))-386267/30*(-(Gt({{0,0},{0,0},{1,-2/3-2/3*mTau2}},1/3,mTau2))+Gt({{0,0},{0,0},{1,2/3+2/3*mTau2}},1/3,mTau2))+386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-2/3*mTau2}},2/3,mTau2))+Gt({{0,0},{0,0},{1,2/3+2/3*mTau2}},2/3,mTau2))+1338/5*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau2}},1/2,mTau2))+288/5*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau2}},2/3,mTau2))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau2}},2/3,mTau2))+6216/5*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau2}},1/6+1/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau2}},1/6+1/6*mTau2,mTau2))+168/5*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau2}},2/3+5/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau2}},2/3+5/6*mTau2,mTau2))-64911/40*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau2}},1/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau2}},1/6*mTau2,mTau2))-4737/40*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau2}},5/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau2}},5/6*mTau2,mTau2))-1579/10*(-(Gt({{0,0},{0,0},{1,-1/6-5/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/6+5/6*mTau2}},1/2*mTau2,mTau2))-165/2*(-(Gt({{0,0},{0,0},{1,-1/6-5/6*mTau2}},2/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/6+5/6*mTau2}},2/3*mTau2,mTau2))+893/5*(-(Gt({{0,0},{0,0},{1,-1/3-5/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/3+5/6*mTau2}},1/2*mTau2,mTau2))-126*(-(Gt({{0,0},{0,0},{1,-1/2-5/6*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/2+5/6*mTau2}},1/2+1/2*mTau2,mTau2))+168/5*(-(Gt({{0,0},{0,0},{1,-1/2-5/6*mTau2}},1/2+2/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/2+5/6*mTau2}},1/2+2/3*mTau2,mTau2))+406/5*(-(Gt({{0,0},{0,0},{1,-1/2-5/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/2+5/6*mTau2}},1/2*mTau2,mTau2))-6666/5*(-(Gt({{0,0},{0,0},{1,-2/3-5/6*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,2/3+5/6*mTau2}},1/2,mTau2))-41*(-(Gt({{0,0},{0,0},{1,-2/3-5/6*mTau2}},1/2+1/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,2/3+5/6*mTau2}},1/2+1/3*mTau2,mTau2))+221/5*(-(Gt({{0,0},{0,0},{1,-2/3-5/6*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,2/3+5/6*mTau2}},1/2+1/2*mTau2,mTau2))-253/15*(-(Gt({{0,0},{0,0},{1,-2/3-5/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,2/3+5/6*mTau2}},1/2*mTau2,mTau2))-37841/225*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau2}},1/3,mTau2))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau2}},1/3,mTau2))+321817/225*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau2}},1/2,mTau2))-136121/225*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau2}},2/3,mTau2))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau2}},2/3,mTau2))+12277/25*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau2}},5/6,mTau2))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau2}},5/6,mTau2))-208783/12*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau2}},1/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau2}},1/6*mTau2,mTau2))+1078601/180*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau2}},1/2*mTau2,mTau2))-846/5*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau2}},5/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau2}},5/6*mTau2,mTau2))-13319/8*(-(Gt({{0,0},{0,0},{1,-1/6*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,1/6*mTau2}},1/2,mTau2))+8037/40*(-(Gt({{0,0},{0,0},{1,-1/6*mTau2}},5/6+2/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/6*mTau2}},5/6+2/3*mTau2,mTau2))-462/5*(-(Gt({{0,0},{0,0},{1,-1/3*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,1/3*mTau2}},1/2,mTau2))-755827/600*(-(Gt({{0,0},{0,0},{1,-1/2*mTau2}},5/6,mTau2))+Gt({{0,0},{0,0},{1,1/2*mTau2}},5/6,mTau2))-22841/10*(-(Gt({{0,0},{0,0},{1,-1/2*mTau2}},2/3+1/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/2*mTau2}},2/3+1/6*mTau2,mTau2))-1371547/180*(-(Gt({{0,0},{0,0},{1,-1/2*mTau2}},5/6+1/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/2*mTau2}},5/6+1/3*mTau2,mTau2))+969431/60*(-(Gt({{0,0},{0,0},{1,-1/2*mTau2}},5/6+1/2*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/2*mTau2}},5/6+1/2*mTau2,mTau2))+20261/120*(-(Gt({{0,0},{0,0},{1,-1/2*mTau2}},1/6+2/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/2*mTau2}},1/6+2/3*mTau2,mTau2))+577/5*(-(Gt({{0,0},{0,0},{1,-1/2*mTau2}},1/3+5/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/2*mTau2}},1/3+5/6*mTau2,mTau2))+1639/15*(-(Gt({{0,0},{0,0},{1,-1/2*mTau2}},2/3+5/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/2*mTau2}},2/3+5/6*mTau2,mTau2))-1011209/180*(-(Gt({{0,0},{0,0},{1,-1/2*mTau2}},5/6+5/6*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/2*mTau2}},5/6+5/6*mTau2,mTau2))-11644/5*(-(Gt({{0,0},{0,0},{1,-1/2*mTau2}},1/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,1/2*mTau2}},1/3*mTau2,mTau2))+231/5*(-(Gt({{0,0},{0,0},{1,-2/3*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,2/3*mTau2}},1/2,mTau2))-70291/40*(-(Gt({{0,0},{0,0},{1,-5/6*mTau2}},1/2,mTau2))+Gt({{0,0},{0,0},{1,5/6*mTau2}},1/2,mTau2))+8037/40*(-(Gt({{0,0},{0,0},{1,-5/6*mTau2}},5/6+2/3*mTau2,mTau2))+Gt({{0,0},{0,0},{1,5/6*mTau2}},5/6+2/3*mTau2,mTau2))",
		"72.7438554014587368055360431381966310365001120163133888967613568497810648614604679648242062201092120381597518639048471-55.6179104763007951729712931555293719483587635401541546610854380198493368713558944022584064895364103557121883167608595*I");
	test(5,"2197/50*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6}},5/6+5/6*mTau2,mTau2))+Gt({{0,0},{2,1/6}},5/6+5/6*mTau2,mTau2))-10409/15*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6}},1/2*mTau2,mTau2))+Gt({{0,0},{2,1/6}},1/2*mTau2,mTau2))-6591/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3}},1/6,mTau2))+Gt({{0,0},{2,1/3}},1/6,mTau2))-386267/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3}},2/3+1/3*mTau2,mTau2))+Gt({{0,0},{2,1/3}},2/3+1/3*mTau2,mTau2))+386267/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3}},2/3+2/3*mTau2,mTau2))+Gt({{0,0},{2,1/3}},2/3+2/3*mTau2,mTau2))+18621/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3}},1/2*mTau2,mTau2))+Gt({{0,0},{2,1/3}},1/2*mTau2,mTau2))-386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3}},2/3*mTau2,mTau2))+Gt({{0,0},{2,1/3}},2/3*mTau2,mTau2))+665*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/6*mTau2}},1/2,mTau2))+Gt({{0,0},{2,1/6+1/6*mTau2}},1/2,mTau2))-893/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{2,1/6+1/6*mTau2}},1/2*mTau2,mTau2))+1271/4*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/6*mTau2}},1/2,mTau2))+Gt({{0,0},{2,2/3+1/6*mTau2}},1/2,mTau2))-1271/4*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/6*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{2,2/3+1/6*mTau2}},1/2+1/2*mTau2,mTau2))+8141/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{2,2/3+1/6*mTau2}},1/2*mTau2,mTau2))+420*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/6*mTau2}},2/3,mTau2))+Gt({{0,0},{2,5/6+1/6*mTau2}},2/3,mTau2))+1008/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/6*mTau2}},5/6,mTau2))+Gt({{0,0},{2,5/6+1/6*mTau2}},5/6,mTau2))-231*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/6*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{2,5/6+1/6*mTau2}},1/2+1/2*mTau2,mTau2))+288/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/6*mTau2}},1/6*mTau2,mTau2))+Gt({{0,0},{2,5/6+1/6*mTau2}},1/6*mTau2,mTau2))+1557/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{2,5/6+1/6*mTau2}},1/2*mTau2,mTau2))-280*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/3*mTau2}},1/2,mTau2))+Gt({{0,0},{2,1/6+1/3*mTau2}},1/2,mTau2))+84/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/3*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{2,1/6+1/3*mTau2}},1/2+1/2*mTau2,mTau2))-57739/48*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/3*mTau2}},1/6*mTau2,mTau2))+Gt({{0,0},{2,1/6+1/3*mTau2}},1/6*mTau2,mTau2))+36031/240*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/3*mTau2}},5/6*mTau2,mTau2))+Gt({{0,0},{2,1/6+1/3*mTau2}},5/6*mTau2,mTau2))+386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/3*mTau2}},1/3,mTau2))+Gt({{0,0},{2,2/3+1/3*mTau2}},1/3,mTau2))-665*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/3*mTau2}},1/2,mTau2))+Gt({{0,0},{2,2/3+1/3*mTau2}},1/2,mTau2))-386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/3*mTau2}},2/3,mTau2))+Gt({{0,0},{2,2/3+1/3*mTau2}},2/3,mTau2))+386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/3*mTau2}},1/3+1/3*mTau2,mTau2))+Gt({{0,0},{2,2/3+1/3*mTau2}},1/3+1/3*mTau2,mTau2))-386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/3*mTau2}},1/3*mTau2,mTau2))+Gt({{0,0},{2,2/3+1/3*mTau2}},1/3*mTau2,mTau2))+1127/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/3*mTau2}},1/2,mTau2))+Gt({{0,0},{2,5/6+1/3*mTau2}},1/2,mTau2))+3108/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/3*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{2,5/6+1/3*mTau2}},1/2+1/2*mTau2,mTau2))-1045553/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/3*mTau2}},1/3*mTau2,mTau2))+Gt({{0,0},{2,5/6+1/3*mTau2}},1/3*mTau2,mTau2))+231407/45*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/3*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{2,5/6+1/3*mTau2}},1/2*mTau2,mTau2))+22867/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/2*mTau2}},1/2,mTau2))+Gt({{0,0},{2,1/6+1/2*mTau2}},1/2,mTau2))-6207/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/2*mTau2}},2/3+1/2*mTau2,mTau2))+Gt({{0,0},{2,1/6+1/2*mTau2}},2/3+1/2*mTau2,mTau2))-72913/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3-1/2*mTau2}},1/2,mTau2))+Gt({{0,0},{2,1/3+1/2*mTau2}},1/2,mTau2))+4443/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3-1/2*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{2,1/3+1/2*mTau2}},1/2+1/2*mTau2,mTau2))-188113/1800*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau2}},1/3,mTau2))+Gt({{0,0},{2,1/2+1/2*mTau2}},1/3,mTau2))+1582769/1800*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau2}},2/3,mTau2))+Gt({{0,0},{2,1/2+1/2*mTau2}},2/3,mTau2))-4665/4*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau2}},5/6,mTau2))+Gt({{0,0},{2,1/2+1/2*mTau2}},5/6,mTau2))+315*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau2}},1/2+1/6*mTau2,mTau2))+Gt({{0,0},{2,1/2+1/2*mTau2}},1/2+1/6*mTau2,mTau2))+526*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau2}},1/2+1/3*mTau2,mTau2))+Gt({{0,0},{2,1/2+1/2*mTau2}},1/2+1/3*mTau2,mTau2))-1367/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau2}},5/6*mTau2,mTau2))+Gt({{0,0},{2,1/2+1/2*mTau2}},5/6*mTau2,mTau2))-31277/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/2*mTau2}},1/2,mTau2))+Gt({{0,0},{2,2/3+1/2*mTau2}},1/2,mTau2))-882/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/2*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{2,2/3+1/2*mTau2}},1/2+1/2*mTau2,mTau2))-9379/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/2*mTau2}},1/2,mTau2))+Gt({{0,0},{2,5/6+1/2*mTau2}},1/2,mTau2))-756/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/2*mTau2}},2/3,mTau2))+Gt({{0,0},{2,5/6+1/2*mTau2}},2/3,mTau2))+756/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/2*mTau2}},5/6,mTau2))+Gt({{0,0},{2,5/6+1/2*mTau2}},5/6,mTau2))+6207/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/2*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{2,5/6+1/2*mTau2}},1/2*mTau2,mTau2))+7343/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-2/3*mTau2}},1/2,mTau2))+Gt({{0,0},{2,1/6+2/3*mTau2}},1/2,mTau2))-4281/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-2/3*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{2,1/6+2/3*mTau2}},1/2*mTau2,mTau2))+2541/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-2/3*mTau2}},2/3*mTau2,mTau2))+Gt({{0,0},{2,1/6+2/3*mTau2}},2/3*mTau2,mTau2))+665/2*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3-2/3*mTau2}},1/2,mTau2))+Gt({{0,0},{2,1/3+2/3*mTau2}},1/2,mTau2))+231/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-2/3*mTau2}},1/2,mTau2))+Gt({{0,0},{2,1/2+2/3*mTau2}},1/2,mTau2))-386267/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-2/3*mTau2}},1/3,mTau2))+Gt({{0,0},{2,2/3+2/3*mTau2}},1/3,mTau2))+386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-2/3*mTau2}},2/3,mTau2))+Gt({{0,0},{2,2/3+2/3*mTau2}},2/3,mTau2))+669/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau2}},1/2,mTau2))+Gt({{0,0},{2,5/6+2/3*mTau2}},1/2,mTau2))+144/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau2}},2/3,mTau2))+Gt({{0,0},{2,5/6+2/3*mTau2}},2/3,mTau2))+3108/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau2}},1/6+1/6*mTau2,mTau2))+Gt({{0,0},{2,5/6+2/3*mTau2}},1/6+1/6*mTau2,mTau2))+84/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau2}},2/3+5/6*mTau2,mTau2))+Gt({{0,0},{2,5/6+2/3*mTau2}},2/3+5/6*mTau2,mTau2))-64911/80*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau2}},1/6*mTau2,mTau2))+Gt({{0,0},{2,5/6+2/3*mTau2}},1/6*mTau2,mTau2))-4737/80*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau2}},5/6*mTau2,mTau2))+Gt({{0,0},{2,5/6+2/3*mTau2}},5/6*mTau2,mTau2))-1579/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-5/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{2,1/6+5/6*mTau2}},1/2*mTau2,mTau2))-165/4*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-5/6*mTau2}},2/3*mTau2,mTau2))+Gt({{0,0},{2,1/6+5/6*mTau2}},2/3*mTau2,mTau2))+893/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3-5/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{2,1/3+5/6*mTau2}},1/2*mTau2,mTau2))-63*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-5/6*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{2,1/2+5/6*mTau2}},1/2+1/2*mTau2,mTau2))+84/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-5/6*mTau2}},1/2+2/3*mTau2,mTau2))+Gt({{0,0},{2,1/2+5/6*mTau2}},1/2+2/3*mTau2,mTau2))+203/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-5/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{2,1/2+5/6*mTau2}},1/2*mTau2,mTau2))-3333/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-5/6*mTau2}},1/2,mTau2))+Gt({{0,0},{2,2/3+5/6*mTau2}},1/2,mTau2))-41/2*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-5/6*mTau2}},1/2+1/3*mTau2,mTau2))+Gt({{0,0},{2,2/3+5/6*mTau2}},1/2+1/3*mTau2,mTau2))+221/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-5/6*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{0,0},{2,2/3+5/6*mTau2}},1/2+1/2*mTau2,mTau2))-253/30*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-5/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{2,2/3+5/6*mTau2}},1/2*mTau2,mTau2))-37841/450*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau2}},1/3,mTau2))+Gt({{0,0},{2,5/6+5/6*mTau2}},1/3,mTau2))+321817/450*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau2}},1/2,mTau2))+Gt({{0,0},{2,5/6+5/6*mTau2}},1/2,mTau2))-136121/450*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau2}},2/3,mTau2))+Gt({{0,0},{2,5/6+5/6*mTau2}},2/3,mTau2))+12277/50*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau2}},5/6,mTau2))+Gt({{0,0},{2,5/6+5/6*mTau2}},5/6,mTau2))-208783/24*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau2}},1/6*mTau2,mTau2))+Gt({{0,0},{2,5/6+5/6*mTau2}},1/6*mTau2,mTau2))+1078601/360*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau2}},1/2*mTau2,mTau2))+Gt({{0,0},{2,5/6+5/6*mTau2}},1/2*mTau2,mTau2))-423/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau2}},5/6*mTau2,mTau2))+Gt({{0,0},{2,5/6+5/6*mTau2}},5/6*mTau2,mTau2))-13319/16*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6*mTau2}},1/2,mTau2))+Gt({{0,0},{2,1/6*mTau2}},1/2,mTau2))+8037/80*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6*mTau2}},5/6+2/3*mTau2,mTau2))+Gt({{0,0},{2,1/6*mTau2}},5/6+2/3*mTau2,mTau2))-231/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3*mTau2}},1/2,mTau2))+Gt({{0,0},{2,1/3*mTau2}},1/2,mTau2))-755827/1200*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau2}},5/6,mTau2))+Gt({{0,0},{2,1/2*mTau2}},5/6,mTau2))-22841/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau2}},2/3+1/6*mTau2,mTau2))+Gt({{0,0},{2,1/2*mTau2}},2/3+1/6*mTau2,mTau2))-1371547/360*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau2}},5/6+1/3*mTau2,mTau2))+Gt({{0,0},{2,1/2*mTau2}},5/6+1/3*mTau2,mTau2))+969431/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau2}},5/6+1/2*mTau2,mTau2))+Gt({{0,0},{2,1/2*mTau2}},5/6+1/2*mTau2,mTau2))+20261/240*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau2}},1/6+2/3*mTau2,mTau2))+Gt({{0,0},{2,1/2*mTau2}},1/6+2/3*mTau2,mTau2))+577/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau2}},1/3+5/6*mTau2,mTau2))+Gt({{0,0},{2,1/2*mTau2}},1/3+5/6*mTau2,mTau2))+1639/30*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau2}},2/3+5/6*mTau2,mTau2))+Gt({{0,0},{2,1/2*mTau2}},2/3+5/6*mTau2,mTau2))-1011209/360*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau2}},5/6+5/6*mTau2,mTau2))+Gt({{0,0},{2,1/2*mTau2}},5/6+5/6*mTau2,mTau2))-5822/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau2}},1/3*mTau2,mTau2))+Gt({{0,0},{2,1/2*mTau2}},1/3*mTau2,mTau2))+231/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3*mTau2}},1/2,mTau2))+Gt({{0,0},{2,2/3*mTau2}},1/2,mTau2))-70291/80*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6*mTau2}},1/2,mTau2))+Gt({{0,0},{2,5/6*mTau2}},1/2,mTau2))+8037/80*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6*mTau2}},5/6+2/3*mTau2,mTau2))+Gt({{0,0},{2,5/6*mTau2}},5/6+2/3*mTau2,mTau2))",
		"161.2681146783657633123567891341936452158363749498713077149574430263258311059399904471332277791136192321234729378945194-75.397689707044340898088338207529508917496977089736006842308618879712026963207096810379266912390326942869809822673558*I");
	test(6,"2197/300*(pow(Pi,-2))*(-(Gt({{3,-1/6}},5/6+5/6*mTau2,mTau2))+Gt({{3,1/6}},5/6+5/6*mTau2,mTau2))-10409/90*(pow(Pi,-2))*(-(Gt({{3,-1/6}},1/2*mTau2,mTau2))+Gt({{3,1/6}},1/2*mTau2,mTau2))-2197/40*(pow(Pi,-2))*(-(Gt({{3,-1/3}},1/6,mTau2))+Gt({{3,1/3}},1/6,mTau2))-386267/360*(pow(Pi,-2))*(-(Gt({{3,-1/3}},2/3+1/3*mTau2,mTau2))+Gt({{3,1/3}},2/3+1/3*mTau2,mTau2))+386267/360*(pow(Pi,-2))*(-(Gt({{3,-1/3}},2/3+2/3*mTau2,mTau2))+Gt({{3,1/3}},2/3+2/3*mTau2,mTau2))+6207/40*(pow(Pi,-2))*(-(Gt({{3,-1/3}},1/2*mTau2,mTau2))+Gt({{3,1/3}},1/2*mTau2,mTau2))-386267/720*(pow(Pi,-2))*(-(Gt({{3,-1/3}},2/3*mTau2,mTau2))+Gt({{3,1/3}},2/3*mTau2,mTau2))+665/6*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/6*mTau2}},1/2,mTau2))+Gt({{3,1/6+1/6*mTau2}},1/2,mTau2))-893/120*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/6*mTau2}},1/2*mTau2,mTau2))+Gt({{3,1/6+1/6*mTau2}},1/2*mTau2,mTau2))+1271/24*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/6*mTau2}},1/2,mTau2))+Gt({{3,2/3+1/6*mTau2}},1/2,mTau2))-1271/24*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/6*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{3,2/3+1/6*mTau2}},1/2+1/2*mTau2,mTau2))+8141/120*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/6*mTau2}},1/2*mTau2,mTau2))+Gt({{3,2/3+1/6*mTau2}},1/2*mTau2,mTau2))+70*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/6*mTau2}},2/3,mTau2))+Gt({{3,5/6+1/6*mTau2}},2/3,mTau2))+168/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/6*mTau2}},5/6,mTau2))+Gt({{3,5/6+1/6*mTau2}},5/6,mTau2))-77/2*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/6*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{3,5/6+1/6*mTau2}},1/2+1/2*mTau2,mTau2))+48/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/6*mTau2}},1/6*mTau2,mTau2))+Gt({{3,5/6+1/6*mTau2}},1/6*mTau2,mTau2))+519/40*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/6*mTau2}},1/2*mTau2,mTau2))+Gt({{3,5/6+1/6*mTau2}},1/2*mTau2,mTau2))-140/3*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/3*mTau2}},1/2,mTau2))+Gt({{3,1/6+1/3*mTau2}},1/2,mTau2))+14/5*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/3*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{3,1/6+1/3*mTau2}},1/2+1/2*mTau2,mTau2))-57739/288*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/3*mTau2}},1/6*mTau2,mTau2))+Gt({{3,1/6+1/3*mTau2}},1/6*mTau2,mTau2))+36031/1440*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/3*mTau2}},5/6*mTau2,mTau2))+Gt({{3,1/6+1/3*mTau2}},5/6*mTau2,mTau2))+386267/720*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/3*mTau2}},1/3,mTau2))+Gt({{3,2/3+1/3*mTau2}},1/3,mTau2))-665/6*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/3*mTau2}},1/2,mTau2))+Gt({{3,2/3+1/3*mTau2}},1/2,mTau2))-386267/720*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/3*mTau2}},2/3,mTau2))+Gt({{3,2/3+1/3*mTau2}},2/3,mTau2))+386267/720*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/3*mTau2}},1/3+1/3*mTau2,mTau2))+Gt({{3,2/3+1/3*mTau2}},1/3+1/3*mTau2,mTau2))-386267/720*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/3*mTau2}},1/3*mTau2,mTau2))+Gt({{3,2/3+1/3*mTau2}},1/3*mTau2,mTau2))+1127/60*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/3*mTau2}},1/2,mTau2))+Gt({{3,5/6+1/3*mTau2}},1/2,mTau2))+518/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/3*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{3,5/6+1/3*mTau2}},1/2+1/2*mTau2,mTau2))-1045553/720*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/3*mTau2}},1/3*mTau2,mTau2))+Gt({{3,5/6+1/3*mTau2}},1/3*mTau2,mTau2))+231407/270*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/3*mTau2}},1/2*mTau2,mTau2))+Gt({{3,5/6+1/3*mTau2}},1/2*mTau2,mTau2))+22867/360*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/2*mTau2}},1/2,mTau2))+Gt({{3,1/6+1/2*mTau2}},1/2,mTau2))-2069/40*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/2*mTau2}},2/3+1/2*mTau2,mTau2))+Gt({{3,1/6+1/2*mTau2}},2/3+1/2*mTau2,mTau2))-72913/360*(pow(Pi,-2))*(-(Gt({{3,-1/3-1/2*mTau2}},1/2,mTau2))+Gt({{3,1/3+1/2*mTau2}},1/2,mTau2))+1481/20*(pow(Pi,-2))*(-(Gt({{3,-1/3-1/2*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{3,1/3+1/2*mTau2}},1/2+1/2*mTau2,mTau2))-188113/10800*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau2}},1/3,mTau2))+Gt({{3,1/2+1/2*mTau2}},1/3,mTau2))+1582769/10800*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau2}},2/3,mTau2))+Gt({{3,1/2+1/2*mTau2}},2/3,mTau2))-1555/8*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau2}},5/6,mTau2))+Gt({{3,1/2+1/2*mTau2}},5/6,mTau2))+105/2*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau2}},1/2+1/6*mTau2,mTau2))+Gt({{3,1/2+1/2*mTau2}},1/2+1/6*mTau2,mTau2))+263/3*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau2}},1/2+1/3*mTau2,mTau2))+Gt({{3,1/2+1/2*mTau2}},1/2+1/3*mTau2,mTau2))-1367/30*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau2}},5/6*mTau2,mTau2))+Gt({{3,1/2+1/2*mTau2}},5/6*mTau2,mTau2))-31277/360*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/2*mTau2}},1/2,mTau2))+Gt({{3,2/3+1/2*mTau2}},1/2,mTau2))-147/5*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/2*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{3,2/3+1/2*mTau2}},1/2+1/2*mTau2,mTau2))-9379/360*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/2*mTau2}},1/2,mTau2))+Gt({{3,5/6+1/2*mTau2}},1/2,mTau2))-126/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/2*mTau2}},2/3,mTau2))+Gt({{3,5/6+1/2*mTau2}},2/3,mTau2))+126/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/2*mTau2}},5/6,mTau2))+Gt({{3,5/6+1/2*mTau2}},5/6,mTau2))+2069/40*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/2*mTau2}},1/2*mTau2,mTau2))+Gt({{3,5/6+1/2*mTau2}},1/2*mTau2,mTau2))+7343/60*(pow(Pi,-2))*(-(Gt({{3,-1/6-2/3*mTau2}},1/2,mTau2))+Gt({{3,1/6+2/3*mTau2}},1/2,mTau2))-1427/40*(pow(Pi,-2))*(-(Gt({{3,-1/6-2/3*mTau2}},1/2*mTau2,mTau2))+Gt({{3,1/6+2/3*mTau2}},1/2*mTau2,mTau2))+847/40*(pow(Pi,-2))*(-(Gt({{3,-1/6-2/3*mTau2}},2/3*mTau2,mTau2))+Gt({{3,1/6+2/3*mTau2}},2/3*mTau2,mTau2))+665/12*(pow(Pi,-2))*(-(Gt({{3,-1/3-2/3*mTau2}},1/2,mTau2))+Gt({{3,1/3+2/3*mTau2}},1/2,mTau2))+77/10*(pow(Pi,-2))*(-(Gt({{3,-1/2-2/3*mTau2}},1/2,mTau2))+Gt({{3,1/2+2/3*mTau2}},1/2,mTau2))-386267/360*(pow(Pi,-2))*(-(Gt({{3,-2/3-2/3*mTau2}},1/3,mTau2))+Gt({{3,2/3+2/3*mTau2}},1/3,mTau2))+386267/720*(pow(Pi,-2))*(-(Gt({{3,-2/3-2/3*mTau2}},2/3,mTau2))+Gt({{3,2/3+2/3*mTau2}},2/3,mTau2))+223/10*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau2}},1/2,mTau2))+Gt({{3,5/6+2/3*mTau2}},1/2,mTau2))+24/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau2}},2/3,mTau2))+Gt({{3,5/6+2/3*mTau2}},2/3,mTau2))+518/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau2}},1/6+1/6*mTau2,mTau2))+Gt({{3,5/6+2/3*mTau2}},1/6+1/6*mTau2,mTau2))+14/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau2}},2/3+5/6*mTau2,mTau2))+Gt({{3,5/6+2/3*mTau2}},2/3+5/6*mTau2,mTau2))-21637/160*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau2}},1/6*mTau2,mTau2))+Gt({{3,5/6+2/3*mTau2}},1/6*mTau2,mTau2))-1579/160*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau2}},5/6*mTau2,mTau2))+Gt({{3,5/6+2/3*mTau2}},5/6*mTau2,mTau2))-1579/120*(pow(Pi,-2))*(-(Gt({{3,-1/6-5/6*mTau2}},1/2*mTau2,mTau2))+Gt({{3,1/6+5/6*mTau2}},1/2*mTau2,mTau2))-55/8*(pow(Pi,-2))*(-(Gt({{3,-1/6-5/6*mTau2}},2/3*mTau2,mTau2))+Gt({{3,1/6+5/6*mTau2}},2/3*mTau2,mTau2))+893/60*(pow(Pi,-2))*(-(Gt({{3,-1/3-5/6*mTau2}},1/2*mTau2,mTau2))+Gt({{3,1/3+5/6*mTau2}},1/2*mTau2,mTau2))-21/2*(pow(Pi,-2))*(-(Gt({{3,-1/2-5/6*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{3,1/2+5/6*mTau2}},1/2+1/2*mTau2,mTau2))+14/5*(pow(Pi,-2))*(-(Gt({{3,-1/2-5/6*mTau2}},1/2+2/3*mTau2,mTau2))+Gt({{3,1/2+5/6*mTau2}},1/2+2/3*mTau2,mTau2))+203/30*(pow(Pi,-2))*(-(Gt({{3,-1/2-5/6*mTau2}},1/2*mTau2,mTau2))+Gt({{3,1/2+5/6*mTau2}},1/2*mTau2,mTau2))-1111/10*(pow(Pi,-2))*(-(Gt({{3,-2/3-5/6*mTau2}},1/2,mTau2))+Gt({{3,2/3+5/6*mTau2}},1/2,mTau2))-41/12*(pow(Pi,-2))*(-(Gt({{3,-2/3-5/6*mTau2}},1/2+1/3*mTau2,mTau2))+Gt({{3,2/3+5/6*mTau2}},1/2+1/3*mTau2,mTau2))+221/60*(pow(Pi,-2))*(-(Gt({{3,-2/3-5/6*mTau2}},1/2+1/2*mTau2,mTau2))+Gt({{3,2/3+5/6*mTau2}},1/2+1/2*mTau2,mTau2))-253/180*(pow(Pi,-2))*(-(Gt({{3,-2/3-5/6*mTau2}},1/2*mTau2,mTau2))+Gt({{3,2/3+5/6*mTau2}},1/2*mTau2,mTau2))-37841/2700*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau2}},1/3,mTau2))+Gt({{3,5/6+5/6*mTau2}},1/3,mTau2))+321817/2700*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau2}},1/2,mTau2))+Gt({{3,5/6+5/6*mTau2}},1/2,mTau2))-136121/2700*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau2}},2/3,mTau2))+Gt({{3,5/6+5/6*mTau2}},2/3,mTau2))+12277/300*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau2}},5/6,mTau2))+Gt({{3,5/6+5/6*mTau2}},5/6,mTau2))-208783/144*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau2}},1/6*mTau2,mTau2))+Gt({{3,5/6+5/6*mTau2}},1/6*mTau2,mTau2))+1078601/2160*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau2}},1/2*mTau2,mTau2))+Gt({{3,5/6+5/6*mTau2}},1/2*mTau2,mTau2))-141/10*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau2}},5/6*mTau2,mTau2))+Gt({{3,5/6+5/6*mTau2}},5/6*mTau2,mTau2))-13319/96*(pow(Pi,-2))*(-(Gt({{3,-1/6*mTau2}},1/2,mTau2))+Gt({{3,1/6*mTau2}},1/2,mTau2))+2679/160*(pow(Pi,-2))*(-(Gt({{3,-1/6*mTau2}},5/6+2/3*mTau2,mTau2))+Gt({{3,1/6*mTau2}},5/6+2/3*mTau2,mTau2))-77/10*(pow(Pi,-2))*(-(Gt({{3,-1/3*mTau2}},1/2,mTau2))+Gt({{3,1/3*mTau2}},1/2,mTau2))-755827/7200*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau2}},5/6,mTau2))+Gt({{3,1/2*mTau2}},5/6,mTau2))-22841/120*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau2}},2/3+1/6*mTau2,mTau2))+Gt({{3,1/2*mTau2}},2/3+1/6*mTau2,mTau2))-1371547/2160*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau2}},5/6+1/3*mTau2,mTau2))+Gt({{3,1/2*mTau2}},5/6+1/3*mTau2,mTau2))+969431/720*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau2}},5/6+1/2*mTau2,mTau2))+Gt({{3,1/2*mTau2}},5/6+1/2*mTau2,mTau2))+20261/1440*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau2}},1/6+2/3*mTau2,mTau2))+Gt({{3,1/2*mTau2}},1/6+2/3*mTau2,mTau2))+577/60*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau2}},1/3+5/6*mTau2,mTau2))+Gt({{3,1/2*mTau2}},1/3+5/6*mTau2,mTau2))+1639/180*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau2}},2/3+5/6*mTau2,mTau2))+Gt({{3,1/2*mTau2}},2/3+5/6*mTau2,mTau2))-1011209/2160*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau2}},5/6+5/6*mTau2,mTau2))+Gt({{3,1/2*mTau2}},5/6+5/6*mTau2,mTau2))-2911/15*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau2}},1/3*mTau2,mTau2))+Gt({{3,1/2*mTau2}},1/3*mTau2,mTau2))+77/20*(pow(Pi,-2))*(-(Gt({{3,-2/3*mTau2}},1/2,mTau2))+Gt({{3,2/3*mTau2}},1/2,mTau2))-70291/480*(pow(Pi,-2))*(-(Gt({{3,-5/6*mTau2}},1/2,mTau2))+Gt({{3,5/6*mTau2}},1/2,mTau2))+2679/160*(pow(Pi,-2))*(-(Gt({{3,-5/6*mTau2}},5/6+2/3*mTau2,mTau2))+Gt({{3,5/6*mTau2}},5/6+2/3*mTau2,mTau2))",
		"37.6994345499473993723993405197402645983806487830044068389581324734880470771367570884253497446296767233220812628101391-8.3781396701486908058483975342436320242607431292104996791368633596419181043750126741870181861679272229863116614548161*I");
	test(7,"2197/25*(-(Gt({{0,0},{0,0},{1,-1/6}},5/6+5/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/6}},5/6+5/6*mTau3,mTau3))-20818/15*(-(Gt({{0,0},{0,0},{1,-1/6}},1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/6}},1/2*mTau3,mTau3))-6591/10*(-(Gt({{0,0},{0,0},{1,-1/3}},1/6,mTau3))+Gt({{0,0},{0,0},{1,1/3}},1/6,mTau3))-386267/30*(-(Gt({{0,0},{0,0},{1,-1/3}},2/3+1/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/3}},2/3+1/3*mTau3,mTau3))+386267/30*(-(Gt({{0,0},{0,0},{1,-1/3}},2/3+2/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/3}},2/3+2/3*mTau3,mTau3))+18621/10*(-(Gt({{0,0},{0,0},{1,-1/3}},1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/3}},1/2*mTau3,mTau3))-386267/60*(-(Gt({{0,0},{0,0},{1,-1/3}},2/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/3}},2/3*mTau3,mTau3))+1330*(-(Gt({{0,0},{0,0},{1,-1/6-1/6*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,1/6+1/6*mTau3}},1/2,mTau3))-893/10*(-(Gt({{0,0},{0,0},{1,-1/6-1/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/6+1/6*mTau3}},1/2*mTau3,mTau3))+1271/2*(-(Gt({{0,0},{0,0},{1,-2/3-1/6*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,2/3+1/6*mTau3}},1/2,mTau3))-1271/2*(-(Gt({{0,0},{0,0},{1,-2/3-1/6*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,2/3+1/6*mTau3}},1/2+1/2*mTau3,mTau3))+8141/10*(-(Gt({{0,0},{0,0},{1,-2/3-1/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,2/3+1/6*mTau3}},1/2*mTau3,mTau3))+840*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau3}},2/3,mTau3))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau3}},2/3,mTau3))+2016/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau3}},5/6,mTau3))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau3}},5/6,mTau3))-462*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau3}},1/2+1/2*mTau3,mTau3))+576/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau3}},1/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau3}},1/6*mTau3,mTau3))+1557/10*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau3}},1/2*mTau3,mTau3))-560*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau3}},1/2,mTau3))+168/5*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau3}},1/2+1/2*mTau3,mTau3))-57739/24*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau3}},1/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau3}},1/6*mTau3,mTau3))+36031/120*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau3}},5/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau3}},5/6*mTau3,mTau3))+386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau3}},1/3,mTau3))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau3}},1/3,mTau3))-1330*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau3}},1/2,mTau3))-386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau3}},2/3,mTau3))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau3}},2/3,mTau3))+386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau3}},1/3+1/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau3}},1/3+1/3*mTau3,mTau3))-386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau3}},1/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau3}},1/3*mTau3,mTau3))+1127/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau3}},1/2,mTau3))+6216/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau3}},1/2+1/2*mTau3,mTau3))-1045553/60*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau3}},1/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau3}},1/3*mTau3,mTau3))+462814/45*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau3}},1/2*mTau3,mTau3))+22867/30*(-(Gt({{0,0},{0,0},{1,-1/6-1/2*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,1/6+1/2*mTau3}},1/2,mTau3))-6207/10*(-(Gt({{0,0},{0,0},{1,-1/6-1/2*mTau3}},2/3+1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/6+1/2*mTau3}},2/3+1/2*mTau3,mTau3))-72913/30*(-(Gt({{0,0},{0,0},{1,-1/3-1/2*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,1/3+1/2*mTau3}},1/2,mTau3))+4443/5*(-(Gt({{0,0},{0,0},{1,-1/3-1/2*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/3+1/2*mTau3}},1/2+1/2*mTau3,mTau3))-188113/900*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau3}},1/3,mTau3))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau3}},1/3,mTau3))+1582769/900*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau3}},2/3,mTau3))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau3}},2/3,mTau3))-4665/2*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau3}},5/6,mTau3))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau3}},5/6,mTau3))+630*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau3}},1/2+1/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau3}},1/2+1/6*mTau3,mTau3))+1052*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau3}},1/2+1/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau3}},1/2+1/3*mTau3,mTau3))-2734/5*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau3}},5/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau3}},5/6*mTau3,mTau3))-31277/30*(-(Gt({{0,0},{0,0},{1,-2/3-1/2*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,2/3+1/2*mTau3}},1/2,mTau3))-1764/5*(-(Gt({{0,0},{0,0},{1,-2/3-1/2*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,2/3+1/2*mTau3}},1/2+1/2*mTau3,mTau3))-9379/30*(-(Gt({{0,0},{0,0},{1,-5/6-1/2*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,5/6+1/2*mTau3}},1/2,mTau3))-1512/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/2*mTau3}},2/3,mTau3))+Gt({{0,0},{0,0},{1,5/6+1/2*mTau3}},2/3,mTau3))+1512/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/2*mTau3}},5/6,mTau3))+Gt({{0,0},{0,0},{1,5/6+1/2*mTau3}},5/6,mTau3))+6207/10*(-(Gt({{0,0},{0,0},{1,-5/6-1/2*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,5/6+1/2*mTau3}},1/2*mTau3,mTau3))+7343/5*(-(Gt({{0,0},{0,0},{1,-1/6-2/3*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,1/6+2/3*mTau3}},1/2,mTau3))-4281/10*(-(Gt({{0,0},{0,0},{1,-1/6-2/3*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/6+2/3*mTau3}},1/2*mTau3,mTau3))+2541/10*(-(Gt({{0,0},{0,0},{1,-1/6-2/3*mTau3}},2/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/6+2/3*mTau3}},2/3*mTau3,mTau3))+665*(-(Gt({{0,0},{0,0},{1,-1/3-2/3*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,1/3+2/3*mTau3}},1/2,mTau3))+462/5*(-(Gt({{0,0},{0,0},{1,-1/2-2/3*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,1/2+2/3*mTau3}},1/2,mTau3))-386267/30*(-(Gt({{0,0},{0,0},{1,-2/3-2/3*mTau3}},1/3,mTau3))+Gt({{0,0},{0,0},{1,2/3+2/3*mTau3}},1/3,mTau3))+386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-2/3*mTau3}},2/3,mTau3))+Gt({{0,0},{0,0},{1,2/3+2/3*mTau3}},2/3,mTau3))+1338/5*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau3}},1/2,mTau3))+288/5*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau3}},2/3,mTau3))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau3}},2/3,mTau3))+6216/5*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau3}},1/6+1/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau3}},1/6+1/6*mTau3,mTau3))+168/5*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau3}},2/3+5/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau3}},2/3+5/6*mTau3,mTau3))-64911/40*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau3}},1/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau3}},1/6*mTau3,mTau3))-4737/40*(-(Gt({{0,0},{0,0},{1,-5/6-2/3*mTau3}},5/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,5/6+2/3*mTau3}},5/6*mTau3,mTau3))-1579/10*(-(Gt({{0,0},{0,0},{1,-1/6-5/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/6+5/6*mTau3}},1/2*mTau3,mTau3))-165/2*(-(Gt({{0,0},{0,0},{1,-1/6-5/6*mTau3}},2/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/6+5/6*mTau3}},2/3*mTau3,mTau3))+893/5*(-(Gt({{0,0},{0,0},{1,-1/3-5/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/3+5/6*mTau3}},1/2*mTau3,mTau3))-126*(-(Gt({{0,0},{0,0},{1,-1/2-5/6*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/2+5/6*mTau3}},1/2+1/2*mTau3,mTau3))+168/5*(-(Gt({{0,0},{0,0},{1,-1/2-5/6*mTau3}},1/2+2/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/2+5/6*mTau3}},1/2+2/3*mTau3,mTau3))+406/5*(-(Gt({{0,0},{0,0},{1,-1/2-5/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/2+5/6*mTau3}},1/2*mTau3,mTau3))-6666/5*(-(Gt({{0,0},{0,0},{1,-2/3-5/6*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,2/3+5/6*mTau3}},1/2,mTau3))-41*(-(Gt({{0,0},{0,0},{1,-2/3-5/6*mTau3}},1/2+1/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,2/3+5/6*mTau3}},1/2+1/3*mTau3,mTau3))+221/5*(-(Gt({{0,0},{0,0},{1,-2/3-5/6*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,2/3+5/6*mTau3}},1/2+1/2*mTau3,mTau3))-253/15*(-(Gt({{0,0},{0,0},{1,-2/3-5/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,2/3+5/6*mTau3}},1/2*mTau3,mTau3))-37841/225*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau3}},1/3,mTau3))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau3}},1/3,mTau3))+321817/225*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau3}},1/2,mTau3))-136121/225*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau3}},2/3,mTau3))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau3}},2/3,mTau3))+12277/25*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau3}},5/6,mTau3))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau3}},5/6,mTau3))-208783/12*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau3}},1/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau3}},1/6*mTau3,mTau3))+1078601/180*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau3}},1/2*mTau3,mTau3))-846/5*(-(Gt({{0,0},{0,0},{1,-5/6-5/6*mTau3}},5/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,5/6+5/6*mTau3}},5/6*mTau3,mTau3))-13319/8*(-(Gt({{0,0},{0,0},{1,-1/6*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,1/6*mTau3}},1/2,mTau3))+8037/40*(-(Gt({{0,0},{0,0},{1,-1/6*mTau3}},5/6+2/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/6*mTau3}},5/6+2/3*mTau3,mTau3))-462/5*(-(Gt({{0,0},{0,0},{1,-1/3*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,1/3*mTau3}},1/2,mTau3))-755827/600*(-(Gt({{0,0},{0,0},{1,-1/2*mTau3}},5/6,mTau3))+Gt({{0,0},{0,0},{1,1/2*mTau3}},5/6,mTau3))-22841/10*(-(Gt({{0,0},{0,0},{1,-1/2*mTau3}},2/3+1/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/2*mTau3}},2/3+1/6*mTau3,mTau3))-1371547/180*(-(Gt({{0,0},{0,0},{1,-1/2*mTau3}},5/6+1/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/2*mTau3}},5/6+1/3*mTau3,mTau3))+969431/60*(-(Gt({{0,0},{0,0},{1,-1/2*mTau3}},5/6+1/2*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/2*mTau3}},5/6+1/2*mTau3,mTau3))+20261/120*(-(Gt({{0,0},{0,0},{1,-1/2*mTau3}},1/6+2/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/2*mTau3}},1/6+2/3*mTau3,mTau3))+577/5*(-(Gt({{0,0},{0,0},{1,-1/2*mTau3}},1/3+5/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/2*mTau3}},1/3+5/6*mTau3,mTau3))+1639/15*(-(Gt({{0,0},{0,0},{1,-1/2*mTau3}},2/3+5/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/2*mTau3}},2/3+5/6*mTau3,mTau3))-1011209/180*(-(Gt({{0,0},{0,0},{1,-1/2*mTau3}},5/6+5/6*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/2*mTau3}},5/6+5/6*mTau3,mTau3))-11644/5*(-(Gt({{0,0},{0,0},{1,-1/2*mTau3}},1/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,1/2*mTau3}},1/3*mTau3,mTau3))+231/5*(-(Gt({{0,0},{0,0},{1,-2/3*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,2/3*mTau3}},1/2,mTau3))-70291/40*(-(Gt({{0,0},{0,0},{1,-5/6*mTau3}},1/2,mTau3))+Gt({{0,0},{0,0},{1,5/6*mTau3}},1/2,mTau3))+8037/40*(-(Gt({{0,0},{0,0},{1,-5/6*mTau3}},5/6+2/3*mTau3,mTau3))+Gt({{0,0},{0,0},{1,5/6*mTau3}},5/6+2/3*mTau3,mTau3))",
		"32.4399013032879086369066796217648087627280771844848609732065788384798183963043337424976099790797297395721080849366384");
	test(8,"2197/50*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6}},5/6+5/6*mTau3,mTau3))+Gt({{0,0},{2,1/6}},5/6+5/6*mTau3,mTau3))-10409/15*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6}},1/2*mTau3,mTau3))+Gt({{0,0},{2,1/6}},1/2*mTau3,mTau3))-6591/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3}},1/6,mTau3))+Gt({{0,0},{2,1/3}},1/6,mTau3))-386267/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3}},2/3+1/3*mTau3,mTau3))+Gt({{0,0},{2,1/3}},2/3+1/3*mTau3,mTau3))+386267/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3}},2/3+2/3*mTau3,mTau3))+Gt({{0,0},{2,1/3}},2/3+2/3*mTau3,mTau3))+18621/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3}},1/2*mTau3,mTau3))+Gt({{0,0},{2,1/3}},1/2*mTau3,mTau3))-386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3}},2/3*mTau3,mTau3))+Gt({{0,0},{2,1/3}},2/3*mTau3,mTau3))+665*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/6*mTau3}},1/2,mTau3))+Gt({{0,0},{2,1/6+1/6*mTau3}},1/2,mTau3))-893/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{2,1/6+1/6*mTau3}},1/2*mTau3,mTau3))+1271/4*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/6*mTau3}},1/2,mTau3))+Gt({{0,0},{2,2/3+1/6*mTau3}},1/2,mTau3))-1271/4*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/6*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{2,2/3+1/6*mTau3}},1/2+1/2*mTau3,mTau3))+8141/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{2,2/3+1/6*mTau3}},1/2*mTau3,mTau3))+420*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/6*mTau3}},2/3,mTau3))+Gt({{0,0},{2,5/6+1/6*mTau3}},2/3,mTau3))+1008/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/6*mTau3}},5/6,mTau3))+Gt({{0,0},{2,5/6+1/6*mTau3}},5/6,mTau3))-231*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/6*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{2,5/6+1/6*mTau3}},1/2+1/2*mTau3,mTau3))+288/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/6*mTau3}},1/6*mTau3,mTau3))+Gt({{0,0},{2,5/6+1/6*mTau3}},1/6*mTau3,mTau3))+1557/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{2,5/6+1/6*mTau3}},1/2*mTau3,mTau3))-280*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/3*mTau3}},1/2,mTau3))+Gt({{0,0},{2,1/6+1/3*mTau3}},1/2,mTau3))+84/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/3*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{2,1/6+1/3*mTau3}},1/2+1/2*mTau3,mTau3))-57739/48*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/3*mTau3}},1/6*mTau3,mTau3))+Gt({{0,0},{2,1/6+1/3*mTau3}},1/6*mTau3,mTau3))+36031/240*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/3*mTau3}},5/6*mTau3,mTau3))+Gt({{0,0},{2,1/6+1/3*mTau3}},5/6*mTau3,mTau3))+386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/3*mTau3}},1/3,mTau3))+Gt({{0,0},{2,2/3+1/3*mTau3}},1/3,mTau3))-665*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/3*mTau3}},1/2,mTau3))+Gt({{0,0},{2,2/3+1/3*mTau3}},1/2,mTau3))-386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/3*mTau3}},2/3,mTau3))+Gt({{0,0},{2,2/3+1/3*mTau3}},2/3,mTau3))+386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/3*mTau3}},1/3+1/3*mTau3,mTau3))+Gt({{0,0},{2,2/3+1/3*mTau3}},1/3+1/3*mTau3,mTau3))-386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/3*mTau3}},1/3*mTau3,mTau3))+Gt({{0,0},{2,2/3+1/3*mTau3}},1/3*mTau3,mTau3))+1127/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/3*mTau3}},1/2,mTau3))+Gt({{0,0},{2,5/6+1/3*mTau3}},1/2,mTau3))+3108/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/3*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{2,5/6+1/3*mTau3}},1/2+1/2*mTau3,mTau3))-1045553/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/3*mTau3}},1/3*mTau3,mTau3))+Gt({{0,0},{2,5/6+1/3*mTau3}},1/3*mTau3,mTau3))+231407/45*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/3*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{2,5/6+1/3*mTau3}},1/2*mTau3,mTau3))+22867/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/2*mTau3}},1/2,mTau3))+Gt({{0,0},{2,1/6+1/2*mTau3}},1/2,mTau3))-6207/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-1/2*mTau3}},2/3+1/2*mTau3,mTau3))+Gt({{0,0},{2,1/6+1/2*mTau3}},2/3+1/2*mTau3,mTau3))-72913/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3-1/2*mTau3}},1/2,mTau3))+Gt({{0,0},{2,1/3+1/2*mTau3}},1/2,mTau3))+4443/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3-1/2*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{2,1/3+1/2*mTau3}},1/2+1/2*mTau3,mTau3))-188113/1800*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau3}},1/3,mTau3))+Gt({{0,0},{2,1/2+1/2*mTau3}},1/3,mTau3))+1582769/1800*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau3}},2/3,mTau3))+Gt({{0,0},{2,1/2+1/2*mTau3}},2/3,mTau3))-4665/4*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau3}},5/6,mTau3))+Gt({{0,0},{2,1/2+1/2*mTau3}},5/6,mTau3))+315*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau3}},1/2+1/6*mTau3,mTau3))+Gt({{0,0},{2,1/2+1/2*mTau3}},1/2+1/6*mTau3,mTau3))+526*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau3}},1/2+1/3*mTau3,mTau3))+Gt({{0,0},{2,1/2+1/2*mTau3}},1/2+1/3*mTau3,mTau3))-1367/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-1/2*mTau3}},5/6*mTau3,mTau3))+Gt({{0,0},{2,1/2+1/2*mTau3}},5/6*mTau3,mTau3))-31277/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/2*mTau3}},1/2,mTau3))+Gt({{0,0},{2,2/3+1/2*mTau3}},1/2,mTau3))-882/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-1/2*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{2,2/3+1/2*mTau3}},1/2+1/2*mTau3,mTau3))-9379/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/2*mTau3}},1/2,mTau3))+Gt({{0,0},{2,5/6+1/2*mTau3}},1/2,mTau3))-756/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/2*mTau3}},2/3,mTau3))+Gt({{0,0},{2,5/6+1/2*mTau3}},2/3,mTau3))+756/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/2*mTau3}},5/6,mTau3))+Gt({{0,0},{2,5/6+1/2*mTau3}},5/6,mTau3))+6207/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-1/2*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{2,5/6+1/2*mTau3}},1/2*mTau3,mTau3))+7343/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-2/3*mTau3}},1/2,mTau3))+Gt({{0,0},{2,1/6+2/3*mTau3}},1/2,mTau3))-4281/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-2/3*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{2,1/6+2/3*mTau3}},1/2*mTau3,mTau3))+2541/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-2/3*mTau3}},2/3*mTau3,mTau3))+Gt({{0,0},{2,1/6+2/3*mTau3}},2/3*mTau3,mTau3))+665/2*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3-2/3*mTau3}},1/2,mTau3))+Gt({{0,0},{2,1/3+2/3*mTau3}},1/2,mTau3))+231/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-2/3*mTau3}},1/2,mTau3))+Gt({{0,0},{2,1/2+2/3*mTau3}},1/2,mTau3))-386267/60*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-2/3*mTau3}},1/3,mTau3))+Gt({{0,0},{2,2/3+2/3*mTau3}},1/3,mTau3))+386267/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-2/3*mTau3}},2/3,mTau3))+Gt({{0,0},{2,2/3+2/3*mTau3}},2/3,mTau3))+669/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau3}},1/2,mTau3))+Gt({{0,0},{2,5/6+2/3*mTau3}},1/2,mTau3))+144/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau3}},2/3,mTau3))+Gt({{0,0},{2,5/6+2/3*mTau3}},2/3,mTau3))+3108/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau3}},1/6+1/6*mTau3,mTau3))+Gt({{0,0},{2,5/6+2/3*mTau3}},1/6+1/6*mTau3,mTau3))+84/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau3}},2/3+5/6*mTau3,mTau3))+Gt({{0,0},{2,5/6+2/3*mTau3}},2/3+5/6*mTau3,mTau3))-64911/80*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau3}},1/6*mTau3,mTau3))+Gt({{0,0},{2,5/6+2/3*mTau3}},1/6*mTau3,mTau3))-4737/80*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-2/3*mTau3}},5/6*mTau3,mTau3))+Gt({{0,0},{2,5/6+2/3*mTau3}},5/6*mTau3,mTau3))-1579/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-5/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{2,1/6+5/6*mTau3}},1/2*mTau3,mTau3))-165/4*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6-5/6*mTau3}},2/3*mTau3,mTau3))+Gt({{0,0},{2,1/6+5/6*mTau3}},2/3*mTau3,mTau3))+893/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3-5/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{2,1/3+5/6*mTau3}},1/2*mTau3,mTau3))-63*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-5/6*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{2,1/2+5/6*mTau3}},1/2+1/2*mTau3,mTau3))+84/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-5/6*mTau3}},1/2+2/3*mTau3,mTau3))+Gt({{0,0},{2,1/2+5/6*mTau3}},1/2+2/3*mTau3,mTau3))+203/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2-5/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{2,1/2+5/6*mTau3}},1/2*mTau3,mTau3))-3333/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-5/6*mTau3}},1/2,mTau3))+Gt({{0,0},{2,2/3+5/6*mTau3}},1/2,mTau3))-41/2*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-5/6*mTau3}},1/2+1/3*mTau3,mTau3))+Gt({{0,0},{2,2/3+5/6*mTau3}},1/2+1/3*mTau3,mTau3))+221/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-5/6*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{0,0},{2,2/3+5/6*mTau3}},1/2+1/2*mTau3,mTau3))-253/30*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3-5/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{2,2/3+5/6*mTau3}},1/2*mTau3,mTau3))-37841/450*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau3}},1/3,mTau3))+Gt({{0,0},{2,5/6+5/6*mTau3}},1/3,mTau3))+321817/450*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau3}},1/2,mTau3))+Gt({{0,0},{2,5/6+5/6*mTau3}},1/2,mTau3))-136121/450*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau3}},2/3,mTau3))+Gt({{0,0},{2,5/6+5/6*mTau3}},2/3,mTau3))+12277/50*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau3}},5/6,mTau3))+Gt({{0,0},{2,5/6+5/6*mTau3}},5/6,mTau3))-208783/24*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau3}},1/6*mTau3,mTau3))+Gt({{0,0},{2,5/6+5/6*mTau3}},1/6*mTau3,mTau3))+1078601/360*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau3}},1/2*mTau3,mTau3))+Gt({{0,0},{2,5/6+5/6*mTau3}},1/2*mTau3,mTau3))-423/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6-5/6*mTau3}},5/6*mTau3,mTau3))+Gt({{0,0},{2,5/6+5/6*mTau3}},5/6*mTau3,mTau3))-13319/16*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6*mTau3}},1/2,mTau3))+Gt({{0,0},{2,1/6*mTau3}},1/2,mTau3))+8037/80*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/6*mTau3}},5/6+2/3*mTau3,mTau3))+Gt({{0,0},{2,1/6*mTau3}},5/6+2/3*mTau3,mTau3))-231/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/3*mTau3}},1/2,mTau3))+Gt({{0,0},{2,1/3*mTau3}},1/2,mTau3))-755827/1200*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau3}},5/6,mTau3))+Gt({{0,0},{2,1/2*mTau3}},5/6,mTau3))-22841/20*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau3}},2/3+1/6*mTau3,mTau3))+Gt({{0,0},{2,1/2*mTau3}},2/3+1/6*mTau3,mTau3))-1371547/360*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau3}},5/6+1/3*mTau3,mTau3))+Gt({{0,0},{2,1/2*mTau3}},5/6+1/3*mTau3,mTau3))+969431/120*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau3}},5/6+1/2*mTau3,mTau3))+Gt({{0,0},{2,1/2*mTau3}},5/6+1/2*mTau3,mTau3))+20261/240*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau3}},1/6+2/3*mTau3,mTau3))+Gt({{0,0},{2,1/2*mTau3}},1/6+2/3*mTau3,mTau3))+577/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau3}},1/3+5/6*mTau3,mTau3))+Gt({{0,0},{2,1/2*mTau3}},1/3+5/6*mTau3,mTau3))+1639/30*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau3}},2/3+5/6*mTau3,mTau3))+Gt({{0,0},{2,1/2*mTau3}},2/3+5/6*mTau3,mTau3))-1011209/360*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau3}},5/6+5/6*mTau3,mTau3))+Gt({{0,0},{2,1/2*mTau3}},5/6+5/6*mTau3,mTau3))-5822/5*(pow(Pi,-1))*(-(Gt({{0,0},{2,-1/2*mTau3}},1/3*mTau3,mTau3))+Gt({{0,0},{2,1/2*mTau3}},1/3*mTau3,mTau3))+231/10*(pow(Pi,-1))*(-(Gt({{0,0},{2,-2/3*mTau3}},1/2,mTau3))+Gt({{0,0},{2,2/3*mTau3}},1/2,mTau3))-70291/80*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6*mTau3}},1/2,mTau3))+Gt({{0,0},{2,5/6*mTau3}},1/2,mTau3))+8037/80*(pow(Pi,-1))*(-(Gt({{0,0},{2,-5/6*mTau3}},5/6+2/3*mTau3,mTau3))+Gt({{0,0},{2,5/6*mTau3}},5/6+2/3*mTau3,mTau3))",
		"88.4966278247746787196625855663374721614175509106773646656811130976590626174303474043642730651481441738440508884436345");
	test(9,"2197/300*(pow(Pi,-2))*(-(Gt({{3,-1/6}},5/6+5/6*mTau3,mTau3))+Gt({{3,1/6}},5/6+5/6*mTau3,mTau3))-10409/90*(pow(Pi,-2))*(-(Gt({{3,-1/6}},1/2*mTau3,mTau3))+Gt({{3,1/6}},1/2*mTau3,mTau3))-2197/40*(pow(Pi,-2))*(-(Gt({{3,-1/3}},1/6,mTau3))+Gt({{3,1/3}},1/6,mTau3))-386267/360*(pow(Pi,-2))*(-(Gt({{3,-1/3}},2/3+1/3*mTau3,mTau3))+Gt({{3,1/3}},2/3+1/3*mTau3,mTau3))+386267/360*(pow(Pi,-2))*(-(Gt({{3,-1/3}},2/3+2/3*mTau3,mTau3))+Gt({{3,1/3}},2/3+2/3*mTau3,mTau3))+6207/40*(pow(Pi,-2))*(-(Gt({{3,-1/3}},1/2*mTau3,mTau3))+Gt({{3,1/3}},1/2*mTau3,mTau3))-386267/720*(pow(Pi,-2))*(-(Gt({{3,-1/3}},2/3*mTau3,mTau3))+Gt({{3,1/3}},2/3*mTau3,mTau3))+665/6*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/6*mTau3}},1/2,mTau3))+Gt({{3,1/6+1/6*mTau3}},1/2,mTau3))-893/120*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/6*mTau3}},1/2*mTau3,mTau3))+Gt({{3,1/6+1/6*mTau3}},1/2*mTau3,mTau3))+1271/24*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/6*mTau3}},1/2,mTau3))+Gt({{3,2/3+1/6*mTau3}},1/2,mTau3))-1271/24*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/6*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{3,2/3+1/6*mTau3}},1/2+1/2*mTau3,mTau3))+8141/120*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/6*mTau3}},1/2*mTau3,mTau3))+Gt({{3,2/3+1/6*mTau3}},1/2*mTau3,mTau3))+70*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/6*mTau3}},2/3,mTau3))+Gt({{3,5/6+1/6*mTau3}},2/3,mTau3))+168/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/6*mTau3}},5/6,mTau3))+Gt({{3,5/6+1/6*mTau3}},5/6,mTau3))-77/2*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/6*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{3,5/6+1/6*mTau3}},1/2+1/2*mTau3,mTau3))+48/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/6*mTau3}},1/6*mTau3,mTau3))+Gt({{3,5/6+1/6*mTau3}},1/6*mTau3,mTau3))+519/40*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/6*mTau3}},1/2*mTau3,mTau3))+Gt({{3,5/6+1/6*mTau3}},1/2*mTau3,mTau3))-140/3*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/3*mTau3}},1/2,mTau3))+Gt({{3,1/6+1/3*mTau3}},1/2,mTau3))+14/5*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/3*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{3,1/6+1/3*mTau3}},1/2+1/2*mTau3,mTau3))-57739/288*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/3*mTau3}},1/6*mTau3,mTau3))+Gt({{3,1/6+1/3*mTau3}},1/6*mTau3,mTau3))+36031/1440*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/3*mTau3}},5/6*mTau3,mTau3))+Gt({{3,1/6+1/3*mTau3}},5/6*mTau3,mTau3))+386267/720*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/3*mTau3}},1/3,mTau3))+Gt({{3,2/3+1/3*mTau3}},1/3,mTau3))-665/6*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/3*mTau3}},1/2,mTau3))+Gt({{3,2/3+1/3*mTau3}},1/2,mTau3))-386267/720*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/3*mTau3}},2/3,mTau3))+Gt({{3,2/3+1/3*mTau3}},2/3,mTau3))+386267/720*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/3*mTau3}},1/3+1/3*mTau3,mTau3))+Gt({{3,2/3+1/3*mTau3}},1/3+1/3*mTau3,mTau3))-386267/720*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/3*mTau3}},1/3*mTau3,mTau3))+Gt({{3,2/3+1/3*mTau3}},1/3*mTau3,mTau3))+1127/60*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/3*mTau3}},1/2,mTau3))+Gt({{3,5/6+1/3*mTau3}},1/2,mTau3))+518/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/3*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{3,5/6+1/3*mTau3}},1/2+1/2*mTau3,mTau3))-1045553/720*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/3*mTau3}},1/3*mTau3,mTau3))+Gt({{3,5/6+1/3*mTau3}},1/3*mTau3,mTau3))+231407/270*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/3*mTau3}},1/2*mTau3,mTau3))+Gt({{3,5/6+1/3*mTau3}},1/2*mTau3,mTau3))+22867/360*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/2*mTau3}},1/2,mTau3))+Gt({{3,1/6+1/2*mTau3}},1/2,mTau3))-2069/40*(pow(Pi,-2))*(-(Gt({{3,-1/6-1/2*mTau3}},2/3+1/2*mTau3,mTau3))+Gt({{3,1/6+1/2*mTau3}},2/3+1/2*mTau3,mTau3))-72913/360*(pow(Pi,-2))*(-(Gt({{3,-1/3-1/2*mTau3}},1/2,mTau3))+Gt({{3,1/3+1/2*mTau3}},1/2,mTau3))+1481/20*(pow(Pi,-2))*(-(Gt({{3,-1/3-1/2*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{3,1/3+1/2*mTau3}},1/2+1/2*mTau3,mTau3))-188113/10800*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau3}},1/3,mTau3))+Gt({{3,1/2+1/2*mTau3}},1/3,mTau3))+1582769/10800*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau3}},2/3,mTau3))+Gt({{3,1/2+1/2*mTau3}},2/3,mTau3))-1555/8*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau3}},5/6,mTau3))+Gt({{3,1/2+1/2*mTau3}},5/6,mTau3))+105/2*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau3}},1/2+1/6*mTau3,mTau3))+Gt({{3,1/2+1/2*mTau3}},1/2+1/6*mTau3,mTau3))+263/3*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau3}},1/2+1/3*mTau3,mTau3))+Gt({{3,1/2+1/2*mTau3}},1/2+1/3*mTau3,mTau3))-1367/30*(pow(Pi,-2))*(-(Gt({{3,-1/2-1/2*mTau3}},5/6*mTau3,mTau3))+Gt({{3,1/2+1/2*mTau3}},5/6*mTau3,mTau3))-31277/360*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/2*mTau3}},1/2,mTau3))+Gt({{3,2/3+1/2*mTau3}},1/2,mTau3))-147/5*(pow(Pi,-2))*(-(Gt({{3,-2/3-1/2*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{3,2/3+1/2*mTau3}},1/2+1/2*mTau3,mTau3))-9379/360*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/2*mTau3}},1/2,mTau3))+Gt({{3,5/6+1/2*mTau3}},1/2,mTau3))-126/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/2*mTau3}},2/3,mTau3))+Gt({{3,5/6+1/2*mTau3}},2/3,mTau3))+126/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/2*mTau3}},5/6,mTau3))+Gt({{3,5/6+1/2*mTau3}},5/6,mTau3))+2069/40*(pow(Pi,-2))*(-(Gt({{3,-5/6-1/2*mTau3}},1/2*mTau3,mTau3))+Gt({{3,5/6+1/2*mTau3}},1/2*mTau3,mTau3))+7343/60*(pow(Pi,-2))*(-(Gt({{3,-1/6-2/3*mTau3}},1/2,mTau3))+Gt({{3,1/6+2/3*mTau3}},1/2,mTau3))-1427/40*(pow(Pi,-2))*(-(Gt({{3,-1/6-2/3*mTau3}},1/2*mTau3,mTau3))+Gt({{3,1/6+2/3*mTau3}},1/2*mTau3,mTau3))+847/40*(pow(Pi,-2))*(-(Gt({{3,-1/6-2/3*mTau3}},2/3*mTau3,mTau3))+Gt({{3,1/6+2/3*mTau3}},2/3*mTau3,mTau3))+665/12*(pow(Pi,-2))*(-(Gt({{3,-1/3-2/3*mTau3}},1/2,mTau3))+Gt({{3,1/3+2/3*mTau3}},1/2,mTau3))+77/10*(pow(Pi,-2))*(-(Gt({{3,-1/2-2/3*mTau3}},1/2,mTau3))+Gt({{3,1/2+2/3*mTau3}},1/2,mTau3))-386267/360*(pow(Pi,-2))*(-(Gt({{3,-2/3-2/3*mTau3}},1/3,mTau3))+Gt({{3,2/3+2/3*mTau3}},1/3,mTau3))+386267/720*(pow(Pi,-2))*(-(Gt({{3,-2/3-2/3*mTau3}},2/3,mTau3))+Gt({{3,2/3+2/3*mTau3}},2/3,mTau3))+223/10*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau3}},1/2,mTau3))+Gt({{3,5/6+2/3*mTau3}},1/2,mTau3))+24/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau3}},2/3,mTau3))+Gt({{3,5/6+2/3*mTau3}},2/3,mTau3))+518/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau3}},1/6+1/6*mTau3,mTau3))+Gt({{3,5/6+2/3*mTau3}},1/6+1/6*mTau3,mTau3))+14/5*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau3}},2/3+5/6*mTau3,mTau3))+Gt({{3,5/6+2/3*mTau3}},2/3+5/6*mTau3,mTau3))-21637/160*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau3}},1/6*mTau3,mTau3))+Gt({{3,5/6+2/3*mTau3}},1/6*mTau3,mTau3))-1579/160*(pow(Pi,-2))*(-(Gt({{3,-5/6-2/3*mTau3}},5/6*mTau3,mTau3))+Gt({{3,5/6+2/3*mTau3}},5/6*mTau3,mTau3))-1579/120*(pow(Pi,-2))*(-(Gt({{3,-1/6-5/6*mTau3}},1/2*mTau3,mTau3))+Gt({{3,1/6+5/6*mTau3}},1/2*mTau3,mTau3))-55/8*(pow(Pi,-2))*(-(Gt({{3,-1/6-5/6*mTau3}},2/3*mTau3,mTau3))+Gt({{3,1/6+5/6*mTau3}},2/3*mTau3,mTau3))+893/60*(pow(Pi,-2))*(-(Gt({{3,-1/3-5/6*mTau3}},1/2*mTau3,mTau3))+Gt({{3,1/3+5/6*mTau3}},1/2*mTau3,mTau3))-21/2*(pow(Pi,-2))*(-(Gt({{3,-1/2-5/6*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{3,1/2+5/6*mTau3}},1/2+1/2*mTau3,mTau3))+14/5*(pow(Pi,-2))*(-(Gt({{3,-1/2-5/6*mTau3}},1/2+2/3*mTau3,mTau3))+Gt({{3,1/2+5/6*mTau3}},1/2+2/3*mTau3,mTau3))+203/30*(pow(Pi,-2))*(-(Gt({{3,-1/2-5/6*mTau3}},1/2*mTau3,mTau3))+Gt({{3,1/2+5/6*mTau3}},1/2*mTau3,mTau3))-1111/10*(pow(Pi,-2))*(-(Gt({{3,-2/3-5/6*mTau3}},1/2,mTau3))+Gt({{3,2/3+5/6*mTau3}},1/2,mTau3))-41/12*(pow(Pi,-2))*(-(Gt({{3,-2/3-5/6*mTau3}},1/2+1/3*mTau3,mTau3))+Gt({{3,2/3+5/6*mTau3}},1/2+1/3*mTau3,mTau3))+221/60*(pow(Pi,-2))*(-(Gt({{3,-2/3-5/6*mTau3}},1/2+1/2*mTau3,mTau3))+Gt({{3,2/3+5/6*mTau3}},1/2+1/2*mTau3,mTau3))-253/180*(pow(Pi,-2))*(-(Gt({{3,-2/3-5/6*mTau3}},1/2*mTau3,mTau3))+Gt({{3,2/3+5/6*mTau3}},1/2*mTau3,mTau3))-37841/2700*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau3}},1/3,mTau3))+Gt({{3,5/6+5/6*mTau3}},1/3,mTau3))+321817/2700*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau3}},1/2,mTau3))+Gt({{3,5/6+5/6*mTau3}},1/2,mTau3))-136121/2700*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau3}},2/3,mTau3))+Gt({{3,5/6+5/6*mTau3}},2/3,mTau3))+12277/300*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau3}},5/6,mTau3))+Gt({{3,5/6+5/6*mTau3}},5/6,mTau3))-208783/144*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau3}},1/6*mTau3,mTau3))+Gt({{3,5/6+5/6*mTau3}},1/6*mTau3,mTau3))+1078601/2160*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau3}},1/2*mTau3,mTau3))+Gt({{3,5/6+5/6*mTau3}},1/2*mTau3,mTau3))-141/10*(pow(Pi,-2))*(-(Gt({{3,-5/6-5/6*mTau3}},5/6*mTau3,mTau3))+Gt({{3,5/6+5/6*mTau3}},5/6*mTau3,mTau3))-13319/96*(pow(Pi,-2))*(-(Gt({{3,-1/6*mTau3}},1/2,mTau3))+Gt({{3,1/6*mTau3}},1/2,mTau3))+2679/160*(pow(Pi,-2))*(-(Gt({{3,-1/6*mTau3}},5/6+2/3*mTau3,mTau3))+Gt({{3,1/6*mTau3}},5/6+2/3*mTau3,mTau3))-77/10*(pow(Pi,-2))*(-(Gt({{3,-1/3*mTau3}},1/2,mTau3))+Gt({{3,1/3*mTau3}},1/2,mTau3))-755827/7200*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau3}},5/6,mTau3))+Gt({{3,1/2*mTau3}},5/6,mTau3))-22841/120*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau3}},2/3+1/6*mTau3,mTau3))+Gt({{3,1/2*mTau3}},2/3+1/6*mTau3,mTau3))-1371547/2160*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau3}},5/6+1/3*mTau3,mTau3))+Gt({{3,1/2*mTau3}},5/6+1/3*mTau3,mTau3))+969431/720*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau3}},5/6+1/2*mTau3,mTau3))+Gt({{3,1/2*mTau3}},5/6+1/2*mTau3,mTau3))+20261/1440*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau3}},1/6+2/3*mTau3,mTau3))+Gt({{3,1/2*mTau3}},1/6+2/3*mTau3,mTau3))+577/60*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau3}},1/3+5/6*mTau3,mTau3))+Gt({{3,1/2*mTau3}},1/3+5/6*mTau3,mTau3))+1639/180*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau3}},2/3+5/6*mTau3,mTau3))+Gt({{3,1/2*mTau3}},2/3+5/6*mTau3,mTau3))-1011209/2160*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau3}},5/6+5/6*mTau3,mTau3))+Gt({{3,1/2*mTau3}},5/6+5/6*mTau3,mTau3))-2911/15*(pow(Pi,-2))*(-(Gt({{3,-1/2*mTau3}},1/3*mTau3,mTau3))+Gt({{3,1/2*mTau3}},1/3*mTau3,mTau3))+77/20*(pow(Pi,-2))*(-(Gt({{3,-2/3*mTau3}},1/2,mTau3))+Gt({{3,2/3*mTau3}},1/2,mTau3))-70291/480*(pow(Pi,-2))*(-(Gt({{3,-5/6*mTau3}},1/2,mTau3))+Gt({{3,5/6*mTau3}},1/2,mTau3))+2679/160*(pow(Pi,-2))*(-(Gt({{3,-5/6*mTau3}},5/6+2/3*mTau3,mTau3))+Gt({{3,5/6*mTau3}},5/6+2/3*mTau3,mTau3))",
		"27.2183205702836204668751253993542526706711163749996263072481607480909262770095359241825445181141874395461742128276062");
	test(10,"2197/25*(-(Gt({{0,0},{0,0},{1,-1/6}},5/6+5/6*mTau4,mTau4))+Gt({{0,0},{0,0},{1,1/6}},5/6+5/6*mTau4,mTau4))-20818/15*(-(Gt({{0,0},{0,0},{1,-1/6}},1/2*mTau4,mTau4))+Gt({{0,0},{0,0},{1,1/6}},1/2*mTau4,mTau4))-6591/10*(-(Gt({{0,0},{0,0},{1,-1/3}},1/6,mTau4))+Gt({{0,0},{0,0},{1,1/3}},1/6,mTau4))-386267/30*(-(Gt({{0,0},{0,0},{1,-1/3}},2/3+1/3*mTau4,mTau4))+Gt({{0,0},{0,0},{1,1/3}},2/3+1/3*mTau4,mTau4))+386267/30*(-(Gt({{0,0},{0,0},{1,-1/3}},2/3+2/3*mTau4,mTau4))+Gt({{0,0},{0,0},{1,1/3}},2/3+2/3*mTau4,mTau4))+18621/10*(-(Gt({{0,0},{0,0},{1,-1/3}},1/2*mTau4,mTau4))+Gt({{0,0},{0,0},{1,1/3}},1/2*mTau4,mTau4))-386267/60*(-(Gt({{0,0},{0,0},{1,-1/3}},2/3*mTau4,mTau4))+Gt({{0,0},{0,0},{1,1/3}},2/3*mTau4,mTau4))+1330*(-(Gt({{0,0},{0,0},{1,-1/6-1/6*mTau4}},1/2,mTau4))+Gt({{0,0},{0,0},{1,1/6+1/6*mTau4}},1/2,mTau4))-893/10*(-(Gt({{0,0},{0,0},{1,-1/6-1/6*mTau4}},1/2*mTau4,mTau4))+Gt({{0,0},{0,0},{1,1/6+1/6*mTau4}},1/2*mTau4,mTau4))+1271/2*(-(Gt({{0,0},{0,0},{1,-2/3-1/6*mTau4}},1/2,mTau4))+Gt({{0,0},{0,0},{1,2/3+1/6*mTau4}},1/2,mTau4))-1271/2*(-(Gt({{0,0},{0,0},{1,-2/3-1/6*mTau4}},1/2+1/2*mTau4,mTau4))+Gt({{0,0},{0,0},{1,2/3+1/6*mTau4}},1/2+1/2*mTau4,mTau4))+8141/10*(-(Gt({{0,0},{0,0},{1,-2/3-1/6*mTau4}},1/2*mTau4,mTau4))+Gt({{0,0},{0,0},{1,2/3+1/6*mTau4}},1/2*mTau4,mTau4))+840*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau4}},2/3,mTau4))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau4}},2/3,mTau4))+2016/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau4}},5/6,mTau4))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau4}},5/6,mTau4))-462*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau4}},1/2+1/2*mTau4,mTau4))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau4}},1/2+1/2*mTau4,mTau4))+576/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau4}},1/6*mTau4,mTau4))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau4}},1/6*mTau4,mTau4))+1557/10*(-(Gt({{0,0},{0,0},{1,-5/6-1/6*mTau4}},1/2*mTau4,mTau4))+Gt({{0,0},{0,0},{1,5/6+1/6*mTau4}},1/2*mTau4,mTau4))-560*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau4}},1/2,mTau4))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau4}},1/2,mTau4))+168/5*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau4}},1/2+1/2*mTau4,mTau4))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau4}},1/2+1/2*mTau4,mTau4))-57739/24*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau4}},1/6*mTau4,mTau4))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau4}},1/6*mTau4,mTau4))+36031/120*(-(Gt({{0,0},{0,0},{1,-1/6-1/3*mTau4}},5/6*mTau4,mTau4))+Gt({{0,0},{0,0},{1,1/6+1/3*mTau4}},5/6*mTau4,mTau4))+386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau4}},1/3,mTau4))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau4}},1/3,mTau4))-1330*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau4}},1/2,mTau4))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau4}},1/2,mTau4))-386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau4}},2/3,mTau4))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau4}},2/3,mTau4))+386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau4}},1/3+1/3*mTau4,mTau4))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau4}},1/3+1/3*mTau4,mTau4))-386267/60*(-(Gt({{0,0},{0,0},{1,-2/3-1/3*mTau4}},1/3*mTau4,mTau4))+Gt({{0,0},{0,0},{1,2/3+1/3*mTau4}},1/3*mTau4,mTau4))+1127/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau4}},1/2,mTau4))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau4}},1/2,mTau4))+6216/5*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau4}},1/2+1/2*mTau4,mTau4))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau4}},1/2+1/2*mTau4,mTau4))-1045553/60*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau4}},1/3*mTau4,mTau4))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau4}},1/3*mTau4,mTau4))+462814/45*(-(Gt({{0,0},{0,0},{1,-5/6-1/3*mTau4}},1/2*mTau4,mTau4))+Gt({{0,0},{0,0},{1,5/6+1/3*mTau4}},1/2*mTau4,mTau4))+22867/30*(-(Gt({{0,0},{0,0},{1,-1/6-1/2*mTau4}},1/2,mTau4))+Gt({{0,0},{0,0},{1,1/6+1/2*mTau4}},1/2,mTau4))-6207/10*(-(Gt({{0,0},{0,0},{1,-1/6-1/2*mTau4}},2/3+1/2*mTau4,mTau4))+Gt({{0,0},{0,0},{1,1/6+1/2*mTau4}},2/3+1/2*mTau4,mTau4))-72913/30*(-(Gt({{0,0},{0,0},{1,-1/3-1/2*mTau4}},1/2,mTau4))+Gt({{0,0},{0,0},{1,1/3+1/2*mTau4}},1/2,mTau4))+4443/5*(-(Gt({{0,0},{0,0},{1,-1/3-1/2*mTau4}},1/2+1/2*mTau4,mTau4))+Gt({{0,0},{0,0},{1,1/3+1/2*mTau4}},1/2+1/2*mTau4,mTau4))-188113/900*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau4}},1/3,mTau4))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau4}},1/3,mTau4))+1582769/900*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau4}},2/3,mTau4))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau4}},2/3,mTau4))-4665/2*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau4}},5/6,mTau4))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau4}},5/6,mTau4))+630*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau4}},1/2+1/6*mTau4,mTau4))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau4}},1/2+1/6*mTau4,mTau4))+1052*(-(Gt({{0,0},{0,0},{1,-1/2-1/2*mTau4}},1/2+1/3*mTau4,mTau4))+Gt({{0,0},{0,0},{1,1/2+1/2*mTau4}},1/2+1/3*mTau4,mTau4))-