/* -*- Mode: Vala; indent-tabs-mode: nil; tab-width: 2 -*-
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 * SPDX-FileCopyrightText: Michael Terry
 */

using GLib;

[GtkTemplate (ui = "/org/gnome/DejaDup/BrowserPage.ui")]
public class BrowserPage : Adw.NavigationPage
{
  public Gtk.Window window {get; set;}

  public void scan(string tag, DejaDup.Operation.State? state = null)
  {
    if (tag != "" && browser.tag == tag) {
      stack.visible_child_name = "browser";
      return;
    }

    saved_state = state;
    browser.clear_tree();

    try {
      var op = OperationLauncher.get_instance().create_list_files(tag, state);
      operation.set(op);

      op.listed_current_files.connect(handle_listed_current_files);
      status_page.connect_to_operation(op);

      op.start.begin();
    } catch (Error e) {
      status_page.set_error_state(e.message);
    }

    stack.visible_child_name = "operation";
  }

  [GtkChild]
  unowned RestoreToolbarView restore_view;
  [GtkChild]
  unowned Gtk.Stack stack;
  [GtkChild]
  unowned OperationStatusPage status_page;
  [GtkChild]
  unowned Browser browser;

  const ActionEntry[] ACTIONS = {
    {"select-all", select_all},
    {"go-up", go_up},
    {"search", activate_search},
  };

  SimpleActionGroup action_group;
  WeakRef operation;
  DejaDup.Operation.State saved_state;

  construct {
    // Set up actions
    action_group = new SimpleActionGroup();
    action_group.add_action_entries(ACTIONS, this);

    var go_up_action = action_group.lookup_action("go-up");
    browser.bind_property("can-go-up", go_up_action, "enabled", BindingFlags.SYNC_CREATE);

    var select_all_action = action_group.lookup_action("select-all");
    browser.bind_property("has-content", select_all_action, "enabled", BindingFlags.SYNC_CREATE);
  }

  void select_all() {
    browser.select_all();
  }

  void go_up() {
    browser.go_up();
  }

  void activate_search() {
    restore_view.activate_search();
  }

  void handle_listed_current_files(DejaDup.OperationFiles op, DejaDup.FileTree tree) {
    browser.set_tree(op.tag, tree);
    saved_state = op.get_state();
    stack.visible_child_name = "browser";
  }

  [GtkCallback]
  void start_restore()
  {
    start_restore_async.begin();
  }

  async void start_restore_async()
  {
    var files = browser.get_selected_files();
    if (files == null)
      return;

    var dlg = new RestoreTargetDialog(browser.get_tree(), files);
    dlg.accepted.connect(() => {
      OperationLauncher.get_instance().start_restore(
        saved_state, dlg.restore_location, browser.get_tree(), browser.tag, files
      );
    });
    dlg.present(this);
  }

  [GtkCallback]
  void handle_showing()
  {
    window.insert_action_group("restore", action_group);
  }

  [GtkCallback]
  void handle_hidden()
  {
    window.insert_action_group("restore", null);

    var op = operation.get() as DejaDup.Operation;
    if (op != null)
      op.stop();
    operation.set(null);
  }

  [GtkCallback]
  string calculate_title(string? folder_name, string? search_filter)
  {
    if (search_filter != null && search_filter != "")
      return C_("noun", "Search");
    else if (folder_name != null && folder_name != "")
      return folder_name;
    else
      return _("Files");
  }
}
