/* SPDX-License-Identifier: BSD-3-Clause
 * Copyright (C) 2019 Marvell International Ltd.
 */

#include <rte_common.h>
#include <rte_malloc.h>

#include "cpt_mcode_defines.h"
#include "cpt_pmd_logs.h"
#include "cpt_pmd_ops_helper.h"

/*
 * CPT FPM table sizes Enumeration
 *
 * 15 table entries * (X, Y, Z coordinates) * Coordinate Offset
 * Coordinate Offset depends on elliptic curve as mentioned below,
 * 6 quadwords for P-192, P-224 and P-256
 * 7 quadwords for P-384
 * 9 quadwords for P-521
 */
typedef enum {
	CPT_P192_LEN = 2160,
	CPT_P224_LEN = 2160,
	CPT_P256_LEN = 2160,
	CPT_P384_LEN = 2520,
	CPT_P521_LEN = 3240
} cpt_fpm_len_t;

/* FPM table address and length */
struct fpm_entry {
	const uint8_t *data;
	int len;
};

/*
 * Pre-computed ECC FMUL tables needed by cpt microcode
 * for NIST curves P-192, P-256, P-384, P-521, P-224.
 */

static const uint8_t fpm_table_p192[CPT_P192_LEN] = {
	0xf4, 0xff, 0x0a, 0xfd, 0x82, 0xff, 0x10, 0x12, 0x7c, 0xbf, 0x20, 0xeb,
	0x43, 0xa1, 0x88, 0x00, 0x18, 0x8d, 0xa8, 0x0e, 0xb0, 0x30, 0x90, 0xf6,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x73, 0xf9, 0x77, 0xa1, 0x1e, 0x79, 0x48, 0x11, 0x63, 0x10, 0x11, 0xed,
	0x6b, 0x24, 0xcd, 0xd5, 0x07, 0x19, 0x2b, 0x95, 0xff, 0xc8, 0xda, 0x78,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xc3, 0x96, 0x49, 0xc5, 0x5d, 0x7c, 0x48, 0xd8, 0xeb, 0x2c, 0xdf, 0xae,
	0x5a, 0x92, 0x7c, 0x35, 0x67, 0xe3, 0x0c, 0xbd, 0xcb, 0xa6, 0x71, 0xfb,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x7a, 0x83, 0xce, 0xe1, 0xec, 0xbf, 0xbe, 0x7d, 0xce, 0x32, 0xd0, 0x3c,
	0x06, 0x30, 0x15, 0x77, 0xa9, 0x35, 0x49, 0xc4, 0x58, 0x10, 0xf5, 0xc3,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x6f, 0x5e, 0xf8, 0x89, 0x66, 0xe3, 0xea, 0xd3, 0xf2, 0x9e, 0x6f, 0xea,
	0xdf, 0xc9, 0xbf, 0x1a, 0xce, 0x21, 0x6b, 0xb8, 0x45, 0x20, 0x06, 0xe0,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x46, 0xb9, 0x09, 0x2d, 0x92, 0x7b, 0x37, 0x79, 0x1d, 0x0a, 0xeb, 0x4b,
	0xb5, 0xb8, 0x0a, 0x20, 0xd9, 0x8a, 0x2e, 0xe2, 0x5a, 0xae, 0xc9, 0x58,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xb1, 0x99, 0x63, 0xd8, 0xc0, 0xa1, 0xe3, 0x40, 0x47, 0x30, 0xd4, 0xf4,
	0x80, 0xd1, 0x09, 0x0b, 0x51, 0xa5, 0x81, 0xd9, 0x18, 0x4a, 0xc7, 0x37,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xec, 0xc5, 0x67, 0x31, 0xe6, 0x99, 0x12, 0xa5, 0x7c, 0xdf, 0xce, 0xa0,
	0x2f, 0x68, 0x3f, 0x16, 0x5b, 0xd8, 0x1e, 0xe2, 0xe0, 0xbb, 0x9f, 0x6e,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xe4, 0xb1, 0x5a, 0x2d, 0xd4, 0xf4, 0x33, 0x74, 0x07, 0x57, 0xee, 0xa7,
	0xf2, 0x92, 0xc3, 0x41, 0x0c, 0x73, 0x06, 0x91, 0xd0, 0xf8, 0xdc, 0x24,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xdf, 0x79, 0x78, 0x90, 0xbb, 0xf4, 0x5e, 0x00, 0x00, 0x8a, 0x9e, 0x83,
	0xe9, 0xde, 0x87, 0x08, 0x31, 0xb2, 0x4c, 0x31, 0x93, 0x54, 0xde, 0x3e,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xcb, 0x5e, 0xc0, 0x43, 0xdd, 0xf6, 0x3a, 0xba, 0xc9, 0x4c, 0x21, 0xd9,
	0xf8, 0x4f, 0x41, 0xe1, 0xf0, 0xf4, 0x08, 0x83, 0x61, 0xd2, 0x44, 0x16,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xf3, 0x75, 0x85, 0xb0, 0x40, 0x64, 0x95, 0xf7, 0xe5, 0xde, 0x3b, 0x5b,
	0x16, 0xbc, 0xd0, 0xca, 0x27, 0x85, 0x3c, 0x1a, 0xe1, 0x3e, 0xa4, 0x88,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xd0, 0x74, 0x23, 0x2a, 0x8e, 0x8a, 0xe6, 0x8f, 0x74, 0x9e, 0x52, 0x8e,
	0xee, 0x29, 0xf7, 0xa9, 0x06, 0x11, 0xde, 0xa3, 0x97, 0x16, 0x46, 0x9f,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x66, 0xb8, 0x67, 0xdd, 0x0d, 0x80, 0x43, 0xcc, 0x6a, 0x65, 0x46, 0x54,
	0x3a, 0x72, 0x7d, 0xe6, 0xf9, 0x54, 0x60, 0x52, 0x83, 0x38, 0xbd, 0xc9,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xb6, 0xeb, 0x71, 0x93, 0x0c, 0x5d, 0x8f, 0x50, 0x1c, 0x24, 0x5c, 0x02,
	0xb9, 0x04, 0xb5, 0x96, 0x04, 0xbc, 0x1f, 0x71, 0x95, 0x1f, 0x75, 0x13,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xa4, 0xd0, 0x91, 0x6e, 0xbe, 0x34, 0x80, 0x3d, 0x8b, 0xec, 0x94, 0x8a,
	0x8c, 0x21, 0x96, 0x2a, 0x15, 0x00, 0x96, 0xe7, 0xfd, 0x69, 0xf8, 0xd0,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xbd, 0x44, 0xff, 0xe8, 0xe7, 0x1a, 0xac, 0x0c, 0x7d, 0x69, 0xa0, 0xb0,
	0x43, 0x22, 0xd0, 0x65, 0x9f, 0x56, 0xd9, 0x6c, 0xec, 0xa3, 0xba, 0x2a,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xee, 0x59, 0xf0, 0xd1, 0x25, 0xa5, 0x9d, 0xce, 0x83, 0x7d, 0x62, 0xdd,
	0xc3, 0xf4, 0x57, 0x5a, 0xa4, 0xe0, 0x7f, 0xb3, 0x35, 0xde, 0x73, 0xd9,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xec, 0x76, 0x76, 0x0c, 0x1c, 0xf4, 0x6a, 0xe2, 0xff, 0x54, 0x98, 0x32,
	0xa3, 0x3d, 0x44, 0xb0, 0xe9, 0x5a, 0xd2, 0x10, 0xf3, 0x18, 0x5c, 0x11,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x27, 0x3e, 0x5e, 0xc5, 0x38, 0xed, 0x37, 0x2e, 0x51, 0xd3, 0x91, 0x36,
	0xb0, 0xab, 0x11, 0x69, 0xa5, 0xea, 0x86, 0xf6, 0x8f, 0x3a, 0x27, 0xc8,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x29, 0x12, 0x37, 0xea, 0x74, 0xd2, 0xd7, 0xd5, 0x95, 0x36, 0x36, 0xee,
	0x56, 0x33, 0x8e, 0x9b, 0x0d, 0xa6, 0x5e, 0x86, 0x28, 0x5c, 0x12, 0x0c,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x13, 0x02, 0xf0, 0x4c, 0xf1, 0x3c, 0x32, 0x33, 0xfc, 0x89, 0x8a, 0xb9,
	0x97, 0x83, 0x91, 0xb2, 0x26, 0xd6, 0x5c, 0x2e, 0x3a, 0xa0, 0x62, 0x72,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xd5, 0x09, 0x47, 0xa8, 0x18, 0xc5, 0xef, 0xe6, 0x45, 0xdb, 0x23, 0xae,
	0xfe, 0x11, 0x3c, 0x6c, 0x91, 0xf1, 0x99, 0xf2, 0xe5, 0xbb, 0xe8, 0x6d,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x37, 0x68, 0x81, 0xb6, 0x60, 0xfe, 0xc0, 0x64, 0x38, 0x73, 0x43, 0xe9,
	0x47, 0x5d, 0xae, 0xa4, 0xec, 0xcd, 0x57, 0xe8, 0xac, 0x8d, 0x8a, 0x19,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xc9, 0xfe, 0xf5, 0xb9, 0x5b, 0x51, 0x02, 0x28, 0x37, 0x4c, 0x0a, 0x4c,
	0x19, 0x2e, 0xbc, 0xd6, 0x22, 0x98, 0xf2, 0x04, 0xce, 0x6a, 0x83, 0xf9,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x46, 0xe4, 0xb8, 0x20, 0xf4, 0xc5, 0x74, 0xd0, 0x06, 0xd5, 0x86, 0x44,
	0xef, 0xeb, 0x2c, 0xc0, 0xe7, 0x13, 0xa4, 0x00, 0x10, 0xc3, 0xc9, 0x49,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x61, 0x78, 0xcb, 0x0e, 0x2d, 0x64, 0xee, 0xdf, 0x27, 0xaf, 0x7d, 0x5e,
	0xb8, 0x5e, 0x1f, 0x99, 0xd8, 0x73, 0xce, 0xd7, 0x6c, 0xb7, 0xbe, 0x1f,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xef, 0xc9, 0x12, 0x9c, 0x52, 0xa6, 0x7f, 0x9c, 0xa3, 0xd7, 0xb9, 0x57,
	0x60, 0x04, 0xd9, 0xad, 0xfc, 0x59, 0x98, 0x08, 0xdc, 0x41, 0xf8, 0xe2,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xbb, 0x6c, 0x6b, 0x59, 0x7f, 0xdf, 0x92, 0x8a, 0xad, 0x16, 0x7e, 0xf0,
	0xd7, 0xf9, 0x3b, 0xf4, 0xfa, 0xa9, 0xe4, 0x32, 0x15, 0x4e, 0x06, 0x1c,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x0c, 0x3d, 0x0d, 0x63, 0xd5, 0x2c, 0x8f, 0x3f, 0x61, 0x01, 0xb2, 0xbe,
	0xd5, 0xf7, 0xe0, 0x8f, 0xd8, 0x77, 0xcd, 0xdd, 0xd6, 0xae, 0x3c, 0xf3,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
};

static const uint8_t fpm_table_p224[CPT_P224_LEN] = {
	0x34, 0x32, 0x80, 0xd6, 0x11, 0x5c, 0x1d, 0x21, 0x4a, 0x03, 0xc1, 0xd3,
	0x56, 0xc2, 0x11, 0x22, 0x6b, 0xb4, 0xbf, 0x7f, 0x32, 0x13, 0x90, 0xb9,
	0x00, 0x00, 0x00, 0x00, 0xb7, 0x0e, 0x0c, 0xbd, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x44, 0xd5, 0x81, 0x99, 0x85, 0x00, 0x7e, 0x34, 0xcd, 0x43, 0x75, 0xa0,
	0x5a, 0x07, 0x47, 0x64, 0xb5, 0xf7, 0x23, 0xfb, 0x4c, 0x22, 0xdf, 0xe6,
	0x00, 0x00, 0x00, 0x00, 0xbd, 0x37, 0x63, 0x88, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xc8, 0xae, 0x4d, 0x94, 0xff, 0x48, 0xdb, 0xc5, 0xb5, 0xc8, 0x8b, 0x66,
	0x32, 0xc8, 0x7a, 0x44, 0x66, 0xc7, 0x27, 0x87, 0x2b, 0x8d, 0x08, 0x1c,
	0x00, 0x00, 0x00, 0x00, 0x5b, 0xe5, 0xde, 0x8b, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xc6, 0xf8, 0x1e, 0x08, 0x47, 0xfb, 0x64, 0xdb, 0xc8, 0xe3, 0x75, 0x3e,
	0x9d, 0x5a, 0x58, 0x31, 0xa2, 0x13, 0x38, 0x8c, 0x65, 0x8a, 0x02, 0xae,
	0x00, 0x00, 0x00, 0x00, 0xde, 0x52, 0x6c, 0x0d, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xee, 0x8f, 0x93, 0x0d, 0x2b, 0x30, 0x9e, 0xe8, 0xb6, 0x78, 0xea, 0x1a,
	0x0f, 0x59, 0x7e, 0x02, 0x14, 0x74, 0x52, 0x56, 0x6c, 0x25, 0x7d, 0x3e,
	0x00, 0x00, 0x00, 0x00, 0x09, 0xbe, 0x54, 0xb7, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xf6, 0x12, 0x1f, 0xdd, 0x96, 0xa2, 0x05, 0xda, 0x12, 0xa8, 0xe4, 0xf9,
	0x98, 0x15, 0x8e, 0xe1, 0x1b, 0x1d, 0x05, 0x44, 0x47, 0xf2, 0xc3, 0x3a,
	0x00, 0x00, 0x00, 0x00, 0x32, 0xf7, 0x1c, 0x32, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x26, 0x73, 0x99, 0x28, 0x0e, 0x4e, 0x09, 0x58, 0x79, 0xab, 0xae, 0x5c,
	0xa8, 0xeb, 0x9c, 0x0b, 0xe9, 0xa8, 0xac, 0xf0, 0x74, 0x0e, 0xa3, 0x35,
	0x00, 0x00, 0x00, 0x00, 0x0f, 0xb6, 0xce, 0x42, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x92, 0x09, 0xdc, 0xfe, 0x40, 0x85, 0x7c, 0x64, 0xa2, 0x3f, 0xe4, 0x34,
	0x50, 0xb4, 0x25, 0x87, 0x2a, 0x6f, 0x38, 0x62, 0xb6, 0xfe, 0x44, 0xb1,
	0x00, 0x00, 0x00, 0x00, 0x9e, 0xd1, 0x3b, 0x1b, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xf9, 0xfb, 0xf4, 0x91, 0x9a, 0x5f, 0x1c, 0x42, 0x56, 0x8b, 0xc4, 0x34,
	0x8a, 0x69, 0xdd, 0x65, 0x3d, 0x01, 0x11, 0x6e, 0x47, 0x78, 0xdf, 0x49,
	0x00, 0x00, 0x00, 0x00, 0x56, 0x65, 0xff, 0xd2, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xca, 0x4b, 0x80, 0x12, 0xe5, 0x3d, 0x3f, 0xb4, 0xe7, 0x61, 0x44, 0x25,
	0x89, 0xec, 0x86, 0x76, 0x1f, 0xde, 0x69, 0x6f, 0xcb, 0x2b, 0xe1, 0x15,
	0x00, 0x00, 0x00, 0x00, 0xc5, 0xf0, 0x98, 0xeb, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xe6, 0xf2, 0x88, 0xf1, 0x39, 0xb0, 0xa5, 0xad, 0x0c, 0x76, 0x16, 0x85,
	0x58, 0x72, 0xd0, 0x94, 0x8a, 0xbc, 0x69, 0xb6, 0x6d, 0x7a, 0xc7, 0xfb,
	0x00, 0x00, 0x00, 0x00, 0x64, 0xe1, 0x68, 0x02, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xe1, 0x94, 0x7b, 0x17, 0x96, 0xa9, 0x76, 0xcf, 0x67, 0x78, 0x38, 0x8a,
	0xd9, 0xb7, 0x68, 0xd7, 0x75, 0x39, 0xb7, 0x24, 0x17, 0x76, 0xa8, 0x39,
	0x00, 0x00, 0x00, 0x00, 0xb4, 0xdb, 0x5a, 0x94, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x86, 0x5a, 0x09, 0xbe, 0xdf, 0x5b, 0x10, 0x74, 0x74, 0xc0, 0xce, 0xf5,
	0x95, 0xab, 0xef, 0xf7, 0xba, 0x69, 0x9c, 0xe4, 0x14, 0xb3, 0x90, 0xb1,
	0x00, 0x00, 0x00, 0x00, 0xc0, 0xca, 0x10, 0xcb, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x55, 0x87, 0xbb, 0x19, 0x41, 0x02, 0x67, 0xde, 0xa7, 0x71, 0xb8, 0xce,
	0xa6, 0x3f, 0xcc, 0x78, 0xbc, 0xa5, 0x91, 0x7d, 0x0e, 0x8d, 0x16, 0x65,
	0x00, 0x00, 0x00, 0x00, 0x63, 0x9e, 0x9e, 0x20, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xe8, 0xb7, 0x37, 0x97, 0x4e, 0x1b, 0xfb, 0x82, 0x16, 0x71, 0x58, 0x39,
	0x55, 0xb6, 0x32, 0xb8, 0xf6, 0x75, 0x2f, 0xdd, 0x34, 0x7a, 0x0d, 0x7b,
	0x00, 0x00, 0x00, 0x00, 0xff, 0x24, 0xb5, 0x67, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xea, 0xce, 0x1f, 0x14, 0x00, 0xe2, 0xe0, 0x16, 0x71, 0xe5, 0x9b, 0x8c,
	0x60, 0x8d, 0x20, 0x97, 0x2b, 0x07, 0xec, 0x89, 0x89, 0x37, 0x5d, 0x09,
	0x00, 0x00, 0x00, 0x00, 0xc3, 0xe2, 0x8c, 0xdc, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x9a, 0x88, 0x5f, 0xdf, 0xf5, 0x12, 0x07, 0xbf, 0xc5, 0xbd, 0xd8, 0x15,
	0x01, 0x55, 0x75, 0x02, 0xf1, 0x96, 0x50, 0x03, 0x6b, 0xbd, 0xd0, 0x72,
	0x00, 0x00, 0x00, 0x00, 0xda, 0x9b, 0x7e, 0x0c, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xf2, 0xa7, 0x4c, 0xce, 0x47, 0xb9, 0x7d, 0x42, 0xf7, 0xb5, 0x5d, 0x63,
	0x28, 0xf5, 0xde, 0x75, 0x06, 0x13, 0xe9, 0x5d, 0x9b, 0x48, 0x88, 0x67,
	0x00, 0x00, 0x00, 0x00, 0xd6, 0x05, 0x07, 0x13, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xa4, 0xee, 0x4e, 0xe7, 0xb4, 0x4a, 0xda, 0x48, 0xe3, 0x67, 0x5f, 0xf9,
	0x15, 0xda, 0x1a, 0x27, 0x33, 0x6b, 0x97, 0x6a, 0x82, 0x0d, 0xa0, 0x86,
	0x00, 0x00, 0x00, 0x00, 0x5b, 0x6e, 0x9f, 0xfd, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xf5, 0x2c, 0xd8, 0x87, 0x2f, 0xf4, 0xa8, 0x85, 0x53, 0x29, 0x86, 0xf1,
	0xb9, 0x1a, 0x28, 0xdc, 0xaf, 0x35, 0x40, 0xdd, 0x75, 0xbf, 0x86, 0x56,
	0x00, 0x00, 0x00, 0x00, 0xce, 0x4f, 0x78, 0xa9, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x4d, 0x51, 0xfa, 0xb9, 0xe8, 0x49, 0xeb, 0x79, 0xd4, 0xc3, 0x0d, 0x04,
	0xdd, 0xf6, 0xb1, 0xcd, 0x6d, 0x51, 0xe5, 0x58, 0x06, 0x2a, 0x7e, 0x65,
	0x00, 0x00, 0x00, 0x00, 0xe6, 0xe4, 0xf0, 0x9a, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xe0, 0x85, 0xf1, 0x4f, 0xe7, 0x07, 0x60, 0xb5, 0x45, 0x7a, 0xc8, 0x3d,
	0xf3, 0x26, 0xd8, 0xb0, 0xc7, 0x94, 0x33, 0x72, 0x7e, 0xef, 0x87, 0x70,
	0x00, 0x00, 0x00, 0x00, 0x74, 0x6d, 0x83, 0xb0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x7a, 0x24, 0xee, 0x7e, 0x21, 0x7b, 0x0a, 0x33, 0x4e, 0xa1, 0x1e, 0x5c,
	0xca, 0x30, 0xf9, 0x68, 0xc2, 0xc4, 0x7d, 0x0a, 0xa0, 0xc7, 0xbe, 0xeb,
	0x00, 0x00, 0x00, 0x00, 0xca, 0x62, 0xa6, 0x8b, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x5c, 0x26, 0x7c, 0x9c, 0xfb, 0xe5, 0x6a, 0x33, 0x89, 0x37, 0x74, 0x21,
	0x13, 0x71, 0x46, 0x6b, 0x60, 0xd0, 0x38, 0xc4, 0x90, 0xef, 0x7d, 0xec,
	0x00, 0x00, 0x00, 0x00, 0x27, 0xde, 0xcf, 0x82, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x23, 0x14, 0xf5, 0x42, 0xbe, 0x9e, 0x07, 0x9c, 0x4a, 0x60, 0x56, 0x3b,
	0xcf, 0xe2, 0x06, 0x81, 0xb0, 0xc0, 0x46, 0x49, 0xfb, 0x97, 0x61, 0x5a,
	0x00, 0x00, 0x00, 0x00, 0x2f, 0xa4, 0x2d, 0x2b, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x86, 0xe5, 0x99, 0xb4, 0xcf, 0x64, 0x05, 0x25, 0xa3, 0x44, 0xe4, 0x18,
	0x5f, 0x72, 0x58, 0x47, 0x7f, 0xbd, 0x84, 0xd7, 0x0a, 0x38, 0xa0, 0xd4,
	0x00, 0x00, 0x00, 0x00, 0x11, 0x8d, 0x15, 0xd5, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xe8, 0x5b, 0x13, 0xf0, 0x09, 0xc6, 0x14, 0x58, 0xc4, 0xf3, 0x1b, 0x3d,
	0x17, 0x0f, 0x1c, 0xfa, 0x7d, 0x61, 0x7e, 0x7e, 0x9c, 0xea, 0x52, 0x0a,
	0x00, 0x00, 0x00, 0x00, 0x89, 0x05, 0xdb, 0xb7, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x26, 0xf8, 0xab, 0x68, 0xb4, 0x27, 0x56, 0x1d, 0x04, 0xed, 0x8c, 0x65,
	0xfd, 0xd1, 0x62, 0x2e, 0x80, 0x4c, 0x4a, 0x1d, 0x67, 0x90, 0x50, 0xed,
	0x00, 0x00, 0x00, 0x00, 0x3b, 0x02, 0x2a, 0x4b, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x11, 0xe9, 0x36, 0x25, 0x8d, 0x97, 0x84, 0xea, 0xe9, 0x50, 0x4a, 0x27,
	0x66, 0x6e, 0x0c, 0xd2, 0xce, 0x40, 0xfe, 0xfb, 0xf2, 0xc6, 0x53, 0xb4,
	0x00, 0x00, 0x00, 0x00, 0xb6, 0x32, 0x22, 0x76, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xfc, 0x88, 0xcf, 0x76, 0x3c, 0x0b, 0x0d, 0x76, 0xb2, 0xc3, 0xc7, 0x8c,
	0x8c, 0x53, 0x5f, 0x4c, 0xba, 0x0d, 0x13, 0xdb, 0x7b, 0xac, 0xf0, 0x19,
	0x00, 0x00, 0x00, 0x00, 0x7e, 0x11, 0x95, 0x29, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
};

static const uint8_t fpm_table_p256[CPT_P256_LEN] = {
	0xf4, 0xa1, 0x39, 0x45, 0xd8, 0x98, 0xc2, 0x96, 0x77, 0x03, 0x7d, 0x81,
	0x2d, 0xeb, 0x33, 0xa0, 0xf8, 0xbc, 0xe6, 0xe5, 0x63, 0xa4, 0x40, 0xf2,
	0x6b, 0x17, 0xd1, 0xf2, 0xe1, 0x2c, 0x42, 0x47, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xcb, 0xb6, 0x40, 0x68, 0x37, 0xbf, 0x51, 0xf5, 0x2b, 0xce, 0x33, 0x57,
	0x6b, 0x31, 0x5e, 0xce, 0x8e, 0xe7, 0xeb, 0x4a, 0x7c, 0x0f, 0x9e, 0x16,
	0x4f, 0xe3, 0x42, 0xe2, 0xfe, 0x1a, 0x7f, 0x9b, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x90, 0xe7, 0x5c, 0xb4, 0x8e, 0x14, 0xdb, 0x63, 0x29, 0x49, 0x3b, 0xaa,
	0xad, 0x65, 0x1f, 0x7e, 0x84, 0x92, 0x59, 0x2e, 0x32, 0x6e, 0x25, 0xde,
	0x0f, 0xa8, 0x22, 0xbc, 0x28, 0x11, 0xaa, 0xa5, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xe4, 0x11, 0x24, 0x54, 0x5f, 0x46, 0x2e, 0xe7, 0x34, 0xb1, 0xa6, 0x50,
	0x50, 0xfe, 0x82, 0xf5, 0x6f, 0x4a, 0xd4, 0xbc, 0xb3, 0xdf, 0x18, 0x8b,
	0xbf, 0xf4, 0x4a, 0xe8, 0xf5, 0xdb, 0xa8, 0x0d, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x93, 0x39, 0x1c, 0xe2, 0x09, 0x79, 0x92, 0xaf, 0xe9, 0x6c, 0x98, 0xfd,
	0x0d, 0x35, 0xf1, 0xfa, 0xb2, 0x57, 0xc0, 0xde, 0x95, 0xe0, 0x27, 0x89,
	0x30, 0x0a, 0x4b, 0xbc, 0x89, 0xd6, 0x72, 0x6f, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xaa, 0x54, 0xa2, 0x91, 0xc0, 0x81, 0x27, 0xa0, 0x5b, 0xb1, 0xee, 0xad,
	0xa9, 0xd8, 0x06, 0xa5, 0x7f, 0x1d, 0xdb, 0x25, 0xff, 0x1e, 0x3c, 0x6f,
	0x72, 0xaa, 0xc7, 0xe0, 0xd0, 0x9b, 0x46, 0x44, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x57, 0xc8, 0x4f, 0xc9, 0xd7, 0x89, 0xbd, 0x85, 0xfc, 0x35, 0xff, 0x7d,
	0xc2, 0x97, 0xea, 0xc3, 0xfb, 0x98, 0x2f, 0xd5, 0x88, 0xc6, 0x76, 0x6e,
	0x44, 0x7d, 0x73, 0x9b, 0xee, 0xdb, 0x5e, 0x67, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x0c, 0x7e, 0x33, 0xc9, 0x72, 0xe2, 0x5b, 0x32, 0x3d, 0x34, 0x9b, 0x95,
	0xa7, 0xfa, 0xe5, 0x00, 0xe1, 0x2e, 0x9d, 0x95, 0x3a, 0x4a, 0xaf, 0xf7,
	0x2d, 0x48, 0x25, 0xab, 0x83, 0x41, 0x31, 0xee, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x13, 0x94, 0x9c, 0x93, 0x2a, 0x1d, 0x36, 0x7f, 0xef, 0x7f, 0xbd, 0x2b,
	0x1a, 0x0a, 0x11, 0xb7, 0xdd, 0xc6, 0x06, 0x8b, 0xb9, 0x1d, 0xfc, 0x60,
	0xef, 0x95, 0x19, 0x32, 0x8a, 0x9c, 0x72, 0xff, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x19, 0x60, 0x35, 0xa7, 0x73, 0x76, 0xd8, 0xa8, 0x23, 0x18, 0x3b, 0x08,
	0x95, 0xca, 0x17, 0x40, 0xc1, 0xee, 0x98, 0x07, 0x02, 0x2c, 0x21, 0x9c,
	0x61, 0x1e, 0x9f, 0xc3, 0x7d, 0xbb, 0x2c, 0x9b, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xca, 0xe2, 0xb1, 0x92, 0x0b, 0x57, 0xf4, 0xbc, 0x29, 0x36, 0xdf, 0x5e,
	0xc6, 0xc9, 0xbc, 0x36, 0x7d, 0xea, 0x64, 0x82, 0xe1, 0x12, 0x38, 0xbf,
	0x55, 0x06, 0x63, 0x79, 0x7b, 0x51, 0xf5, 0xd8, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x44, 0xff, 0xe2, 0x16, 0x34, 0x8a, 0x96, 0x4c, 0x9f, 0xb3, 0xd5, 0x76,
	0xdb, 0xde, 0xfb, 0xe1, 0x0a, 0xfa, 0x40, 0x01, 0x8d, 0x9d, 0x50, 0xe5,
	0x15, 0x71, 0x64, 0x84, 0x8a, 0xec, 0xb8, 0x51, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xe4, 0x8e, 0xca, 0xff, 0xfc, 0x5c, 0xde, 0x01, 0x7c, 0xcd, 0x84, 0xe7,
	0x0d, 0x71, 0x5f, 0x26, 0xa2, 0xe8, 0xf4, 0x83, 0xf4, 0x3e, 0x43, 0x91,
	0xeb, 0x5d, 0x77, 0x45, 0xb2, 0x11, 0x41, 0xea, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xca, 0xc9, 0x17, 0xe2, 0x73, 0x1a, 0x34, 0x79, 0x85, 0xf2, 0x2c, 0xfe,
	0x28, 0x44, 0xb6, 0x45, 0x09, 0x90, 0xe6, 0xa1, 0x58, 0x00, 0x6c, 0xee,
	0xea, 0xfd, 0x72, 0xeb, 0xdb, 0xec, 0xc1, 0x7b, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x6c, 0xf2, 0x0f, 0xfb, 0x31, 0x37, 0x28, 0xbe, 0x96, 0x43, 0x95, 0x91,
	0xa3, 0xc6, 0xb9, 0x4a, 0x27, 0x36, 0xff, 0x83, 0x44, 0x31, 0x5f, 0xc5,
	0xa6, 0xd3, 0x96, 0x77, 0xa7, 0x84, 0x92, 0x76, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xf2, 0xba, 0xb8, 0x33, 0xc3, 0x57, 0xf5, 0xf4, 0x82, 0x4a, 0x92, 0x0c,
	0x22, 0x84, 0x05, 0x9b, 0x66, 0xb8, 0xba, 0xbd, 0x2d, 0x27, 0xec, 0xdf,
	0x67, 0x4f, 0x84, 0x74, 0x9b, 0x0b, 0x88, 0x16, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x2d, 0xf4, 0x8c, 0x04, 0x67, 0x7c, 0x8a, 0x3e, 0x74, 0xe0, 0x2f, 0x08,
	0x02, 0x03, 0xa5, 0x6b, 0x31, 0x85, 0x5f, 0x7d, 0xb8, 0xc7, 0xfe, 0xdb,
	0x4e, 0x76, 0x9e, 0x76, 0x72, 0xc9, 0xdd, 0xad, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xa4, 0xc3, 0x61, 0x65, 0xb8, 0x24, 0xbb, 0xb0, 0xfb, 0x9a, 0xe1, 0x6f,
	0x3b, 0x91, 0x22, 0xa5, 0x1e, 0xc0, 0x05, 0x72, 0x06, 0x94, 0x72, 0x81,
	0x42, 0xb9, 0x90, 0x82, 0xde, 0x83, 0x06, 0x63, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x6e, 0xf9, 0x51, 0x50, 0xdd, 0xa8, 0x68, 0xb9, 0xd1, 0xf8, 0x9e, 0x79,
	0x9c, 0x0c, 0xe1, 0x31, 0x7f, 0xdc, 0x1c, 0xa0, 0x08, 0xa1, 0xc4, 0x78,
	0x78, 0x87, 0x8e, 0xf6, 0x1c, 0x6c, 0xe0, 0x4d, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x9c, 0x62, 0xb9, 0x12, 0x1f, 0xe0, 0xd9, 0x76, 0x6a, 0xce, 0x57, 0x0e,
	0xbd, 0xe0, 0x8d, 0x4f, 0xde, 0x53, 0x14, 0x2c, 0x12, 0x30, 0x9d, 0xef,
	0xb6, 0xcb, 0x3f, 0x5d, 0x7b, 0x72, 0xc3, 0x21, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x7f, 0x99, 0x1e, 0xd2, 0xc3, 0x1a, 0x35, 0x73, 0x5b, 0x82, 0xdd, 0x5b,
	0xd5, 0x4f, 0xb4, 0x96, 0x59, 0x5c, 0x52, 0x20, 0x81, 0x2f, 0xfc, 0xae,
	0x0c, 0x88, 0xbc, 0x4d, 0x71, 0x6b, 0x12, 0x87, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x3a, 0x57, 0xbf, 0x63, 0x5f, 0x48, 0xac, 0xa8, 0x7c, 0x81, 0x81, 0xf4,
	0xdf, 0x25, 0x64, 0xf3, 0x18, 0xd1, 0xb5, 0xb3, 0x9c, 0x04, 0xe6, 0xaa,
	0xdd, 0x5d, 0xde, 0xa3, 0xf3, 0x90, 0x1d, 0xc6, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xe9, 0x6a, 0x79, 0xfb, 0x3e, 0x72, 0xad, 0x0c, 0x43, 0xa0, 0xa2, 0x8c,
	0x42, 0xba, 0x79, 0x2f, 0xef, 0xe0, 0xa4, 0x23, 0x08, 0x3e, 0x49, 0xf3,
	0x68, 0xf3, 0x44, 0xaf, 0x6b, 0x31, 0x74, 0x66, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xcd, 0xfe, 0x17, 0xdb, 0x3f, 0xb2, 0x4d, 0x4a, 0x66, 0x8b, 0xfc, 0x22,
	0x71, 0xf5, 0xc6, 0x26, 0x60, 0x4e, 0xd9, 0x3c, 0x24, 0xd6, 0x7f, 0xf3,
	0x31, 0xb9, 0xc4, 0x05, 0xf8, 0x54, 0x0a, 0x20, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xd3, 0x6b, 0x47, 0x89, 0xa2, 0x58, 0x2e, 0x7f, 0x0d, 0x1a, 0x10, 0x14,
	0x4e, 0xc3, 0x9c, 0x28, 0x66, 0x3c, 0x62, 0xc3, 0xed, 0xba, 0xd7, 0xa0,
	0x40, 0x52, 0xbf, 0x4b, 0x6f, 0x46, 0x1d, 0xb9, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x23, 0x5a, 0x27, 0xc3, 0x18, 0x8d, 0x25, 0xeb, 0xe7, 0x24, 0xf3, 0x39,
	0x99, 0xbf, 0xcc, 0x5b, 0x86, 0x2b, 0xe6, 0xbd, 0x71, 0xd7, 0x0c, 0xc8,
	0xfe, 0xcf, 0x4d, 0x51, 0x90, 0xb0, 0xfc, 0x61, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x74, 0x34, 0x6c, 0x10, 0xa1, 0xd4, 0xcf, 0xac, 0xaf, 0xdf, 0x5c, 0xc0,
	0x85, 0x26, 0xa7, 0xa4, 0x12, 0x32, 0x02, 0xa8, 0xf6, 0x2b, 0xff, 0x7a,
	0x1e, 0xdd, 0xba, 0xe2, 0xc8, 0x02, 0xe4, 0x1a, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x8f, 0xa0, 0xaf, 0x2d, 0xd6, 0x03, 0xf8, 0x44, 0x36, 0xe0, 0x6b, 0x7e,
	0x4c, 0x70, 0x19, 0x17, 0x0c, 0x45, 0xf4, 0x52, 0x73, 0xdb, 0x33, 0xa0,
	0x43, 0x10, 0x4d, 0x86, 0x56, 0x0e, 0xbc, 0xfc, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x96, 0x15, 0xb5, 0x11, 0x0d, 0x1d, 0x78, 0xe5, 0x66, 0xb0, 0xde, 0x32,
	0x25, 0xc4, 0x74, 0x4b, 0x0a, 0x4a, 0x46, 0xfb, 0x6a, 0xaf, 0x36, 0x3a,
	0xb4, 0x8e, 0x26, 0xb4, 0x84, 0xf7, 0xa2, 0x1c, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x06, 0xeb, 0xb0, 0xf6, 0x21, 0xa0, 0x1b, 0x2d, 0xc0, 0x04, 0xe4, 0x04,
	0x8b, 0x7b, 0x0f, 0x98, 0x64, 0x13, 0x1b, 0xcd, 0xfe, 0xd6, 0xf6, 0x68,
	0xfa, 0xc0, 0x15, 0x40, 0x4d, 0x4d, 0x3d, 0xab, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
};

static const uint8_t fpm_table_p384[CPT_P384_LEN] = {
	0x3a, 0x54, 0x5e, 0x38, 0x72, 0x76, 0x0a, 0xb7, 0x55, 0x02, 0xf2, 0x5d,
	0xbf, 0x55, 0x29, 0x6c, 0x59, 0xf7, 0x41, 0xe0, 0x82, 0x54, 0x2a, 0x38,
	0x6e, 0x1d, 0x3b, 0x62, 0x8b, 0xa7, 0x9b, 0x98, 0x8e, 0xb1, 0xc7, 0x1e,
	0xf3, 0x20, 0xad, 0x74, 0xaa, 0x87, 0xca, 0x22, 0xbe, 0x8b, 0x05, 0x37,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7a, 0x43, 0x1d, 0x7c,
	0x90, 0xea, 0x0e, 0x5f, 0x0a, 0x60, 0xb1, 0xce, 0x1d, 0x7e, 0x81, 0x9d,
	0xe9, 0xda, 0x31, 0x13, 0xb5, 0xf0, 0xb8, 0xc0, 0xf8, 0xf4, 0x1d, 0xbd,
	0x28, 0x9a, 0x14, 0x7c, 0x5d, 0x9e, 0x98, 0xbf, 0x92, 0x92, 0xdc, 0x29,
	0x36, 0x17, 0xde, 0x4a, 0x96, 0x26, 0x2c, 0x6f, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x39, 0xc1, 0xb3, 0x28, 0xd8, 0xee, 0x21, 0xc9, 0x2c, 0x3e, 0x0c, 0x91,
	0x55, 0x87, 0x17, 0xdb, 0x4b, 0x58, 0x80, 0x8b, 0x3f, 0x86, 0x86, 0xa9,
	0x43, 0x60, 0x39, 0x09, 0x18, 0x14, 0x1b, 0x1a, 0xd6, 0xe9, 0x8b, 0x0d,
	0x37, 0xca, 0x7a, 0xbc, 0xf5, 0x32, 0x38, 0x9a, 0x06, 0x0c, 0xbd, 0x1b,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7a, 0x7e, 0x18, 0x39,
	0x23, 0xd8, 0x6e, 0xcd, 0x31, 0xea, 0x31, 0xb1, 0x08, 0x5a, 0x4e, 0x9a,
	0xbc, 0x40, 0xce, 0x5a, 0xbe, 0x64, 0x36, 0x03, 0xbd, 0x22, 0xcf, 0xb2,
	0xa2, 0x12, 0x41, 0x63, 0x6f, 0x04, 0xca, 0xa2, 0xde, 0x3a, 0x82, 0xba,
	0xb9, 0xd2, 0x85, 0x2c, 0xc3, 0xb3, 0x8e, 0x69, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x26, 0x4e, 0x52, 0x46, 0xeb, 0x09, 0xa0, 0xe5, 0xf8, 0xf4, 0xbe, 0x11,
	0x32, 0xcd, 0xf0, 0x3c, 0xda, 0x9d, 0x54, 0x83, 0x5f, 0xae, 0xfa, 0x4f,
	0xbb, 0xbc, 0x4f, 0xd0, 0x17, 0xa3, 0x1b, 0x22, 0xc3, 0xde, 0xcd, 0x0c,
	0x86, 0xf0, 0x61, 0x45, 0x52, 0x8e, 0xf1, 0x67, 0x0a, 0x5f, 0x2c, 0xab,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8a, 0x1e, 0x98, 0x58,
	0xc1, 0x4f, 0x0d, 0xd6, 0x55, 0x05, 0x38, 0xa8, 0x09, 0xcb, 0x75, 0x24,
	0xbd, 0x60, 0xca, 0xb4, 0xc8, 0x7f, 0xed, 0x22, 0xf8, 0xb7, 0x6f, 0xdd,
	0x63, 0x1d, 0x05, 0x8d, 0x58, 0x03, 0xea, 0xa1, 0x1a, 0x1d, 0xcf, 0x14,
	0x7b, 0x9b, 0x1f, 0xbe, 0x7b, 0xcc, 0xf5, 0x6c, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xa6, 0x28, 0xb0, 0x9a, 0xaa, 0x03, 0xbd, 0x53, 0xba, 0x06, 0x54, 0x58,
	0xa4, 0xf5, 0x2d, 0x78, 0xdb, 0x29, 0x87, 0x89, 0x4d, 0x10, 0xdd, 0xea,
	0xb4, 0x2a, 0x31, 0xaf, 0x8a, 0x3e, 0x29, 0x7d, 0x40, 0xf7, 0xf9, 0xe7,
	0x06, 0x42, 0x12, 0x79, 0xc1, 0x9e, 0x0b, 0x4c, 0x80, 0x01, 0x19, 0xc4,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x82, 0x2d, 0x0f, 0xc5,
	0xe6, 0xc8, 0x8c, 0x41, 0xaf, 0x68, 0xaa, 0x6d, 0xe6, 0x39, 0xd8, 0x58,
	0xc1, 0xc7, 0xca, 0xd1, 0x35, 0xf6, 0xeb, 0xf2, 0x57, 0x7a, 0x30, 0xea,
	0xe3, 0x56, 0x7a, 0xf9, 0xe5, 0xa0, 0x19, 0x1d, 0x1f, 0x5b, 0x77, 0xf6,
	0x16, 0xf3, 0xfd, 0xbf, 0x03, 0x56, 0xb3, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x30, 0x99, 0x15, 0x60, 0xaa, 0x13, 0x39, 0x09, 0x90, 0x97, 0xdb, 0xb1,
	0xc6, 0xcb, 0x00, 0x17, 0xd3, 0x7d, 0xe4, 0x24, 0xb8, 0x60, 0xfa, 0xe6,
	0x9b, 0xb1, 0x83, 0xb2, 0x70, 0xb3, 0x75, 0xdd, 0x56, 0x7a, 0x62, 0x33,
	0xcd, 0x6c, 0xe3, 0xa3, 0xaa, 0xb8, 0xbb, 0x9f, 0x0f, 0xdc, 0x30, 0x88,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x16, 0xc5, 0xb9, 0x81,
	0x60, 0x0a, 0xd5, 0xa6, 0xeb, 0xdf, 0x73, 0xf2, 0xd6, 0x2f, 0xaa, 0x44,
	0x6d, 0x95, 0x5b, 0xb3, 0xc9, 0x74, 0x7b, 0xf3, 0xf6, 0x00, 0x5f, 0xc8,
	0x15, 0xeb, 0x04, 0xac, 0xf0, 0xaf, 0x01, 0xd1, 0x28, 0x20, 0x50, 0xb5,
	0x48, 0x94, 0x2f, 0x81, 0x31, 0x4f, 0x6d, 0x28, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x22, 0x11, 0x21, 0x77, 0x16, 0x60, 0x5e, 0x23, 0x47, 0xd2, 0xc8,
	0x9e, 0xf2, 0x81, 0xc8, 0x54, 0xba, 0x45, 0x99, 0x56, 0x7d, 0x63, 0x42,
	0xce, 0x0f, 0xba, 0x30, 0x77, 0xc0, 0xf0, 0x3f, 0x70, 0x22, 0xf8, 0x02,
	0xcb, 0x36, 0x74, 0x44, 0x73, 0x34, 0xa9, 0x36, 0xa9, 0xa6, 0xa0, 0x52,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb5, 0x46, 0x1f, 0x68,
	0xd6, 0x58, 0xa0, 0x1a, 0x0a, 0x64, 0xd5, 0x19, 0xc2, 0xbd, 0x0e, 0xfa,
	0x9e, 0x2e, 0xee, 0x8f, 0x69, 0x7a, 0x92, 0x80, 0x8e, 0x5d, 0x9b, 0x89,
	0x7d, 0x0e, 0x01, 0x7a, 0x1f, 0x7c, 0x5c, 0x36, 0x7c, 0xbd, 0x4c, 0xcd,
	0x7f, 0xfc, 0xef, 0xf7, 0xf6, 0x32, 0xc9, 0x26, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x30, 0x0a, 0xe2, 0xe6, 0x0e, 0x75, 0x83, 0x44, 0x45, 0x1c, 0x70, 0x7a,
	0x37, 0x1a, 0x2c, 0xa5, 0x25, 0x65, 0x1d, 0x10, 0x50, 0x52, 0xdd, 0x32,
	0xbf, 0x88, 0xde, 0x7f, 0x48, 0x62, 0xb9, 0x54, 0xfa, 0xfc, 0xe2, 0x6e,
	0x03, 0x81, 0xef, 0x13, 0xdc, 0x91, 0x6c, 0x17, 0x96, 0x0e, 0x09, 0x0e,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xed, 0x17, 0xcc, 0x44,
	0x02, 0x6b, 0x08, 0x89, 0x95, 0xc0, 0x1f, 0xf1, 0x9b, 0x42, 0x44, 0x1b,
	0x40, 0x89, 0x64, 0x78, 0xcc, 0x16, 0x06, 0x97, 0x52, 0xd1, 0x54, 0xb8,
	0x0b, 0xa0, 0x4a, 0x35, 0xb3, 0xd9, 0x2e, 0xa4, 0x70, 0x1c, 0x29, 0x52,
	0x26, 0x6e, 0x8a, 0x40, 0xd6, 0x9e, 0xca, 0x0a, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xe4, 0xbf, 0xc2, 0xc0, 0x49, 0x05, 0xca, 0x71, 0xf3, 0x3a, 0x45, 0x0a,
	0xd1, 0x56, 0xf7, 0x61, 0x3d, 0x8b, 0x29, 0xdb, 0xd0, 0x88, 0x48, 0xc2,
	0x09, 0x7d, 0xa3, 0x95, 0xa2, 0x30, 0x96, 0x86, 0x21, 0x19, 0x05, 0x03,
	0x5f, 0x49, 0x72, 0xd7, 0xb2, 0xd1, 0x05, 0x58, 0x17, 0xcb, 0xaa, 0x12,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xdd, 0xce, 0xbb, 0x55,
	0x75, 0x3e, 0xe3, 0x24, 0xe8, 0x7a, 0xb0, 0x7c, 0x69, 0x24, 0x66, 0x6f,
	0x9b, 0x47, 0x5d, 0x74, 0x4e, 0xcf, 0x1a, 0x68, 0xf8, 0x2b, 0xe8, 0xf5,
	0x2e, 0x62, 0x36, 0xc0, 0x23, 0x7c, 0x0d, 0xba, 0x3c, 0xfd, 0x05, 0x6b,
	0x35, 0x4c, 0xd8, 0x72, 0xc3, 0xc6, 0xcb, 0xd2, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x8d, 0x10, 0x4d, 0x24, 0x70, 0x8d, 0x4c, 0xee, 0x19, 0x7d, 0x69, 0x58,
	0x81, 0x9c, 0xf0, 0x43, 0x47, 0xfc, 0x87, 0xfa, 0xf0, 0x71, 0x22, 0x10,
	0x10, 0x3d, 0xf7, 0x85, 0x5c, 0x20, 0x15, 0x58, 0x30, 0xb0, 0xa9, 0xe8,
	0x61, 0x1e, 0xf6, 0x38, 0x00, 0xb1, 0x9a, 0xc8, 0xfd, 0xfe, 0xbf, 0xec,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd4, 0x0e, 0x8d, 0x6f,
	0xd2, 0x01, 0xe0, 0x3e, 0xbb, 0x7c, 0x96, 0x9c, 0x22, 0x28, 0xff, 0x5f,
	0x68, 0x81, 0x02, 0x82, 0x63, 0x61, 0x64, 0xc5, 0xcd, 0xbb, 0x3c, 0xd2,
	0xe7, 0x54, 0x22, 0x0d, 0x14, 0x18, 0xfe, 0x25, 0xe9, 0xf6, 0xed, 0xc4,
	0xa7, 0x2f, 0x91, 0x05, 0x9e, 0xe3, 0x60, 0x31, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x64, 0xd2, 0xc2, 0x73, 0xb7, 0x69, 0x73, 0x7a, 0x2c, 0xc0, 0x24, 0x51,
	0x97, 0xd5, 0x3f, 0xfd, 0xc3, 0xb6, 0xac, 0x4b, 0xe8, 0x6c, 0x46, 0xbd,
	0x17, 0xe9, 0x41, 0x1f, 0x68, 0x5e, 0x92, 0x6d, 0x13, 0x6d, 0xf3, 0x6b,
	0x75, 0x20, 0x3a, 0x36, 0x3f, 0x95, 0x61, 0xe0, 0x8b, 0xf0, 0xb2, 0x7e,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xdd, 0x6f, 0xf8, 0xd5,
	0x27, 0xe9, 0x90, 0xa7, 0xc3, 0x4b, 0xe5, 0x86, 0xf9, 0x86, 0x7a, 0x60,
	0xea, 0x08, 0x87, 0x47, 0x85, 0x54, 0xe0, 0x14, 0xcf, 0xce, 0xd6, 0x64,
	0x6f, 0x52, 0xe4, 0xcb, 0x4b, 0x1a, 0x5a, 0x20, 0x41, 0x2a, 0xb6, 0x41,
	0x0b, 0x06, 0xf0, 0x06, 0x39, 0x62, 0x95, 0x87, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x04, 0x4c, 0x0d, 0xd2, 0x85, 0x65, 0x1f, 0x82, 0x32, 0x5c, 0x51, 0xe7,
	0x78, 0x5d, 0x3e, 0xf7, 0xb8, 0x3a, 0x18, 0x61, 0x88, 0xe9, 0x55, 0x32,
	0x53, 0x9f, 0x94, 0xad, 0x52, 0x2c, 0x29, 0x31, 0x15, 0x27, 0x4e, 0x5b,
	0x89, 0x80, 0xf1, 0x37, 0x9f, 0xd7, 0xb0, 0x10, 0xdf, 0x0f, 0x66, 0xd7,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe4, 0xa7, 0xb9, 0x4a,
	0x40, 0x64, 0xe4, 0xc0, 0xd4, 0x4e, 0xba, 0x45, 0x25, 0xd7, 0xd2, 0x11,
	0x0a, 0x80, 0x6b, 0x54, 0xbe, 0x8a, 0x04, 0xe3, 0x92, 0x92, 0x26, 0xbd,
	0x14, 0x90, 0x33, 0xde, 0x79, 0x5f, 0x6f, 0xa3, 0xc9, 0x73, 0x92, 0x46,
	0x32, 0x1a, 0xa9, 0xa3, 0xb9, 0x26, 0x02, 0x25, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x49, 0xbc, 0xc2, 0xf5, 0x8b, 0x70, 0x7b, 0x8e, 0x29, 0x01, 0xb5, 0x19,
	0x1d, 0x92, 0x89, 0x83, 0x2e, 0x4c, 0x29, 0x56, 0x7d, 0x49, 0xc7, 0x80,
	0xeb, 0xd1, 0xcf, 0xf8, 0x4c, 0x6a, 0x99, 0x64, 0x2c, 0xae, 0xbb, 0xd3,
	0x16, 0xee, 0x3e, 0x13, 0x36, 0xa5, 0x43, 0xee, 0xa8, 0x7a, 0x68, 0xf7,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x75, 0xb4, 0x1c, 0x29,
	0xb5, 0x69, 0x94, 0x6d, 0x15, 0x10, 0xe7, 0xd4, 0x3e, 0xf2, 0x26, 0x7e,
	0x91, 0x23, 0x50, 0x72, 0xd4, 0xb3, 0x39, 0x4d, 0x58, 0xea, 0xff, 0x04,
	0x8f, 0xbd, 0x85, 0xd1, 0xd3, 0x49, 0xab, 0x03, 0x78, 0xa6, 0x78, 0x47,
	0xf2, 0x77, 0xba, 0xcd, 0xa5, 0x0e, 0xe4, 0x1c, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0xb0, 0x56, 0x58, 0x5f, 0x86, 0x3b, 0xbd, 0xe9, 0x2c, 0xdc, 0x5a,
	0xb4, 0x83, 0x28, 0x3d, 0xeb, 0xb3, 0x12, 0x09, 0xdc, 0x7c, 0x42, 0x1d,
	0x3a, 0xfc, 0xbd, 0x79, 0x6d, 0x01, 0xa5, 0xa8, 0xe2, 0xb0, 0x67, 0xca,
	0xa0, 0x8b, 0x6a, 0x51, 0x02, 0x6e, 0x0d, 0xc2, 0xe8, 0xcb, 0x7a, 0xeb,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0xc3, 0x50, 0x29,
	0x02, 0xdd, 0xe1, 0x8a, 0x64, 0xc1, 0x5f, 0xac, 0xd8, 0xc6, 0xcf, 0x36,
	0x17, 0xea, 0x27, 0x01, 0x10, 0x78, 0x1e, 0x45, 0xd6, 0x8d, 0x1f, 0xfc,
	0x1f, 0x34, 0x43, 0xd8, 0x4b, 0xe2, 0x56, 0x37, 0x8c, 0x74, 0x61, 0xa5,
	0xae, 0x88, 0x66, 0xba, 0xd8, 0xef, 0x24, 0xe1, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xac, 0x3a, 0x78, 0xd0, 0xd2, 0x65, 0xa9, 0x1c, 0x1a, 0x29, 0xf8, 0xef,
	0x6c, 0x8f, 0x83, 0xd3, 0xef, 0x98, 0xfd, 0xde, 0x8f, 0xd8, 0xd8, 0x17,
	0xdf, 0x45, 0x9e, 0xa1, 0xc4, 0x2b, 0xf7, 0x48, 0x14, 0xda, 0xfc, 0x39,
	0x81, 0xa7, 0x3d, 0xc7, 0xb0, 0x3d, 0xfa, 0x54, 0xc5, 0x2a, 0xfa, 0x2d,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xcc, 0x40, 0x6f, 0x6e,
	0x6c, 0x0d, 0x2c, 0xe7, 0xcd, 0x12, 0x0b, 0x2b, 0x41, 0xfd, 0x72, 0xca,
	0xef, 0x5d, 0x90, 0x06, 0x78, 0xf6, 0x02, 0xdd, 0xf5, 0xf8, 0xa2, 0xd1,
	0x8a, 0xcc, 0xf2, 0x29, 0xaa, 0xfd, 0x1f, 0xcf, 0xce, 0x6d, 0x90, 0x8a,
	0x2c, 0xe2, 0x88, 0x5a, 0x0e, 0x6d, 0x85, 0xf2, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x89, 0x10, 0x9a, 0x0e, 0xc6, 0x26, 0x66, 0xde, 0xc8, 0xc1, 0x2e, 0x75,
	0x7f, 0xfc, 0xd0, 0x1e, 0xa8, 0x20, 0x61, 0x69, 0xc4, 0x8b, 0x5a, 0xb0,
	0x4b, 0xc2, 0xfd, 0xcf, 0xf9, 0x83, 0xac, 0x6c, 0x59, 0xcf, 0xca, 0x71,
	0x55, 0x97, 0x7d, 0x23, 0x12, 0x64, 0xcb, 0x33, 0x57, 0x66, 0xc9, 0x6a,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6b, 0x69, 0x13, 0x81,
	0x2e, 0x01, 0x4b, 0x4b, 0x31, 0xd2, 0x87, 0x07, 0xe4, 0x48, 0x3e, 0xc5,
	0xcb, 0xf7, 0x19, 0x0c, 0xff, 0xb1, 0x97, 0x58, 0xb6, 0x67, 0x17, 0xa0,
	0x65, 0xa5, 0xf2, 0x48, 0xd9, 0x4a, 0xd8, 0xfa, 0xc5, 0x3b, 0x4f, 0x69,
	0x11, 0x9e, 0xbe, 0xee, 0xa1, 0xa1, 0xa3, 0x76, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
};

static const uint8_t fpm_table_p521[CPT_P521_LEN] = {
	0xf9, 0x7e, 0x7e, 0x31, 0xc2, 0xe5, 0xbd, 0x66, 0x33, 0x48, 0xb3, 0xc1,
	0x85, 0x6a, 0x42, 0x9b, 0xfe, 0x1d, 0xc1, 0x27, 0xa2, 0xff, 0xa8, 0xde,
	0xa1, 0x4b, 0x5e, 0x77, 0xef, 0xe7, 0x59, 0x28, 0xf8, 0x28, 0xaf, 0x60,
	0x6b, 0x4d, 0x3d, 0xba, 0x9c, 0x64, 0x81, 0x39, 0x05, 0x3f, 0xb5, 0x21,
	0x9e, 0x3e, 0xcb, 0x66, 0x23, 0x95, 0xb4, 0x42, 0x85, 0x8e, 0x06, 0xb7,
	0x04, 0x04, 0xe9, 0xcd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6,
	0x88, 0xbe, 0x94, 0x76, 0x9f, 0xd1, 0x66, 0x50, 0x35, 0x3c, 0x70, 0x86,
	0xa2, 0x72, 0xc2, 0x40, 0xc5, 0x50, 0xb9, 0x01, 0x3f, 0xad, 0x07, 0x61,
	0x97, 0xee, 0x72, 0x99, 0x5e, 0xf4, 0x26, 0x40, 0x17, 0xaf, 0xbd, 0x17,
	0x27, 0x3e, 0x66, 0x2c, 0x98, 0xf5, 0x44, 0x49, 0x57, 0x9b, 0x44, 0x68,
	0x5c, 0x8a, 0x5f, 0xb4, 0x2c, 0x7d, 0x1b, 0xd9, 0x39, 0x29, 0x6a, 0x78,
	0x9a, 0x3b, 0xc0, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x18,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x69, 0xca, 0xd3, 0xcc, 0xc4, 0xd6, 0xab, 0x08, 0x3a, 0xdb, 0x57, 0x77,
	0x3b, 0x89, 0x90, 0xb8, 0xd0, 0xca, 0xd8, 0xce, 0x8d, 0x95, 0x88, 0x01,
	0xcb, 0x57, 0x2e, 0x66, 0x6d, 0x72, 0x8f, 0x9e, 0xe3, 0xd9, 0xe7, 0xc4,
	0xcd, 0x51, 0x31, 0xfc, 0xaf, 0xce, 0xb6, 0xb0, 0x61, 0x45, 0xdc, 0x06,
	0x12, 0xec, 0xd3, 0x92, 0xe2, 0x13, 0x04, 0x3a, 0xbd, 0x59, 0x92, 0x94,
	0x3a, 0x64, 0xc8, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b,
	0x86, 0x23, 0xbd, 0xbb, 0xf6, 0xea, 0x9c, 0xf1, 0x3a, 0xad, 0x94, 0x95,
	0x54, 0x7a, 0xa6, 0x50, 0xd3, 0xd8, 0x53, 0xfc, 0xbe, 0xb2, 0x71, 0x59,
	0x3d, 0x25, 0xa6, 0x48, 0x30, 0xb4, 0x08, 0x33, 0x12, 0xd1, 0x88, 0xe8,
	0xde, 0xc5, 0x1b, 0xd1, 0x83, 0x63, 0x30, 0xd2, 0xb3, 0x48, 0xc3, 0xfa,
	0x9d, 0xf5, 0x0c, 0xfe, 0x73, 0xc2, 0xea, 0x59, 0xb5, 0xdf, 0xfb, 0x20,
	0x61, 0xde, 0xd0, 0xb8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x8a,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xdc, 0x33, 0x41, 0x6e, 0xc8, 0xc5, 0xbc, 0xb2, 0xde, 0xfb, 0x4a, 0x9c,
	0xe2, 0x44, 0x91, 0x0b, 0x27, 0xc3, 0x56, 0x1b, 0x53, 0xa8, 0xf7, 0xb9,
	0x10, 0x88, 0xbb, 0x9e, 0xf6, 0x94, 0xd7, 0xb1, 0x98, 0xfa, 0x92, 0xaa,
	0xa6, 0xd2, 0xc7, 0x82, 0x53, 0xc2, 0xa3, 0xdb, 0x3b, 0xa3, 0x7d, 0xd4,
	0x67, 0xfc, 0x7c, 0xab, 0xd5, 0x93, 0x4b, 0xbc, 0x0c, 0x72, 0xcf, 0x96,
	0x93, 0xbb, 0x09, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xaf,
	0xf3, 0x2e, 0xbb, 0x1f, 0x13, 0xda, 0xb4, 0x57, 0x7c, 0x36, 0x11, 0xdf,
	0xad, 0x23, 0x53, 0x70, 0xd8, 0x6d, 0x54, 0xdb, 0xab, 0x9e, 0x13, 0x10,
	0xbf, 0x40, 0x10, 0xf1, 0x61, 0x85, 0xbf, 0x0d, 0x94, 0x6d, 0xb5, 0x6e,
	0x31, 0x3c, 0x69, 0xf5, 0x3b, 0x67, 0x3c, 0x92, 0xe3, 0x77, 0x73, 0x27,
	0x58, 0x7a, 0x4e, 0xa7, 0x47, 0x7a, 0xbd, 0xe6, 0xae, 0x87, 0xa6, 0x00,
	0xd8, 0xaa, 0xa4, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x09,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xbf, 0x2f, 0xfc, 0xb7, 0x6e, 0x64, 0x0a, 0x8d, 0x63, 0x47, 0x95, 0x01,
	0xa0, 0xb5, 0xa0, 0x7e, 0x55, 0xbb, 0x30, 0x01, 0x5f, 0x36, 0xf2, 0xe7,
	0x98, 0x90, 0xf9, 0x99, 0x05, 0x8a, 0x67, 0x6a, 0xd9, 0xee, 0x34, 0x1b,
	0x45, 0x5c, 0x0d, 0x27, 0x6c, 0x95, 0x78, 0x0c, 0x18, 0xe0, 0x8f, 0xc8,
	0xeb, 0x63, 0xa6, 0x75, 0x44, 0x2a, 0x07, 0x5d, 0xce, 0x46, 0xa1, 0xa5,
	0xfb, 0x69, 0xda, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6b,
	0x8c, 0x61, 0x89, 0x34, 0xf3, 0xed, 0x62, 0x53, 0xe0, 0x03, 0x42, 0x44,
	0x6e, 0x94, 0x33, 0xb4, 0x98, 0x9b, 0x99, 0x21, 0x42, 0x60, 0xb2, 0xc5,
	0x11, 0x69, 0x98, 0x04, 0xd9, 0xdf, 0x47, 0x47, 0x12, 0x5f, 0xe6, 0x1f,
	0x76, 0x3a, 0x7f, 0x63, 0xa9, 0x72, 0x78, 0x6d, 0xc6, 0xec, 0x39, 0x5a,
	0x66, 0x2f, 0x9f, 0xe7, 0xb7, 0xb7, 0xb8, 0xc6, 0xfb, 0x64, 0x4a, 0x61,
	0x54, 0x56, 0x55, 0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8a,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x52, 0x3c, 0xf9, 0xb2, 0xfd, 0xe7, 0x22, 0x35, 0x5c, 0x90, 0xde, 0xf0,
	0xe8, 0xe0, 0x99, 0x59, 0xfe, 0x13, 0x1b, 0x9c, 0xd3, 0xcf, 0x46, 0x66,
	0xc9, 0x14, 0x31, 0x30, 0xc1, 0x32, 0x76, 0xad, 0xa7, 0xdc, 0xdd, 0xc1,
	0x85, 0xe2, 0x36, 0x37, 0x09, 0x45, 0x74, 0xcc, 0xf5, 0x14, 0x11, 0xd7,
	0xf3, 0xfc, 0x87, 0xc4, 0xbd, 0x29, 0xfe, 0xd7, 0x2c, 0xc3, 0x2d, 0x3f,
	0x17, 0x1c, 0xef, 0xb7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x32,
	0xdf, 0xa9, 0x2d, 0x3e, 0x35, 0xcc, 0x7a, 0x6c, 0xdd, 0x6d, 0xc6, 0x86,
	0xd4, 0xe4, 0x78, 0x8b, 0xcb, 0x66, 0xcd, 0xe2, 0x1f, 0x74, 0xbb, 0xe0,
	0xb0, 0xe9, 0xff, 0x6a, 0xf6, 0x7e, 0xc3, 0x95, 0x18, 0x6c, 0xfa, 0x86,
	0x07, 0xb9, 0xdd, 0xff, 0xe8, 0x67, 0xde, 0x2f, 0xcf, 0x2d, 0xfd, 0x72,
	0x49, 0x8c, 0x21, 0x91, 0xe2, 0x4e, 0xd3, 0x15, 0x2d, 0xf0, 0xac, 0xf8,
	0xf7, 0x37, 0xe8, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9f,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x52, 0x6f, 0x00, 0x95, 0x74, 0x31, 0x82, 0x2a, 0x18, 0x5d, 0x92, 0xc3,
	0xeb, 0x0c, 0x4e, 0xf8, 0xc8, 0x78, 0x13, 0x76, 0x38, 0x89, 0x30, 0x98,
	0x32, 0x54, 0x7e, 0xec, 0x6a, 0x55, 0x72, 0xd0, 0xe1, 0xe8, 0xea, 0xe8,
	0xf5, 0x94, 0x62, 0x73, 0x9a, 0x9e, 0x24, 0x00, 0xc8, 0x2f, 0x4f, 0x17,
	0xfb, 0x98, 0xab, 0xff, 0xdb, 0x9f, 0x0e, 0x9b, 0x3c, 0x20, 0x1a, 0xa5,
	0x83, 0x28, 0x87, 0xa5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x95,
	0x58, 0x76, 0x12, 0xa4, 0x41, 0xc2, 0xb1, 0x4a, 0x11, 0x91, 0xb7, 0x1d,
	0xfd, 0xbf, 0x12, 0x43, 0x97, 0x39, 0x6e, 0xe7, 0xbc, 0xf5, 0x3f, 0x43,
	0xd1, 0x4b, 0xf1, 0xa7, 0x90, 0xec, 0xf9, 0x76, 0x7f, 0x14, 0x7a, 0x72,
	0x0b, 0xc6, 0xa0, 0xea, 0x40, 0x95, 0x18, 0xf8, 0xaf, 0xcb, 0xff, 0x46,
	0x30, 0x21, 0xdc, 0xa5, 0x32, 0x17, 0x0c, 0x93, 0x88, 0x16, 0xd3, 0xee,
	0x33, 0xf2, 0x46, 0xdd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xcc, 0x7d, 0x6e, 0x15, 0x2f, 0x97, 0x03, 0x30, 0x50, 0xb1, 0xf6, 0x9e,
	0x03, 0x00, 0x75, 0x86, 0xfc, 0x0e, 0x37, 0x04, 0x58, 0x25, 0x83, 0x51,
	0xa8, 0x5d, 0x47, 0xe3, 0x56, 0xd8, 0xaf, 0x60, 0x1c, 0x89, 0x3b, 0x86,
	0x8a, 0xc7, 0x42, 0x7f, 0x8e, 0x70, 0xdd, 0xd9, 0x5c, 0xcf, 0x72, 0xde,
	0x1e, 0xd8, 0x3e, 0x13, 0x16, 0x06, 0x0a, 0x41, 0xc9, 0x00, 0xb8, 0x12,
	0x6d, 0xa8, 0x80, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f,
	0x3a, 0xc0, 0x08, 0x85, 0x5c, 0x6c, 0x32, 0x0b, 0x56, 0xa9, 0xb0, 0x12,
	0x24, 0x1f, 0x0c, 0xd6, 0x47, 0x94, 0x1b, 0x89, 0x97, 0x5d, 0xe7, 0x17,
	0xe7, 0x09, 0xef, 0x78, 0xdc, 0xe4, 0x8a, 0x06, 0x48, 0x2f, 0x2a, 0x6a,
	0x01, 0x56, 0xf2, 0xbd, 0xe7, 0xbd, 0xca, 0x0d, 0xae, 0x0a, 0x8a, 0x52,
	0x8e, 0x41, 0xf9, 0x41, 0x3f, 0x45, 0x94, 0x01, 0xea, 0x6d, 0x1c, 0x40,
	0x60, 0x16, 0x6a, 0x6e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa2,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xa6, 0x6c, 0xcc, 0xc8, 0xd6, 0xe1, 0xbd, 0x72, 0x1e, 0xa4, 0xe9, 0x21,
	0x4a, 0xe2, 0xfa, 0x5c, 0x66, 0x77, 0x5a, 0xf2, 0x2d, 0x02, 0x1f, 0xa7,
	0x6d, 0x71, 0x1d, 0xfb, 0x2a, 0x4c, 0x46, 0x77, 0xdd, 0xaa, 0xe8, 0xbb,
	0x5a, 0xe3, 0x80, 0xb3, 0x53, 0x15, 0x89, 0x94, 0x60, 0x0f, 0x11, 0xfc,
	0xfe, 0xb1, 0x22, 0xdb, 0xda, 0x94, 0xd4, 0x43, 0x7c, 0xbf, 0x1a, 0xfa,
	0xdf, 0xfc, 0x21, 0xdd, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06,
	0x03, 0xcf, 0xa7, 0x31, 0x83, 0x4b, 0xf8, 0x91, 0x4e, 0x01, 0x60, 0x85,
	0x63, 0x0b, 0x80, 0x32, 0x90, 0xcf, 0x9b, 0x59, 0x49, 0xdb, 0x4d, 0x96,
	0x96, 0xfd, 0x26, 0x14, 0x33, 0x5c, 0x9d, 0xdd, 0xc0, 0x21, 0x45, 0x10,
	0x8e, 0x3b, 0x98, 0xfb, 0x6d, 0xed, 0x06, 0x33, 0x1d, 0xa2, 0xea, 0x2f,
	0x2b, 0xda, 0x6d, 0x76, 0x9d, 0x0e, 0xad, 0x76, 0x4b, 0xa0, 0x0e, 0x99,
	0xf3, 0xe4, 0xfb, 0xf2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x69,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xb0, 0x31, 0x1b, 0x24, 0xcd, 0xed, 0xb3, 0xd3, 0x61, 0x14, 0xed, 0xd7,
	0xe5, 0xc3, 0xfb, 0x9d, 0x8e, 0x02, 0x64, 0x37, 0x1a, 0x6b, 0xb2, 0xa1,
	0xd7, 0xcc, 0x12, 0x87, 0x5e, 0xcd, 0xbe, 0x72, 0x6f, 0x71, 0x72, 0x87,
	0x15, 0x58, 0x28, 0xe1, 0xb8, 0x3a, 0xef, 0xaa, 0x5f, 0x9b, 0xf5, 0x5d,
	0x29, 0xd8, 0xc7, 0x42, 0x7e, 0x0b, 0x8a, 0xce, 0x18, 0xfe, 0x72, 0xa4,
	0x64, 0x25, 0x9f, 0x35, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x8a,
	0x80, 0xf7, 0xfb, 0xb6, 0x2a, 0xa4, 0x37, 0x5d, 0x8d, 0x40, 0xb9, 0xee,
	0x6d, 0x0b, 0xec, 0x41, 0x55, 0xdd, 0x1c, 0xe0, 0x41, 0xdb, 0xbd, 0x4b,
	0xba, 0x77, 0x95, 0x12, 0x29, 0x4b, 0xc6, 0xcc, 0x38, 0x3d, 0x3a, 0x90,
	0x4a, 0xa6, 0x6d, 0xb2, 0x3c, 0x46, 0x18, 0x62, 0x6c, 0xa8, 0xd7, 0x01,
	0xf8, 0x0e, 0x3b, 0x59, 0x19, 0xbc, 0xfc, 0x0b, 0x9b, 0x63, 0x7d, 0x37,
	0x5f, 0x56, 0xcd, 0xc1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x5e,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x74, 0x3b, 0x45, 0x1f, 0x86, 0xeb, 0x01, 0xd4, 0x63, 0x9c, 0x35, 0xde,
	0x4d, 0x93, 0xf5, 0x7f, 0xab, 0x5e, 0xaa, 0xc4, 0x71, 0xd1, 0x5e, 0x15,
	0xb7, 0x7b, 0xd7, 0x5b, 0xbe, 0x29, 0x19, 0xc6, 0x34, 0xb5, 0x39, 0x64,
	0x2d, 0x2e, 0x8f, 0xa7, 0x1c, 0x04, 0xb3, 0xbf, 0xc5, 0x70, 0xd5, 0x49,
	0x1b, 0x77, 0x64, 0x58, 0x6e, 0xaf, 0x33, 0xdc, 0xbd, 0x40, 0x33, 0xb2,
	0x4d, 0xf5, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x77,
	0x35, 0xe3, 0xa8, 0x3d, 0x9d, 0x79, 0x2a, 0xa8, 0x5e, 0xb6, 0x8e, 0xa9,
	0x73, 0x85, 0xfb, 0xe3, 0xcc, 0x3b, 0xf6, 0xb7, 0x91, 0xd7, 0x82, 0x0f,
	0x53, 0x9e, 0x98, 0xd0, 0x0b, 0x57, 0x18, 0x65, 0xdc, 0xc8, 0x13, 0x13,
	0x2f, 0xdc, 0x47, 0x3e, 0x52, 0x40, 0x20, 0x14, 0xb8, 0xa3, 0xef, 0xf6,
	0x93, 0x7b, 0x43, 0xd3, 0x30, 0x8e, 0x48, 0x85, 0x86, 0xd0, 0xac, 0xd5,
	0xbb, 0x8e, 0x72, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x77,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x49, 0x2d, 0xc8, 0x61, 0xce, 0x07, 0x6e, 0x91, 0x1b, 0xed, 0x06, 0xac,
	0xb5, 0xa5, 0x79, 0x7a, 0x80, 0x25, 0x4f, 0x8a, 0x10, 0xd2, 0xef, 0x33,
	0x97, 0x66, 0x0e, 0x4f, 0xd1, 0xa3, 0x6b, 0x3e, 0x7d, 0xa6, 0xe9, 0x2f,
	0x35, 0xaa, 0xee, 0xfe, 0xed, 0x65, 0x1c, 0x3f, 0x89, 0x08, 0xbd, 0xe5,
	0x99, 0xed, 0x82, 0xc0, 0x14, 0xb0, 0xc8, 0xd7, 0x76, 0xd6, 0xd5, 0x2e,
	0x23, 0x42, 0x44, 0xe8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbb,
	0x49, 0xed, 0x49, 0xd7, 0x20, 0x06, 0x0e, 0x54, 0x5d, 0xae, 0xaa, 0x77,
	0x53, 0x5d, 0x38, 0x1c, 0x02, 0x7b, 0x07, 0x4e, 0x2f, 0x84, 0x36, 0x83,
	0xe1, 0xc5, 0x2f, 0x6c, 0x4f, 0x54, 0x94, 0x1d, 0xc5, 0xa9, 0x18, 0x83,
	0x69, 0xbe, 0x08, 0xb0, 0xd8, 0xe8, 0x81, 0x44, 0x0f, 0xc0, 0x0b, 0x32,
	0x5c, 0x14, 0x9a, 0x7f, 0x39, 0x77, 0x6c, 0xe6, 0x45, 0xab, 0xab, 0x75,
	0x48, 0x39, 0xd3, 0xc5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x27,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xaf, 0xe2, 0xc9, 0x9e, 0x41, 0xc5, 0xad, 0x4e, 0x91, 0xbd, 0x0f, 0x74,
	0x04, 0x97, 0x37, 0x24, 0x11, 0x0d, 0x39, 0x02, 0x70, 0x29, 0xce, 0x18,
	0x31, 0x39, 0x96, 0x1a, 0xe4, 0xd8, 0xcd, 0x86, 0x20, 0x41, 0xf8, 0xf2,
	0x65, 0x04, 0x09, 0xad, 0xf6, 0x03, 0x67, 0x47, 0x5c, 0x3a, 0xb7, 0xe7,
	0x0c, 0xfa, 0xe7, 0xe4, 0x5f, 0xc5, 0x9d, 0x58, 0xa2, 0xa1, 0x41, 0x9d,
	0xb0, 0x1d, 0xc2, 0x3d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xa8,
	0xe5, 0x88, 0xb5, 0xf2, 0xd7, 0x71, 0x52, 0xe1, 0x94, 0xf5, 0xe2, 0x1e,
	0x3d, 0x65, 0x7a, 0x29, 0x91, 0xbe, 0x93, 0xd9, 0xb2, 0x9b, 0xea, 0xa5,
	0x8c, 0xcc, 0xd4, 0x2c, 0xe4, 0x96, 0xf3, 0x7a, 0xbf, 0x29, 0x5a, 0x89,
	0x7c, 0xc5, 0xff, 0xc9, 0x38, 0x64, 0xd8, 0x5e, 0xd8, 0xe3, 0x06, 0x5a,
	0xe1, 0x35, 0xda, 0x88, 0x59, 0x3c, 0xc5, 0x50, 0xd2, 0xe2, 0x1f, 0x06,
	0xa2, 0xac, 0xff, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x3a,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x9f, 0x12, 0xaf, 0x8f, 0xf1, 0x27, 0x32, 0x22, 0x13, 0xd3, 0x23, 0xa3,
	0xc7, 0xb0, 0xbf, 0x6d, 0x7b, 0xbb, 0x66, 0x7d, 0xa8, 0x35, 0xd8, 0xea,
	0xd3, 0x14, 0x5f, 0xdd, 0xb3, 0x0e, 0x21, 0x3c, 0xe1, 0xd3, 0x17, 0x0f,
	0x86, 0x13, 0x38, 0x9c, 0x50, 0x37, 0x17, 0x64, 0x48, 0x12, 0x41, 0xf8,
	0x53, 0x7c, 0x14, 0xdd, 0xc2, 0xae, 0x9d, 0xbe, 0x86, 0x81, 0x31, 0x96,
	0x29, 0x64, 0x59, 0xc1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79,
	0xa4, 0xf4, 0x01, 0x3e, 0x71, 0xb7, 0x6e, 0x68, 0x31, 0x86, 0x62, 0xbc,
	0xf0, 0x1d, 0xcb, 0xa9, 0xd0, 0x55, 0xec, 0xb8, 0x18, 0xe8, 0x7f, 0x1f,
	0xc7, 0xcc, 0xf5, 0xfa, 0xbc, 0xf1, 0x44, 0x74, 0x3a, 0x39, 0x90, 0x76,
	0xf8, 0xa9, 0x58, 0x8a, 0x9f, 0x38, 0x62, 0x02, 0xa5, 0x38, 0x72, 0x4b,
	0x15, 0x80, 0xc0, 0xbb, 0x18, 0x20, 0xa4, 0xdd, 0xed, 0xb8, 0x90, 0x86,
	0xca, 0x8f, 0xdd, 0xa7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd8,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xf4, 0xab, 0xf8, 0x15, 0x29, 0x73, 0x14, 0x27, 0xf5, 0xb8, 0x6b, 0x12,
	0xf9, 0xe4, 0x22, 0xc5, 0x4d, 0xd1, 0xa9, 0x71, 0x21, 0x97, 0xfd, 0x42,
	0x02, 0x10, 0x34, 0x7f, 0x6f, 0xfd, 0x80, 0xff, 0x22, 0x5c, 0xd3, 0xa0,
	0x7e, 0x37, 0x56, 0x30, 0xc2, 0x59, 0xfe, 0x53, 0xff, 0x95, 0x16, 0xb6,
	0xc0, 0xb8, 0x01, 0x10, 0xa6, 0x89, 0xdb, 0x24, 0xc3, 0xbd, 0xce, 0xb5,
	0x77, 0x0e, 0x1f, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6e,
	0x0b, 0x89, 0x2f, 0x5c, 0xdd, 0x8c, 0x05, 0xb9, 0x41, 0x7c, 0x1a, 0xce,
	0xbf, 0x23, 0x01, 0x0c, 0xe9, 0x74, 0x9c, 0x6c, 0xbb, 0xf7, 0xf7, 0x53,
	0xfe, 0xc2, 0xd2, 0xe4, 0xb3, 0x3f, 0x5a, 0x06, 0x41, 0x02, 0x30, 0xdc,
	0x2f, 0xa4, 0x1c, 0x6b, 0x29, 0x27, 0x10, 0x68, 0xa1, 0x8a, 0x14, 0x16,
	0xab, 0xb9, 0xaf, 0xe4, 0x73, 0xb9, 0xec, 0x29, 0x72, 0xd9, 0x03, 0x5c,
	0x26, 0x79, 0x2a, 0x47, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x0e,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x28, 0xff, 0x7b, 0xe0, 0xfc, 0x9e, 0x23, 0xd2, 0xf5, 0xe0, 0x7f, 0xef,
	0xb8, 0x63, 0xa2, 0x40, 0x1b, 0x61, 0x96, 0xe4, 0x67, 0xcb, 0x5b, 0x0e,
	0x30, 0xa9, 0xa3, 0x6b, 0x9e, 0xc2, 0xfb, 0xfc, 0x06, 0xef, 0x3f, 0x4e,
	0xdf, 0x56, 0x80, 0x15, 0x72, 0x9b, 0xb1, 0x97, 0xc9, 0xf5, 0x26, 0x0b,
	0x52, 0xb0, 0xb4, 0xfe, 0xb6, 0x04, 0x15, 0x86, 0x26, 0x51, 0xb3, 0x20,
	0x63, 0xf1, 0x99, 0xba, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x34,
	0xdc, 0xfa, 0xf4, 0x3a, 0x05, 0xb6, 0x4e, 0xa9, 0x1d, 0xb7, 0x31, 0x9c,
	0x19, 0x16, 0x32, 0xe8, 0x3a, 0x60, 0xe8, 0xab, 0x97, 0x7a, 0x9c, 0x9d,
	0x85, 0x42, 0x8e, 0x55, 0xee, 0x3a, 0x97, 0x81, 0x71, 0xc3, 0x42, 0x1b,
	0x5b, 0x6d, 0x51, 0xc0, 0x01, 0xed, 0x96, 0x12, 0x52, 0x56, 0x02, 0x26,
	0x6c, 0xc1, 0xdb, 0xed, 0x90, 0x72, 0x2e, 0x36, 0xfa, 0xa6, 0x4f, 0x19,
	0xc2, 0xc7, 0x0c, 0xba, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5b,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
};

static const struct fpm_entry fpm_table_scalar[CPT_EC_ID_PMAX] = {
	{
		.data = fpm_table_p192,
		.len = sizeof(fpm_table_p192)
	},
	{
		.data = fpm_table_p224,
		.len = sizeof(fpm_table_p224)
	},
	{
		.data = fpm_table_p256,
		.len = sizeof(fpm_table_p256)
	},
	{
		.data = fpm_table_p384,
		.len = sizeof(fpm_table_p384)
	},
	{
		.data = fpm_table_p521,
		.len = sizeof(fpm_table_p521)
	}
};

static rte_spinlock_t lock = RTE_SPINLOCK_INITIALIZER;
static uint8_t *fpm_table;
static int nb_devs;

int cpt_fpm_init(uint64_t *fpm_table_iova)
{
	int i, len = 0;
	uint8_t *data;

	if (fpm_table_iova == NULL)
		return -EINVAL;

	rte_spinlock_lock(&lock);

	if (nb_devs != 0)
		goto update_nb_devs;

	/* Compute FPM table size for all supported curves */
	for (i = 0; i < CPT_EC_ID_PMAX; i++)
		len += fpm_table_scalar[i].len;

	/* Allocate shared FPM table */
	fpm_table = rte_malloc(NULL, len, 0);

	if (fpm_table == NULL) {
		rte_spinlock_unlock(&lock);
		return -ENOMEM;
	}
	data = fpm_table;

	for (i = CPT_EC_ID_P192; i < CPT_EC_ID_PMAX; i++) {
		memcpy(data, fpm_table_scalar[i].data, fpm_table_scalar[i].len);
		/* Convert iova address to big endian to be used by cpt */
		fpm_table_iova[i] = rte_cpu_to_be_64(rte_mem_virt2iova(data));

		data = RTE_PTR_ADD(data, fpm_table_scalar[i].len);
	}

update_nb_devs:

	/* Increment number of devices using FPM table */
	nb_devs++;

	rte_spinlock_unlock(&lock);

	return 0;
}

void cpt_fpm_clear(void)
{
	rte_spinlock_lock(&lock);

	/* Decrement number of devices using FPM table */
	nb_devs--;
	if ((nb_devs == 0) && (fpm_table != NULL))
		rte_free(fpm_table);

	rte_spinlock_unlock(&lock);
}
